/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/
import {SuperMap} from '../SuperMap';
import toPairs from 'lodash.topairs';

/**
 * @class CartoCSS
 * @deprecatedclass SuperMap.CartoCSS
 * @classdesc CartoCSS 解析类，其主要功能为将 CartoCSS 字符串解析为 CartoCSS 的 shader 属性风格对象。
 * @category BaseTypes Style
 * @param {string} cartoStr -  样式表字符串。
 * @example
 * var cartocss = "@provinceLineColor:#ddd;
 *                 #China_Provinces_L___China400{
 *                      line-dasharray:10,10;
 *                      line-color:@provinceLineColor;
 *                      line-width:1;
 *                 }";
 * new CartoCSS(cartocss);
 * @usage
 */
/*eslint no-useless-escape: "off"*/
export class CartoCSS {

    constructor(cartoStr) {
        this.env = null;

        /**
         * @member CartoCSS.prototype.parser
         * @description 解析器。
         */
        this.parser = null;

        /**
         * @member CartoCSS.prototype.ruleSet
         * @description CartoCSS 规则对象。
         */
        this.ruleSet = null;

        /**
         * @member CartoCSS.prototype.cartoStr
         * @description CartoCSS 样式表字符串。
         */
        this.cartoStr = "";

        /**
         * @member CartoCSS.prototype.shaders
         * @description Carto 着色器集。
         */
        this.shaders = null;

        if (typeof cartoStr === "string") {
            this.cartoStr = cartoStr;
            this.env = {
                frames: [],
                errors: [],
                error: function (obj) {
                    this.errors.push(obj);
                }
            };
            this.parser = this.getParser(this.env);
            this.parse(cartoStr);
            this.shaders = this.toShaders();
        }
    }

    /**
     * @function CartoCSS.prototype.getParser
     * @description 获取 CartoCSS 解析器。
     */
    getParser(env) {
        var input,       // LeSS input string
            i,           // current index in `input`
            j,           // current chunk
            temp,        // temporarily holds a chunk's state, for backtracking
            memo,        // temporarily holds `i`, when backtracking
            furthest,    // furthest index the parser has gone to
            chunks,      // chunkified input
            current,     // index of current chunk, in `input`
            parser;

        var that = this;

        // This function is called after all files
        // have been imported through `@import`.
        var finish = function () {//NOSONAR
            //所有文件导入完成之后调用
        };

        function save() {
            temp = chunks[j];
            memo = i;
            current = i;
        }

        function restore() {
            chunks[j] = temp;
            i = memo;
            current = i;
        }

        function sync() {
            if (i > current) {
                chunks[j] = chunks[j].slice(i - current);
                current = i;
            }
        }

        //
        // Parse from a token, regexp or string, and move forward if match
        //
        function _match(tok) {
            var match, length, c, endIndex;

            // Non-terminal
            if (tok instanceof Function) {
                return tok.call(parser.parsers);
                // Terminal
                // Either match a single character in the input,
                // or match a regexp in the current chunk (chunk[j]).
            } else if (typeof(tok) === 'string') {
                match = input.charAt(i) === tok ? tok : null;
                length = 1;
                sync();
            } else {
                sync();

                match = tok.exec(chunks[j]);
                if (match) {
                    length = match[0].length;
                } else {
                    return null;
                }
            }

            // The match is confirmed, add the match length to `i`,
            // and consume any extra white-space characters (' ' || '\n')
            // which come after that. The reason for this is that LeSS's
            // grammar is mostly white-space insensitive.
            if (match) {
                var mem = i += length;
                endIndex = i + chunks[j].length - length;

                while (i < endIndex) {
                    c = input.charCodeAt(i);
                    if (!(c === 32 || c === 10 || c === 9)) {
                        break;
                    }
                    i++;
                }
                chunks[j] = chunks[j].slice(length + (i - mem));
                current = i;

                if (chunks[j].length === 0 && j < chunks.length - 1) {
                    j++;
                }

                if (typeof(match) === 'string') {
                    return match;
                } else {
                    return match.length === 1 ? match[0] : match;
                }
            }
        }

        // Same as _match(), but don't change the state of the parser,
        // just return the match.
        function peek(tok) {
            if (typeof(tok) === 'string') {
                return input.charAt(i) === tok;
            } else {
                return !!tok.test(chunks[j]);
            }
        }

        // Make an error object from a passed set of properties.
        // Accepted properties:
        // - `message`: Text of the error message.
        // - `filename`: Filename where the error occurred.
        // - `index`: Char. index where the error occurred.
        function makeError(err) {
            var einput;

            var defautls = {
                index: furthest,
                filename: env.filename,
                message: 'Parse error.',
                line: 0,
                column: -1
            };
            for (var prop in defautls) {
                if (err[prop] === 0) {
                    err[prop] = defautls[prop];
                }
            }

            if (err.filename && that.env.inputs && that.env.inputs[err.filename]) {
                einput = that.env.inputs[err.filename];
            } else {
                einput = input;
            }

            err.line = (einput.slice(0, err.index).match(/\n/g) || '').length + 1;
            for (var n = err.index; n >= 0 && einput.charAt(n) !== '\n'; n--) {
                err.column++;
            }
            return new Error([err.filename, err.line, err.column, err.message].join(";"));
        }

        this.env = env = env || {};
        this.env.filename = this.env.filename || null;
        this.env.inputs = this.env.inputs || {};

        // The Parser
        parser = {

            // Parse an input string into an abstract syntax tree.
            // Throws an error on parse errors.
            parse: function (str) {
                var root, error = null;

                i = j = current = furthest = 0;
                chunks = [];
                input = str.replace(/\r\n/g, '\n');
                if (env.filename) {
                    that.env.inputs[env.filename] = input;
                }

                // Split the input into chunks.
                chunks = (function (chunks) {
                    var j = 0,
                        skip = /(?:@\{[\w-]+\}|[^"'`\{\}\/\(\)\\])+/g,
                        comment = /\/\*(?:[^*]|\*+[^\/*])*\*+\/|\/\/.*/g,
                        string = /"((?:[^"\\\r\n]|\\.)*)"|'((?:[^'\\\r\n]|\\.)*)'|`((?:[^`]|\\.)*)`/g,
                        level = 0,
                        match,
                        chunk = chunks[0],
                        inParam;

                    for (var i = 0, c, cc; i < input.length;) {
                        skip.lastIndex = i;
                        if (match = skip.exec(input)) {
                            if (match.index === i) {
                                i += match[0].length;
                                chunk.push(match[0]);
                            }
                        }
                        c = input.charAt(i);
                        comment.lastIndex = string.lastIndex = i;

                        if (match = string.exec(input)) {
                            if (match.index === i) {
                                i += match[0].length;
                                chunk.push(match[0]);
                                continue;
                            }
                        }

                        if (!inParam && c === '/') {
                            cc = input.charAt(i + 1);
                            if (cc === '/' || cc === '*') {
                                if (match = comment.exec(input)) {
                                    if (match.index === i) {
                                        i += match[0].length;
                                        chunk.push(match[0]);
                                        continue;
                                    }
                                }
                            }
                        }

                        switch (c) {
                            case '{'://NOSONAR
                                if (!inParam) {
                                    level++;
                                    chunk.push(c);
                                    break;
                                }
                            case '}'://NOSONAR
                                if (!inParam) {
                                    level--;
                                    chunk.push(c);
                                    chunks[++j] = chunk = [];
                                    break;
                                }
                            case '('://NOSONAR
                                if (!inParam) {
                                    inParam = true;
                                    chunk.push(c);
                                    break;
                                }
                            case ')'://NOSONAR
                                if (inParam) {
                                    inParam = false;
                                    chunk.push(c);
                                    break;
                                }
                            default:
                                chunk.push(c);
                                break;
                        }

                        i++;
                    }
                    if (level !== 0) {
                        error = {
                            index: i - 1,
                            type: 'Parse',
                            message: (level > 0) ? "missing closing `}`" : "missing opening `{`"
                        };
                    }

                    return chunks.map(function (c) {
                        return c.join('');
                    });
                })([[]]);

                if (error) {
                    throw makeError(error);
                }

                // Sort rules by specificity: this function expects selectors to be
                // split already.
                //
                // Written to be used as a .sort(Function);
                // argument.
                //
                // [1, 0, 0, 467] > [0, 0, 1, 520]
                var specificitySort = function (a, b) {
                    var as = a.specificity;
                    var bs = b.specificity;

                    if (as[0] != bs[0]) {return bs[0] - as[0];}
                    if (as[1] != bs[1]) {return bs[1] - as[1];}
                    if (as[2] != bs[2]) {return bs[2] - as[2];}
                    return bs[3] - as[3];
                };

                // Start with the primary rule.
                // The whole syntax tree is held under a Ruleset node,
                // with the `root` property set to true, so no `{}` are
                // output.
                root = new CartoCSS.Tree.Ruleset([], _match(this.parsers.primary));
                root.root = true;

                // Get an array of Ruleset objects, flattened
                // and sorted according to specificitySort
                root.toList = (function () {
                    return function (env) {
                        env.error = function (e) {
                            if (!env.errors) {env.errors = new Error('');}
                            if (env.errors.message) {
                                env.errors.message += '\n' + makeError(e).message;
                            } else {
                                env.errors.message = makeError(e).message;
                            }
                        };
                        env.frames = env.frames || [];


                        // call populates Invalid-caused errors
                        var definitions = this.flatten([], [], env);
                        definitions.sort(specificitySort);
                        return definitions;
                    };
                })();

                return root;
            },

            // Here in, the parsing rules/functions
            //
            // The basic structure of the syntax tree generated is as follows:
            //
            //   Ruleset ->  Rule -> Value -> Expression -> Entity
            //
            //  In general, most rules will try to parse a token with the `_match()` function, and if the return
            //  value is truly, will return a new node, of the relevant type. Sometimes, we need to check
            //  first, before parsing, that's when we use `peek()`.
            parsers: {
                // The `primary` rule is the *entry* and *exit* point of the parser.
                // The rules here can appear at any level of the parse tree.
                //
                // The recursive nature of the grammar is an interplay between the `block`
                // rule, which represents `{ ... }`, the `ruleset` rule, and this `primary` rule,
                // as represented by this simplified grammar:
                //
                //     primary  →  (ruleset | rule)+
                //     ruleset  →  selector+ block
                //     block    →  '{' primary '}'
                //
                // Only at one point is the primary rule not called from the
                // block rule: at the root level.
                primary: function () {
                    var node, root = [];

                    while ((node = _match(this.rule) || _match(this.ruleset) ||
                        _match(this.comment)) ||
                    _match(/^[\s\n]+/) || (node = _match(this.invalid))) {
                        if (node) {root.push(node);}
                    }
                    return root;
                },

                invalid: function () {
                    var chunk = _match(/^[^;\n]*[;\n]/);

                    // To fail gracefully, match everything until a semicolon or linebreak.
                    if (chunk) {
                        return new CartoCSS.Tree.Invalid(chunk, memo);
                    }
                },

                // We create a Comment node for CSS comments `/* */`,
                // but keep the LeSS comments `//` silent, by just skipping
                // over them.
                comment: function () {
                    var comment;

                    if (input.charAt(i) !== '/') {return;}

                    if (input.charAt(i + 1) === '/') {
                        return new CartoCSS.Tree.Comment(_match(/^\/\/.*/), true);
                    } else if (comment = _match(/^\/\*(?:[^*]|\*+[^\/*])*\*+\/\n?/)) {
                        return new CartoCSS.Tree.Comment(comment);
                    }
                },

                // Entities are tokens which can be found inside an Expression
                entities: {

                    // A string, which supports escaping " and ' "milky way" 'he\'s the one!'
                    quoted: function () {
                        if (input.charAt(i) !== '"' && input.charAt(i) !== "'") {return;}
                        var str = _match(/^"((?:[^"\\\r\n]|\\.)*)"|'((?:[^'\\\r\n]|\\.)*)'/);
                        if (str) {
                            return new CartoCSS.Tree.Quoted(str[1] || str[2]);
                        }
                    },

                    // A reference to a Mapnik field, like [NAME]
                    // Behind the scenes, this has the same representation, but Carto
                    // needs to be careful to warn when unsupported operations are used.
                    field: function () {
                        var l = '[', r = ']';
                        if (!_match(l)) {return;}
                        var field_name = _match(/(^[^\]]+)/);
                        if (!_match(r)) {return;}
                        if (field_name) {return new CartoCSS.Tree.Field(field_name[1]);}
                    },

                    // This is a comparison operator
                    comparison: function () {
                        var str = _match(/^=~|=|!=|<=|>=|<|>/);
                        if (str) {
                            return str;
                        }
                    },

                    // A catch-all word, such as: hard-light
                    // These can start with either a letter or a dash (-),
                    // and then contain numbers, underscores, and letters.
                    keyword: function () {
                        var k = _match(/^[A-Za-z\u4e00-\u9fa5-]+[A-Za-z-0-9\u4e00-\u9fa5_]*/);
                        if (k) {
                            return new CartoCSS.Tree.Keyword(k);
                        }
                    },

                    // A function call like rgb(255, 0, 255)
                    // The arguments are parsed with the `entities.arguments` parser.
                    call: function () {
                        var name, args;

                        if (!(name = /^([\w\-]+|%)\(/.exec(chunks[j]))) {return;}

                        name = name[1];

                        if (name === 'url') {
                            // url() is handled by the url parser instead
                            return null;
                        } else {
                            i += name.length;
                        }

                        var l = '(', r = ')';
                        _match(l); // Parse the '(' and consume whitespace.

                        args = _match(this.entities['arguments']);

                        if (!_match(r)) {return;}

                        if (name) {
                            return new CartoCSS.Tree.Call(name, args, i);
                        }
                    },
                    // Arguments are comma-separated expressions
                    'arguments': function () {
                        var args = [], arg;

                        while (arg = _match(this.expression)) {
                            args.push(arg);
                            var q = ',';
                            if (!_match(q)) {
                                break;
                            }
                        }

                        return args;
                    },
                    literal: function () {
                        return _match(this.entities.dimension) ||
                            _match(this.entities.keywordcolor) ||
                            _match(this.entities.hexcolor) ||
                            _match(this.entities.quoted);
                    },

                    // Parse url() tokens
                    //
                    // We use a specific rule for urls, because they don't really behave like
                    // standard function calls. The difference is that the argument doesn't have
                    // to be enclosed within a string, so it can't be parsed as an Expression.
                    url: function () {
                        var value;

                        if (input.charAt(i) !== 'u' || !_match(/^url\(/)) {return;}
                        value = _match(this.entities.quoted) || _match(this.entities.variable) ||
                            _match(/^[\-\w%@_match\/.&=:;#+?~]+/) || '';
                        var r = ')';
                        if (!_match(r)) {
                            return new CartoCSS.Tree.Invalid(value, memo, 'Missing closing ) in URL.');
                        } else {
                            return new CartoCSS.Tree.URL((typeof value.value !== 'undefined' ||
                                value instanceof CartoCSS.Tree.Variable) ?
                                value : new CartoCSS.Tree.Quoted(value));
                        }
                    },

                    // A Variable entity, such as `@fink`, in
                    //
                    //     width: @fink + 2px
                    //
                    // We use a different parser for variable definitions,
                    // see `parsers.variable`.
                    variable: function () {
                        var name, index = i;

                        if (input.charAt(i) === '@' && (name = _match(/^@[\w-]+/))) {
                            return new CartoCSS.Tree.Variable(name, index, env.filename);
                        }
                    },

                    hexcolor: function () {
                        var rgb;
                        if (input.charAt(i) === '#' && (rgb = _match(/^#([a-fA-F0-9]{6}|[a-fA-F0-9]{3})/))) {
                            return new CartoCSS.Tree.Color(rgb[1]);
                        }
                    },

                    keywordcolor: function () {
                        var rgb = chunks[j].match(/^[a-z]+/);
                        if (rgb && rgb[0] in CartoCSS.Tree.Reference.data.colors) {
                            return new CartoCSS.Tree.Color(CartoCSS.Tree.Reference.data.colors[_match(/^[a-z]+/)]);
                        }
                    },

                    // A Dimension, that is, a number and a unit. The only
                    // unit that has an effect is %
                    dimension: function () {
                        var c = input.charCodeAt(i);
                        if ((c > 57 || c < 45) || c === 47) {return;}
                        var value = _match(/^(-?\d*\.?\d+(?:[eE][-+]?\d+)?)(\%|\w+)?/);
                        if (value) {
                            return new CartoCSS.Tree.Dimension(value[1], value[2], memo);
                        }
                    }
                },

                // The variable part of a variable definition.
                // Used in the `rule` parser. Like @fink:
                variable: function () {
                    var name;

                    if (input.charAt(i) === '@' && (name = _match(/^(@[\w-]+)\s*:/))) {
                        return name[1];
                    }
                },

                // Entities are the smallest recognized token,
                // and can be found inside a rule's value.
                entity: function () {
                    var property1 = _match(this.entities.call) || _match(this.entities.literal);
                    var property2 = _match(this.entities.field) || _match(this.entities.variable);
                    var property3 = _match(this.entities.url) || _match(this.entities.keyword);
                    return property1 || property2 || property3;
                },

                // A Rule terminator. Note that we use `peek()` to check for '}',
                // because the `block` rule will be expecting it, but we still need to make sure
                // it's there, if ';' was ommitted.
                end: function () {
                    var q = ';';
                    return _match(q) || peek('}');
                },

                // Elements are the building blocks for Selectors. They consist of
                // an element name, such as a tag a class, or `*`.
                //增加对中文的支持，[\u4e00-\u9fa5]
                element: function () {
                    var e = _match(/^(?:[.#][\w\u4e00-\u9fa5\-]+|\*|Map)/);
                    if (e) {return new CartoCSS.Tree.Element(e);}
                },

                // Attachments allow adding multiple lines, polygons etc. to an
                // object. There can only be one attachment per selector.
                attachment: function () {
                    var s = _match(/^::([\w\-]+(?:\/[\w\-]+)*)/);
                    if (s) {return s[1];}
                },

                // Selectors are made out of one or more Elements, see above.
                selector: function () {
                    var a, attachment,
                        e, elements = [],
                        f, filters = new CartoCSS.Tree.Filterset(),
                        z, zooms = [],
                        segments = 0, conditions = 0;

                    while (
                        (e = _match(this.element)) ||
                        (z = _match(this.zoom)) ||
                        (f = _match(this.filter)) ||
                        (a = _match(this.attachment))
                        ) {
                        segments++;
                        if (e) {
                            elements.push(e);
                        } else if (z) {
                            zooms.push(z);
                            conditions++;
                        } else if (f) {
                            var err = filters.add(f);
                            if (err) {
                                throw makeError({
                                    message: err,
                                    index: i - 1
                                });
                            }
                            conditions++;
                        } else if (attachment) {
                            throw makeError({
                                message: 'Encountered second attachment name.',
                                index: i - 1
                            });
                        } else {
                            attachment = a;
                        }

                        var c = input.charAt(i);
                        if (c === '{' || c === '}' || c === ';' || c === ',') {
                            break;
                        }
                    }

                    if (segments) {
                        return new CartoCSS.Tree.Selector(filters, zooms, elements, attachment, conditions, memo);
                    }
                },

                filter: function () {
                    save();
                    var key, op, val, l = '[', r = ']';
                    if (!_match(l)) {return;}
                    if (key = _match(/^[a-zA-Z0-9\-_]+/) ||
                            _match(this.entities.quoted) ||
                            _match(this.entities.variable) ||
                            _match(this.entities.keyword) ||
                            _match(this.entities.field)) {
                        if (key instanceof CartoCSS.Tree.Quoted) {
                            key = new CartoCSS.Tree.Field(key.toString());
                        }
                        if ((op = _match(this.entities.comparison)) &&
                            (val = _match(this.entities.quoted) ||
                                _match(this.entities.variable) ||
                                _match(this.entities.dimension) ||
                                _match(this.entities.keyword) ||
                                _match(this.entities.field))) {
                            if (!_match(r)) {
                                throw makeError({
                                    message: 'Missing closing ] of filter.',
                                    index: memo - 1
                                });
                            }
                            if (!key.is) {key = new CartoCSS.Tree.Field(key);}
                            return new CartoCSS.Tree.Filter(key, op, val, memo, env.filename);
                        }
                    }
                },

                zoom: function () {
                    save();
                    var op, val, r = ']';
                    if (_match(/^\[\s*zoom/g) &&
                        (op = _match(this.entities.comparison)) &&
                        (val = _match(this.entities.variable) || _match(this.entities.dimension)) && _match(r)) {
                        return new CartoCSS.Tree.Zoom(op, val, memo);
                    } else {
                        // backtrack
                        restore();
                    }
                },

                // The `block` rule is used by `ruleset`
                // It's a wrapper around the `primary` rule, with added `{}`.
                block: function () {
                    var content, l = '{', r = '}';

                    if (_match(l) && (content = _match(this.primary)) && _match(r)) {
                        return content;
                    }
                },

                // div, .class, body > p {...}
                ruleset: function () {
                    var selectors = [], s, rules, q = ',';
                    save();

                    while (s = _match(this.selector)) {
                        selectors.push(s);
                        while (_match(this.comment)) {//NOSONAR
                        }
                        if (!_match(q)) {
                            break;
                        }
                        while (_match(this.comment)) {//NOSONAR
                        }
                    }
                    if (s) {
                        while (_match(this.comment)) {//NOSONAR
                        }
                    }

                    if (selectors.length > 0 && (rules = _match(this.block))) {
                        if (selectors.length === 1 &&
                            selectors[0].elements.length &&
                            selectors[0].elements[0].value === 'Map') {
                            var rs = new CartoCSS.Tree.Ruleset(selectors, rules);
                            rs.isMap = true;
                            return rs;
                        }
                        return new CartoCSS.Tree.Ruleset(selectors, rules);
                    } else {
                        // Backtrack
                        restore();
                    }
                },

                rule: function () {
                    var name, value, c = input.charAt(i);
                    save();

                    if (c === '.' || c === '#') {
                        return;
                    }

                    if (name = _match(this.variable) || _match(this.property)) {
                        value = _match(this.value);

                        if (value && _match(this.end)) {
                            return new CartoCSS.Tree.Rule(name, value, memo, env.filename);
                        } else {
                            furthest = i;
                            restore();
                        }
                    }
                },

                font: function () {
                    var value = [], expression = [], e, q = ',';

                    while (e = _match(this.entity)) {
                        expression.push(e);
                    }

                    value.push(new CartoCSS.Tree.Expression(expression));

                    if (_match(q)) {
                        while (e = _match(this.expression)) {
                            value.push(e);
                            if (!_match(q)) {
                                break;
                            }
                        }
                    }
                    return new CartoCSS.Tree.Value(value);
                },

                // A Value is a comma-delimited list of Expressions
                // In a Rule, a Value represents everything after the `:`,
                // and before the `;`.
                value: function () {
                    var e, expressions = [], q = ',';

                    while (e = _match(this.expression)) {
                        expressions.push(e);
                        if (!_match(q)) {
                            break;
                        }
                    }

                    if (expressions.length > 1) {
                        return new CartoCSS.Tree.Value(expressions.map(function (e) {
                            return e.value[0];
                        }));
                    } else if (expressions.length === 1) {
                        return new CartoCSS.Tree.Value(expressions);
                    }
                },
                // A sub-expression, contained by parenthensis
                sub: function () {
                    var e, l = '(', r = ")";
                    if (_match(l) && (e = _match(this.expression)) && _match(r)) {
                        return e;
                    }
                },
                // This is a misnomer because it actually handles multiplication
                // and division.
                multiplication: function () {
                    var m, a, op, operation, q = '/';
                    if (m = _match(this.operand)) {
                        while ((op = (_match(q) || _match('*') || _match('%'))) && (a = _match(this.operand))) {
                            operation = new CartoCSS.Tree.Operation(op, [operation || m, a], memo);
                        }
                        return operation || m;
                    }
                },
                addition: function () {
                    var m, a, op, operation, plus = '+';
                    if (m = _match(this.multiplication)) {
                        while ((op = _match(/^[-+]\s+/) || (input.charAt(i - 1) != ' ' && (_match(plus) || _match('-')))) &&
                        (a = _match(this.multiplication))) {
                            operation = new CartoCSS.Tree.Operation(op, [operation || m, a], memo);
                        }
                        return operation || m;
                    }
                },

                // An operand is anything that can be part of an operation,
                // such as a Color, or a Variable
                operand: function () {
                    return _match(this.sub) || _match(this.entity);
                },

                // Expressions either represent mathematical operations,
                // or white-space delimited Entities.  @var * 2
                expression: function () {
                    var e, entities = [];

                    while (e = _match(this.addition) || _match(this.entity)) {
                        entities.push(e);
                    }

                    if (entities.length > 0) {
                        return new CartoCSS.Tree.Expression(entities);
                    }
                },
                property: function () {
                    var name = _match(/^(([a-z][-a-z_0-9]*\/)?\*?-?[-a-z_0-9]+)\s*:/);
                    if (name) {return name[1];}
                }
            }
        };
        return parser;
    }


    /**
     * @function CartoCSS.prototype.parse
     * @description 利用CartoCSS解析器里面的parse方法，将CartoCSS样式表字符串转化为CartoCSS规则集。
     * @returns {Object} CartoCSS规则集。
     */
    parse(str) {
        var parser = this.parser;
        var ruleSet = this.ruleSet = parser.parse(str);
        return ruleSet;
    }


    /**
     * @function CartoCSS.prototype.toShaders
     * @description 将CartoCSS规则集转化为着色器。
     * @returns {Array} CartoCSS着色器集。
     */
    toShaders() {
        if (this.ruleSet) {
            var ruleset = this.ruleSet;
            if (ruleset) {
                var defs = ruleset.toList(this.env);
                defs.reverse();

                var shaders = {};
                var keys = [];
                this._toShaders(shaders,keys,defs);

                var ordered_shaders = [];

                var done = {};
                for (var i = 0, len0 = defs.length; i < len0; ++i) {
                    var def = defs[i];
                    var k = def.attachment;
                    var shader = shaders[keys[i]];
                    var shaderArray = [];
                    if (!done[k]) {
                        var j = 0;
                        for (var prop in shader) {
                            if (prop !== 'zoom' && prop !== 'frames' && prop !== "attachment" && prop != "elements") {
                                //对layer-index作特殊处理以实现图层的控制
                                if (prop === "layer-index") {
                                    /*var getLayerIndex = Function("attributes", "zoom", "var _value = null;" + shader[prop].join('\n') + "; return _value; ");*/
                                    var getLayerIndex = function (attributes, zoom) {//NOSONAR
                                        var _value = null;
                                        shader[prop].join('\n');
                                        return _value;
                                    };
                                    var layerIndex = getLayerIndex();
                                    Object.defineProperty(shaderArray, "layerIndex", {
                                        configurable: true,
                                        enumerable: false,
                                        value: layerIndex
                                    });
                                } else {
                                    shaderArray[j++] = function (ops, shaderArray) {//NOSONAR
                                        if (!Array.isArray(ops)) {
                                            return ops;
                                        }
                                        var body = ops.join('\n');
                                        var myKeyword = 'attributes["FEATUREID"]&&attributes["FEATUREID"]';
                                        var index = body.indexOf(myKeyword);
                                        if (index >= 0) {
                                            //对featureID作一些特殊处理，以将featureID提取出来
                                            if (!shaderArray.featureFilter) {
                                                var featureFilterStart = index + myKeyword.length;
                                                var featureFilterEnd = body.indexOf(")", featureFilterStart + 1);
                                                var featureFilterStr = "featureId&&(featureId" + body.substring(featureFilterStart, featureFilterEnd) + ")";
                                                /*var featureFilter = Function("featureId", "if(" + featureFilterStr + "){return true;}return false;");*/
                                                var featureFilter = function (featureId) {
                                                    if (featureFilterStr) {
                                                        return true;
                                                    }
                                                    return false;
                                                }
                                                Object.defineProperty(shaderArray, "featureFilter", {
                                                    configurable: true,
                                                    enumerable: false,
                                                    value: featureFilter
                                                });
                                            }
                                            return {
                                                "property": prop,
                                                "getValue": Function("attributes", "zoom", "seftFilter", "var _value = null; var isExcute=typeof seftFilter=='function'?sefgFilter():seftFilter;if(isExcute){" + body + ";} return _value; ")//NOSONAR
                                            };
                                        } else {
                                            return {
                                                "property": prop,
                                                "getValue": Function("attributes", "zoom", "var _value = null;" + body + "; return _value; ")//NOSONAR
                                            };
                                        }
                                    }(shader[prop], shaderArray);
                                }
                            }
                        }
                        Object.defineProperty(shaderArray, "attachment", {
                            configurable: true,
                            enumerable: false,
                            value: k
                        });
                        Object.defineProperty(shaderArray, "elements", {
                            configurable: true,
                            enumerable: false,
                            value: def.elements
                        });
                        ordered_shaders.push(shaderArray);
                        done[keys[i]] = true;
                    }
                    Object.defineProperty(shaderArray, "zoom", {
                        configurable: true,
                        enumerable: false,
                        value: def.zoom
                    });
                    //shader.frames.push(def.frame_offset);
                }
                return ordered_shaders;
            }
        }
        return null;
    }

    _toShaders(shaders, keys,defs) {
        for (let i = 0, len0 = defs.length; i < len0; ++i) {
            let def = defs[i];
            let element_str = [];
            for (let j = 0, len1 = def.elements.length; j < len1; j++) {
                element_str.push(def.elements[j]);
            }
            let filters = def.filters.filters;
            let filterStr = [];
            for (let attr in filters) {
                filterStr.push(filters[attr].id);
            }
            let key = element_str.join("/") + "::" + def.attachment + "_" + filterStr.join("_");
            keys.push(key);
            let shader = shaders[key] = (shaders[key] || {});
            //shader.frames = [];
            shader.zoom = CartoCSS.Tree.Zoom.all;
            let props = def.toJS(this.env);
            for (let v in props) {
                (shader[v] = (shader[v] || [])).push(props[v].join('\n'))
            }
        }
    }
    /**
     * @function CartoCSS.prototype.getShaders
     * @description 获取CartoCSS着色器。
     * @returns {Array} 着色器集。
     * @example
     *   //shaders的结构大概如下：
     *   var shaders=[
     *   {
     *       attachment:"one",
     *       elements:[],
     *       zoom:23,
     *       length:2,
     *       0:{property:"line-color",value:function(attribute,zoom){var _value=null;if(zoom){_value="#123456"}return _vlaue;}},
     *       1:{preoperty:"line-width",value:function(attribute,zoom){var _value=null;if(zoom){_value=3}return _vlaue;}}
     *   },
     *   {
     *       attachment:"two",
     *       elements:[],
     *       zoom:23,
     *       length:2,
     *       0:{property:"polygon-color",value:function(attribute,zoom){var _value=null;if(zoom){_value="#123456"}return _vlaue;}},
     *       1:{property:"line-width",value:function(attribute,zoom){var _value=null;if(zoom){_value=3}return _vlaue;}}
     *   }
     *   ];
     */
    getShaders() {
        return this.shaders;
    }

    /**
     * @function CartoCSS.prototype.destroy
     * @description CartoCSS解析对象的析构函数，用于销毁CartoCSS解析对象。
     */
    destroy() {
        this.cartoStr = null;
        this.env = null;
        this.ruleSet = null;
        this.parser = null;
        this.shaders = null;
    }

}

var _mapnik_reference_latest = {
    "version": "2.1.1",
    "style": {
        "filter-mode": {
            "type": [
                "all",
                "first"
            ],
            "doc": "Control the processing behavior of Rule filters within a Style. If 'all' is used then all Rules are processed sequentially independent of whether any previous filters matched. If 'first' is used then it means processing ends after the first match (a positive filter evaluation) and no further Rules in the Style are processed ('first' is usually the default for CSS implementations on top of Mapnik to simplify translation from CSS to Mapnik XML)",
            "default-value": "all",
            "default-meaning": "All Rules in a Style are processed whether they have filters or not and whether or not the filter conditions evaluate to true."
        },
        "image-filters": {
            "css": "image-filters",
            "default-value": "none",
            "default-meaning": "no filters",
            "type": "functions",
            "functions": [
                ["agg-stack-blur", 2],
                ["emboss", 0],
                ["blur", 0],
                ["gray", 0],
                ["sobel", 0],
                ["edge-detect", 0],
                ["x-gradient", 0],
                ["y-gradient", 0],
                ["invert", 0],
                ["sharpen", 0]
            ],
            "doc": "A list of image filters."
        },
        "comp-op": {
            "css": "comp-op",
            "default-value": "src-over",
            "default-meaning": "add the current layer on top of other layers",
            "doc": "Composite operation. This defines how this layer should behave relative to layers atop or below it.",
            "type": ["clear",
                "src",
                "dst",
                "src-over",
                "dst-over",
                "src-in",
                "dst-in",
                "src-out",
                "dst-out",
                "src-atop",
                "dst-atop",
                "xor",
                "plus",
                "minus",
                "multiply",
                "screen",
                "overlay",
                "darken",
                "lighten",
                "color-dodge",
                "color-burn",
                "hard-light",
                "soft-light",
                "difference",
                "exclusion",
                "contrast",
                "invert",
                "invert-rgb",
                "grain-merge",
                "grain-extract",
                "hue",
                "saturation",
                "color",
                "value"
            ]
        },
        "opacity": {
            "css": "opacity",
            "type": "float",
            "doc": "An alpha value for the style (which means an alpha applied to all features in separate buffer and then composited back to main buffer)",
            "default-value": 1,
            "default-meaning": "no separate buffer will be used and no alpha will be applied to the style after rendering"
        }
    },
    "layer": {
        "name": {
            "default-value": "",
            "type": "string",
            "required": true,
            "default-meaning": "No layer name has been provided",
            "doc": "The name of a layer. Can be anything you wish and is not strictly validated, but ideally unique  in the map"
        },
        "srs": {
            "default-value": "",
            "type": "string",
            "default-meaning": "No srs value is provided and the value will be inherited from the Map's srs",
            "doc": "The spatial reference system definition for the layer, aka the projection. Can either be a proj4 literal string like '+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs' or, if the proper proj4 epsg/nad/etc identifier files are installed, a string that uses an id like: '+init=epsg:4326'"
        },
        "status": {
            "default-value": true,
            "type": "boolean",
            "default-meaning": "This layer will be marked as active and available for processing",
            "doc": "A property that can be set to false to disable this layer from being processed"
        },
        "minzoom": {
            "default-value": "0",
            "type": "float",
            "default-meaning": "The layer will be visible at the minimum possible scale",
            "doc": "The minimum scale denominator that this layer will be visible at. A layer's visibility is determined by whether its status is true and if the Map scale >= minzoom - 1e-6 and scale < maxzoom + 1e-6"
        },
        "maxzoom": {
            "default-value": "1.79769e+308",
            "type": "float",
            "default-meaning": "The layer will be visible at the maximum possible scale",
            "doc": "The maximum scale denominator that this layer will be visible at. The default is the numeric limit of the C++ double type, which may vary slightly by system, but is likely a massive number like 1.79769e+308 and ensures that this layer will always be visible unless the value is reduced. A layer's visibility is determined by whether its status is true and if the Map scale >= minzoom - 1e-6 and scale < maxzoom + 1e-6"
        },
        "queryable": {
            "default-value": false,
            "type": "boolean",
            "default-meaning": "The layer will not be available for the direct querying of data values",
            "doc": "This property was added for GetFeatureInfo/WMS compatibility and is rarely used. It is off by default meaning that in a WMS context the layer will not be able to be queried unless the property is explicitly set to true"
        },
        "clear-label-cache": {
            "default-value": false,
            "type": "boolean",
            "default-meaning": "The renderer's collision detector cache (used for avoiding duplicate labels and overlapping markers) will not be cleared immediately before processing this layer",
            "doc": "This property, by default off, can be enabled to allow a user to clear the collision detector cache before a given layer is processed. This may be desirable to ensure that a given layers data shows up on the map even if it normally would not because of collisions with previously rendered labels or markers"
        },
        "group-by": {
            "default-value": "",
            "type": "string",
            "default-meaning": "No special layer grouping will be used during rendering",
            "doc": "https://github.com/mapnik/mapnik/wiki/Grouped-rendering"
        },
        "buffer-size": {
            "default-value": "0",
            "type": "float",
            "default-meaning": "No buffer will be used",
            "doc": "Extra tolerance around the Layer extent (in pixels) used to when querying and (potentially) clipping the layer data during rendering"
        },
        "maximum-extent": {
            "default-value": "none",
            "type": "bbox",
            "default-meaning": "No clipping extent will be used",
            "doc": "An extent to be used to limit the bounds used to query this specific layer data during rendering. Should be minx, miny, maxx, maxy in the coordinates of the Layer."
        }
    },
    "symbolizers": {
        "*": {
            "image-filters": {
                "css": "image-filters",
                "default-value": "none",
                "default-meaning": "no filters",
                "type": "functions",
                "functions": [
                    ["agg-stack-blur", 2],
                    ["emboss", 0],
                    ["blur", 0],
                    ["gray", 0],
                    ["sobel", 0],
                    ["edge-detect", 0],
                    ["x-gradient", 0],
                    ["y-gradient", 0],
                    ["invert", 0],
                    ["sharpen", 0]
                ],
                "doc": "A list of image filters."
            },
            "comp-op": {
                "css": "comp-op",
                "default-value": "src-over",
                "default-meaning": "add the current layer on top of other layers",
                "doc": "Composite operation. This defines how this layer should behave relative to layers atop or below it.",
                "type": ["clear",
                    "src",
                    "dst",
                    "src-over",
                    "dst-over",
                    "src-in",
                    "dst-in",
                    "src-out",
                    "dst-out",
                    "src-atop",
                    "dst-atop",
                    "xor",
                    "plus",
                    "minus",
                    "multiply",
                    "screen",
                    "overlay",
                    "darken",
                    "lighten",
                    "color-dodge",
                    "color-burn",
                    "hard-light",
                    "soft-light",
                    "difference",
                    "exclusion",
                    "contrast",
                    "invert",
                    "invert-rgb",
                    "grain-merge",
                    "grain-extract",
                    "hue",
                    "saturation",
                    "color",
                    "value"
                ]
            },
            "opacity": {
                "css": "opacity",
                "type": "float",
                "doc": "An alpha value for the style (which means an alpha applied to all features in separate buffer and then composited back to main buffer)",
                "default-value": 1,
                "default-meaning": "no separate buffer will be used and no alpha will be applied to the style after rendering"
            }
        },
        "map": {
            "background-color": {
                "css": "background-color",
                "default-value": "none",
                "default-meaning": "transparent",
                "type": "color",
                "doc": "Map Background color"
            },
            "background-image": {
                "css": "background-image",
                "type": "uri",
                "default-value": "",
                "default-meaning": "transparent",
                "doc": "An image that is repeated below all features on a map as a background.",
                "description": "Map Background image"
            },
            "srs": {
                "css": "srs",
                "type": "string",
                "default-value": "+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs",
                "default-meaning": "The proj4 literal of EPSG:4326 is assumed to be the Map's spatial reference and all data from layers within this map will be plotted using this coordinate system. If any layers do not declare an srs value then they will be assumed to be in the same srs as the Map and not transformations will be needed to plot them in the Map's coordinate space",
                "doc": "Map spatial reference (proj4 string)"
            },
            "buffer-size": {
                "css": "buffer-size",
                "default-value": "0",
                "type": "float",
                "default-meaning": "No buffer will be used",
                "doc": "Extra tolerance around the map (in pixels) used to ensure labels crossing tile boundaries are equally rendered in each tile (e.g. cut in each tile). Not intended to be used in combination with \"avoid-edges\"."
            },
            "maximum-extent": {
                "css": "",
                "default-value": "none",
                "type": "bbox",
                "default-meaning": "No clipping extent will be used",
                "doc": "An extent to be used to limit the bounds used to query all layers during rendering. Should be minx, miny, maxx, maxy in the coordinates of the Map."
            },
            "base": {
                "css": "base",
                "default-value": "",
                "default-meaning": "This base path defaults to an empty string meaning that any relative paths to files referenced in styles or layers will be interpreted relative to the application process.",
                "type": "string",
                "doc": "Any relative paths used to reference files will be understood as relative to this directory path if the map is loaded from an in memory object rather than from the filesystem. If the map is loaded from the filesystem and this option is not provided it will be set to the directory of the stylesheet."
            },
            "paths-from-xml": {
                "css": "",
                "default-value": true,
                "default-meaning": "Paths read from XML will be interpreted from the location of the XML",
                "type": "boolean",
                "doc": "value to control whether paths in the XML will be interpreted from the location of the XML or from the working directory of the program that calls load_map()"
            },
            "minimum-version": {
                "css": "",
                "default-value": "none",
                "default-meaning": "Mapnik version will not be detected and no error will be thrown about compatibility",
                "type": "string",
                "doc": "The minumum Mapnik version (e.g. 0.7.2) needed to use certain functionality in the stylesheet"
            },
            "font-directory": {
                "css": "font-directory",
                "type": "uri",
                "default-value": "none",
                "default-meaning": "No map-specific fonts will be registered",
                "doc": "Path to a directory which holds fonts which should be registered when the Map is loaded (in addition to any fonts that may be automatically registered)."
            }
        },
        "polygon": {
            "fill": {
                "css": "polygon-fill",
                "type": "color",
                "default-value": "rgba(128,128,128,1)",
                "default-meaning": "gray and fully opaque (alpha = 1), same as rgb(128,128,128)",
                "doc": "Fill color to assign to a polygon"
            },
            "fill-opacity": {
                "css": "polygon-opacity",
                "type": "float",
                "doc": "The opacity of the polygon",
                "default-value": 1,
                "default-meaning": "opaque"
            },
            "gamma": {
                "css": "polygon-gamma",
                "type": "float",
                "default-value": 1,
                "default-meaning": "fully antialiased",
                "range": "0-1",
                "doc": "Level of antialiasing of polygon edges"
            },
            "gamma-method": {
                "css": "polygon-gamma-method",
                "type": [
                    "power",
                    "linear",
                    "none",
                    "threshold",
                    "multiply"
                ],
                "default-value": "power",
                "default-meaning": "pow(x,gamma) is used to calculate pixel gamma, which produces slightly smoother line and polygon antialiasing than the 'linear' method, while other methods are usually only used to disable AA",
                "doc": "An Antigrain Geometry specific rendering hint to control the quality of antialiasing. Under the hood in Mapnik this method is used in combination with the 'gamma' value (which defaults to 1). The methods are in the AGG source at https://github.com/mapnik/mapnik/blob/master/deps/agg/include/agg_gamma_functions.h"
            },
            "clip": {
                "css": "polygon-clip",
                "type": "boolean",
                "default-value": true,
                "default-meaning": "geometry will be clipped to map bounds before rendering",
                "doc": "geometries are clipped to map bounds by default for best rendering performance. In some cases users may wish to disable this to avoid rendering artifacts."
            },
            "smooth": {
                "css": "polygon-smooth",
                "type": "float",
                "default-value": 0,
                "default-meaning": "no smoothing",
                "range": "0-1",
                "doc": "Smooths out geometry angles. 0 is no smoothing, 1 is fully smoothed. Values greater than 1 will produce wild, looping geometries."
            },
            "geometry-transform": {
                "css": "polygon-geometry-transform",
                "type": "functions",
                "default-value": "none",
                "default-meaning": "geometry will not be transformed",
                "doc": "Allows transformation functions to be applied to the geometry.",
                "functions": [
                    ["matrix", 6],
                    ["translate", 2],
                    ["scale", 2],
                    ["rotate", 3],
                    ["skewX", 1],
                    ["skewY", 1]
                ]
            },
            "comp-op": {
                "css": "polygon-comp-op",
                "default-value": "src-over",
                "default-meaning": "add the current symbolizer on top of other symbolizer",
                "doc": "Composite operation. This defines how this symbolizer should behave relative to symbolizers atop or below it.",
                "type": ["clear",
                    "src",
                    "dst",
                    "src-over",
                    "dst-over",
                    "src-in",
                    "dst-in",
                    "src-out",
                    "dst-out",
                    "src-atop",
                    "dst-atop",
                    "xor",
                    "plus",
                    "minus",
                    "multiply",
                    "screen",
                    "overlay",
                    "darken",
                    "lighten",
                    "color-dodge",
                    "color-burn",
                    "hard-light",
                    "soft-light",
                    "difference",
                    "exclusion",
                    "contrast",
                    "invert",
                    "invert-rgb",
                    "grain-merge",
                    "grain-extract",
                    "hue",
                    "saturation",
                    "color",
                    "value"
                ]
            }
        },
        "line": {
            "stroke": {
                "css": "line-color",
                "default-value": "rgba(0,0,0,1)",
                "type": "color",
                "default-meaning": "black and fully opaque (alpha = 1), same as rgb(0,0,0)",
                "doc": "The color of a drawn line"
            },
            "stroke-width": {
                "css": "line-width",
                "default-value": 1,
                "type": "float",
                "doc": "The width of a line in pixels"
            },
            "stroke-opacity": {
                "css": "line-opacity",
                "default-value": 1,
                "type": "float",
                "default-meaning": "opaque",
                "doc": "The opacity of a line"
            },
            "stroke-linejoin": {
                "css": "line-join",
                "default-value": "miter",
                "type": [
                    "miter",
                    "round",
                    "bevel"
                ],
                "doc": "The behavior of lines when joining"
            },
            "stroke-linecap": {
                "css": "line-cap",
                "default-value": "butt",
                "type": [
                    "butt",
                    "round",
                    "square"
                ],
                "doc": "The display of line endings"
            },
            "stroke-gamma": {
                "css": "line-gamma",
                "type": "float",
                "default-value": 1,
                "default-meaning": "fully antialiased",
                "range": "0-1",
                "doc": "Level of antialiasing of stroke line"
            },
            "stroke-gamma-method": {
                "css": "line-gamma-method",
                "type": [
                    "power",
                    "linear",
                    "none",
                    "threshold",
                    "multiply"
                ],
                "default-value": "power",
                "default-meaning": "pow(x,gamma) is used to calculate pixel gamma, which produces slightly smoother line and polygon antialiasing than the 'linear' method, while other methods are usually only used to disable AA",
                "doc": "An Antigrain Geometry specific rendering hint to control the quality of antialiasing. Under the hood in Mapnik this method is used in combination with the 'gamma' value (which defaults to 1). The methods are in the AGG source at https://github.com/mapnik/mapnik/blob/master/deps/agg/include/agg_gamma_functions.h"
            },
            "stroke-dasharray": {
                "css": "line-dasharray",
                "type": "numbers",
                "doc": "A pair of length values [a,b], where (a) is the dash length and (b) is the gap length respectively. More than two values are supported for more complex patterns.",
                "default-value": "none",
                "default-meaning": "solid line"
            },
            "stroke-dashoffset": {
                "css": "line-dash-offset",
                "type": "numbers",
                "doc": "valid parameter but not currently used in renderers (only exists for experimental svg support in Mapnik which is not yet enabled)",
                "default-value": "none",
                "default-meaning": "solid line"
            },
            "stroke-miterlimit": {
                "css": "line-miterlimit",
                "type": "float",
                "doc": "The limit on the ratio of the miter length to the stroke-width. Used to automatically convert miter joins to bevel joins for sharp angles to avoid the miter extending beyond the thickness of the stroking path. Normally will not need to be set, but a larger value can sometimes help avoid jaggy artifacts.",
                "default-value": 4.0,
                "default-meaning": "Will auto-convert miters to bevel line joins when theta is less than 29 degrees as per the SVG spec: 'miterLength / stroke-width = 1 / sin ( theta / 2 )'"
            },
            "clip": {
                "css": "line-clip",
                "type": "boolean",
                "default-value": true,
                "default-meaning": "geometry will be clipped to map bounds before rendering",
                "doc": "geometries are clipped to map bounds by default for best rendering performance. In some cases users may wish to disable this to avoid rendering artifacts."
            },
            "smooth": {
                "css": "line-smooth",
                "type": "float",
                "default-value": 0,
                "default-meaning": "no smoothing",
                "range": "0-1",
                "doc": "Smooths out geometry angles. 0 is no smoothing, 1 is fully smoothed. Values greater than 1 will produce wild, looping geometries."
            },
            "offset": {
                "css": "line-offset",
                "type": "float",
                "default-value": 0,
                "default-meaning": "no offset",
                "doc": "Offsets a line a number of pixels parallel to its actual path. Postive values move the line left, negative values move it right (relative to the directionality of the line)."
            },
            "rasterizer": {
                "css": "line-rasterizer",
                "type": [
                    "full",
                    "fast"
                ],
                "default-value": "full",
                "doc": "Exposes an alternate AGG rendering method that sacrifices some accuracy for speed."
            },
            "geometry-transform": {
                "css": "line-geometry-transform",
                "type": "functions",
                "default-value": "none",
                "default-meaning": "geometry will not be transformed",
                "doc": "Allows transformation functions to be applied to the geometry.",
                "functions": [
                    ["matrix", 6],
                    ["translate", 2],
                    ["scale", 2],
                    ["rotate", 3],
                    ["skewX", 1],
                    ["skewY", 1]
                ]
            },
            "comp-op": {
                "css": "line-comp-op",
                "default-value": "src-over",
                "default-meaning": "add the current symbolizer on top of other symbolizer",
                "doc": "Composite operation. This defines how this symbolizer should behave relative to symbolizers atop or below it.",
                "type": ["clear",
                    "src",
                    "dst",
                    "src-over",
                    "dst-over",
                    "src-in",
                    "dst-in",
                    "src-out",
                    "dst-out",
                    "src-atop",
                    "dst-atop",
                    "xor",
                    "plus",
                    "minus",
                    "multiply",
                    "screen",
                    "overlay",
                    "darken",
                    "lighten",
                    "color-dodge",
                    "color-burn",
                    "hard-light",
                    "soft-light",
                    "difference",
                    "exclusion",
                    "contrast",
                    "invert",
                    "invert-rgb",
                    "grain-merge",
                    "grain-extract",
                    "hue",
                    "saturation",
                    "color",
                    "value"
                ]
            }
        },
        "markers": {
            "file": {
                "css": "marker-file",
                "doc": "An SVG file that this marker shows at each placement. If no file is given, the marker will show an ellipse.",
                "default-value": "",
                "default-meaning": "An ellipse or circle, if width equals height",
                "type": "uri"
            },
            "opacity": {
                "css": "marker-opacity",
                "doc": "The overall opacity of the marker, if set, overrides both the opacity of both the fill and stroke",
                "default-value": 1,
                "default-meaning": "The stroke-opacity and fill-opacity will be used",
                "type": "float"
            },
            "fill-opacity": {
                "css": "marker-fill-opacity",
                "doc": "The fill opacity of the marker",
                "default-value": 1,
                "default-meaning": "opaque",
                "type": "float"
            },
            "stroke": {
                "css": "marker-line-color",
                "doc": "The color of the stroke around a marker shape.",
                "default-value": "black",
                "type": "color"
            },
            "stroke-width": {
                "css": "marker-line-width",
                "doc": "The width of the stroke around a marker shape, in pixels. This is positioned on the boundary, so high values can cover the area itself.",
                "type": "float"
            },
            "stroke-opacity": {
                "css": "marker-line-opacity",
                "default-value": 1,
                "default-meaning": "opaque",
                "doc": "The opacity of a line",
                "type": "float"
            },
            "placement": {
                "css": "marker-placement",
                "type": [
                    "point",
                    "line",
                    "interior"
                ],
                "default-value": "point",
                "default-meaning": "Place markers at the center point (centroid) of the geometry",
                "doc": "Attempt to place markers on a point, in the center of a polygon, or if markers-placement:line, then multiple times along a line. 'interior' placement can be used to ensure that points placed on polygons are forced to be inside the polygon interior"
            },
            "multi-policy": {
                "css": "marker-multi-policy",
                "type": [
                    "each",
                    "whole",
                    "largest"
                ],
                "default-value": "each",
                "default-meaning": "If a feature contains multiple geometries and the placement type is either point or interior then a marker will be rendered for each",
                "doc": "A special setting to allow the user to control rendering behavior for 'multi-geometries' (when a feature contains multiple geometries). This setting does not apply to markers placed along lines. The 'each' policy is default and means all geometries will get a marker. The 'whole' policy means that the aggregate centroid between all geometries will be used. The 'largest' policy means that only the largest (by bounding box areas) feature will get a rendered marker (this is how text labeling behaves by default)."
            },
            "marker-type": {
                "css": "marker-type",
                "type": [
                    "arrow",
                    "ellipse"
                ],
                "default-value": "ellipse",
                "doc": "The default marker-type. If a SVG file is not given as the marker-file parameter, the renderer provides either an arrow or an ellipse (a circle if height is equal to width)"
            },
            "width": {
                "css": "marker-width",
                "default-value": 10,
                "doc": "The width of the marker, if using one of the default types.",
                "type": "expression"
            },
            "height": {
                "css": "marker-height",
                "default-value": 10,
                "doc": "The height of the marker, if using one of the default types.",
                "type": "expression"
            },
            "fill": {
                "css": "marker-fill",
                "default-value": "blue",
                "doc": "The color of the area of the marker.",
                "type": "color"
            },
            "allow-overlap": {
                "css": "marker-allow-overlap",
                "type": "boolean",
                "default-value": false,
                "doc": "Control whether overlapping markers are shown or hidden.",
                "default-meaning": "Do not allow makers to overlap with each other - overlapping markers will not be shown."
            },
            "ignore-placement": {
                "css": "marker-ignore-placement",
                "type": "boolean",
                "default-value": false,
                "default-meaning": "do not store the bbox of this geometry in the collision detector cache",
                "doc": "value to control whether the placement of the feature will prevent the placement of other features"
            },
            "spacing": {
                "css": "marker-spacing",
                "doc": "Space between repeated labels",
                "default-value": 100,
                "type": "float"
            },
            "max-error": {
                "css": "marker-max-error",
                "type": "float",
                "default-value": 0.2,
                "doc": "The maximum difference between actual marker placement and the marker-spacing parameter. Setting a high value can allow the renderer to try to resolve placement conflicts with other symbolizers."
            },
            "transform": {
                "css": "marker-transform",
                "type": "functions",
                "functions": [
                    ["matrix", 6],
                    ["translate", 2],
                    ["scale", 2],
                    ["rotate", 3],
                    ["skewX", 1],
                    ["skewY", 1]
                ],
                "default-value": "",
                "default-meaning": "No transformation",
                "doc": "SVG transformation definition"
            },
            "clip": {
                "css": "marker-clip",
                "type": "boolean",
                "default-value": true,
                "default-meaning": "geometry will be clipped to map bounds before rendering",
                "doc": "geometries are clipped to map bounds by default for best rendering performance. In some cases users may wish to disable this to avoid rendering artifacts."
            },
            "smooth": {
                "css": "marker-smooth",
                "type": "float",
                "default-value": 0,
                "default-meaning": "no smoothing",
                "range": "0-1",
                "doc": "Smooths out geometry angles. 0 is no smoothing, 1 is fully smoothed. Values greater than 1 will produce wild, looping geometries."
            },
            "geometry-transform": {
                "css": "marker-geometry-transform",
                "type": "functions",
                "default-value": "none",
                "default-meaning": "geometry will not be transformed",
                "doc": "Allows transformation functions to be applied to the geometry.",
                "functions": [
                    ["matrix", 6],
                    ["translate", 2],
                    ["scale", 2],
                    ["rotate", 3],
                    ["skewX", 1],
                    ["skewY", 1]
                ]
            },
            "comp-op": {
                "css": "marker-comp-op",
                "default-value": "src-over",
                "default-meaning": "add the current symbolizer on top of other symbolizer",
                "doc": "Composite operation. This defines how this symbolizer should behave relative to symbolizers atop or below it.",
                "type": ["clear",
                    "src",
                    "dst",
                    "src-over",
                    "dst-over",
                    "src-in",
                    "dst-in",
                    "src-out",
                    "dst-out",
                    "src-atop",
                    "dst-atop",
                    "xor",
                    "plus",
                    "minus",
                    "multiply",
                    "screen",
                    "overlay",
                    "darken",
                    "lighten",
                    "color-dodge",
                    "color-burn",
                    "hard-light",
                    "soft-light",
                    "difference",
                    "exclusion",
                    "contrast",
                    "invert",
                    "invert-rgb",
                    "grain-merge",
                    "grain-extract",
                    "hue",
                    "saturation",
                    "color",
                    "value"
                ]
            }
        },
        "shield": {
            "name": {
                "css": "shield-name",
                "type": "expression",
                "serialization": "content",
                "doc": "Value to use for a shield\"s text label. Data columns are specified using brackets like [column_name]"
            },
            "file": {
                "css": "shield-file",
                "required": true,
                "type": "uri",
                "default-value": "none",
                "doc": "Image file to render behind the shield text"
            },
            "face-name": {
                "css": "shield-face-name",
                "type": "string",
                "validate": "font",
                "doc": "Font name and style to use for the shield text",
                "default-value": "",
                "required": true
            },
            "unlock-image": {
                "css": "shield-unlock-image",
                "type": "boolean",
                "doc": "This parameter should be set to true if you are trying to position text beside rather than on top of the shield image",
                "default-value": false,
                "default-meaning": "text alignment relative to the shield image uses the center of the image as the anchor for text positioning."
            },
            "size": {
                "css": "shield-size",
                "type": "float",
                "doc": "The size of the shield text in pixels"
            },
            "fill": {
                "css": "shield-fill",
                "type": "color",
                "doc": "The color of the shield text"
            },
            "placement": {
                "css": "shield-placement",
                "type": [
                    "point",
                    "line",
                    "vertex",
                    "interior"
                ],
                "default-value": "point",
                "doc": "How this shield should be placed. Point placement attempts to place it on top of points, line places along lines multiple times per feature, vertex places on the vertexes of polygons, and interior attempts to place inside of polygons."
            },
            "avoid-edges": {
                "css": "shield-avoid-edges",
                "doc": "Tell positioning algorithm to avoid labeling near intersection edges.",
                "type": "boolean",
                "default-value": false
            },
            "allow-overlap": {
                "css": "shield-allow-overlap",
                "type": "boolean",
                "default-value": false,
                "doc": "Control whether overlapping shields are shown or hidden.",
                "default-meaning": "Do not allow shields to overlap with other map elements already placed."
            },
            "minimum-distance": {
                "css": "shield-min-distance",
                "type": "float",
                "default-value": 0,
                "doc": "Minimum distance to the next shield symbol, not necessarily the same shield."
            },
            "spacing": {
                "css": "shield-spacing",
                "type": "float",
                "default-value": 0,
                "doc": "The spacing between repeated occurrences of the same shield on a line"
            },
            "minimum-padding": {
                "css": "shield-min-padding",
                "default-value": 0,
                "doc": "Determines the minimum amount of padding that a shield gets relative to other shields",
                "type": "float"
            },
            "wrap-width": {
                "css": "shield-wrap-width",
                "type": "unsigned",
                "default-value": 0,
                "doc": "Length of a chunk of text in characters before wrapping text"
            },
            "wrap-before": {
                "css": "shield-wrap-before",
                "type": "boolean",
                "default-value": false,
                "doc": "Wrap text before wrap-width is reached. If false, wrapped lines will be a bit longer than wrap-width."
            },
            "wrap-character": {
                "css": "shield-wrap-character",
                "type": "string",
                "default-value": " ",
                "doc": "Use this character instead of a space to wrap long names."
            },
            "halo-fill": {
                "css": "shield-halo-fill",
                "type": "color",
                "default-value": "#FFFFFF",
                "default-meaning": "white",
                "doc": "Specifies the color of the halo around the text."
            },
            "halo-radius": {
                "css": "shield-halo-radius",
                "doc": "Specify the radius of the halo in pixels",
                "default-value": 0,
                "default-meaning": "no halo",
                "type": "float"
            },
            "character-spacing": {
                "css": "shield-character-spacing",
                "type": "unsigned",
                "default-value": 0,
                "doc": "Horizontal spacing between characters (in pixels). Currently works for point placement only, not line placement."
            },
            "line-spacing": {
                "css": "shield-line-spacing",
                "doc": "Vertical spacing between lines of multiline labels (in pixels)",
                "type": "unsigned"
            },
            "dx": {
                "css": "shield-text-dx",
                "type": "float",
                "doc": "Displace text within shield by fixed amount, in pixels, +/- along the X axis.  A positive value will shift the text right",
                "default-value": 0
            },
            "dy": {
                "css": "shield-text-dy",
                "type": "float",
                "doc": "Displace text within shield by fixed amount, in pixels, +/- along the Y axis.  A positive value will shift the text down",
                "default-value": 0
            },
            "shield-dx": {
                "css": "shield-dx",
                "type": "float",
                "doc": "Displace shield by fixed amount, in pixels, +/- along the X axis.  A positive value will shift the text right",
                "default-value": 0
            },
            "shield-dy": {
                "css": "shield-dy",
                "type": "float",
                "doc": "Displace shield by fixed amount, in pixels, +/- along the Y axis.  A positive value will shift the text down",
                "default-value": 0
            },
            "opacity": {
                "css": "shield-opacity",
                "type": "float",
                "doc": "(Default 1.0) - opacity of the image used for the shield",
                "default-value": 1
            },
            "text-opacity": {
                "css": "shield-text-opacity",
                "type": "float",
                "doc": "(Default 1.0) - opacity of the text placed on top of the shield",
                "default-value": 1
            },
            "horizontal-alignment": {
                "css": "shield-horizontal-alignment",
                "type": [
                    "left",
                    "middle",
                    "right",
                    "auto"
                ],
                "doc": "The shield's horizontal alignment from its centerpoint",
                "default-value": "auto"
            },
            "vertical-alignment": {
                "css": "shield-vertical-alignment",
                "type": [
                    "top",
                    "middle",
                    "bottom",
                    "auto"
                ],
                "doc": "The shield's vertical alignment from its centerpoint",
                "default-value": "middle"
            },
            "text-transform": {
                "css": "shield-text-transform",
                "type": [
                    "none",
                    "uppercase",
                    "lowercase",
                    "capitalize"
                ],
                "doc": "Transform the case of the characters",
                "default-value": "none"
            },
            "justify-alignment": {
                "css": "shield-justify-alignment",
                "type": [
                    "left",
                    "center",
                    "right",
                    "auto"
                ],
                "doc": "Define how text in a shield's label is justified",
                "default-value": "auto"
            },
            "clip": {
                "css": "shield-clip",
                "type": "boolean",
                "default-value": true,
                "default-meaning": "geometry will be clipped to map bounds before rendering",
                "doc": "geometries are clipped to map bounds by default for best rendering performance. In some cases users may wish to disable this to avoid rendering artifacts."
            },
            "comp-op": {
                "css": "shield-comp-op",
                "default-value": "src-over",
                "default-meaning": "add the current symbolizer on top of other symbolizer",
                "doc": "Composite operation. This defines how this symbolizer should behave relative to symbolizers atop or below it.",
                "type": ["clear",
                    "src",
                    "dst",
                    "src-over",
                    "dst-over",
                    "src-in",
                    "dst-in",
                    "src-out",
                    "dst-out",
                    "src-atop",
                    "dst-atop",
                    "xor",
                    "plus",
                    "minus",
                    "multiply",
                    "screen",
                    "overlay",
                    "darken",
                    "lighten",
                    "color-dodge",
                    "color-burn",
                    "hard-light",
                    "soft-light",
                    "difference",
                    "exclusion",
                    "contrast",
                    "invert",
                    "invert-rgb",
                    "grain-merge",
                    "grain-extract",
                    "hue",
                    "saturation",
                    "color",
                    "value"
                ]
            }
        },
        "line-pattern": {
            "file": {
                "css": "line-pattern-file",
                "type": "uri",
                "default-value": "none",
                "required": true,
                "doc": "An image file to be repeated and warped along a line"
            },
            "clip": {
                "css": "line-pattern-clip",
                "type": "boolean",
                "default-value": true,
                "default-meaning": "geometry will be clipped to map bounds before rendering",
                "doc": "geometries are clipped to map bounds by default for best rendering performance. In some cases users may wish to disable this to avoid rendering artifacts."
            },
            "smooth": {
                "css": "line-pattern-smooth",
                "type": "float",
                "default-value": 0,
                "default-meaning": "no smoothing",
                "range": "0-1",
                "doc": "Smooths out geometry angles. 0 is no smoothing, 1 is fully smoothed. Values greater than 1 will produce wild, looping geometries."
            },
            "geometry-transform": {
                "css": "line-pattern-geometry-transform",
                "type": "functions",
                "default-value": "none",
                "default-meaning": "geometry will not be transformed",
                "doc": "Allows transformation functions to be applied to the geometry.",
                "functions": [
                    ["matrix", 6],
                    ["translate", 2],
                    ["scale", 2],
                    ["rotate", 3],
                    ["skewX", 1],
                    ["skewY", 1]
                ]
            },
            "comp-op": {
                "css": "line-pattern-comp-op",
                "default-value": "src-over",
                "default-meaning": "add the current symbolizer on top of other symbolizer",
                "doc": "Composite operation. This defines how this symbolizer should behave relative to symbolizers atop or below it.",
                "type": ["clear",
                    "src",
                    "dst",
                    "src-over",
                    "dst-over",
                    "src-in",
                    "dst-in",
                    "src-out",
                    "dst-out",
                    "src-atop",
                    "dst-atop",
                    "xor",
                    "plus",
                    "minus",
                    "multiply",
                    "screen",
                    "overlay",
                    "darken",
                    "lighten",
                    "color-dodge",
                    "color-burn",
                    "hard-light",
                    "soft-light",
                    "difference",
                    "exclusion",
                    "contrast",
                    "invert",
                    "invert-rgb",
                    "grain-merge",
                    "grain-extract",
                    "hue",
                    "saturation",
                    "color",
                    "value"
                ]
            }
        },
        "polygon-pattern": {
            "file": {
                "css": "polygon-pattern-file",
                "type": "uri",
                "default-value": "none",
                "required": true,
                "doc": "Image to use as a repeated pattern fill within a polygon"
            },
            "alignment": {
                "css": "polygon-pattern-alignment",
                "type": [
                    "local",
                    "global"
                ],
                "default-value": "local",
                "doc": "Specify whether to align pattern fills to the layer or to the map."
            },
            "gamma": {
                "css": "polygon-pattern-gamma",
                "type": "float",
                "default-value": 1,
                "default-meaning": "fully antialiased",
                "range": "0-1",
                "doc": "Level of antialiasing of polygon pattern edges"
            },
            "opacity": {
                "css": "polygon-pattern-opacity",
                "type": "float",
                "doc": "(Default 1.0) - Apply an opacity level to the image used for the pattern",
                "default-value": 1,
                "default-meaning": "The image is rendered without modifications"
            },
            "clip": {
                "css": "polygon-pattern-clip",
                "type": "boolean",
                "default-value": true,
                "default-meaning": "geometry will be clipped to map bounds before rendering",
                "doc": "geometries are clipped to map bounds by default for best rendering performance. In some cases users may wish to disable this to avoid rendering artifacts."
            },
            "smooth": {
                "css": "polygon-pattern-smooth",
                "type": "float",
                "default-value": 0,
                "default-meaning": "no smoothing",
                "range": "0-1",
                "doc": "Smooths out geometry angles. 0 is no smoothing, 1 is fully smoothed. Values greater than 1 will produce wild, looping geometries."
            },
            "geometry-transform": {
                "css": "polygon-pattern-geometry-transform",
                "type": "functions",
                "default-value": "none",
                "default-meaning": "geometry will not be transformed",
                "doc": "Allows transformation functions to be applied to the geometry.",
                "functions": [
                    ["matrix", 6],
                    ["translate", 2],
                    ["scale", 2],
                    ["rotate", 3],
                    ["skewX", 1],
                    ["skewY", 1]
                ]
            },
            "comp-op": {
                "css": "polygon-pattern-comp-op",
                "default-value": "src-over",
                "default-meaning": "add the current symbolizer on top of other symbolizer",
                "doc": "Composite operation. This defines how this symbolizer should behave relative to symbolizers atop or below it.",
                "type": ["clear",
                    "src",
                    "dst",
                    "src-over",
                    "dst-over",
                    "src-in",
                    "dst-in",
                    "src-out",
                    "dst-out",
                    "src-atop",
                    "dst-atop",
                    "xor",
                    "plus",
                    "minus",
                    "multiply",
                    "screen",
                    "overlay",
                    "darken",
                    "lighten",
                    "color-dodge",
                    "color-burn",
                    "hard-light",
                    "soft-light",
                    "difference",
                    "exclusion",
                    "contrast",
                    "invert",
                    "invert-rgb",
                    "grain-merge",
                    "grain-extract",
                    "hue",
                    "saturation",
                    "color",
                    "value"
                ]
            }
        },
        "raster": {
            "opacity": {
                "css": "raster-opacity",
                "default-value": 1,
                "default-meaning": "opaque",
                "type": "float",
                "doc": "The opacity of the raster symbolizer on top of other symbolizers."
            },
            "filter-factor": {
                "css": "raster-filter-factor",
                "default-value": -1,
                "default-meaning": "Allow the datasource to choose appropriate downscaling.",
                "type": "float",
                "doc": "This is used by the Raster or Gdal datasources to pre-downscale images using overviews. Higher numbers can sometimes cause much better scaled image output, at the cost of speed."
            },
            "scaling": {
                "css": "raster-scaling",
                "type": [
                    "near",
                    "fast",
                    "bilinear",
                    "bilinear8",
                    "bicubic",
                    "spline16",
                    "spline36",
                    "hanning",
                    "hamming",
                    "hermite",
                    "kaiser",
                    "quadric",
                    "catrom",
                    "gaussian",
                    "bessel",
                    "mitchell",
                    "sinc",
                    "lanczos",
                    "blackman"
                ],
                "default-value": "near",
                "doc": "The scaling algorithm used to making different resolution versions of this raster layer. Bilinear is a good compromise between speed and accuracy, while lanczos gives the highest quality."
            },
            "mesh-size": {
                "css": "raster-mesh-size",
                "default-value": 16,
                "default-meaning": "Reprojection mesh will be 1/16 of the resolution of the source image",
                "type": "unsigned",
                "doc": "A reduced resolution mesh is used for raster reprojection, and the total image size is divided by the mesh-size to determine the quality of that mesh. Values for mesh-size larger than the default will result in faster reprojection but might lead to distortion."
            },
            "comp-op": {
                "css": "raster-comp-op",
                "default-value": "src-over",
                "default-meaning": "add the current symbolizer on top of other symbolizer",
                "doc": "Composite operation. This defines how this symbolizer should behave relative to symbolizers atop or below it.",
                "type": ["clear",
                    "src",
                    "dst",
                    "src-over",
                    "dst-over",
                    "src-in",
                    "dst-in",
                    "src-out",
                    "dst-out",
                    "src-atop",
                    "dst-atop",
                    "xor",
                    "plus",
                    "minus",
                    "multiply",
                    "screen",
                    "overlay",
                    "darken",
                    "lighten",
                    "color-dodge",
                    "color-burn",
                    "hard-light",
                    "soft-light",
                    "difference",
                    "exclusion",
                    "contrast",
                    "invert",
                    "invert-rgb",
                    "grain-merge",
                    "grain-extract",
                    "hue",
                    "saturation",
                    "color",
                    "value"
                ]
            }
        },
        "point": {
            "file": {
                "css": "point-file",
                "type": "uri",
                "required": false,
                "default-value": "none",
                "doc": "Image file to represent a point"
            },
            "allow-overlap": {
                "css": "point-allow-overlap",
                "type": "boolean",
                "default-value": false,
                "doc": "Control whether overlapping points are shown or hidden.",
                "default-meaning": "Do not allow points to overlap with each other - overlapping markers will not be shown."
            },
            "ignore-placement": {
                "css": "point-ignore-placement",
                "type": "boolean",
                "default-value": false,
                "default-meaning": "do not store the bbox of this geometry in the collision detector cache",
                "doc": "value to control whether the placement of the feature will prevent the placement of other features"
            },
            "opacity": {
                "css": "point-opacity",
                "type": "float",
                "default-value": 1.0,
                "default-meaning": "Fully opaque",
                "doc": "A value from 0 to 1 to control the opacity of the point"
            },
            "placement": {
                "css": "point-placement",
                "type": [
                    "centroid",
                    "interior"
                ],
                "doc": "How this point should be placed. Centroid calculates the geometric center of a polygon, which can be outside of it, while interior always places inside of a polygon.",
                "default-value": "centroid"
            },
            "transform": {
                "css": "point-transform",
                "type": "functions",
                "functions": [
                    ["matrix", 6],
                    ["translate", 2],
                    ["scale", 2],
                    ["rotate", 3],
                    ["skewX", 1],
                    ["skewY", 1]
                ],
                "default-value": "",
                "default-meaning": "No transformation",
                "doc": "SVG transformation definition"
            },
            "comp-op": {
                "css": "point-comp-op",
                "default-value": "src-over",
                "default-meaning": "add the current symbolizer on top of other symbolizer",
                "doc": "Composite operation. This defines how this symbolizer should behave relative to symbolizers atop or below it.",
                "type": ["clear",
                    "src",
                    "dst",
                    "src-over",
                    "dst-over",
                    "src-in",
                    "dst-in",
                    "src-out",
                    "dst-out",
                    "src-atop",
                    "dst-atop",
                    "xor",
                    "plus",
                    "minus",
                    "multiply",
                    "screen",
                    "overlay",
                    "darken",
                    "lighten",
                    "color-dodge",
                    "color-burn",
                    "hard-light",
                    "soft-light",
                    "difference",
                    "exclusion",
                    "contrast",
                    "invert",
                    "invert-rgb",
                    "grain-merge",
                    "grain-extract",
                    "hue",
                    "saturation",
                    "color",
                    "value"
                ]
            }
        },
        "text": {
            "name": {
                "css": "text-name",
                "type": "expression",
                "required": true,
                "default-value": "",
                "serialization": "content",
                "doc": "Value to use for a text label. Data columns are specified using brackets like [column_name]"
            },
            "face-name": {
                "css": "text-face-name",
                "type": "string",
                "validate": "font",
                "doc": "Font name and style to render a label in",
                "required": true
            },
            "size": {
                "css": "text-size",
                "type": "float",
                "default-value": 10,
                "doc": "Text size in pixels"
            },
            "text-ratio": {
                "css": "text-ratio",
                "doc": "Define the amount of text (of the total) present on successive lines when wrapping occurs",
                "default-value": 0,
                "type": "unsigned"
            },
            "wrap-width": {
                "css": "text-wrap-width",
                "doc": "Length of a chunk of text in characters before wrapping text",
                "default-value": 0,
                "type": "unsigned"
            },
            "wrap-before": {
                "css": "text-wrap-before",
                "type": "boolean",
                "default-value": false,
                "doc": "Wrap text before wrap-width is reached. If false, wrapped lines will be a bit longer than wrap-width."
            },
            "wrap-character": {
                "css": "text-wrap-character",
                "type": "string",
                "default-value": " ",
                "doc": "Use this character instead of a space to wrap long text."
            },
            "spacing": {
                "css": "text-spacing",
                "type": "unsigned",
                "doc": "Distance between repeated text labels on a line (aka. label-spacing)"
            },
            "character-spacing": {
                "css": "text-character-spacing",
                "type": "float",
                "default-value": 0,
                "doc": "Horizontal spacing adjustment between characters in pixels"
            },
            "line-spacing": {
                "css": "text-line-spacing",
                "default-value": 0,
                "type": "unsigned",
                "doc": "Vertical spacing adjustment between lines in pixels"
            },
            "label-position-tolerance": {
                "css": "text-label-position-tolerance",
                "default-value": 0,
                "type": "unsigned",
                "doc": "Allows the label to be displaced from its ideal position by a number of pixels (only works with placement:line)"
            },
            "max-char-angle-delta": {
                "css": "text-max-char-angle-delta",
                "type": "float",
                "default-value": "22.5",
                "doc": "The maximum angle change, in degrees, allowed between adjacent characters in a label. This value internally is converted to radians to the default is 22.5*math.pi/180.0. The higher the value the fewer labels will be placed around around sharp corners."
            },
            "fill": {
                "css": "text-fill",
                "doc": "Specifies the color for the text",
                "default-value": "#000000",
                "type": "color"
            },
            "opacity": {
                "css": "text-opacity",
                "doc": "A number from 0 to 1 specifying the opacity for the text",
                "default-value": 1.0,
                "default-meaning": "Fully opaque",
                "type": "float"
            },
            "halo-fill": {
                "css": "text-halo-fill",
                "type": "color",
                "default-value": "#FFFFFF",
                "default-meaning": "white",
                "doc": "Specifies the color of the halo around the text."
            },
            "halo-radius": {
                "css": "text-halo-radius",
                "doc": "Specify the radius of the halo in pixels",
                "default-value": 0,
                "default-meaning": "no halo",
                "type": "float"
            },
            "dx": {
                "css": "text-dx",
                "type": "float",
                "doc": "Displace text by fixed amount, in pixels, +/- along the X axis.  A positive value will shift the text right",
                "default-value": 0
            },
            "dy": {
                "css": "text-dy",
                "type": "float",
                "doc": "Displace text by fixed amount, in pixels, +/- along the Y axis.  A positive value will shift the text down",
                "default-value": 0
            },
            "vertical-alignment": {
                "css": "text-vertical-alignment",
                "type": [
                    "top",
                    "middle",
                    "bottom",
                    "auto"
                ],
                "doc": "Position of label relative to point position.",
                "default-value": "auto",
                "default-meaning": "Default affected by value of dy; \"bottom\" for dy>0, \"top\" for dy<0."
            },
            "avoid-edges": {
                "css": "text-avoid-edges",
                "doc": "Tell positioning algorithm to avoid labeling near intersection edges.",
                "default-value": false,
                "type": "boolean"
            },
            "minimum-distance": {
                "css": "text-min-distance",
                "doc": "Minimum permitted distance to the next text symbolizer.",
                "type": "float"
            },
            "minimum-padding": {
                "css": "text-min-padding",
                "doc": "Determines the minimum amount of padding that a text symbolizer gets relative to other text",
                "type": "float"
            },
            "minimum-path-length": {
                "css": "text-min-path-length",
                "type": "float",
                "default-value": 0,
                "default-meaning": "place labels on all paths",
                "doc": "Place labels only on paths longer than this value."
            },
            "allow-overlap": {
                "css": "text-allow-overlap",
                "type": "boolean",
                "default-value": false,
                "doc": "Control whether overlapping text is shown or hidden.",
                "default-meaning": "Do not allow text to overlap with other text - overlapping markers will not be shown."
            },
            "orientation": {
                "css": "text-orientation",
                "type": "expression",
                "doc": "Rotate the text."
            },
            "placement": {
                "css": "text-placement",
                "type": [
                    "point",
                    "line",
                    "vertex",
                    "interior"
                ],
                "default-value": "point",
                "doc": "Control the style of placement of a point versus the geometry it is attached to."
            },
            "placement-type": {
                "css": "text-placement-type",
                "doc": "Re-position and/or re-size text to avoid overlaps. \"simple\" for basic algorithm (using text-placements string,) \"dummy\" to turn this feature off.",
                "type": [
                    "dummy",
                    "simple"
                ],
                "default-value": "dummy"
            },
            "placements": {
                "css": "text-placements",
                "type": "string",
                "default-value": "",
                "doc": "If \"placement-type\" is set to \"simple\", use this \"POSITIONS,[SIZES]\" string. An example is `text-placements: \"E,NE,SE,W,NW,SW\";` "
            },
            "text-transform": {
                "css": "text-transform",
                "type": [
                    "none",
                    "uppercase",
                    "lowercase",
                    "capitalize"
                ],
                "doc": "Transform the case of the characters",
                "default-value": "none"
            },
            "horizontal-alignment": {
                "css": "text-horizontal-alignment",
                "type": [
                    "left",
                    "middle",
                    "right",
                    "auto"
                ],
                "doc": "The text's horizontal alignment from its centerpoint",
                "default-value": "auto"
            },
            "justify-alignment": {
                "css": "text-align",
                "type": [
                    "left",
                    "right",
                    "center",
                    "auto"
                ],
                "doc": "Define how text is justified",
                "default-value": "auto",
                "default-meaning": "Auto alignment means that text will be centered by default except when using the `placement-type` parameter - in that case either right or left justification will be used automatically depending on where the text could be fit given the `text-placements` directives"
            },
            "clip": {
                "css": "text-clip",
                "type": "boolean",
                "default-value": true,
                "default-meaning": "geometry will be clipped to map bounds before rendering",
                "doc": "geometries are clipped to map bounds by default for best rendering performance. In some cases users may wish to disable this to avoid rendering artifacts."
            },
            "comp-op": {
                "css": "text-comp-op",
                "default-value": "src-over",
                "default-meaning": "add the current symbolizer on top of other symbolizer",
                "doc": "Composite operation. This defines how this symbolizer should behave relative to symbolizers atop or below it.",
                "type": ["clear",
                    "src",
                    "dst",
                    "src-over",
                    "dst-over",
                    "src-in",
                    "dst-in",
                    "src-out",
                    "dst-out",
                    "src-atop",
                    "dst-atop",
                    "xor",
                    "plus",
                    "minus",
                    "multiply",
                    "screen",
                    "overlay",
                    "darken",
                    "lighten",
                    "color-dodge",
                    "color-burn",
                    "hard-light",
                    "soft-light",
                    "difference",
                    "exclusion",
                    "contrast",
                    "invert",
                    "invert-rgb",
                    "grain-merge",
                    "grain-extract",
                    "hue",
                    "saturation",
                    "color",
                    "value"
                ]
            }
        },
        "building": {
            "fill": {
                "css": "building-fill",
                "default-value": "#FFFFFF",
                "doc": "The color of the buildings walls.",
                "type": "color"
            },
            "fill-opacity": {
                "css": "building-fill-opacity",
                "type": "float",
                "doc": "The opacity of the building as a whole, including all walls.",
                "default-value": 1
            },
            "height": {
                "css": "building-height",
                "doc": "The height of the building in pixels.",
                "type": "expression",
                "default-value": "0"
            }
        }
    },
    "colors": {
        "aliceblue": [240, 248, 255],
        "antiquewhite": [250, 235, 215],
        "aqua": [0, 255, 255],
        "aquamarine": [127, 255, 212],
        "azure": [240, 255, 255],
        "beige": [245, 245, 220],
        "bisque": [255, 228, 196],
        "black": [0, 0, 0],
        "blanchedalmond": [255, 235, 205],
        "blue": [0, 0, 255],
        "blueviolet": [138, 43, 226],
        "brown": [165, 42, 42],
        "burlywood": [222, 184, 135],
        "cadetblue": [95, 158, 160],
        "chartreuse": [127, 255, 0],
        "chocolate": [210, 105, 30],
        "coral": [255, 127, 80],
        "cornflowerblue": [100, 149, 237],
        "cornsilk": [255, 248, 220],
        "crimson": [220, 20, 60],
        "cyan": [0, 255, 255],
        "darkblue": [0, 0, 139],
        "darkcyan": [0, 139, 139],
        "darkgoldenrod": [184, 134, 11],
        "darkgray": [169, 169, 169],
        "darkgreen": [0, 100, 0],
        "darkgrey": [169, 169, 169],
        "darkkhaki": [189, 183, 107],
        "darkmagenta": [139, 0, 139],
        "darkolivegreen": [85, 107, 47],
        "darkorange": [255, 140, 0],
        "darkorchid": [153, 50, 204],
        "darkred": [139, 0, 0],
        "darksalmon": [233, 150, 122],
        "darkseagreen": [143, 188, 143],
        "darkslateblue": [72, 61, 139],
        "darkslategrey": [47, 79, 79],
        "darkturquoise": [0, 206, 209],
        "darkviolet": [148, 0, 211],
        "deeppink": [255, 20, 147],
        "deepskyblue": [0, 191, 255],
        "dimgray": [105, 105, 105],
        "dimgrey": [105, 105, 105],
        "dodgerblue": [30, 144, 255],
        "firebrick": [178, 34, 34],
        "floralwhite": [255, 250, 240],
        "forestgreen": [34, 139, 34],
        "fuchsia": [255, 0, 255],
        "gainsboro": [220, 220, 220],
        "ghostwhite": [248, 248, 255],
        "gold": [255, 215, 0],
        "goldenrod": [218, 165, 32],
        "gray": [128, 128, 128],
        "grey": [128, 128, 128],
        "green": [0, 128, 0],
        "greenyellow": [173, 255, 47],
        "honeydew": [240, 255, 240],
        "hotpink": [255, 105, 180],
        "indianred": [205, 92, 92],
        "indigo": [75, 0, 130],
        "ivory": [255, 255, 240],
        "khaki": [240, 230, 140],
        "lavender": [230, 230, 250],
        "lavenderblush": [255, 240, 245],
        "lawngreen": [124, 252, 0],
        "lemonchiffon": [255, 250, 205],
        "lightblue": [173, 216, 230],
        "lightcoral": [240, 128, 128],
        "lightcyan": [224, 255, 255],
        "lightgoldenrodyellow": [250, 250, 210],
        "lightgray": [211, 211, 211],
        "lightgreen": [144, 238, 144],
        "lightgrey": [211, 211, 211],
        "lightpink": [255, 182, 193],
        "lightsalmon": [255, 160, 122],
        "lightseagreen": [32, 178, 170],
        "lightskyblue": [135, 206, 250],
        "lightslategray": [119, 136, 153],
        "lightslategrey": [119, 136, 153],
        "lightsteelblue": [176, 196, 222],
        "lightyellow": [255, 255, 224],
        "lime": [0, 255, 0],
        "limegreen": [50, 205, 50],
        "linen": [250, 240, 230],
        "magenta": [255, 0, 255],
        "maroon": [128, 0, 0],
        "mediumaquamarine": [102, 205, 170],
        "mediumblue": [0, 0, 205],
        "mediumorchid": [186, 85, 211],
        "mediumpurple": [147, 112, 219],
        "mediumseagreen": [60, 179, 113],
        "mediumslateblue": [123, 104, 238],
        "mediumspringgreen": [0, 250, 154],
        "mediumturquoise": [72, 209, 204],
        "mediumvioletred": [199, 21, 133],
        "midnightblue": [25, 25, 112],
        "mintcream": [245, 255, 250],
        "mistyrose": [255, 228, 225],
        "moccasin": [255, 228, 181],
        "navajowhite": [255, 222, 173],
        "navy": [0, 0, 128],
        "oldlace": [253, 245, 230],
        "olive": [128, 128, 0],
        "olivedrab": [107, 142, 35],
        "orange": [255, 165, 0],
        "orangered": [255, 69, 0],
        "orchid": [218, 112, 214],
        "palegoldenrod": [238, 232, 170],
        "palegreen": [152, 251, 152],
        "paleturquoise": [175, 238, 238],
        "palevioletred": [219, 112, 147],
        "papayawhip": [255, 239, 213],
        "peachpuff": [255, 218, 185],
        "peru": [205, 133, 63],
        "pink": [255, 192, 203],
        "plum": [221, 160, 221],
        "powderblue": [176, 224, 230],
        "purple": [128, 0, 128],
        "red": [255, 0, 0],
        "rosybrown": [188, 143, 143],
        "royalblue": [65, 105, 225],
        "saddlebrown": [139, 69, 19],
        "salmon": [250, 128, 114],
        "sandybrown": [244, 164, 96],
        "seagreen": [46, 139, 87],
        "seashell": [255, 245, 238],
        "sienna": [160, 82, 45],
        "silver": [192, 192, 192],
        "skyblue": [135, 206, 235],
        "slateblue": [106, 90, 205],
        "slategray": [112, 128, 144],
        "slategrey": [112, 128, 144],
        "snow": [255, 250, 250],
        "springgreen": [0, 255, 127],
        "steelblue": [70, 130, 180],
        "tan": [210, 180, 140],
        "teal": [0, 128, 128],
        "thistle": [216, 191, 216],
        "tomato": [255, 99, 71],
        "turquoise": [64, 224, 208],
        "violet": [238, 130, 238],
        "wheat": [245, 222, 179],
        "white": [255, 255, 255],
        "whitesmoke": [245, 245, 245],
        "yellow": [255, 255, 0],
        "yellowgreen": [154, 205, 50],
        "transparent": [0, 0, 0, 0]
    },
    "filter": {
        "value": [
            "true",
            "false",
            "null",
            "point",
            "linestring",
            "polygon",
            "collection"
        ]
    }
};

CartoCSS['mapnik_reference'] = {
    version: {
        latest: _mapnik_reference_latest,
        '2.1.1': _mapnik_reference_latest
    }
};

CartoCSS.Tree = {};
CartoCSS.Tree.operate = function(op, a, b) {
    switch (op) {
        case '+': return a + b;
        case '-': return a - b;
        case '*': return a * b;
        case '%': return a % b;
        case '/': return a / b;
    }
};
CartoCSS.Tree.functions = {
    rgb: function (r, g, b) {
        return this.rgba(r, g, b, 1.0);
    },
    rgba: function (r, g, b, a) {
        var me = this;
        var rgb = [r, g, b].map(function (c) {
            return me.number(c);
        });
        a = me.number(a);
        if (rgb.some(isNaN) || isNaN(a)) {return null;}
        return new CartoCSS.Tree.Color(rgb, a);
    },
    // Only require val
    stop: function (val) {
        var color, mode;
        if (arguments.length > 1) {color = arguments[1];}
        if (arguments.length > 2) {mode = arguments[2];}

        return {
            is: 'tag',
            val: val,
            color: color,
            mode: mode,
            toString(env) {
                return '\n\t<stop value="' + val.ev(env) + '"' +
                    (color ? ' color="' + color.ev(env) + '" ' : '') +
                    (mode ? ' mode="' + mode.ev(env) + '" ' : '') +
                    '/>';
            }
        };
    },
    hsl: function (h, s, l) {
        return this.hsla(h, s, l, 1.0);
    },
    hsla: function (h, s, l, a) {
        h = (this.number(h) % 360) / 360;
        s = this.number(s);
        l = this.number(l);
        a = this.number(a);
        if ([h, s, l, a].some(isNaN)) {return null;}

        var m2 = l <= 0.5 ? l * (s + 1) : l + s - l * s,
            m1 = l * 2 - m2;

        return this.rgba(hue(h + 1 / 3) * 255,
            hue(h) * 255,
            hue(h - 1 / 3) * 255,
            a);

        function hue(h) {
            h = h < 0 ? h + 1 : (h > 1 ? h - 1 : h);
            if (h * 6 < 1) {return m1 + (m2 - m1) * h * 6;} else if (h * 2 < 1) {return m2;} else if (h * 3 < 2) {return m1 + (m2 - m1) * (2 / 3 - h) * 6;} else {return m1;}
        }
    },
    hue: function (color) {
        if (!('toHSL' in color)) {return null;}
        return new CartoCSS.Tree.Dimension(Math.round(color.toHSL().h));
    },
    saturation: function (color) {
        if (!('toHSL' in color)) {return null;}
        return new CartoCSS.Tree.Dimension(Math.round(color.toHSL().s * 100), '%');
    },
    lightness: function (color) {
        if (!('toHSL' in color)) {return null;}
        return new CartoCSS.Tree.Dimension(Math.round(color.toHSL().l * 100), '%');
    },
    alpha: function (color) {
        if (!('toHSL' in color)) {return null;}
        return new CartoCSS.Tree.Dimension(color.toHSL().a);
    },
    saturate: function (color, amount) {
        if (!('toHSL' in color)) {return null;}
        var hsl = color.toHSL();

        hsl.s += amount.value / 100;
        hsl.s = this.clamp(hsl.s);
        return this.hsla_simple(hsl);
    },
    desaturate: function (color, amount) {
        if (!('toHSL' in color)) {return null;}
        var hsl = color.toHSL();

        hsl.s -= amount.value / 100;
        hsl.s = this.clamp(hsl.s);
        return this.hsla_simple(hsl);
    },
    lighten: function (color, amount) {
        if (!('toHSL' in color)) {return null;}
        var hsl = color.toHSL();

        hsl.l += amount.value / 100;
        hsl.l = this.clamp(hsl.l);
        return this.hsla_simple(hsl);
    },
    darken: function (color, amount) {
        if (!('toHSL' in color)) {return null;}
        var hsl = color.toHSL();

        hsl.l -= amount.value / 100;
        hsl.l = this.clamp(hsl.l);
        return this.hsla_simple(hsl);
    },
    fadein: function (color, amount) {
        if (!('toHSL' in color)) {return null;}
        var hsl = color.toHSL();

        hsl.a += amount.value / 100;
        hsl.a = this.clamp(hsl.a);
        return this.hsla_simple(hsl);
    },
    fadeout: function (color, amount) {
        if (!('toHSL' in color)) {return null;}
        var hsl = color.toHSL();

        hsl.a -= amount.value / 100;
        hsl.a = this.clamp(hsl.a);
        return this.hsla_simple(hsl);
    },
    spin: function (color, amount) {
        if (!('toHSL' in color)) {return null;}
        var hsl = color.toHSL();
        var hue = (hsl.h + amount.value) % 360;

        hsl.h = hue < 0 ? 360 + hue : hue;

        return this.hsla_simple(hsl);
    },
    replace: function (entity, a, b) {
        if (entity.is === 'field') {
            return entity.toString + '.replace(' + a.toString() + ', ' + b.toString() + ')';
        } else {
            return entity.replace(a, b);
        }
    },
    //
    // Copyright (c) 2006-2009 Hampton Catlin, Nathan Weizenbaum, and Chris Eppstein
    // http://sass-lang.com
    //
    mix: function (color1, color2, weight) {
        var p = weight.value / 100.0;
        var w = p * 2 - 1;
        var a = color1.toHSL().a - color2.toHSL().a;

        var w1 = (((w * a == -1) ? w : (w + a) / (1 + w * a)) + 1) / 2.0;
        var w2 = 1 - w1;

        var rgb = [color1.rgb[0] * w1 + color2.rgb[0] * w2,
            color1.rgb[1] * w1 + color2.rgb[1] * w2,
            color1.rgb[2] * w1 + color2.rgb[2] * w2];

        var alpha = color1.alpha * p + color2.alpha * (1 - p);

        return new CartoCSS.Tree.Color(rgb, alpha);
    },
    greyscale: function (color) {
        return this.desaturate(color, new CartoCSS.Tree.Dimension(100));
    },
    '%': function (quoted /* arg, arg, ...*/) {
        var args = Array.prototype.slice.call(arguments, 1),
            str = quoted.value;

        for (var i = 0; i < args.length; i++) {
            str = str.replace(/%s/, args[i].value)
                .replace(/%[da]/, args[i].toString());
        }
        str = str.replace(/%%/g, '%');
        return new CartoCSS.Tree.Quoted(str);
    },

    hsla_simple: function (h) {
        return this.hsla(h.h, h.s, h.l, h.a);
    },

    number: function (n) {
        if (n instanceof CartoCSS.Tree.Dimension) {
            return parseFloat(n.unit === '%' ? n.value / 100 : n.value);
        } else if (typeof(n) === 'number') {
            return n;
        } else {
            return NaN;
        }
    },

    clamp: function (val) {
        return Math.min(1, Math.max(0, val));
    }
};

CartoCSS.Tree.Call = class Call {

    constructor(name, args, index) {
        this.is = 'call';
        this.name = name;
        this.args = args;
        this.index = index;
    }

    // When evuating a function call,
    // we either find the function in `tree.functions` [1],
    // in which case we call it, passing the  evaluated arguments,
    // or we simply print it out as it appeared originally [2].
    // The *functions.js* file contains the built-in functions.
    // The reason why we evaluate the arguments, is in the case where
    // we try to pass a variable to a function, like: `saturate(@color)`.
    // The function should receive the value, not the variable.
    'ev'(env) {
        var args = this.args.map(function (a) {
            return a.ev(env);
        });

        for (var i = 0; i < args.length; i++) {
            if (args[i].is === 'undefined') {
                return {
                    is: 'undefined',
                    value: 'undefined'
                };
            }
        }

        if (this.name in CartoCSS.Tree.functions) {
            if (CartoCSS.Tree.functions[this.name].length <= args.length) {
                var val = CartoCSS.Tree.functions[this.name].apply(CartoCSS.Tree.functions, args);
                if (val === null) {
                    env.error({
                        message: 'incorrect arguments given to ' + this.name + '()',
                        index: this.index,
                        type: 'runtime',
                        filename: this.filename
                    });
                    return {is: 'undefined', value: 'undefined'};
                }
                return val;
            } else {
                env.error({
                    message: 'incorrect number of arguments for ' + this.name +
                    '(). ' + CartoCSS.Tree.functions[this.name].length + ' expected.',
                    index: this.index,
                    type: 'runtime',
                    filename: this.filename
                });
                return {
                    is: 'undefined',
                    value: 'undefined'
                };
            }
        } else {
            var fn = CartoCSS.Tree.Reference.mapnikFunctions[this.name];
            if (fn === undefined) {
                var functions = toPairs(CartoCSS.Tree.Reference.mapnikFunctions);
                // cheap closest, needs improvement.
                var name = this.name;
                var mean = functions.map(function (f) {
                    return [f[0], CartoCSS.Tree.Reference.editDistance(name, f[0]), f[1]];
                }).sort(function (a, b) {
                    return a[1] - b[1];
                });
                env.error({
                    message: 'unknown function ' + this.name + '(), did you mean ' +
                    mean[0][0] + '(' + mean[0][2] + ')',
                    index: this.index,
                    type: 'runtime',
                    filename: this.filename
                });
                return {
                    is: 'undefined',
                    value: 'undefined'
                };
            }
            if (fn !== args.length &&
                // support variable-arg functions like `colorize-alpha`
                fn !== -1) {
                env.error({
                    message: 'function ' + this.name + '() takes ' +
                    fn + ' arguments and was given ' + args.length,
                    index: this.index,
                    type: 'runtime',
                    filename: this.filename
                });
                return {
                    is: 'undefined',
                    value: 'undefined'
                };
            } else {
                // Save the evaluated versions of arguments
                this.args = args;
                return this;
            }
        }
    }

    toString(env, format) {
        if (this.args.length) {
            return this.name + '(' + this.args.join(',') + ')';
        } else {
            return this.name;
        }
    }
};

CartoCSS.Tree.Color = class Color {

    constructor(rgb, a) {
        this.is = 'color';
        // The end goal here, is to parse the arguments
        // into an integer triplet, such as `128, 255, 0`
        //
        // This facilitates operations and conversions.
        if (Array.isArray(rgb)) {
            this.rgb = rgb.slice(0, 3);
        } else if (rgb.length == 6) {
            this.rgb = rgb.match(/.{2}/g).map(function (c) {
                return parseInt(c, 16);
            });
        } else {
            this.rgb = rgb.split('').map(function (c) {
                return parseInt(c + c, 16);
            });
        }

        if (typeof(a) === 'number') {
            this.alpha = a;
        } else if (rgb.length === 4) {
            this.alpha = rgb[3];
        } else {
            this.alpha = 1;
        }
    }

    'ev'() {
        return this;
    }

    // If we have some transparency, the only way to represent it
    // is via `rgba`. Otherwise, we use the hex representation,
    // which has better compatibility with older browsers.
    // Values are capped between `0` and `255`, rounded and zero-padded.
    toString() {
        /* if (this.alpha < 1.0) {*/
        return 'rgba(' + this.rgb.map(function (c) {
            return Math.round(c);
        }).concat(this.alpha).join(', ') + ')';
        /*} else {
         return '#' + this.rgb.map(function(i) {
         i = Math.round(i);
         i = (i > 255 ? 255 : (i < 0 ? 0 : i)).toString(16);
         return i.length === 1 ? '0' + i : i;
         }).join('');
         }*/
    }

    // Operations have to be done per-channel, if not,
    // channels will spill onto each other. Once we have
    // our result, in the form of an integer triplet,
    // we create a new Color node to hold the result.
    operate(env, op, other) {
        var result = [];

        if (!(other instanceof CartoCSS.Tree.Color)) {
            other = other.toColor();
        }

        for (var c = 0; c < 3; c++) {
            result[c] = CartoCSS.Tree.operate(op, this.rgb[c], other.rgb[c]);
        }
        return new CartoCSS.Tree.Color(result);
    }

    toHSL() {
        var r = this.rgb[0] / 255,
            g = this.rgb[1] / 255,
            b = this.rgb[2] / 255,
            a = this.alpha;

        var max = Math.max(r, g, b), min = Math.min(r, g, b);
        var h, s, l = (max + min) / 2, d = max - min;

        if (max === min) {
            h = s = 0;
        } else {
            s = l > 0.5 ? d / (2 - max - min) : d / (max + min);

            switch (max) {
                case r:
                    h = (g - b) / d + (g < b ? 6 : 0);
                    break;
                case g:
                    h = (b - r) / d + 2;
                    break;
                case b:
                    h = (r - g) / d + 4;
                    break;
                default:
                    break;
            }
            h /= 6;
        }
        return {h: h * 360, s: s, l: l, a: a};
    }
};

CartoCSS.Tree.Comment = class Comment {
    constructor(value, silent) {
        this.value = value;
        this.silent = !!silent;
    }

    toString(env) {
        return '<!--' + this.value + '-->';
    }

    'ev'() {
        return this;
    }
};

CartoCSS.Tree.Definition = class Definition {
    constructor(selector, rules) {
        this.elements = selector.elements;
        //assert.ok(selector.filters instanceof CartoCSS.Tree.Filterset);
        this.rules = rules;
        this.ruleIndex = {};
        for (var i = 0; i < this.rules.length; i++) {
            if ('zoom' in this.rules[i]) {this.rules[i] = this.rules[i].clone();}
            this.rules[i].zoom = selector.zoom;
            this.ruleIndex[this.rules[i].updateID()] = true;
        }
        this.filters = selector.filters;
        this.zoom = selector.zoom;
        this.attachment = selector.attachment || '__default__';
        this.specificity = selector.specificity();
    }

    toString() {
        var str = this.filters.toString();
        for (var i = 0; i < this.rules.length; i++) {
            str += '\n    ' + this.rules[i];
        }
        return str;
    }

    toJS(env) {
        var shaderAttrs = {};

        // merge conditions from filters with zoom condition of the
        // definition
        var zoom = this.zoom;
        //var frame_offset = this.frame_offset;
        var _if = this.filters.toJS(env);
        var filters = [zoom];
        if (_if) {filters.push(_if);}
        //if(frame_offset) filters.push('ctx["frame-offset"] === ' + frame_offset);
        _if = filters.join(" && ");

        function eachRule(rule) {
            if (rule instanceof CartoCSS.Tree.Rule) {
                shaderAttrs[rule.name] = shaderAttrs[rule.name] || [];
                if (_if) {
                    shaderAttrs[rule.name].push(
                        "if(" + _if + "){" + rule.value.toJS(env) + "}"
                    );
                } else {
                    shaderAttrs[rule.name].push(rule.value.toJS(env));
                }
            } else {
                if (rule instanceof CartoCSS.Tree.Ruleset) {
                    var sh = rule.toJS(env);
                    for (var v in sh) {
                        shaderAttrs[v] = shaderAttrs[v] || [];
                        for (var attr in sh[v]) {
                            shaderAttrs[v].push(sh[v][attr]);
                        }
                    }
                }
            }
        }
        for (var id in this.rules) {
            eachRule(this.rules[id]);
        }
        return shaderAttrs;
    }
};

CartoCSS.Tree.Dimension = class Dimension {


    constructor(value, unit, index) {
        this.is = 'float';
        this.physical_units = ['m', 'cm', 'in', 'mm', 'pt', 'pc'];
        this.screen_units = ['px', '%'];
        this.all_units = ['m', 'cm', 'in', 'mm', 'pt', 'pc', 'px', '%'];
        this.densities = {
            m: 0.0254,
            mm: 25.4,
            cm: 2.54,
            pt: 72,
            pc: 6
        };
        this.value = parseFloat(value);
        this.unit = unit || null;
        this.index = index;
    }

    ev(env) {
        if (this.unit && this.all_units.indexOf(this.unit)<0) {
            env.error({
                message: "Invalid unit: '" + this.unit + "'",
                index: this.index
            });
            return {is: 'undefined', value: 'undefined'};
        }

        // normalize units which are not px or %
        if (this.unit && this.physical_units.indexOf(this.unit)>=0) {
            if (!env.ppi) {
                env.error({
                    message: "ppi is not set, so metric units can't be used",
                    index: this.index
                });
                return {is: 'undefined', value: 'undefined'};
            }
            // convert all units to inch
            // convert inch to px using ppi
            this.value = (this.value / this.densities[this.unit]) * env.ppi;
            this.unit = 'px';
        }

        return this;
    }

    toColor() {
        return new CartoCSS.Tree.Color([this.value, this.value, this.value]);
    }

    round() {
        this.value = Math.round(this.value);
        return this;
    }

    toString() {
        return this.value.toString();
    }

    operate(env, op, other) {
        if (this.unit === '%' && other.unit !== '%') {
            env.error({
                message: 'If two operands differ, the first must not be %',
                index: this.index
            });
            return {
                is: 'undefined',
                value: 'undefined'
            };
        }

        if (this.unit !== '%' && other.unit === '%') {
            if (op === '*' || op === '/' || op === '%') {
                env.error({
                    message: 'Percent values can only be added or subtracted from other values',
                    index: this.index
                });
                return {
                    is: 'undefined',
                    value: 'undefined'
                };
            }

            return new CartoCSS.Tree.Dimension(CartoCSS.Tree.operate(op,
                this.value, this.value * other.value * 0.01),
                this.unit);
        }

        //here the operands are either the same (% or undefined or px), or one is undefined and the other is px
        return new CartoCSS.Tree.Dimension(CartoCSS.Tree.operate(op, this.value, other.value),
            this.unit || other.unit);
    }
};

CartoCSS.Tree.Element = class Element {
    constructor(value) {
        this.value = value.trim();
        if (this.value[0] === '#') {
            this.type = 'id';
            this.clean = this.value.replace(/^#/, '');
        }
        if (this.value[0] === '.') {
            this.type = 'class';
            this.clean = this.value.replace(/^\./, '');
        }
        if (this.value.indexOf('*') !== -1) {
            this.type = 'wildcard';
        }
    }

    specificity() {
        return [
            (this.type === 'id') ? 1 : 0, // a
            (this.type === 'class') ? 1 : 0  // b
        ];
    }

    toString() {
        return this.value;
    }
};

CartoCSS.Tree.Expression = class Expression {


    constructor(value) {
        this.is = 'expression';
        this.value = value;
    }

    ev(env) {
        if (this.value.length > 1) {
            return new CartoCSS.Tree.Expression(this.value.map(function (e) {
                return e.ev(env);
            }));
        } else {
            return this.value[0].ev(env);
        }
    }

    toString(env) {
        return this.value.map(function (e) {
            return e.toString(env);
        }).join(' ');
    }
};

CartoCSS.Tree.Field = class Field {

    constructor(content) {
        this.is = 'field';
        this.value = content || '';
    }

    toString() {
        return '["' + this.value.toUpperCase() + '"]';
    }

    'ev'() {
        return this;
    }
};

CartoCSS.Tree.Filter = class Filter {

    constructor(key, op, val, index, filename) {
        this.ops = {
            '<': [' &lt; ', 'numeric'],
            '>': [' &gt; ', 'numeric'],
            '=': [' = ', 'both'],
            '!=': [' != ', 'both'],
            '<=': [' &lt;= ', 'numeric'],
            '>=': [' &gt;= ', 'numeric'],
            '=~': ['.match(', 'string', ')']
        };

        this.key = key;
        this.op = op;
        this.val = val;
        this.index = index;
        this.filename = filename;

        this.id = this.key + this.op + this.val;
    }

    ev(env) {
        this.key = this.key.ev(env);
        this.val = this.val.ev(env);
        return this;
    }

    toString() {
        return '[' + this.id + ']';
    }
};

CartoCSS.Tree.Filterset = class Filterset {
    constructor() {
        this.filters = {};
    }

    toJS(env) {
        function eachFilter(filter) {
            var op = filter.op;
            if (op === "=") {
                op = "==";
            }
            var val = filter.val;
            if (filter._val !== undefined) {
                val = filter._val.toString(true);
            }

            //对scale进行特殊处理，将值转换成数值
            if (filter.key && filter.key.value === 'scale') {
                val = +val;
            } else if (typeof val === 'string' || typeof val === 'object') {
                val = "'" + val + "'";
            }
            var attrs = "attributes";
            return attrs + "&&" + attrs + filter.key + "&&" + attrs + filter.key + " " + op + val;
        }
        var results = [];
        for (var id in this.filters) {
            results.push(eachFilter(this.filters[id]));
        }
        return results.join(' && ');
    }

    toString() {
        var arr = [];
        for (var id in this.filters) {arr.push(this.filters[id].id);}
        return arr.sort().join('\t');
    }

    ev(env) {
        for (var i in this.filters) {
            this.filters[i].ev(env);
        }
        return this;
    }

    clone() {
        var clone = new CartoCSS.Tree.Filterset();
        for (var id in this.filters) {
            clone.filters[id] = this.filters[id];
        }
        return clone;
    }

    cloneWith(other) {
        var additions = [];
        for (var id in other.filters) {
            var status = this.addable(other.filters[id]);
            // status is true, false or null. if it's null we don't fail this
            // clone nor do we add the filter.
            if (status === false) {
                return false;
            }
            if (status === true) {
                // Adding the filter will override another value.
                additions.push(other.filters[id]);
            }
        }

        // Adding the other filters doesn't make this filterset invalid, but it
        // doesn't add anything to it either.
        if (!additions.length) {
            return null;
        }

        // We can successfully add all filters. Now clone the filterset and add the
        // new rules.
        var clone = new CartoCSS.Tree.Filterset();

        // We can add the rules that are already present without going through the
        // add function as a Filterset is always in it's simplest canonical form.
        for (id in this.filters) {
            clone.filters[id] = this.filters[id];
        }

        // Only add new filters that actually change the filter.
        while (id = additions.shift()) {
            clone.add(id);
        }

        return clone;
    }

    addable(filter) {
        var key = filter.key.toString(),
            value = filter.val.toString();

        if (value.match(/^[0-9]+(\.[0-9]*)?_match/)) {value = parseFloat(value);}

        switch (filter.op) {
            case '=':
                // if there is already foo= and we're adding foo=
                if (this.filters[key + '='] !== undefined) {
                    if (this.filters[key + '='].val.toString() != value) {
                        return false;
                    } else {
                        return null;
                    }
                }
                if (this.filters[key + '!=' + value] !== undefined) {return false;}
                if (this.filters[key + '>'] !== undefined && this.filters[key + '>'].val >= value) {return false;}
                if (this.filters[key + '<'] !== undefined && this.filters[key + '<'].val <= value) {return false;}
                if (this.filters[key + '>='] !== undefined && this.filters[key + '>='].val > value) {return false;}
                if (this.filters[key + '<='] !== undefined && this.filters[key + '<='].val < value) {return false;}
                return true;

            case '=~':
                return true;

            case '!=':
                if (this.filters[key + '='] !== undefined) {return (this.filters[key + '='].val === value) ? false : null;}
                if (this.filters[key + '!=' + value] !== undefined) {return null;}
                if (this.filters[key + '>'] !== undefined && this.filters[key + '>'].val >= value) {return null;}
                if (this.filters[key + '<'] !== undefined && this.filters[key + '<'].val <= value) {return null;}
                if (this.filters[key + '>='] !== undefined && this.filters[key + '>='].val > value) {return null;}
                if (this.filters[key + '<='] !== undefined && this.filters[key + '<='].val < value) {return null;}
                return true;

            case '>':
                if (key + '=' in this.filters) {
                    if (this.filters[key + '='].val <= value) {
                        return false;
                    } else {
                        return null;
                    }
                }
                if (this.filters[key + '<'] !== undefined && this.filters[key + '<'].val <= value) {return false;}
                if (this.filters[key + '<='] !== undefined && this.filters[key + '<='].val <= value) {return false;}
                if (this.filters[key + '>'] !== undefined && this.filters[key + '>'].val >= value) {return null;}
                if (this.filters[key + '>='] !== undefined && this.filters[key + '>='].val > value) {return null;}
                return true;

            case '>=':
                if (this.filters[key + '='] !== undefined) {return (this.filters[key + '='].val < value) ? false : null;}
                if (this.filters[key + '<'] !== undefined && this.filters[key + '<'].val <= value) {return false;}
                if (this.filters[key + '<='] !== undefined && this.filters[key + '<='].val < value) {return false;}
                if (this.filters[key + '>'] !== undefined && this.filters[key + '>'].val >= value) {return null;}
                if (this.filters[key + '>='] !== undefined && this.filters[key + '>='].val >= value) {return null;}
                return true;

            case '<':
                if (this.filters[key + '='] !== undefined) {return (this.filters[key + '='].val >= value) ? false : null;}
                if (this.filters[key + '>'] !== undefined && this.filters[key + '>'].val >= value) {return false;}
                if (this.filters[key + '>='] !== undefined && this.filters[key + '>='].val >= value) {return false;}
                if (this.filters[key + '<'] !== undefined && this.filters[key + '<'].val <= value) {return null;}
                if (this.filters[key + '<='] !== undefined && this.filters[key + '<='].val < value) {return null;}
                return true;

            case '<=':
                if (this.filters[key + '='] !== undefined) {return (this.filters[key + '='].val > value) ? false : null;}
                if (this.filters[key + '>'] !== undefined && this.filters[key + '>'].val >= value) {return false;}
                if (this.filters[key + '>='] !== undefined && this.filters[key + '>='].val > value) {return false;}
                if (this.filters[key + '<'] !== undefined && this.filters[key + '<'].val <= value) {return null;}
                if (this.filters[key + '<='] !== undefined && this.filters[key + '<='].val <= value) {return null;}
                return true;

            default:
                break;
        }
    }

    conflict(filter) {
        var key = filter.key.toString(),
            value = filter.val.toString();

        if (!isNaN(parseFloat(value))) {value = parseFloat(value);}

        // if (a=b) && (a=c)
        // if (a=b) && (a!=b)
        // or (a!=b) && (a=b)
        if ((filter.op === '=' && this.filters[key + '='] !== undefined &&
                value != this.filters[key + '='].val.toString()) ||
            (filter.op === '!=' && this.filters[key + '='] !== undefined &&
                value == this.filters[key + '='].val.toString()) ||
            (filter.op === '=' && this.filters[key + '!='] !== undefined &&
                value === this.filters[key + '!='].val.toString())) {
            return filter.toString() + ' added to ' + this.toString() + ' produces an invalid filter';
        }

        return false;
    }

    add(filter, env) {
        var key = filter.key.toString(),
            op = filter.op,
            conflict = this.conflict(filter),
            numval;
        if (conflict) {return conflict;}

        if (op === '=') {
            for (var i in this.filters) {
                if (this.filters[i].key === key) {delete this.filters[i];}
            }
            this.filters[key + '='] = filter;
        } else if (op === '!=') {
            this.filters[key + '!=' + filter.val] = filter;
        } else if (op === '=~') {
            this.filters[key + '=~' + filter.val] = filter;
        } else if (op === '>') {
            // If there are other filters that are also >
            // but are less than this one, they don't matter, so
            // remove them.
            for (var j in this.filters) {
                if (this.filters[j].key === key && this.filters[j].val <= filter.val) {
                    delete this.filters[j];
                }
            }
            this.filters[key + '>'] = filter;
        } else if (op === '>=') {
            for (var k in this.filters) {
                numval = (+this.filters[k].val.toString());
                if (this.filters[k].key === key && numval < filter.val) {
                    delete this.filters[k];
                }
            }
            if (this.filters[key + '!=' + filter.val] !== undefined) {
                delete this.filters[key + '!=' + filter.val];
                filter.op = '>';
                this.filters[key + '>'] = filter;
            } else {
                this.filters[key + '>='] = filter;
            }
        } else if (op === '<') {
            for (var l in this.filters) {
                numval = (+this.filters[l].val.toString());
                if (this.filters[l].key === key && numval >= filter.val) {
                    delete this.filters[l];
                }
            }
            this.filters[key + '<'] = filter;
        } else if (op === '<=') {
            for (var m in this.filters) {
                numval = (+this.filters[m].val.toString());
                if (this.filters[m].key === key && numval > filter.val) {
                    delete this.filters[m];
                }
            }
            if (this.filters[key + '!=' + filter.val] !== undefined) {
                delete this.filters[key + '!=' + filter.val];
                filter.op = '<';
                this.filters[key + '<'] = filter;
            } else {
                this.filters[key + '<='] = filter;
            }
        }
    }
};

CartoCSS.Tree.Fontset = class Fontset {
    constructor(env, fonts) {
        this.fonts = fonts;
        this.name = 'fontset-' + env.effects.length;
    }
};
CartoCSS.Tree.Invalid = class Invalid {

    constructor(chunk, index, message) {
        this.is = 'invalid';
        this.chunk = chunk;
        this.index = index;
        this.type = 'syntax';
        this.message = message || "Invalid code: " + this.chunk;
    }

    ev(env) {
        env.error({
            chunk: this.chunk,
            index: this.index,
            type: 'syntax',
            message: this.message || "Invalid code: " + this.chunk
        });
        return {
            is: 'undefined'
        };
    }
};

CartoCSS.Tree.Keyword = class Keyword {
    ev() {
        return this;
    }

    constructor(value) {
        this.value = value;
        var special = {
            'transparent': 'color',
            'true': 'boolean',
            'false': 'boolean'
        };
        this.is = special[value] ? special[value] : 'keyword';
    }

    toString() {
        return this.value;
    }
};

/*Layer:class Invalid ),*/

CartoCSS.Tree.Literal = class Literal {
    constructor(content) {
        this.value = content || '';
        this.is = 'field';
    }

    toString() {
        return this.value;
    }

    'ev'() {
        return this;
    }
};

CartoCSS.Tree.Operation = class Operation {

    constructor(op, operands, index) {
        this.is = 'operation';
        this.op = op.trim();
        this.operands = operands;
        this.index = index;
    }

    ev(env) {
        var a = this.operands[0].ev(env),
            b = this.operands[1].ev(env),
            temp;

        if (a.is === 'undefined' || b.is === 'undefined') {
            return {
                is: 'undefined',
                value: 'undefined'
            };
        }

        if (a instanceof CartoCSS.Tree.Dimension && b instanceof CartoCSS.Tree.Color) {
            if (this.op === '*' || this.op === '+') {
                temp = b;
                b = a;
                a = temp;
            } else {
                env.error({
                    name: "OperationError",
                    message: "Can't substract or divide a color from a number",
                    index: this.index
                });
            }
        }

        // Only concatenate plain strings, because this is easily
        // pre-processed
        if (a instanceof CartoCSS.Tree.Quoted && b instanceof CartoCSS.Tree.Quoted && this.op !== '+') {
            env.error({
                message: "Can't subtract, divide, or multiply strings.",
                index: this.index,
                type: 'runtime',
                filename: this.filename
            });
            return {
                is: 'undefined',
                value: 'undefined'
            };
        }

        // Fields, literals, dimensions, and quoted strings can be combined.
        if (a instanceof CartoCSS.Tree.Field || b instanceof CartoCSS.Tree.Field ||
            a instanceof CartoCSS.Tree.Literal || b instanceof CartoCSS.Tree.Literal) {
            if (a.is === 'color' || b.is === 'color') {
                env.error({
                    message: "Can't subtract, divide, or multiply colors in expressions.",
                    index: this.index,
                    type: 'runtime',
                    filename: this.filename
                });
                return {
                    is: 'undefined',
                    value: 'undefined'
                };
            } else {
                return new CartoCSS.Tree.Literal(a.ev(env).toString(true) + this.op + b.ev(env).toString(true));
            }
        }

        if (a.operate === undefined) {
            env.error({
                message: 'Cannot do math with type ' + a.is + '.',
                index: this.index,
                type: 'runtime',
                filename: this.filename
            });
            return {
                is: 'undefined',
                value: 'undefined'
            };
        }

        return a.operate(env, this.op, b);
    }
};

CartoCSS.Tree.Quoted = class Quoted {

    constructor(content) {
        this.is = 'string';
        this.value = content || '';
    }

    toString(quotes) {
        var escapedValue = this.value
            .replace(/&/g, '&amp;')
        var xmlvalue = escapedValue
            .replace(/\'/g, '\\\'')
            .replace(/\"/g, '&quot;')
            .replace(/</g, '&lt;')
            .replace(/\>/g, '&gt;');
        return (quotes === true) ? "'" + xmlvalue + "'" : escapedValue;
    }

    ev() {
        return this;
    }

    operate(env, op, other) {
        return new CartoCSS.Tree.Quoted(CartoCSS.Tree.operate(op, this.toString(), other.toString(this.contains_field)));
    }
};

CartoCSS.Tree.Reference = {
    _validateValue: {
        'font': function (env, value) {
            if (env.validation_data && env.validation_data.fonts) {
                return env.validation_data.fonts.indexOf(value) != -1;
            } else {
                return true;
            }
        }
    },
    setData: function (data) {
        this.data = data;
        this.selector_cache = generateSelectorCache(data);
        this.mapnikFunctions = generateMapnikFunctions(data);
        this.required_cache = generateRequiredProperties(data);

        function generateSelectorCache(data) {
            var index = {};
            for (var i in data.symbolizers) {
                for (var j in data.symbolizers[i]) {
                    if (data.symbolizers[i][j].hasOwnProperty('css')) {
                        index[data.symbolizers[i][j].css] = [data.symbolizers[i][j], i, j];
                    }
                }
            }
            return index;
        }

        function generateMapnikFunctions(data) {
            var functions = {};
            for (var i in data.symbolizers) {
                for (var j in data.symbolizers[i]) {
                    if (data.symbolizers[i][j].type === 'functions') {
                        for (var k = 0; k < data.symbolizers[i][j].functions.length; k++) {
                            var fn = data.symbolizers[i][j].functions[k];
                            functions[fn[0]] = fn[1];
                        }
                    }
                }
            }
            return functions;
        }

        function generateRequiredProperties(data) {
            var cache = {};
            for (var symbolizer_name in data.symbolizers) {
                cache[symbolizer_name] = [];
                for (var j in data.symbolizers[symbolizer_name]) {
                    if (data.symbolizers[symbolizer_name][j].required) {
                        cache[symbolizer_name].push(data.symbolizers[symbolizer_name][j].css);
                    }
                }
            }
            return cache;
        }
    },
    setVersion: function (version) {
        if (CartoCSS.mapnik_reference.version.hasOwnProperty(version)) {
            this.setData(CartoCSS.mapnik_reference.version[version]);
            return true;
        }
        return false;
    },
    selectorData: function (selector, i) {
        if (this.selector_cache && this.selector_cache[selector]) {return this.selector_cache[selector][i];}
    },
    validSelector: function (selector) {
        return !!this.selector_cache[selector];
    },
    selectorName: function (selector) {
        return this.selectorData(selector, 2);
    },
    selector: function (selector) {
        return this.selectorData(selector, 0);
    },
    symbolizer: function (selector) {
        return this.selectorData(selector, 1);
    },
    requiredProperties: function (symbolizer_name, rules) {
        var req = this.required_cache[symbolizer_name];
        for (var i in req) {
            if (!(req[i] in rules)) {
                return 'Property ' + req[i] + ' required for defining ' +
                    symbolizer_name + ' styles.';
            }
        }
    },
    isFont: function (selector) {
        return this.selector(selector).validate === 'font';
    },
    editDistance: function (a, b) {
        if (a.length === 0) {return b.length;}
        if (b.length === 0) {return a.length;}
        var matrix = [];
        for (var i = 0; i <= b.length; i++) {
            matrix[i] = [i];
        }
        for (var j = 0; j <= a.length; j++) {
            matrix[0][j] = j;
        }
        for (i = 1; i <= b.length; i++) {
            for (j = 1; j <= a.length; j++) {
                if (b.charAt(i - 1) === a.charAt(j - 1)) {
                    matrix[i][j] = matrix[i - 1][j - 1];
                } else {
                    matrix[i][j] = Math.min(matrix[i - 1][j - 1] + 1, // substitution
                        Math.min(matrix[i][j - 1] + 1, // insertion
                            matrix[i - 1][j] + 1)); // deletion
                }
            }
        }
        return matrix[b.length][a.length];
    },
    validValue: function (env, selector, value) {
        function validateFunctions(value, selector) {
            if (value.value[0].is === 'string') {return true;}
            for (var i in value.value) {
                for (var j in value.value[i].value) {
                    if (value.value[i].value[j].is !== 'call') {return false;}
                    var f = find(this.selector(selector).functions, function (x) {//NOSONAR
                        return x[0] === value.value[i].value[j].name;
                    });
                    if (!(f && f[1] === -1)) {
                        // This filter is unknown or given an incorrect number of arguments
                        if (!f || f[1] !== value.value[i].value[j].args.length) {return false;}
                    }
                }
            }
            return true;
        }

        function validateKeyword(value, selector) {
            if (typeof this.selector(selector).type === 'object') {
                return this.selector(selector).type
                    .indexOf(value.value[0].value) !== -1;
            } else {
                // allow unquoted keywords as strings
                return this.selector(selector).type === 'string';
            }
        }

        var i;
        if (!this.selector(selector)) {
            return false;
        } else if (value.value[0].is === 'keyword') {
            return validateKeyword(value, selector);
        } else if (value.value[0].is === 'undefined') {
            // caught earlier in the chain - ignore here so that
            // error is not overridden
            return true;
        } else if (this.selector(selector).type === 'numbers') {
            for (i in value.value) {
                if (value.value[i].is !== 'float') {
                    return false;
                }
            }
            return true;
        } else if (this.selector(selector).type === 'tags') {
            if (!value.value) {return false;}
            if (!value.value[0].value) {
                return value.value[0].is === 'tag';
            }
            for (i = 0; i < value.value[0].value.length; i++) {
                if (value.value[0].value[i].is !== 'tag') {return false;}
            }
            return true;
        } else if (this.selector(selector).type == 'functions') {
            // For backwards compatibility, you can specify a string for `functions`-compatible
            // values, though they will not be validated.
            return validateFunctions(value, selector);
        } else if (this.selector(selector).type === 'expression') {
            return true;
        } else if (this.selector(selector).type === 'unsigned') {
            if (value.value[0].is === 'float') {
                value.value[0].round();
                return true;
            } else {
                return false;
            }
        } else {
            if (this.selector(selector).validate) {
                var valid = false;
                for (i = 0; i < value.value.length; i++) {
                    if (this.selector(selector).type === value.value[i].is &&
                        this._validateValue[this.selector(selector).validate](env, value.value[i].value)) {
                        return true;
                    }
                }
                return valid;
            } else {
                return this.selector(selector).type === value.value[0].is;
            }
        }
    }
};
CartoCSS.Tree.Reference.setVersion("latest");

CartoCSS.Tree.Rule = class Rule {

    constructor(name, value, index, filename) {
        this.is = 'rule';
        var parts = name.split('/');
        this.name = parts.pop();
        this.instance = parts.length ? parts[0] : '__default__';
        this.value = (value instanceof CartoCSS.Tree.Value) ?
            value : new CartoCSS.Tree.Value([value]);
        this.index = index;
        this.symbolizer = CartoCSS.Tree.Reference.symbolizer(this.name);
        this.filename = filename;
        this.variable = (name.charAt(0) === '@');
    }

    clone() {
        var clone = Object.create(CartoCSS.Tree.Rule.prototype);
        clone.name = this.name;
        clone.value = this.value;
        clone.index = this.index;
        clone.instance = this.instance;
        clone.symbolizer = this.symbolizer;
        clone.filename = this.filename;
        clone.variable = this.variable;
        return clone;
    }

    updateID() {
        return this.id = this.zoom + '#' + this.instance + '#' + this.name;
    }

    toString() {
        return '[' + CartoCSS.Tree.Zoom.toString(this.zoom) + '] ' + this.name + ': ' + this.value;
    }
    ev(context) {
        return new CartoCSS.Tree.Rule(this.name,
            this.value.ev(context),
            this.index,
            this.filename);
    }
};

CartoCSS.Tree.Ruleset = class Ruleset {

    constructor(selectors, rules) {
        this.is = 'ruleset';
        this.selectors = selectors;
        this.rules = rules;
        // static cache of find() function
        this._lookups = {};
    }

    ev(env) {
        var i,
            rule,
            ruleset = new CartoCSS.Tree.Ruleset(this.selectors, this.rules.slice(0));
        ruleset.root = this.root;

        // push the current ruleset to the frames stack
        env.frames.unshift(ruleset);

        // Evaluate everything else
        for (i = 0, rule; i < ruleset.rules.length; i++) {
            rule = ruleset.rules[i];
            ruleset.rules[i] = rule.ev ? rule.ev(env) : rule;
        }

        // Pop the stack
        env.frames.shift();

        return ruleset;
    }

    match(args) {
        return !args || args.length === 0;
    }

    variables() {
        if (this._variables) {
            return this._variables;
        } else {
            return this._variables = this.rules.reduce(function (hash, r) {
                if (r instanceof CartoCSS.Tree.Rule && r.variable === true) {
                    hash[r.name] = r;
                }
                return hash;
            }, {});
        }
    }

    variable(name) {
        return this.variables()[name];
    }

    rulesets() {
        if (this._rulesets) {
            return this._rulesets;
        } else {
            return this._rulesets = this.rules.filter(function (r) {
                return (r instanceof CartoCSS.Tree.Ruleset);
            });
        }
    }

    find(selector, self) {
        self = self || this;
        var rules = [], match,
            key = selector.toString();

        if (key in this._lookups) {
            return this._lookups[key];
        }

        this.rulesets().forEach(function (rule) {
            if (rule !== self) {
                for (var j = 0; j < rule.selectors.length; j++) {
                    match = selector.match(rule.selectors[j]);
                    if (match) {
                        if (selector.elements.length > 1) {
                            Array.prototype.push.apply(rules, rule.find(
                                new CartoCSS.Tree.Selector(null, null, selector.elements.slice(1)), self));
                        } else {
                            rules.push(rule);
                        }
                        break;
                    }
                }
            }
        });
        return this._lookups[key] = rules;
    }

    // Zooms can use variables. This replaces CartoCSS.Tree.Zoom objects on selectors
    // with simple bit-arrays that we can compare easily.
    evZooms(env) {
        for (var i = 0; i < this.selectors.length; i++) {
            var zval = CartoCSS.Tree.Zoom.all;
            for (var z = 0; z < this.selectors[i].zoom.length; z++) {
                zval = this.selectors[i].zoom[z].ev(env).zoom;
            }
            this.selectors[i].zoom = zval;
        }
    }

    flatten(result, parents, env) {
        var selectors = [], i, j;
        if (this.selectors.length === 0) {
            env.frames = env.frames.concat(this.rules);
        }
        // evaluate zoom variables on this object.
        this.evZooms(env);
        for (i = 0; i < this.selectors.length; i++) {
            var child = this.selectors[i];

            if (!child.filters) {
                // This is an invalid filterset.
                continue;
            }

            if (parents.length) {
                for (j = 0; j < parents.length; j++) {
                    var parent = parents[j];

                    var mergedFilters = parent.filters.cloneWith(child.filters);
                    if (mergedFilters === null) {
                        // Filters could be added, but they didn't change the
                        // filters. This means that we only have to clone when
                        // the zoom levels or the attachment is different too.
                        if (parent.zoom === child.zoom &&
                            parent.attachment === child.attachment &&
                            parent.elements.join() === child.elements.join()) {
                            selectors.push(parent);
                            continue;
                        } else {
                            mergedFilters = parent.filters;
                        }
                    } else if (!mergedFilters) {
                        // The merged filters are invalid, that means we don't
                        // have to clone.
                        continue;
                    }

                    var clone = Object.create(CartoCSS.Tree.Selector.prototype);
                    clone.filters = mergedFilters;
                    clone.zoom = child.zoom;
                    clone.elements = parent.elements.concat(child.elements);
                    if (parent.attachment && child.attachment) {
                        clone.attachment = parent.attachment + '/' + child.attachment;
                    } else {clone.attachment = child.attachment || parent.attachment;}
                    clone.conditions = parent.conditions + child.conditions;
                    clone.index = child.index;
                    selectors.push(clone);
                }
            } else {
                selectors.push(child);
            }
        }

        var rules = [];
        for (i = 0; i < this.rules.length; i++) {
            var rule = this.rules[i];

            // Recursively flatten any nested rulesets
            if (rule instanceof CartoCSS.Tree.Ruleset) {
                rule.flatten(result, selectors, env);
            } else if (rule instanceof CartoCSS.Tree.Rule) {
                rules.push(rule);
            } else if (rule instanceof CartoCSS.Tree.Invalid) {
                env.error(rule);
            }
        }

        var index = rules.length ? rules[0].index : false;
        for (i = 0; i < selectors.length; i++) {
            // For specificity sort, use the position of the first rule to allow
            // defining attachments that are under current element as a descendant
            // selector.
            if (index !== false) {
                selectors[i].index = index;
            }
            result.push(new CartoCSS.Tree.Definition(selectors[i], rules.slice()));
        }

        return result;
    }
};

CartoCSS.Tree.Selector = class Selector {
    constructor(filters, zoom, elements, attachment, conditions, index) {
        this.elements = elements || [];
        this.attachment = attachment;
        this.filters = filters || {};
        this.zoom = typeof zoom !== 'undefined' ? zoom : CartoCSS.Tree.Zoom.all;
        this.conditions = conditions;
        this.index = index;
    }

    specificity() {
        return this.elements.reduce(function (memo, e) {
            var spec = e.specificity();
            memo[0] += spec[0];
            memo[1] += spec[1];
            return memo;
        }, [0, 0, this.conditions, this.index]);
    }
};

/*style:class Invalid ),*/

CartoCSS.Tree.URL = class URL {

    constructor(val, paths) {
        this.is = 'uri';
        this.value = val;
        this.paths = paths;
    }

    toString() {
        return this.value.toString();
    }

    ev(ctx) {
        return new CartoCSS.Tree.URL(this.value.ev(ctx), this.paths);
    }
};

CartoCSS.Tree.Value = class Value {

    constructor(value) {
        this.is = 'value';
        this.value = value;
    }

    ev(env) {
        if (this.value.length === 1) {
            return this.value[0].ev(env);
        } else {
            return new CartoCSS.Tree.Value(this.value.map(function (v) {
                return v.ev(env);
            }));
        }
    }

    toJS(env) {
        //var v = this.value[0].value[0];
        var val = this.ev(env);
        var v = val.toString();
        if (val.is === "color" || val.is === 'uri' || val.is === 'string' || val.is === 'keyword') {
            v = "'" + v + "'";
        } else if (val.is === 'field') {
            // replace [varuable] by ctx['variable']
            v = v.replace(/\[(.*)\]/g, "attributes['\_match1']")
        } else if (val.value && typeof val.value === "object") {
            v = "[" + v + "]";
        }

        return "_value = " + v + ";";
    }

    toString(env, selector, sep, format) {
        return this.value.map(function (e) {
            return e.toString(env, format);
        }).join(sep || ', ');
    }

    clone() {
        var obj = Object.create(CartoCSS.Tree.Value.prototype);
        if (Array.isArray(obj)) {obj.value = this.value.slice();} else {obj.value = this.value;}
        obj.is = this.is;
        return obj;
    }
};

CartoCSS.Tree.Variable = class Variable {
    constructor(name, index, filename) {
        this.is = 'variable';
        this.name = name;
        this.index = index;
        this.filename = filename;
    }

    toString() {
        return this.name;
    }

    ev(env) {
        if (this._css) {return this._css;}

        var thisframe = env.frames.filter(function (f) {
            return f.name === this.name;
        }.bind(this));
        if (thisframe.length) {
            return thisframe[0].value.ev(env);
        } else {
            env.error({
                message: 'variable ' + this.name + ' is undefined',
                index: this.index,
                type: 'runtime',
                filename: this.filename
            });
            return {
                is: 'undefined',
                value: 'undefined'
            };
        }
    }
};

CartoCSS.Tree.Zoom = class Zoom {
    constructor(op, value, index) {
        this.op = op;
        this.value = value;
        this.index = index;
    }

    setZoom(zoom) {
        this.zoom = zoom;
        return this;
    }

    ev(env) {
        var value = parseInt(this.value.ev(env).toString(), 10);


        if (value > CartoCSS.Tree.Zoom.maxZoom || value < 0) {
            env.error({
                message: 'Only zoom levels between 0 and ' +
                CartoCSS.Tree.Zoom.maxZoom + ' supported.',
                index: this.index
            });
        }

        switch (this.op) {
            case '=':
                this.zoom = "zoom && zoom === " + value;
                return this;
            case '>':
                this.zoom = "zoom && zoom > " + value;
                break;
            case '>=':
                this.zoom = "zoom && zoom >= " + value;
                break;
            case '<':
                this.zoom = "zoom && zoom < " + value;
                break;
            case '<=':
                this.zoom = "zoom && zoom <= " + value;
                break;
            default:
                break;
        }
        /*
         for (var i = 0; i <= CartoCSS.Tree.Zoom.maxZoom; i++) {
         if (i >= start && i <= end) {
         zoom |= (1 << i);
         }
         }
         this.zoom = zoom;
         this.zoom=value+this.op+"zoom";*/
        return this;
    }

    toString() {
        var str = '';
        for (var i = 0; i <= CartoCSS.Tree.Zoom.maxZoom; i++) {
            str += (this.zoom & (1 << i)) ? 'X' : '.';
        }
        return str;
    }
};

// Covers all zoomlevels from 0 to 22
CartoCSS.Tree.Zoom.all = 23;

CartoCSS.Tree.Zoom.maxZoom = 22;

CartoCSS.Tree.Zoom.ranges = {
    0: 1000000000,
    1: 500000000,
    2: 200000000,
    3: 100000000,
    4: 50000000,
    5: 25000000,
    6: 12500000,
    7: 6500000,
    8: 3000000,
    9: 1500000,
    10: 750000,
    11: 400000,
    12: 200000,
    13: 100000,
    14: 50000,
    15: 25000,
    16: 12500,
    17: 5000,
    18: 2500,
    19: 1500,
    20: 750,
    21: 500,
    22: 250,
    23: 100
};
