/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/
 import { DataFormat } from '../REST';
 import { KernelDensityJobsService } from './KernelDensityJobsService';
 import { SingleObjectQueryJobsService } from './SingleObjectQueryJobsService';
 import { SummaryMeshJobsService } from './SummaryMeshJobsService';
 import { VectorClipJobsService } from './VectorClipJobsService';
 import { OverlayGeoJobsService } from './OverlayGeoJobsService';
 import { SummaryRegionJobsService } from './SummaryRegionJobsService';
 import { BuffersAnalystJobsService } from './BuffersAnalystJobsService';
 import { TopologyValidatorJobsService } from './TopologyValidatorJobsService';
 import { SummaryAttributesJobsService } from './SummaryAttributesJobsService';
 
 /**
  * @class ProcessingService
  * @category  iServer ProcessingService
  * @classdesc 分布式分析相关服务类。
  * @extends {ServiceBase}
  * @example
  * new ProcessingService(url,options)
  *  .getKernelDensityJobs(function(result){
  *     //doSomething
  * })
  * @param {string} url - 服务地址。 
  * @param {Object} options - 参数。
  * @param {string} [options.proxy] - 服务代理地址。
  * @param {boolean} [options.withCredentials=false] - 请求是否携带cookie。
  * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
  * @param {Object} [options.headers] - 请求头。
  * @usage
  */
 export class ProcessingService {
 
     constructor(url, options) {
         this.url = url;
         this.options = options || {};
         this.kernelDensityJobs = {};
         this.summaryMeshJobs = {};
         this.queryJobs = {};
         this.summaryRegionJobs = {};
         this.vectorClipJobs = {};
         this.overlayGeoJobs = {};
         this.buffersJobs = {};
         this.topologyValidatorJobs = {};
         this.summaryAttributesJobs = {};
     }
 
     /**
      * @function ProcessingService.prototype.getKernelDensityJobs
      * @description 获取密度分析的列表。
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat} [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getKernelDensityJobs(callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var kernelDensityJobsService = new KernelDensityJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return kernelDensityJobsService.getKernelDensityJobs(callback);
     }
 
     /**
      * @function ProcessingService.prototype.getKernelDensityJob
      * @description 获取某个密度分析。
      * @param {string} id - 空间分析的ID。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat} [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getKernelDensityJob(id, callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var kernelDensityJobsService = new KernelDensityJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return kernelDensityJobsService.getKernelDensityJob(id, callback);
     }
 
     /**
      * @function ProcessingService.prototype.addKernelDensityJob
      * @description 密度分析。
      * @param {KernelDensityJobParameter} params -密度分析参数类。 
      * @param {RequestCallback} callback 回调函数。 
      * @param {number} [seconds=1000] - 获取创建成功结果的时间间隔。 
      * @param {DataFormat} [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     addKernelDensityJob(params, callback, seconds, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var kernelDensityJobsService = new KernelDensityJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return kernelDensityJobsService.addKernelDensityJob(params, seconds, callback, function (job) {
          me.kernelDensityJobs[job.id] = job.state;
        });
     }
 
     /**
      * @function ProcessingService.prototype.getKernelDensityJobState
      * @description 获取密度分析的状态。
      * @param {string} id - 密度分析的id。
      * @returns {Object} 密度分析的状态。
      */
     getKernelDensityJobState(id) {
         return this.kernelDensityJobs[id];
     }
 
     /**
      * @function ProcessingService.prototype.getSummaryMeshJobs
      * @description 获取点聚合分析的列表。
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat} [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getSummaryMeshJobs(callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var summaryMeshJobsService = new SummaryMeshJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return summaryMeshJobsService.getSummaryMeshJobs(callback);
     }
 
     /**
      * @function ProcessingService.prototype.getSummaryMeshJob
      * @description 获取某个点聚合分析。
      * @param {string} id - 空间分析的 ID。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat} [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getSummaryMeshJob(id, callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var summaryMeshJobsService = new SummaryMeshJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return summaryMeshJobsService.getSummaryMeshJob(id, callback);
     }
 
     /**
      * @function ProcessingService.prototype.addSummaryMeshJob
      * @description 点聚合分析。
      * @param {SummaryMeshJobParameter} params - 点聚合分析任务参数类。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {number} [seconds=1000] - 获取创建成功结果的时间间隔。
      * @param {DataFormat} [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     addSummaryMeshJob(params, callback, seconds, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var summaryMeshJobsService = new SummaryMeshJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return summaryMeshJobsService.addSummaryMeshJob(params, seconds, callback, function (job) {
          me.summaryMeshJobs[job.id] = job.state;
        });
     }
 
     /**
      * @function ProcessingService.prototype.getSummaryMeshJobState
      * @description 获取点聚合分析的状态。
      * @param {string} id - 点聚合分析的 ID。
      * @returns {Object} 点聚合分析的状态。
      */
     getSummaryMeshJobState(id) {
         return this.summaryMeshJobs[id];
     }
 
     /**
      * @function ProcessingService.prototype.getQueryJobs
      * @description 获取单对象查询分析的列表。
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat} [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getQueryJobs(callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var singleObjectQueryJobsService = new SingleObjectQueryJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return singleObjectQueryJobsService.getQueryJobs(callback);
     }
 
     /**
      * @function ProcessingService.prototype.getQueryJob
      * @description 获取某个单对象查询分析。
      * @param {string} id - 空间分析的 ID。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat} [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getQueryJob(id, callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var singleObjectQueryJobsService = new SingleObjectQueryJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return singleObjectQueryJobsService.getQueryJob(id, callback);
     }
 
     /**
      * @function ProcessingService.prototype.addQueryJob
      * @description 单对象查询分析。
      * @param {SingleObjectQueryJobsParameter} params - 单对象查询分析的请求参数。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {number} [seconds=1000] - 获取创建成功结果的时间间隔。 
      * @param {DataFormat} [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     addQueryJob(params, callback, seconds, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var singleObjectQueryJobsService = new SingleObjectQueryJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return singleObjectQueryJobsService.addQueryJob(params, seconds, callback, function (job) {
          me.queryJobs[job.id] = job.state;
        });
     }
 
     /**
      * @function ProcessingService.prototype.getQueryJobState
      * @description 获取单对象查询分析的状态。
      * @param {string} id - 单对象查询分析的 ID。
      * @returns {Object} 单对象查询分析的状态。
      */
     getQueryJobState(id) {
         return this.queryJobs[id];
     }
 
     /**
      * @function ProcessingService.prototype.getSummaryRegionJobs
      * @description 获取区域汇总分析的列表。
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat} [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getSummaryRegionJobs(callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var summaryRegionJobsService = new SummaryRegionJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return summaryRegionJobsService.getSummaryRegionJobs(callback);
     }
 
     /**
      * @function ProcessingService.prototype.getSummaryRegionJob
      * @description 获取某个区域汇总分析。
      * @param {string} id - 区域汇总分析的 ID。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat} [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getSummaryRegionJob(id, callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var summaryRegionJobsService = new SummaryRegionJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return summaryRegionJobsService.getSummaryRegionJob(id, callback);
     }
 
     /**
      * @function ProcessingService.prototype.addSummaryRegionJob
      * @description 区域汇总分析。
      * @param {SummaryRegionJobParameter} params - 区域汇总分析参数类。
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {number} [seconds=1000] - 获取创建成功结果的时间间隔。
      * @param {DataFormat} [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     addSummaryRegionJob(params, callback, seconds, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var summaryRegionJobsService = new SummaryRegionJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return summaryRegionJobsService.addSummaryRegionJob(params, seconds, callback, function (job) {
          me.summaryRegionJobs[job.id] = job.state;
      });
     }
 
     /**
      * @function ProcessingService.prototype.getSummaryRegionJobState
      * @description 获取区域汇总分析的状态。
      * @param {string} id - 生成区域汇总分析的 ID。
      * @returns {Object} 区域汇总分析的状态。
      */
     getSummaryRegionJobState(id) {
         return this.summaryRegionJobs[id];
     }
 
     /**
      * @function ProcessingService.prototype.getVectorClipJobs
      * @description 获取矢量裁剪分析的列表。
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getVectorClipJobs(callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var vectorClipJobsService = new VectorClipJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return vectorClipJobsService.getVectorClipJobs(callback);
     }
 
     /**
      * @function ProcessingService.prototype.getVectorClipJob
      * @description 获取某个矢量裁剪分析。
      * @param {string} id - 空间分析的 ID。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getVectorClipJob(id, callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var vectorClipJobsService = new VectorClipJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return vectorClipJobsService.getVectorClipJob(id, callback);
     }
 
     /**
      * @function ProcessingService.prototype.addVectorClipJob
      * @description 矢量裁剪分析。
      * @param {VectorClipJobsParameter} params - 矢量裁剪分析请求参数类。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {number} [seconds=1000] - 获取创建成功结果的时间间隔。 
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     addVectorClipJob(params, callback, seconds, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var vectorClipJobsService = new VectorClipJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return vectorClipJobsService.addVectorClipJob(params, seconds, callback, function (job) {
          me.vectorClipJobs[job.id] = job.state;
        });
     }
 
     /**
      * @function ProcessingService.prototype.getVectorClipJobState
      * @description 获取矢量裁剪分析的状态。
      * @param {number} id - 矢量裁剪分析的ID。
      * @returns {Object} 矢量裁剪分析的状态。
      */
     getVectorClipJobState(id) {
         return this.vectorClipJobs[id];
     }
 
     /**
      * @function ProcessingService.prototype.getOverlayGeoJobs
      * @description 获取叠加分析的列表。
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getOverlayGeoJobs(callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var overlayGeoJobsService = new OverlayGeoJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return overlayGeoJobsService.getOverlayGeoJobs(callback);
     }
 
     /**
      * @function ProcessingService.prototype.getOverlayGeoJob
      * @description 获取某个叠加分析。
      * @param {string} id - 空间分析的 ID。
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getOverlayGeoJob(id, callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var overlayGeoJobsService = new OverlayGeoJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return overlayGeoJobsService.getOverlayGeoJob(id, callback);
     }
 
     /**
      * @function ProcessingService.prototype.addOverlayGeoJob
      * @description 叠加分析。
      * @param {OverlayGeoJobParameter} params - 叠加分析请求参数类。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {number} [seconds=1000] - 获取创建成功结果的时间间隔。 
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     addOverlayGeoJob(params, callback, seconds, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var overlayGeoJobsService = new OverlayGeoJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return overlayGeoJobsService.addOverlayGeoJob(params, seconds, callback, function (job) {
          me.overlayGeoJobs[job.id] = job.state;
        });
     }
 
     /**
      * @function ProcessingService.prototype.getoverlayGeoJobState
      * @description 获取叠加分析的状态。
      * @param {string} id - 叠加分析的 ID。
      * @returns {Object} 叠加分析的状态。
      */
     getoverlayGeoJobState(id) {
         return this.overlayGeoJobs[id];
     }
 
     /**
      * @function ProcessingService.prototype.getBuffersJobs
      * @description 获取缓冲区分析的列表。
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getBuffersJobs(callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var buffersAnalystJobsService = new BuffersAnalystJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return buffersAnalystJobsService.getBuffersJobs(callback);
     }
 
     /**
      * @function ProcessingService.prototype.getBuffersJob
      * @description 获取某个缓冲区分析。
      * @param {string} id - 空间分析的 ID。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getBuffersJob(id, callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var buffersAnalystJobsService = new BuffersAnalystJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return buffersAnalystJobsService.getBuffersJob(id, callback);
     }
 
     /**
      * @function ProcessingService.prototype.addBuffersJob
      * @description 缓冲区分析。
      * @param {BuffersAnalystJobsParameter} params - 缓冲区分析请求参数类。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {number} seconds - 获取创建成功结果的时间间隔。 
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     addBuffersJob(params, callback, seconds, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var buffersAnalystJobsService = new BuffersAnalystJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return buffersAnalystJobsService.addBuffersJob(params, seconds, callback, function (job) {
          me.buffersJobs[job.id] = job.state;
      });
     }
 
     /**
      * @function ProcessingService.prototype.getBuffersJobState
      * @description 获取缓冲区分析的状态。
      * @param {string} id - 缓冲区分析的 ID。
      * @returns {Object} 缓冲区分析的状态。
      */
     getBuffersJobState(id) {
         return this.buffersJobs[id];
     }
 
     /**
      * @function ProcessingService.prototype.getTopologyValidatorJobs
      * @description 获取拓扑检查分析的列表。
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getTopologyValidatorJobs(callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var topologyValidatorJobsService = new TopologyValidatorJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return topologyValidatorJobsService.getTopologyValidatorJobs(callback);
     }
 
     /**
      * @function ProcessingService.prototype.getTopologyValidatorJob
      * @description 获取某个拓扑检查分析。
      * @param {string} id - 空间分析的 ID。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getTopologyValidatorJob(id, callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var topologyValidatorJobsService = new TopologyValidatorJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return topologyValidatorJobsService.getTopologyValidatorJob(id, callback);
     }
 
     /**
      * @function ProcessingService.prototype.addTopologyValidatorJob
      * @description 拓扑检查分析。
      * @param {TopologyValidatorJobsParameter} params - 拓扑检查分析请求参数类。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {number} [seconds=1000] - 获取创建成功结果的时间间隔。 
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     addTopologyValidatorJob(params, callback, seconds, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var topologyValidatorJobsService = new TopologyValidatorJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return topologyValidatorJobsService.addTopologyValidatorJob(params, seconds, callback, function (job) {
          me.topologyValidatorJobs[job.id] = job.state;
      });
     }
 
     /**
      * @function ProcessingService.prototype.getTopologyValidatorJobState
      * @description 获取拓扑检查分析的状态。
      * @param {string} id - 拓扑检查分析的 ID。
      * @returns {Object} 拓扑检查分析的状态。
      */
     getTopologyValidatorJobState(id) {
         return this.topologyValidatorJobs[id];
     }
 
     /**
      * @function ProcessingService.prototype.getSummaryAttributesJobs
      * @description 获取属性汇总分析的列表。
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getSummaryAttributesJobs(callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var summaryAttributesJobsService = new SummaryAttributesJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return summaryAttributesJobsService.getSummaryAttributesJobs(callback);
     }
 
     /**
      * @function ProcessingService.prototype.getSummaryAttributesJob
      * @description 获取某个属性汇总分析。
      * @param {string} id - 空间分析的 ID。
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     getSummaryAttributesJob(id, callback, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var summaryAttributesJobsService = new SummaryAttributesJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return summaryAttributesJobsService.getSummaryAttributesJob(id, callback);
     }
 
     /**
      * @function ProcessingService.prototype.addSummaryAttributesJob
      * @description 属性汇总分析。
      * @param {SummaryAttributesJobsParameter} params - 属性汇总分析参数类。 
      * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
      * @param {number} [seconds=1000] - 获取创建成功结果的时间间隔。
      * @param {DataFormat}  [resultFormat=DataFormat.GEOJSON] - 返回结果类型。
      * @returns {Promise} Promise 对象。
      */
     addSummaryAttributesJob(params, callback, seconds, resultFormat) {
         var me = this,
             format = me._processFormat(resultFormat);
         var summaryAttributesJobsService = new SummaryAttributesJobsService(me.url, {
             proxy: me.options.proxy,
             withCredentials: me.options.withCredentials,
             crossOrigin: me.options.crossOrigin,
             headers: me.options.headers,
             format: format
         });
         return summaryAttributesJobsService.addSummaryAttributesJob(params, seconds, callback, function (job) {
          me.summaryAttributesJobs[job.id] = job.state;
      });
     }
 
     /**
      * @function ProcessingService.prototype.getSummaryAttributesJobState
      * @description 获取属性汇总分析的状态。
      * @param {string} id - 属性汇总分析的 ID。 
      * @returns {Object} 属性汇总分析的状态
      */
     getSummaryAttributesJobState(id) {
         return this.summaryAttributesJobs[id];
     }
 
     _processFormat(resultFormat) {
         return (resultFormat) ? resultFormat : DataFormat.GEOJSON;
     }
 }
 