/*!
 * 
 *          iclient-leaflet
 *          Copyright© 2000 - 2023 SuperMap Software Co.Ltd
 *          license: Apache-2.0
 *          version: v11.1.1
 *
 */
/******/ (() => { // webpackBootstrap
/******/ 	var __webpack_modules__ = ({

/***/ 550:
/***/ ((module) => {

"use strict";


module.exports = Point;

/**
 * A standalone point geometry with useful accessor, comparison, and
 * modification methods.
 *
 * @class Point
 * @param {Number} x the x-coordinate. this could be longitude or screen
 * pixels, or any other sort of unit.
 * @param {Number} y the y-coordinate. this could be latitude or screen
 * pixels, or any other sort of unit.
 * @example
 * var point = new Point(-77, 38);
 */
function Point(x, y) {
    this.x = x;
    this.y = y;
}

Point.prototype = {

    /**
     * Clone this point, returning a new point that can be modified
     * without affecting the old one.
     * @return {Point} the clone
     */
    clone: function() { return new Point(this.x, this.y); },

    /**
     * Add this point's x & y coordinates to another point,
     * yielding a new point.
     * @param {Point} p the other point
     * @return {Point} output point
     */
    add:     function(p) { return this.clone()._add(p); },

    /**
     * Subtract this point's x & y coordinates to from point,
     * yielding a new point.
     * @param {Point} p the other point
     * @return {Point} output point
     */
    sub:     function(p) { return this.clone()._sub(p); },

    /**
     * Multiply this point's x & y coordinates by point,
     * yielding a new point.
     * @param {Point} p the other point
     * @return {Point} output point
     */
    multByPoint:    function(p) { return this.clone()._multByPoint(p); },

    /**
     * Divide this point's x & y coordinates by point,
     * yielding a new point.
     * @param {Point} p the other point
     * @return {Point} output point
     */
    divByPoint:     function(p) { return this.clone()._divByPoint(p); },

    /**
     * Multiply this point's x & y coordinates by a factor,
     * yielding a new point.
     * @param {Point} k factor
     * @return {Point} output point
     */
    mult:    function(k) { return this.clone()._mult(k); },

    /**
     * Divide this point's x & y coordinates by a factor,
     * yielding a new point.
     * @param {Point} k factor
     * @return {Point} output point
     */
    div:     function(k) { return this.clone()._div(k); },

    /**
     * Rotate this point around the 0, 0 origin by an angle a,
     * given in radians
     * @param {Number} a angle to rotate around, in radians
     * @return {Point} output point
     */
    rotate:  function(a) { return this.clone()._rotate(a); },

    /**
     * Rotate this point around p point by an angle a,
     * given in radians
     * @param {Number} a angle to rotate around, in radians
     * @param {Point} p Point to rotate around
     * @return {Point} output point
     */
    rotateAround:  function(a,p) { return this.clone()._rotateAround(a,p); },

    /**
     * Multiply this point by a 4x1 transformation matrix
     * @param {Array<Number>} m transformation matrix
     * @return {Point} output point
     */
    matMult: function(m) { return this.clone()._matMult(m); },

    /**
     * Calculate this point but as a unit vector from 0, 0, meaning
     * that the distance from the resulting point to the 0, 0
     * coordinate will be equal to 1 and the angle from the resulting
     * point to the 0, 0 coordinate will be the same as before.
     * @return {Point} unit vector point
     */
    unit:    function() { return this.clone()._unit(); },

    /**
     * Compute a perpendicular point, where the new y coordinate
     * is the old x coordinate and the new x coordinate is the old y
     * coordinate multiplied by -1
     * @return {Point} perpendicular point
     */
    perp:    function() { return this.clone()._perp(); },

    /**
     * Return a version of this point with the x & y coordinates
     * rounded to integers.
     * @return {Point} rounded point
     */
    round:   function() { return this.clone()._round(); },

    /**
     * Return the magitude of this point: this is the Euclidean
     * distance from the 0, 0 coordinate to this point's x and y
     * coordinates.
     * @return {Number} magnitude
     */
    mag: function() {
        return Math.sqrt(this.x * this.x + this.y * this.y);
    },

    /**
     * Judge whether this point is equal to another point, returning
     * true or false.
     * @param {Point} other the other point
     * @return {boolean} whether the points are equal
     */
    equals: function(other) {
        return this.x === other.x &&
               this.y === other.y;
    },

    /**
     * Calculate the distance from this point to another point
     * @param {Point} p the other point
     * @return {Number} distance
     */
    dist: function(p) {
        return Math.sqrt(this.distSqr(p));
    },

    /**
     * Calculate the distance from this point to another point,
     * without the square root step. Useful if you're comparing
     * relative distances.
     * @param {Point} p the other point
     * @return {Number} distance
     */
    distSqr: function(p) {
        var dx = p.x - this.x,
            dy = p.y - this.y;
        return dx * dx + dy * dy;
    },

    /**
     * Get the angle from the 0, 0 coordinate to this point, in radians
     * coordinates.
     * @return {Number} angle
     */
    angle: function() {
        return Math.atan2(this.y, this.x);
    },

    /**
     * Get the angle from this point to another point, in radians
     * @param {Point} b the other point
     * @return {Number} angle
     */
    angleTo: function(b) {
        return Math.atan2(this.y - b.y, this.x - b.x);
    },

    /**
     * Get the angle between this point and another point, in radians
     * @param {Point} b the other point
     * @return {Number} angle
     */
    angleWith: function(b) {
        return this.angleWithSep(b.x, b.y);
    },

    /*
     * Find the angle of the two vectors, solving the formula for
     * the cross product a x b = |a||b|sin(θ) for θ.
     * @param {Number} x the x-coordinate
     * @param {Number} y the y-coordinate
     * @return {Number} the angle in radians
     */
    angleWithSep: function(x, y) {
        return Math.atan2(
            this.x * y - this.y * x,
            this.x * x + this.y * y);
    },

    _matMult: function(m) {
        var x = m[0] * this.x + m[1] * this.y,
            y = m[2] * this.x + m[3] * this.y;
        this.x = x;
        this.y = y;
        return this;
    },

    _add: function(p) {
        this.x += p.x;
        this.y += p.y;
        return this;
    },

    _sub: function(p) {
        this.x -= p.x;
        this.y -= p.y;
        return this;
    },

    _mult: function(k) {
        this.x *= k;
        this.y *= k;
        return this;
    },

    _div: function(k) {
        this.x /= k;
        this.y /= k;
        return this;
    },

    _multByPoint: function(p) {
        this.x *= p.x;
        this.y *= p.y;
        return this;
    },

    _divByPoint: function(p) {
        this.x /= p.x;
        this.y /= p.y;
        return this;
    },

    _unit: function() {
        this._div(this.mag());
        return this;
    },

    _perp: function() {
        var y = this.y;
        this.y = this.x;
        this.x = -y;
        return this;
    },

    _rotate: function(angle) {
        var cos = Math.cos(angle),
            sin = Math.sin(angle),
            x = cos * this.x - sin * this.y,
            y = sin * this.x + cos * this.y;
        this.x = x;
        this.y = y;
        return this;
    },

    _rotateAround: function(angle, p) {
        var cos = Math.cos(angle),
            sin = Math.sin(angle),
            x = p.x + cos * (this.x - p.x) - sin * (this.y - p.y),
            y = p.y + sin * (this.x - p.x) + cos * (this.y - p.y);
        this.x = x;
        this.y = y;
        return this;
    },

    _round: function() {
        this.x = Math.round(this.x);
        this.y = Math.round(this.y);
        return this;
    }
};

/**
 * Construct a point from an array if necessary, otherwise if the input
 * is already a Point, or an unknown type, return it unchanged
 * @param {Array<Number>|Point|*} a any kind of input value
 * @return {Point} constructed point, or passed-through value.
 * @example
 * // this
 * var point = Point.convert([0, 1]);
 * // is equivalent to
 * var point = new Point(0, 1);
 */
Point.convert = function (a) {
    if (a instanceof Point) {
        return a;
    }
    if (Array.isArray(a)) {
        return new Point(a[0], a[1]);
    }
    return a;
};


/***/ }),

/***/ 929:
/***/ ((module, __unused_webpack_exports, __webpack_require__) => {

module.exports.VectorTile = __webpack_require__(779);
/* unused reexport */ __webpack_require__(24);
/* unused reexport */ __webpack_require__(701);


/***/ }),

/***/ 779:
/***/ ((module, __unused_webpack_exports, __webpack_require__) => {

"use strict";


var VectorTileLayer = __webpack_require__(701);

module.exports = VectorTile;

function VectorTile(pbf, end) {
    this.layers = pbf.readFields(readTile, {}, end);
}

function readTile(tag, layers, pbf) {
    if (tag === 3) {
        var layer = new VectorTileLayer(pbf, pbf.readVarint() + pbf.pos);
        if (layer.length) layers[layer.name] = layer;
    }
}



/***/ }),

/***/ 24:
/***/ ((module, __unused_webpack_exports, __webpack_require__) => {

"use strict";


var Point = __webpack_require__(550);

module.exports = VectorTileFeature;

function VectorTileFeature(pbf, end, extent, keys, values) {
    // Public
    this.properties = {};
    this.extent = extent;
    this.type = 0;

    // Private
    this._pbf = pbf;
    this._geometry = -1;
    this._keys = keys;
    this._values = values;

    pbf.readFields(readFeature, this, end);
}

function readFeature(tag, feature, pbf) {
    if (tag == 1) feature.id = pbf.readVarint();
    else if (tag == 2) readTag(pbf, feature);
    else if (tag == 3) feature.type = pbf.readVarint();
    else if (tag == 4) feature._geometry = pbf.pos;
}

function readTag(pbf, feature) {
    var end = pbf.readVarint() + pbf.pos;

    while (pbf.pos < end) {
        var key = feature._keys[pbf.readVarint()],
            value = feature._values[pbf.readVarint()];
        feature.properties[key] = value;
    }
}

VectorTileFeature.types = ['Unknown', 'Point', 'LineString', 'Polygon'];

VectorTileFeature.prototype.loadGeometry = function() {
    var pbf = this._pbf;
    pbf.pos = this._geometry;

    var end = pbf.readVarint() + pbf.pos,
        cmd = 1,
        length = 0,
        x = 0,
        y = 0,
        lines = [],
        line;

    while (pbf.pos < end) {
        if (length <= 0) {
            var cmdLen = pbf.readVarint();
            cmd = cmdLen & 0x7;
            length = cmdLen >> 3;
        }

        length--;

        if (cmd === 1 || cmd === 2) {
            x += pbf.readSVarint();
            y += pbf.readSVarint();

            if (cmd === 1) { // moveTo
                if (line) lines.push(line);
                line = [];
            }

            line.push(new Point(x, y));

        } else if (cmd === 7) {

            // Workaround for https://github.com/mapbox/mapnik-vector-tile/issues/90
            if (line) {
                line.push(line[0].clone()); // closePolygon
            }

        } else {
            throw new Error('unknown command ' + cmd);
        }
    }

    if (line) lines.push(line);

    return lines;
};

VectorTileFeature.prototype.bbox = function() {
    var pbf = this._pbf;
    pbf.pos = this._geometry;

    var end = pbf.readVarint() + pbf.pos,
        cmd = 1,
        length = 0,
        x = 0,
        y = 0,
        x1 = Infinity,
        x2 = -Infinity,
        y1 = Infinity,
        y2 = -Infinity;

    while (pbf.pos < end) {
        if (length <= 0) {
            var cmdLen = pbf.readVarint();
            cmd = cmdLen & 0x7;
            length = cmdLen >> 3;
        }

        length--;

        if (cmd === 1 || cmd === 2) {
            x += pbf.readSVarint();
            y += pbf.readSVarint();
            if (x < x1) x1 = x;
            if (x > x2) x2 = x;
            if (y < y1) y1 = y;
            if (y > y2) y2 = y;

        } else if (cmd !== 7) {
            throw new Error('unknown command ' + cmd);
        }
    }

    return [x1, y1, x2, y2];
};

VectorTileFeature.prototype.toGeoJSON = function(x, y, z) {
    var size = this.extent * Math.pow(2, z),
        x0 = this.extent * x,
        y0 = this.extent * y,
        coords = this.loadGeometry(),
        type = VectorTileFeature.types[this.type],
        i, j;

    function project(line) {
        for (var j = 0; j < line.length; j++) {
            var p = line[j], y2 = 180 - (p.y + y0) * 360 / size;
            line[j] = [
                (p.x + x0) * 360 / size - 180,
                360 / Math.PI * Math.atan(Math.exp(y2 * Math.PI / 180)) - 90
            ];
        }
    }

    switch (this.type) {
    case 1:
        var points = [];
        for (i = 0; i < coords.length; i++) {
            points[i] = coords[i][0];
        }
        coords = points;
        project(coords);
        break;

    case 2:
        for (i = 0; i < coords.length; i++) {
            project(coords[i]);
        }
        break;

    case 3:
        coords = classifyRings(coords);
        for (i = 0; i < coords.length; i++) {
            for (j = 0; j < coords[i].length; j++) {
                project(coords[i][j]);
            }
        }
        break;
    }

    if (coords.length === 1) {
        coords = coords[0];
    } else {
        type = 'Multi' + type;
    }

    var result = {
        type: "Feature",
        geometry: {
            type: type,
            coordinates: coords
        },
        properties: this.properties
    };

    if ('id' in this) {
        result.id = this.id;
    }

    return result;
};

// classifies an array of rings into polygons with outer rings and holes

function classifyRings(rings) {
    var len = rings.length;

    if (len <= 1) return [rings];

    var polygons = [],
        polygon,
        ccw;

    for (var i = 0; i < len; i++) {
        var area = signedArea(rings[i]);
        if (area === 0) continue;

        if (ccw === undefined) ccw = area < 0;

        if (ccw === area < 0) {
            if (polygon) polygons.push(polygon);
            polygon = [rings[i]];

        } else {
            polygon.push(rings[i]);
        }
    }
    if (polygon) polygons.push(polygon);

    return polygons;
}

function signedArea(ring) {
    var sum = 0;
    for (var i = 0, len = ring.length, j = len - 1, p1, p2; i < len; j = i++) {
        p1 = ring[i];
        p2 = ring[j];
        sum += (p2.x - p1.x) * (p1.y + p2.y);
    }
    return sum;
}


/***/ }),

/***/ 701:
/***/ ((module, __unused_webpack_exports, __webpack_require__) => {

"use strict";


var VectorTileFeature = __webpack_require__(24);

module.exports = VectorTileLayer;

function VectorTileLayer(pbf, end) {
    // Public
    this.version = 1;
    this.name = null;
    this.extent = 4096;
    this.length = 0;

    // Private
    this._pbf = pbf;
    this._keys = [];
    this._values = [];
    this._features = [];

    pbf.readFields(readLayer, this, end);

    this.length = this._features.length;
}

function readLayer(tag, layer, pbf) {
    if (tag === 15) layer.version = pbf.readVarint();
    else if (tag === 1) layer.name = pbf.readString();
    else if (tag === 5) layer.extent = pbf.readVarint();
    else if (tag === 2) layer._features.push(pbf.pos);
    else if (tag === 3) layer._keys.push(pbf.readString());
    else if (tag === 4) layer._values.push(readValueMessage(pbf));
}

function readValueMessage(pbf) {
    var value = null,
        end = pbf.readVarint() + pbf.pos;

    while (pbf.pos < end) {
        var tag = pbf.readVarint() >> 3;

        value = tag === 1 ? pbf.readString() :
            tag === 2 ? pbf.readFloat() :
            tag === 3 ? pbf.readDouble() :
            tag === 4 ? pbf.readVarint64() :
            tag === 5 ? pbf.readVarint() :
            tag === 6 ? pbf.readSVarint() :
            tag === 7 ? pbf.readBoolean() : null;
    }

    return value;
}

// return feature `i` from this layer as a `VectorTileFeature`
VectorTileLayer.prototype.feature = function(i) {
    if (i < 0 || i >= this._features.length) throw new Error('feature index out of bounds');

    this._pbf.pos = this._features[i];

    var end = this._pbf.readVarint() + this._pbf.pos;
    return new VectorTileFeature(this._pbf, end, this.extent, this._keys, this._values);
};


/***/ }),

/***/ 693:
/***/ (function(module) {

(function(self) {
  'use strict';

  // if __disableNativeFetch is set to true, the it will always polyfill fetch
  // with Ajax.
  if (!self.__disableNativeFetch && self.fetch) {
    return
  }

  function normalizeName(name) {
    if (typeof name !== 'string') {
      name = String(name)
    }
    if (/[^a-z0-9\-#$%&'*+.\^_`|~]/i.test(name)) {
      throw new TypeError('Invalid character in header field name')
    }
    return name.toLowerCase()
  }

  function normalizeValue(value) {
    if (typeof value !== 'string') {
      value = String(value)
    }
    return value
  }

  function Headers(headers) {
    this.map = {}

    if (headers instanceof Headers) {
      headers.forEach(function(value, name) {
        this.append(name, value)
      }, this)

    } else if (headers) {
      Object.getOwnPropertyNames(headers).forEach(function(name) {
        this.append(name, headers[name])
      }, this)
    }
  }

  Headers.prototype.append = function(name, value) {
    name = normalizeName(name)
    value = normalizeValue(value)
    var list = this.map[name]
    if (!list) {
      list = []
      this.map[name] = list
    }
    list.push(value)
  }

  Headers.prototype['delete'] = function(name) {
    delete this.map[normalizeName(name)]
  }

  Headers.prototype.get = function(name) {
    var values = this.map[normalizeName(name)]
    return values ? values[0] : null
  }

  Headers.prototype.getAll = function(name) {
    return this.map[normalizeName(name)] || []
  }

  Headers.prototype.has = function(name) {
    return this.map.hasOwnProperty(normalizeName(name))
  }

  Headers.prototype.set = function(name, value) {
    this.map[normalizeName(name)] = [normalizeValue(value)]
  }

  Headers.prototype.forEach = function(callback, thisArg) {
    Object.getOwnPropertyNames(this.map).forEach(function(name) {
      this.map[name].forEach(function(value) {
        callback.call(thisArg, value, name, this)
      }, this)
    }, this)
  }

  function consumed(body) {
    if (body.bodyUsed) {
      return Promise.reject(new TypeError('Already read'))
    }
    body.bodyUsed = true
  }

  function fileReaderReady(reader) {
    return new Promise(function(resolve, reject) {
      reader.onload = function() {
        resolve(reader.result)
      }
      reader.onerror = function() {
        reject(reader.error)
      }
    })
  }

  function readBlobAsArrayBuffer(blob) {
    var reader = new FileReader()
    reader.readAsArrayBuffer(blob)
    return fileReaderReady(reader)
  }

  function readBlobAsText(blob, options) {
    var reader = new FileReader()
    var contentType = options.headers.map['content-type'] ? options.headers.map['content-type'].toString() : ''
    var regex = /charset\=[0-9a-zA-Z\-\_]*;?/
    var _charset = blob.type.match(regex) || contentType.match(regex)
    var args = [blob]

    if(_charset) {
      args.push(_charset[0].replace(/^charset\=/, '').replace(/;$/, ''))
    }

    reader.readAsText.apply(reader, args)
    return fileReaderReady(reader)
  }

  var support = {
    blob: 'FileReader' in self && 'Blob' in self && (function() {
      try {
        new Blob();
        return true
      } catch(e) {
        return false
      }
    })(),
    formData: 'FormData' in self,
    arrayBuffer: 'ArrayBuffer' in self
  }

  function Body() {
    this.bodyUsed = false


    this._initBody = function(body, options) {
      this._bodyInit = body
      if (typeof body === 'string') {
        this._bodyText = body
      } else if (support.blob && Blob.prototype.isPrototypeOf(body)) {
        this._bodyBlob = body
        this._options = options
      } else if (support.formData && FormData.prototype.isPrototypeOf(body)) {
        this._bodyFormData = body
      } else if (!body) {
        this._bodyText = ''
      } else if (support.arrayBuffer && ArrayBuffer.prototype.isPrototypeOf(body)) {
        // Only support ArrayBuffers for POST method.
        // Receiving ArrayBuffers happens via Blobs, instead.
      } else {
        throw new Error('unsupported BodyInit type')
      }
    }

    if (support.blob) {
      this.blob = function() {
        var rejected = consumed(this)
        if (rejected) {
          return rejected
        }

        if (this._bodyBlob) {
          return Promise.resolve(this._bodyBlob)
        } else if (this._bodyFormData) {
          throw new Error('could not read FormData body as blob')
        } else {
          return Promise.resolve(new Blob([this._bodyText]))
        }
      }

      this.arrayBuffer = function() {
        return this.blob().then(readBlobAsArrayBuffer)
      }

      this.text = function() {
        var rejected = consumed(this)
        if (rejected) {
          return rejected
        }

        if (this._bodyBlob) {
          return readBlobAsText(this._bodyBlob, this._options)
        } else if (this._bodyFormData) {
          throw new Error('could not read FormData body as text')
        } else {
          return Promise.resolve(this._bodyText)
        }
      }
    } else {
      this.text = function() {
        var rejected = consumed(this)
        return rejected ? rejected : Promise.resolve(this._bodyText)
      }
    }

    if (support.formData) {
      this.formData = function() {
        return this.text().then(decode)
      }
    }

    this.json = function() {
      return this.text().then(JSON.parse)
    }

    return this
  }

  // HTTP methods whose capitalization should be normalized
  var methods = ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'POST', 'PUT']

  function normalizeMethod(method) {
    var upcased = method.toUpperCase()
    return (methods.indexOf(upcased) > -1) ? upcased : method
  }

  function Request(input, options) {
    options = options || {}
    var body = options.body
    if (Request.prototype.isPrototypeOf(input)) {
      if (input.bodyUsed) {
        throw new TypeError('Already read')
      }
      this.url = input.url
      this.credentials = input.credentials
      if (!options.headers) {
        this.headers = new Headers(input.headers)
      }
      this.method = input.method
      this.mode = input.mode
      if (!body) {
        body = input._bodyInit
        input.bodyUsed = true
      }
    } else {
      this.url = input
    }

    this.credentials = options.credentials || this.credentials || 'omit'
    if (options.headers || !this.headers) {
      this.headers = new Headers(options.headers)
    }
    this.method = normalizeMethod(options.method || this.method || 'GET')
    this.mode = options.mode || this.mode || null
    this.referrer = null

    if ((this.method === 'GET' || this.method === 'HEAD') && body) {
      throw new TypeError('Body not allowed for GET or HEAD requests')
    }
    this._initBody(body, options)
  }

  Request.prototype.clone = function() {
    return new Request(this)
  }

  function decode(body) {
    var form = new FormData()
    body.trim().split('&').forEach(function(bytes) {
      if (bytes) {
        var split = bytes.split('=')
        var name = split.shift().replace(/\+/g, ' ')
        var value = split.join('=').replace(/\+/g, ' ')
        form.append(decodeURIComponent(name), decodeURIComponent(value))
      }
    })
    return form
  }

  function headers(xhr) {
    var head = new Headers()
    var pairs = xhr.getAllResponseHeaders().trim().split('\n')
    pairs.forEach(function(header) {
      var split = header.trim().split(':')
      var key = split.shift().trim()
      var value = split.join(':').trim()
      head.append(key, value)
    })
    return head
  }

  Body.call(Request.prototype)

  function Response(bodyInit, options) {
    if (!options) {
      options = {}
    }

    this._initBody(bodyInit, options)
    this.type = 'default'
    this.status = options.status
    this.ok = this.status >= 200 && this.status < 300
    this.statusText = options.statusText
    this.headers = options.headers instanceof Headers ? options.headers : new Headers(options.headers)
    this.url = options.url || ''
  }

  Body.call(Response.prototype)

  Response.prototype.clone = function() {
    return new Response(this._bodyInit, {
      status: this.status,
      statusText: this.statusText,
      headers: new Headers(this.headers),
      url: this.url
    })
  }

  Response.error = function() {
    var response = new Response(null, {status: 0, statusText: ''})
    response.type = 'error'
    return response
  }

  var redirectStatuses = [301, 302, 303, 307, 308]

  Response.redirect = function(url, status) {
    if (redirectStatuses.indexOf(status) === -1) {
      throw new RangeError('Invalid status code')
    }

    return new Response(null, {status: status, headers: {location: url}})
  }

  self.Headers = Headers;
  self.Request = Request;
  self.Response = Response;

  self.fetch = function(input, init) {
    return new Promise(function(resolve, reject) {
      var request
      if (Request.prototype.isPrototypeOf(input) && !init) {
        request = input
      } else {
        request = new Request(input, init)
      }

      var xhr = new XMLHttpRequest()

      function responseURL() {
        if ('responseURL' in xhr) {
          return xhr.responseURL
        }

        // Avoid security warnings on getResponseHeader when not allowed by CORS
        if (/^X-Request-URL:/m.test(xhr.getAllResponseHeaders())) {
          return xhr.getResponseHeader('X-Request-URL')
        }

        return;
      }

      var __onLoadHandled = false;

      function onload() {
        if (xhr.readyState !== 4) {
          return
        }
        var status = (xhr.status === 1223) ? 204 : xhr.status
        if (status < 100 || status > 599) {
          if (__onLoadHandled) { return; } else { __onLoadHandled = true; }
          reject(new TypeError('Network request failed'))
          return
        }
        var options = {
          status: status,
          statusText: xhr.statusText,
          headers: headers(xhr),
          url: responseURL()
        }
        var body = 'response' in xhr ? xhr.response : xhr.responseText;

        if (__onLoadHandled) { return; } else { __onLoadHandled = true; }
        resolve(new Response(body, options))
      }
      xhr.onreadystatechange = onload;
      xhr.onload = onload;
      xhr.onerror = function() {
        if (__onLoadHandled) { return; } else { __onLoadHandled = true; }
        reject(new TypeError('Network request failed'))
      }

      xhr.open(request.method, request.url, true)

      // `withCredentials` should be setted after calling `.open` in IE10
      // http://stackoverflow.com/a/19667959/1219343
      try {
        if (request.credentials === 'include') {
          if ('withCredentials' in xhr) {
            xhr.withCredentials = true;
          } else {
            console && console.warn && console.warn('withCredentials is not supported, you can ignore this warning');
          }
        }
      } catch (e) {
        console && console.warn && console.warn('set withCredentials error:' + e);
      }

      if ('responseType' in xhr && support.blob) {
        xhr.responseType = 'blob'
      }

      request.headers.forEach(function(value, name) {
        xhr.setRequestHeader(name, value)
      })

      xhr.send(typeof request._bodyInit === 'undefined' ? null : request._bodyInit)
    })
  }
  self.fetch.polyfill = true

  // Support CommonJS
  if ( true && module.exports) {
    module.exports = self.fetch;
  }
})(typeof self !== 'undefined' ? self : this);


/***/ }),

/***/ 144:
/***/ (function(module, exports) {

var __WEBPACK_AMD_DEFINE_FACTORY__, __WEBPACK_AMD_DEFINE_ARRAY__, __WEBPACK_AMD_DEFINE_RESULT__;(function (global, factory) {
  if (true) {
    !(__WEBPACK_AMD_DEFINE_ARRAY__ = [exports, module], __WEBPACK_AMD_DEFINE_FACTORY__ = (factory),
		__WEBPACK_AMD_DEFINE_RESULT__ = (typeof __WEBPACK_AMD_DEFINE_FACTORY__ === 'function' ?
		(__WEBPACK_AMD_DEFINE_FACTORY__.apply(exports, __WEBPACK_AMD_DEFINE_ARRAY__)) : __WEBPACK_AMD_DEFINE_FACTORY__),
		__WEBPACK_AMD_DEFINE_RESULT__ !== undefined && (module.exports = __WEBPACK_AMD_DEFINE_RESULT__));
  } else { var mod; }
})(this, function (exports, module) {
  'use strict';

  var defaultOptions = {
    timeout: 5000,
    jsonpCallback: 'callback',
    jsonpCallbackFunction: null
  };

  function generateCallbackFunction() {
    return 'jsonp_' + Date.now() + '_' + Math.ceil(Math.random() * 100000);
  }

  function clearFunction(functionName) {
    // IE8 throws an exception when you try to delete a property on window
    // http://stackoverflow.com/a/1824228/751089
    try {
      delete window[functionName];
    } catch (e) {
      window[functionName] = undefined;
    }
  }

  function removeScript(scriptId) {
    var script = document.getElementById(scriptId);
    if (script) {
      document.getElementsByTagName('head')[0].removeChild(script);
    }
  }

  function fetchJsonp(_url) {
    var options = arguments.length <= 1 || arguments[1] === undefined ? {} : arguments[1];

    // to avoid param reassign
    var url = _url;
    var timeout = options.timeout || defaultOptions.timeout;
    var jsonpCallback = options.jsonpCallback || defaultOptions.jsonpCallback;

    var timeoutId = undefined;

    return new Promise(function (resolve, reject) {
      var callbackFunction = options.jsonpCallbackFunction || generateCallbackFunction();
      var scriptId = jsonpCallback + '_' + callbackFunction;

      window[callbackFunction] = function (response) {
        resolve({
          ok: true,
          // keep consistent with fetch API
          json: function json() {
            return Promise.resolve(response);
          }
        });

        if (timeoutId) clearTimeout(timeoutId);

        removeScript(scriptId);

        clearFunction(callbackFunction);
      };

      // Check if the user set their own params, and if not add a ? to start a list of params
      url += url.indexOf('?') === -1 ? '?' : '&';

      var jsonpScript = document.createElement('script');
      jsonpScript.setAttribute('src', '' + url + jsonpCallback + '=' + callbackFunction);
      if (options.charset) {
        jsonpScript.setAttribute('charset', options.charset);
      }
      jsonpScript.id = scriptId;
      document.getElementsByTagName('head')[0].appendChild(jsonpScript);

      timeoutId = setTimeout(function () {
        reject(new Error('JSONP request to ' + _url + ' timed out'));

        clearFunction(callbackFunction);
        removeScript(scriptId);
        window[callbackFunction] = function () {
          clearFunction(callbackFunction);
        };
      }, timeout);

      // Caught if got 404/500
      jsonpScript.onerror = function () {
        reject(new Error('JSONP request to ' + _url + ' failed'));

        clearFunction(callbackFunction);
        removeScript(scriptId);
        if (timeoutId) clearTimeout(timeoutId);
      };
    });
  }

  // export as global function
  /*
  let local;
  if (typeof global !== 'undefined') {
    local = global;
  } else if (typeof self !== 'undefined') {
    local = self;
  } else {
    try {
      local = Function('return this')();
    } catch (e) {
      throw new Error('polyfill failed because global object is unavailable in this environment');
    }
  }
  local.fetchJsonp = fetchJsonp;
  */

  module.exports = fetchJsonp;
});

/***/ }),

/***/ 645:
/***/ ((__unused_webpack_module, exports) => {

/*! ieee754. BSD-3-Clause License. Feross Aboukhadijeh <https://feross.org/opensource> */
exports.read = function (buffer, offset, isLE, mLen, nBytes) {
  var e, m
  var eLen = (nBytes * 8) - mLen - 1
  var eMax = (1 << eLen) - 1
  var eBias = eMax >> 1
  var nBits = -7
  var i = isLE ? (nBytes - 1) : 0
  var d = isLE ? -1 : 1
  var s = buffer[offset + i]

  i += d

  e = s & ((1 << (-nBits)) - 1)
  s >>= (-nBits)
  nBits += eLen
  for (; nBits > 0; e = (e * 256) + buffer[offset + i], i += d, nBits -= 8) {}

  m = e & ((1 << (-nBits)) - 1)
  e >>= (-nBits)
  nBits += mLen
  for (; nBits > 0; m = (m * 256) + buffer[offset + i], i += d, nBits -= 8) {}

  if (e === 0) {
    e = 1 - eBias
  } else if (e === eMax) {
    return m ? NaN : ((s ? -1 : 1) * Infinity)
  } else {
    m = m + Math.pow(2, mLen)
    e = e - eBias
  }
  return (s ? -1 : 1) * m * Math.pow(2, e - mLen)
}

exports.write = function (buffer, value, offset, isLE, mLen, nBytes) {
  var e, m, c
  var eLen = (nBytes * 8) - mLen - 1
  var eMax = (1 << eLen) - 1
  var eBias = eMax >> 1
  var rt = (mLen === 23 ? Math.pow(2, -24) - Math.pow(2, -77) : 0)
  var i = isLE ? 0 : (nBytes - 1)
  var d = isLE ? 1 : -1
  var s = value < 0 || (value === 0 && 1 / value < 0) ? 1 : 0

  value = Math.abs(value)

  if (isNaN(value) || value === Infinity) {
    m = isNaN(value) ? 1 : 0
    e = eMax
  } else {
    e = Math.floor(Math.log(value) / Math.LN2)
    if (value * (c = Math.pow(2, -e)) < 1) {
      e--
      c *= 2
    }
    if (e + eBias >= 1) {
      value += rt / c
    } else {
      value += rt * Math.pow(2, 1 - eBias)
    }
    if (value * c >= 2) {
      e++
      c /= 2
    }

    if (e + eBias >= eMax) {
      m = 0
      e = eMax
    } else if (e + eBias >= 1) {
      m = ((value * c) - 1) * Math.pow(2, mLen)
      e = e + eBias
    } else {
      m = value * Math.pow(2, eBias - 1) * Math.pow(2, mLen)
      e = 0
    }
  }

  for (; mLen >= 8; buffer[offset + i] = m & 0xff, i += d, m /= 256, mLen -= 8) {}

  e = (e << mLen) | m
  eLen += mLen
  for (; eLen > 0; buffer[offset + i] = e & 0xff, i += d, e /= 256, eLen -= 8) {}

  buffer[offset + i - d] |= s * 128
}


/***/ }),

/***/ 186:
/***/ ((module) => {

var containers = []; // will store container HTMLElement references
var styleElements = []; // will store {prepend: HTMLElement, append: HTMLElement}

var usage = 'insert-css: You need to provide a CSS string. Usage: insertCss(cssString[, options]).';

function insertCss(css, options) {
    options = options || {};

    if (css === undefined) {
        throw new Error(usage);
    }

    var position = options.prepend === true ? 'prepend' : 'append';
    var container = options.container !== undefined ? options.container : document.querySelector('head');
    var containerId = containers.indexOf(container);

    // first time we see this container, create the necessary entries
    if (containerId === -1) {
        containerId = containers.push(container) - 1;
        styleElements[containerId] = {};
    }

    // try to get the correponding container + position styleElement, create it otherwise
    var styleElement;

    if (styleElements[containerId] !== undefined && styleElements[containerId][position] !== undefined) {
        styleElement = styleElements[containerId][position];
    } else {
        styleElement = styleElements[containerId][position] = createStyleElement();

        if (position === 'prepend') {
            container.insertBefore(styleElement, container.childNodes[0]);
        } else {
            container.appendChild(styleElement);
        }
    }

    // strip potential UTF-8 BOM if css was read from a file
    if (css.charCodeAt(0) === 0xFEFF) { css = css.substr(1, css.length); }

    // actually add the stylesheet
    if (styleElement.styleSheet) {
        styleElement.styleSheet.cssText += css
    } else {
        styleElement.textContent += css;
    }

    return styleElement;
};

function createStyleElement() {
    var styleElement = document.createElement('style');
    styleElement.setAttribute('type', 'text/css');
    return styleElement;
}

module.exports = insertCss;
module.exports.insertCss = insertCss;


/***/ }),

/***/ 545:
/***/ ((module, __unused_webpack_exports, __webpack_require__) => {

var utils = __webpack_require__(820)

var AND = '&&'
  , OR = '||'
  , AND_STR = 'and'
  , OR_STR = 'or'
  , NOT = '!'
  , EQUAL = '='
  , LIKE = '~'
  , NOTEQUAL = NOT + EQUAL
  , NOTLIKE = NOT + LIKE
  , GT = '>'
  , GE = '>='
  , LT = '<'
  , LE = '<='
  , WILDCARD = '*'
  , COMMA = ','
  , DELIMITER = '.'
  , LEFT = '('
  , RIGHT = ')'
  , WHERE = 'where'
  , synopsis = {
  pathway: [],
  groups: {}
}
  , AST = {}
  , options = {};

var print = console.log;
// ------------------ splitter -------------------- //

function Tokenize(query) {
  var parts = __splitTrim(query, WHERE);
  var pathway = parts[0];
  var where = parts[1];

  synopsis.pathway = __splitTrim(pathway, COMMA);
  for (var i = 0, len = synopsis.pathway.length; i < len; i++) {
    synopsis.pathway[i] = __splitTrim(synopsis.pathway[i], DELIMITER);
    if (synopsis.pathway[i][0] == WILDCARD)
      synopsis.pathway[i].shift();
    if (synopsis.pathway[i].length === 0)
      synopsis.pathway.splice(i, 1);
  }

  var lastLeft = -1,
    lastRight = -1,
    current = 0;
  while (current < where.length) {
    if (where[current] === LEFT) {
      lastLeft = current;
    } else if (where[current] === RIGHT) {
      lastRight = current;
      if (lastRight > lastLeft && lastLeft !== -1) {
        var k = 'gr' + '_' + new Date().getTime();
        synopsis.groups[k] = where.substring(lastLeft + 1, lastRight);
        where = where.replace(LEFT + synopsis.groups[k] + RIGHT, k);
        current = -1;
      }
    }
    current += 1;
  }
  LogicalGrouping(AST, where);
}
function LogicalGrouping(current, where) {
  var lastAnd = __findIndex(where, AND),
    lastOr = __findIndex(where, OR);

  if (lastAnd !== Number.MAX_VALUE || lastOr !== Number.MAX_VALUE) {
    if (lastAnd < lastOr) {
      current.and = current.and || [];
      var parts = __splitTrim(where, AND);
      current.and.push(parts[0]);
      LogicalGrouping(current.and, parts[1]);
    } else {
      current.or = current.or || [];
      var parts = __splitTrim(where, OR);
      current.or.push(parts[0]);
      LogicalGrouping(current.or, parts[1]);
    }
  } else {
    if (synopsis.groups[where]) {
      where = synopsis.groups[where];
      LogicalGrouping(current, where);
    } else {
      if (Array.isArray(current))
        current.push(where);
      else
        current.or = [where];
      ExtractExpression(AST.or ? AST.or : AST.and)
    }
  }
}
function ExtractExpression(logicalGroup) {
  for (var k in logicalGroup) {
    if (logicalGroup.hasOwnProperty(k)) {
      if (Array.isArray(logicalGroup[k])) {
        ExtractExpression(logicalGroup[k]);
      }
      else if (typeof logicalGroup[k] === 'string') {
        if (__contains(logicalGroup[k], NOTEQUAL)) {
          var parts = __splitTrim(logicalGroup[k], NOTEQUAL);
          logicalGroup[k] = {
            ne: [
              parts[0],
              parts[1]
            ]
          };
        } else if (__contains(logicalGroup[k], NOTLIKE)) {
          var parts = __splitTrim(logicalGroup[k], NOTLIKE);
          logicalGroup[k] = {
            nreq: [
              parts[0],
              parts[1]
            ]
          };
        } else if (__contains(logicalGroup[k], LIKE)) {
          var parts = __splitTrim(logicalGroup[k], LIKE);
          logicalGroup[k] = { // rough eq
            req: [
              parts[0],
              parts[1]
            ]
          };
        } else if (__contains(logicalGroup[k], GE)) {
          var parts = __splitTrim(logicalGroup[k], GE);
          logicalGroup[k] = { // greater than or equal
            ge: [
              parts[0],
              parts[1]
            ]
          };
        } else if (__contains(logicalGroup[k], GT)) {
          var parts = __splitTrim(logicalGroup[k], GT);
          logicalGroup[k] = { // greater than
            gt: [
              parts[0],
              parts[1]
            ]
          };
        } else if (__contains(logicalGroup[k], LE)) {
          var parts = __splitTrim(logicalGroup[k], LE);
          logicalGroup[k] = { // less than or equal
            le: [
              parts[0],
              parts[1]
            ]
          };
        } else if (__contains(logicalGroup[k], LT)) {
          var parts = __splitTrim(logicalGroup[k], LT);
          logicalGroup[k] = { // less than
            lt: [
              parts[0],
              parts[1]
            ]
          };
        } else if (__contains(logicalGroup[k], EQUAL)) {
          var parts = __splitTrim(logicalGroup[k], EQUAL);
          logicalGroup[k] = {
            eq: [
              parts[0],
              parts[1]
            ]
          };
        }
      }
    }
  }
}

function __findIndex(str, token) {
  var index = str.indexOf(token);
  return index === -1 ? Number.MAX_VALUE : index;
}
function __splitTrim(str, token) {
  return str.split(token).map(function (p) {
    return p.trim();
  });
}
function __contains(a, b) {
  return a.indexOf(b) > -1;
}
function __hierarchize(obj, dottedPath) {
  var parts = __splitTrim(dottedPath, DELIMITER);
  var res = obj;
  for (var p in parts) {
    if (res.hasOwnProperty(parts[p]))
      res = res[parts[p]];
    else
      return '';
  }
  // support comparison for Date/DateString
  if(utils.isDate(res)) res = res.valueOf()
  else if(utils.isDateString(res)) res = utils.parseDateFromString(res)
  else res = res.toString()

  return res
}

function FilterOR(ASTNode, row) {
  var res = false;
  for (var k in ASTNode) {
    var filterFunc = (k === AND_STR ? FilterAND : (k === OR_STR ? FilterOR : Filter));
    res = res || filterFunc(ASTNode[k], row);
    if (options.trace)
      print(synopsis.step, '======((( or', ASTNode[k], res);
    if (res) return res;
  }
  return res;
}
function FilterAND(ASTNode, row) {
  var res = true;
  for (var k in ASTNode) {
    var filterFunc = (k === AND_STR ? FilterAND : (k === OR_STR ? FilterOR : Filter));
    res = res && filterFunc(ASTNode[k], row);
    if (options.trace)
      print(synopsis.step, '======((( and', ASTNode[k], res);
    if (!res) return res;
  }
  return res;
}
function Filter(ASTNode, row) {
  synopsis.step += 1;
  if (ASTNode.or) {
    var res = FilterOR(ASTNode.or, row);
    if (options.trace)
      print(synopsis.step, 'OR', ASTNode, res);
    return res;
  } else if (ASTNode.and) {
    var res = FilterAND(ASTNode.and, row);
    if (options.trace)
      print(synopsis.step, 'AND', ASTNode, res);
    return res;
  } else if (typeof ASTNode === 'object') {
    if (ASTNode.eq) { // =
      return __hierarchize(row, ASTNode.eq[0]) === ASTNode.eq[1];
    } else if (ASTNode.ne) { // !=
      return __hierarchize(row, ASTNode.ne[0]) !== ASTNode.ne[1];
    } else if (ASTNode.req) { // ~
      return __contains(__hierarchize(row, ASTNode.req[0]), ASTNode.req[1]);
    } else if (ASTNode.nreq) { // ~
      return !__contains(__hierarchize(row, ASTNode.nreq[0]), ASTNode.nreq[1]);
    } else if (ASTNode.gt) { // >
      return __hierarchize(row, ASTNode.gt[0]) > ASTNode.gt[1];
    } else if (ASTNode.ge) { // >=
      return __hierarchize(row, ASTNode.ge[0]) >= ASTNode.ge[1];
    } else if (ASTNode.lt) { // <
      return __hierarchize(row, ASTNode.lt[0]) < ASTNode.lt[1];
    } else if (ASTNode.le) { // <=
      return __hierarchize(row, ASTNode.le[0]) <= ASTNode.le[1];
    } else {
      return Filter(ASTNode, row);
    }
  }
}
function Parse(dataSource) {
  var result = [];
  for (var k in dataSource)
    if (Filter(AST, dataSource[k]))
      result.push(dataSource[k]);
  return result;
}
function Fields(result) {
  if (result && synopsis.pathway.length > 0) {
    //print(synopsis.pathway);
    return result.map(function (ele) {
      var res = {};
      for (var i = 0, len = synopsis.pathway.length; i < len; i++) {
        var key = synopsis.pathway[i].join(DELIMITER);
        res[key] = __hierarchize(ele, key);
      }
      return res;
    });
  }
  return result;
}
function Query(dataSource, query, opts) {
  synopsis = {
    pathway: [],
    groups: {},
    step: 0
  };
  AST = {};
  opts = opts || {
      trace: false
    };
  options = opts;
  Tokenize(query);
  return Fields(Parse(dataSource));
}

if ( true && typeof(module.exports) != 'undefined') module.exports = Query;
if (typeof(window) != 'undefined') window.Query = Query;

/***/ }),

/***/ 820:
/***/ ((module) => {

function getObjectType(obj) {
  return Object.prototype.toString.call(obj);
}
function isDate(obj) {
  return getObjectType(obj) === '[object Date]';
}
function isString(obj) {
  return getObjectType(obj) === '[object String]';
}
function isDateString(obj) {
  return isString(obj) && !isNaN(Date.parse(obj))
}
function isNumber(obj) {
  return typeof obj === 'number'
}
function parseDateFromString(str) {
  return Date.parse(str)
}
module.exports = {
  getObjectType: getObjectType,
  isDate: isDate,
  isString: isString,
  isDateString: isDateString,
  parseDateFromString: parseDateFromString,
  isNumber: isNumber
}


/***/ }),

/***/ 465:
/***/ ((module, exports, __webpack_require__) => {

/* module decorator */ module = __webpack_require__.nmd(module);
/**
 * lodash (Custom Build) <https://lodash.com/>
 * Build: `lodash modularize exports="npm" -o ./`
 * Copyright jQuery Foundation and other contributors <https://jquery.org/>
 * Released under MIT license <https://lodash.com/license>
 * Based on Underscore.js 1.8.3 <http://underscorejs.org/LICENSE>
 * Copyright Jeremy Ashkenas, DocumentCloud and Investigative Reporters & Editors
 */

/** Used as the size to enable large array optimizations. */
var LARGE_ARRAY_SIZE = 200;

/** Used to stand-in for `undefined` hash values. */
var HASH_UNDEFINED = '__lodash_hash_undefined__';

/** Used as references for various `Number` constants. */
var MAX_SAFE_INTEGER = 9007199254740991;

/** `Object#toString` result references. */
var argsTag = '[object Arguments]',
    arrayTag = '[object Array]',
    boolTag = '[object Boolean]',
    dateTag = '[object Date]',
    errorTag = '[object Error]',
    funcTag = '[object Function]',
    genTag = '[object GeneratorFunction]',
    mapTag = '[object Map]',
    numberTag = '[object Number]',
    objectTag = '[object Object]',
    promiseTag = '[object Promise]',
    regexpTag = '[object RegExp]',
    setTag = '[object Set]',
    stringTag = '[object String]',
    symbolTag = '[object Symbol]',
    weakMapTag = '[object WeakMap]';

var arrayBufferTag = '[object ArrayBuffer]',
    dataViewTag = '[object DataView]',
    float32Tag = '[object Float32Array]',
    float64Tag = '[object Float64Array]',
    int8Tag = '[object Int8Array]',
    int16Tag = '[object Int16Array]',
    int32Tag = '[object Int32Array]',
    uint8Tag = '[object Uint8Array]',
    uint8ClampedTag = '[object Uint8ClampedArray]',
    uint16Tag = '[object Uint16Array]',
    uint32Tag = '[object Uint32Array]';

/**
 * Used to match `RegExp`
 * [syntax characters](http://ecma-international.org/ecma-262/7.0/#sec-patterns).
 */
var reRegExpChar = /[\\^$.*+?()[\]{}|]/g;

/** Used to match `RegExp` flags from their coerced string values. */
var reFlags = /\w*$/;

/** Used to detect host constructors (Safari). */
var reIsHostCtor = /^\[object .+?Constructor\]$/;

/** Used to detect unsigned integer values. */
var reIsUint = /^(?:0|[1-9]\d*)$/;

/** Used to identify `toStringTag` values supported by `_.clone`. */
var cloneableTags = {};
cloneableTags[argsTag] = cloneableTags[arrayTag] =
cloneableTags[arrayBufferTag] = cloneableTags[dataViewTag] =
cloneableTags[boolTag] = cloneableTags[dateTag] =
cloneableTags[float32Tag] = cloneableTags[float64Tag] =
cloneableTags[int8Tag] = cloneableTags[int16Tag] =
cloneableTags[int32Tag] = cloneableTags[mapTag] =
cloneableTags[numberTag] = cloneableTags[objectTag] =
cloneableTags[regexpTag] = cloneableTags[setTag] =
cloneableTags[stringTag] = cloneableTags[symbolTag] =
cloneableTags[uint8Tag] = cloneableTags[uint8ClampedTag] =
cloneableTags[uint16Tag] = cloneableTags[uint32Tag] = true;
cloneableTags[errorTag] = cloneableTags[funcTag] =
cloneableTags[weakMapTag] = false;

/** Detect free variable `global` from Node.js. */
var freeGlobal = typeof __webpack_require__.g == 'object' && __webpack_require__.g && __webpack_require__.g.Object === Object && __webpack_require__.g;

/** Detect free variable `self`. */
var freeSelf = typeof self == 'object' && self && self.Object === Object && self;

/** Used as a reference to the global object. */
var root = freeGlobal || freeSelf || Function('return this')();

/** Detect free variable `exports`. */
var freeExports =  true && exports && !exports.nodeType && exports;

/** Detect free variable `module`. */
var freeModule = freeExports && "object" == 'object' && module && !module.nodeType && module;

/** Detect the popular CommonJS extension `module.exports`. */
var moduleExports = freeModule && freeModule.exports === freeExports;

/**
 * Adds the key-value `pair` to `map`.
 *
 * @private
 * @param {Object} map The map to modify.
 * @param {Array} pair The key-value pair to add.
 * @returns {Object} Returns `map`.
 */
function addMapEntry(map, pair) {
  // Don't return `map.set` because it's not chainable in IE 11.
  map.set(pair[0], pair[1]);
  return map;
}

/**
 * Adds `value` to `set`.
 *
 * @private
 * @param {Object} set The set to modify.
 * @param {*} value The value to add.
 * @returns {Object} Returns `set`.
 */
function addSetEntry(set, value) {
  // Don't return `set.add` because it's not chainable in IE 11.
  set.add(value);
  return set;
}

/**
 * A specialized version of `_.forEach` for arrays without support for
 * iteratee shorthands.
 *
 * @private
 * @param {Array} [array] The array to iterate over.
 * @param {Function} iteratee The function invoked per iteration.
 * @returns {Array} Returns `array`.
 */
function arrayEach(array, iteratee) {
  var index = -1,
      length = array ? array.length : 0;

  while (++index < length) {
    if (iteratee(array[index], index, array) === false) {
      break;
    }
  }
  return array;
}

/**
 * Appends the elements of `values` to `array`.
 *
 * @private
 * @param {Array} array The array to modify.
 * @param {Array} values The values to append.
 * @returns {Array} Returns `array`.
 */
function arrayPush(array, values) {
  var index = -1,
      length = values.length,
      offset = array.length;

  while (++index < length) {
    array[offset + index] = values[index];
  }
  return array;
}

/**
 * A specialized version of `_.reduce` for arrays without support for
 * iteratee shorthands.
 *
 * @private
 * @param {Array} [array] The array to iterate over.
 * @param {Function} iteratee The function invoked per iteration.
 * @param {*} [accumulator] The initial value.
 * @param {boolean} [initAccum] Specify using the first element of `array` as
 *  the initial value.
 * @returns {*} Returns the accumulated value.
 */
function arrayReduce(array, iteratee, accumulator, initAccum) {
  var index = -1,
      length = array ? array.length : 0;

  if (initAccum && length) {
    accumulator = array[++index];
  }
  while (++index < length) {
    accumulator = iteratee(accumulator, array[index], index, array);
  }
  return accumulator;
}

/**
 * The base implementation of `_.times` without support for iteratee shorthands
 * or max array length checks.
 *
 * @private
 * @param {number} n The number of times to invoke `iteratee`.
 * @param {Function} iteratee The function invoked per iteration.
 * @returns {Array} Returns the array of results.
 */
function baseTimes(n, iteratee) {
  var index = -1,
      result = Array(n);

  while (++index < n) {
    result[index] = iteratee(index);
  }
  return result;
}

/**
 * Gets the value at `key` of `object`.
 *
 * @private
 * @param {Object} [object] The object to query.
 * @param {string} key The key of the property to get.
 * @returns {*} Returns the property value.
 */
function getValue(object, key) {
  return object == null ? undefined : object[key];
}

/**
 * Checks if `value` is a host object in IE < 9.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a host object, else `false`.
 */
function isHostObject(value) {
  // Many host objects are `Object` objects that can coerce to strings
  // despite having improperly defined `toString` methods.
  var result = false;
  if (value != null && typeof value.toString != 'function') {
    try {
      result = !!(value + '');
    } catch (e) {}
  }
  return result;
}

/**
 * Converts `map` to its key-value pairs.
 *
 * @private
 * @param {Object} map The map to convert.
 * @returns {Array} Returns the key-value pairs.
 */
function mapToArray(map) {
  var index = -1,
      result = Array(map.size);

  map.forEach(function(value, key) {
    result[++index] = [key, value];
  });
  return result;
}

/**
 * Creates a unary function that invokes `func` with its argument transformed.
 *
 * @private
 * @param {Function} func The function to wrap.
 * @param {Function} transform The argument transform.
 * @returns {Function} Returns the new function.
 */
function overArg(func, transform) {
  return function(arg) {
    return func(transform(arg));
  };
}

/**
 * Converts `set` to an array of its values.
 *
 * @private
 * @param {Object} set The set to convert.
 * @returns {Array} Returns the values.
 */
function setToArray(set) {
  var index = -1,
      result = Array(set.size);

  set.forEach(function(value) {
    result[++index] = value;
  });
  return result;
}

/** Used for built-in method references. */
var arrayProto = Array.prototype,
    funcProto = Function.prototype,
    objectProto = Object.prototype;

/** Used to detect overreaching core-js shims. */
var coreJsData = root['__core-js_shared__'];

/** Used to detect methods masquerading as native. */
var maskSrcKey = (function() {
  var uid = /[^.]+$/.exec(coreJsData && coreJsData.keys && coreJsData.keys.IE_PROTO || '');
  return uid ? ('Symbol(src)_1.' + uid) : '';
}());

/** Used to resolve the decompiled source of functions. */
var funcToString = funcProto.toString;

/** Used to check objects for own properties. */
var hasOwnProperty = objectProto.hasOwnProperty;

/**
 * Used to resolve the
 * [`toStringTag`](http://ecma-international.org/ecma-262/7.0/#sec-object.prototype.tostring)
 * of values.
 */
var objectToString = objectProto.toString;

/** Used to detect if a method is native. */
var reIsNative = RegExp('^' +
  funcToString.call(hasOwnProperty).replace(reRegExpChar, '\\$&')
  .replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g, '$1.*?') + '$'
);

/** Built-in value references. */
var Buffer = moduleExports ? root.Buffer : undefined,
    Symbol = root.Symbol,
    Uint8Array = root.Uint8Array,
    getPrototype = overArg(Object.getPrototypeOf, Object),
    objectCreate = Object.create,
    propertyIsEnumerable = objectProto.propertyIsEnumerable,
    splice = arrayProto.splice;

/* Built-in method references for those with the same name as other `lodash` methods. */
var nativeGetSymbols = Object.getOwnPropertySymbols,
    nativeIsBuffer = Buffer ? Buffer.isBuffer : undefined,
    nativeKeys = overArg(Object.keys, Object);

/* Built-in method references that are verified to be native. */
var DataView = getNative(root, 'DataView'),
    Map = getNative(root, 'Map'),
    Promise = getNative(root, 'Promise'),
    Set = getNative(root, 'Set'),
    WeakMap = getNative(root, 'WeakMap'),
    nativeCreate = getNative(Object, 'create');

/** Used to detect maps, sets, and weakmaps. */
var dataViewCtorString = toSource(DataView),
    mapCtorString = toSource(Map),
    promiseCtorString = toSource(Promise),
    setCtorString = toSource(Set),
    weakMapCtorString = toSource(WeakMap);

/** Used to convert symbols to primitives and strings. */
var symbolProto = Symbol ? Symbol.prototype : undefined,
    symbolValueOf = symbolProto ? symbolProto.valueOf : undefined;

/**
 * Creates a hash object.
 *
 * @private
 * @constructor
 * @param {Array} [entries] The key-value pairs to cache.
 */
function Hash(entries) {
  var index = -1,
      length = entries ? entries.length : 0;

  this.clear();
  while (++index < length) {
    var entry = entries[index];
    this.set(entry[0], entry[1]);
  }
}

/**
 * Removes all key-value entries from the hash.
 *
 * @private
 * @name clear
 * @memberOf Hash
 */
function hashClear() {
  this.__data__ = nativeCreate ? nativeCreate(null) : {};
}

/**
 * Removes `key` and its value from the hash.
 *
 * @private
 * @name delete
 * @memberOf Hash
 * @param {Object} hash The hash to modify.
 * @param {string} key The key of the value to remove.
 * @returns {boolean} Returns `true` if the entry was removed, else `false`.
 */
function hashDelete(key) {
  return this.has(key) && delete this.__data__[key];
}

/**
 * Gets the hash value for `key`.
 *
 * @private
 * @name get
 * @memberOf Hash
 * @param {string} key The key of the value to get.
 * @returns {*} Returns the entry value.
 */
function hashGet(key) {
  var data = this.__data__;
  if (nativeCreate) {
    var result = data[key];
    return result === HASH_UNDEFINED ? undefined : result;
  }
  return hasOwnProperty.call(data, key) ? data[key] : undefined;
}

/**
 * Checks if a hash value for `key` exists.
 *
 * @private
 * @name has
 * @memberOf Hash
 * @param {string} key The key of the entry to check.
 * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
 */
function hashHas(key) {
  var data = this.__data__;
  return nativeCreate ? data[key] !== undefined : hasOwnProperty.call(data, key);
}

/**
 * Sets the hash `key` to `value`.
 *
 * @private
 * @name set
 * @memberOf Hash
 * @param {string} key The key of the value to set.
 * @param {*} value The value to set.
 * @returns {Object} Returns the hash instance.
 */
function hashSet(key, value) {
  var data = this.__data__;
  data[key] = (nativeCreate && value === undefined) ? HASH_UNDEFINED : value;
  return this;
}

// Add methods to `Hash`.
Hash.prototype.clear = hashClear;
Hash.prototype['delete'] = hashDelete;
Hash.prototype.get = hashGet;
Hash.prototype.has = hashHas;
Hash.prototype.set = hashSet;

/**
 * Creates an list cache object.
 *
 * @private
 * @constructor
 * @param {Array} [entries] The key-value pairs to cache.
 */
function ListCache(entries) {
  var index = -1,
      length = entries ? entries.length : 0;

  this.clear();
  while (++index < length) {
    var entry = entries[index];
    this.set(entry[0], entry[1]);
  }
}

/**
 * Removes all key-value entries from the list cache.
 *
 * @private
 * @name clear
 * @memberOf ListCache
 */
function listCacheClear() {
  this.__data__ = [];
}

/**
 * Removes `key` and its value from the list cache.
 *
 * @private
 * @name delete
 * @memberOf ListCache
 * @param {string} key The key of the value to remove.
 * @returns {boolean} Returns `true` if the entry was removed, else `false`.
 */
function listCacheDelete(key) {
  var data = this.__data__,
      index = assocIndexOf(data, key);

  if (index < 0) {
    return false;
  }
  var lastIndex = data.length - 1;
  if (index == lastIndex) {
    data.pop();
  } else {
    splice.call(data, index, 1);
  }
  return true;
}

/**
 * Gets the list cache value for `key`.
 *
 * @private
 * @name get
 * @memberOf ListCache
 * @param {string} key The key of the value to get.
 * @returns {*} Returns the entry value.
 */
function listCacheGet(key) {
  var data = this.__data__,
      index = assocIndexOf(data, key);

  return index < 0 ? undefined : data[index][1];
}

/**
 * Checks if a list cache value for `key` exists.
 *
 * @private
 * @name has
 * @memberOf ListCache
 * @param {string} key The key of the entry to check.
 * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
 */
function listCacheHas(key) {
  return assocIndexOf(this.__data__, key) > -1;
}

/**
 * Sets the list cache `key` to `value`.
 *
 * @private
 * @name set
 * @memberOf ListCache
 * @param {string} key The key of the value to set.
 * @param {*} value The value to set.
 * @returns {Object} Returns the list cache instance.
 */
function listCacheSet(key, value) {
  var data = this.__data__,
      index = assocIndexOf(data, key);

  if (index < 0) {
    data.push([key, value]);
  } else {
    data[index][1] = value;
  }
  return this;
}

// Add methods to `ListCache`.
ListCache.prototype.clear = listCacheClear;
ListCache.prototype['delete'] = listCacheDelete;
ListCache.prototype.get = listCacheGet;
ListCache.prototype.has = listCacheHas;
ListCache.prototype.set = listCacheSet;

/**
 * Creates a map cache object to store key-value pairs.
 *
 * @private
 * @constructor
 * @param {Array} [entries] The key-value pairs to cache.
 */
function MapCache(entries) {
  var index = -1,
      length = entries ? entries.length : 0;

  this.clear();
  while (++index < length) {
    var entry = entries[index];
    this.set(entry[0], entry[1]);
  }
}

/**
 * Removes all key-value entries from the map.
 *
 * @private
 * @name clear
 * @memberOf MapCache
 */
function mapCacheClear() {
  this.__data__ = {
    'hash': new Hash,
    'map': new (Map || ListCache),
    'string': new Hash
  };
}

/**
 * Removes `key` and its value from the map.
 *
 * @private
 * @name delete
 * @memberOf MapCache
 * @param {string} key The key of the value to remove.
 * @returns {boolean} Returns `true` if the entry was removed, else `false`.
 */
function mapCacheDelete(key) {
  return getMapData(this, key)['delete'](key);
}

/**
 * Gets the map value for `key`.
 *
 * @private
 * @name get
 * @memberOf MapCache
 * @param {string} key The key of the value to get.
 * @returns {*} Returns the entry value.
 */
function mapCacheGet(key) {
  return getMapData(this, key).get(key);
}

/**
 * Checks if a map value for `key` exists.
 *
 * @private
 * @name has
 * @memberOf MapCache
 * @param {string} key The key of the entry to check.
 * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
 */
function mapCacheHas(key) {
  return getMapData(this, key).has(key);
}

/**
 * Sets the map `key` to `value`.
 *
 * @private
 * @name set
 * @memberOf MapCache
 * @param {string} key The key of the value to set.
 * @param {*} value The value to set.
 * @returns {Object} Returns the map cache instance.
 */
function mapCacheSet(key, value) {
  getMapData(this, key).set(key, value);
  return this;
}

// Add methods to `MapCache`.
MapCache.prototype.clear = mapCacheClear;
MapCache.prototype['delete'] = mapCacheDelete;
MapCache.prototype.get = mapCacheGet;
MapCache.prototype.has = mapCacheHas;
MapCache.prototype.set = mapCacheSet;

/**
 * Creates a stack cache object to store key-value pairs.
 *
 * @private
 * @constructor
 * @param {Array} [entries] The key-value pairs to cache.
 */
function Stack(entries) {
  this.__data__ = new ListCache(entries);
}

/**
 * Removes all key-value entries from the stack.
 *
 * @private
 * @name clear
 * @memberOf Stack
 */
function stackClear() {
  this.__data__ = new ListCache;
}

/**
 * Removes `key` and its value from the stack.
 *
 * @private
 * @name delete
 * @memberOf Stack
 * @param {string} key The key of the value to remove.
 * @returns {boolean} Returns `true` if the entry was removed, else `false`.
 */
function stackDelete(key) {
  return this.__data__['delete'](key);
}

/**
 * Gets the stack value for `key`.
 *
 * @private
 * @name get
 * @memberOf Stack
 * @param {string} key The key of the value to get.
 * @returns {*} Returns the entry value.
 */
function stackGet(key) {
  return this.__data__.get(key);
}

/**
 * Checks if a stack value for `key` exists.
 *
 * @private
 * @name has
 * @memberOf Stack
 * @param {string} key The key of the entry to check.
 * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
 */
function stackHas(key) {
  return this.__data__.has(key);
}

/**
 * Sets the stack `key` to `value`.
 *
 * @private
 * @name set
 * @memberOf Stack
 * @param {string} key The key of the value to set.
 * @param {*} value The value to set.
 * @returns {Object} Returns the stack cache instance.
 */
function stackSet(key, value) {
  var cache = this.__data__;
  if (cache instanceof ListCache) {
    var pairs = cache.__data__;
    if (!Map || (pairs.length < LARGE_ARRAY_SIZE - 1)) {
      pairs.push([key, value]);
      return this;
    }
    cache = this.__data__ = new MapCache(pairs);
  }
  cache.set(key, value);
  return this;
}

// Add methods to `Stack`.
Stack.prototype.clear = stackClear;
Stack.prototype['delete'] = stackDelete;
Stack.prototype.get = stackGet;
Stack.prototype.has = stackHas;
Stack.prototype.set = stackSet;

/**
 * Creates an array of the enumerable property names of the array-like `value`.
 *
 * @private
 * @param {*} value The value to query.
 * @param {boolean} inherited Specify returning inherited property names.
 * @returns {Array} Returns the array of property names.
 */
function arrayLikeKeys(value, inherited) {
  // Safari 8.1 makes `arguments.callee` enumerable in strict mode.
  // Safari 9 makes `arguments.length` enumerable in strict mode.
  var result = (isArray(value) || isArguments(value))
    ? baseTimes(value.length, String)
    : [];

  var length = result.length,
      skipIndexes = !!length;

  for (var key in value) {
    if ((inherited || hasOwnProperty.call(value, key)) &&
        !(skipIndexes && (key == 'length' || isIndex(key, length)))) {
      result.push(key);
    }
  }
  return result;
}

/**
 * Assigns `value` to `key` of `object` if the existing value is not equivalent
 * using [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
 * for equality comparisons.
 *
 * @private
 * @param {Object} object The object to modify.
 * @param {string} key The key of the property to assign.
 * @param {*} value The value to assign.
 */
function assignValue(object, key, value) {
  var objValue = object[key];
  if (!(hasOwnProperty.call(object, key) && eq(objValue, value)) ||
      (value === undefined && !(key in object))) {
    object[key] = value;
  }
}

/**
 * Gets the index at which the `key` is found in `array` of key-value pairs.
 *
 * @private
 * @param {Array} array The array to inspect.
 * @param {*} key The key to search for.
 * @returns {number} Returns the index of the matched value, else `-1`.
 */
function assocIndexOf(array, key) {
  var length = array.length;
  while (length--) {
    if (eq(array[length][0], key)) {
      return length;
    }
  }
  return -1;
}

/**
 * The base implementation of `_.assign` without support for multiple sources
 * or `customizer` functions.
 *
 * @private
 * @param {Object} object The destination object.
 * @param {Object} source The source object.
 * @returns {Object} Returns `object`.
 */
function baseAssign(object, source) {
  return object && copyObject(source, keys(source), object);
}

/**
 * The base implementation of `_.clone` and `_.cloneDeep` which tracks
 * traversed objects.
 *
 * @private
 * @param {*} value The value to clone.
 * @param {boolean} [isDeep] Specify a deep clone.
 * @param {boolean} [isFull] Specify a clone including symbols.
 * @param {Function} [customizer] The function to customize cloning.
 * @param {string} [key] The key of `value`.
 * @param {Object} [object] The parent object of `value`.
 * @param {Object} [stack] Tracks traversed objects and their clone counterparts.
 * @returns {*} Returns the cloned value.
 */
function baseClone(value, isDeep, isFull, customizer, key, object, stack) {
  var result;
  if (customizer) {
    result = object ? customizer(value, key, object, stack) : customizer(value);
  }
  if (result !== undefined) {
    return result;
  }
  if (!isObject(value)) {
    return value;
  }
  var isArr = isArray(value);
  if (isArr) {
    result = initCloneArray(value);
    if (!isDeep) {
      return copyArray(value, result);
    }
  } else {
    var tag = getTag(value),
        isFunc = tag == funcTag || tag == genTag;

    if (isBuffer(value)) {
      return cloneBuffer(value, isDeep);
    }
    if (tag == objectTag || tag == argsTag || (isFunc && !object)) {
      if (isHostObject(value)) {
        return object ? value : {};
      }
      result = initCloneObject(isFunc ? {} : value);
      if (!isDeep) {
        return copySymbols(value, baseAssign(result, value));
      }
    } else {
      if (!cloneableTags[tag]) {
        return object ? value : {};
      }
      result = initCloneByTag(value, tag, baseClone, isDeep);
    }
  }
  // Check for circular references and return its corresponding clone.
  stack || (stack = new Stack);
  var stacked = stack.get(value);
  if (stacked) {
    return stacked;
  }
  stack.set(value, result);

  if (!isArr) {
    var props = isFull ? getAllKeys(value) : keys(value);
  }
  arrayEach(props || value, function(subValue, key) {
    if (props) {
      key = subValue;
      subValue = value[key];
    }
    // Recursively populate clone (susceptible to call stack limits).
    assignValue(result, key, baseClone(subValue, isDeep, isFull, customizer, key, value, stack));
  });
  return result;
}

/**
 * The base implementation of `_.create` without support for assigning
 * properties to the created object.
 *
 * @private
 * @param {Object} prototype The object to inherit from.
 * @returns {Object} Returns the new object.
 */
function baseCreate(proto) {
  return isObject(proto) ? objectCreate(proto) : {};
}

/**
 * The base implementation of `getAllKeys` and `getAllKeysIn` which uses
 * `keysFunc` and `symbolsFunc` to get the enumerable property names and
 * symbols of `object`.
 *
 * @private
 * @param {Object} object The object to query.
 * @param {Function} keysFunc The function to get the keys of `object`.
 * @param {Function} symbolsFunc The function to get the symbols of `object`.
 * @returns {Array} Returns the array of property names and symbols.
 */
function baseGetAllKeys(object, keysFunc, symbolsFunc) {
  var result = keysFunc(object);
  return isArray(object) ? result : arrayPush(result, symbolsFunc(object));
}

/**
 * The base implementation of `getTag`.
 *
 * @private
 * @param {*} value The value to query.
 * @returns {string} Returns the `toStringTag`.
 */
function baseGetTag(value) {
  return objectToString.call(value);
}

/**
 * The base implementation of `_.isNative` without bad shim checks.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a native function,
 *  else `false`.
 */
function baseIsNative(value) {
  if (!isObject(value) || isMasked(value)) {
    return false;
  }
  var pattern = (isFunction(value) || isHostObject(value)) ? reIsNative : reIsHostCtor;
  return pattern.test(toSource(value));
}

/**
 * The base implementation of `_.keys` which doesn't treat sparse arrays as dense.
 *
 * @private
 * @param {Object} object The object to query.
 * @returns {Array} Returns the array of property names.
 */
function baseKeys(object) {
  if (!isPrototype(object)) {
    return nativeKeys(object);
  }
  var result = [];
  for (var key in Object(object)) {
    if (hasOwnProperty.call(object, key) && key != 'constructor') {
      result.push(key);
    }
  }
  return result;
}

/**
 * Creates a clone of  `buffer`.
 *
 * @private
 * @param {Buffer} buffer The buffer to clone.
 * @param {boolean} [isDeep] Specify a deep clone.
 * @returns {Buffer} Returns the cloned buffer.
 */
function cloneBuffer(buffer, isDeep) {
  if (isDeep) {
    return buffer.slice();
  }
  var result = new buffer.constructor(buffer.length);
  buffer.copy(result);
  return result;
}

/**
 * Creates a clone of `arrayBuffer`.
 *
 * @private
 * @param {ArrayBuffer} arrayBuffer The array buffer to clone.
 * @returns {ArrayBuffer} Returns the cloned array buffer.
 */
function cloneArrayBuffer(arrayBuffer) {
  var result = new arrayBuffer.constructor(arrayBuffer.byteLength);
  new Uint8Array(result).set(new Uint8Array(arrayBuffer));
  return result;
}

/**
 * Creates a clone of `dataView`.
 *
 * @private
 * @param {Object} dataView The data view to clone.
 * @param {boolean} [isDeep] Specify a deep clone.
 * @returns {Object} Returns the cloned data view.
 */
function cloneDataView(dataView, isDeep) {
  var buffer = isDeep ? cloneArrayBuffer(dataView.buffer) : dataView.buffer;
  return new dataView.constructor(buffer, dataView.byteOffset, dataView.byteLength);
}

/**
 * Creates a clone of `map`.
 *
 * @private
 * @param {Object} map The map to clone.
 * @param {Function} cloneFunc The function to clone values.
 * @param {boolean} [isDeep] Specify a deep clone.
 * @returns {Object} Returns the cloned map.
 */
function cloneMap(map, isDeep, cloneFunc) {
  var array = isDeep ? cloneFunc(mapToArray(map), true) : mapToArray(map);
  return arrayReduce(array, addMapEntry, new map.constructor);
}

/**
 * Creates a clone of `regexp`.
 *
 * @private
 * @param {Object} regexp The regexp to clone.
 * @returns {Object} Returns the cloned regexp.
 */
function cloneRegExp(regexp) {
  var result = new regexp.constructor(regexp.source, reFlags.exec(regexp));
  result.lastIndex = regexp.lastIndex;
  return result;
}

/**
 * Creates a clone of `set`.
 *
 * @private
 * @param {Object} set The set to clone.
 * @param {Function} cloneFunc The function to clone values.
 * @param {boolean} [isDeep] Specify a deep clone.
 * @returns {Object} Returns the cloned set.
 */
function cloneSet(set, isDeep, cloneFunc) {
  var array = isDeep ? cloneFunc(setToArray(set), true) : setToArray(set);
  return arrayReduce(array, addSetEntry, new set.constructor);
}

/**
 * Creates a clone of the `symbol` object.
 *
 * @private
 * @param {Object} symbol The symbol object to clone.
 * @returns {Object} Returns the cloned symbol object.
 */
function cloneSymbol(symbol) {
  return symbolValueOf ? Object(symbolValueOf.call(symbol)) : {};
}

/**
 * Creates a clone of `typedArray`.
 *
 * @private
 * @param {Object} typedArray The typed array to clone.
 * @param {boolean} [isDeep] Specify a deep clone.
 * @returns {Object} Returns the cloned typed array.
 */
function cloneTypedArray(typedArray, isDeep) {
  var buffer = isDeep ? cloneArrayBuffer(typedArray.buffer) : typedArray.buffer;
  return new typedArray.constructor(buffer, typedArray.byteOffset, typedArray.length);
}

/**
 * Copies the values of `source` to `array`.
 *
 * @private
 * @param {Array} source The array to copy values from.
 * @param {Array} [array=[]] The array to copy values to.
 * @returns {Array} Returns `array`.
 */
function copyArray(source, array) {
  var index = -1,
      length = source.length;

  array || (array = Array(length));
  while (++index < length) {
    array[index] = source[index];
  }
  return array;
}

/**
 * Copies properties of `source` to `object`.
 *
 * @private
 * @param {Object} source The object to copy properties from.
 * @param {Array} props The property identifiers to copy.
 * @param {Object} [object={}] The object to copy properties to.
 * @param {Function} [customizer] The function to customize copied values.
 * @returns {Object} Returns `object`.
 */
function copyObject(source, props, object, customizer) {
  object || (object = {});

  var index = -1,
      length = props.length;

  while (++index < length) {
    var key = props[index];

    var newValue = customizer
      ? customizer(object[key], source[key], key, object, source)
      : undefined;

    assignValue(object, key, newValue === undefined ? source[key] : newValue);
  }
  return object;
}

/**
 * Copies own symbol properties of `source` to `object`.
 *
 * @private
 * @param {Object} source The object to copy symbols from.
 * @param {Object} [object={}] The object to copy symbols to.
 * @returns {Object} Returns `object`.
 */
function copySymbols(source, object) {
  return copyObject(source, getSymbols(source), object);
}

/**
 * Creates an array of own enumerable property names and symbols of `object`.
 *
 * @private
 * @param {Object} object The object to query.
 * @returns {Array} Returns the array of property names and symbols.
 */
function getAllKeys(object) {
  return baseGetAllKeys(object, keys, getSymbols);
}

/**
 * Gets the data for `map`.
 *
 * @private
 * @param {Object} map The map to query.
 * @param {string} key The reference key.
 * @returns {*} Returns the map data.
 */
function getMapData(map, key) {
  var data = map.__data__;
  return isKeyable(key)
    ? data[typeof key == 'string' ? 'string' : 'hash']
    : data.map;
}

/**
 * Gets the native function at `key` of `object`.
 *
 * @private
 * @param {Object} object The object to query.
 * @param {string} key The key of the method to get.
 * @returns {*} Returns the function if it's native, else `undefined`.
 */
function getNative(object, key) {
  var value = getValue(object, key);
  return baseIsNative(value) ? value : undefined;
}

/**
 * Creates an array of the own enumerable symbol properties of `object`.
 *
 * @private
 * @param {Object} object The object to query.
 * @returns {Array} Returns the array of symbols.
 */
var getSymbols = nativeGetSymbols ? overArg(nativeGetSymbols, Object) : stubArray;

/**
 * Gets the `toStringTag` of `value`.
 *
 * @private
 * @param {*} value The value to query.
 * @returns {string} Returns the `toStringTag`.
 */
var getTag = baseGetTag;

// Fallback for data views, maps, sets, and weak maps in IE 11,
// for data views in Edge < 14, and promises in Node.js.
if ((DataView && getTag(new DataView(new ArrayBuffer(1))) != dataViewTag) ||
    (Map && getTag(new Map) != mapTag) ||
    (Promise && getTag(Promise.resolve()) != promiseTag) ||
    (Set && getTag(new Set) != setTag) ||
    (WeakMap && getTag(new WeakMap) != weakMapTag)) {
  getTag = function(value) {
    var result = objectToString.call(value),
        Ctor = result == objectTag ? value.constructor : undefined,
        ctorString = Ctor ? toSource(Ctor) : undefined;

    if (ctorString) {
      switch (ctorString) {
        case dataViewCtorString: return dataViewTag;
        case mapCtorString: return mapTag;
        case promiseCtorString: return promiseTag;
        case setCtorString: return setTag;
        case weakMapCtorString: return weakMapTag;
      }
    }
    return result;
  };
}

/**
 * Initializes an array clone.
 *
 * @private
 * @param {Array} array The array to clone.
 * @returns {Array} Returns the initialized clone.
 */
function initCloneArray(array) {
  var length = array.length,
      result = array.constructor(length);

  // Add properties assigned by `RegExp#exec`.
  if (length && typeof array[0] == 'string' && hasOwnProperty.call(array, 'index')) {
    result.index = array.index;
    result.input = array.input;
  }
  return result;
}

/**
 * Initializes an object clone.
 *
 * @private
 * @param {Object} object The object to clone.
 * @returns {Object} Returns the initialized clone.
 */
function initCloneObject(object) {
  return (typeof object.constructor == 'function' && !isPrototype(object))
    ? baseCreate(getPrototype(object))
    : {};
}

/**
 * Initializes an object clone based on its `toStringTag`.
 *
 * **Note:** This function only supports cloning values with tags of
 * `Boolean`, `Date`, `Error`, `Number`, `RegExp`, or `String`.
 *
 * @private
 * @param {Object} object The object to clone.
 * @param {string} tag The `toStringTag` of the object to clone.
 * @param {Function} cloneFunc The function to clone values.
 * @param {boolean} [isDeep] Specify a deep clone.
 * @returns {Object} Returns the initialized clone.
 */
function initCloneByTag(object, tag, cloneFunc, isDeep) {
  var Ctor = object.constructor;
  switch (tag) {
    case arrayBufferTag:
      return cloneArrayBuffer(object);

    case boolTag:
    case dateTag:
      return new Ctor(+object);

    case dataViewTag:
      return cloneDataView(object, isDeep);

    case float32Tag: case float64Tag:
    case int8Tag: case int16Tag: case int32Tag:
    case uint8Tag: case uint8ClampedTag: case uint16Tag: case uint32Tag:
      return cloneTypedArray(object, isDeep);

    case mapTag:
      return cloneMap(object, isDeep, cloneFunc);

    case numberTag:
    case stringTag:
      return new Ctor(object);

    case regexpTag:
      return cloneRegExp(object);

    case setTag:
      return cloneSet(object, isDeep, cloneFunc);

    case symbolTag:
      return cloneSymbol(object);
  }
}

/**
 * Checks if `value` is a valid array-like index.
 *
 * @private
 * @param {*} value The value to check.
 * @param {number} [length=MAX_SAFE_INTEGER] The upper bounds of a valid index.
 * @returns {boolean} Returns `true` if `value` is a valid index, else `false`.
 */
function isIndex(value, length) {
  length = length == null ? MAX_SAFE_INTEGER : length;
  return !!length &&
    (typeof value == 'number' || reIsUint.test(value)) &&
    (value > -1 && value % 1 == 0 && value < length);
}

/**
 * Checks if `value` is suitable for use as unique object key.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is suitable, else `false`.
 */
function isKeyable(value) {
  var type = typeof value;
  return (type == 'string' || type == 'number' || type == 'symbol' || type == 'boolean')
    ? (value !== '__proto__')
    : (value === null);
}

/**
 * Checks if `func` has its source masked.
 *
 * @private
 * @param {Function} func The function to check.
 * @returns {boolean} Returns `true` if `func` is masked, else `false`.
 */
function isMasked(func) {
  return !!maskSrcKey && (maskSrcKey in func);
}

/**
 * Checks if `value` is likely a prototype object.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a prototype, else `false`.
 */
function isPrototype(value) {
  var Ctor = value && value.constructor,
      proto = (typeof Ctor == 'function' && Ctor.prototype) || objectProto;

  return value === proto;
}

/**
 * Converts `func` to its source code.
 *
 * @private
 * @param {Function} func The function to process.
 * @returns {string} Returns the source code.
 */
function toSource(func) {
  if (func != null) {
    try {
      return funcToString.call(func);
    } catch (e) {}
    try {
      return (func + '');
    } catch (e) {}
  }
  return '';
}

/**
 * This method is like `_.clone` except that it recursively clones `value`.
 *
 * @static
 * @memberOf _
 * @since 1.0.0
 * @category Lang
 * @param {*} value The value to recursively clone.
 * @returns {*} Returns the deep cloned value.
 * @see _.clone
 * @example
 *
 * var objects = [{ 'a': 1 }, { 'b': 2 }];
 *
 * var deep = _.cloneDeep(objects);
 * console.log(deep[0] === objects[0]);
 * // => false
 */
function cloneDeep(value) {
  return baseClone(value, true, true);
}

/**
 * Performs a
 * [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
 * comparison between two values to determine if they are equivalent.
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to compare.
 * @param {*} other The other value to compare.
 * @returns {boolean} Returns `true` if the values are equivalent, else `false`.
 * @example
 *
 * var object = { 'a': 1 };
 * var other = { 'a': 1 };
 *
 * _.eq(object, object);
 * // => true
 *
 * _.eq(object, other);
 * // => false
 *
 * _.eq('a', 'a');
 * // => true
 *
 * _.eq('a', Object('a'));
 * // => false
 *
 * _.eq(NaN, NaN);
 * // => true
 */
function eq(value, other) {
  return value === other || (value !== value && other !== other);
}

/**
 * Checks if `value` is likely an `arguments` object.
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is an `arguments` object,
 *  else `false`.
 * @example
 *
 * _.isArguments(function() { return arguments; }());
 * // => true
 *
 * _.isArguments([1, 2, 3]);
 * // => false
 */
function isArguments(value) {
  // Safari 8.1 makes `arguments.callee` enumerable in strict mode.
  return isArrayLikeObject(value) && hasOwnProperty.call(value, 'callee') &&
    (!propertyIsEnumerable.call(value, 'callee') || objectToString.call(value) == argsTag);
}

/**
 * Checks if `value` is classified as an `Array` object.
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is an array, else `false`.
 * @example
 *
 * _.isArray([1, 2, 3]);
 * // => true
 *
 * _.isArray(document.body.children);
 * // => false
 *
 * _.isArray('abc');
 * // => false
 *
 * _.isArray(_.noop);
 * // => false
 */
var isArray = Array.isArray;

/**
 * Checks if `value` is array-like. A value is considered array-like if it's
 * not a function and has a `value.length` that's an integer greater than or
 * equal to `0` and less than or equal to `Number.MAX_SAFE_INTEGER`.
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is array-like, else `false`.
 * @example
 *
 * _.isArrayLike([1, 2, 3]);
 * // => true
 *
 * _.isArrayLike(document.body.children);
 * // => true
 *
 * _.isArrayLike('abc');
 * // => true
 *
 * _.isArrayLike(_.noop);
 * // => false
 */
function isArrayLike(value) {
  return value != null && isLength(value.length) && !isFunction(value);
}

/**
 * This method is like `_.isArrayLike` except that it also checks if `value`
 * is an object.
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is an array-like object,
 *  else `false`.
 * @example
 *
 * _.isArrayLikeObject([1, 2, 3]);
 * // => true
 *
 * _.isArrayLikeObject(document.body.children);
 * // => true
 *
 * _.isArrayLikeObject('abc');
 * // => false
 *
 * _.isArrayLikeObject(_.noop);
 * // => false
 */
function isArrayLikeObject(value) {
  return isObjectLike(value) && isArrayLike(value);
}

/**
 * Checks if `value` is a buffer.
 *
 * @static
 * @memberOf _
 * @since 4.3.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a buffer, else `false`.
 * @example
 *
 * _.isBuffer(new Buffer(2));
 * // => true
 *
 * _.isBuffer(new Uint8Array(2));
 * // => false
 */
var isBuffer = nativeIsBuffer || stubFalse;

/**
 * Checks if `value` is classified as a `Function` object.
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a function, else `false`.
 * @example
 *
 * _.isFunction(_);
 * // => true
 *
 * _.isFunction(/abc/);
 * // => false
 */
function isFunction(value) {
  // The use of `Object#toString` avoids issues with the `typeof` operator
  // in Safari 8-9 which returns 'object' for typed array and other constructors.
  var tag = isObject(value) ? objectToString.call(value) : '';
  return tag == funcTag || tag == genTag;
}

/**
 * Checks if `value` is a valid array-like length.
 *
 * **Note:** This method is loosely based on
 * [`ToLength`](http://ecma-international.org/ecma-262/7.0/#sec-tolength).
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a valid length, else `false`.
 * @example
 *
 * _.isLength(3);
 * // => true
 *
 * _.isLength(Number.MIN_VALUE);
 * // => false
 *
 * _.isLength(Infinity);
 * // => false
 *
 * _.isLength('3');
 * // => false
 */
function isLength(value) {
  return typeof value == 'number' &&
    value > -1 && value % 1 == 0 && value <= MAX_SAFE_INTEGER;
}

/**
 * Checks if `value` is the
 * [language type](http://www.ecma-international.org/ecma-262/7.0/#sec-ecmascript-language-types)
 * of `Object`. (e.g. arrays, functions, objects, regexes, `new Number(0)`, and `new String('')`)
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is an object, else `false`.
 * @example
 *
 * _.isObject({});
 * // => true
 *
 * _.isObject([1, 2, 3]);
 * // => true
 *
 * _.isObject(_.noop);
 * // => true
 *
 * _.isObject(null);
 * // => false
 */
function isObject(value) {
  var type = typeof value;
  return !!value && (type == 'object' || type == 'function');
}

/**
 * Checks if `value` is object-like. A value is object-like if it's not `null`
 * and has a `typeof` result of "object".
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is object-like, else `false`.
 * @example
 *
 * _.isObjectLike({});
 * // => true
 *
 * _.isObjectLike([1, 2, 3]);
 * // => true
 *
 * _.isObjectLike(_.noop);
 * // => false
 *
 * _.isObjectLike(null);
 * // => false
 */
function isObjectLike(value) {
  return !!value && typeof value == 'object';
}

/**
 * Creates an array of the own enumerable property names of `object`.
 *
 * **Note:** Non-object values are coerced to objects. See the
 * [ES spec](http://ecma-international.org/ecma-262/7.0/#sec-object.keys)
 * for more details.
 *
 * @static
 * @since 0.1.0
 * @memberOf _
 * @category Object
 * @param {Object} object The object to query.
 * @returns {Array} Returns the array of property names.
 * @example
 *
 * function Foo() {
 *   this.a = 1;
 *   this.b = 2;
 * }
 *
 * Foo.prototype.c = 3;
 *
 * _.keys(new Foo);
 * // => ['a', 'b'] (iteration order is not guaranteed)
 *
 * _.keys('hi');
 * // => ['0', '1']
 */
function keys(object) {
  return isArrayLike(object) ? arrayLikeKeys(object) : baseKeys(object);
}

/**
 * This method returns a new empty array.
 *
 * @static
 * @memberOf _
 * @since 4.13.0
 * @category Util
 * @returns {Array} Returns the new empty array.
 * @example
 *
 * var arrays = _.times(2, _.stubArray);
 *
 * console.log(arrays);
 * // => [[], []]
 *
 * console.log(arrays[0] === arrays[1]);
 * // => false
 */
function stubArray() {
  return [];
}

/**
 * This method returns `false`.
 *
 * @static
 * @memberOf _
 * @since 4.13.0
 * @category Util
 * @returns {boolean} Returns `false`.
 * @example
 *
 * _.times(2, _.stubFalse);
 * // => [false, false]
 */
function stubFalse() {
  return false;
}

module.exports = cloneDeep;


/***/ }),

/***/ 52:
/***/ ((module, __unused_webpack_exports, __webpack_require__) => {

/**
 * lodash (Custom Build) <https://lodash.com/>
 * Build: `lodash modularize exports="npm" -o ./`
 * Copyright jQuery Foundation and other contributors <https://jquery.org/>
 * Released under MIT license <https://lodash.com/license>
 * Based on Underscore.js 1.8.3 <http://underscorejs.org/LICENSE>
 * Copyright Jeremy Ashkenas, DocumentCloud and Investigative Reporters & Editors
 */

/** Used as references for various `Number` constants. */
var MAX_SAFE_INTEGER = 9007199254740991;

/** `Object#toString` result references. */
var argsTag = '[object Arguments]',
    funcTag = '[object Function]',
    genTag = '[object GeneratorFunction]',
    mapTag = '[object Map]',
    objectTag = '[object Object]',
    promiseTag = '[object Promise]',
    setTag = '[object Set]',
    weakMapTag = '[object WeakMap]';

var dataViewTag = '[object DataView]';

/**
 * Used to match `RegExp`
 * [syntax characters](http://ecma-international.org/ecma-262/7.0/#sec-patterns).
 */
var reRegExpChar = /[\\^$.*+?()[\]{}|]/g;

/** Used to detect host constructors (Safari). */
var reIsHostCtor = /^\[object .+?Constructor\]$/;

/** Used to detect unsigned integer values. */
var reIsUint = /^(?:0|[1-9]\d*)$/;

/** Detect free variable `global` from Node.js. */
var freeGlobal = typeof __webpack_require__.g == 'object' && __webpack_require__.g && __webpack_require__.g.Object === Object && __webpack_require__.g;

/** Detect free variable `self`. */
var freeSelf = typeof self == 'object' && self && self.Object === Object && self;

/** Used as a reference to the global object. */
var root = freeGlobal || freeSelf || Function('return this')();

/**
 * A specialized version of `_.map` for arrays without support for iteratee
 * shorthands.
 *
 * @private
 * @param {Array} [array] The array to iterate over.
 * @param {Function} iteratee The function invoked per iteration.
 * @returns {Array} Returns the new mapped array.
 */
function arrayMap(array, iteratee) {
  var index = -1,
      length = array ? array.length : 0,
      result = Array(length);

  while (++index < length) {
    result[index] = iteratee(array[index], index, array);
  }
  return result;
}

/**
 * The base implementation of `_.times` without support for iteratee shorthands
 * or max array length checks.
 *
 * @private
 * @param {number} n The number of times to invoke `iteratee`.
 * @param {Function} iteratee The function invoked per iteration.
 * @returns {Array} Returns the array of results.
 */
function baseTimes(n, iteratee) {
  var index = -1,
      result = Array(n);

  while (++index < n) {
    result[index] = iteratee(index);
  }
  return result;
}

/**
 * The base implementation of `_.toPairs` and `_.toPairsIn` which creates an array
 * of key-value pairs for `object` corresponding to the property names of `props`.
 *
 * @private
 * @param {Object} object The object to query.
 * @param {Array} props The property names to get values for.
 * @returns {Object} Returns the key-value pairs.
 */
function baseToPairs(object, props) {
  return arrayMap(props, function(key) {
    return [key, object[key]];
  });
}

/**
 * Gets the value at `key` of `object`.
 *
 * @private
 * @param {Object} [object] The object to query.
 * @param {string} key The key of the property to get.
 * @returns {*} Returns the property value.
 */
function getValue(object, key) {
  return object == null ? undefined : object[key];
}

/**
 * Checks if `value` is a host object in IE < 9.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a host object, else `false`.
 */
function isHostObject(value) {
  // Many host objects are `Object` objects that can coerce to strings
  // despite having improperly defined `toString` methods.
  var result = false;
  if (value != null && typeof value.toString != 'function') {
    try {
      result = !!(value + '');
    } catch (e) {}
  }
  return result;
}

/**
 * Converts `map` to its key-value pairs.
 *
 * @private
 * @param {Object} map The map to convert.
 * @returns {Array} Returns the key-value pairs.
 */
function mapToArray(map) {
  var index = -1,
      result = Array(map.size);

  map.forEach(function(value, key) {
    result[++index] = [key, value];
  });
  return result;
}

/**
 * Creates a unary function that invokes `func` with its argument transformed.
 *
 * @private
 * @param {Function} func The function to wrap.
 * @param {Function} transform The argument transform.
 * @returns {Function} Returns the new function.
 */
function overArg(func, transform) {
  return function(arg) {
    return func(transform(arg));
  };
}

/**
 * Converts `set` to its value-value pairs.
 *
 * @private
 * @param {Object} set The set to convert.
 * @returns {Array} Returns the value-value pairs.
 */
function setToPairs(set) {
  var index = -1,
      result = Array(set.size);

  set.forEach(function(value) {
    result[++index] = [value, value];
  });
  return result;
}

/** Used for built-in method references. */
var funcProto = Function.prototype,
    objectProto = Object.prototype;

/** Used to detect overreaching core-js shims. */
var coreJsData = root['__core-js_shared__'];

/** Used to detect methods masquerading as native. */
var maskSrcKey = (function() {
  var uid = /[^.]+$/.exec(coreJsData && coreJsData.keys && coreJsData.keys.IE_PROTO || '');
  return uid ? ('Symbol(src)_1.' + uid) : '';
}());

/** Used to resolve the decompiled source of functions. */
var funcToString = funcProto.toString;

/** Used to check objects for own properties. */
var hasOwnProperty = objectProto.hasOwnProperty;

/**
 * Used to resolve the
 * [`toStringTag`](http://ecma-international.org/ecma-262/7.0/#sec-object.prototype.tostring)
 * of values.
 */
var objectToString = objectProto.toString;

/** Used to detect if a method is native. */
var reIsNative = RegExp('^' +
  funcToString.call(hasOwnProperty).replace(reRegExpChar, '\\$&')
  .replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g, '$1.*?') + '$'
);

/** Built-in value references. */
var propertyIsEnumerable = objectProto.propertyIsEnumerable;

/* Built-in method references for those with the same name as other `lodash` methods. */
var nativeKeys = overArg(Object.keys, Object);

/* Built-in method references that are verified to be native. */
var DataView = getNative(root, 'DataView'),
    Map = getNative(root, 'Map'),
    Promise = getNative(root, 'Promise'),
    Set = getNative(root, 'Set'),
    WeakMap = getNative(root, 'WeakMap');

/** Used to detect maps, sets, and weakmaps. */
var dataViewCtorString = toSource(DataView),
    mapCtorString = toSource(Map),
    promiseCtorString = toSource(Promise),
    setCtorString = toSource(Set),
    weakMapCtorString = toSource(WeakMap);

/**
 * Creates an array of the enumerable property names of the array-like `value`.
 *
 * @private
 * @param {*} value The value to query.
 * @param {boolean} inherited Specify returning inherited property names.
 * @returns {Array} Returns the array of property names.
 */
function arrayLikeKeys(value, inherited) {
  // Safari 8.1 makes `arguments.callee` enumerable in strict mode.
  // Safari 9 makes `arguments.length` enumerable in strict mode.
  var result = (isArray(value) || isArguments(value))
    ? baseTimes(value.length, String)
    : [];

  var length = result.length,
      skipIndexes = !!length;

  for (var key in value) {
    if ((inherited || hasOwnProperty.call(value, key)) &&
        !(skipIndexes && (key == 'length' || isIndex(key, length)))) {
      result.push(key);
    }
  }
  return result;
}

/**
 * The base implementation of `getTag`.
 *
 * @private
 * @param {*} value The value to query.
 * @returns {string} Returns the `toStringTag`.
 */
function baseGetTag(value) {
  return objectToString.call(value);
}

/**
 * The base implementation of `_.isNative` without bad shim checks.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a native function,
 *  else `false`.
 */
function baseIsNative(value) {
  if (!isObject(value) || isMasked(value)) {
    return false;
  }
  var pattern = (isFunction(value) || isHostObject(value)) ? reIsNative : reIsHostCtor;
  return pattern.test(toSource(value));
}

/**
 * The base implementation of `_.keys` which doesn't treat sparse arrays as dense.
 *
 * @private
 * @param {Object} object The object to query.
 * @returns {Array} Returns the array of property names.
 */
function baseKeys(object) {
  if (!isPrototype(object)) {
    return nativeKeys(object);
  }
  var result = [];
  for (var key in Object(object)) {
    if (hasOwnProperty.call(object, key) && key != 'constructor') {
      result.push(key);
    }
  }
  return result;
}

/**
 * Creates a `_.toPairs` or `_.toPairsIn` function.
 *
 * @private
 * @param {Function} keysFunc The function to get the keys of a given object.
 * @returns {Function} Returns the new pairs function.
 */
function createToPairs(keysFunc) {
  return function(object) {
    var tag = getTag(object);
    if (tag == mapTag) {
      return mapToArray(object);
    }
    if (tag == setTag) {
      return setToPairs(object);
    }
    return baseToPairs(object, keysFunc(object));
  };
}

/**
 * Gets the native function at `key` of `object`.
 *
 * @private
 * @param {Object} object The object to query.
 * @param {string} key The key of the method to get.
 * @returns {*} Returns the function if it's native, else `undefined`.
 */
function getNative(object, key) {
  var value = getValue(object, key);
  return baseIsNative(value) ? value : undefined;
}

/**
 * Gets the `toStringTag` of `value`.
 *
 * @private
 * @param {*} value The value to query.
 * @returns {string} Returns the `toStringTag`.
 */
var getTag = baseGetTag;

// Fallback for data views, maps, sets, and weak maps in IE 11,
// for data views in Edge < 14, and promises in Node.js.
if ((DataView && getTag(new DataView(new ArrayBuffer(1))) != dataViewTag) ||
    (Map && getTag(new Map) != mapTag) ||
    (Promise && getTag(Promise.resolve()) != promiseTag) ||
    (Set && getTag(new Set) != setTag) ||
    (WeakMap && getTag(new WeakMap) != weakMapTag)) {
  getTag = function(value) {
    var result = objectToString.call(value),
        Ctor = result == objectTag ? value.constructor : undefined,
        ctorString = Ctor ? toSource(Ctor) : undefined;

    if (ctorString) {
      switch (ctorString) {
        case dataViewCtorString: return dataViewTag;
        case mapCtorString: return mapTag;
        case promiseCtorString: return promiseTag;
        case setCtorString: return setTag;
        case weakMapCtorString: return weakMapTag;
      }
    }
    return result;
  };
}

/**
 * Checks if `value` is a valid array-like index.
 *
 * @private
 * @param {*} value The value to check.
 * @param {number} [length=MAX_SAFE_INTEGER] The upper bounds of a valid index.
 * @returns {boolean} Returns `true` if `value` is a valid index, else `false`.
 */
function isIndex(value, length) {
  length = length == null ? MAX_SAFE_INTEGER : length;
  return !!length &&
    (typeof value == 'number' || reIsUint.test(value)) &&
    (value > -1 && value % 1 == 0 && value < length);
}

/**
 * Checks if `func` has its source masked.
 *
 * @private
 * @param {Function} func The function to check.
 * @returns {boolean} Returns `true` if `func` is masked, else `false`.
 */
function isMasked(func) {
  return !!maskSrcKey && (maskSrcKey in func);
}

/**
 * Checks if `value` is likely a prototype object.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a prototype, else `false`.
 */
function isPrototype(value) {
  var Ctor = value && value.constructor,
      proto = (typeof Ctor == 'function' && Ctor.prototype) || objectProto;

  return value === proto;
}

/**
 * Converts `func` to its source code.
 *
 * @private
 * @param {Function} func The function to process.
 * @returns {string} Returns the source code.
 */
function toSource(func) {
  if (func != null) {
    try {
      return funcToString.call(func);
    } catch (e) {}
    try {
      return (func + '');
    } catch (e) {}
  }
  return '';
}

/**
 * Checks if `value` is likely an `arguments` object.
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is an `arguments` object,
 *  else `false`.
 * @example
 *
 * _.isArguments(function() { return arguments; }());
 * // => true
 *
 * _.isArguments([1, 2, 3]);
 * // => false
 */
function isArguments(value) {
  // Safari 8.1 makes `arguments.callee` enumerable in strict mode.
  return isArrayLikeObject(value) && hasOwnProperty.call(value, 'callee') &&
    (!propertyIsEnumerable.call(value, 'callee') || objectToString.call(value) == argsTag);
}

/**
 * Checks if `value` is classified as an `Array` object.
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is an array, else `false`.
 * @example
 *
 * _.isArray([1, 2, 3]);
 * // => true
 *
 * _.isArray(document.body.children);
 * // => false
 *
 * _.isArray('abc');
 * // => false
 *
 * _.isArray(_.noop);
 * // => false
 */
var isArray = Array.isArray;

/**
 * Checks if `value` is array-like. A value is considered array-like if it's
 * not a function and has a `value.length` that's an integer greater than or
 * equal to `0` and less than or equal to `Number.MAX_SAFE_INTEGER`.
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is array-like, else `false`.
 * @example
 *
 * _.isArrayLike([1, 2, 3]);
 * // => true
 *
 * _.isArrayLike(document.body.children);
 * // => true
 *
 * _.isArrayLike('abc');
 * // => true
 *
 * _.isArrayLike(_.noop);
 * // => false
 */
function isArrayLike(value) {
  return value != null && isLength(value.length) && !isFunction(value);
}

/**
 * This method is like `_.isArrayLike` except that it also checks if `value`
 * is an object.
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is an array-like object,
 *  else `false`.
 * @example
 *
 * _.isArrayLikeObject([1, 2, 3]);
 * // => true
 *
 * _.isArrayLikeObject(document.body.children);
 * // => true
 *
 * _.isArrayLikeObject('abc');
 * // => false
 *
 * _.isArrayLikeObject(_.noop);
 * // => false
 */
function isArrayLikeObject(value) {
  return isObjectLike(value) && isArrayLike(value);
}

/**
 * Checks if `value` is classified as a `Function` object.
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a function, else `false`.
 * @example
 *
 * _.isFunction(_);
 * // => true
 *
 * _.isFunction(/abc/);
 * // => false
 */
function isFunction(value) {
  // The use of `Object#toString` avoids issues with the `typeof` operator
  // in Safari 8-9 which returns 'object' for typed array and other constructors.
  var tag = isObject(value) ? objectToString.call(value) : '';
  return tag == funcTag || tag == genTag;
}

/**
 * Checks if `value` is a valid array-like length.
 *
 * **Note:** This method is loosely based on
 * [`ToLength`](http://ecma-international.org/ecma-262/7.0/#sec-tolength).
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a valid length, else `false`.
 * @example
 *
 * _.isLength(3);
 * // => true
 *
 * _.isLength(Number.MIN_VALUE);
 * // => false
 *
 * _.isLength(Infinity);
 * // => false
 *
 * _.isLength('3');
 * // => false
 */
function isLength(value) {
  return typeof value == 'number' &&
    value > -1 && value % 1 == 0 && value <= MAX_SAFE_INTEGER;
}

/**
 * Checks if `value` is the
 * [language type](http://www.ecma-international.org/ecma-262/7.0/#sec-ecmascript-language-types)
 * of `Object`. (e.g. arrays, functions, objects, regexes, `new Number(0)`, and `new String('')`)
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is an object, else `false`.
 * @example
 *
 * _.isObject({});
 * // => true
 *
 * _.isObject([1, 2, 3]);
 * // => true
 *
 * _.isObject(_.noop);
 * // => true
 *
 * _.isObject(null);
 * // => false
 */
function isObject(value) {
  var type = typeof value;
  return !!value && (type == 'object' || type == 'function');
}

/**
 * Checks if `value` is object-like. A value is object-like if it's not `null`
 * and has a `typeof` result of "object".
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is object-like, else `false`.
 * @example
 *
 * _.isObjectLike({});
 * // => true
 *
 * _.isObjectLike([1, 2, 3]);
 * // => true
 *
 * _.isObjectLike(_.noop);
 * // => false
 *
 * _.isObjectLike(null);
 * // => false
 */
function isObjectLike(value) {
  return !!value && typeof value == 'object';
}

/**
 * Creates an array of the own enumerable property names of `object`.
 *
 * **Note:** Non-object values are coerced to objects. See the
 * [ES spec](http://ecma-international.org/ecma-262/7.0/#sec-object.keys)
 * for more details.
 *
 * @static
 * @since 0.1.0
 * @memberOf _
 * @category Object
 * @param {Object} object The object to query.
 * @returns {Array} Returns the array of property names.
 * @example
 *
 * function Foo() {
 *   this.a = 1;
 *   this.b = 2;
 * }
 *
 * Foo.prototype.c = 3;
 *
 * _.keys(new Foo);
 * // => ['a', 'b'] (iteration order is not guaranteed)
 *
 * _.keys('hi');
 * // => ['0', '1']
 */
function keys(object) {
  return isArrayLike(object) ? arrayLikeKeys(object) : baseKeys(object);
}

/**
 * Creates an array of own enumerable string keyed-value pairs for `object`
 * which can be consumed by `_.fromPairs`. If `object` is a map or set, its
 * entries are returned.
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @alias entries
 * @category Object
 * @param {Object} object The object to query.
 * @returns {Array} Returns the key-value pairs.
 * @example
 *
 * function Foo() {
 *   this.a = 1;
 *   this.b = 2;
 * }
 *
 * Foo.prototype.c = 3;
 *
 * _.toPairs(new Foo);
 * // => [['a', 1], ['b', 2]] (iteration order is not guaranteed)
 */
var toPairs = createToPairs(keys);

module.exports = toPairs;


/***/ }),

/***/ 618:
/***/ ((module, exports, __webpack_require__) => {

/* module decorator */ module = __webpack_require__.nmd(module);
/**
 * lodash (Custom Build) <https://lodash.com/>
 * Build: `lodash modularize exports="npm" -o ./`
 * Copyright jQuery Foundation and other contributors <https://jquery.org/>
 * Released under MIT license <https://lodash.com/license>
 * Based on Underscore.js 1.8.3 <http://underscorejs.org/LICENSE>
 * Copyright Jeremy Ashkenas, DocumentCloud and Investigative Reporters & Editors
 */

/** Used as the size to enable large array optimizations. */
var LARGE_ARRAY_SIZE = 200;

/** Used as the `TypeError` message for "Functions" methods. */
var FUNC_ERROR_TEXT = 'Expected a function';

/** Used to stand-in for `undefined` hash values. */
var HASH_UNDEFINED = '__lodash_hash_undefined__';

/** Used to compose bitmasks for comparison styles. */
var UNORDERED_COMPARE_FLAG = 1,
    PARTIAL_COMPARE_FLAG = 2;

/** Used as references for various `Number` constants. */
var INFINITY = 1 / 0,
    MAX_SAFE_INTEGER = 9007199254740991;

/** `Object#toString` result references. */
var argsTag = '[object Arguments]',
    arrayTag = '[object Array]',
    boolTag = '[object Boolean]',
    dateTag = '[object Date]',
    errorTag = '[object Error]',
    funcTag = '[object Function]',
    genTag = '[object GeneratorFunction]',
    mapTag = '[object Map]',
    numberTag = '[object Number]',
    objectTag = '[object Object]',
    promiseTag = '[object Promise]',
    regexpTag = '[object RegExp]',
    setTag = '[object Set]',
    stringTag = '[object String]',
    symbolTag = '[object Symbol]',
    weakMapTag = '[object WeakMap]';

var arrayBufferTag = '[object ArrayBuffer]',
    dataViewTag = '[object DataView]',
    float32Tag = '[object Float32Array]',
    float64Tag = '[object Float64Array]',
    int8Tag = '[object Int8Array]',
    int16Tag = '[object Int16Array]',
    int32Tag = '[object Int32Array]',
    uint8Tag = '[object Uint8Array]',
    uint8ClampedTag = '[object Uint8ClampedArray]',
    uint16Tag = '[object Uint16Array]',
    uint32Tag = '[object Uint32Array]';

/** Used to match property names within property paths. */
var reIsDeepProp = /\.|\[(?:[^[\]]*|(["'])(?:(?!\1)[^\\]|\\.)*?\1)\]/,
    reIsPlainProp = /^\w*$/,
    reLeadingDot = /^\./,
    rePropName = /[^.[\]]+|\[(?:(-?\d+(?:\.\d+)?)|(["'])((?:(?!\2)[^\\]|\\.)*?)\2)\]|(?=(?:\.|\[\])(?:\.|\[\]|$))/g;

/**
 * Used to match `RegExp`
 * [syntax characters](http://ecma-international.org/ecma-262/7.0/#sec-patterns).
 */
var reRegExpChar = /[\\^$.*+?()[\]{}|]/g;

/** Used to match backslashes in property paths. */
var reEscapeChar = /\\(\\)?/g;

/** Used to detect host constructors (Safari). */
var reIsHostCtor = /^\[object .+?Constructor\]$/;

/** Used to detect unsigned integer values. */
var reIsUint = /^(?:0|[1-9]\d*)$/;

/** Used to identify `toStringTag` values of typed arrays. */
var typedArrayTags = {};
typedArrayTags[float32Tag] = typedArrayTags[float64Tag] =
typedArrayTags[int8Tag] = typedArrayTags[int16Tag] =
typedArrayTags[int32Tag] = typedArrayTags[uint8Tag] =
typedArrayTags[uint8ClampedTag] = typedArrayTags[uint16Tag] =
typedArrayTags[uint32Tag] = true;
typedArrayTags[argsTag] = typedArrayTags[arrayTag] =
typedArrayTags[arrayBufferTag] = typedArrayTags[boolTag] =
typedArrayTags[dataViewTag] = typedArrayTags[dateTag] =
typedArrayTags[errorTag] = typedArrayTags[funcTag] =
typedArrayTags[mapTag] = typedArrayTags[numberTag] =
typedArrayTags[objectTag] = typedArrayTags[regexpTag] =
typedArrayTags[setTag] = typedArrayTags[stringTag] =
typedArrayTags[weakMapTag] = false;

/** Detect free variable `global` from Node.js. */
var freeGlobal = typeof __webpack_require__.g == 'object' && __webpack_require__.g && __webpack_require__.g.Object === Object && __webpack_require__.g;

/** Detect free variable `self`. */
var freeSelf = typeof self == 'object' && self && self.Object === Object && self;

/** Used as a reference to the global object. */
var root = freeGlobal || freeSelf || Function('return this')();

/** Detect free variable `exports`. */
var freeExports =  true && exports && !exports.nodeType && exports;

/** Detect free variable `module`. */
var freeModule = freeExports && "object" == 'object' && module && !module.nodeType && module;

/** Detect the popular CommonJS extension `module.exports`. */
var moduleExports = freeModule && freeModule.exports === freeExports;

/** Detect free variable `process` from Node.js. */
var freeProcess = moduleExports && freeGlobal.process;

/** Used to access faster Node.js helpers. */
var nodeUtil = (function() {
  try {
    return freeProcess && freeProcess.binding('util');
  } catch (e) {}
}());

/* Node.js helper references. */
var nodeIsTypedArray = nodeUtil && nodeUtil.isTypedArray;

/**
 * A specialized version of `_.includes` for arrays without support for
 * specifying an index to search from.
 *
 * @private
 * @param {Array} [array] The array to inspect.
 * @param {*} target The value to search for.
 * @returns {boolean} Returns `true` if `target` is found, else `false`.
 */
function arrayIncludes(array, value) {
  var length = array ? array.length : 0;
  return !!length && baseIndexOf(array, value, 0) > -1;
}

/**
 * This function is like `arrayIncludes` except that it accepts a comparator.
 *
 * @private
 * @param {Array} [array] The array to inspect.
 * @param {*} target The value to search for.
 * @param {Function} comparator The comparator invoked per element.
 * @returns {boolean} Returns `true` if `target` is found, else `false`.
 */
function arrayIncludesWith(array, value, comparator) {
  var index = -1,
      length = array ? array.length : 0;

  while (++index < length) {
    if (comparator(value, array[index])) {
      return true;
    }
  }
  return false;
}

/**
 * A specialized version of `_.some` for arrays without support for iteratee
 * shorthands.
 *
 * @private
 * @param {Array} [array] The array to iterate over.
 * @param {Function} predicate The function invoked per iteration.
 * @returns {boolean} Returns `true` if any element passes the predicate check,
 *  else `false`.
 */
function arraySome(array, predicate) {
  var index = -1,
      length = array ? array.length : 0;

  while (++index < length) {
    if (predicate(array[index], index, array)) {
      return true;
    }
  }
  return false;
}

/**
 * The base implementation of `_.findIndex` and `_.findLastIndex` without
 * support for iteratee shorthands.
 *
 * @private
 * @param {Array} array The array to inspect.
 * @param {Function} predicate The function invoked per iteration.
 * @param {number} fromIndex The index to search from.
 * @param {boolean} [fromRight] Specify iterating from right to left.
 * @returns {number} Returns the index of the matched value, else `-1`.
 */
function baseFindIndex(array, predicate, fromIndex, fromRight) {
  var length = array.length,
      index = fromIndex + (fromRight ? 1 : -1);

  while ((fromRight ? index-- : ++index < length)) {
    if (predicate(array[index], index, array)) {
      return index;
    }
  }
  return -1;
}

/**
 * The base implementation of `_.indexOf` without `fromIndex` bounds checks.
 *
 * @private
 * @param {Array} array The array to inspect.
 * @param {*} value The value to search for.
 * @param {number} fromIndex The index to search from.
 * @returns {number} Returns the index of the matched value, else `-1`.
 */
function baseIndexOf(array, value, fromIndex) {
  if (value !== value) {
    return baseFindIndex(array, baseIsNaN, fromIndex);
  }
  var index = fromIndex - 1,
      length = array.length;

  while (++index < length) {
    if (array[index] === value) {
      return index;
    }
  }
  return -1;
}

/**
 * The base implementation of `_.isNaN` without support for number objects.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is `NaN`, else `false`.
 */
function baseIsNaN(value) {
  return value !== value;
}

/**
 * The base implementation of `_.property` without support for deep paths.
 *
 * @private
 * @param {string} key The key of the property to get.
 * @returns {Function} Returns the new accessor function.
 */
function baseProperty(key) {
  return function(object) {
    return object == null ? undefined : object[key];
  };
}

/**
 * The base implementation of `_.times` without support for iteratee shorthands
 * or max array length checks.
 *
 * @private
 * @param {number} n The number of times to invoke `iteratee`.
 * @param {Function} iteratee The function invoked per iteration.
 * @returns {Array} Returns the array of results.
 */
function baseTimes(n, iteratee) {
  var index = -1,
      result = Array(n);

  while (++index < n) {
    result[index] = iteratee(index);
  }
  return result;
}

/**
 * The base implementation of `_.unary` without support for storing metadata.
 *
 * @private
 * @param {Function} func The function to cap arguments for.
 * @returns {Function} Returns the new capped function.
 */
function baseUnary(func) {
  return function(value) {
    return func(value);
  };
}

/**
 * Checks if a cache value for `key` exists.
 *
 * @private
 * @param {Object} cache The cache to query.
 * @param {string} key The key of the entry to check.
 * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
 */
function cacheHas(cache, key) {
  return cache.has(key);
}

/**
 * Gets the value at `key` of `object`.
 *
 * @private
 * @param {Object} [object] The object to query.
 * @param {string} key The key of the property to get.
 * @returns {*} Returns the property value.
 */
function getValue(object, key) {
  return object == null ? undefined : object[key];
}

/**
 * Checks if `value` is a host object in IE < 9.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a host object, else `false`.
 */
function isHostObject(value) {
  // Many host objects are `Object` objects that can coerce to strings
  // despite having improperly defined `toString` methods.
  var result = false;
  if (value != null && typeof value.toString != 'function') {
    try {
      result = !!(value + '');
    } catch (e) {}
  }
  return result;
}

/**
 * Converts `map` to its key-value pairs.
 *
 * @private
 * @param {Object} map The map to convert.
 * @returns {Array} Returns the key-value pairs.
 */
function mapToArray(map) {
  var index = -1,
      result = Array(map.size);

  map.forEach(function(value, key) {
    result[++index] = [key, value];
  });
  return result;
}

/**
 * Creates a unary function that invokes `func` with its argument transformed.
 *
 * @private
 * @param {Function} func The function to wrap.
 * @param {Function} transform The argument transform.
 * @returns {Function} Returns the new function.
 */
function overArg(func, transform) {
  return function(arg) {
    return func(transform(arg));
  };
}

/**
 * Converts `set` to an array of its values.
 *
 * @private
 * @param {Object} set The set to convert.
 * @returns {Array} Returns the values.
 */
function setToArray(set) {
  var index = -1,
      result = Array(set.size);

  set.forEach(function(value) {
    result[++index] = value;
  });
  return result;
}

/** Used for built-in method references. */
var arrayProto = Array.prototype,
    funcProto = Function.prototype,
    objectProto = Object.prototype;

/** Used to detect overreaching core-js shims. */
var coreJsData = root['__core-js_shared__'];

/** Used to detect methods masquerading as native. */
var maskSrcKey = (function() {
  var uid = /[^.]+$/.exec(coreJsData && coreJsData.keys && coreJsData.keys.IE_PROTO || '');
  return uid ? ('Symbol(src)_1.' + uid) : '';
}());

/** Used to resolve the decompiled source of functions. */
var funcToString = funcProto.toString;

/** Used to check objects for own properties. */
var hasOwnProperty = objectProto.hasOwnProperty;

/**
 * Used to resolve the
 * [`toStringTag`](http://ecma-international.org/ecma-262/7.0/#sec-object.prototype.tostring)
 * of values.
 */
var objectToString = objectProto.toString;

/** Used to detect if a method is native. */
var reIsNative = RegExp('^' +
  funcToString.call(hasOwnProperty).replace(reRegExpChar, '\\$&')
  .replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g, '$1.*?') + '$'
);

/** Built-in value references. */
var Symbol = root.Symbol,
    Uint8Array = root.Uint8Array,
    propertyIsEnumerable = objectProto.propertyIsEnumerable,
    splice = arrayProto.splice;

/* Built-in method references for those with the same name as other `lodash` methods. */
var nativeKeys = overArg(Object.keys, Object);

/* Built-in method references that are verified to be native. */
var DataView = getNative(root, 'DataView'),
    Map = getNative(root, 'Map'),
    Promise = getNative(root, 'Promise'),
    Set = getNative(root, 'Set'),
    WeakMap = getNative(root, 'WeakMap'),
    nativeCreate = getNative(Object, 'create');

/** Used to detect maps, sets, and weakmaps. */
var dataViewCtorString = toSource(DataView),
    mapCtorString = toSource(Map),
    promiseCtorString = toSource(Promise),
    setCtorString = toSource(Set),
    weakMapCtorString = toSource(WeakMap);

/** Used to convert symbols to primitives and strings. */
var symbolProto = Symbol ? Symbol.prototype : undefined,
    symbolValueOf = symbolProto ? symbolProto.valueOf : undefined,
    symbolToString = symbolProto ? symbolProto.toString : undefined;

/**
 * Creates a hash object.
 *
 * @private
 * @constructor
 * @param {Array} [entries] The key-value pairs to cache.
 */
function Hash(entries) {
  var index = -1,
      length = entries ? entries.length : 0;

  this.clear();
  while (++index < length) {
    var entry = entries[index];
    this.set(entry[0], entry[1]);
  }
}

/**
 * Removes all key-value entries from the hash.
 *
 * @private
 * @name clear
 * @memberOf Hash
 */
function hashClear() {
  this.__data__ = nativeCreate ? nativeCreate(null) : {};
}

/**
 * Removes `key` and its value from the hash.
 *
 * @private
 * @name delete
 * @memberOf Hash
 * @param {Object} hash The hash to modify.
 * @param {string} key The key of the value to remove.
 * @returns {boolean} Returns `true` if the entry was removed, else `false`.
 */
function hashDelete(key) {
  return this.has(key) && delete this.__data__[key];
}

/**
 * Gets the hash value for `key`.
 *
 * @private
 * @name get
 * @memberOf Hash
 * @param {string} key The key of the value to get.
 * @returns {*} Returns the entry value.
 */
function hashGet(key) {
  var data = this.__data__;
  if (nativeCreate) {
    var result = data[key];
    return result === HASH_UNDEFINED ? undefined : result;
  }
  return hasOwnProperty.call(data, key) ? data[key] : undefined;
}

/**
 * Checks if a hash value for `key` exists.
 *
 * @private
 * @name has
 * @memberOf Hash
 * @param {string} key The key of the entry to check.
 * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
 */
function hashHas(key) {
  var data = this.__data__;
  return nativeCreate ? data[key] !== undefined : hasOwnProperty.call(data, key);
}

/**
 * Sets the hash `key` to `value`.
 *
 * @private
 * @name set
 * @memberOf Hash
 * @param {string} key The key of the value to set.
 * @param {*} value The value to set.
 * @returns {Object} Returns the hash instance.
 */
function hashSet(key, value) {
  var data = this.__data__;
  data[key] = (nativeCreate && value === undefined) ? HASH_UNDEFINED : value;
  return this;
}

// Add methods to `Hash`.
Hash.prototype.clear = hashClear;
Hash.prototype['delete'] = hashDelete;
Hash.prototype.get = hashGet;
Hash.prototype.has = hashHas;
Hash.prototype.set = hashSet;

/**
 * Creates an list cache object.
 *
 * @private
 * @constructor
 * @param {Array} [entries] The key-value pairs to cache.
 */
function ListCache(entries) {
  var index = -1,
      length = entries ? entries.length : 0;

  this.clear();
  while (++index < length) {
    var entry = entries[index];
    this.set(entry[0], entry[1]);
  }
}

/**
 * Removes all key-value entries from the list cache.
 *
 * @private
 * @name clear
 * @memberOf ListCache
 */
function listCacheClear() {
  this.__data__ = [];
}

/**
 * Removes `key` and its value from the list cache.
 *
 * @private
 * @name delete
 * @memberOf ListCache
 * @param {string} key The key of the value to remove.
 * @returns {boolean} Returns `true` if the entry was removed, else `false`.
 */
function listCacheDelete(key) {
  var data = this.__data__,
      index = assocIndexOf(data, key);

  if (index < 0) {
    return false;
  }
  var lastIndex = data.length - 1;
  if (index == lastIndex) {
    data.pop();
  } else {
    splice.call(data, index, 1);
  }
  return true;
}

/**
 * Gets the list cache value for `key`.
 *
 * @private
 * @name get
 * @memberOf ListCache
 * @param {string} key The key of the value to get.
 * @returns {*} Returns the entry value.
 */
function listCacheGet(key) {
  var data = this.__data__,
      index = assocIndexOf(data, key);

  return index < 0 ? undefined : data[index][1];
}

/**
 * Checks if a list cache value for `key` exists.
 *
 * @private
 * @name has
 * @memberOf ListCache
 * @param {string} key The key of the entry to check.
 * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
 */
function listCacheHas(key) {
  return assocIndexOf(this.__data__, key) > -1;
}

/**
 * Sets the list cache `key` to `value`.
 *
 * @private
 * @name set
 * @memberOf ListCache
 * @param {string} key The key of the value to set.
 * @param {*} value The value to set.
 * @returns {Object} Returns the list cache instance.
 */
function listCacheSet(key, value) {
  var data = this.__data__,
      index = assocIndexOf(data, key);

  if (index < 0) {
    data.push([key, value]);
  } else {
    data[index][1] = value;
  }
  return this;
}

// Add methods to `ListCache`.
ListCache.prototype.clear = listCacheClear;
ListCache.prototype['delete'] = listCacheDelete;
ListCache.prototype.get = listCacheGet;
ListCache.prototype.has = listCacheHas;
ListCache.prototype.set = listCacheSet;

/**
 * Creates a map cache object to store key-value pairs.
 *
 * @private
 * @constructor
 * @param {Array} [entries] The key-value pairs to cache.
 */
function MapCache(entries) {
  var index = -1,
      length = entries ? entries.length : 0;

  this.clear();
  while (++index < length) {
    var entry = entries[index];
    this.set(entry[0], entry[1]);
  }
}

/**
 * Removes all key-value entries from the map.
 *
 * @private
 * @name clear
 * @memberOf MapCache
 */
function mapCacheClear() {
  this.__data__ = {
    'hash': new Hash,
    'map': new (Map || ListCache),
    'string': new Hash
  };
}

/**
 * Removes `key` and its value from the map.
 *
 * @private
 * @name delete
 * @memberOf MapCache
 * @param {string} key The key of the value to remove.
 * @returns {boolean} Returns `true` if the entry was removed, else `false`.
 */
function mapCacheDelete(key) {
  return getMapData(this, key)['delete'](key);
}

/**
 * Gets the map value for `key`.
 *
 * @private
 * @name get
 * @memberOf MapCache
 * @param {string} key The key of the value to get.
 * @returns {*} Returns the entry value.
 */
function mapCacheGet(key) {
  return getMapData(this, key).get(key);
}

/**
 * Checks if a map value for `key` exists.
 *
 * @private
 * @name has
 * @memberOf MapCache
 * @param {string} key The key of the entry to check.
 * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
 */
function mapCacheHas(key) {
  return getMapData(this, key).has(key);
}

/**
 * Sets the map `key` to `value`.
 *
 * @private
 * @name set
 * @memberOf MapCache
 * @param {string} key The key of the value to set.
 * @param {*} value The value to set.
 * @returns {Object} Returns the map cache instance.
 */
function mapCacheSet(key, value) {
  getMapData(this, key).set(key, value);
  return this;
}

// Add methods to `MapCache`.
MapCache.prototype.clear = mapCacheClear;
MapCache.prototype['delete'] = mapCacheDelete;
MapCache.prototype.get = mapCacheGet;
MapCache.prototype.has = mapCacheHas;
MapCache.prototype.set = mapCacheSet;

/**
 *
 * Creates an array cache object to store unique values.
 *
 * @private
 * @constructor
 * @param {Array} [values] The values to cache.
 */
function SetCache(values) {
  var index = -1,
      length = values ? values.length : 0;

  this.__data__ = new MapCache;
  while (++index < length) {
    this.add(values[index]);
  }
}

/**
 * Adds `value` to the array cache.
 *
 * @private
 * @name add
 * @memberOf SetCache
 * @alias push
 * @param {*} value The value to cache.
 * @returns {Object} Returns the cache instance.
 */
function setCacheAdd(value) {
  this.__data__.set(value, HASH_UNDEFINED);
  return this;
}

/**
 * Checks if `value` is in the array cache.
 *
 * @private
 * @name has
 * @memberOf SetCache
 * @param {*} value The value to search for.
 * @returns {number} Returns `true` if `value` is found, else `false`.
 */
function setCacheHas(value) {
  return this.__data__.has(value);
}

// Add methods to `SetCache`.
SetCache.prototype.add = SetCache.prototype.push = setCacheAdd;
SetCache.prototype.has = setCacheHas;

/**
 * Creates a stack cache object to store key-value pairs.
 *
 * @private
 * @constructor
 * @param {Array} [entries] The key-value pairs to cache.
 */
function Stack(entries) {
  this.__data__ = new ListCache(entries);
}

/**
 * Removes all key-value entries from the stack.
 *
 * @private
 * @name clear
 * @memberOf Stack
 */
function stackClear() {
  this.__data__ = new ListCache;
}

/**
 * Removes `key` and its value from the stack.
 *
 * @private
 * @name delete
 * @memberOf Stack
 * @param {string} key The key of the value to remove.
 * @returns {boolean} Returns `true` if the entry was removed, else `false`.
 */
function stackDelete(key) {
  return this.__data__['delete'](key);
}

/**
 * Gets the stack value for `key`.
 *
 * @private
 * @name get
 * @memberOf Stack
 * @param {string} key The key of the value to get.
 * @returns {*} Returns the entry value.
 */
function stackGet(key) {
  return this.__data__.get(key);
}

/**
 * Checks if a stack value for `key` exists.
 *
 * @private
 * @name has
 * @memberOf Stack
 * @param {string} key The key of the entry to check.
 * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
 */
function stackHas(key) {
  return this.__data__.has(key);
}

/**
 * Sets the stack `key` to `value`.
 *
 * @private
 * @name set
 * @memberOf Stack
 * @param {string} key The key of the value to set.
 * @param {*} value The value to set.
 * @returns {Object} Returns the stack cache instance.
 */
function stackSet(key, value) {
  var cache = this.__data__;
  if (cache instanceof ListCache) {
    var pairs = cache.__data__;
    if (!Map || (pairs.length < LARGE_ARRAY_SIZE - 1)) {
      pairs.push([key, value]);
      return this;
    }
    cache = this.__data__ = new MapCache(pairs);
  }
  cache.set(key, value);
  return this;
}

// Add methods to `Stack`.
Stack.prototype.clear = stackClear;
Stack.prototype['delete'] = stackDelete;
Stack.prototype.get = stackGet;
Stack.prototype.has = stackHas;
Stack.prototype.set = stackSet;

/**
 * Creates an array of the enumerable property names of the array-like `value`.
 *
 * @private
 * @param {*} value The value to query.
 * @param {boolean} inherited Specify returning inherited property names.
 * @returns {Array} Returns the array of property names.
 */
function arrayLikeKeys(value, inherited) {
  // Safari 8.1 makes `arguments.callee` enumerable in strict mode.
  // Safari 9 makes `arguments.length` enumerable in strict mode.
  var result = (isArray(value) || isArguments(value))
    ? baseTimes(value.length, String)
    : [];

  var length = result.length,
      skipIndexes = !!length;

  for (var key in value) {
    if ((inherited || hasOwnProperty.call(value, key)) &&
        !(skipIndexes && (key == 'length' || isIndex(key, length)))) {
      result.push(key);
    }
  }
  return result;
}

/**
 * Gets the index at which the `key` is found in `array` of key-value pairs.
 *
 * @private
 * @param {Array} array The array to inspect.
 * @param {*} key The key to search for.
 * @returns {number} Returns the index of the matched value, else `-1`.
 */
function assocIndexOf(array, key) {
  var length = array.length;
  while (length--) {
    if (eq(array[length][0], key)) {
      return length;
    }
  }
  return -1;
}

/**
 * The base implementation of `_.get` without support for default values.
 *
 * @private
 * @param {Object} object The object to query.
 * @param {Array|string} path The path of the property to get.
 * @returns {*} Returns the resolved value.
 */
function baseGet(object, path) {
  path = isKey(path, object) ? [path] : castPath(path);

  var index = 0,
      length = path.length;

  while (object != null && index < length) {
    object = object[toKey(path[index++])];
  }
  return (index && index == length) ? object : undefined;
}

/**
 * The base implementation of `getTag`.
 *
 * @private
 * @param {*} value The value to query.
 * @returns {string} Returns the `toStringTag`.
 */
function baseGetTag(value) {
  return objectToString.call(value);
}

/**
 * The base implementation of `_.hasIn` without support for deep paths.
 *
 * @private
 * @param {Object} [object] The object to query.
 * @param {Array|string} key The key to check.
 * @returns {boolean} Returns `true` if `key` exists, else `false`.
 */
function baseHasIn(object, key) {
  return object != null && key in Object(object);
}

/**
 * The base implementation of `_.isEqual` which supports partial comparisons
 * and tracks traversed objects.
 *
 * @private
 * @param {*} value The value to compare.
 * @param {*} other The other value to compare.
 * @param {Function} [customizer] The function to customize comparisons.
 * @param {boolean} [bitmask] The bitmask of comparison flags.
 *  The bitmask may be composed of the following flags:
 *     1 - Unordered comparison
 *     2 - Partial comparison
 * @param {Object} [stack] Tracks traversed `value` and `other` objects.
 * @returns {boolean} Returns `true` if the values are equivalent, else `false`.
 */
function baseIsEqual(value, other, customizer, bitmask, stack) {
  if (value === other) {
    return true;
  }
  if (value == null || other == null || (!isObject(value) && !isObjectLike(other))) {
    return value !== value && other !== other;
  }
  return baseIsEqualDeep(value, other, baseIsEqual, customizer, bitmask, stack);
}

/**
 * A specialized version of `baseIsEqual` for arrays and objects which performs
 * deep comparisons and tracks traversed objects enabling objects with circular
 * references to be compared.
 *
 * @private
 * @param {Object} object The object to compare.
 * @param {Object} other The other object to compare.
 * @param {Function} equalFunc The function to determine equivalents of values.
 * @param {Function} [customizer] The function to customize comparisons.
 * @param {number} [bitmask] The bitmask of comparison flags. See `baseIsEqual`
 *  for more details.
 * @param {Object} [stack] Tracks traversed `object` and `other` objects.
 * @returns {boolean} Returns `true` if the objects are equivalent, else `false`.
 */
function baseIsEqualDeep(object, other, equalFunc, customizer, bitmask, stack) {
  var objIsArr = isArray(object),
      othIsArr = isArray(other),
      objTag = arrayTag,
      othTag = arrayTag;

  if (!objIsArr) {
    objTag = getTag(object);
    objTag = objTag == argsTag ? objectTag : objTag;
  }
  if (!othIsArr) {
    othTag = getTag(other);
    othTag = othTag == argsTag ? objectTag : othTag;
  }
  var objIsObj = objTag == objectTag && !isHostObject(object),
      othIsObj = othTag == objectTag && !isHostObject(other),
      isSameTag = objTag == othTag;

  if (isSameTag && !objIsObj) {
    stack || (stack = new Stack);
    return (objIsArr || isTypedArray(object))
      ? equalArrays(object, other, equalFunc, customizer, bitmask, stack)
      : equalByTag(object, other, objTag, equalFunc, customizer, bitmask, stack);
  }
  if (!(bitmask & PARTIAL_COMPARE_FLAG)) {
    var objIsWrapped = objIsObj && hasOwnProperty.call(object, '__wrapped__'),
        othIsWrapped = othIsObj && hasOwnProperty.call(other, '__wrapped__');

    if (objIsWrapped || othIsWrapped) {
      var objUnwrapped = objIsWrapped ? object.value() : object,
          othUnwrapped = othIsWrapped ? other.value() : other;

      stack || (stack = new Stack);
      return equalFunc(objUnwrapped, othUnwrapped, customizer, bitmask, stack);
    }
  }
  if (!isSameTag) {
    return false;
  }
  stack || (stack = new Stack);
  return equalObjects(object, other, equalFunc, customizer, bitmask, stack);
}

/**
 * The base implementation of `_.isMatch` without support for iteratee shorthands.
 *
 * @private
 * @param {Object} object The object to inspect.
 * @param {Object} source The object of property values to match.
 * @param {Array} matchData The property names, values, and compare flags to match.
 * @param {Function} [customizer] The function to customize comparisons.
 * @returns {boolean} Returns `true` if `object` is a match, else `false`.
 */
function baseIsMatch(object, source, matchData, customizer) {
  var index = matchData.length,
      length = index,
      noCustomizer = !customizer;

  if (object == null) {
    return !length;
  }
  object = Object(object);
  while (index--) {
    var data = matchData[index];
    if ((noCustomizer && data[2])
          ? data[1] !== object[data[0]]
          : !(data[0] in object)
        ) {
      return false;
    }
  }
  while (++index < length) {
    data = matchData[index];
    var key = data[0],
        objValue = object[key],
        srcValue = data[1];

    if (noCustomizer && data[2]) {
      if (objValue === undefined && !(key in object)) {
        return false;
      }
    } else {
      var stack = new Stack;
      if (customizer) {
        var result = customizer(objValue, srcValue, key, object, source, stack);
      }
      if (!(result === undefined
            ? baseIsEqual(srcValue, objValue, customizer, UNORDERED_COMPARE_FLAG | PARTIAL_COMPARE_FLAG, stack)
            : result
          )) {
        return false;
      }
    }
  }
  return true;
}

/**
 * The base implementation of `_.isNative` without bad shim checks.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a native function,
 *  else `false`.
 */
function baseIsNative(value) {
  if (!isObject(value) || isMasked(value)) {
    return false;
  }
  var pattern = (isFunction(value) || isHostObject(value)) ? reIsNative : reIsHostCtor;
  return pattern.test(toSource(value));
}

/**
 * The base implementation of `_.isTypedArray` without Node.js optimizations.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a typed array, else `false`.
 */
function baseIsTypedArray(value) {
  return isObjectLike(value) &&
    isLength(value.length) && !!typedArrayTags[objectToString.call(value)];
}

/**
 * The base implementation of `_.iteratee`.
 *
 * @private
 * @param {*} [value=_.identity] The value to convert to an iteratee.
 * @returns {Function} Returns the iteratee.
 */
function baseIteratee(value) {
  // Don't store the `typeof` result in a variable to avoid a JIT bug in Safari 9.
  // See https://bugs.webkit.org/show_bug.cgi?id=156034 for more details.
  if (typeof value == 'function') {
    return value;
  }
  if (value == null) {
    return identity;
  }
  if (typeof value == 'object') {
    return isArray(value)
      ? baseMatchesProperty(value[0], value[1])
      : baseMatches(value);
  }
  return property(value);
}

/**
 * The base implementation of `_.keys` which doesn't treat sparse arrays as dense.
 *
 * @private
 * @param {Object} object The object to query.
 * @returns {Array} Returns the array of property names.
 */
function baseKeys(object) {
  if (!isPrototype(object)) {
    return nativeKeys(object);
  }
  var result = [];
  for (var key in Object(object)) {
    if (hasOwnProperty.call(object, key) && key != 'constructor') {
      result.push(key);
    }
  }
  return result;
}

/**
 * The base implementation of `_.matches` which doesn't clone `source`.
 *
 * @private
 * @param {Object} source The object of property values to match.
 * @returns {Function} Returns the new spec function.
 */
function baseMatches(source) {
  var matchData = getMatchData(source);
  if (matchData.length == 1 && matchData[0][2]) {
    return matchesStrictComparable(matchData[0][0], matchData[0][1]);
  }
  return function(object) {
    return object === source || baseIsMatch(object, source, matchData);
  };
}

/**
 * The base implementation of `_.matchesProperty` which doesn't clone `srcValue`.
 *
 * @private
 * @param {string} path The path of the property to get.
 * @param {*} srcValue The value to match.
 * @returns {Function} Returns the new spec function.
 */
function baseMatchesProperty(path, srcValue) {
  if (isKey(path) && isStrictComparable(srcValue)) {
    return matchesStrictComparable(toKey(path), srcValue);
  }
  return function(object) {
    var objValue = get(object, path);
    return (objValue === undefined && objValue === srcValue)
      ? hasIn(object, path)
      : baseIsEqual(srcValue, objValue, undefined, UNORDERED_COMPARE_FLAG | PARTIAL_COMPARE_FLAG);
  };
}

/**
 * A specialized version of `baseProperty` which supports deep paths.
 *
 * @private
 * @param {Array|string} path The path of the property to get.
 * @returns {Function} Returns the new accessor function.
 */
function basePropertyDeep(path) {
  return function(object) {
    return baseGet(object, path);
  };
}

/**
 * The base implementation of `_.toString` which doesn't convert nullish
 * values to empty strings.
 *
 * @private
 * @param {*} value The value to process.
 * @returns {string} Returns the string.
 */
function baseToString(value) {
  // Exit early for strings to avoid a performance hit in some environments.
  if (typeof value == 'string') {
    return value;
  }
  if (isSymbol(value)) {
    return symbolToString ? symbolToString.call(value) : '';
  }
  var result = (value + '');
  return (result == '0' && (1 / value) == -INFINITY) ? '-0' : result;
}

/**
 * The base implementation of `_.uniqBy` without support for iteratee shorthands.
 *
 * @private
 * @param {Array} array The array to inspect.
 * @param {Function} [iteratee] The iteratee invoked per element.
 * @param {Function} [comparator] The comparator invoked per element.
 * @returns {Array} Returns the new duplicate free array.
 */
function baseUniq(array, iteratee, comparator) {
  var index = -1,
      includes = arrayIncludes,
      length = array.length,
      isCommon = true,
      result = [],
      seen = result;

  if (comparator) {
    isCommon = false;
    includes = arrayIncludesWith;
  }
  else if (length >= LARGE_ARRAY_SIZE) {
    var set = iteratee ? null : createSet(array);
    if (set) {
      return setToArray(set);
    }
    isCommon = false;
    includes = cacheHas;
    seen = new SetCache;
  }
  else {
    seen = iteratee ? [] : result;
  }
  outer:
  while (++index < length) {
    var value = array[index],
        computed = iteratee ? iteratee(value) : value;

    value = (comparator || value !== 0) ? value : 0;
    if (isCommon && computed === computed) {
      var seenIndex = seen.length;
      while (seenIndex--) {
        if (seen[seenIndex] === computed) {
          continue outer;
        }
      }
      if (iteratee) {
        seen.push(computed);
      }
      result.push(value);
    }
    else if (!includes(seen, computed, comparator)) {
      if (seen !== result) {
        seen.push(computed);
      }
      result.push(value);
    }
  }
  return result;
}

/**
 * Casts `value` to a path array if it's not one.
 *
 * @private
 * @param {*} value The value to inspect.
 * @returns {Array} Returns the cast property path array.
 */
function castPath(value) {
  return isArray(value) ? value : stringToPath(value);
}

/**
 * Creates a set object of `values`.
 *
 * @private
 * @param {Array} values The values to add to the set.
 * @returns {Object} Returns the new set.
 */
var createSet = !(Set && (1 / setToArray(new Set([,-0]))[1]) == INFINITY) ? noop : function(values) {
  return new Set(values);
};

/**
 * A specialized version of `baseIsEqualDeep` for arrays with support for
 * partial deep comparisons.
 *
 * @private
 * @param {Array} array The array to compare.
 * @param {Array} other The other array to compare.
 * @param {Function} equalFunc The function to determine equivalents of values.
 * @param {Function} customizer The function to customize comparisons.
 * @param {number} bitmask The bitmask of comparison flags. See `baseIsEqual`
 *  for more details.
 * @param {Object} stack Tracks traversed `array` and `other` objects.
 * @returns {boolean} Returns `true` if the arrays are equivalent, else `false`.
 */
function equalArrays(array, other, equalFunc, customizer, bitmask, stack) {
  var isPartial = bitmask & PARTIAL_COMPARE_FLAG,
      arrLength = array.length,
      othLength = other.length;

  if (arrLength != othLength && !(isPartial && othLength > arrLength)) {
    return false;
  }
  // Assume cyclic values are equal.
  var stacked = stack.get(array);
  if (stacked && stack.get(other)) {
    return stacked == other;
  }
  var index = -1,
      result = true,
      seen = (bitmask & UNORDERED_COMPARE_FLAG) ? new SetCache : undefined;

  stack.set(array, other);
  stack.set(other, array);

  // Ignore non-index properties.
  while (++index < arrLength) {
    var arrValue = array[index],
        othValue = other[index];

    if (customizer) {
      var compared = isPartial
        ? customizer(othValue, arrValue, index, other, array, stack)
        : customizer(arrValue, othValue, index, array, other, stack);
    }
    if (compared !== undefined) {
      if (compared) {
        continue;
      }
      result = false;
      break;
    }
    // Recursively compare arrays (susceptible to call stack limits).
    if (seen) {
      if (!arraySome(other, function(othValue, othIndex) {
            if (!seen.has(othIndex) &&
                (arrValue === othValue || equalFunc(arrValue, othValue, customizer, bitmask, stack))) {
              return seen.add(othIndex);
            }
          })) {
        result = false;
        break;
      }
    } else if (!(
          arrValue === othValue ||
            equalFunc(arrValue, othValue, customizer, bitmask, stack)
        )) {
      result = false;
      break;
    }
  }
  stack['delete'](array);
  stack['delete'](other);
  return result;
}

/**
 * A specialized version of `baseIsEqualDeep` for comparing objects of
 * the same `toStringTag`.
 *
 * **Note:** This function only supports comparing values with tags of
 * `Boolean`, `Date`, `Error`, `Number`, `RegExp`, or `String`.
 *
 * @private
 * @param {Object} object The object to compare.
 * @param {Object} other The other object to compare.
 * @param {string} tag The `toStringTag` of the objects to compare.
 * @param {Function} equalFunc The function to determine equivalents of values.
 * @param {Function} customizer The function to customize comparisons.
 * @param {number} bitmask The bitmask of comparison flags. See `baseIsEqual`
 *  for more details.
 * @param {Object} stack Tracks traversed `object` and `other` objects.
 * @returns {boolean} Returns `true` if the objects are equivalent, else `false`.
 */
function equalByTag(object, other, tag, equalFunc, customizer, bitmask, stack) {
  switch (tag) {
    case dataViewTag:
      if ((object.byteLength != other.byteLength) ||
          (object.byteOffset != other.byteOffset)) {
        return false;
      }
      object = object.buffer;
      other = other.buffer;

    case arrayBufferTag:
      if ((object.byteLength != other.byteLength) ||
          !equalFunc(new Uint8Array(object), new Uint8Array(other))) {
        return false;
      }
      return true;

    case boolTag:
    case dateTag:
    case numberTag:
      // Coerce booleans to `1` or `0` and dates to milliseconds.
      // Invalid dates are coerced to `NaN`.
      return eq(+object, +other);

    case errorTag:
      return object.name == other.name && object.message == other.message;

    case regexpTag:
    case stringTag:
      // Coerce regexes to strings and treat strings, primitives and objects,
      // as equal. See http://www.ecma-international.org/ecma-262/7.0/#sec-regexp.prototype.tostring
      // for more details.
      return object == (other + '');

    case mapTag:
      var convert = mapToArray;

    case setTag:
      var isPartial = bitmask & PARTIAL_COMPARE_FLAG;
      convert || (convert = setToArray);

      if (object.size != other.size && !isPartial) {
        return false;
      }
      // Assume cyclic values are equal.
      var stacked = stack.get(object);
      if (stacked) {
        return stacked == other;
      }
      bitmask |= UNORDERED_COMPARE_FLAG;

      // Recursively compare objects (susceptible to call stack limits).
      stack.set(object, other);
      var result = equalArrays(convert(object), convert(other), equalFunc, customizer, bitmask, stack);
      stack['delete'](object);
      return result;

    case symbolTag:
      if (symbolValueOf) {
        return symbolValueOf.call(object) == symbolValueOf.call(other);
      }
  }
  return false;
}

/**
 * A specialized version of `baseIsEqualDeep` for objects with support for
 * partial deep comparisons.
 *
 * @private
 * @param {Object} object The object to compare.
 * @param {Object} other The other object to compare.
 * @param {Function} equalFunc The function to determine equivalents of values.
 * @param {Function} customizer The function to customize comparisons.
 * @param {number} bitmask The bitmask of comparison flags. See `baseIsEqual`
 *  for more details.
 * @param {Object} stack Tracks traversed `object` and `other` objects.
 * @returns {boolean} Returns `true` if the objects are equivalent, else `false`.
 */
function equalObjects(object, other, equalFunc, customizer, bitmask, stack) {
  var isPartial = bitmask & PARTIAL_COMPARE_FLAG,
      objProps = keys(object),
      objLength = objProps.length,
      othProps = keys(other),
      othLength = othProps.length;

  if (objLength != othLength && !isPartial) {
    return false;
  }
  var index = objLength;
  while (index--) {
    var key = objProps[index];
    if (!(isPartial ? key in other : hasOwnProperty.call(other, key))) {
      return false;
    }
  }
  // Assume cyclic values are equal.
  var stacked = stack.get(object);
  if (stacked && stack.get(other)) {
    return stacked == other;
  }
  var result = true;
  stack.set(object, other);
  stack.set(other, object);

  var skipCtor = isPartial;
  while (++index < objLength) {
    key = objProps[index];
    var objValue = object[key],
        othValue = other[key];

    if (customizer) {
      var compared = isPartial
        ? customizer(othValue, objValue, key, other, object, stack)
        : customizer(objValue, othValue, key, object, other, stack);
    }
    // Recursively compare objects (susceptible to call stack limits).
    if (!(compared === undefined
          ? (objValue === othValue || equalFunc(objValue, othValue, customizer, bitmask, stack))
          : compared
        )) {
      result = false;
      break;
    }
    skipCtor || (skipCtor = key == 'constructor');
  }
  if (result && !skipCtor) {
    var objCtor = object.constructor,
        othCtor = other.constructor;

    // Non `Object` object instances with different constructors are not equal.
    if (objCtor != othCtor &&
        ('constructor' in object && 'constructor' in other) &&
        !(typeof objCtor == 'function' && objCtor instanceof objCtor &&
          typeof othCtor == 'function' && othCtor instanceof othCtor)) {
      result = false;
    }
  }
  stack['delete'](object);
  stack['delete'](other);
  return result;
}

/**
 * Gets the data for `map`.
 *
 * @private
 * @param {Object} map The map to query.
 * @param {string} key The reference key.
 * @returns {*} Returns the map data.
 */
function getMapData(map, key) {
  var data = map.__data__;
  return isKeyable(key)
    ? data[typeof key == 'string' ? 'string' : 'hash']
    : data.map;
}

/**
 * Gets the property names, values, and compare flags of `object`.
 *
 * @private
 * @param {Object} object The object to query.
 * @returns {Array} Returns the match data of `object`.
 */
function getMatchData(object) {
  var result = keys(object),
      length = result.length;

  while (length--) {
    var key = result[length],
        value = object[key];

    result[length] = [key, value, isStrictComparable(value)];
  }
  return result;
}

/**
 * Gets the native function at `key` of `object`.
 *
 * @private
 * @param {Object} object The object to query.
 * @param {string} key The key of the method to get.
 * @returns {*} Returns the function if it's native, else `undefined`.
 */
function getNative(object, key) {
  var value = getValue(object, key);
  return baseIsNative(value) ? value : undefined;
}

/**
 * Gets the `toStringTag` of `value`.
 *
 * @private
 * @param {*} value The value to query.
 * @returns {string} Returns the `toStringTag`.
 */
var getTag = baseGetTag;

// Fallback for data views, maps, sets, and weak maps in IE 11,
// for data views in Edge < 14, and promises in Node.js.
if ((DataView && getTag(new DataView(new ArrayBuffer(1))) != dataViewTag) ||
    (Map && getTag(new Map) != mapTag) ||
    (Promise && getTag(Promise.resolve()) != promiseTag) ||
    (Set && getTag(new Set) != setTag) ||
    (WeakMap && getTag(new WeakMap) != weakMapTag)) {
  getTag = function(value) {
    var result = objectToString.call(value),
        Ctor = result == objectTag ? value.constructor : undefined,
        ctorString = Ctor ? toSource(Ctor) : undefined;

    if (ctorString) {
      switch (ctorString) {
        case dataViewCtorString: return dataViewTag;
        case mapCtorString: return mapTag;
        case promiseCtorString: return promiseTag;
        case setCtorString: return setTag;
        case weakMapCtorString: return weakMapTag;
      }
    }
    return result;
  };
}

/**
 * Checks if `path` exists on `object`.
 *
 * @private
 * @param {Object} object The object to query.
 * @param {Array|string} path The path to check.
 * @param {Function} hasFunc The function to check properties.
 * @returns {boolean} Returns `true` if `path` exists, else `false`.
 */
function hasPath(object, path, hasFunc) {
  path = isKey(path, object) ? [path] : castPath(path);

  var result,
      index = -1,
      length = path.length;

  while (++index < length) {
    var key = toKey(path[index]);
    if (!(result = object != null && hasFunc(object, key))) {
      break;
    }
    object = object[key];
  }
  if (result) {
    return result;
  }
  var length = object ? object.length : 0;
  return !!length && isLength(length) && isIndex(key, length) &&
    (isArray(object) || isArguments(object));
}

/**
 * Checks if `value` is a valid array-like index.
 *
 * @private
 * @param {*} value The value to check.
 * @param {number} [length=MAX_SAFE_INTEGER] The upper bounds of a valid index.
 * @returns {boolean} Returns `true` if `value` is a valid index, else `false`.
 */
function isIndex(value, length) {
  length = length == null ? MAX_SAFE_INTEGER : length;
  return !!length &&
    (typeof value == 'number' || reIsUint.test(value)) &&
    (value > -1 && value % 1 == 0 && value < length);
}

/**
 * Checks if `value` is a property name and not a property path.
 *
 * @private
 * @param {*} value The value to check.
 * @param {Object} [object] The object to query keys on.
 * @returns {boolean} Returns `true` if `value` is a property name, else `false`.
 */
function isKey(value, object) {
  if (isArray(value)) {
    return false;
  }
  var type = typeof value;
  if (type == 'number' || type == 'symbol' || type == 'boolean' ||
      value == null || isSymbol(value)) {
    return true;
  }
  return reIsPlainProp.test(value) || !reIsDeepProp.test(value) ||
    (object != null && value in Object(object));
}

/**
 * Checks if `value` is suitable for use as unique object key.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is suitable, else `false`.
 */
function isKeyable(value) {
  var type = typeof value;
  return (type == 'string' || type == 'number' || type == 'symbol' || type == 'boolean')
    ? (value !== '__proto__')
    : (value === null);
}

/**
 * Checks if `func` has its source masked.
 *
 * @private
 * @param {Function} func The function to check.
 * @returns {boolean} Returns `true` if `func` is masked, else `false`.
 */
function isMasked(func) {
  return !!maskSrcKey && (maskSrcKey in func);
}

/**
 * Checks if `value` is likely a prototype object.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a prototype, else `false`.
 */
function isPrototype(value) {
  var Ctor = value && value.constructor,
      proto = (typeof Ctor == 'function' && Ctor.prototype) || objectProto;

  return value === proto;
}

/**
 * Checks if `value` is suitable for strict equality comparisons, i.e. `===`.
 *
 * @private
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` if suitable for strict
 *  equality comparisons, else `false`.
 */
function isStrictComparable(value) {
  return value === value && !isObject(value);
}

/**
 * A specialized version of `matchesProperty` for source values suitable
 * for strict equality comparisons, i.e. `===`.
 *
 * @private
 * @param {string} key The key of the property to get.
 * @param {*} srcValue The value to match.
 * @returns {Function} Returns the new spec function.
 */
function matchesStrictComparable(key, srcValue) {
  return function(object) {
    if (object == null) {
      return false;
    }
    return object[key] === srcValue &&
      (srcValue !== undefined || (key in Object(object)));
  };
}

/**
 * Converts `string` to a property path array.
 *
 * @private
 * @param {string} string The string to convert.
 * @returns {Array} Returns the property path array.
 */
var stringToPath = memoize(function(string) {
  string = toString(string);

  var result = [];
  if (reLeadingDot.test(string)) {
    result.push('');
  }
  string.replace(rePropName, function(match, number, quote, string) {
    result.push(quote ? string.replace(reEscapeChar, '$1') : (number || match));
  });
  return result;
});

/**
 * Converts `value` to a string key if it's not a string or symbol.
 *
 * @private
 * @param {*} value The value to inspect.
 * @returns {string|symbol} Returns the key.
 */
function toKey(value) {
  if (typeof value == 'string' || isSymbol(value)) {
    return value;
  }
  var result = (value + '');
  return (result == '0' && (1 / value) == -INFINITY) ? '-0' : result;
}

/**
 * Converts `func` to its source code.
 *
 * @private
 * @param {Function} func The function to process.
 * @returns {string} Returns the source code.
 */
function toSource(func) {
  if (func != null) {
    try {
      return funcToString.call(func);
    } catch (e) {}
    try {
      return (func + '');
    } catch (e) {}
  }
  return '';
}

/**
 * This method is like `_.uniq` except that it accepts `iteratee` which is
 * invoked for each element in `array` to generate the criterion by which
 * uniqueness is computed. The iteratee is invoked with one argument: (value).
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Array
 * @param {Array} array The array to inspect.
 * @param {Function} [iteratee=_.identity]
 *  The iteratee invoked per element.
 * @returns {Array} Returns the new duplicate free array.
 * @example
 *
 * _.uniqBy([2.1, 1.2, 2.3], Math.floor);
 * // => [2.1, 1.2]
 *
 * // The `_.property` iteratee shorthand.
 * _.uniqBy([{ 'x': 1 }, { 'x': 2 }, { 'x': 1 }], 'x');
 * // => [{ 'x': 1 }, { 'x': 2 }]
 */
function uniqBy(array, iteratee) {
  return (array && array.length)
    ? baseUniq(array, baseIteratee(iteratee, 2))
    : [];
}

/**
 * Creates a function that memoizes the result of `func`. If `resolver` is
 * provided, it determines the cache key for storing the result based on the
 * arguments provided to the memoized function. By default, the first argument
 * provided to the memoized function is used as the map cache key. The `func`
 * is invoked with the `this` binding of the memoized function.
 *
 * **Note:** The cache is exposed as the `cache` property on the memoized
 * function. Its creation may be customized by replacing the `_.memoize.Cache`
 * constructor with one whose instances implement the
 * [`Map`](http://ecma-international.org/ecma-262/7.0/#sec-properties-of-the-map-prototype-object)
 * method interface of `delete`, `get`, `has`, and `set`.
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Function
 * @param {Function} func The function to have its output memoized.
 * @param {Function} [resolver] The function to resolve the cache key.
 * @returns {Function} Returns the new memoized function.
 * @example
 *
 * var object = { 'a': 1, 'b': 2 };
 * var other = { 'c': 3, 'd': 4 };
 *
 * var values = _.memoize(_.values);
 * values(object);
 * // => [1, 2]
 *
 * values(other);
 * // => [3, 4]
 *
 * object.a = 2;
 * values(object);
 * // => [1, 2]
 *
 * // Modify the result cache.
 * values.cache.set(object, ['a', 'b']);
 * values(object);
 * // => ['a', 'b']
 *
 * // Replace `_.memoize.Cache`.
 * _.memoize.Cache = WeakMap;
 */
function memoize(func, resolver) {
  if (typeof func != 'function' || (resolver && typeof resolver != 'function')) {
    throw new TypeError(FUNC_ERROR_TEXT);
  }
  var memoized = function() {
    var args = arguments,
        key = resolver ? resolver.apply(this, args) : args[0],
        cache = memoized.cache;

    if (cache.has(key)) {
      return cache.get(key);
    }
    var result = func.apply(this, args);
    memoized.cache = cache.set(key, result);
    return result;
  };
  memoized.cache = new (memoize.Cache || MapCache);
  return memoized;
}

// Assign cache to `_.memoize`.
memoize.Cache = MapCache;

/**
 * Performs a
 * [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
 * comparison between two values to determine if they are equivalent.
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to compare.
 * @param {*} other The other value to compare.
 * @returns {boolean} Returns `true` if the values are equivalent, else `false`.
 * @example
 *
 * var object = { 'a': 1 };
 * var other = { 'a': 1 };
 *
 * _.eq(object, object);
 * // => true
 *
 * _.eq(object, other);
 * // => false
 *
 * _.eq('a', 'a');
 * // => true
 *
 * _.eq('a', Object('a'));
 * // => false
 *
 * _.eq(NaN, NaN);
 * // => true
 */
function eq(value, other) {
  return value === other || (value !== value && other !== other);
}

/**
 * Checks if `value` is likely an `arguments` object.
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is an `arguments` object,
 *  else `false`.
 * @example
 *
 * _.isArguments(function() { return arguments; }());
 * // => true
 *
 * _.isArguments([1, 2, 3]);
 * // => false
 */
function isArguments(value) {
  // Safari 8.1 makes `arguments.callee` enumerable in strict mode.
  return isArrayLikeObject(value) && hasOwnProperty.call(value, 'callee') &&
    (!propertyIsEnumerable.call(value, 'callee') || objectToString.call(value) == argsTag);
}

/**
 * Checks if `value` is classified as an `Array` object.
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is an array, else `false`.
 * @example
 *
 * _.isArray([1, 2, 3]);
 * // => true
 *
 * _.isArray(document.body.children);
 * // => false
 *
 * _.isArray('abc');
 * // => false
 *
 * _.isArray(_.noop);
 * // => false
 */
var isArray = Array.isArray;

/**
 * Checks if `value` is array-like. A value is considered array-like if it's
 * not a function and has a `value.length` that's an integer greater than or
 * equal to `0` and less than or equal to `Number.MAX_SAFE_INTEGER`.
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is array-like, else `false`.
 * @example
 *
 * _.isArrayLike([1, 2, 3]);
 * // => true
 *
 * _.isArrayLike(document.body.children);
 * // => true
 *
 * _.isArrayLike('abc');
 * // => true
 *
 * _.isArrayLike(_.noop);
 * // => false
 */
function isArrayLike(value) {
  return value != null && isLength(value.length) && !isFunction(value);
}

/**
 * This method is like `_.isArrayLike` except that it also checks if `value`
 * is an object.
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is an array-like object,
 *  else `false`.
 * @example
 *
 * _.isArrayLikeObject([1, 2, 3]);
 * // => true
 *
 * _.isArrayLikeObject(document.body.children);
 * // => true
 *
 * _.isArrayLikeObject('abc');
 * // => false
 *
 * _.isArrayLikeObject(_.noop);
 * // => false
 */
function isArrayLikeObject(value) {
  return isObjectLike(value) && isArrayLike(value);
}

/**
 * Checks if `value` is classified as a `Function` object.
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a function, else `false`.
 * @example
 *
 * _.isFunction(_);
 * // => true
 *
 * _.isFunction(/abc/);
 * // => false
 */
function isFunction(value) {
  // The use of `Object#toString` avoids issues with the `typeof` operator
  // in Safari 8-9 which returns 'object' for typed array and other constructors.
  var tag = isObject(value) ? objectToString.call(value) : '';
  return tag == funcTag || tag == genTag;
}

/**
 * Checks if `value` is a valid array-like length.
 *
 * **Note:** This method is loosely based on
 * [`ToLength`](http://ecma-international.org/ecma-262/7.0/#sec-tolength).
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a valid length, else `false`.
 * @example
 *
 * _.isLength(3);
 * // => true
 *
 * _.isLength(Number.MIN_VALUE);
 * // => false
 *
 * _.isLength(Infinity);
 * // => false
 *
 * _.isLength('3');
 * // => false
 */
function isLength(value) {
  return typeof value == 'number' &&
    value > -1 && value % 1 == 0 && value <= MAX_SAFE_INTEGER;
}

/**
 * Checks if `value` is the
 * [language type](http://www.ecma-international.org/ecma-262/7.0/#sec-ecmascript-language-types)
 * of `Object`. (e.g. arrays, functions, objects, regexes, `new Number(0)`, and `new String('')`)
 *
 * @static
 * @memberOf _
 * @since 0.1.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is an object, else `false`.
 * @example
 *
 * _.isObject({});
 * // => true
 *
 * _.isObject([1, 2, 3]);
 * // => true
 *
 * _.isObject(_.noop);
 * // => true
 *
 * _.isObject(null);
 * // => false
 */
function isObject(value) {
  var type = typeof value;
  return !!value && (type == 'object' || type == 'function');
}

/**
 * Checks if `value` is object-like. A value is object-like if it's not `null`
 * and has a `typeof` result of "object".
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is object-like, else `false`.
 * @example
 *
 * _.isObjectLike({});
 * // => true
 *
 * _.isObjectLike([1, 2, 3]);
 * // => true
 *
 * _.isObjectLike(_.noop);
 * // => false
 *
 * _.isObjectLike(null);
 * // => false
 */
function isObjectLike(value) {
  return !!value && typeof value == 'object';
}

/**
 * Checks if `value` is classified as a `Symbol` primitive or object.
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a symbol, else `false`.
 * @example
 *
 * _.isSymbol(Symbol.iterator);
 * // => true
 *
 * _.isSymbol('abc');
 * // => false
 */
function isSymbol(value) {
  return typeof value == 'symbol' ||
    (isObjectLike(value) && objectToString.call(value) == symbolTag);
}

/**
 * Checks if `value` is classified as a typed array.
 *
 * @static
 * @memberOf _
 * @since 3.0.0
 * @category Lang
 * @param {*} value The value to check.
 * @returns {boolean} Returns `true` if `value` is a typed array, else `false`.
 * @example
 *
 * _.isTypedArray(new Uint8Array);
 * // => true
 *
 * _.isTypedArray([]);
 * // => false
 */
var isTypedArray = nodeIsTypedArray ? baseUnary(nodeIsTypedArray) : baseIsTypedArray;

/**
 * Converts `value` to a string. An empty string is returned for `null`
 * and `undefined` values. The sign of `-0` is preserved.
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Lang
 * @param {*} value The value to process.
 * @returns {string} Returns the string.
 * @example
 *
 * _.toString(null);
 * // => ''
 *
 * _.toString(-0);
 * // => '-0'
 *
 * _.toString([1, 2, 3]);
 * // => '1,2,3'
 */
function toString(value) {
  return value == null ? '' : baseToString(value);
}

/**
 * Gets the value at `path` of `object`. If the resolved value is
 * `undefined`, the `defaultValue` is returned in its place.
 *
 * @static
 * @memberOf _
 * @since 3.7.0
 * @category Object
 * @param {Object} object The object to query.
 * @param {Array|string} path The path of the property to get.
 * @param {*} [defaultValue] The value returned for `undefined` resolved values.
 * @returns {*} Returns the resolved value.
 * @example
 *
 * var object = { 'a': [{ 'b': { 'c': 3 } }] };
 *
 * _.get(object, 'a[0].b.c');
 * // => 3
 *
 * _.get(object, ['a', '0', 'b', 'c']);
 * // => 3
 *
 * _.get(object, 'a.b.c', 'default');
 * // => 'default'
 */
function get(object, path, defaultValue) {
  var result = object == null ? undefined : baseGet(object, path);
  return result === undefined ? defaultValue : result;
}

/**
 * Checks if `path` is a direct or inherited property of `object`.
 *
 * @static
 * @memberOf _
 * @since 4.0.0
 * @category Object
 * @param {Object} object The object to query.
 * @param {Array|string} path The path to check.
 * @returns {boolean} Returns `true` if `path` exists, else `false`.
 * @example
 *
 * var object = _.create({ 'a': _.create({ 'b': 2 }) });
 *
 * _.hasIn(object, 'a');
 * // => true
 *
 * _.hasIn(object, 'a.b');
 * // => true
 *
 * _.hasIn(object, ['a', 'b']);
 * // => true
 *
 * _.hasIn(object, 'b');
 * // => false
 */
function hasIn(object, path) {
  return object != null && hasPath(object, path, baseHasIn);
}

/**
 * Creates an array of the own enumerable property names of `object`.
 *
 * **Note:** Non-object values are coerced to objects. See the
 * [ES spec](http://ecma-international.org/ecma-262/7.0/#sec-object.keys)
 * for more details.
 *
 * @static
 * @since 0.1.0
 * @memberOf _
 * @category Object
 * @param {Object} object The object to query.
 * @returns {Array} Returns the array of property names.
 * @example
 *
 * function Foo() {
 *   this.a = 1;
 *   this.b = 2;
 * }
 *
 * Foo.prototype.c = 3;
 *
 * _.keys(new Foo);
 * // => ['a', 'b'] (iteration order is not guaranteed)
 *
 * _.keys('hi');
 * // => ['0', '1']
 */
function keys(object) {
  return isArrayLike(object) ? arrayLikeKeys(object) : baseKeys(object);
}

/**
 * This method returns the first argument it receives.
 *
 * @static
 * @since 0.1.0
 * @memberOf _
 * @category Util
 * @param {*} value Any value.
 * @returns {*} Returns `value`.
 * @example
 *
 * var object = { 'a': 1 };
 *
 * console.log(_.identity(object) === object);
 * // => true
 */
function identity(value) {
  return value;
}

/**
 * This method returns `undefined`.
 *
 * @static
 * @memberOf _
 * @since 2.3.0
 * @category Util
 * @example
 *
 * _.times(2, _.noop);
 * // => [undefined, undefined]
 */
function noop() {
  // No operation performed.
}

/**
 * Creates a function that returns the value at `path` of a given object.
 *
 * @static
 * @memberOf _
 * @since 2.4.0
 * @category Util
 * @param {Array|string} path The path of the property to get.
 * @returns {Function} Returns the new accessor function.
 * @example
 *
 * var objects = [
 *   { 'a': { 'b': 2 } },
 *   { 'a': { 'b': 1 } }
 * ];
 *
 * _.map(objects, _.property('a.b'));
 * // => [2, 1]
 *
 * _.map(_.sortBy(objects, _.property(['a', 'b'])), 'a.b');
 * // => [1, 2]
 */
function property(path) {
  return isKey(path) ? baseProperty(toKey(path)) : basePropertyDeep(path);
}

module.exports = uniqBy;


/***/ }),

/***/ 614:
/***/ ((module, __unused_webpack_exports, __webpack_require__) => {

"use strict";


module.exports = Pbf;

var ieee754 = __webpack_require__(645);

function Pbf(buf) {
    this.buf = ArrayBuffer.isView && ArrayBuffer.isView(buf) ? buf : new Uint8Array(buf || 0);
    this.pos = 0;
    this.type = 0;
    this.length = this.buf.length;
}

Pbf.Varint  = 0; // varint: int32, int64, uint32, uint64, sint32, sint64, bool, enum
Pbf.Fixed64 = 1; // 64-bit: double, fixed64, sfixed64
Pbf.Bytes   = 2; // length-delimited: string, bytes, embedded messages, packed repeated fields
Pbf.Fixed32 = 5; // 32-bit: float, fixed32, sfixed32

var SHIFT_LEFT_32 = (1 << 16) * (1 << 16),
    SHIFT_RIGHT_32 = 1 / SHIFT_LEFT_32;

// Threshold chosen based on both benchmarking and knowledge about browser string
// data structures (which currently switch structure types at 12 bytes or more)
var TEXT_DECODER_MIN_LENGTH = 12;
var utf8TextDecoder = typeof TextDecoder === 'undefined' ? null : new TextDecoder('utf8');

Pbf.prototype = {

    destroy: function() {
        this.buf = null;
    },

    // === READING =================================================================

    readFields: function(readField, result, end) {
        end = end || this.length;

        while (this.pos < end) {
            var val = this.readVarint(),
                tag = val >> 3,
                startPos = this.pos;

            this.type = val & 0x7;
            readField(tag, result, this);

            if (this.pos === startPos) this.skip(val);
        }
        return result;
    },

    readMessage: function(readField, result) {
        return this.readFields(readField, result, this.readVarint() + this.pos);
    },

    readFixed32: function() {
        var val = readUInt32(this.buf, this.pos);
        this.pos += 4;
        return val;
    },

    readSFixed32: function() {
        var val = readInt32(this.buf, this.pos);
        this.pos += 4;
        return val;
    },

    // 64-bit int handling is based on github.com/dpw/node-buffer-more-ints (MIT-licensed)

    readFixed64: function() {
        var val = readUInt32(this.buf, this.pos) + readUInt32(this.buf, this.pos + 4) * SHIFT_LEFT_32;
        this.pos += 8;
        return val;
    },

    readSFixed64: function() {
        var val = readUInt32(this.buf, this.pos) + readInt32(this.buf, this.pos + 4) * SHIFT_LEFT_32;
        this.pos += 8;
        return val;
    },

    readFloat: function() {
        var val = ieee754.read(this.buf, this.pos, true, 23, 4);
        this.pos += 4;
        return val;
    },

    readDouble: function() {
        var val = ieee754.read(this.buf, this.pos, true, 52, 8);
        this.pos += 8;
        return val;
    },

    readVarint: function(isSigned) {
        var buf = this.buf,
            val, b;

        b = buf[this.pos++]; val  =  b & 0x7f;        if (b < 0x80) return val;
        b = buf[this.pos++]; val |= (b & 0x7f) << 7;  if (b < 0x80) return val;
        b = buf[this.pos++]; val |= (b & 0x7f) << 14; if (b < 0x80) return val;
        b = buf[this.pos++]; val |= (b & 0x7f) << 21; if (b < 0x80) return val;
        b = buf[this.pos];   val |= (b & 0x0f) << 28;

        return readVarintRemainder(val, isSigned, this);
    },

    readVarint64: function() { // for compatibility with v2.0.1
        return this.readVarint(true);
    },

    readSVarint: function() {
        var num = this.readVarint();
        return num % 2 === 1 ? (num + 1) / -2 : num / 2; // zigzag encoding
    },

    readBoolean: function() {
        return Boolean(this.readVarint());
    },

    readString: function() {
        var end = this.readVarint() + this.pos;
        var pos = this.pos;
        this.pos = end;

        if (end - pos >= TEXT_DECODER_MIN_LENGTH && utf8TextDecoder) {
            // longer strings are fast with the built-in browser TextDecoder API
            return readUtf8TextDecoder(this.buf, pos, end);
        }
        // short strings are fast with our custom implementation
        return readUtf8(this.buf, pos, end);
    },

    readBytes: function() {
        var end = this.readVarint() + this.pos,
            buffer = this.buf.subarray(this.pos, end);
        this.pos = end;
        return buffer;
    },

    // verbose for performance reasons; doesn't affect gzipped size

    readPackedVarint: function(arr, isSigned) {
        if (this.type !== Pbf.Bytes) return arr.push(this.readVarint(isSigned));
        var end = readPackedEnd(this);
        arr = arr || [];
        while (this.pos < end) arr.push(this.readVarint(isSigned));
        return arr;
    },
    readPackedSVarint: function(arr) {
        if (this.type !== Pbf.Bytes) return arr.push(this.readSVarint());
        var end = readPackedEnd(this);
        arr = arr || [];
        while (this.pos < end) arr.push(this.readSVarint());
        return arr;
    },
    readPackedBoolean: function(arr) {
        if (this.type !== Pbf.Bytes) return arr.push(this.readBoolean());
        var end = readPackedEnd(this);
        arr = arr || [];
        while (this.pos < end) arr.push(this.readBoolean());
        return arr;
    },
    readPackedFloat: function(arr) {
        if (this.type !== Pbf.Bytes) return arr.push(this.readFloat());
        var end = readPackedEnd(this);
        arr = arr || [];
        while (this.pos < end) arr.push(this.readFloat());
        return arr;
    },
    readPackedDouble: function(arr) {
        if (this.type !== Pbf.Bytes) return arr.push(this.readDouble());
        var end = readPackedEnd(this);
        arr = arr || [];
        while (this.pos < end) arr.push(this.readDouble());
        return arr;
    },
    readPackedFixed32: function(arr) {
        if (this.type !== Pbf.Bytes) return arr.push(this.readFixed32());
        var end = readPackedEnd(this);
        arr = arr || [];
        while (this.pos < end) arr.push(this.readFixed32());
        return arr;
    },
    readPackedSFixed32: function(arr) {
        if (this.type !== Pbf.Bytes) return arr.push(this.readSFixed32());
        var end = readPackedEnd(this);
        arr = arr || [];
        while (this.pos < end) arr.push(this.readSFixed32());
        return arr;
    },
    readPackedFixed64: function(arr) {
        if (this.type !== Pbf.Bytes) return arr.push(this.readFixed64());
        var end = readPackedEnd(this);
        arr = arr || [];
        while (this.pos < end) arr.push(this.readFixed64());
        return arr;
    },
    readPackedSFixed64: function(arr) {
        if (this.type !== Pbf.Bytes) return arr.push(this.readSFixed64());
        var end = readPackedEnd(this);
        arr = arr || [];
        while (this.pos < end) arr.push(this.readSFixed64());
        return arr;
    },

    skip: function(val) {
        var type = val & 0x7;
        if (type === Pbf.Varint) while (this.buf[this.pos++] > 0x7f) {}
        else if (type === Pbf.Bytes) this.pos = this.readVarint() + this.pos;
        else if (type === Pbf.Fixed32) this.pos += 4;
        else if (type === Pbf.Fixed64) this.pos += 8;
        else throw new Error('Unimplemented type: ' + type);
    },

    // === WRITING =================================================================

    writeTag: function(tag, type) {
        this.writeVarint((tag << 3) | type);
    },

    realloc: function(min) {
        var length = this.length || 16;

        while (length < this.pos + min) length *= 2;

        if (length !== this.length) {
            var buf = new Uint8Array(length);
            buf.set(this.buf);
            this.buf = buf;
            this.length = length;
        }
    },

    finish: function() {
        this.length = this.pos;
        this.pos = 0;
        return this.buf.subarray(0, this.length);
    },

    writeFixed32: function(val) {
        this.realloc(4);
        writeInt32(this.buf, val, this.pos);
        this.pos += 4;
    },

    writeSFixed32: function(val) {
        this.realloc(4);
        writeInt32(this.buf, val, this.pos);
        this.pos += 4;
    },

    writeFixed64: function(val) {
        this.realloc(8);
        writeInt32(this.buf, val & -1, this.pos);
        writeInt32(this.buf, Math.floor(val * SHIFT_RIGHT_32), this.pos + 4);
        this.pos += 8;
    },

    writeSFixed64: function(val) {
        this.realloc(8);
        writeInt32(this.buf, val & -1, this.pos);
        writeInt32(this.buf, Math.floor(val * SHIFT_RIGHT_32), this.pos + 4);
        this.pos += 8;
    },

    writeVarint: function(val) {
        val = +val || 0;

        if (val > 0xfffffff || val < 0) {
            writeBigVarint(val, this);
            return;
        }

        this.realloc(4);

        this.buf[this.pos++] =           val & 0x7f  | (val > 0x7f ? 0x80 : 0); if (val <= 0x7f) return;
        this.buf[this.pos++] = ((val >>>= 7) & 0x7f) | (val > 0x7f ? 0x80 : 0); if (val <= 0x7f) return;
        this.buf[this.pos++] = ((val >>>= 7) & 0x7f) | (val > 0x7f ? 0x80 : 0); if (val <= 0x7f) return;
        this.buf[this.pos++] =   (val >>> 7) & 0x7f;
    },

    writeSVarint: function(val) {
        this.writeVarint(val < 0 ? -val * 2 - 1 : val * 2);
    },

    writeBoolean: function(val) {
        this.writeVarint(Boolean(val));
    },

    writeString: function(str) {
        str = String(str);
        this.realloc(str.length * 4);

        this.pos++; // reserve 1 byte for short string length

        var startPos = this.pos;
        // write the string directly to the buffer and see how much was written
        this.pos = writeUtf8(this.buf, str, this.pos);
        var len = this.pos - startPos;

        if (len >= 0x80) makeRoomForExtraLength(startPos, len, this);

        // finally, write the message length in the reserved place and restore the position
        this.pos = startPos - 1;
        this.writeVarint(len);
        this.pos += len;
    },

    writeFloat: function(val) {
        this.realloc(4);
        ieee754.write(this.buf, val, this.pos, true, 23, 4);
        this.pos += 4;
    },

    writeDouble: function(val) {
        this.realloc(8);
        ieee754.write(this.buf, val, this.pos, true, 52, 8);
        this.pos += 8;
    },

    writeBytes: function(buffer) {
        var len = buffer.length;
        this.writeVarint(len);
        this.realloc(len);
        for (var i = 0; i < len; i++) this.buf[this.pos++] = buffer[i];
    },

    writeRawMessage: function(fn, obj) {
        this.pos++; // reserve 1 byte for short message length

        // write the message directly to the buffer and see how much was written
        var startPos = this.pos;
        fn(obj, this);
        var len = this.pos - startPos;

        if (len >= 0x80) makeRoomForExtraLength(startPos, len, this);

        // finally, write the message length in the reserved place and restore the position
        this.pos = startPos - 1;
        this.writeVarint(len);
        this.pos += len;
    },

    writeMessage: function(tag, fn, obj) {
        this.writeTag(tag, Pbf.Bytes);
        this.writeRawMessage(fn, obj);
    },

    writePackedVarint:   function(tag, arr) { if (arr.length) this.writeMessage(tag, writePackedVarint, arr);   },
    writePackedSVarint:  function(tag, arr) { if (arr.length) this.writeMessage(tag, writePackedSVarint, arr);  },
    writePackedBoolean:  function(tag, arr) { if (arr.length) this.writeMessage(tag, writePackedBoolean, arr);  },
    writePackedFloat:    function(tag, arr) { if (arr.length) this.writeMessage(tag, writePackedFloat, arr);    },
    writePackedDouble:   function(tag, arr) { if (arr.length) this.writeMessage(tag, writePackedDouble, arr);   },
    writePackedFixed32:  function(tag, arr) { if (arr.length) this.writeMessage(tag, writePackedFixed32, arr);  },
    writePackedSFixed32: function(tag, arr) { if (arr.length) this.writeMessage(tag, writePackedSFixed32, arr); },
    writePackedFixed64:  function(tag, arr) { if (arr.length) this.writeMessage(tag, writePackedFixed64, arr);  },
    writePackedSFixed64: function(tag, arr) { if (arr.length) this.writeMessage(tag, writePackedSFixed64, arr); },

    writeBytesField: function(tag, buffer) {
        this.writeTag(tag, Pbf.Bytes);
        this.writeBytes(buffer);
    },
    writeFixed32Field: function(tag, val) {
        this.writeTag(tag, Pbf.Fixed32);
        this.writeFixed32(val);
    },
    writeSFixed32Field: function(tag, val) {
        this.writeTag(tag, Pbf.Fixed32);
        this.writeSFixed32(val);
    },
    writeFixed64Field: function(tag, val) {
        this.writeTag(tag, Pbf.Fixed64);
        this.writeFixed64(val);
    },
    writeSFixed64Field: function(tag, val) {
        this.writeTag(tag, Pbf.Fixed64);
        this.writeSFixed64(val);
    },
    writeVarintField: function(tag, val) {
        this.writeTag(tag, Pbf.Varint);
        this.writeVarint(val);
    },
    writeSVarintField: function(tag, val) {
        this.writeTag(tag, Pbf.Varint);
        this.writeSVarint(val);
    },
    writeStringField: function(tag, str) {
        this.writeTag(tag, Pbf.Bytes);
        this.writeString(str);
    },
    writeFloatField: function(tag, val) {
        this.writeTag(tag, Pbf.Fixed32);
        this.writeFloat(val);
    },
    writeDoubleField: function(tag, val) {
        this.writeTag(tag, Pbf.Fixed64);
        this.writeDouble(val);
    },
    writeBooleanField: function(tag, val) {
        this.writeVarintField(tag, Boolean(val));
    }
};

function readVarintRemainder(l, s, p) {
    var buf = p.buf,
        h, b;

    b = buf[p.pos++]; h  = (b & 0x70) >> 4;  if (b < 0x80) return toNum(l, h, s);
    b = buf[p.pos++]; h |= (b & 0x7f) << 3;  if (b < 0x80) return toNum(l, h, s);
    b = buf[p.pos++]; h |= (b & 0x7f) << 10; if (b < 0x80) return toNum(l, h, s);
    b = buf[p.pos++]; h |= (b & 0x7f) << 17; if (b < 0x80) return toNum(l, h, s);
    b = buf[p.pos++]; h |= (b & 0x7f) << 24; if (b < 0x80) return toNum(l, h, s);
    b = buf[p.pos++]; h |= (b & 0x01) << 31; if (b < 0x80) return toNum(l, h, s);

    throw new Error('Expected varint not more than 10 bytes');
}

function readPackedEnd(pbf) {
    return pbf.type === Pbf.Bytes ?
        pbf.readVarint() + pbf.pos : pbf.pos + 1;
}

function toNum(low, high, isSigned) {
    if (isSigned) {
        return high * 0x100000000 + (low >>> 0);
    }

    return ((high >>> 0) * 0x100000000) + (low >>> 0);
}

function writeBigVarint(val, pbf) {
    var low, high;

    if (val >= 0) {
        low  = (val % 0x100000000) | 0;
        high = (val / 0x100000000) | 0;
    } else {
        low  = ~(-val % 0x100000000);
        high = ~(-val / 0x100000000);

        if (low ^ 0xffffffff) {
            low = (low + 1) | 0;
        } else {
            low = 0;
            high = (high + 1) | 0;
        }
    }

    if (val >= 0x10000000000000000 || val < -0x10000000000000000) {
        throw new Error('Given varint doesn\'t fit into 10 bytes');
    }

    pbf.realloc(10);

    writeBigVarintLow(low, high, pbf);
    writeBigVarintHigh(high, pbf);
}

function writeBigVarintLow(low, high, pbf) {
    pbf.buf[pbf.pos++] = low & 0x7f | 0x80; low >>>= 7;
    pbf.buf[pbf.pos++] = low & 0x7f | 0x80; low >>>= 7;
    pbf.buf[pbf.pos++] = low & 0x7f | 0x80; low >>>= 7;
    pbf.buf[pbf.pos++] = low & 0x7f | 0x80; low >>>= 7;
    pbf.buf[pbf.pos]   = low & 0x7f;
}

function writeBigVarintHigh(high, pbf) {
    var lsb = (high & 0x07) << 4;

    pbf.buf[pbf.pos++] |= lsb         | ((high >>>= 3) ? 0x80 : 0); if (!high) return;
    pbf.buf[pbf.pos++]  = high & 0x7f | ((high >>>= 7) ? 0x80 : 0); if (!high) return;
    pbf.buf[pbf.pos++]  = high & 0x7f | ((high >>>= 7) ? 0x80 : 0); if (!high) return;
    pbf.buf[pbf.pos++]  = high & 0x7f | ((high >>>= 7) ? 0x80 : 0); if (!high) return;
    pbf.buf[pbf.pos++]  = high & 0x7f | ((high >>>= 7) ? 0x80 : 0); if (!high) return;
    pbf.buf[pbf.pos++]  = high & 0x7f;
}

function makeRoomForExtraLength(startPos, len, pbf) {
    var extraLen =
        len <= 0x3fff ? 1 :
        len <= 0x1fffff ? 2 :
        len <= 0xfffffff ? 3 : Math.floor(Math.log(len) / (Math.LN2 * 7));

    // if 1 byte isn't enough for encoding message length, shift the data to the right
    pbf.realloc(extraLen);
    for (var i = pbf.pos - 1; i >= startPos; i--) pbf.buf[i + extraLen] = pbf.buf[i];
}

function writePackedVarint(arr, pbf)   { for (var i = 0; i < arr.length; i++) pbf.writeVarint(arr[i]);   }
function writePackedSVarint(arr, pbf)  { for (var i = 0; i < arr.length; i++) pbf.writeSVarint(arr[i]);  }
function writePackedFloat(arr, pbf)    { for (var i = 0; i < arr.length; i++) pbf.writeFloat(arr[i]);    }
function writePackedDouble(arr, pbf)   { for (var i = 0; i < arr.length; i++) pbf.writeDouble(arr[i]);   }
function writePackedBoolean(arr, pbf)  { for (var i = 0; i < arr.length; i++) pbf.writeBoolean(arr[i]);  }
function writePackedFixed32(arr, pbf)  { for (var i = 0; i < arr.length; i++) pbf.writeFixed32(arr[i]);  }
function writePackedSFixed32(arr, pbf) { for (var i = 0; i < arr.length; i++) pbf.writeSFixed32(arr[i]); }
function writePackedFixed64(arr, pbf)  { for (var i = 0; i < arr.length; i++) pbf.writeFixed64(arr[i]);  }
function writePackedSFixed64(arr, pbf) { for (var i = 0; i < arr.length; i++) pbf.writeSFixed64(arr[i]); }

// Buffer code below from https://github.com/feross/buffer, MIT-licensed

function readUInt32(buf, pos) {
    return ((buf[pos]) |
        (buf[pos + 1] << 8) |
        (buf[pos + 2] << 16)) +
        (buf[pos + 3] * 0x1000000);
}

function writeInt32(buf, val, pos) {
    buf[pos] = val;
    buf[pos + 1] = (val >>> 8);
    buf[pos + 2] = (val >>> 16);
    buf[pos + 3] = (val >>> 24);
}

function readInt32(buf, pos) {
    return ((buf[pos]) |
        (buf[pos + 1] << 8) |
        (buf[pos + 2] << 16)) +
        (buf[pos + 3] << 24);
}

function readUtf8(buf, pos, end) {
    var str = '';
    var i = pos;

    while (i < end) {
        var b0 = buf[i];
        var c = null; // codepoint
        var bytesPerSequence =
            b0 > 0xEF ? 4 :
            b0 > 0xDF ? 3 :
            b0 > 0xBF ? 2 : 1;

        if (i + bytesPerSequence > end) break;

        var b1, b2, b3;

        if (bytesPerSequence === 1) {
            if (b0 < 0x80) {
                c = b0;
            }
        } else if (bytesPerSequence === 2) {
            b1 = buf[i + 1];
            if ((b1 & 0xC0) === 0x80) {
                c = (b0 & 0x1F) << 0x6 | (b1 & 0x3F);
                if (c <= 0x7F) {
                    c = null;
                }
            }
        } else if (bytesPerSequence === 3) {
            b1 = buf[i + 1];
            b2 = buf[i + 2];
            if ((b1 & 0xC0) === 0x80 && (b2 & 0xC0) === 0x80) {
                c = (b0 & 0xF) << 0xC | (b1 & 0x3F) << 0x6 | (b2 & 0x3F);
                if (c <= 0x7FF || (c >= 0xD800 && c <= 0xDFFF)) {
                    c = null;
                }
            }
        } else if (bytesPerSequence === 4) {
            b1 = buf[i + 1];
            b2 = buf[i + 2];
            b3 = buf[i + 3];
            if ((b1 & 0xC0) === 0x80 && (b2 & 0xC0) === 0x80 && (b3 & 0xC0) === 0x80) {
                c = (b0 & 0xF) << 0x12 | (b1 & 0x3F) << 0xC | (b2 & 0x3F) << 0x6 | (b3 & 0x3F);
                if (c <= 0xFFFF || c >= 0x110000) {
                    c = null;
                }
            }
        }

        if (c === null) {
            c = 0xFFFD;
            bytesPerSequence = 1;

        } else if (c > 0xFFFF) {
            c -= 0x10000;
            str += String.fromCharCode(c >>> 10 & 0x3FF | 0xD800);
            c = 0xDC00 | c & 0x3FF;
        }

        str += String.fromCharCode(c);
        i += bytesPerSequence;
    }

    return str;
}

function readUtf8TextDecoder(buf, pos, end) {
    return utf8TextDecoder.decode(buf.subarray(pos, end));
}

function writeUtf8(buf, str, pos) {
    for (var i = 0, c, lead; i < str.length; i++) {
        c = str.charCodeAt(i); // code point

        if (c > 0xD7FF && c < 0xE000) {
            if (lead) {
                if (c < 0xDC00) {
                    buf[pos++] = 0xEF;
                    buf[pos++] = 0xBF;
                    buf[pos++] = 0xBD;
                    lead = c;
                    continue;
                } else {
                    c = lead - 0xD800 << 10 | c - 0xDC00 | 0x10000;
                    lead = null;
                }
            } else {
                if (c > 0xDBFF || (i + 1 === str.length)) {
                    buf[pos++] = 0xEF;
                    buf[pos++] = 0xBF;
                    buf[pos++] = 0xBD;
                } else {
                    lead = c;
                }
                continue;
            }
        } else if (lead) {
            buf[pos++] = 0xEF;
            buf[pos++] = 0xBF;
            buf[pos++] = 0xBD;
            lead = null;
        }

        if (c < 0x80) {
            buf[pos++] = c;
        } else {
            if (c < 0x800) {
                buf[pos++] = c >> 0x6 | 0xC0;
            } else {
                if (c < 0x10000) {
                    buf[pos++] = c >> 0xC | 0xE0;
                } else {
                    buf[pos++] = c >> 0x12 | 0xF0;
                    buf[pos++] = c >> 0xC & 0x3F | 0x80;
                }
                buf[pos++] = c >> 0x6 & 0x3F | 0x80;
            }
            buf[pos++] = c & 0x3F | 0x80;
        }
    }
    return pos;
}


/***/ }),

/***/ 107:
/***/ (function(__unused_webpack_module, __unused_webpack_exports, __webpack_require__) {

(function (global, factory) {
	 true ? factory() :
	0;
}(this, (function () { 'use strict';

/**
 * @this {Promise}
 */
function finallyConstructor(callback) {
  var constructor = this.constructor;
  return this.then(
    function(value) {
      // @ts-ignore
      return constructor.resolve(callback()).then(function() {
        return value;
      });
    },
    function(reason) {
      // @ts-ignore
      return constructor.resolve(callback()).then(function() {
        // @ts-ignore
        return constructor.reject(reason);
      });
    }
  );
}

function allSettled(arr) {
  var P = this;
  return new P(function(resolve, reject) {
    if (!(arr && typeof arr.length !== 'undefined')) {
      return reject(
        new TypeError(
          typeof arr +
            ' ' +
            arr +
            ' is not iterable(cannot read property Symbol(Symbol.iterator))'
        )
      );
    }
    var args = Array.prototype.slice.call(arr);
    if (args.length === 0) return resolve([]);
    var remaining = args.length;

    function res(i, val) {
      if (val && (typeof val === 'object' || typeof val === 'function')) {
        var then = val.then;
        if (typeof then === 'function') {
          then.call(
            val,
            function(val) {
              res(i, val);
            },
            function(e) {
              args[i] = { status: 'rejected', reason: e };
              if (--remaining === 0) {
                resolve(args);
              }
            }
          );
          return;
        }
      }
      args[i] = { status: 'fulfilled', value: val };
      if (--remaining === 0) {
        resolve(args);
      }
    }

    for (var i = 0; i < args.length; i++) {
      res(i, args[i]);
    }
  });
}

// Store setTimeout reference so promise-polyfill will be unaffected by
// other code modifying setTimeout (like sinon.useFakeTimers())
var setTimeoutFunc = setTimeout;

function isArray(x) {
  return Boolean(x && typeof x.length !== 'undefined');
}

function noop() {}

// Polyfill for Function.prototype.bind
function bind(fn, thisArg) {
  return function() {
    fn.apply(thisArg, arguments);
  };
}

/**
 * @constructor
 * @param {Function} fn
 */
function Promise(fn) {
  if (!(this instanceof Promise))
    throw new TypeError('Promises must be constructed via new');
  if (typeof fn !== 'function') throw new TypeError('not a function');
  /** @type {!number} */
  this._state = 0;
  /** @type {!boolean} */
  this._handled = false;
  /** @type {Promise|undefined} */
  this._value = undefined;
  /** @type {!Array<!Function>} */
  this._deferreds = [];

  doResolve(fn, this);
}

function handle(self, deferred) {
  while (self._state === 3) {
    self = self._value;
  }
  if (self._state === 0) {
    self._deferreds.push(deferred);
    return;
  }
  self._handled = true;
  Promise._immediateFn(function() {
    var cb = self._state === 1 ? deferred.onFulfilled : deferred.onRejected;
    if (cb === null) {
      (self._state === 1 ? resolve : reject)(deferred.promise, self._value);
      return;
    }
    var ret;
    try {
      ret = cb(self._value);
    } catch (e) {
      reject(deferred.promise, e);
      return;
    }
    resolve(deferred.promise, ret);
  });
}

function resolve(self, newValue) {
  try {
    // Promise Resolution Procedure: https://github.com/promises-aplus/promises-spec#the-promise-resolution-procedure
    if (newValue === self)
      throw new TypeError('A promise cannot be resolved with itself.');
    if (
      newValue &&
      (typeof newValue === 'object' || typeof newValue === 'function')
    ) {
      var then = newValue.then;
      if (newValue instanceof Promise) {
        self._state = 3;
        self._value = newValue;
        finale(self);
        return;
      } else if (typeof then === 'function') {
        doResolve(bind(then, newValue), self);
        return;
      }
    }
    self._state = 1;
    self._value = newValue;
    finale(self);
  } catch (e) {
    reject(self, e);
  }
}

function reject(self, newValue) {
  self._state = 2;
  self._value = newValue;
  finale(self);
}

function finale(self) {
  if (self._state === 2 && self._deferreds.length === 0) {
    Promise._immediateFn(function() {
      if (!self._handled) {
        Promise._unhandledRejectionFn(self._value);
      }
    });
  }

  for (var i = 0, len = self._deferreds.length; i < len; i++) {
    handle(self, self._deferreds[i]);
  }
  self._deferreds = null;
}

/**
 * @constructor
 */
function Handler(onFulfilled, onRejected, promise) {
  this.onFulfilled = typeof onFulfilled === 'function' ? onFulfilled : null;
  this.onRejected = typeof onRejected === 'function' ? onRejected : null;
  this.promise = promise;
}

/**
 * Take a potentially misbehaving resolver function and make sure
 * onFulfilled and onRejected are only called once.
 *
 * Makes no guarantees about asynchrony.
 */
function doResolve(fn, self) {
  var done = false;
  try {
    fn(
      function(value) {
        if (done) return;
        done = true;
        resolve(self, value);
      },
      function(reason) {
        if (done) return;
        done = true;
        reject(self, reason);
      }
    );
  } catch (ex) {
    if (done) return;
    done = true;
    reject(self, ex);
  }
}

Promise.prototype['catch'] = function(onRejected) {
  return this.then(null, onRejected);
};

Promise.prototype.then = function(onFulfilled, onRejected) {
  // @ts-ignore
  var prom = new this.constructor(noop);

  handle(this, new Handler(onFulfilled, onRejected, prom));
  return prom;
};

Promise.prototype['finally'] = finallyConstructor;

Promise.all = function(arr) {
  return new Promise(function(resolve, reject) {
    if (!isArray(arr)) {
      return reject(new TypeError('Promise.all accepts an array'));
    }

    var args = Array.prototype.slice.call(arr);
    if (args.length === 0) return resolve([]);
    var remaining = args.length;

    function res(i, val) {
      try {
        if (val && (typeof val === 'object' || typeof val === 'function')) {
          var then = val.then;
          if (typeof then === 'function') {
            then.call(
              val,
              function(val) {
                res(i, val);
              },
              reject
            );
            return;
          }
        }
        args[i] = val;
        if (--remaining === 0) {
          resolve(args);
        }
      } catch (ex) {
        reject(ex);
      }
    }

    for (var i = 0; i < args.length; i++) {
      res(i, args[i]);
    }
  });
};

Promise.allSettled = allSettled;

Promise.resolve = function(value) {
  if (value && typeof value === 'object' && value.constructor === Promise) {
    return value;
  }

  return new Promise(function(resolve) {
    resolve(value);
  });
};

Promise.reject = function(value) {
  return new Promise(function(resolve, reject) {
    reject(value);
  });
};

Promise.race = function(arr) {
  return new Promise(function(resolve, reject) {
    if (!isArray(arr)) {
      return reject(new TypeError('Promise.race accepts an array'));
    }

    for (var i = 0, len = arr.length; i < len; i++) {
      Promise.resolve(arr[i]).then(resolve, reject);
    }
  });
};

// Use polyfill for setImmediate for performance gains
Promise._immediateFn =
  // @ts-ignore
  (typeof setImmediate === 'function' &&
    function(fn) {
      // @ts-ignore
      setImmediate(fn);
    }) ||
  function(fn) {
    setTimeoutFunc(fn, 0);
  };

Promise._unhandledRejectionFn = function _unhandledRejectionFn(err) {
  if (typeof console !== 'undefined' && console) {
    console.warn('Possible Unhandled Promise Rejection:', err); // eslint-disable-line no-console
  }
};

/** @suppress {undefinedVars} */
var globalNS = (function() {
  // the only reliable means to get the global object is
  // `Function('return this')()`
  // However, this causes CSP violations in Chrome apps.
  if (typeof self !== 'undefined') {
    return self;
  }
  if (typeof window !== 'undefined') {
    return window;
  }
  if (typeof __webpack_require__.g !== 'undefined') {
    return __webpack_require__.g;
  }
  throw new Error('unable to locate global object');
})();

// Expose the polyfill if Promise is undefined or set to a
// non-function value. The latter can be due to a named HTMLElement
// being exposed by browsers for legacy reasons.
// https://github.com/taylorhakes/promise-polyfill/issues/114
if (typeof globalNS['Promise'] !== 'function') {
  globalNS['Promise'] = Promise;
} else {
  if (!globalNS.Promise.prototype['finally']) {
    globalNS.Promise.prototype['finally'] = finallyConstructor;
  } 
  if (!globalNS.Promise.allSettled) {
    globalNS.Promise.allSettled = allSettled;
  }
}

})));


/***/ }),

/***/ 166:
/***/ (function(module) {

(function (global, factory) {
	 true ? module.exports = factory() :
	0;
}(this, (function () { 'use strict';

function quickselect(arr, k, left, right, compare) {
    quickselectStep(arr, k, left || 0, right || (arr.length - 1), compare || defaultCompare);
}

function quickselectStep(arr, k, left, right, compare) {

    while (right > left) {
        if (right - left > 600) {
            var n = right - left + 1;
            var m = k - left + 1;
            var z = Math.log(n);
            var s = 0.5 * Math.exp(2 * z / 3);
            var sd = 0.5 * Math.sqrt(z * s * (n - s) / n) * (m - n / 2 < 0 ? -1 : 1);
            var newLeft = Math.max(left, Math.floor(k - m * s / n + sd));
            var newRight = Math.min(right, Math.floor(k + (n - m) * s / n + sd));
            quickselectStep(arr, k, newLeft, newRight, compare);
        }

        var t = arr[k];
        var i = left;
        var j = right;

        swap(arr, left, k);
        if (compare(arr[right], t) > 0) swap(arr, left, right);

        while (i < j) {
            swap(arr, i, j);
            i++;
            j--;
            while (compare(arr[i], t) < 0) i++;
            while (compare(arr[j], t) > 0) j--;
        }

        if (compare(arr[left], t) === 0) swap(arr, left, j);
        else {
            j++;
            swap(arr, j, right);
        }

        if (j <= k) left = j + 1;
        if (k <= j) right = j - 1;
    }
}

function swap(arr, i, j) {
    var tmp = arr[i];
    arr[i] = arr[j];
    arr[j] = tmp;
}

function defaultCompare(a, b) {
    return a < b ? -1 : a > b ? 1 : 0;
}

return quickselect;

})));


/***/ }),

/***/ 510:
/***/ ((module, __unused_webpack_exports, __webpack_require__) => {

"use strict";


module.exports = rbush;
module.exports["default"] = rbush;

var quickselect = __webpack_require__(166);

function rbush(maxEntries, format) {
    if (!(this instanceof rbush)) return new rbush(maxEntries, format);

    // max entries in a node is 9 by default; min node fill is 40% for best performance
    this._maxEntries = Math.max(4, maxEntries || 9);
    this._minEntries = Math.max(2, Math.ceil(this._maxEntries * 0.4));

    if (format) {
        this._initFormat(format);
    }

    this.clear();
}

rbush.prototype = {

    all: function () {
        return this._all(this.data, []);
    },

    search: function (bbox) {

        var node = this.data,
            result = [],
            toBBox = this.toBBox;

        if (!intersects(bbox, node)) return result;

        var nodesToSearch = [],
            i, len, child, childBBox;

        while (node) {
            for (i = 0, len = node.children.length; i < len; i++) {

                child = node.children[i];
                childBBox = node.leaf ? toBBox(child) : child;

                if (intersects(bbox, childBBox)) {
                    if (node.leaf) result.push(child);
                    else if (contains(bbox, childBBox)) this._all(child, result);
                    else nodesToSearch.push(child);
                }
            }
            node = nodesToSearch.pop();
        }

        return result;
    },

    collides: function (bbox) {

        var node = this.data,
            toBBox = this.toBBox;

        if (!intersects(bbox, node)) return false;

        var nodesToSearch = [],
            i, len, child, childBBox;

        while (node) {
            for (i = 0, len = node.children.length; i < len; i++) {

                child = node.children[i];
                childBBox = node.leaf ? toBBox(child) : child;

                if (intersects(bbox, childBBox)) {
                    if (node.leaf || contains(bbox, childBBox)) return true;
                    nodesToSearch.push(child);
                }
            }
            node = nodesToSearch.pop();
        }

        return false;
    },

    load: function (data) {
        if (!(data && data.length)) return this;

        if (data.length < this._minEntries) {
            for (var i = 0, len = data.length; i < len; i++) {
                this.insert(data[i]);
            }
            return this;
        }

        // recursively build the tree with the given data from scratch using OMT algorithm
        var node = this._build(data.slice(), 0, data.length - 1, 0);

        if (!this.data.children.length) {
            // save as is if tree is empty
            this.data = node;

        } else if (this.data.height === node.height) {
            // split root if trees have the same height
            this._splitRoot(this.data, node);

        } else {
            if (this.data.height < node.height) {
                // swap trees if inserted one is bigger
                var tmpNode = this.data;
                this.data = node;
                node = tmpNode;
            }

            // insert the small tree into the large tree at appropriate level
            this._insert(node, this.data.height - node.height - 1, true);
        }

        return this;
    },

    insert: function (item) {
        if (item) this._insert(item, this.data.height - 1);
        return this;
    },

    clear: function () {
        this.data = createNode([]);
        return this;
    },

    remove: function (item, equalsFn) {
        if (!item) return this;

        var node = this.data,
            bbox = this.toBBox(item),
            path = [],
            indexes = [],
            i, parent, index, goingUp;

        // depth-first iterative tree traversal
        while (node || path.length) {

            if (!node) { // go up
                node = path.pop();
                parent = path[path.length - 1];
                i = indexes.pop();
                goingUp = true;
            }

            if (node.leaf) { // check current node
                index = findItem(item, node.children, equalsFn);

                if (index !== -1) {
                    // item found, remove the item and condense tree upwards
                    node.children.splice(index, 1);
                    path.push(node);
                    this._condense(path);
                    return this;
                }
            }

            if (!goingUp && !node.leaf && contains(node, bbox)) { // go down
                path.push(node);
                indexes.push(i);
                i = 0;
                parent = node;
                node = node.children[0];

            } else if (parent) { // go right
                i++;
                node = parent.children[i];
                goingUp = false;

            } else node = null; // nothing found
        }

        return this;
    },

    toBBox: function (item) { return item; },

    compareMinX: compareNodeMinX,
    compareMinY: compareNodeMinY,

    toJSON: function () { return this.data; },

    fromJSON: function (data) {
        this.data = data;
        return this;
    },

    _all: function (node, result) {
        var nodesToSearch = [];
        while (node) {
            if (node.leaf) result.push.apply(result, node.children);
            else nodesToSearch.push.apply(nodesToSearch, node.children);

            node = nodesToSearch.pop();
        }
        return result;
    },

    _build: function (items, left, right, height) {

        var N = right - left + 1,
            M = this._maxEntries,
            node;

        if (N <= M) {
            // reached leaf level; return leaf
            node = createNode(items.slice(left, right + 1));
            calcBBox(node, this.toBBox);
            return node;
        }

        if (!height) {
            // target height of the bulk-loaded tree
            height = Math.ceil(Math.log(N) / Math.log(M));

            // target number of root entries to maximize storage utilization
            M = Math.ceil(N / Math.pow(M, height - 1));
        }

        node = createNode([]);
        node.leaf = false;
        node.height = height;

        // split the items into M mostly square tiles

        var N2 = Math.ceil(N / M),
            N1 = N2 * Math.ceil(Math.sqrt(M)),
            i, j, right2, right3;

        multiSelect(items, left, right, N1, this.compareMinX);

        for (i = left; i <= right; i += N1) {

            right2 = Math.min(i + N1 - 1, right);

            multiSelect(items, i, right2, N2, this.compareMinY);

            for (j = i; j <= right2; j += N2) {

                right3 = Math.min(j + N2 - 1, right2);

                // pack each entry recursively
                node.children.push(this._build(items, j, right3, height - 1));
            }
        }

        calcBBox(node, this.toBBox);

        return node;
    },

    _chooseSubtree: function (bbox, node, level, path) {

        var i, len, child, targetNode, area, enlargement, minArea, minEnlargement;

        while (true) {
            path.push(node);

            if (node.leaf || path.length - 1 === level) break;

            minArea = minEnlargement = Infinity;

            for (i = 0, len = node.children.length; i < len; i++) {
                child = node.children[i];
                area = bboxArea(child);
                enlargement = enlargedArea(bbox, child) - area;

                // choose entry with the least area enlargement
                if (enlargement < minEnlargement) {
                    minEnlargement = enlargement;
                    minArea = area < minArea ? area : minArea;
                    targetNode = child;

                } else if (enlargement === minEnlargement) {
                    // otherwise choose one with the smallest area
                    if (area < minArea) {
                        minArea = area;
                        targetNode = child;
                    }
                }
            }

            node = targetNode || node.children[0];
        }

        return node;
    },

    _insert: function (item, level, isNode) {

        var toBBox = this.toBBox,
            bbox = isNode ? item : toBBox(item),
            insertPath = [];

        // find the best node for accommodating the item, saving all nodes along the path too
        var node = this._chooseSubtree(bbox, this.data, level, insertPath);

        // put the item into the node
        node.children.push(item);
        extend(node, bbox);

        // split on node overflow; propagate upwards if necessary
        while (level >= 0) {
            if (insertPath[level].children.length > this._maxEntries) {
                this._split(insertPath, level);
                level--;
            } else break;
        }

        // adjust bboxes along the insertion path
        this._adjustParentBBoxes(bbox, insertPath, level);
    },

    // split overflowed node into two
    _split: function (insertPath, level) {

        var node = insertPath[level],
            M = node.children.length,
            m = this._minEntries;

        this._chooseSplitAxis(node, m, M);

        var splitIndex = this._chooseSplitIndex(node, m, M);

        var newNode = createNode(node.children.splice(splitIndex, node.children.length - splitIndex));
        newNode.height = node.height;
        newNode.leaf = node.leaf;

        calcBBox(node, this.toBBox);
        calcBBox(newNode, this.toBBox);

        if (level) insertPath[level - 1].children.push(newNode);
        else this._splitRoot(node, newNode);
    },

    _splitRoot: function (node, newNode) {
        // split root node
        this.data = createNode([node, newNode]);
        this.data.height = node.height + 1;
        this.data.leaf = false;
        calcBBox(this.data, this.toBBox);
    },

    _chooseSplitIndex: function (node, m, M) {

        var i, bbox1, bbox2, overlap, area, minOverlap, minArea, index;

        minOverlap = minArea = Infinity;

        for (i = m; i <= M - m; i++) {
            bbox1 = distBBox(node, 0, i, this.toBBox);
            bbox2 = distBBox(node, i, M, this.toBBox);

            overlap = intersectionArea(bbox1, bbox2);
            area = bboxArea(bbox1) + bboxArea(bbox2);

            // choose distribution with minimum overlap
            if (overlap < minOverlap) {
                minOverlap = overlap;
                index = i;

                minArea = area < minArea ? area : minArea;

            } else if (overlap === minOverlap) {
                // otherwise choose distribution with minimum area
                if (area < minArea) {
                    minArea = area;
                    index = i;
                }
            }
        }

        return index;
    },

    // sorts node children by the best axis for split
    _chooseSplitAxis: function (node, m, M) {

        var compareMinX = node.leaf ? this.compareMinX : compareNodeMinX,
            compareMinY = node.leaf ? this.compareMinY : compareNodeMinY,
            xMargin = this._allDistMargin(node, m, M, compareMinX),
            yMargin = this._allDistMargin(node, m, M, compareMinY);

        // if total distributions margin value is minimal for x, sort by minX,
        // otherwise it's already sorted by minY
        if (xMargin < yMargin) node.children.sort(compareMinX);
    },

    // total margin of all possible split distributions where each node is at least m full
    _allDistMargin: function (node, m, M, compare) {

        node.children.sort(compare);

        var toBBox = this.toBBox,
            leftBBox = distBBox(node, 0, m, toBBox),
            rightBBox = distBBox(node, M - m, M, toBBox),
            margin = bboxMargin(leftBBox) + bboxMargin(rightBBox),
            i, child;

        for (i = m; i < M - m; i++) {
            child = node.children[i];
            extend(leftBBox, node.leaf ? toBBox(child) : child);
            margin += bboxMargin(leftBBox);
        }

        for (i = M - m - 1; i >= m; i--) {
            child = node.children[i];
            extend(rightBBox, node.leaf ? toBBox(child) : child);
            margin += bboxMargin(rightBBox);
        }

        return margin;
    },

    _adjustParentBBoxes: function (bbox, path, level) {
        // adjust bboxes along the given tree path
        for (var i = level; i >= 0; i--) {
            extend(path[i], bbox);
        }
    },

    _condense: function (path) {
        // go through the path, removing empty nodes and updating bboxes
        for (var i = path.length - 1, siblings; i >= 0; i--) {
            if (path[i].children.length === 0) {
                if (i > 0) {
                    siblings = path[i - 1].children;
                    siblings.splice(siblings.indexOf(path[i]), 1);

                } else this.clear();

            } else calcBBox(path[i], this.toBBox);
        }
    },

    _initFormat: function (format) {
        // data format (minX, minY, maxX, maxY accessors)

        // uses eval-type function compilation instead of just accepting a toBBox function
        // because the algorithms are very sensitive to sorting functions performance,
        // so they should be dead simple and without inner calls

        var compareArr = ['return a', ' - b', ';'];

        this.compareMinX = new Function('a', 'b', compareArr.join(format[0]));
        this.compareMinY = new Function('a', 'b', compareArr.join(format[1]));

        this.toBBox = new Function('a',
            'return {minX: a' + format[0] +
            ', minY: a' + format[1] +
            ', maxX: a' + format[2] +
            ', maxY: a' + format[3] + '};');
    }
};

function findItem(item, items, equalsFn) {
    if (!equalsFn) return items.indexOf(item);

    for (var i = 0; i < items.length; i++) {
        if (equalsFn(item, items[i])) return i;
    }
    return -1;
}

// calculate node's bbox from bboxes of its children
function calcBBox(node, toBBox) {
    distBBox(node, 0, node.children.length, toBBox, node);
}

// min bounding rectangle of node children from k to p-1
function distBBox(node, k, p, toBBox, destNode) {
    if (!destNode) destNode = createNode(null);
    destNode.minX = Infinity;
    destNode.minY = Infinity;
    destNode.maxX = -Infinity;
    destNode.maxY = -Infinity;

    for (var i = k, child; i < p; i++) {
        child = node.children[i];
        extend(destNode, node.leaf ? toBBox(child) : child);
    }

    return destNode;
}

function extend(a, b) {
    a.minX = Math.min(a.minX, b.minX);
    a.minY = Math.min(a.minY, b.minY);
    a.maxX = Math.max(a.maxX, b.maxX);
    a.maxY = Math.max(a.maxY, b.maxY);
    return a;
}

function compareNodeMinX(a, b) { return a.minX - b.minX; }
function compareNodeMinY(a, b) { return a.minY - b.minY; }

function bboxArea(a)   { return (a.maxX - a.minX) * (a.maxY - a.minY); }
function bboxMargin(a) { return (a.maxX - a.minX) + (a.maxY - a.minY); }

function enlargedArea(a, b) {
    return (Math.max(b.maxX, a.maxX) - Math.min(b.minX, a.minX)) *
           (Math.max(b.maxY, a.maxY) - Math.min(b.minY, a.minY));
}

function intersectionArea(a, b) {
    var minX = Math.max(a.minX, b.minX),
        minY = Math.max(a.minY, b.minY),
        maxX = Math.min(a.maxX, b.maxX),
        maxY = Math.min(a.maxY, b.maxY);

    return Math.max(0, maxX - minX) *
           Math.max(0, maxY - minY);
}

function contains(a, b) {
    return a.minX <= b.minX &&
           a.minY <= b.minY &&
           b.maxX <= a.maxX &&
           b.maxY <= a.maxY;
}

function intersects(a, b) {
    return b.minX <= a.maxX &&
           b.minY <= a.maxY &&
           b.maxX >= a.minX &&
           b.maxY >= a.minY;
}

function createNode(children) {
    return {
        children: children,
        height: 1,
        leaf: true,
        minX: Infinity,
        minY: Infinity,
        maxX: -Infinity,
        maxY: -Infinity
    };
}

// sort an array so that items come in groups of n unsorted items, with groups sorted between each other;
// combines selection algorithm with binary divide & conquer approach

function multiSelect(arr, left, right, n, compare) {
    var stack = [left, right],
        mid;

    while (stack.length) {
        right = stack.pop();
        left = stack.pop();

        if (right - left <= n) continue;

        mid = left + Math.ceil((right - left) / n / 2) * n;
        quickselect(arr, mid, left, right, compare);

        stack.push(left, mid, mid, right);
    }
}


/***/ })

/******/ 	});
/************************************************************************/
/******/ 	// The module cache
/******/ 	var __webpack_module_cache__ = {};
/******/ 	
/******/ 	// The require function
/******/ 	function __webpack_require__(moduleId) {
/******/ 		// Check if module is in cache
/******/ 		var cachedModule = __webpack_module_cache__[moduleId];
/******/ 		if (cachedModule !== undefined) {
/******/ 			return cachedModule.exports;
/******/ 		}
/******/ 		// Create a new module (and put it into the cache)
/******/ 		var module = __webpack_module_cache__[moduleId] = {
/******/ 			id: moduleId,
/******/ 			loaded: false,
/******/ 			exports: {}
/******/ 		};
/******/ 	
/******/ 		// Execute the module function
/******/ 		__webpack_modules__[moduleId].call(module.exports, module, module.exports, __webpack_require__);
/******/ 	
/******/ 		// Flag the module as loaded
/******/ 		module.loaded = true;
/******/ 	
/******/ 		// Return the exports of the module
/******/ 		return module.exports;
/******/ 	}
/******/ 	
/************************************************************************/
/******/ 	/* webpack/runtime/compat get default export */
/******/ 	(() => {
/******/ 		// getDefaultExport function for compatibility with non-harmony modules
/******/ 		__webpack_require__.n = (module) => {
/******/ 			var getter = module && module.__esModule ?
/******/ 				() => (module['default']) :
/******/ 				() => (module);
/******/ 			__webpack_require__.d(getter, { a: getter });
/******/ 			return getter;
/******/ 		};
/******/ 	})();
/******/ 	
/******/ 	/* webpack/runtime/define property getters */
/******/ 	(() => {
/******/ 		// define getter functions for harmony exports
/******/ 		__webpack_require__.d = (exports, definition) => {
/******/ 			for(var key in definition) {
/******/ 				if(__webpack_require__.o(definition, key) && !__webpack_require__.o(exports, key)) {
/******/ 					Object.defineProperty(exports, key, { enumerable: true, get: definition[key] });
/******/ 				}
/******/ 			}
/******/ 		};
/******/ 	})();
/******/ 	
/******/ 	/* webpack/runtime/global */
/******/ 	(() => {
/******/ 		__webpack_require__.g = (function() {
/******/ 			if (typeof globalThis === 'object') return globalThis;
/******/ 			try {
/******/ 				return this || new Function('return this')();
/******/ 			} catch (e) {
/******/ 				if (typeof window === 'object') return window;
/******/ 			}
/******/ 		})();
/******/ 	})();
/******/ 	
/******/ 	/* webpack/runtime/hasOwnProperty shorthand */
/******/ 	(() => {
/******/ 		__webpack_require__.o = (obj, prop) => (Object.prototype.hasOwnProperty.call(obj, prop))
/******/ 	})();
/******/ 	
/******/ 	/* webpack/runtime/node module decorator */
/******/ 	(() => {
/******/ 		__webpack_require__.nmd = (module) => {
/******/ 			module.paths = [];
/******/ 			if (!module.children) module.children = [];
/******/ 			return module;
/******/ 		};
/******/ 	})();
/******/ 	
/************************************************************************/
var __webpack_exports__ = {};
// This entry need to be wrapped in an IIFE because it need to be in strict mode.
(() => {
"use strict";

// UNUSED EXPORTS: AddressMatchService, AggregationParameter, AggregationTypes, AlongLineDirection, AnalystAreaUnit, AnalystSizeUnit, AreaSolarRadiationParameters, ArrayStatistic, AttributesPopContainer, BaiduCRS, BaiduTileLayer, Bounds, BoundsType, Browser, BucketAggParameter, BucketAggType, BufferAnalystParameters, BufferDistance, BufferEndType, BufferRadiusUnit, BufferSetting, BuffersAnalystJobsParameter, BurstPipelineAnalystParameters, CRS, CartoCSS, CellSizeType, ChangeTileVersion, ChartQueryFilterParameter, ChartQueryParameters, ChartService, ChartType, ChartView, ChartViewModel, CircleStyle, CityTabsPage, ClientComputationLayer, ClientComputationView, ClientComputationViewModel, ClientType, ClipAnalystMode, ClipParameter, CloudTileLayer, CloverStyle, ColorDictionary, ColorGradientType, ColorSpaceType, ColorsPickerUtil, CommonContainer, CommonServiceBase, CommonTheme, CommonUtil, CommontypesConversion, ComponentsUtil, ComponentsViewBase, ComputeWeightMatrixParameters, ConnectedEdgesAnalystParameters, ConvexHullAnalystParameters, CreateDatasetParameters, Credential, DataFlowLayer, DataFlowService, DataFlowView, DataFlowViewModel, DataFormat, DataItemOrderBy, DataItemType, DataReturnMode, DataReturnOption, DataServiceQueryView, DataServiceQueryViewModel, DatasetBufferAnalystParameters, DatasetInfo, DatasetMinDistanceAnalystParameters, DatasetOverlayAnalystParameters, DatasetService, DatasetSurfaceAnalystParameters, DatasetThiessenAnalystParameters, DatasourceConnectionInfo, DatasourceService, DensityKernelAnalystParameters, DirectionType, DistributedAnalysisView, DistributedAnalysisViewModel, DropDownBox, EchartsLayer, EditFeaturesParameters, EditType, ElasticSearch, EngineType, EntityType, Event, Events, Exponent, FGBLayer, FacilityAnalyst3DParameters, FacilityAnalystSinks3DParameters, FacilityAnalystSources3DParameters, FacilityAnalystStreamParameters, FacilityAnalystTracedown3DParameters, FacilityAnalystTraceup3DParameters, FacilityAnalystUpstream3DParameters, Feature, FeatureService, FeatureShapeFactory, FeatureTheme, FeatureThemeGraph, FeatureThemeRankSymbol, FeatureThemeVector, FeatureVector, FetchRequest, FieldParameters, FieldService, FieldStatisticsParameters, FieldsFilter, FileReaderUtil, FillGradientMode, FilterField, FilterParameter, FindClosestFacilitiesParameters, FindLocationParameters, FindMTSPPathsParameters, FindPathParameters, FindServiceAreasParameters, FindTSPPathsParameters, Format, GenerateSpatialDataParameters, GeoCodingParameter, GeoDecodingParameter, GeoFeatureThemeLayer, GeoHashGridAggParameter, GeoJSONFormat, GeoJSONLayerWithName, GeoJsonLayerDataModel, GeoRelationAnalystParameters, Geometry, Geometry3D, GeometryBufferAnalystParameters, GeometryCollection, GeometryCurve, GeometryGeoText, GeometryLineString, GeometryLinearRing, GeometryMinDistanceAnalystParameters, GeometryMultiLineString, GeometryMultiPoint, GeometryMultiPolygon, GeometryOverlayAnalystParameters, GeometryPoint, GeometryPolygon, GeometryRectangle, GeometrySurfaceAnalystParameters, GeometryThiessenAnalystParameters, GeometryType, GeoprocessingService, GetFeatureMode, GetFeaturesByBoundsParameters, GetFeaturesByBufferParameters, GetFeaturesByGeometryParameters, GetFeaturesByIDsParameters, GetFeaturesBySQLParameters, GetFeaturesParametersBase, GetFeaturesServiceBase, GetGridCellInfosParameters, GetLayersLegendInfoParameters, GetResolutionFromScaleDpi, GraduatedMode, GraphAxesTextDisplayMode, GraphMap, GraphThemeLayer, Graphic, GraphicLayer, Grid, GridCellInfosService, GridType, HeatMapFeature, HeatMapLayer, HillshadeParameter, IManager, IManagerCreateNodeParam, IManagerServiceBase, IPortal, IPortalAddDataParam, IPortalAddResourceParam, IPortalDataConnectionInfoParam, IPortalDataMetaInfoParam, IPortalDataStoreInfoParam, IPortalQueryParam, IPortalQueryResult, IPortalRegisterServiceParam, IPortalResource, IPortalServiceBase, IPortalShareEntity, IPortalShareParam, IPortalUser, ImageCollectionService, ImageGFAspect, ImageGFHillShade, ImageGFOrtho, ImageGFSlope, ImageMapLayer, ImageRenderingRule, ImageSearchParameter, ImageService, ImageStretchOption, ImageStyle, ImageTileLayer, IndexTabsPageContainer, InterpolationAlgorithmType, InterpolationAnalystParameters, InterpolationDensityAnalystParameters, InterpolationIDWAnalystParameters, InterpolationKrigingAnalystParameters, InterpolationRBFAnalystParameters, JSONFormat, JoinItem, JoinType, KernelDensityJobParameter, KeyServiceParameter, KnowledgeGraph, KnowledgeGraphService, LabelBackShape, LabelImageCell, LabelMatrixCell, LabelMixedTextStyle, LabelOverLengthMode, LabelSymbolCell, LabelThemeCell, LabelThemeLayer, Lang, LayerInfoService, LayerStatus, LayerType, LinkItem, Logo, LonLat, MapService, MapVLayer, MapVRenderer, MappingParameters, MapvRenderer, MathExpressionAnalysisParameters, MeasureMode, MeasureParameters, MeasureService, MessageBox, MetricsAggParameter, MetricsAggType, NDVIParameter, NavTabsPage, NetworkAnalyst3DService, NetworkAnalystService, NetworkAnalystServiceBase, NonEarthCRS, NormalRenderer, NormalizeScale, Online, OnlineData, OnlineQueryDatasParameter, OnlineServiceBase, OpenFileView, OpenFileViewModel, OrderBy, OrderType, OutputSetting, OutputType, OverlapDisplayedOptions, OverlayAnalystParameters, OverlayGeoJobParameter, OverlayOperationType, PaginationContainer, PermissionType, Pixel, PixelFormat, PointWithMeasure, PopContainer, ProcessingService, ProcessingServiceBase, QueryByBoundsParameters, QueryByDistanceParameters, QueryByGeometryParameters, QueryBySQLParameters, QueryOption, QueryParameters, QueryService, RangeMode, RangeThemeLayer, RankSymbolThemeLayer, RasterFunctionParameter, RasterFunctionType, ResourceType, Route, RouteCalculateMeasureParameters, RouteLocatorParameters, SVGRenderer, SearchMode, SearchType, SearchView, SearchViewModel, SecurityManager, Select, ServerColor, ServerFeature, ServerGeometry, ServerInfo, ServerStyle, ServerTextStyle, ServerTheme, ServerType, ServiceBase, ServiceStatus, SetDatasourceParameters, SetLayerInfoParameters, SetLayerStatusParameters, SetLayersInfoParameters, ShapeParameters, ShapeParametersCircle, ShapeParametersImage, ShapeParametersLabel, ShapeParametersLine, ShapeParametersPoint, ShapeParametersPolygon, ShapeParametersRectangle, ShapeParametersSector, SideType, SingleObjectQueryJobsParameter, Size, SmoothMethod, Sortby, SpatialAnalystBase, SpatialAnalystService, SpatialQueryMode, SpatialRelationType, StatisticAnalystMode, StatisticMode, StopQueryParameters, SummaryAttributesJobsParameter, SummaryMeshJobParameter, SummaryRegionJobParameter, SummaryType, SuperMap, SupplyCenter, SupplyCenterType, SurfaceAnalystMethod, SurfaceAnalystParameters, SurfaceAnalystParametersSetting, TemplateBase, TerrainAspectCalculationParameters, TerrainCurvatureCalculationParameters, TerrainCutFillCalculationParameters, TerrainSlopeCalculationParameters, TextAlignment, ThemeDotDensity, ThemeFeature, ThemeGraduatedSymbol, ThemeGraduatedSymbolStyle, ThemeGraph, ThemeGraphAxes, ThemeGraphItem, ThemeGraphSize, ThemeGraphText, ThemeGraphTextFormat, ThemeGraphType, ThemeGridRange, ThemeGridRangeItem, ThemeGridUnique, ThemeGridUniqueItem, ThemeLabel, ThemeLabelAlongLine, ThemeLabelBackground, ThemeLabelItem, ThemeLabelText, ThemeLabelUniqueItem, ThemeLayer, ThemeMemoryData, ThemeOffset, ThemeParameters, ThemeRange, ThemeRangeItem, ThemeService, ThemeStyle, ThemeType, ThemeUnique, ThemeUniqueItem, ThiessenAnalystParameters, TianDiTu_MercatorCRS, TianDiTu_WGS84CRS, TiandituTileLayer, TileVectorLayer, TiledMapLayer, TiledVectorLayer, TimeControlBase, TimeFlowControl, TokenServiceParameter, TopologyValidatorJobsParameter, TopologyValidatorRule, TraceAnalystParameters, TrafficTransferAnalystService, TransferLine, TransferPathParameters, TransferPreference, TransferSolutionParameters, TransferTactic, Transform, TransportationAnalystParameter, TransportationAnalystResultSetting, TurfLayer, TurnType, UGCLayer, UGCLayerType, UGCMapLayer, UGCSubLayer, UnicodeMarker, UniqueThemeLayer, Unit, UpdateDatasetParameters, UpdateEdgeWeightParameters, UpdateTurnNodeWeightParameters, VariogramMode, Vector, VectorClipJobsParameter, VectorGrid, VectorTileFormat, WKTFormat, WMTSLayer, WebExportFormatType, WebMap, WebPrintingJobContent, WebPrintingJobCustomItems, WebPrintingJobExportOptions, WebPrintingJobImage, WebPrintingJobLayers, WebPrintingJobLayoutOptions, WebPrintingJobLegendOptions, WebPrintingJobLittleMapOptions, WebPrintingJobNorthArrowOptions, WebPrintingJobParameters, WebPrintingJobScaleBarOptions, WebPrintingJobService, WebScaleOrientationType, WebScaleType, WebScaleUnit, addressMatchService, baiduTileLayer, changeTileVersion, chartService, circleStyle, clientComputationLayer, clientComputationView, cloudTileLayer, cloverStyle, componentsViewBase, conversionDegree, crs, crsFromMapJSON, dataFlowLayer, dataFlowService, dataFlowView, dataFlowViewModel, dataServiceQueryView, dataServiceQueryViewModel, datasetService, datasourceService, distributedAnalysisView, echartsLayer, featureService, fgbLayer, fieldService, geoJSONLayerWithName, geoprocessingService, getMeterPerMapUnit, getResolutionFromScaleDpi, getWrapNum, graphThemeLayer, graphic, graphicLayer, gridCellInfosService, heatMapFeature, heatMapLayer, imageMapLayer, imageStyle, imageTileLayer, initMap, isCORS, labelThemeLayer, layerInfoService, logo, mapService, mapVLayer, measureService, networkAnalyst3DService, networkAnalystService, nonEarthCRS, normalizeScale, openFileView, openFileViewModel, processingService, queryService, rangeThemeLayer, rankSymbolThemeLayer, resolutionToScale, scaleToResolution, searchView, searchViewModel, setCORS, spatialAnalystService, terrainAnalystSetting, themeFeature, themeService, tiandituTileLayer, tiledMapLayer, tiledVectorLayer, toGeoJSON, toSuperMapGeometry, trafficTransferAnalystService, transform, turfLayer, unicodeMarker, uniqueThemeLayer, webMap, webPrintingJobService, wmtsLayer

;// CONCATENATED MODULE: ./src/common/SuperMap.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/
var SuperMap = window.SuperMap = window.SuperMap || {};
SuperMap.Components = window.SuperMap.Components || {};

;// CONCATENATED MODULE: ./src/common/REST.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/

 /**
 * @enum DataFormat
 * @description 服务请求返回结果数据类型
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.DataFormat.GEOJSON;
 *
 * </script>
 * // ES6 Import
 * import { DataFormat } from '{npm}';
 *
 * const result = DataFormat.GEOJSON;
 * ```
 */
var DataFormat = {
    /** GEOJSON */
    GEOJSON: "GEOJSON",
    /** ISERVER */
    ISERVER: "ISERVER",
    /** FGB */
    FGB: "FGB"
};

/**
 * @enum ServerType
 * @description 服务器类型
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.ServerType.ISERVER;
 *
 * </script>
 * // ES6 Import
 * import { ServerType } from '{npm}';
 *
 * const result = ServerType.ISERVER;
 * ```
 */
var ServerType = {
    /** ISERVER */
    ISERVER: "ISERVER",
    /** IPORTAL */
    IPORTAL: "IPORTAL",
    /** ONLINE */
    ONLINE: "ONLINE"
};

/**
 * @enum GeometryType
 * @description 几何对象枚举,定义了一系列几何对象类型。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.GeometryType.LINE;
 *
 * </script>
 * // ES6 Import
 * import { GeometryType } from '{npm}';
 *
 * const result = GeometryType.LINE;
 * ```
 */
var REST_GeometryType = {
    /** 线几何对象类型。 */
    LINE: "LINE",
    /** 路由对象。 */
    LINEM: "LINEM",
    /** 点几何对象类型。 */
    POINT: "POINT",
    /** 面几何对象类型。 */
    REGION: "REGION",
    /** EPS点几何对象。 */
    POINTEPS: "POINTEPS",
    /** EPS线几何对象。 */
    LINEEPS: "LINEEPS",
    /** EPS面几何对象。 */
    REGIONEPS: "REGIONEPS",
    /** 椭圆。 */
    ELLIPSE: "ELLIPSE",
    /** 圆。 */
    CIRCLE: "CIRCLE",
    /** 文本几何对象类型。 */
    TEXT: "TEXT",
    /** 矩形。 */
    RECTANGLE: "RECTANGLE",
    /** 未定义。 */
    UNKNOWN: "UNKNOWN",
    /** 复合几何对象类型。 */
    GEOCOMPOUND:"GEOCOMPOUND"
};

/**
 * @enum QueryOption
 * @description 查询结果类型枚举，描述查询结果返回类型，包括只返回属性、只返回几何实体以及返回属性和几何实体。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.QueryOption.ATTRIBUTE;
 *
 * </script>
 * // ES6 Import
 * import { QueryOption } from '{npm}';
 *
 * const result = QueryOption.ATTRIBUTE;
 * ```
 */
var QueryOption = {
    /** 属性。 */
    ATTRIBUTE: "ATTRIBUTE",
    /** 属性和几何对象。 */
    ATTRIBUTEANDGEOMETRY: "ATTRIBUTEANDGEOMETRY",
    /** 几何对象。 */
    GEOMETRY: "GEOMETRY"
};

/**
 * @enum JoinType
 * @description 关联查询时的关联类型常量。
 * 该类定义了两个表之间的连接类型常量，决定了对两个表之间进行连接查询时，查询结果中得到的记录的情况。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.JoinType.INNERJOIN;
 *
 * </script>
 * // ES6 Import
 * import { JoinType } from '{npm}';
 *
 * const result = JoinType.INNERJOIN;
 * ```
 */
var JoinType = {
    /** 内连接。 */
    INNERJOIN: "INNERJOIN",
    /** 左连接。 */
    LEFTJOIN: "LEFTJOIN"
};

/**
 * @enum SpatialQueryMode
 * @description  空间查询模式枚举。该类定义了空间查询操作模式常量。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.SpatialQueryMode.CONTAIN;
 *
 * </script>
 * // ES6 Import
 * import { SpatialQueryMode } from '{npm}';
 *
 * const result = SpatialQueryMode.CONTAIN;
 * ```
 */
var SpatialQueryMode = {
    /** 包含空间查询模式。 */
    CONTAIN: "CONTAIN",
    /** 交叉空间查询模式。 */
    CROSS: "CROSS",
    /** 分离空间查询模式。 */
    DISJOINT: "DISJOINT",
    /** 重合空间查询模式。 */
    IDENTITY: "IDENTITY",
    /** 相交空间查询模式。 */
    INTERSECT: "INTERSECT",
    /** 无空间查询。 */
    NONE: "NONE",
    /** 叠加空间查询模式。 */
    OVERLAP: "OVERLAP",
    /** 邻接空间查询模式。 */
    TOUCH: "TOUCH",
    /** 被包含空间查询模式。 */
    WITHIN: "WITHIN"
};

/**
 * @enum SpatialRelationType
 * @description  数据集对象间的空间关系枚举。
 * 该类定义了数据集对象间的空间关系类型常量。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.SpatialRelationType.CONTAIN;
 *
 * </script>
 * // ES6 Import
 * import { SpatialRelationType } from '{npm}';
 *
 * const result = {namespace}.SpatialRelationType.CONTAIN;
 * ```
 */
var SpatialRelationType = {
    /** 包含关系。 */
    CONTAIN: "CONTAIN",
    /** 相交关系。 */
    INTERSECT: "INTERSECT",
    /** 被包含关系。 */
    WITHIN: "WITHIN"
};

/**
 * @enum MeasureMode
 * @type {string}
 * @description  量算模式枚举。
 * @category BaseTypes Constant
 * 该类定义了两种测量模式：距离测量和面积测量。
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.MeasureMode.DISTANCE;
 *
 * </script>
 * // ES6 Import
 * import { MeasureMode } from '{npm}';
 *
 * const result = MeasureMode.DISTANCE;
 * ```
 */
var MeasureMode = {
    /** 距离测量。 */
    DISTANCE: "DISTANCE",
    /** 面积测量。 */
    AREA: "AREA"
};

/**
 * @enum Unit
 * @description  距离单位枚举。
 * 该类定义了一系列距离单位类型。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.Unit.METER;
 *
 * </script>
 * // ES6 Import
 * import { Unit } from '{npm}';
 *
 * const result = Unit.METER;
 * ```
 */
var Unit = {
    /**  米。 */
    METER: "METER",
    /**  千米。 */
    KILOMETER: "KILOMETER",
    /**  英里。 */
    MILE: "MILE",
    /**  码。 */
    YARD: "YARD",
    /**  度。 */
    DEGREE: "DEGREE",
    /**  毫米。 */
    MILLIMETER: "MILLIMETER",
    /**  厘米。 */
    CENTIMETER: "CENTIMETER",
    /**  英寸。 */
    INCH: "INCH",
    /**  分米。 */
    DECIMETER: "DECIMETER",
    /**  英尺。 */
    FOOT: "FOOT",
    /**  秒。 */
    SECOND: "SECOND",
    /**  分。 */
    MINUTE: "MINUTE",
    /**  弧度。 */
    RADIAN: "RADIAN"
};

/**
 * @enum BufferRadiusUnit
 * @description  缓冲区距离单位枚举。该类定义了一系列缓冲距离单位类型。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.BufferRadiusUnit.CENTIMETER;
 *
 * </script>
 * // ES6 Import
 * import { BufferRadiusUnit } from '{npm}';
 *
 * const result = BufferRadiusUnit.CENTIMETER;
 * ```
 */
var BufferRadiusUnit = {
    /**  厘米。 */
    CENTIMETER: "CENTIMETER",
    /**  分米。 */
    DECIMETER: "DECIMETER",
    /**  英尺。 */
    FOOT: "FOOT",
    /**  英寸。 */
    INCH: "INCH",
    /**  千米。 */
    KILOMETER: "KILOMETER",
    /**  米。 */
    METER: "METER",
    /**  英里。 */
    MILE: "MILE",
    /**  毫米。 */
    MILLIMETER: "MILLIMETER",
    /**  码。 */
    YARD: "YARD"
}

/**
 * @enum EngineType
 * @description  数据源引擎类型枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.EngineType.IMAGEPLUGINS;
 *
 * </script>
 * // ES6 Import
 * import { EngineType } from '{npm}';
 *
 * const result = EngineType.IMAGEPLUGINS;
 * ```
 */
var EngineType = {
    /** 影像只读引擎类型，文件引擎，针对通用影像格式如 BMP，JPG，TIFF 以及超图自定义影像格式 SIT 等。 */
    IMAGEPLUGINS: "IMAGEPLUGINS",
    /**  OGC 引擎类型，针对于 Web 数据源，Web 引擎，目前支持的类型有 WMS，WFS，WCS。 */
    OGC: "OGC",
    /**  Oracle 引擎类型，针对 Oracle 数据源，数据库引擎。 */
    ORACLEPLUS: "ORACLEPLUS",
    /**  SDB 引擎类型，文件引擎，即 SDB 数据源。 */
    SDBPLUS: "SDBPLUS",
    /**  SQL Server 引擎类型，针对 SQL Server 数据源，数据库引擎。 */
    SQLPLUS: "SQLPLUS",
    /**  UDB 引擎类型，文件引擎。 */
    UDB: "UDB"
};

/**
 * @enum ThemeGraphTextFormat
 * @description  统计专题图文本显示格式枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.ThemeGraphTextFormat.CAPTION;
 *
 * </script>
 * // ES6 Import
 * import { ThemeGraphTextFormat } from '{npm}';
 *
 * const result = ThemeGraphTextFormat.CAPTION;
 * ```
 */
var ThemeGraphTextFormat = {
    /**  标题。以各子项的标题来进行标注。 */
    CAPTION: "CAPTION",
    /**  标题 + 百分数。以各子项的标题和所占的百分比来进行标注。 */
    CAPTION_PERCENT: "CAPTION_PERCENT",
    /**  标题 + 实际数值。以各子项的标题和真实数值来进行标注。 */
    CAPTION_VALUE: "CAPTION_VALUE",
    /**  百分数。以各子项所占的百分比来进行标注。 */
    PERCENT: "PERCENT",
    /**  实际数值。以各子项的真实数值来进行标注。 */
    VALUE: "VALUE"

};

/**
 * @enum ThemeGraphType
 * @description  统计专题图类型枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.ThemeGraphType.AREA;
 *
 * </script>
 * // ES6 Import
 * import { ThemeGraphType } from '{npm}';
 *
 * const result = ThemeGraphType.AREA;
 * ```
 */
var ThemeGraphType = {
    /**  面积图。 */
    AREA: "AREA",
    /**  柱状图。 */
    BAR: "BAR",
    /**  三维柱状图。 */
    BAR3D: "BAR3D",
    /**  折线图。 */
    LINE: "LINE",
    /**  饼图。 */
    PIE: "PIE",
    /**  三维饼图。 */
    PIE3D: "PIE3D",
    /**  点状图。 */
    POINT: "POINT",
    /**  环状图。 */
    RING: "RING",
    /**  玫瑰图。 */
    ROSE: "ROSE",
    /**  三维玫瑰图。 */
    ROSE3D: "ROSE3D",
    /**  堆叠柱状图。 */
    STACK_BAR: "STACK_BAR",
    /**  三维堆叠柱状图。 */
    STACK_BAR3D: "STACK_BAR3D",
    /**  阶梯图。 */
    STEP: "STEP"
};

/**
 * @enum GraphAxesTextDisplayMode
 * @description  统计专题图坐标轴文本显示模式。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.GraphAxesTextDisplayMode.ALL;
 *
 * </script>
 * // ES6 Import
 * import { GraphAxesTextDisplayMode } from '{npm}';
 *
 * const result = GraphAxesTextDisplayMode.ALL;
 * ```
 */
var GraphAxesTextDisplayMode = {
    /**  显示全部文本。 */
    ALL: "ALL",
    /**  不显示。 */
    NONE: "NONE",
    /**  显示Y轴的文本。 */
    YAXES: "YAXES"
};

/**
 * @enum GraduatedMode
 * @description  专题图分级模式枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.GraduatedMode.CONSTANT;
 *
 * </script>
 * // ES6 Import
 * import { GraduatedMode } from '{npm}';
 *
 * const result = GraduatedMode.CONSTANT;
 * ```
 */
var GraduatedMode = {
    /**  常量分级模式。 */
    CONSTANT: "CONSTANT",
    /** 对数分级模式。 */
    LOGARITHM: "LOGARITHM",
    /**  平方根分级模式。 */
    SQUAREROOT: "SQUAREROOT"
};

/**
 * @enum RangeMode
 * @description  范围分段专题图分段方式枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.RangeMode.CUSTOMINTERVAL;
 *
 * </script>
 * // ES6 Import
 * import { RangeMode } from '{npm}';
 *
 * const result = RangeMode.CUSTOMINTERVAL;
 * ```
 */
var RangeMode = {
    /**  自定义分段法。 */
    CUSTOMINTERVAL: "CUSTOMINTERVAL",
    /**  等距离分段法。 */
    EQUALINTERVAL: "EQUALINTERVAL",
    /**  对数分段法。 */
    LOGARITHM: "LOGARITHM",
    /**  等计数分段法。 */
    QUANTILE: "QUANTILE",
    /**  平方根分段法。 */
    SQUAREROOT: "SQUAREROOT",
    /**  标准差分段法。 */
    STDDEVIATION: "STDDEVIATION"
};

/**
 * @enum ThemeType
 * @description  专题图类型枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.ThemeType.DOTDENSITY;
 *
 * </script>
 * // ES6 Import
 * import { ThemeType } from '{npm}';
 *
 * const result = ThemeType.DOTDENSITY;
 * ```
 */
var ThemeType = {
    /** 点密度专题图。 */
    DOTDENSITY: "DOTDENSITY",
    /** 等级符号专题图。 */
    GRADUATEDSYMBOL: "GRADUATEDSYMBOL",
    /** 统计专题图。 */
    GRAPH: "GRAPH",
    /** 标签专题图。 */
    LABEL: "LABEL",
    /** 分段专题图。 */
    RANGE: "RANGE",
    /** 単值专题图。 */
    UNIQUE: "UNIQUE"
};

/**
 * @enum ColorGradientType
 * @description  渐变颜色枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.ColorGradientType.BLACK_WHITE;
 *
 * </script>
 * // ES6 Import
 * import { ColorGradientType } from '{npm}';
 *
 * const result = ColorGradientType.BLACK_WHITE;
 * ```
 */
var ColorGradientType = {
    /** 黑白渐变色。 */
    BLACK_WHITE: "BLACKWHITE",
    /** 蓝黑渐变色。 */
    BLUE_BLACK: "BLUEBLACK",
    /** 蓝红渐变色。 */
    BLUE_RED: "BLUERED",
    /** 蓝白渐变色。 */
    BLUE_WHITE: "BLUEWHITE",
    /** 青黑渐变色。 */
    CYAN_BLACK: "CYANBLACK",
    /** 青蓝渐变色。 */
    CYAN_BLUE: "CYANBLUE",
    /** 青绿渐变色。 */
    CYAN_GREEN: "CYANGREEN",
    /** 青白渐变色。 */
    CYAN_WHITE: "CYANWHITE",
    /** 绿黑渐变色。 */
    GREEN_BLACK: "GREENBLACK",
    /** 绿蓝渐变色。 */
    GREEN_BLUE: "GREENBLUE",
    /** 绿橙紫渐变色。 */
    GREEN_ORANGE_VIOLET: "GREENORANGEVIOLET",
    /** 绿红渐变色。 */
    GREEN_RED: "GREENRED",
    /** 蓝红渐变色。 */
    GREEN_WHITE: "GREENWHITE",
    /** 粉黑渐变色。 */
    PINK_BLACK: "PINKBLACK",
    /** 粉蓝渐变色。 */
    PINK_BLUE: "PINKBLUE",
    /** 粉红渐变色。 */
    PINK_RED: "PINKRED",
    /** 粉白渐变色。 */
    PINK_WHITE: "PINKWHITE",
    /** 彩虹色。 */
    RAIN_BOW: "RAINBOW",
    /** 红黑渐变色。 */
    RED_BLACK: "REDBLACK",
    /** 红白渐变色。 */
    RED_WHITE: "REDWHITE",
    /** 光谱渐变。 */
    SPECTRUM: "SPECTRUM",
    /** 地形渐变，用于三维显示效果较好。 */
    TERRAIN: "TERRAIN",
    /** 黄黑渐变色。 */
    YELLOW_BLACK: "YELLOWBLACK",
    /** 黄蓝渐变色。 */
    YELLOW_BLUE: "YELLOWBLUE",
    /** 黄绿渐变色。 */
    YELLOW_GREEN: "YELLOWGREEN",
    /** 黄红渐变色。 */
    YELLOW_RED: "YELLOWRED",
    /** 黄白渐变色。 */
    YELLOW_WHITE: "YELLOWWHITE"
};

/**
 * @enum TextAlignment
 * @description  文本对齐枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.TextAlignment.TOPLEFT;
 *
 * </script>
 * // ES6 Import
 * import { TextAlignment } from '{npm}';
 *
 * const result = TextAlignment.TOPLEFT;
 * ```
 */
var TextAlignment = {
    /** 左上角对齐。 */
    TOPLEFT: "TOPLEFT",
    /** 顶部居中对齐。 */
    TOPCENTER: "TOPCENTER",
    /** 右上角对齐。 */
    TOPRIGHT: "TOPRIGHT",
    /** 基准线左对齐。 */
    BASELINELEFT: "BASELINELEFT",
    /** 基准线居中对齐。 */
    BASELINECENTER: "BASELINECENTER",
    /** 基准线右对齐。 */
    BASELINERIGHT: "BASELINERIGHT",
    /** 左下角对齐。 */
    BOTTOMLEFT: "BOTTOMLEFT",
    /** 底部居中对齐。 */
    BOTTOMCENTER: "BOTTOMCENTER",
    /** 右下角对齐。 */
    BOTTOMRIGHT: "BOTTOMRIGHT",
    /** 左中对齐。 */
    MIDDLELEFT: "MIDDLELEFT",
    /** 中心对齐。 */
    MIDDLECENTER: "MIDDLECENTER",
    /** 右中对齐。 */
    MIDDLERIGHT: "MIDDLERIGHT"
};

/**
 * @enum FillGradientMode
 * @description  渐变填充风格的渐变类型枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.FillGradientMode.NONE;
 *
 * </script>
 * // ES6 Import
 * import { FillGradientMode } from '{npm}';
 *
 * const result = FillGradientMode.NONE;
 * ```
 */
var FillGradientMode = {
    /** 无渐变。 */
    NONE: "NONE",
    /** 线性渐变填充。 */
    LINEAR: "LINEAR",
    /** 辐射渐变填充。 */
    RADIAL: "RADIAL",
    /** 圆锥渐变填充。 */
    CONICAL: "CONICAL",
    /** 四角渐变填充。 */
    SQUARE: "SQUARE"
};

/**
 * @enum AlongLineDirection
 * @description  标签沿线标注方向枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.AlongLineDirection.NORMAL;
 *
 * </script>
 * // ES6 Import
 * import { AlongLineDirection } from '{npm}';
 *
 * const result = AlongLineDirection.NORMAL;
 * ```
 */
var AlongLineDirection = {
    /** 沿线的法线方向放置标签。 */
    NORMAL: "ALONG_LINE_NORMAL",
    /** 从下到上，从左到右放置。 */
    LB_TO_RT: "LEFT_BOTTOM_TO_RIGHT_TOP",
    /** 从上到下，从左到右放置。 */
    LT_TO_RB: "LEFT_TOP_TO_RIGHT_BOTTOM",
    /** 从下到上，从右到左放置。 */
    RB_TO_LT: "RIGHT_BOTTOM_TO_LEFT_TOP",
    /** 从上到下，从右到左放置。 */
    RT_TO_LB: "RIGHT_TOP_TO_LEFT_BOTTOM"
};

/**
 * @enum LabelBackShape
 * @description  标签专题图中标签背景的形状枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.LabelBackShape.DIAMOND;
 *
 * </script>
 * // ES6 Import
 * import { LabelBackShape } from '{npm}';
 *
 * const result = LabelBackShape.DIAMOND;
 * ```
 */
var LabelBackShape = {
    /** 菱形背景，即标签背景的形状为菱形。 */
    DIAMOND: "DIAMOND",
    /** 椭圆形背景，即标签背景的行状为椭圆形。 */
    ELLIPSE: "ELLIPSE",
    /** 符号背景，即标签背景的形状为设定的符号。 */
    MARKER: "MARKER",
    /** 空背景，即不使用任何形状作为标签的背景。 */
    NONE: "NONE",
    /** 矩形背景，即标签背景的形状为矩形。 */
    RECT: "RECT",
    /** 圆角矩形背景，即标签背景的形状为圆角矩形。 */
    ROUNDRECT: "ROUNDRECT",
    /** 三角形背景，即标签背景的形状为三角形。 */
    TRIANGLE: "TRIANGLE"
};

/**
 * @enum LabelOverLengthMode
 * @description  标签专题图中超长标签的处理模式枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.LabelOverLengthMode.NEWLINE;
 *
 * </script>
 * // ES6 Import
 * import { LabelOverLengthMode } from '{npm}';
 *
 * const result = LabelOverLengthMode.NEWLINE;
 * ```
 */
var LabelOverLengthMode = {
    /** 换行显示。 */
    NEWLINE: "NEWLINE",
    /** 对超长标签不进行处理。 */
    NONE: "NONE",
    /** 省略超出部分。 */
    OMIT: "OMIT"
};

/**
 * @enum DirectionType
 * @description  网络分析中方向枚举。
 * 在行驶引导子项中使用。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.DirectionType.EAST;
 *
 * </script>
 * // ES6 Import
 * import { DirectionType } from '{npm}';
 *
 * const result = DirectionType.EAST;
 * ```
 */
var DirectionType = {
    /** 东。 */
    EAST: "EAST",
    /** 无方向。 */
    NONE: "NONE",
    /** 北。 */
    NORTH: "NORTH",
    /** 南。 */
    SOURTH: "SOURTH",
    /** 西。 */
    WEST: "WEST"
};

/**
 * @enum SideType
 * @description  行驶位置枚举。
 * 表示在行驶在路的左边、右边或者路上的枚举，该类用在行驶导引子项类中。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.SideType.LEFT;
 *
 * </script>
 * // ES6 Import
 * import { SideType } from '{npm}';
 *
 * const result = SideType.LEFT;
 * ```
 */
var SideType = {
    /** 路的左侧。 */
    LEFT: "LEFT",
    /** 在路上（即路的中间）。 */
    MIDDLE: "MIDDLE",
    /** 无效值。 */
    NONE: "NONE",
    /** 路的右侧。 */
    RIGHT: "RIGHT"
};

/**
 * @enum SupplyCenterType
 * @description  资源供给中心类型枚举。
 * 该枚举定义了网络分析中资源中心点的类型，主要用于资源分配和选址分区。
 * 资源供给中心点的类型包括非中心，固定中心和可选中心。固定中心用于资源分配分析；固定中心和可选中心用于选址分析；非中心在两种网络分析时都不予考虑。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.SupplyCenterType.FIXEDCENTER;
 *
 * </script>
 * // ES6 Import
 * import { SupplyCenterType } from '{npm}';
 *
 * const result = SupplyCenterType.FIXEDCENTER;
 * ```
 */
var SupplyCenterType = {
    /** 固定中心点。 */
    FIXEDCENTER: "FIXEDCENTER",
    /** 非中心点。 */
    NULL: "NULL",
    /** 可选中心点。 */
    OPTIONALCENTER: "OPTIONALCENTER"
};

/**
 * @enum TurnType
 * @description  转弯方向枚举。
 * 用在行驶引导子项类中，表示转弯的方向。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.TurnType.AHEAD;
 *
 * </script>
 * // ES6 Import
 * import { TurnType } from '{npm}';
 *
 * const result = TurnType.AHEAD;
 * ```
 */
var TurnType = {
    /** 向前直行。 */
    AHEAD: "AHEAD",
    /** 掉头。 */
    BACK: "BACK",
    /** 终点，不拐弯。 */
    END: "END",
    /** 左转弯。 */
    LEFT: "LEFT",
    /** 无效值。 */
    NONE: "NONE",
    /** 右转弯。 */
    RIGHT: "RIGHT"
};

/**
 * @enum BufferEndType
 * @description  缓冲区分析BufferEnd类型。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.BufferEndType.FLAT;
 *
 * </script>
 * // ES6 Import
 * import { BufferEndType } from '{npm}';
 *
 * const result = BufferEndType.FLAT;
 * ```
 */
var BufferEndType = {
    /** 平头缓冲。 */
    FLAT: "FLAT",
    /** 圆头缓冲。 */
    ROUND: "ROUND"
};
/**
 * @enum OverlayOperationType
 * @description  叠加分析类型枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.OverlayOperationType.CLIP;
 *
 * </script>
 * // ES6 Import
 * import { OverlayOperationType } from '{npm}';
 *
 * const result = OverlayOperationType.CLIP;
 * ```
 */
 var OverlayOperationType = {
    /** 操作数据集（几何对象）裁剪被操作数据集（几何对象）。 */
    CLIP: "CLIP",
    /** 在被操作数据集（几何对象）上擦除掉与操作数据集（几何对象）相重合的部分。 */
    ERASE: "ERASE",
    /**对被操作数据集（几何对象）进行同一操作，即操作执行后，被操作数据集（几何对象）包含来自操作数据集（几何对象）的几何形状。 */
    IDENTITY: "IDENTITY",
    /** 对两个数据集（几何对象）求交，返回两个数据集（几何对象）的交集。 */
    INTERSECT: "INTERSECT",
    /** 对两个面数据集（几何对象）进行合并操作。 */
    UNION: "UNION",
    /** 对两个面数据集（几何对象）进行更新操作。 */
    UPDATE: "UPDATE",
    /** 对两个面数据集（几何对象）进行对称差操作。 */
    XOR: "XOR"
};

/**
 * @enum OutputType
 * @description  分布式分析输出类型枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.OutputType.INDEXEDHDFS;
 *
 * </script>
 * // ES6 Import
 * import { OutputType } from '{npm}';
 *
 * const result = OutputType.INDEXEDHDFS;
 * ```
 */
var OutputType =  {
    /** INDEXEDHDFS */
    INDEXEDHDFS: "INDEXEDHDFS",
    /** UDB */
    UDB: "UDB",
    /** MONGODB */
    MONGODB: "MONGODB",
    /** PG */
    PG: "PG"
};

/**
 * @enum SmoothMethod
 * @description  光滑方法枚举。
 * 用于从Grid 或DEM数据生成等值线或等值面时对等值线或者等值面的边界线进行平滑处理的方法。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.SmoothMethod.BSPLINE;
 *
 * </script>
 * // ES6 Import
 * import { SmoothMethod } from '{npm}';
 *
 * const result = SmoothMethod.BSPLINE;
 * ```
 */
 var SmoothMethod = {
    /** B 样条法。 */
    BSPLINE: "BSPLINE",
    /** 磨角法。 */
    POLISH: "POLISH"
};

/**
 * @enum SurfaceAnalystMethod
 * @description  表面分析方法枚举。
 * 通过对数据进行表面分析，能够挖掘原始数据所包含的信息，使某些细节明显化，易于分析。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.SurfaceAnalystMethod.ISOLINE;
 *
 * </script>
 * // ES6 Import
 * import { SurfaceAnalystMethod } from '{npm}';
 *
 * const result = SurfaceAnalystMethod.ISOLINE;
 * ```
 */
var SurfaceAnalystMethod = {
    /** 等值线提取。 */
    ISOLINE: "ISOLINE",
    /** 等值面提取。 */
    ISOREGION: "ISOREGION"
};

/**
 * @enum DataReturnMode
 * @description  数据返回模式枚举。
 * 该枚举用于指定空间分析返回结果模式，包含返回数据集标识和记录集、只返回数据集标识(数据集名称@数据源名称)及只返回记录集三种模式。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.DataReturnMode.DATASET_AND_RECORDSET;
 *
 * </script>
 * // ES6 Import
 * import { DataReturnMode } from '{npm}';
 *
 * const result = DataReturnMode.DATASET_AND_RECORDSET;
 * ```
 */
var DataReturnMode = {
    /** 返回结果数据集标识(数据集名称@数据源名称)和记录集（RecordSet）。 */
    DATASET_AND_RECORDSET: "DATASET_AND_RECORDSET",
    /** 只返回数据集标识（数据集名称@数据源名称）。 */
    DATASET_ONLY: "DATASET_ONLY",
    /** 只返回记录集（RecordSet）。 */
    RECORDSET_ONLY: "RECORDSET_ONLY"
};

/**
 * @enum EditType
 * @description  要素集更新模式枚举。
 * 该枚举用于指定数据服务中要素集更新模式，包含添加要素集、更新要素集和删除要素集。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.EditType.ADD;
 *
 * </script>
 * // ES6 Import
 * import { EditType } from '{npm}';
 *
 * const result = {namespace}.EditType.ADD;
 * ```
 */
var EditType = {
    /** 增加操作。 */
    ADD: "add",
    /** 修改操作。 */
    UPDATE: "update",
    /** 删除操作。 */
    DELETE: "delete"
};

/**
 * @enum TransferTactic
 * @description  公交换乘策略枚举。
 * 该枚举用于指定公交服务中要素集更新模式,包含添加要素集、更新要素集和删除要素集。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.TransferTactic.LESS_TIME;
 *
 * </script>
 * // ES6 Import
 * import { TransferTactic } from '{npm}';
 *
 * const result = TransferTactic.LESS_TIME;
 * ```
 */
var TransferTactic = {
    /** 时间短。 */
    LESS_TIME: "LESS_TIME",
    /** 少换乘。 */
    LESS_TRANSFER: "LESS_TRANSFER",
    /** 少步行。 */
    LESS_WALK: "LESS_WALK",
    /** 距离最短。 */
    MIN_DISTANCE: "MIN_DISTANCE"
};

/**
 * @enum TransferPreference
 * @description  公交换乘策略枚举。
 * 该枚举用于指定交通换乘服务中设置地铁优先、公交优先、不乘地铁、无偏好等偏好设置。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.TransferPreference.BUS;
 *
 * </script>
 * // ES6 Import
 * import { TransferPreference } from '{npm}';
 *
 * const result = TransferPreference.BUS;
 * ```
 */
var TransferPreference = {
    /** 公交汽车优先。 */
    BUS: "BUS",
    /** 地铁优先。 */
    SUBWAY: "SUBWAY",
    /** 不乘坐地铁。 */
    NO_SUBWAY: "NO_SUBWAY",
    /** 无乘车偏好。 */
    NONE: "NONE"
};

/**
 * @enum GridType
 * @description  地图背景格网类型枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.GridType.CROSS;
 *
 * </script>
 * // ES6 Import
 * import { GridType } from '{npm}';
 *
 * const result = GridType.CROSS;
 * ```
 */
var GridType =  {
    /** 十字叉丝。 */
    CROSS: "CROSS",
    /** 网格线。 */
    GRID: "GRID",
    /** 点。 */
    POINT: "POINT"
};

/**
 * @enum ColorSpaceType
 * @description  色彩空间枚举。
 * 由于成色原理的不同，决定了显示器、投影仪这类靠色光直接合成颜色的颜色设备和打印机、
 * 印刷机这类靠使用颜料的印刷设备在生成颜色方式上的区别。
 * 针对上述不同成色方式，SuperMap 提供两种色彩空间，
 * 分别为 RGB 和 CMYK。RGB 主要用于显示系统中，CMYK 主要用于印刷系统中。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.ColorSpaceType.CMYK;
 *
 * </script>
 * // ES6 Import
 * import { ColorSpaceType } from '{npm}';
 *
 * const result = ColorSpaceType.CMYK;
 * ```
 */
var ColorSpaceType = {
    /** 该类型主要在印刷系统使用。 */
    CMYK: "CMYK",
    /** 该类型主要在显示系统中使用。 */
    RGB: "RGB"
};

/**
 * @enum LayerType
 * @description  图层类型。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.LayerType.UGC;
 *
 * </script>
 * // ES6 Import
 * import { LayerType } from '{npm}';
 *
 * const result = LayerType.UGC;
 * ```
 */
var LayerType = {
    /** SuperMap UGC 类型图层。如矢量图层、栅格(Grid)图层、影像图层。 */
    UGC: "UGC",
    /** WMS 图层。 */
    WMS: "WMS",
    /** WFS 图层。 */
    WFS: "WFS",
    /** 自定义图层。 */
    CUSTOM: "CUSTOM"
};

/**
 * @enum UGCLayerType
 * @description  SuperMap 图层类型。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.UGCLayerType.THEME;
 *
 * </script>
 * // ES6 Import
 * import { UGCLayerType } from '{npm}';
 *
 * const result = UGCLayerType.THEME;
 * ```
 */
var UGCLayerType = {
    /** 专题图层。 */
    THEME: "THEME",
    /** 矢量图层。 */
    VECTOR: "VECTOR",
    /** 栅格图层。 */
    GRID: "GRID",
    /** 影像图层。 */
    IMAGE: "IMAGE"
};

/**
 * @enum StatisticMode
 * @description  字段统计方法类型。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.StatisticMode.AVERAGE;
 *
 * </script>
 * // ES6 Import
 * import { StatisticMode } from '{npm}';
 *
 * const result = StatisticMode.AVERAGE;
 * ```
 */
var StatisticMode = {
    /** 统计所选字段的平均值。 */
    AVERAGE: "AVERAGE",
    /** 统计所选字段的最大值。 */
    MAX: "MAX",
    /** 统计所选字段的最小值。 */
    MIN: "MIN",
    /** 统计所选字段的标准差 */
    STDDEVIATION: "STDDEVIATION",
    /** 统计所选字段的总和。 */
    SUM: "SUM",
    /** 统计所选字段的方差。 */
    VARIANCE: "VARIANCE"
};

/**
 * @enum PixelFormat
 * @description  栅格与影像数据存储的像素格式枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.PixelFormat.BIT16;
 *
 * </script>
 * // ES6 Import
 * import { PixelFormat } from '{npm}';
 *
 * const result = PixelFormat.BIT16;
 * ```
 */
var PixelFormat = {
    /** 每个像元用16个比特(即2个字节)表示。 */
    BIT16: "BIT16",
    /** 每个像元用32个比特(即4个字节)表示。 */
    BIT32: "BIT32",
    /** 每个像元用64个比特(即8个字节)表示，只提供给栅格数据集使用。 */
    BIT64: "BIT64",
    /** 每个像元用4个字节来表示，只提供给栅格数据集使用。 */
    SINGLE: "SINGLE",
    /** 每个像元用8个字节来表示，只提供给栅格数据集使用。 */
    DOUBLE: "DOUBLE",
    /** 每个像元用1个比特表示。 */
    UBIT1: "UBIT1",
    /** 每个像元用4个比特来表示。 */
    UBIT4: "UBIT4",
    /** 每个像元用8个比特(即1个字节)来表示。 */
    UBIT8: "UBIT8",
    /** 每个像元用24个比特(即3个字节)来表示。 */
    UBIT24: "UBIT24",
    /** 每个像元用32个比特(即4个字节)来表示。 */
    UBIT32: "UBIT32"
};

/**
 * @enum SearchMode
 * @description  内插时使用的样本点的查找方式枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.SearchMode.KDTREE_FIXED_COUNT;
 *
 * </script>
 * // ES6 Import
 * import { SearchMode } from '{npm}';
 *
 * const result = SearchMode.KDTREE_FIXED_COUNT;
 * ```
 */
var SearchMode = {
    /** 使用 KDTREE 的固定点数方式查找参与内插分析的点。 */
    KDTREE_FIXED_COUNT: "KDTREE_FIXED_COUNT",
    /** 使用 KDTREE 的定长方式查找参与内插分析的点。 */
    KDTREE_FIXED_RADIUS: "KDTREE_FIXED_RADIUS",
    /** 不进行查找，使用所有的输入点进行内插分析。 */
    NONE: "NONE",
    /** 使用 QUADTREE 方式查找参与内插分析的点，仅对样条（RBF）插值和普通克吕金（Kriging）有用。 */
    QUADTREE: "QUADTREE"
};

/**
 * @enum InterpolationAlgorithmType
 * @description  插值分析的算法的类型。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.InterpolationAlgorithmType.KRIGING;
 *
 * </script>
 * // ES6 Import
 * import { InterpolationAlgorithmType } from '{npm}';
 *
 * const result = InterpolationAlgorithmType.KRIGING;
 * ```
 */
var InterpolationAlgorithmType = {
    /** 普通克吕金插值法。 */
    KRIGING: "KRIGING",
    /** 简单克吕金插值法。 */
    SimpleKriging: "SimpleKriging",
    /** 泛克吕金插值法。 */
    UniversalKriging: "UniversalKriging"
};

/**
 * @enum VariogramMode
 * @description  克吕金（Kriging）插值时的半变函数类型枚举。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.VariogramMode.EXPONENTIAL;
 *
 * </script>
 * // ES6 Import
 * import { VariogramMode } from '{npm}';
 *
 * const result = VariogramMode.EXPONENTIAL;
 * ```
 */
var VariogramMode = {
    /** 指数函数。 */
    EXPONENTIAL: "EXPONENTIAL",
    /** 高斯函数。 */
    GAUSSIAN: "GAUSSIAN",
    /** 球型函数。 */
    SPHERICAL: "SPHERICAL"
};

/**
 * @enum Exponent
 * @description  定义了泛克吕金（UniversalKriging）插值时样点数据中趋势面方程的阶数。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.Exponent.EXP1;
 *
 * </script>
 * // ES6 Import
 * import { Exponent } from '{npm}';
 *
 * const result = Exponent.EXP1;
 * ```
 */
var Exponent = {
    /** 阶数为1。 */
    EXP1: "EXP1",
    /** 阶数为2。 */
    EXP2: "EXP2"
};

/**
 * @enum ClientType
 * @description token申请的客户端标识类型。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.ClientType.IP;
 *
 * </script>
 * // ES6 Import
 * import { ClientType } from '{npm}';
 *
 * const result = ClientType.IP;
 * ```
 */
var ClientType = {
    /** 指定的 IP 地址。 */
    IP: "IP",
    /** 指定的 URL。 */
    REFERER: "Referer",
    /** 发送申请令牌请求的客户端 IP。 */
    REQUESTIP: "RequestIP",
    /** 不做任何验证。 */
    NONE: "NONE",
    /** SERVER。 */
    SERVER: "SERVER",
    /** WEB。 */
    WEB: "WEB"
};

/**
 * @enum ChartType
 * @description 客户端专题图图表类型。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.ChartType.BAR;
 *
 * </script>
 * // ES6 Import
 * import { ChartType } from '{npm}';
 *
 * const result = ChartType.BAR;
 * ```
 */
var ChartType = {
    /** 柱状图。 */
    BAR: "Bar",
    /** 三维柱状图。 */
    BAR3D: "Bar3D",
    /** 圆形图。 */
    CIRCLE: "Circle",
    /** 饼图。 */
    PIE: "Pie",
    /** 散点图。 */
    POINT: "Point",
    /** 折线图。 */
    LINE: "Line",
    /** 环状图。 */
    RING: "Ring"
};

/**
 * @enum ClipAnalystMode
 * @description  裁剪分析模式
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.ClipAnalystMode.CLIP;
 *
 * </script>
 * // ES6 Import
 * import { ClipAnalystMode } from '{npm}';
 *
 * const result = ClipAnalystMode.CLIP;
 * ```
 */
var ClipAnalystMode = {
    /** CLIP。 */
    CLIP: "clip",
    /** INTERSECT。 */
    INTERSECT: "intersect"
};

/**
 * @enum AnalystAreaUnit
 * @description 分布式分析面积单位。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.AnalystAreaUnit.SQUAREMETER;
 *
 * </script>
 * // ES6 Import
 * import { AnalystAreaUnit } from '{npm}';
 *
 * const result = AnalystAreaUnit.SQUAREMETER;
 * ```
 */
var AnalystAreaUnit = {
    /** 平方米。 */
    "SQUAREMETER": "SquareMeter",
    /** 平方千米。 */
    "SQUAREKILOMETER": "SquareKiloMeter",
    /** 公顷。 */
    "HECTARE": "Hectare",
    /** 公亩。 */
    "ARE": "Are",
    /** 英亩。 */
    "ACRE": "Acre",
    /** 平方英尺。 */
    "SQUAREFOOT": "SquareFoot",
    /** 平方码。 */
    "SQUAREYARD": "SquareYard",
    /** 平方英里。 */
    "SQUAREMILE": "SquareMile"
};

/**
 * @enum AnalystSizeUnit
 * @description 分布式分析单位。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.AnalystSizeUnit.METER;
 *
 * </script>
 * // ES6 Import
 * import { AnalystSizeUnit } from '{npm}';
 *
 * const result = AnalystSizeUnit.METER;
 * ```
 */
var AnalystSizeUnit = {
    /** 米。 */
    "METER": "Meter",
    /** 千米。 */
    "KILOMETER": "Kilometer",
    /** 码。 */
    "YARD": "Yard",
    /** 英尺。 */
    "FOOT": "Foot",
    /** 英里。 */
    "MILE": "Mile"
};

/**
 * @enum StatisticAnalystMode
 * @description 分布式分析统计模式。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.StatisticAnalystMode.MAX;
 *
 * </script>
 * // ES6 Import
 * import { StatisticAnalystMode } from '{npm}';
 *
 * const result = StatisticAnalystMode.MAX;
 * ```
 */
var StatisticAnalystMode = {
    /** 统计所选字段的最大值。 */
    "MAX": "max",
    /** 统计所选字段的最小值。 */
    "MIN": "min",
    /** 统计所选字段的平均值。 */
    "AVERAGE": "average",
    /** 统计所选字段的总和。 */
    "SUM": "sum",
    /** 统计所选字段的方差。 */
    "VARIANCE": "variance",
    /** 统计所选字段的标准差。 */
    "STDDEVIATION": "stdDeviation"
};

/**
 * @enum SummaryType
 * @description 分布式分析聚合类型。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.SummaryType.SUMMARYMESH;
 *
 * </script>
 * // ES6 Import
 * import { SummaryType } from '{npm}';
 *
 * const result = SummaryType.SUMMARYMESH;
 * ```
 */
var SummaryType = {
    /** 格网聚合。 */
    "SUMMARYMESH": "SUMMARYMESH",
    /** 多边形聚合。 */
    "SUMMARYREGION": "SUMMARYREGION"
};

/**
 * @enum TopologyValidatorRule
 * @description  拓扑检查模式枚举。该类定义了拓扑检查操作模式常量。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.TopologyValidatorRule.REGIONNOOVERLAP;
 *
 * </script>
 * // ES6 Import
 * import { TopologyValidatorRule } from '{npm}';
 *
 * const result = TopologyValidatorRule.REGIONNOOVERLAP;
 * ```
 */
var TopologyValidatorRule = {
    /** 面内无重叠，用于对面数据进行拓扑检查。 */
    REGIONNOOVERLAP: "REGIONNOOVERLAP",
    /** 面与面无重叠，用于对面数据进行拓扑检查。 */
    REGIONNOOVERLAPWITH: "REGIONNOOVERLAPWITH",
    /** 面被面包含，用于对面数据进行拓扑检查。 */
    REGIONCONTAINEDBYREGION: "REGIONCONTAINEDBYREGION",
    /** 面被面覆盖，用于对面数据进行拓扑检查。 */
    REGIONCOVEREDBYREGION: "REGIONCOVEREDBYREGION",
    /** 线与线无重叠，用于对线数据进行拓扑检查。 */
    LINENOOVERLAP: "LINENOOVERLAP",
    /** 线内无重叠，用于对线数据进行拓扑检查。 */
    LINENOOVERLAPWITH: "LINENOOVERLAPWITH",
    /** 点不相同，用于对点数据进行拓扑检查。 */
    POINTNOIDENTICAL: "POINTNOIDENTICAL"
};

/**
 * @enum BucketAggType
 * @description  格网聚合查询枚举类，该类定义了Elasticsearch数据服务中聚合查询模式常量
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.BucketAggType.GEOHASH_GRID;
 *
 * </script>
 * // ES6 Import
 * import { BucketAggType } from '{npm}';
 *
 * const result = BucketAggType.GEOHASH_GRID;
 * ```
 */
var BucketAggType = {
    /** 格网聚合类型。 */
    GEOHASH_GRID: "geohash_grid"
};

/**
 * @enum MetricsAggType
 * @description  指标聚合类型枚举类，该类定义了Elasticsearch数据服务中聚合查询模式常量。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.MetricsAggType.AVG;
 *
 * </script>
 * // ES6 Import
 * import { MetricsAggType } from '{npm}';
 *
 * const result = MetricsAggType.AVG;
 * ```
 */
var MetricsAggType = {
  /** 平均值聚合类型。 */
  AVG:'avg',
  /** 最大值聚合类型。 */
  MAX:'max',
  /** 最小值聚合类型。 */
  MIN:'min',
  /** 求和聚合类型。 */
  SUM:'sum'
};

/**
 * @enum GetFeatureMode
 * @description feature 查询方式。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.GetFeatureMode.BOUNDS;
 *
 * </script>
 * // ES6 Import
 * import { GetFeatureMode } from '{npm}';
 *
 * const result = GetFeatureMode.BOUNDS;
 * ```
 */
var GetFeatureMode = {
    /** 通过范围查询来获取要素。 */
    BOUNDS: "BOUNDS",
    /** 通过几何对象的缓冲区来获取要素。 */
    BUFFER: "BUFFER",
    /** 通过 ID 来获取要素。 */
    ID: "ID",
    /** 通过空间查询模式来获取要素。 */
    SPATIAL: "SPATIAL",
    /** 通过 SQL 查询来获取要素。 */
    SQL: 'SQL'
}

/**
 * @enum RasterFunctionType
 * @description 栅格分析方法。
 * @category BaseTypes Constant
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.GetFeatureMode.NDVI;
 *
 * </script>
 * // ES6 Import
 * import { GetFeatureMode } from '{npm}';
 *
 * const result = GetFeatureMode.NDVI;
 * ```
 */
var RasterFunctionType = {
    /** 归一化植被指数。 */
    NDVI: "NDVI",
    /** 阴影面分析。 */
    HILLSHADE: "HILLSHADE"
}

/**
 * @enum ResourceType
 * @description iportal资源类型。
 * @category BaseTypes Constant
 * @version 10.0.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.GetFeatureMode.MAP;
 *
 * </script>
 * // ES6 Import
 * import { GetFeatureMode } from '{npm}';
 *
 * const result = GetFeatureMode.MAP;
 * ```
 */
var ResourceType = {
    /** 地图。 */
    MAP: "MAP",
    /** 服务。 */
    SERVICE: "SERVICE",
    /** 场景。 */
    SCENE: "SCENE",
    /** 数据。 */
    DATA: "DATA",
    /** 洞察。 */
    INSIGHTS_WORKSPACE: "INSIGHTS_WORKSPACE",
    /** 大屏。 */
    MAP_DASHBOARD: "MAP_DASHBOARD"
}

/**
 * @enum OrderBy
 * @description iportal资源排序字段。
 * @category BaseTypes Constant
 * @version 10.0.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.OrderBy.UPDATETIME;
 *
 * </script>
 * // ES6 Import
 * import { OrderBy } from '{npm}';
 *
 * const result = OrderBy.UPDATETIME;
 * ```
 */
var OrderBy = {
    /** 按更新时间排序。 */
    UPDATETIME: "UPDATETIME",
    /** 按热度(可能是访问量、下载量)排序。 */
    HEATLEVEL: "HEATLEVEL",
    /** 按相关性排序。 */
    RELEVANCE: "RELEVANCE"
}

/**
 * @enum OrderType
 * @description iportal资源升序还是降序过滤。
 * @category BaseTypes Constant
 * @version 10.0.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.OrderType.ASC;
 *
 * </script>
 * // ES6 Import
 * import { OrderType } from '{npm}';
 *
 * const result = OrderType.ASC;
 * ```
 */
var OrderType = {
    /** 升序。 */
    ASC: "ASC",
    /** 降序。 */
    DESC: "DESC"
}

/**
 * @enum SearchType
 * @description iportal资源查询的范围进行过滤。
 * @category BaseTypes Constant
 * @version 10.0.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.SearchType.PUBLIC;
 *
 * </script>
 * // ES6 Import
 * import { SearchType } from '{npm}';
 *
 * const result = SearchType.PUBLIC;
 * ```
 */
var SearchType = {
    /** 公开资源。 */
    PUBLIC: "PUBLIC",
    /** 我的资源。 */
    MY_RES: "MY_RES",
    /** 我的群组资源。 */
    MYGROUP_RES: "MYGROUP_RES",
    /** 我的部门资源。 */
    MYDEPARTMENT_RES: "MYDEPARTMENT_RES",
    /** 分享给我的资源。 */
    SHARETOME_RES: "SHARETOME_RES"
}

/**
 * @enum AggregationTypes
 * @description iportal资源聚合查询的类型。
 * @category BaseTypes Constant
 * @version 10.0.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.AggregationTypes.TAG;
 *
 * </script>
 * // ES6 Import
 * import { AggregationTypes } from '{npm}';
 *
 * const result = AggregationTypes.TAG;
 * ```
 */
var AggregationTypes = {
    /** 标签。 */
    TAG: "TAG",
    /** 资源类型。 */
    TYPE: "TYPE"
}

/**
 * @enum PermissionType
 * @description iportal资源权限类型。
 * @category BaseTypes Constant
 * @version 10.0.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.PermissionType.SEARCH;
 *
 * </script>
 * // ES6 Import
 * import { PermissionType } from '{npm}';
 *
 * const result = PermissionType.SEARCH;
 * ```
 */
var PermissionType = {
    /** 可检索。 */
    SEARCH:"SEARCH",
    /** 可查看。 */
    READ: "READ",
    /** 可编辑。 */
    READWRITE: "READWRITE",
    /** 可删除。 */
    DELETE: "DELETE",
    /** 可下载，包括可读、可检索。 */
    DOWNLOAD:"DOWNLOAD"
}

/**
 * @enum EntityType
 * @description iportal资源实体类型。
 * @category BaseTypes Constant
 * @version 10.0.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.EntityType.DEPARTMENT;
 *
 * </script>
 * // ES6 Import
 * import { EntityType } from '{npm}';
 *
 * const result = EntityType.DEPARTMENT;
 * ```
 */
var EntityType = {
    /** 部门。 */
    DEPARTMENT: "DEPARTMENT",
    /** 用户组。 */
    GROUP: "GROUP",
    /** 群组。 */
    IPORTALGROUP: "IPORTALGROUP",
    /** 角色。 */
    ROLE: "ROLE",
    /** 用户。 */
    USER: "USER"
}

/**
 * @enum DataItemType
 * @description iportal数据类型。
 * @category BaseTypes Constant
 * @version 10.0.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.DataItemType.GEOJSON;
 *
 * </script>
 * // ES6 Import
 * import { DataItemType } from '{npm}';
 *
 * const result = DataItemType.GEOJSON;
 * ```
 */
var DataItemType = {
    /** geojson 数据。 */
    GEOJSON: "GEOJSON",
    /** UGCV5_MVT。  */
    UGCV5_MVT: "UGCV5_MVT",
    /** json数据。  */
    JSON: "JSON",
    /** 音频文件。 */
    AUDIO: "AUDIO",
    /** Color 颜色。 */
    COLOR: "COLOR",
    /** ColorScheme 颜色方案。 */
    COLORSCHEME: "COLORSCHEME",
    /** CSV 数据。 */
    CSV: "CSV",
    /** EXCEL 数据。 */
    EXCEL: "EXCEL",
    /** FillSymbol 填充符号库。 */
    FILLSYMBOL: "FILLSYMBOL",
    /** 图片类型。 */
    IMAGE: "IMAGE",
    /** LayerTemplate 图层模板。 */
    LAYERTEMPLATE: "LAYERTEMPLATE",
    /** LayoutTemplate 布局模板。 */
    LAYOUTTEMPLATE: "LAYOUTTEMPLATE",
    /** LineSymbol 线符号库。 */
    LINESYMBOL: "LINESYMBOL",
    /** MapTemplate 地图模板。 */
    MAPTEMPLATE: "MAPTEMPLATE",
    /** MarkerSymbol 点符号库。 */
    MARKERSYMBOL: "MARKERSYMBOL",
    /** MBTILES。 */
    MBTILES: "MBTILES",
    /** 照片。 */
    PHOTOS: "PHOTOS",
    /** SHP 空间数据。 */
    SHP: "SHP",
    /** SMTILES。 */
    SMTILES: "SMTILES",
    /** SVTILES。 */
    SVTILES: "SVTILES",
    /** ThemeTemplate 专题图模板。 */
    THEMETEMPLATE: "THEMETEMPLATE",
    /** TPK。 */
    TPK: "TPK",
    /** UDB 数据源。 */
    UDB: "UDB",
    /** UGCV5。 */
    UGCV5: "UGCV5",
    /** 其他类型（普通文件）。 */
    UNKNOWN: "UNKNOWN",
    /** 视频文件。 */
    VIDEO: "VIDEO",
    /** WorkEnviroment 工作环境。 */
    WORKENVIRONMENT: "WORKENVIRONMENT",
    /** 工作空间。 */
    WORKSPACE: "WORKSPACE"
}

/**
 * @enum WebExportFormatType
 * @description Web 打印输出的格式。
 * @category BaseTypes Constant
 * @version 10.0.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.WebExportFormatType.PNG;
 *
 * </script>
 * // ES6 Import
 * import { WebExportFormatType } from '{npm}';
 *
 * const result = WebExportFormatType.PNG;
 * ```
 */
var WebExportFormatType = {
    /** PNG */
    PNG: "PNG",
    /** PDF */
    PDF: "PDF"
}

/**
 * @enum WebScaleOrientationType
 * @description Web 比例尺的方位样式。
 * @category BaseTypes Constant
 * @version 10.0.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.WebScaleOrientationType.HORIZONTALLABELSBELOW;
 *
 * </script>
 * // ES6 Import
 * import { WebScaleOrientationType } from '{npm}';
 *
 * const result = WebScaleOrientationType.HORIZONTALLABELSBELOW;
 * ```
 */
var WebScaleOrientationType = {
    /** horizontal labels below. */
    HORIZONTALLABELSBELOW: "HORIZONTALLABELSBELOW",
    /** horizontal labels above. */
    HORIZONTALLABELSABOVE: "HORIZONTALLABELSABOVE",
    /** vertical labels left. */
    VERTICALLABELSLEFT: "VERTICALLABELSLEFT",
    /** vertical labels right. */
    VERTICALLABELSRIGHT: "VERTICALLABELSRIGHT"
}

/**
 * @enum WebScaleType
 * @description Web 比例尺的样式。
 * @category BaseTypes Constant
 * @version 10.0.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.WebScaleType.LINE;
 *
 * </script>
 * // ES6 Import
 * import { WebScaleType } from '{npm}';
 *
 * const result = WebScaleType.LINE;
 * ```
 */
var WebScaleType = {
    /** line. */
    LINE: "LINE",
    /** bar. */
    BAR: "BAR",
    /** bar sub. */
    BAR_SUB: "BAR_SUB"
}

/**
 * @enum WebScaleUnit
 * @description Web 比例尺的单位制。
 * @category BaseTypes Constant
 * @version 10.0.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.WebScaleUnit.METER;
 *
 * </script>
 * // ES6 Import
 * import { WebScaleUnit } from '{npm}';
 *
 * const result = WebScaleUnit.METER;
 * ```
 */
var WebScaleUnit = {
    /** 米。 */
    METER: "METER",
    /** 英尺。 */
    FOOT: "FOOT",
    /** 度。 */
    DEGREES: "DEGREES"
}

/**
 * @enum BoundsType
 * @description 范围类型。
 * @category BaseTypes Constant
 * @version 11.1.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.BoundsType.UNION;
 *
 * </script>
 * // ES6 Import
 * import { BoundsType } from '{npm}';
 *
 * const result = BoundsType.UNION;
 * ```
 */
var BoundsType = {
  /** 自定义范围。 */
  CUSTOM: "CUSTOM",
  /** 输入栅格数据集范围的交集。 */
  INTERSECTION: "INTERSECTION",
  /** 输入栅格数据集范围的并集。 */
  UNION: "UNION"
}

/**
 * @enum CellSizeType
 * @description 单元格类型。
 * @category BaseTypes Constant
 * @version 11.1.1
 * @type {string}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.CellSizeType.MAX;
 *
 * </script>
 * // ES6 Import
 * import { CellSizeType } from '{npm}';
 *
 * const result = CellSizeType.MAX;
 * ```
 */
var CellSizeType = {
  /** 用户自己输入的单元格值大小作为单元格大小类型。 */
  CUSTOM: "CUSTOM",
  /** 输入栅格数据集中单元格最大值作为单元格大小类型。*/
  MAX : "MAX",
  /** 输入栅格数据集中单元格最小值作为单元格大小类型。 */
  MIN : "MIN"
}


;// CONCATENATED MODULE: ./src/common/commontypes/Size.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/

/**
 * @class  Size
 * @deprecatedclass SuperMap.Size
 * @category BaseTypes Style
 * @classdesc 此类描绘一对高宽值的实例。
 * @param {number} [w=0.0] - 宽度。
 * @param {number} [h=0.0] - 高度。
 *
 * @example
 * var size = new Size(31,46);
 * @usage
 */
class Size {

    constructor(w, h) {
        /**
         * @member {number} [Size.prototype.w=0.0]
         * @description 宽度。
         */
        this.w = w ? parseFloat(w) : 0.0;

        /**
         * @member {number} [Size.prototype.h=0.0]
         * @description 高度。
         */
        this.h = w ? parseFloat(h) : 0.0;
        this.CLASS_NAME = "SuperMap.Size";
    }


    /**
     * @function Size.prototype.toString
     * @description 返回字符串形式。
     * @example
     * var size = new Size(10,5);
     * var str = size.toString();
     * @returns {string} 例如："w=10,h=5"。
     */
    toString() {
        return ("w=" + this.w + ",h=" + this.h);
    }


    /**
     * @function Size.prototype.clone
     * @description 克隆当前size对象。
     * @example
     * var size = new Size(31,46);
     * var size2 = size.clone();
     * @returns {Size}  新的与当前 size 对象有相同宽、高的 Size 对象。
     */
    clone() {
        return new Size(this.w, this.h);
    }


    /**
     *
     * @function Size.prototype.equals
     * @description 比较两个 size 对象是否相等。
     * @example
     * var size = new Size(31,46);
     * var size2 = new Size(31,46);
     * var isEquals = size.equals(size2);
     *
     * @param {Size} sz - 用于比较相等的 Size 对象。
     * @returns {boolean} 传入的 size 和当前 size 高宽相等，注意：如果传入的 size 为空则返回 false。
     *
     */
    equals(sz) {
        var equals = false;
        if (sz != null) {
            equals = ((this.w === sz.w && this.h === sz.h) ||
                (isNaN(this.w) && isNaN(this.h) && isNaN(sz.w) && isNaN(sz.h)));
        }
        return equals;
    }

    /**
     *
     * @function Size.prototype.destroy
     * @description 销毁此对象。销毁后此对象的所有属性为 null，而不是初始值。
     * @example
     * var size = new Size(31,46);
     * size.destroy();
     */
    destroy() {
        this.w = null;
        this.h = null;
    }
}

;// CONCATENATED MODULE: ./src/common/commontypes/Pixel.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/

/**
 * @class Pixel
 * @deprecatedclass SuperMap.Pixel
 * @category BaseTypes Geometry
 * @classdesc 用 x,y 坐标描绘屏幕坐标（像素点）。
 * @param {number} [x=0.0] - x 坐标。
 * @param {number} [y=0.0] - y 坐标。
 * @param {Pixel.Mode} [mode=Pixel.Mode.LeftTop] - 坐标模式。
 *
 * @example
 * //单独创建一个对象
 * var pixcel = new Pixel(100,50);
 *
 * //依据 size 创建
 *  var size = new Size(21,25);
 *  var offset = new Pixel(-(size.w/2), -size.h);
 * @usage
 */
class Pixel {
  constructor(x, y, mode) {
    /**
     * @member {number} [Pixel.prototype.x=0.0]
     * @description x 坐标。
     */
    this.x = x ? parseFloat(x) : 0.0;

    /**
     * @member {number} [Pixel.prototype.y=0.0]
     * @description y 坐标。
     */
    this.y = y ? parseFloat(y) : 0.0;

    /**
     * @member {Pixel.Mode} [Pixel.prototype.mode=Pixel.Mode.LeftTop]
     * @description 坐标模式，有左上、右上、右下、左下这几种模式，分别表示相对于左上角、右上角、右下角、左下角的坐标。
     */
    this.mode = mode;
    this.CLASS_NAME = 'SuperMap.Pixel';
  }

  /**
   * @function Pixel.prototype.toString
   * @description 返回此对象的字符串形式。
   * @example
   *
   * var pixcel = new Pixel(100,50);
   * var str = pixcel.toString();
   *
   * @returns {string} 例如: "x=200.4,y=242.2"
   */
  toString() {
    return 'x=' + this.x + ',y=' + this.y;
  }

  /**
   * @function Pixel.prototype.clone
   * @description 克隆当前的 pixel 对象。
   * @example
   * var pixcel = new Pixel(100,50);
   * var pixcel2 = pixcel.clone();
   * @returns {Pixel} 新的与当前 pixel 对象有相同 x、y 坐标的 pixel 对象。
   */
  clone() {
    return new Pixel(this.x, this.y, this.mode);
  }

  /**
   * @function Pixel.prototype.equals
   * @description 比较两 pixel 是否相等。
   * @example
   * var pixcel = new Pixel(100,50);
   * var pixcel2 = new Pixel(100,50);
   * var isEquals = pixcel.equals(pixcel2);
   *
   * @param {Pixel} px - 用于比较相等的 pixel 对象。
   * @returns {boolean} 如果传入的像素点和当前像素点相同返回 true，如果不同或传入参数为 NULL 则返回 false。
   */
  equals(px) {
    var equals = false;
    if (px != null) {
      equals = (this.x == px.x && this.y == px.y) || (isNaN(this.x) && isNaN(this.y) && isNaN(px.x) && isNaN(px.y));
    }
    return equals;
  }

  /**
   * @function Pixel.prototype.distanceTo
   * @description 返回两个 pixel 的距离。
   * @example
   * var pixcel = new Pixel(100,50);
   * var pixcel2 = new Pixel(110,30);
   * var distance = pixcel.distanceTo(pixcel2);
   *
   * @param {Pixel} px - 需要计算的 pixel。
   * @returns {number} 作为参数传入的像素与当前像素点的距离。
   */
  distanceTo(px) {
    return Math.sqrt(Math.pow(this.x - px.x, 2) + Math.pow(this.y - px.y, 2));
  }

  /**
   * @function Pixel.prototype.add
   * @description 在原来像素坐标基础上，x 值加上传入的 x 参数，y 值加上传入的 y 参数。
   * @example
   * var pixcel = new Pixel(100,50);
   * //pixcel2是新的对象
   * var pixcel2 = pixcel.add(20,30);
   *
   * @param {number} x - 传入的 x 值。
   * @param {number} y - 传入的 y 值。
   * @returns {Pixel} 新的 pixel 对象，该 pixel 是由当前的 pixel 与传入的 x，y 相加得到。
   */
  add(x, y) {
    if (x == null || y == null) {
      throw new TypeError('Pixel.add cannot receive null values');
    }
    return new Pixel(this.x + x, this.y + y);
  }

  /**
   * @function Pixel.prototype.offset
   * @description 通过传入的 {@link Pixel} 参数对原屏幕坐标进行偏移。
   * @example
   * var pixcel = new Pixel(100,50);
   * var pixcel2 = new Pixel(130,20);
   * //pixcel3 是新的对象
   * var pixcel3 = pixcel.offset(pixcel2);
   *
   * @param {Pixel} px - 传入的 {@link Pixel} 对象。
   * @returns {Pixel} 新的 pixel，该 pixel 是由当前的 pixel 对象的 x，y 值与传入的 Pixel 对象的 x，y 值相加得到。
   */
  offset(px) {
    var newPx = this.clone();
    if (px) {
      newPx = this.add(px.x, px.y);
    }
    return newPx;
  }

  /**
   *
   * @function Pixel.prototype.destroy
   * @description 销毁此对象。销毁后此对象的所有属性为 null，而不是初始值。
   * @example
   * var pixcel = new Pixel(100,50);
   * pixcel.destroy();
   */
  destroy() {
    this.x = null;
    this.y = null;
    this.mode = null;
  }
}
/**
 * @enum Mode
 * @memberOf Pixel
 * @readonly
 * @description 模式。
 * @type {string}
 */
Pixel.Mode = {
  /** 左上模式。*/
  LeftTop: 'lefttop',
  /** 右上模式。 */
  RightTop: 'righttop',
  /** 右下模式。 */
  RightBottom: 'rightbottom',
  /** 左下模式。 */
  LeftBottom: 'leftbottom'
};

;// CONCATENATED MODULE: ./src/common/commontypes/BaseTypes.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/
 

/**
 * @function inherit
 * @description 除了 C 和 P 两个必要参数外，可以传递任意数量的对象，这些对象都将继承C。
 * @param {Object} C - 继承的类。
 * @param {Object} P - 被继承的父类。
 * @private
 */
var inheritExt = function (C, P) {
    var F = function () {
    };
    F.prototype = P.prototype;
    C.prototype = new F;
    var i, l, o;
    for (i = 2, l = arguments.length; i < l; i++) {
        o = arguments[i];
        if (typeof o === "function") {
            o = o.prototype;
        }
        Util_Util.extend(C.prototype, o);
    }
};


/**
 * @function mixinExt
 * @description 实现多重继承。
 * @param {Class|Object} ...mixins - 继承的类。
 * @private
 */
var mixinExt = function (...mixins) {

    class Mix {
        constructor(options) {
            for (var index = 0; index < mixins.length; index++) {
                copyProperties(this, new mixins[index](options));
            }
        }
    }

    for (var index = 0; index < mixins.length; index++) {
        var mixin = mixins[index];
        copyProperties(Mix, mixin);
        copyProperties(Mix.prototype, mixin.prototype);
        copyProperties(Mix.prototype, new mixin());
    }
    return Mix;

    function copyProperties(target, source) {
        var ownKeys = Object.getOwnPropertyNames(source);
        if (Object.getOwnPropertySymbols) {
            ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source));
        }
        for (var index = 0; index < ownKeys.length; index++) {
            var key = ownKeys[index];
            if (key !== "constructor"
                && key !== "prototype"
                && key !== "name" && key !== "length") {
                let desc = Object.getOwnPropertyDescriptor(source, key);
                if (window["ActiveXObject"]) {
                    Object.defineProperty(target, key, desc || {});
                } else {
                    Object.defineProperty(target, key, desc);
                }
            }
        }
    }
};

/**
 * @name String
 * @namespace
 * @category BaseTypes Util
 * @description 字符串操作的一系列常用扩展函数。
 * @private
 */
var StringExt = {

    /**
     * @function StringExt.startsWith
     * @description 判断目标字符串是否以指定的子字符串开头。
     * @param {string} str - 目标字符串。
     * @param {string} sub - 查找的子字符串。
     * @returns {boolean} 目标字符串以指定的子字符串开头，则返回 true；否则返回 false。
     */
    startsWith: function (str, sub) {
        return (str.indexOf(sub) == 0);
    },

    /**
     * @function StringExt.contains
     * @description 判断目标字符串是否包含指定的子字符串。
     * @param {string} str - 目标字符串。
     * @param {string} sub - 查找的子字符串。
     * @returns {boolean} 目标字符串中包含指定的子字符串，则返回 true；否则返回 false。
     */
    contains: function (str, sub) {
        return (str.indexOf(sub) != -1);
    },

    /**
     * @function StringExt.trim
     * @description 删除一个字符串的开头和结尾处的所有空白字符。
     * @param {string} str - （可能）存在空白字符填塞的字符串。
     * @returns {string} 删除开头和结尾处空白字符后的字符串。
     */
    trim: function (str) {
        return str.replace(/^\s\s*/, '').replace(/\s\s*$/, '');
    },

    /**
     * @function StringExt.camelize
     * @description 骆驼式("-")连字符的字符串处理。
     * 例如："chicken-head" becomes "chickenHead",
     *       "-chicken-head" becomes "ChickenHead"。
     * @param {string} str - 要处理的字符串，原始内容不应被修改。
     * @returns {string}
     */
    camelize: function (str) {
        var oStringList = str.split('-');
        var camelizedString = oStringList[0];
        for (var i = 1, len = oStringList.length; i < len; i++) {
            var s = oStringList[i];
            camelizedString += s.charAt(0).toUpperCase() + s.substring(1);
        }
        return camelizedString;
    },

    /**
     * @function StringExt.format
     * @description 提供带 ${token} 标记的字符串, 返回 context 对象属性中指定标记的属性值。
     * @example
     * 示例：
     * (code)
     * 1、template = "${value,getValue}";
     *         context = {value: {getValue:function(){return Math.max.apply(null,argument);}}};
     *         args = [2,23,12,36,21];
     *       返回值:36
     * (end)
     * 示例:
     * (code)
     * 2、template = "$${{value,getValue}}";
     *         context = {value: {getValue:function(){return Math.max.apply(null,argument);}}};
     *         args = [2,23,12,36,21];
     *       返回值:"${36}"
     * (end)
     * 示例:
     * (code)
     * 3、template = "${a,b}";
     *         context = {a: {b:"format"}};
     *         args = null;
     *       返回值:"format"
     * (end)
     * 示例:
     * (code)
     * 3、template = "${a,b}";
     *         context = null;
     *         args = null;
     *       返回值:"${a.b}"
     * (end)
     * @param {string} template - 带标记的字符串将要被替换。参数 template 格式为"${token}"，此处的 token 标记会替换为 context["token"] 属性的值。
     * @param {Object} [context=window] - 带有属性的可选对象的属性用于匹配格式化字符串中的标记。如果该参数为空，将使用 window 对象。
     * @param {Array.<number>} [args] - 可选参数传递给在 context 对象上找到的函数。
     * @returns {string} 从 context 对象属性中替换字符串标记位的字符串。
     */
    format: function (template, context, args) {
        if (!context) {
            context = window;
        }

        // Example matching:
        // str   = ${foo.bar}
        // match = foo.bar
        var replacer = function (str, match) {
            var replacement;

            // Loop through all subs. Example: ${a.b.c}
            // 0 -> replacement = context[a];
            // 1 -> replacement = context[a][b];
            // 2 -> replacement = context[a][b][c];
            var subs = match.split(/\.+/);
            for (var i = 0; i < subs.length; i++) {
                if (i == 0) {
                    replacement = context;
                }

                replacement = replacement[subs[i]];
            }

            if (typeof replacement === "function") {
                replacement = args ?
                    replacement.apply(null, args) :
                    replacement();
            }

            // If replacement is undefined, return the string 'undefined'.
            // This is a workaround for a bugs in browsers not properly
            // dealing with non-participating groups in regular expressions:
            // http://blog.stevenlevithan.com/archives/npcg-javascript
            if (typeof replacement == 'undefined') {
                return 'undefined';
            } else {
                return replacement;
            }
        };

        return template.replace(StringExt.tokenRegEx, replacer);
    },

    /**
     * @member {RegExp} [StringExt.tokenRegEx]
     * @description 寻找带 token 的字符串，默认为 tokenRegEx=/\$\{([\w.]+?)\}/g。
     * @example
     * Examples: ${a}, ${a.b.c}, ${a-b}, ${5}
     */
    tokenRegEx: /\$\{([\w.]+?)\}/g,

    /**
     * @member {RegExp} [StringExt.numberRegEx]
     * @description 判断一个字符串是否只包含一个数值，默认为 numberRegEx=/^([+-]?)(?=\d|\.\d)\d*(\.\d*)?([Ee]([+-]?\d+))?$/。
     */
    numberRegEx: /^([+-]?)(?=\d|\.\d)\d*(\.\d*)?([Ee]([+-]?\d+))?$/,

    /**
     * @function StringExt.isNumeric
     * @description 判断一个字符串是否只包含一个数值。
     * @example
     * (code)
     * StringExt.isNumeric("6.02e23") // true
     * StringExt.isNumeric("12 dozen") // false
     * StringExt.isNumeric("4") // true
     * StringExt.isNumeric(" 4 ") // false
     * (end)
     * @returns {boolean} 字符串包含唯一的数值，返回 true；否则返回 false。
     */
    isNumeric: function (value) {
        return StringExt.numberRegEx.test(value);
    },

    /**
     * @function StringExt.numericIf
     * @description 把一个看似数值型的字符串转化为一个数值。
     * @returns {(number|string)} 如果能转换为数值则返回数值，否则返回字符串本身。
     */
    numericIf: function (value) {
        return StringExt.isNumeric(value) ? parseFloat(value) : value;
    }

};

/**
 * @name Number
 * @namespace
 * @category BaseTypes Util
 * @description 数值操作的一系列常用扩展函数。
 * @private
 */
var NumberExt = {

    /**
     * @member {string} [NumberExt.decimalSeparator='.']
     * @description 格式化数字时默认的小数点分隔符。
     * @constant
     */
    decimalSeparator: ".",

    /**
     * @member {string} [NumberExt.thousandsSeparator=',']
     * @description 格式化数字时默认的千位分隔符。
     * @constant
     */
    thousandsSeparator: ",",

    /**
     * @function NumberExt.limitSigDigs
     * @description 限制浮点数的有效数字位数。
     * @param {number} num - 浮点数。
     * @param {number} sig - 有效位数。
     * @returns {number} 将数字四舍五入到指定数量的有效位数。
     */
    limitSigDigs: function (num, sig) {
        var fig = 0;
        if (sig > 0) {
            fig = parseFloat(num.toPrecision(sig));
        }
        return fig;
    },

    /**
     * @function NumberExt.format
     * @description 数字格式化输出。
     * @param {number} num - 数字。
     * @param {number} [dec=0]  - 数字的小数部分四舍五入到指定的位数。设置为 null 值时小数部分不变。
     * @param {string} [tsep=','] - 千位分隔符。
     * @param {string} [dsep='.'] - 小数点分隔符。
     * @returns {string} 数字格式化后的字符串。
     */
    format: function (num, dec, tsep, dsep) {
        dec = (typeof dec != "undefined") ? dec : 0;
        tsep = (typeof tsep != "undefined") ? tsep :
        NumberExt.thousandsSeparator;
        dsep = (typeof dsep != "undefined") ? dsep :
        NumberExt.decimalSeparator;

        if (dec != null) {
            num = parseFloat(num.toFixed(dec));
        }

        var parts = num.toString().split(".");
        if (parts.length === 1 && dec == null) {
            // integer where we do not want to touch the decimals
            dec = 0;
        }

        var integer = parts[0];
        if (tsep) {
            var thousands = /(-?[0-9]+)([0-9]{3})/;
            while (thousands.test(integer)) {
                integer = integer.replace(thousands, "$1" + tsep + "$2");
            }
        }

        var str;
        if (dec == 0) {
            str = integer;
        } else {
            var rem = parts.length > 1 ? parts[1] : "0";
            if (dec != null) {
                rem = rem + new Array(dec - rem.length + 1).join("0");
            }
            str = integer + dsep + rem;
        }
        return str;
    }
};

if (!Number.prototype.limitSigDigs) {
    /**
     * APIMethod: Number.limitSigDigs
     * 限制浮点数的有效数字位数.
     * @param {number} sig -有效位数。
     * @returns {number} 将数字四舍五入到指定数量的有效位数。
     *           如果传入值 为 null、0、或者是负数, 返回值 0。
     */
    Number.prototype.limitSigDigs = function (sig) {
        return NumberExt.limitSigDigs(this, sig);
    };
}

/**
 * @name Function
 * @namespace
 * @category BaseTypes Util
 * @description 函数操作的一系列常用扩展函数。
 * @private
 */
var FunctionExt = {
    /**
     * @function FunctionExt.bind
     * @description 绑定函数到对象。方便创建 this 的作用域。
     * @param {function} func - 输入函数。
     * @param {Object} object - 对象绑定到输入函数（作为输入函数的 this 对象）。
     * @returns {function} object 参数作为 func 函数的 this 对象。
     */
    bind: function (func, object) {
        // create a reference to all arguments past the second one
        var args = Array.prototype.slice.apply(arguments, [2]);
        return function () {
            // Push on any additional arguments from the actual function call.
            // These will come after those sent to the bind call.
            var newArgs = args.concat(
                Array.prototype.slice.apply(arguments, [0])
            );
            return func.apply(object, newArgs);
        };
    },

    /**
     * @function FunctionExt.bindAsEventListener
     * @description 绑定函数到对象，在调用该函数时配置并使用事件对象作为第一个参数。
     * @param {function} func - 用于监听事件的函数。
     * @param {Object} object - this 对象的引用。
     * @returns {function}
     */
    bindAsEventListener: function (func, object) {
        return function (event) {
            return func.call(object, event || window.event);
        };
    },

    /**
     * @function FunctionExt.False
     * @description 该函数仅仅返回 false。该函数主要是避免在 IE8 以下浏览中 DOM 事件句柄的匿名函数问题。
     * @example
     * document.onclick = FunctionExt.False;
     * @returns {boolean}
     */
    False: function () {
        return false;
    },

    /**
     * @function FunctionExt.True
     * @description 该函数仅仅返回 true。该函数主要是避免在 IE8 以下浏览中 DOM 事件句柄的匿名函数问题。
     * @example
     * document.onclick = FunctionExt.True;
     * @returns {boolean}
     */
    True: function () {
        return true;
    },

    /**
     * @function FunctionExt.Void
     * @description 可重用函数，仅仅返回 "undefined"。
     * @returns {undefined}
     */
    Void: function () {
    }

};

/**
 * @name Array
 * @namespace
 * @category BaseTypes Util
 * @description 数组操作的一系列常用扩展函数。
 * @private
 */
var ArrayExt = {

    /**
     * @function ArrayExt.filter
     * @description 过滤数组，提供了 ECMA-262 标准中 Array.prototype.filter 函数的扩展。详见：{@link http://developer.mozilla.org/en/Core_JavaScript_1.5_Reference/Global_Objects/Array/filter}
     * @param {Array} array - 要过滤的数组。
     * @param {function} callback - 数组中的每一个元素调用该函数。</br>
     *     如果函数的返回值为 true，该元素将包含在返回的数组中。该函数有三个参数: 数组中的元素，元素的索引，数组自身。</br>
     *     如果设置了可选参数 caller，在调用 callback 时，使用可选参数 caller 设置为 callback 的参数。</br>
     * @param {Object} [caller] - 在调用 callback 时，使用参数 caller 设置为 callback 的参数。
     * @returns {Array} callback 函数返回 true 时的元素将作为返回数组中的元素。
     */
    filter: function (array, callback, caller) {
        var selected = [];
        if (Array.prototype.filter) {
            selected = array.filter(callback, caller);
        } else {
            var len = array.length;
            if (typeof callback != "function") {
                throw new TypeError();
            }
            for (var i = 0; i < len; i++) {
                if (i in array) {
                    var val = array[i];
                    if (callback.call(caller, val, i, array)) {
                        selected.push(val);
                    }
                }
            }
        }
        return selected;
    }

};

;// CONCATENATED MODULE: ./src/common/commontypes/Geometry.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/
// import {WKT} from '../format/WKT';
// import {Vector} from './Vector';


/**
 * @class Geometry
 * @deprecatedclass SuperMap.Geometry
 * @category BaseTypes Geometry
 * @classdesc 几何对象类，描述地理对象的几何图形。
 * @usage
 */
class Geometry_Geometry {


    constructor() {
        this.CLASS_NAME = "SuperMap.Geometry";
        /**
         * @member {string} Geometry.prototype.id
         * @description  几何对象的唯一标识符。
         *
         */
        this.id = Util_Util.createUniqueID(this.CLASS_NAME + "_");

        /**
         * @member {Geometry} Geometry.prototype.parent
         * @description 父类几何对象。
         */
        this.parent = null;

        /**
         * @member {Bounds} Geometry.prototype.bounds
         * @description 几何对象的范围。
         *
         */
        this.bounds = null;

        /**
         * @member {number} Geometry.prototype.SRID
         * @description 投影坐标参数。通过该参数，服务器判断 Geometry 对象的坐标参考系是否与数据集相同，如果不同，则在数据入库前进行投影变换。
         * @example
         * var geometry= new Geometry();
         * geometry. SRID=4326;
         *
         */
        this.SRID = null;
    }


    /**
     * @function Geometry.prototype.destroy
     * @description 解构 Geometry 类，释放资源。
     */
    destroy() {
        this.id = null;
        this.bounds = null;
        this.SRID = null;
    }


    /**
     * @function Geometry.prototype.clone
     * @description 克隆几何图形。克隆的几何图形不设置非标准的属性。
     * @returns {Geometry} 克隆的几何图形。
     */
    clone() {
        return new Geometry_Geometry();
    }


    /**
     * @function Geometry.prototype.setBounds
     * @description 设置几何对象的 bounds。
     * @param {Bounds} bounds - 范围。
     */
    setBounds(bounds) {
        if (bounds) {
            this.bounds = bounds.clone();
        }
    }


    /**
     * @function Geometry.prototype.clearBounds
     * @description 清除几何对象的 bounds。
     * 如果该对象有父类，也会清除父类几何对象的 bounds。
     */
    clearBounds() {
        this.bounds = null;
        if (this.parent) {
            this.parent.clearBounds();
        }
    }


    /**
     * @function Geometry.prototype.extendBounds
     * @description 扩展现有边界以包含新边界。如果尚未设置几何边界，则设置新边界。
     * @param {Bounds} newBounds - 几何对象的 bounds。
     */
    extendBounds(newBounds) {
        var bounds = this.getBounds();
        if (!bounds) {
            this.setBounds(newBounds);
        } else {
            this.bounds.extend(newBounds);
        }
    }


    /**
     * @function Geometry.prototype.getBounds
     * @description 获得几何图形的边界。如果没有设置边界，可通过计算获得。
     * @returns {Bounds} 几何对象的边界。
     */
    getBounds() {
        if (this.bounds == null) {
            this.calculateBounds();
        }
        return this.bounds;
    }


    /**
     * @function Geometry.prototype.calculateBounds
     * @description 重新计算几何图形的边界（需要在子类中实现此方法）。
     */
    calculateBounds() {
        //
        // This should be overridden by subclasses.
        //
    }

    /**
     * @function Geometry.prototype.getVertices
     * @description 返回几何图形的所有顶点的列表（需要在子类中实现此方法）。
     * @param {boolean} [nodes] - 如果是 true，线则只返回线的末端点，如果 false，仅仅返回顶点，如果没有设置，则返回顶点。
     * @returns {Array} 几何图形的顶点列表。
     */
    getVertices(nodes) { // eslint-disable-line no-unused-vars
    }

    /**
     * @function Geometry.prototype.getArea
     * @description 计算几何对象的面积 ，此方法需要在子类中定义。
     * @returns {number} 计算后的对象面积。
     */
    getArea() {
        //to be overridden by geometries that actually have an area
        //
        return 0.0;
    }


    // /**
    //  * @function Geometry.prototype.toString
    //  * @description 返回geometry对象的字符串表述，需要引入{@link WKTFormat}。此方法只能在子类实现，在父类使用会报错。
    //  * @returns {string} geometry对象的字符串表述(Well-Known Text)
    //  */
    // toString() {
    // var string;
    // if (WKT) {
    //     var wkt = new WKT();
    //     string = wkt.write(new Vector(this));
    // } else {
    //     string = Object.prototype.toString.call(this);
    // }
    // return string;
    // }
}

;// CONCATENATED MODULE: ./src/common/commontypes/Util.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @description 浏览器名称，依赖于 userAgent 属性，BROWSER_NAME 可以是空，或者以下浏览器：
 *     * "opera" -- Opera
 *     * "msie"  -- Internet Explorer
 *     * "safari" -- Safari
 *     * "firefox" -- Firefox
 *     * "mozilla" -- Mozilla
 * @category BaseTypes Constant
 * @constant {Object}
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.Browser.name;
 *
 * </script>
 * // ES6 Import
 * import { Browser } from '{npm}';
 *
 * const result = Browser.name;
 * ```
 */
const Browser = (function () {
  var name = '',
    version = '',
    device = 'pc',
    uaMatch;
  //以下进行测试
  var ua = navigator.userAgent.toLowerCase();
  if (ua.indexOf('msie') > -1 || (ua.indexOf('trident') > -1 && ua.indexOf('rv') > -1)) {
    name = 'msie';
    uaMatch = ua.match(/msie ([\d.]+)/) || ua.match(/rv:([\d.]+)/);
  } else if (ua.indexOf('chrome') > -1) {
    name = 'chrome';
    uaMatch = ua.match(/chrome\/([\d.]+)/);
  } else if (ua.indexOf('firefox') > -1) {
    name = 'firefox';
    uaMatch = ua.match(/firefox\/([\d.]+)/);
  } else if (ua.indexOf('opera') > -1) {
    name = 'opera';
    uaMatch = ua.match(/version\/([\d.]+)/);
  } else if (ua.indexOf('safari') > -1) {
    name = 'safari';
    uaMatch = ua.match(/version\/([\d.]+)/);
  }
  version = uaMatch ? uaMatch[1] : '';

  if (ua.indexOf('ipad') > -1 || ua.indexOf('ipod') > -1 || ua.indexOf('iphone') > -1) {
    device = 'apple';
  } else if (ua.indexOf('android') > -1) {
    uaMatch = ua.match(/version\/([\d.]+)/);
    version = uaMatch ? uaMatch[1] : '';
    device = 'android';
  }
  return { name: name, version: version, device: device };
})();

const isSupportCanvas = (function () {
  var checkRes = true,
    broz = Browser;
  if (document.createElement('canvas').getContext) {
    if (broz.name === 'firefox' && parseFloat(broz.version) < 5) {
      checkRes = false;
    }
    if (broz.name === 'safari' && parseFloat(broz.version) < 4) {
      checkRes = false;
    }
    if (broz.name === 'opera' && parseFloat(broz.version) < 10) {
      checkRes = false;
    }
    if (broz.name === 'msie' && parseFloat(broz.version) < 9) {
      checkRes = false;
    }
  } else {
    checkRes = false;
  }
  return checkRes;
})();

/**
 * @description 如果 userAgent 捕获到浏览器使用的是 Gecko 引擎则返回 true。
 * @constant {number}
 * @private
 */
const IS_GECKO = (function () {
  var ua = navigator.userAgent.toLowerCase();
  return ua.indexOf('webkit') === -1 && ua.indexOf('gecko') !== -1;
})();

/**
 * @constant {number}
 * @default
 * @description 分辨率与比例尺之间转换的常量。
 * @private
 */
const DOTS_PER_INCH = 96;

/**
 * @name CommonUtil
 * @namespace
 * @category BaseTypes Util
 * @description common 工具类。
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.CommonUtil.getElement();
 *
 *   // 弃用的写法
 *   const result = SuperMap.Util.getElement();
 *
 * </script>
 *
 * // ES6 Import
 * import { CommonUtil } from '{npm}';
 *
 * const result = CommonUtil.getElement();
 * ```
 */

const Util_Util = {

  /**
     * @memberOf CommonUtil
     * @description 对象拷贝赋值。
     * @param {Object} dest - 目标对象。
     * @param {Object} arguments - 待拷贝的对象。
     * @returns {Object} 赋值后的目标对象。
     */
  assign(dest) {
    for (var index = 0; index < Object.getOwnPropertyNames(arguments).length; index++) {
        var arg = Object.getOwnPropertyNames(arguments)[index];
        if (arg == "caller" || arg == "callee" || arg == "length" || arg == "arguments") {
            continue;
        }
        var obj = arguments[arg];
        if (obj) {
            for (var j = 0; j < Object.getOwnPropertyNames(obj).length; j++) {
                var key = Object.getOwnPropertyNames(obj)[j];
                if (arg == "caller" || arg == "callee" || arg == "length" || arg == "arguments") {
                    continue;
                }
                dest[key] = obj[key];
            }
        }
    }
    return dest;
  },
  /**
   * @memberOf CommonUtil
   * @description 复制源对象的所有属性到目标对象上，源对象上的没有定义的属性在目标对象上也不会被设置。
   * @example
   * 要复制 Size 对象的所有属性到自定义对象上，使用方法如下:
   *     var size = new Size(100, 100);
   *     var obj = {}；
   *     CommonUtil.extend(obj, size);
   * @param {Object} [destination] - 目标对象。
   * @param {Object} source - 源对象，其属性将被设置到目标对象上。
   * @returns {Object} 目标对象。
   */

  extend: function (destination, source) {
    destination = destination || {};
    if (source) {
      for (var property in source) {
        var value = source[property];
        if (value !== undefined) {
          destination[property] = value;
        }
      }

      /**
       * IE doesn't include the toString property when iterating over an object's
       * properties with the for(property in object) syntax.  Explicitly check if
       * the source has its own toString property.
       */

      /*
       * FF/Windows < 2.0.0.13 reports "Illegal operation on WrappedNative
       * prototype object" when calling hawOwnProperty if the source object
       * is an instance of window.Event.
       */

      var sourceIsEvt = typeof window.Event === 'function' && source instanceof window.Event;

      if (!sourceIsEvt && source.hasOwnProperty && source.hasOwnProperty('toString')) {
        destination.toString = source.toString;
      }
    }
    return destination;
  },
  /**
   * @memberOf CommonUtil
   * @description 对象拷贝。
   * @param {Object} [des] - 目标对象。
   * @param {Object} soc - 源对象。
   */
  copy: function (des, soc) {
    des = des || {};
    var v;
    if (soc) {
      for (var p in des) {
        v = soc[p];
        if (typeof v !== 'undefined') {
          des[p] = v;
        }
      }
    }
  },
  /**
   * @memberOf CommonUtil
   * @description 销毁对象，将其属性置空。
   * @param {Object} [obj] - 目标对象。
   */
  reset: function (obj) {
    obj = obj || {};
    for (var p in obj) {
      if (obj.hasOwnProperty(p)) {
        if (typeof obj[p] === 'object' && obj[p] instanceof Array) {
          for (var i in obj[p]) {
            if (obj[p][i].destroy) {
              obj[p][i].destroy();
            }
          }
          obj[p].length = 0;
        } else if (typeof obj[p] === 'object' && obj[p] instanceof Object) {
          if (obj[p].destroy) {
            obj[p].destroy();
          }
        }
        obj[p] = null;
      }
    }
  },

  /**
   * @memberOf CommonUtil
   * @description 获取 HTML 元素数组。
   * @returns {Array.<HTMLElement>} HTML 元素数组。
   */
  getElement: function () {
    var elements = [];

    for (var i = 0, len = arguments.length; i < len; i++) {
      var element = arguments[i];
      if (typeof element === 'string') {
        element = document.getElementById(element);
      }
      if (arguments.length === 1) {
        return element;
      }
      elements.push(element);
    }
    return elements;
  },

  /**
   * @memberOf CommonUtil
   * @description instance of 的跨浏览器实现。
   * @param {Object} o - 对象。
   * @returns {boolean} 是否是页面元素。
   */
  isElement: function (o) {
    return !!(o && o.nodeType === 1);
  },

  /**
   * @memberOf CommonUtil
   * @description 判断一个对象是否是数组。
   * @param {Object} a - 对象。
   * @returns {boolean} 是否是数组。
   */
  isArray: function (a) {
    return Object.prototype.toString.call(a) === '[object Array]';
  },

  /**
   * @memberOf CommonUtil
   * @description 从数组中删除某一项。
   * @param {Array} array - 数组。
   * @param {Object} item - 数组中要删除的一项。
   * @returns {Array} 执行删除操作后的数组。
   */
  removeItem: function (array, item) {
    for (var i = array.length - 1; i >= 0; i--) {
      if (array[i] === item) {
        array.splice(i, 1);
        //break;more than once??
      }
    }
    return array;
  },

  /**
   * @memberOf CommonUtil
   * @description 获取某对象在数组中的索引值。
   * @param {Array.<Object>} array - 数组。
   * @param {Object} obj - 对象。
   * @returns {number} 某对象在数组中的索引值。
   */
  indexOf: function (array, obj) {
    if (array == null) {
      return -1;
    } else {
      // use the build-in function if available.
      if (typeof array.indexOf === 'function') {
        return array.indexOf(obj);
      } else {
        for (var i = 0, len = array.length; i < len; i++) {
          if (array[i] === obj) {
            return i;
          }
        }
        return -1;
      }
    }
  },

  /**
   * @memberOf CommonUtil
   * @description 修改某 DOM 元素的许多属性。
   * @param {HTMLElement} element - 待修改的 DOM 元素。
   * @param {string} [id] - DOM 元素的 ID。
   * @param {Pixel} [px] - DOM 元素的 style 属性的 left 和 top 属性。
   * @param {Size} [sz] - DOM 元素的 width 和 height 属性。
   * @param {string} [position] - DOM 元素的 position 属性。
   * @param {string} [border] - DOM 元素的 style 属性的 border 属性。
   * @param {string} [overflow] - DOM 元素的 style 属性的 overflow 属性。
   * @param {number} [opacity] - 不透明度值。取值范围为(0.0 - 1.0)。
   */
  modifyDOMElement: function (element, id, px, sz, position, border, overflow, opacity) {
    if (id) {
      element.id = id;
    }
    if (px) {
      element.style.left = px.x + 'px';
      element.style.top = px.y + 'px';
    }
    if (sz) {
      element.style.width = sz.w + 'px';
      element.style.height = sz.h + 'px';
    }
    if (position) {
      element.style.position = position;
    }
    if (border) {
      element.style.border = border;
    }
    if (overflow) {
      element.style.overflow = overflow;
    }
    if (parseFloat(opacity) >= 0.0 && parseFloat(opacity) < 1.0) {
      element.style.filter = 'alpha(opacity=' + opacity * 100 + ')';
      element.style.opacity = opacity;
    } else if (parseFloat(opacity) === 1.0) {
      element.style.filter = '';
      element.style.opacity = '';
    }
  },

  /**
   * @memberOf CommonUtil
   * @description 比较两个对象并合并。
   * @param {Object} [to] - 目标对象。
   * @param {Object} from - 源对象。
   * @returns {Object} 返回合并后的对象。
   */
  applyDefaults: function (to, from) {
    to = to || {};
    /*
     * FF/Windows < 2.0.0.13 reports "Illegal operation on WrappedNative
     * prototype object" when calling hawOwnProperty if the source object is an
     * instance of window.Event.
     */
    var fromIsEvt = typeof window.Event === 'function' && from instanceof window.Event;

    for (var key in from) {
      if (
        to[key] === undefined ||
        (!fromIsEvt && from.hasOwnProperty && from.hasOwnProperty(key) && !to.hasOwnProperty(key))
      ) {
        to[key] = from[key];
      }
    }
    /**
     * IE doesn't include the toString property when iterating over an object's
     * properties with the for(property in object) syntax.  Explicitly check if
     * the source has its own toString property.
     */
    if (
      !fromIsEvt &&
      from &&
      from.hasOwnProperty &&
      from.hasOwnProperty('toString') &&
      !to.hasOwnProperty('toString')
    ) {
      to.toString = from.toString;
    }

    return to;
  },

  /**
   * @memberOf CommonUtil
   * @description 将参数对象转换为 HTTP 的 GET 请求中的参数字符串。例如："key1=value1&key2=value2&key3=value3"。
   * @param {Object} params - 参数对象。
   * @returns {string} HTTP 的 GET 请求中的参数字符串。
   */
  getParameterString: function (params) {
    var paramsArray = [];

    for (var key in params) {
      var value = params[key];
      if (value != null && typeof value !== 'function') {
        var encodedValue;
        if (Array.isArray(value) || value.toString() === '[object Object]') {
          encodedValue = encodeURIComponent(JSON.stringify(value));
        } else {
          /* value is a string; simply encode */
          encodedValue = encodeURIComponent(value);
        }
        paramsArray.push(encodeURIComponent(key) + '=' + encodedValue);
      }
    }

    return paramsArray.join('&');
  },

  /**
   * @memberOf CommonUtil
   * @description 给 URL 追加查询参数。
   * @param {string} url - 待追加参数的 URL 字符串。
   * @param {string} paramStr - 待追加的查询参数。
   * @returns {string} 新的 URL。
   */
  urlAppend: function (url, paramStr) {
    var newUrl = url;
    if (paramStr) {
      if (paramStr.indexOf('?') === 0) {
        paramStr = paramStr.substring(1);
      }
      var parts = (url + ' ').split(/[?&]/);
      newUrl += parts.pop() === ' ' ? paramStr : parts.length ? '&' + paramStr : '?' + paramStr;
    }
    return newUrl;
  },

  /**
   * @memberOf CommonUtil
   * @description 给 URL 追加 path 参数。
   * @param {string} url - 待追加参数的 URL 字符串。
   * @param {string} paramStr - 待追加的path参数。
   * @returns {string} 新的 URL。
   */
  urlPathAppend: function (url, pathStr) {
    let newUrl = url;
    if (!pathStr) {
      return newUrl;
    }
    if (pathStr.indexOf('/') === 0) {
      pathStr = pathStr.substring(1);
    }
    const parts = url.split('?');
    if (parts[0].indexOf('/', parts[0].length - 1) < 0) {
      parts[0] += '/';
    }
    newUrl = `${parts[0]}${pathStr}${parts.length > 1 ? `?${parts[1]}` : ''}`;
    return newUrl;
  },

  /**
   * @memberOf CommonUtil
   * @description 为了避免浮点精度错误而保留的有效位数。
   * @type {number}
   * @default 14
   */
  DEFAULT_PRECISION: 14,

  /**
   * @memberOf CommonUtil
   * @description 将字符串以接近的精度转换为数字。
   * @param {string} number - 字符串。
   * @param {number} [precision=14] - 精度。
   * @returns {number} 转化后的数字。
   */
  toFloat: function (number, precision) {
    if (precision == null) {
      precision = Util_Util.DEFAULT_PRECISION;
    }
    if (typeof number !== 'number') {
      number = parseFloat(number);
    }
    return precision === 0 ? number : parseFloat(number.toPrecision(precision));
  },

  /**
   * @memberOf CommonUtil
   * @description 角度转弧度。
   * @param {number} x - 角度。
   * @returns {number} 转化后的弧度。
   */
  rad: function (x) {
    return (x * Math.PI) / 180;
  },

  /**
   * @memberOf CommonUtil
   * @description 从 URL 字符串中解析出参数对象。
   * @param {string} url - URL。
   * @returns {Object} 解析出的参数对象。
   */
  getParameters: function (url) {
    // if no url specified, take it from the location bar
    url = url === null || url === undefined ? window.location.href : url;

    //parse out parameters portion of url string
    var paramsString = '';
    if (StringExt.contains(url, '?')) {
      var start = url.indexOf('?') + 1;
      var end = StringExt.contains(url, '#') ? url.indexOf('#') : url.length;
      paramsString = url.substring(start, end);
    }

    var parameters = {};
    var pairs = paramsString.split(/[&;]/);
    for (var i = 0, len = pairs.length; i < len; ++i) {
      var keyValue = pairs[i].split('=');
      if (keyValue[0]) {
        var key = keyValue[0];
        try {
          key = decodeURIComponent(key);
        } catch (err) {
          key = unescape(key);
        }

        // being liberal by replacing "+" with " "
        var value = (keyValue[1] || '').replace(/\+/g, ' ');

        try {
          value = decodeURIComponent(value);
        } catch (err) {
          value = unescape(value);
        }

        // follow OGC convention of comma delimited values
        value = value.split(',');

        //if there's only one value, do not return as array
        if (value.length == 1) {
          value = value[0];
        }

        parameters[key] = value;
      }
    }
    return parameters;
  },

  /**
   * @memberOf CommonUtil
   * @description 不断递增计数变量，用于生成唯一 ID。
   * @type {number}
   * @default 0
   */
  lastSeqID: 0,

  /**
   * @memberOf CommonUtil
   * @description 创建唯一 ID 值。
   * @param {string} [prefix] - 前缀。
   * @returns {string} 唯一的 ID 值。
   */
  createUniqueID: function (prefix) {
    if (prefix == null) {
      prefix = 'id_';
    }
    Util_Util.lastSeqID += 1;
    return prefix + Util_Util.lastSeqID;
  },

  /**
   * @memberOf CommonUtil
   * @description 判断并转化比例尺。
   * @param {number} scale - 比例尺。
   * @returns {number} 正常的 scale 值。
   */
  normalizeScale: function (scale) {
    var normScale = scale > 1.0 ? 1.0 / scale : scale;
    return normScale;
  },

  /**
   * @memberOf CommonUtil
   * @description 比例尺转分辨率。
   * @param {number} scale - 比例尺。
   * @param {string} [units='degrees'] - 比例尺单位。
   * @returns {number} 转化后的分辨率。
   */
  getResolutionFromScale: function (scale, units) {
    var resolution;
    if (scale) {
      if (units == null) {
        units = 'degrees';
      }
      var normScale = Util_Util.normalizeScale(scale);
      resolution = 1 / (normScale * INCHES_PER_UNIT[units] * DOTS_PER_INCH);
    }
    return resolution;
  },

  /**
   * @memberOf CommonUtil
   * @description 分辨率转比例尺。
   * @param {number} resolution - 分辨率。
   * @param {string} [units='degrees'] - 分辨率单位。
   * @returns {number} 转化后的比例尺。
   */
  getScaleFromResolution: function (resolution, units) {
    if (units == null) {
      units = 'degrees';
    }

    var scale = resolution * INCHES_PER_UNIT[units] * DOTS_PER_INCH;
    return scale;
  },

  /**
   * @memberOf CommonUtil
   * @description 获取浏览器相关信息。支持的浏览器包括：Opera，Internet Explorer，Safari，Firefox。
   * @returns {Object} 浏览器名称、版本、设备名称。对应的属性分别为 name, version, device。
   */
  getBrowser: function () {
    return Browser;
  },

  /**
   * @memberOf CommonUtil
   * @description 浏览器是否支持 Canvas。
   * @returns {boolean} 当前浏览器是否支持 HTML5 Canvas。
   */
  isSupportCanvas,

  /**
   * @memberOf CommonUtil
   * @description 判断浏览器是否支持 Canvas。
   * @returns {boolean} 当前浏览器是否支持 HTML5 Canvas 。
   */
  supportCanvas: function () {
    return Util_Util.isSupportCanvas;
  },

  /**
   * @memberOf CommonUtil
   * @description 判断一个 URL 请求是否在当前域中。
   * @param {string} url - URL 请求字符串。
   * @returns {boolean} URL 请求是否在当前域中。
   */
  isInTheSameDomain: function (url) {
    if (!url) {
      return true;
    }
    var index = url.indexOf('//');
    var documentUrl = document.location.toString();
    var documentIndex = documentUrl.indexOf('//');
    if (index === -1) {
      return true;
    } else {
      var protocol;
      var substring = (protocol = url.substring(0, index));
      var documentSubString = documentUrl.substring(documentIndex + 2);
      documentIndex = documentSubString.indexOf('/');
      var documentPortIndex = documentSubString.indexOf(':');
      var documentDomainWithPort = documentSubString.substring(0, documentIndex);
      //var documentPort;

      var documentprotocol = document.location.protocol;
      if (documentPortIndex !== -1) {
        // documentPort = +documentSubString.substring(documentPortIndex, documentIndex);
      } else {
        documentDomainWithPort += ':' + (documentprotocol.toLowerCase() === 'http:' ? 80 : 443);
      }
      if (documentprotocol.toLowerCase() !== substring.toLowerCase()) {
        return false;
      }
      substring = url.substring(index + 2);
      var portIndex = substring.indexOf(':');
      index = substring.indexOf('/');
      var domainWithPort = substring.substring(0, index);
      var domain;
      if (portIndex !== -1) {
        domain = substring.substring(0, portIndex);
      } else {
        domain = substring.substring(0, index);
        domainWithPort += ':' + (protocol.toLowerCase() === 'http:' ? 80 : 443);
      }
      var documentDomain = document.domain;
      if (domain === documentDomain && domainWithPort === documentDomainWithPort) {
        return true;
      }
    }
    return false;
  },

  /**
   * @memberOf CommonUtil
   * @description 计算 iServer 服务的 REST 图层的显示分辨率，需要从 iServer 的 REST 图层表述中获取 viewBounds、viewer、scale、coordUnit、datumAxis 五个参数，来进行计算。
   * @param {Bounds} viewBounds - 地图的参照可视范围，即地图初始化时默认的地图显示范围。
   * @param {Size} viewer - 地图初始化时默认的地图图片的尺寸。
   * @param {number} scale - 地图初始化时默认的显示比例尺。
   * @param {string} [coordUnit='degrees'] - 投影坐标系统的地图单位。
   * @param {number} [datumAxis=6378137] - 地理坐标系统椭球体长半轴。用户自定义地图的 Options 时，若未指定该参数的值，则系统默认为 WGS84 参考系的椭球体长半轴 6378137。
   * @returns {number} 图层显示分辨率。
   */
  calculateDpi: function (viewBounds, viewer, scale, coordUnit, datumAxis) {
    //10000 是 0.1毫米与米的转换。DPI的计算公式：Viewer / DPI *  0.0254 * 10000 = ViewBounds * scale ，公式中的10000是为了提高计算结果的精度，以下出现的ratio皆为如此。
    if (!viewBounds || !viewer || !scale) {
      return;
    }
    var ratio = 10000,
      rvbWidth = viewBounds.getWidth(),
      rvbHeight = viewBounds.getHeight(),
      rvWidth = viewer.w,
      rvHeight = viewer.h;
    //用户自定义地图的Options时，若未指定该参数的值，则系统默认为6378137米，即WGS84参考系的椭球体长半轴。
    datumAxis = datumAxis || 6378137;
    coordUnit = coordUnit || 'degrees';
    var dpi;
    if (
      coordUnit.toLowerCase() === 'degree' ||
      coordUnit.toLowerCase() === 'degrees' ||
      coordUnit.toLowerCase() === 'dd'
    ) {
      let num1 = rvbWidth / rvWidth,
        num2 = rvbHeight / rvHeight,
        resolution = num1 > num2 ? num1 : num2;
      dpi = (0.0254 * ratio) / resolution / scale / ((Math.PI * 2 * datumAxis) / 360) / ratio;
    } else {
      let resolution = rvbWidth / rvWidth;
      dpi = (0.0254 * ratio) / resolution / scale / ratio;
    }
    return dpi;
  },

  /**
   * @memberOf CommonUtil
   * @description 将对象转换成 JSON 字符串。
   * @param {Object} obj - 要转换成 JSON 的 Object 对象。
   * @returns {string} 转换后的 JSON 对象。
   */
  toJSON: function (obj) {
    var objInn = obj;
    if (objInn == null) {
      return null;
    }
    switch (objInn.constructor) {
      case String:
        //s = "'" + str.replace(/(["\\])/g, "\\$1") + "'";   string含有单引号出错
        objInn = '"' + objInn.replace(/(["\\])/g, '\\$1') + '"';
        objInn = objInn.replace(/\n/g, '\\n');
        objInn = objInn.replace(/\r/g, '\\r');
        objInn = objInn.replace('<', '&lt;');
        objInn = objInn.replace('>', '&gt;');
        objInn = objInn.replace(/%/g, '%25');
        objInn = objInn.replace(/&/g, '%26');
        return objInn;
      case Array:
        var arr = '';
        for (var i = 0, len = objInn.length; i < len; i++) {
          arr += Util_Util.toJSON(objInn[i]);
          if (i !== objInn.length - 1) {
            arr += ',';
          }
        }
        return "[" + arr + "]";
      case Number:
        return isFinite(objInn) ? String(objInn) : null;
      case Boolean:
        return String(objInn);
      case Date:
        var dateStr =
          '{' +
          '\'__type\':"System.DateTime",' +
          "'Year':" +
          objInn.getFullYear() +
          ',' +
          "'Month':" +
          (objInn.getMonth() + 1) +
          ',' +
          "'Day':" +
          objInn.getDate() +
          ',' +
          "'Hour':" +
          objInn.getHours() +
          ',' +
          "'Minute':" +
          objInn.getMinutes() +
          ',' +
          "'Second':" +
          objInn.getSeconds() +
          ',' +
          "'Millisecond':" +
          objInn.getMilliseconds() +
          ',' +
          "'TimezoneOffset':" +
          objInn.getTimezoneOffset() +
          '}';
        return dateStr;
      default:
        if (objInn['toJSON'] != null && typeof objInn['toJSON'] === 'function') {
          return objInn.toJSON();
        }
        if (typeof objInn === 'object') {
          if (objInn.length) {
            let arr = [];
            for (let i = 0, len = objInn.length; i < len; i++) {
              arr.push(Util_Util.toJSON(objInn[i]));
            }
            return '[' + arr.join(',') + ']';
          }
          let arr = [];
          for (let attr in objInn) {
            //为解决Geometry类型头json时堆栈溢出的问题，attr == "parent"时不进行json转换
            if (typeof objInn[attr] !== 'function' && attr !== 'CLASS_NAME' && attr !== 'parent') {
              arr.push("'" + attr + "':" + Util_Util.toJSON(objInn[attr]));
            }
          }

          if (arr.length > 0) {
            return '{' + arr.join(',') + '}';
          } else {
            return '{}';
          }
        }
        return objInn.toString();
    }
  },

  /**
   * @memberOf CommonUtil
   * @description 根据比例尺和 DPI 计算屏幕分辨率。
   * @category BaseTypes Util
   * @param {number} scale - 比例尺。
   * @param {number} dpi - 图像分辨率，表示每英寸内的像素个数。
   * @param {string} [coordUnit] - 投影坐标系统的地图单位。
   * @param {number} [datumAxis=6378137] - 地理坐标系统椭球体长半轴。用户自定义地图的 Options 时，若未指定该参数的值，则 DPI 默认按照 WGS84 参考系的椭球体长半轴 6378137 来计算。
   * @returns {number} 当前比例尺下的屏幕分辨率。
   */
  getResolutionFromScaleDpi: function (scale, dpi, coordUnit, datumAxis) {
    var resolution = null,
      ratio = 10000;
    //用户自定义地图的Options时，若未指定该参数的值，则系统默认为6378137米，即WGS84参考系的椭球体长半轴。
    datumAxis = datumAxis || 6378137;
    coordUnit = coordUnit || '';
    if (scale > 0 && dpi > 0) {
      scale = Util_Util.normalizeScale(scale);
      if (
        coordUnit.toLowerCase() === 'degree' ||
        coordUnit.toLowerCase() === 'degrees' ||
        coordUnit.toLowerCase() === 'dd'
      ) {
        //scale = Util.normalizeScale(scale);
        resolution = (0.0254 * ratio) / dpi / scale / ((Math.PI * 2 * datumAxis) / 360) / ratio;
        return resolution;
      } else {
        resolution = (0.0254 * ratio) / dpi / scale / ratio;
        return resolution;
      }
    }
    return -1;
  },

  /**
   * @memberOf CommonUtil
   * @description 根据 resolution、dpi、coordUnit 和 datumAxis 计算比例尺。
   * @param {number} resolution - 用于计算比例尺的地图分辨率。
   * @param {number} dpi - 图像分辨率，表示每英寸内的像素个数。
   * @param {string} [coordUnit] - 投影坐标系统的地图单位。
   * @param {number} [datumAxis=6378137] - 地理坐标系统椭球体长半轴。用户自定义地图的 Options 时，若未指定该参数的值，则 DPI 默认按照 WGS84 参考系的椭球体长半轴 6378137 来计算。
   * @returns {number} 当前屏幕分辨率下的比例尺。
   */
  getScaleFromResolutionDpi: function (resolution, dpi, coordUnit, datumAxis) {
    var scale = null,
      ratio = 10000;
    //用户自定义地图的Options时，若未指定该参数的值，则系统默认为6378137米，即WGS84参考系的椭球体长半轴。
    datumAxis = datumAxis || 6378137;
    coordUnit = coordUnit || '';
    if (resolution > 0 && dpi > 0) {
      if (
        coordUnit.toLowerCase() === 'degree' ||
        coordUnit.toLowerCase() === 'degrees' ||
        coordUnit.toLowerCase() === 'dd'
      ) {
        scale = (0.0254 * ratio) / dpi / resolution / ((Math.PI * 2 * datumAxis) / 360) / ratio;
        return scale;
      } else {
        scale = (0.0254 * ratio) / dpi / resolution / ratio;
        return scale;
      }
    }
    return -1;
  },

  /**
   * @memberOf CommonUtil
   * @description 转换查询结果。
   * @param {Object} result - 查询结果。
   * @returns {Object} 转换后的查询结果。
   */
  transformResult: function (result) {
    if (result.responseText && typeof result.responseText === 'string') {
      result = JSON.parse(result.responseText);
    }
    return result;
  },

  /**
   * @memberOf CommonUtil
   * @description 属性拷贝，不拷贝方法类名(CLASS_NAME)等。
   * @param {Object} [destination] - 拷贝目标。
   * @param {Object} source - 源对象。
   *
   */
  copyAttributes: function (destination, source) {
    destination = destination || {};
    if (source) {
      for (var property in source) {
        var value = source[property];
        if (value !== undefined && property !== 'CLASS_NAME' && typeof value !== 'function') {
          destination[property] = value;
        }
      }
    }
    return destination;
  },

  /**
   * @memberOf CommonUtil
   * @description 将源对象上的属性拷贝到目标对象上。（不拷贝 CLASS_NAME 和方法）
   * @param {Object} [destination] - 目标对象。
   * @param {Object} source - 源对象。
   * @param {Array.<string>} clip - 源对象中禁止拷贝到目标对象的属性，目的是防止目标对象上不可修改的属性被篡改。
   *
   */
  copyAttributesWithClip: function (destination, source, clip) {
    destination = destination || {};
    if (source) {
      for (var property in source) {
        //去掉禁止拷贝的属性
        var isInClip = false;
        if (clip && clip.length) {
          for (var i = 0, len = clip.length; i < len; i++) {
            if (property === clip[i]) {
              isInClip = true;
              break;
            }
          }
        }
        if (isInClip === true) {
          continue;
        }

        var value = source[property];
        if (value !== undefined && property !== 'CLASS_NAME' && typeof value !== 'function') {
          destination[property] = value;
        }
      }
    }
    return destination;
  },

  /**
   * @memberOf CommonUtil
   * @description 克隆一个 Object 对象。
   * @param {Object} obj - 需要克隆的对象。
   * @returns {Object} 对象的拷贝对象，注意是新的对象，不是指向。
   */
  cloneObject: function (obj) {
    // Handle the 3 simple types, and null or undefined
    if (null === obj || 'object' !== typeof obj) {
      return obj;
    }

    // Handle Date
    if (obj instanceof Date) {
      let copy = new Date();
      copy.setTime(obj.getTime());
      return copy;
    }

    // Handle Array
    if (obj instanceof Array) {
      let copy = obj.slice(0);
      return copy;
    }

    // Handle Object
    if (obj instanceof Object) {
      let copy = {};
      for (var attr in obj) {
        if (obj.hasOwnProperty(attr)) {
          copy[attr] = Util_Util.cloneObject(obj[attr]);
        }
      }
      return copy;
    }

    throw new Error("Unable to copy obj! Its type isn't supported.");
  },

  /**
   * @memberOf CommonUtil
   * @description 判断两条线段是不是有交点。
   * @param {GeometryPoint} a1 - 第一条线段的起始节点。
   * @param {GeometryPoint} a2 - 第一条线段的结束节点。
   * @param {GeometryPoint} b1 - 第二条线段的起始节点。
   * @param {GeometryPoint} b2 - 第二条线段的结束节点。
   * @returns {Object} 如果相交返回交点，如果不相交返回两条线段的位置关系。
   */
  lineIntersection: function (a1, a2, b1, b2) {
    var intersectValue = null;
    var k1;
    var k2;
    var b = (b2.x - b1.x) * (a1.y - b1.y) - (b2.y - b1.y) * (a1.x - b1.x);
    var a = (a2.x - a1.x) * (a1.y - b1.y) - (a2.y - a1.y) * (a1.x - b1.x);
    var ab = (b2.y - b1.y) * (a2.x - a1.x) - (b2.x - b1.x) * (a2.y - a1.y);
    //ab==0代表两条线断的斜率一样
    if (ab != 0) {
      k1 = b / ab;
      k2 = a / ab;

      if (k1 >= 0 && k2 <= 1 && k1 <= 1 && k2 >= 0) {
        intersectValue = new Geometry_Geometry.Point(a1.x + k1 * (a2.x - a1.x), a1.y + k1 * (a2.y - a1.y));
      } else {
        intersectValue = 'No Intersection';
      }
    } else {
      if (b == 0 && a == 0) {
        var maxy = Math.max(a1.y, a2.y);
        var miny = Math.min(a1.y, a2.y);
        var maxx = Math.max(a1.x, a2.x);
        var minx = Math.min(a1.x, a2.x);
        if (
          (((b1.y >= miny && b1.y <= maxy) || (b2.y >= miny && b2.y <= maxy)) && b1.x >= minx && b1.x <= maxx) ||
          (b2.x >= minx && b2.x <= maxx)
        ) {
          intersectValue = 'Coincident'; //重合
        } else {
          intersectValue = 'Parallel'; //平行
        }
      } else {
        intersectValue = 'Parallel'; //平行
      }
    }
    return intersectValue;
  },

  /**
   * @memberOf CommonUtil
   * @description 获取文本外接矩形宽度与高度。
   * @param {ThemeStyle} style - 文本样式。
   * @param {string} text - 文本内容。
   * @param {Object} element - DOM 元素。
   * @returns {Object} 裁剪后的宽度，高度信息。
   */
  getTextBounds: function (style, text, element) {
    document.body.appendChild(element);
    element.style.width = 'auto';
    element.style.height = 'auto';
    if (style.fontSize) {
      element.style.fontSize = style.fontSize;
    }
    if (style.fontFamily) {
      element.style.fontFamily = style.fontFamily;
    }
    if (style.fontWeight) {
      element.style.fontWeight = style.fontWeight;
    }
    element.style.position = 'relative';
    element.style.visibility = 'hidden';
    //fix 在某些情况下，element内的文本变成竖起排列，导致宽度计算不正确的bug
    element.style.display = 'inline-block';
    element.innerHTML = text;
    var textWidth = element.clientWidth;
    var textHeight = element.clientHeight;
    document.body.removeChild(element);
    return {
      textWidth: textWidth,
      textHeight: textHeight
    };
  },
  /**
   * @memberOf CommonUtil
   * @description 获取转换后的path路径。
   * @param {string} path - 待转换的path，包含`{param}`。
   * @param {Object} pathParams - path中待替换的参数。
   * @returns {string} 转换后的path路径。
   */
  convertPath: function (path, pathParams) {
    if (!pathParams) {
      return path;
    }
    return path.replace(/\{([\w-\.]+)\}/g, (fullMatch, key) => {
      var value;
      if (pathParams.hasOwnProperty(key)) {
        value = paramToString(pathParams[key]);
      } else {
        value = fullMatch;
      }
      return encodeURIComponent(value);
    });
  }
};

/**
 * @enum INCHES_PER_UNIT
 * @description 每单位的英尺数。
 * @type {number}
 * @private
 */
const INCHES_PER_UNIT = {
  inches: 1.0,
  ft: 12.0,
  mi: 63360.0,
  m: 39.3701,
  km: 39370.1,
  dd: 4374754,
  yd: 36
};
INCHES_PER_UNIT['in'] = INCHES_PER_UNIT.inches;
INCHES_PER_UNIT['degrees'] = INCHES_PER_UNIT.dd;
INCHES_PER_UNIT['nmi'] = 1852 * INCHES_PER_UNIT.m;

// Units from CS-Map
const METERS_PER_INCH = 0.0254000508001016002;
Util_Util.extend(INCHES_PER_UNIT, {
  Inch: INCHES_PER_UNIT.inches,
  Meter: 1.0 / METERS_PER_INCH, //EPSG:9001
  Foot: 0.30480060960121920243 / METERS_PER_INCH, //EPSG:9003
  IFoot: 0.3048 / METERS_PER_INCH, //EPSG:9002
  ClarkeFoot: 0.3047972651151 / METERS_PER_INCH, //EPSG:9005
  SearsFoot: 0.30479947153867624624 / METERS_PER_INCH, //EPSG:9041
  GoldCoastFoot: 0.30479971018150881758 / METERS_PER_INCH, //EPSG:9094
  IInch: 0.0254 / METERS_PER_INCH,
  MicroInch: 0.0000254 / METERS_PER_INCH,
  Mil: 0.0000000254 / METERS_PER_INCH,
  Centimeter: 0.01 / METERS_PER_INCH,
  Kilometer: 1000.0 / METERS_PER_INCH, //EPSG:9036
  Yard: 0.91440182880365760731 / METERS_PER_INCH,
  SearsYard: 0.914398414616029 / METERS_PER_INCH, //EPSG:9040
  IndianYard: 0.91439853074444079983 / METERS_PER_INCH, //EPSG:9084
  IndianYd37: 0.91439523 / METERS_PER_INCH, //EPSG:9085
  IndianYd62: 0.9143988 / METERS_PER_INCH, //EPSG:9086
  IndianYd75: 0.9143985 / METERS_PER_INCH, //EPSG:9087
  IndianFoot: 0.30479951 / METERS_PER_INCH, //EPSG:9080
  IndianFt37: 0.30479841 / METERS_PER_INCH, //EPSG:9081
  IndianFt62: 0.3047996 / METERS_PER_INCH, //EPSG:9082
  IndianFt75: 0.3047995 / METERS_PER_INCH, //EPSG:9083
  Mile: 1609.34721869443738887477 / METERS_PER_INCH,
  IYard: 0.9144 / METERS_PER_INCH, //EPSG:9096
  IMile: 1609.344 / METERS_PER_INCH, //EPSG:9093
  NautM: 1852.0 / METERS_PER_INCH, //EPSG:9030
  'Lat-66': 110943.316488932731 / METERS_PER_INCH,
  'Lat-83': 110946.25736872234125 / METERS_PER_INCH,
  Decimeter: 0.1 / METERS_PER_INCH,
  Millimeter: 0.001 / METERS_PER_INCH,
  Dekameter: 10.0 / METERS_PER_INCH,
  Decameter: 10.0 / METERS_PER_INCH,
  Hectometer: 100.0 / METERS_PER_INCH,
  GermanMeter: 1.0000135965 / METERS_PER_INCH, //EPSG:9031
  CaGrid: 0.999738 / METERS_PER_INCH,
  ClarkeChain: 20.1166194976 / METERS_PER_INCH, //EPSG:9038
  GunterChain: 20.11684023368047 / METERS_PER_INCH, //EPSG:9033
  BenoitChain: 20.116782494375872 / METERS_PER_INCH, //EPSG:9062
  SearsChain: 20.11676512155 / METERS_PER_INCH, //EPSG:9042
  ClarkeLink: 0.201166194976 / METERS_PER_INCH, //EPSG:9039
  GunterLink: 0.2011684023368047 / METERS_PER_INCH, //EPSG:9034
  BenoitLink: 0.20116782494375872 / METERS_PER_INCH, //EPSG:9063
  SearsLink: 0.2011676512155 / METERS_PER_INCH, //EPSG:9043
  Rod: 5.02921005842012 / METERS_PER_INCH,
  IntnlChain: 20.1168 / METERS_PER_INCH, //EPSG:9097
  IntnlLink: 0.201168 / METERS_PER_INCH, //EPSG:9098
  Perch: 5.02921005842012 / METERS_PER_INCH,
  Pole: 5.02921005842012 / METERS_PER_INCH,
  Furlong: 201.1684023368046 / METERS_PER_INCH,
  Rood: 3.778266898 / METERS_PER_INCH,
  CapeFoot: 0.3047972615 / METERS_PER_INCH,
  Brealey: 375.0 / METERS_PER_INCH,
  ModAmFt: 0.304812252984505969011938 / METERS_PER_INCH,
  Fathom: 1.8288 / METERS_PER_INCH,
  'NautM-UK': 1853.184 / METERS_PER_INCH,
  '50kilometers': 50000.0 / METERS_PER_INCH,
  '150kilometers': 150000.0 / METERS_PER_INCH
});

//unit abbreviations supported by PROJ.4
Util_Util.extend(INCHES_PER_UNIT, {
  mm: INCHES_PER_UNIT['Meter'] / 1000.0,
  cm: INCHES_PER_UNIT['Meter'] / 100.0,
  dm: INCHES_PER_UNIT['Meter'] * 100.0,
  km: INCHES_PER_UNIT['Meter'] * 1000.0,
  kmi: INCHES_PER_UNIT['nmi'], //International Nautical Mile
  fath: INCHES_PER_UNIT['Fathom'], //International Fathom
  ch: INCHES_PER_UNIT['IntnlChain'], //International Chain
  link: INCHES_PER_UNIT['IntnlLink'], //International Link
  'us-in': INCHES_PER_UNIT['inches'], //U.S. Surveyor's Inch
  'us-ft': INCHES_PER_UNIT['Foot'], //U.S. Surveyor's Foot
  'us-yd': INCHES_PER_UNIT['Yard'], //U.S. Surveyor's Yard
  'us-ch': INCHES_PER_UNIT['GunterChain'], //U.S. Surveyor's Chain
  'us-mi': INCHES_PER_UNIT['Mile'], //U.S. Surveyor's Statute Mile
  'ind-yd': INCHES_PER_UNIT['IndianYd37'], //Indian Yard
  'ind-ft': INCHES_PER_UNIT['IndianFt37'], //Indian Foot
  'ind-ch': 20.11669506 / METERS_PER_INCH //Indian Chain
});

//将服务端的地图单位转成SuperMap的地图单位
INCHES_PER_UNIT['degree'] = INCHES_PER_UNIT.dd;
INCHES_PER_UNIT['meter'] = INCHES_PER_UNIT.m;
INCHES_PER_UNIT['foot'] = INCHES_PER_UNIT.ft;
INCHES_PER_UNIT['inch'] = INCHES_PER_UNIT.inches;
INCHES_PER_UNIT['mile'] = INCHES_PER_UNIT.mi;
INCHES_PER_UNIT['kilometer'] = INCHES_PER_UNIT.km;
INCHES_PER_UNIT['yard'] = INCHES_PER_UNIT.yd;

function paramToString(param) {
  if (param == undefined || param == null) {
    return '';
  }
  if (param instanceof Date) {
    return param.toJSON();
  }
  if (canBeJsonified(param)) {
    return JSON.stringify(param);
  }

  return param.toString();
}

function canBeJsonified(str) {
  if (typeof str !== 'string' && typeof str !== 'object') {
    return false;
  }
  try {
    const type = str.toString();
    return type === '[object Object]' || type === '[object Array]';
  } catch (err) {
    return false;
  }
}



;// CONCATENATED MODULE: ./src/common/commontypes/LonLat.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class LonLat
 * @category BaseTypes Geometry
 * @classdesc  这个类用来表示经度和纬度对。
 * @param {number|Array.<number>} [lon=0.0] - 地图单位上的 X 轴坐标或者横纵坐标组成的数组；如果地图是地理投影，则此值是经度，否则，此值是地图地理位置的 x 坐标。
 * @param {number} [lat=0.0] - 地图单位上的 Y 轴坐标，如果地图是地理投影，则此值是纬度，否则，此值是地图地理位置的 y 坐标。
 * @example
 * var lonLat = new LonLat(30,45);
 * @usage
 */
class LonLat {


    constructor(lon, lat) {
        if (Util_Util.isArray(lon)) {
            lat = lon[1];
            lon = lon[0];
        }
        /**
         * @member {number} [LonLat.prototype.lon=0.0]
         * @description 地图的单位的 X 轴（横轴）坐标。
         */
        this.lon = lon ? Util_Util.toFloat(lon) : 0.0;

        /**
         * @member {number} [LonLat.prototype.lat=0.0]
         * @description 地图的单位的 Y 轴（纵轴）坐标。
         */
        this.lat = lat ? Util_Util.toFloat(lat) : 0.0;
        this.CLASS_NAME = "SuperMap.LonLat";
    }

    /**
     * @function LonLat.prototype.toString
     * @description 返回此对象的字符串形式。
     * @example
     * var lonLat = new LonLat(100,50);
     * var str = lonLat.toString();
     * @returns {string} 例如: "lon=100,lat=50"。
     */
    toString() {
        return ("lon=" + this.lon + ",lat=" + this.lat);
    }

    /**
     * @function LonLat.prototype.toShortString
     * @description 将经度纬度转换成简单字符串。
     * @example
     * var lonLat = new LonLat(100,50);
     * var str = lonLat.toShortString();
     * @returns {string} 处理后的经纬度字符串。例如："100,50"。
     */
    toShortString() {
        return (this.lon + "," + this.lat);
    }

    /**
     * @function LonLat.prototype.clone
     * @description 复制坐标对象，并返回复制后的新对象。
     * @example
     * var lonLat1 = new LonLat(100,50);
     * var lonLat2 = lonLat1.clone();
     * @returns {LonLat}  相同坐标值的新的坐标对象。
     */
    clone() {
        return new LonLat(this.lon, this.lat);
    }

    /**
     * @function LonLat.prototype.add
     * @description 在已有坐标对象的经纬度基础上加上新的坐标经纬度，并返回新的坐标对象。
     * @example
     * var lonLat1 = new LonLat(100,50);
     * //lonLat2 是新的对象
     * var lonLat2 = lonLat1.add(100,50);
     * @param {number} lon - 经度参数。
     * @param {number} lat - 纬度参数。
     * @returns {LonLat} 新的 LonLat 对象，此对象的经纬度是由传入的经纬度与当前的经纬度相加所得。
     */
    add(lon, lat) {
        if ((lon == null) || (lat == null)) {
            throw new TypeError('LonLat.add cannot receive null values');
        }
        return new LonLat(this.lon + Util_Util.toFloat(lon),
            this.lat + Util_Util.toFloat(lat));
    }

    /**
     * @function LonLat.prototype.equals
     * @description 判断两个坐标对象是否相等。
     * @example
     * var lonLat1 = new LonLat(100,50);
     * var lonLat2 = new LonLat(100,50);
     * var isEquals = lonLat1.equals(lonLat2);
     * @param {LonLat} ll - 需要进行比较的坐标对象。
     * @returns {boolean} 如果LonLat对象的经纬度和传入的经纬度一致则返回true,不一
     *      致或传入的ll参数为NULL则返回false。
     */
    equals(ll) {
        var equals = false;
        if (ll != null) {
            equals = ((this.lon === ll.lon && this.lat === ll.lat) ||
                (isNaN(this.lon) && isNaN(this.lat) && isNaN(ll.lon) && isNaN(ll.lat)));
        }
        return equals;
    }

    /**
     * @function LonLat.prototype.wrapDateLine
     * @description 通过传入的范围对象对坐标对象转换到该范围内。
     * 如果经度小于给定范围最小精度，则在原经度基础上加上范围宽度，直到精度在范围内为止，如果经度大于给定范围则在原经度基础上减去范围宽度。
     * 即指将不在经度范围内的坐标转换到范围以内（只会转换 lon，不会转换 lat，主要用于转移到日界线以内）。
     * @example
     * var lonLat1 = new LonLat(420,50);
     * var lonLat2 = lonLat1.wrapDateLine(
     *      new Bounds(-180,-90,180,90)
     *  );
     * @param {Bounds} maxExtent - 最大边界的范围。
     * @returns {LonLat} 将坐标转换到范围对象以内，并返回新的坐标。
     */
    wrapDateLine(maxExtent) {

        var newLonLat = this.clone();

        if (maxExtent) {
            //shift right?
            while (newLonLat.lon < maxExtent.left) {
                newLonLat.lon += maxExtent.getWidth();
            }

            //shift left?
            while (newLonLat.lon > maxExtent.right) {
                newLonLat.lon -= maxExtent.getWidth();
            }
        }

        return newLonLat;
    }

    /**
     *
     * @function LonLat.prototype.destroy
     * @description 销毁此对象。
     * 销毁后此对象的所有属性为 null，而不是初始值。
     * @example
     * var lonLat = new LonLat(100,50);
     * lonLat.destroy();
     */
    destroy() {
        this.lon = null;
        this.lat = null;
    }

    /**
     * @function LonLat.fromString
     * @description 通过字符串生成一个 {@link LonLat} 对象。
     * @example
     * var str = "100,50";
     * var lonLat = LonLat.fromString(str);
     * @param {string} str - 字符串的格式：Lon+","+Lat。如："100,50"。
     * @returns {LonLat} {@link LonLat} 对象。
     */
    static fromString(str) {
        var pair = str.split(",");
        return new LonLat(pair[0], pair[1]);
    }

    /**
     * @function LonLat.fromArray
     * @description 通过数组生成一个 {@link LonLat} 对象。
     * @param {Array.<number>} arr - 数组的格式，长度只能为2,：[Lon,Lat]。如：[5,-42]。
     * @returns {LonLat} {@link LonLat} 对象。
     */
    static fromArray(arr) {
        var gotArr = Util_Util.isArray(arr),
            lon = gotArr && arr[0],
            lat = gotArr && arr[1];
        return new LonLat(lon, lat);
    }


}


;// CONCATENATED MODULE: ./src/common/commontypes/Bounds.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class Bounds
 * @deprecatedclass SuperMap.Bounds
 * @category BaseTypes Geometry
 * @classdesc 表示边界类实例。使用 bounds 之前需要设置 left，bottom，right，top 四个属性，这些属性的初始值为 null。
 * @param {number|Array.<number>} [left] - 如果是number，则表示左边界，注意考虑宽度，理论上小于 right 值。如果是数组，则表示 [left, bottom, right, top] 左下右上组成的数组。
 * @param {number} [bottom] - 下边界。考虑高度，理论上小于 top 值。
 * @param {number} [right] - 右边界。
 * @param {number} [top] - 上边界。
 * @example
 * var bounds = new Bounds();
 * bounds.extend(new LonLat(4,5));
 * bounds.extend(new LonLat(5,6));
 * bounds.toBBOX(); // returns 4,5,5,6
 * @usage
 */
class Bounds {


    constructor(left, bottom, right, top) {
        if (Util_Util.isArray(left)) {
            top = left[3];
            right = left[2];
            bottom = left[1];
            left = left[0];
        }
        /**
         * @member {number} Bounds.prototype.left
         * @description 最小的水平坐标系。
         */
        this.left = left != null ? Util_Util.toFloat(left) : this.left;

        /**
         * @member {number} Bounds.prototype.bottom
         * @description 最小的垂直坐标系。
         */
        this.bottom = bottom != null ? Util_Util.toFloat(bottom) : this.bottom;

        /**
         * @member {number} Bounds.prototype.right
         * @description 最大的水平坐标系。
         */
        this.right = right != null ? Util_Util.toFloat(right) : this.right;

        /**
         * @member {number} Bounds.prototype.top
         * @description 最大的垂直坐标系。
         */
        this.top = top != null ? Util_Util.toFloat(top) : this.top;

        /**
         * @member {LonLat} Bounds.prototype.centerLonLat
         * @description bounds 的地图空间的中心点。用 getCenterLonLat() 获得。
         */
        this.centerLonLat = null;
        this.CLASS_NAME = "SuperMap.Bounds";
    }

    /**
     * @function Bounds.prototype.clone
     * @description 复制当前 bounds 对象。
     * @example
     * var bounds1 = new Bounds(-180,-90,180,90);
     * var bounds2 = bounds1.clone();
     * @returns {Bounds} 克隆后的 bounds。
     */
    clone() {
        return new Bounds(this.left, this.bottom,
            this.right, this.top);
    }

    /**
     * @function Bounds.prototype.equals
     * @description 判断两个 bounds 对象是否相等。
     * @example
     * var bounds1 = new Bounds(-180,-90,180,90);
     * var bounds2 = new Bounds(-180,-90,180,90);
     * var isEquals = bounds1.equals(bounds2);
     * @param {Bounds} bounds - 需要进行计较的 bounds。
     * @returns {boolean} 如果 bounds 对象的边和传入的 bounds 一致则返回 true，不一致或传入的 bounds 参数为 NULL 则返回 false。
     */
    equals(bounds) {
        var equals = false;
        if (bounds != null) {
            equals = ((this.left === bounds.left) &&
                (this.right === bounds.right) &&
                (this.top === bounds.top) &&
                (this.bottom === bounds.bottom));
        }
        return equals;
    }

    /**
     * @function Bounds.prototype.toString
     * @description 返回此对象的字符串形式。
     * @example
     * var bounds = new Bounds(-180,-90,180,90);
     * var str = bounds.toString();
     * @returns {string} 边界对象的字符串表示形式（left,bottom,right,top），例如: "-180,-90,180,90"。
     */
    toString() {
        return [this.left, this.bottom, this.right, this.top].join(",");
    }

    /**
     * @function Bounds.prototype.toArray
     * @description 边界对象的数组表示形式。
     * @example
     * var bounds = new Bounds(-180,-90,100,80);
     * //array1 = [-180,-90,100,80];
     * var array1 = bounds.toArray();
     * //array1 = [-90,-180,80,100];
     * var array2 = bounds.toArray(true);
     * @param {boolean} [reverseAxisOrder=false] - 是否反转轴顺序。
     * 如果设为 true，则倒转顺序（bottom,left,top,right），否则按正常轴顺序（left,bottom,right,top）。
     * @returns {Array.<number>} left, bottom, right, top 数组。
     */
    toArray(reverseAxisOrder) {
        if (reverseAxisOrder === true) {
            return [this.bottom, this.left, this.top, this.right];
        } else {
            return [this.left, this.bottom, this.right, this.top];
        }
    }

    /**
     * @function Bounds.prototype.toBBOX
     * @description 取小数点后 decimal 位数字进行四舍五入再转换为 BBOX 字符串。
     * @example
     * var bounds = new Bounds(-1.1234567,-1.7654321,1.4444444,1.5555555);
     * //str1 = "-1.123457,-1.765432,1.444444,1.555556";
     * var str1 = bounds.toBBOX();
     * //str2 = "-1.1,-1.8,1.4,1.6";
     * var str2 = bounds.toBBOX(1);
     * //str2 = "-1.8,-1.1,1.6,1.4";
     * var str2 = bounds.toBBOX(1,true);
     * @param {number} [decimal=6] - 边界方位坐标的有效数字个数。
     * @param {boolean} [reverseAxisOrder=false] - 是否是反转轴顺序。
     * 如果设为true，则倒转顺序（bottom,left,top,right）,否则按正常轴顺序（left,bottom,right,top）。
     * @returns {string} 边界对象的字符串表示形式，如："5,42,10,45"。
     */
    toBBOX(decimal, reverseAxisOrder) {
        if (decimal == null) {
            decimal = 6;
        }
        var mult = Math.pow(10, decimal);
        var xmin = Math.round(this.left * mult) / mult;
        var ymin = Math.round(this.bottom * mult) / mult;
        var xmax = Math.round(this.right * mult) / mult;
        var ymax = Math.round(this.top * mult) / mult;
        if (reverseAxisOrder === true) {
            return ymin + "," + xmin + "," + ymax + "," + xmax;
        } else {
            return xmin + "," + ymin + "," + xmax + "," + ymax;
        }
    }

    ///**
    // * @function Bounds.prototype.toGeometry
    // * @description 基于当前边界范围创建一个新的多边形对象。
    // * @example
    // * var bounds = new Bounds(-180,-90,100,80);
    // * // Polygon对象
    // * var geo = bounds.toGeometry();
    // * @returns {GeometryPolygon} 基于当前 bounds 坐标创建的新的多边形。
    // */
    // toGeometry() {
    //     return new Polygon([
    //         new LinearRing([
    //             new Point(this.left, this.bottom),
    //             new Point(this.right, this.bottom),
    //             new Point(this.right, this.top),
    //             new Point(this.left, this.top)
    //         ])
    //     ]);
    // }

    /**
     * @function Bounds.prototype.getWidth
     * @description 获取 bounds 的宽度。
     * @example
     * var bounds = new Bounds(-180,-90,100,80);
     * //width = 280;
     * var width = bounds.getWidth();
     * @returns {number} 获取当前 bounds 的宽度（right 减去 left）。
     */
    getWidth() {
        return (this.right - this.left);
    }

    /**
     * @function Bounds.prototype.getHeight
     * @description 获取 bounds 的高度。
     * @example
     * var bounds = new Bounds(-180,-90,100,80);
     * //height = 170;
     * var height = bounds.getHeight();
     * @returns {number} 边界高度（top 减去 bottom）。
     */
    getHeight() {
        return (this.top - this.bottom);
    }

    /**
     * @function Bounds.prototype.getSize
     * @description 获取边框大小。
     * @example
     * var bounds = new Bounds(-180,-90,100,80);
     * var size = bounds.getSize();
     * @returns {Size} 边框大小。
     */
    getSize() {
        return new Size(this.getWidth(), this.getHeight());
    }

    /**
     * @function Bounds.prototype.getCenterPixel
     * @description 获取像素格式的范围中心点。
     * @example
     * var bounds = new Bounds(-180,-90,100,80);
     * var pixel = bounds.getCenterPixel();
     * @returns {Pixel} 像素格式的当前范围的中心点。
     */
    getCenterPixel() {
        return new Pixel((this.left + this.right) / 2,
            (this.bottom + this.top) / 2);
    }

    /**
     * @function Bounds.prototype.getCenterLonLat
     * @description 获取地理格式的范围中心点。
     * @example
     * var bounds = new Bounds(-180,-90,100,80);
     * var lonlat = bounds.getCenterLonLat();
     * @returns {LonLat} 当前地理范围的中心点。
     */
    getCenterLonLat() {
        if (!this.centerLonLat) {
            this.centerLonLat = new LonLat(
                (this.left + this.right) / 2, (this.bottom + this.top) / 2
            );
        }
        return this.centerLonLat;
    }

    /**
     * @function Bounds.prototype.scale
     * @description 按照比例扩大/缩小出一个新的 bounds。
     * @example
     * var bounds = new Bounds(-50,-50,40,40);
     * var bounds2 = bounds.scale(2);
     * @param {number} [ratio=1] - 需要扩大的比例。
     * @param {(Pixel|LonLat)} [origin] - 扩大时的基准点，默认为当前 bounds 的中心点。
     * @returns {Bounds} 通过 ratio、origin 计算得到的新的边界范围。
     */
    scale(ratio, origin) {
        ratio = ratio ? ratio : 1;
        if (origin == null) {
            origin = this.getCenterLonLat();
        }

        var origx, origy;

        // get origin coordinates
        if (origin.CLASS_NAME === "SuperMap.LonLat") {
            origx = origin.lon;
            origy = origin.lat;
        } else {
            origx = origin.x;
            origy = origin.y;
        }

        var left = (this.left - origx) * ratio + origx;
        var bottom = (this.bottom - origy) * ratio + origy;
        var right = (this.right - origx) * ratio + origx;
        var top = (this.top - origy) * ratio + origy;

        return new Bounds(left, bottom, right, top);
    }

    /**
     * @function Bounds.prototype.add
     * @description 在当前的 Bounds 上按照传入的坐标点进行平移，返回新的范围。
     * @example
     * var bounds1 = new Bounds(-50,-50,40,40);
     * //bounds2 是新的 bounds
     * var bounds2 = bounds.add(20,10);
     * @param {number} x - 坐标点的 x 坐标。
     * @param {number} y - 坐标点的 y 坐标。
     * @returns {Bounds} 新的 bounds，此 bounds 的坐标是由传入的 x，y 参数与当前 bounds 坐标计算所得。
     */
    add(x, y) {
        if ((x == null) || (y == null)) {
            throw new TypeError('Bounds.add cannot receive null values');
        }
        return new Bounds(this.left + x, this.bottom + y,
            this.right + x, this.top + y);
    }

    /**
     * @function Bounds.prototype.extend
     * @description 在当前 bounds 上扩展 bounds，支持 point，lonlat 和 bounds。扩展后的 bounds 的范围是两者的结合。
     * @example
     * var bounds1 = new Bounds(-50,-50,40,40);
     * //bounds 改变
     * bounds.extend(new LonLat(50,60));
     * @param {GeometryPoint|LonLat|Bounds} object - 可以是 point、lonlat 和 bounds。
     */
    extend(object) {
        var bounds = null;
        if (object) {
            // clear cached center location
            switch (object.CLASS_NAME) {
                case "SuperMap.LonLat":
                    bounds = new Bounds(object.lon, object.lat,
                        object.lon, object.lat);
                    break;
                case "SuperMap.Geometry.Point":
                    bounds = new Bounds(object.x, object.y,
                        object.x, object.y);
                    break;

                case "SuperMap.Bounds":
                    bounds = object;
                    break;
            }

            if (bounds) {
                this.centerLonLat = null;
                if ((this.left == null) || (bounds.left < this.left)) {
                    this.left = bounds.left;
                }
                if ((this.bottom == null) || (bounds.bottom < this.bottom)) {
                    this.bottom = bounds.bottom;
                }
                if ((this.right == null) || (bounds.right > this.right)) {
                    this.right = bounds.right;
                }
                if ((this.top == null) || (bounds.top > this.top)) {
                    this.top = bounds.top;
                }
            }
        }
    }

    /**
     * @function Bounds.prototype.containsLonLat
     * @description 判断传入的坐标是否在范围内。
     * @example
     * var bounds1 = new Bounds(-50,-50,40,40);
     * //isContains1 = true
     * //这里的第二个参数可以直接为 boolean 类型，也就是inclusive
     * var isContains1 = bounds.containsLonLat(new LonLat(40,40),true);
     *
     * //(40,40)在范围内，同样(40+360,40)也在范围内
     * var bounds2 = new Bounds(-50,-50,40,40);
     * //isContains2 = true;
     * var isContains2 = bounds2.containsLonLat(
     *      new LonLat(400,40),
     *      {
     *           inclusive:true,
     *           //全球的范围
     *           worldBounds: new Bounds(-180,-90,180,90)
     *      }
     * );
     * @param {(LonLat|Object)} ll - <LonLat> 对象或者是一个包含 'lon' 与 'lat' 属性的对象。
     * @param {Object} options - 可选参数。
     * @param {boolean} [options.inclusive=true] - 是否包含边界。
     * @param {Bounds} [options.worldBounds] - 如果提供 worldBounds 参数, 如果 ll 参数提供的坐标超出了世界边界（worldBounds），
     *        但是通过日界线的转化可以被包含, 它将被认为是包含在该范围内的。
     * @returns {boolean} 传入坐标是否包含在范围内。
     */
    containsLonLat(ll, options) {
        if (typeof options === "boolean") {
            options = {inclusive: options};
        }
        options = options || {};
        var contains = this.contains(ll.lon, ll.lat, options.inclusive),
            worldBounds = options.worldBounds;
        //日界线以外的也有可能算包含，
        if (worldBounds && !contains) {
            var worldWidth = worldBounds.getWidth();
            var worldCenterX = (worldBounds.left + worldBounds.right) / 2;
            //这一步很关键
            var worldsAway = Math.round((ll.lon - worldCenterX) / worldWidth);
            contains = this.containsLonLat({
                lon: ll.lon - worldsAway * worldWidth,
                lat: ll.lat
            }, {inclusive: options.inclusive});
        }
        return contains;
    }

    /**
     * @function Bounds.prototype.containsPixel
     * @description 判断传入的像素是否在范围内。直接匹配大小，不涉及像素和地理转换。
     * @example
     * var bounds = new Bounds(-50,-50,40,40);
     * //isContains = true
     * var isContains = bounds.containsPixel(new Pixel(40,40),true);
     * @param {Pixel} px - 提供的像素参数。
     * @param {boolean} [inclusive=true] - 是否包含边界。
     * @returns {boolean} 传入的 pixel 在当前边界范围之内。
     */
    containsPixel(px, inclusive) {
        return this.contains(px.x, px.y, inclusive);
    }

    /**
     * @function Bounds.prototype.contains
     * @description 判断传入的 x，y 坐标值是否在范围内。
     * @example
     * var bounds = new Bounds(-50,-50,40,40);
     * //isContains = true
     * var isContains = bounds.contains(40,40,true);
     * @param {number} x - x 坐标值。
     * @param {number} y - y 坐标值。
     * @param {boolean} [inclusive=true] - 是否包含边界。
     * @returns {boolean} 传入的 x，y 坐标是否在当前范围内。
     */
    contains(x, y, inclusive) {
        //set default
        if (inclusive == null) {
            inclusive = true;
        }

        if (x == null || y == null) {
            return false;
        }

        //x = Util.toFloat(x);
        //y = Util.toFloat(y);

        var contains = false;
        if (inclusive) {
            contains = ((x >= this.left) && (x <= this.right) &&
                (y >= this.bottom) && (y <= this.top));
        } else {
            contains = ((x > this.left) && (x < this.right) &&
                (y > this.bottom) && (y < this.top));
        }
        return contains;
    }

    /**
     * @function Bounds.prototype.intersectsBounds
     * @description 判断目标边界范围是否与当前边界范围相交。如果两个边界范围中的任意边缘相交或者一个边界包含了另外一个就认为这两个边界相交。
     * @example
     * var bounds = new Bounds(-180,-90,100,80);
     * var isIntersects = bounds.intersectsBounds(
     *      new Bounds(-170,-90,120,80)
     *  );
     * @param {Bounds} bounds - 目标边界。
     * @param {Object} options - 参数。
     * @param {boolean} [options.inclusive=true] - 边缘重合也看成相交。如果是false，两个边界范围没有重叠部分仅仅是在边缘相接（重合），这种情况被认为没有相交。
     * @param {Bounds} [options.worldBounds] - 提供了 worldBounds 参数，如果他们相交时是在全球范围内，两个边界将被视为相交。这仅适用于交叉或完全不在世界范围的边界。
     * @returns {boolean} 传入的 bounds 对象与当前 bounds 相交。
     */
    intersectsBounds(bounds, options) {
        if (typeof options === "boolean") {
            options = {inclusive: options};
        }
        options = options || {};
        if (options.worldBounds) {
            var self = this.wrapDateLine(options.worldBounds);
            bounds = bounds.wrapDateLine(options.worldBounds);
        } else {
            self = this;
        }
        if (options.inclusive == null) {
            options.inclusive = true;
        }
        var intersects = false;
        var mightTouch = (
            self.left === bounds.right ||
            self.right === bounds.left ||
            self.top === bounds.bottom ||
            self.bottom === bounds.top
        );

        // if the two bounds only touch at an edge, and inclusive is false,
        // then the bounds don't *really* intersect.
        if (options.inclusive || !mightTouch) {
            // otherwise, if one of the boundaries even partially contains another,
            // inclusive of the edges, then they do intersect.
            var inBottom = (
                ((bounds.bottom >= self.bottom) && (bounds.bottom <= self.top)) ||
                ((self.bottom >= bounds.bottom) && (self.bottom <= bounds.top))
            );
            var inTop = (
                ((bounds.top >= self.bottom) && (bounds.top <= self.top)) ||
                ((self.top > bounds.bottom) && (self.top < bounds.top))
            );
            var inLeft = (
                ((bounds.left >= self.left) && (bounds.left <= self.right)) ||
                ((self.left >= bounds.left) && (self.left <= bounds.right))
            );
            var inRight = (
                ((bounds.right >= self.left) && (bounds.right <= self.right)) ||
                ((self.right >= bounds.left) && (self.right <= bounds.right))
            );
            intersects = ((inBottom || inTop) && (inLeft || inRight));
        }
        // document me
        if (options.worldBounds && !intersects) {
            var world = options.worldBounds;
            var width = world.getWidth();
            var selfCrosses = !world.containsBounds(self);
            var boundsCrosses = !world.containsBounds(bounds);
            if (selfCrosses && !boundsCrosses) {
                bounds = bounds.add(-width, 0);
                intersects = self.intersectsBounds(bounds, {inclusive: options.inclusive});
            } else if (boundsCrosses && !selfCrosses) {
                self = self.add(-width, 0);
                intersects = bounds.intersectsBounds(self, {inclusive: options.inclusive});
            }
        }
        return intersects;
    }

    /**
     * @function Bounds.prototype.containsBounds
     * @description 判断目标边界是否被当前边界包含在内。
     * @example
     * var bounds = new Bounds(-180,-90,100,80);
     * var isContains = bounds.containsBounds(
     *      new Bounds(-170,-90,100,80),true,true
     *  );
     * @param {Bounds} bounds - 目标边界。
     * @param {boolean} [partial=false] - 目标边界的任意部分都包含在当前边界中则被认为是包含关系。
     * 如果设为 false，整个目标边界全部被包含在当前边界范围内。
     * @param {boolean} [inclusive=true] - 边缘共享是否被视为包含。
     * @returns {boolean} 传入的边界是否被当前边界包含。
     */
    containsBounds(bounds, partial, inclusive) {
        if (partial == null) {
            partial = false;
        }
        if (inclusive == null) {
            inclusive = true;
        }
        var bottomLeft = this.contains(bounds.left, bounds.bottom, inclusive);
        var bottomRight = this.contains(bounds.right, bounds.bottom, inclusive);
        var topLeft = this.contains(bounds.left, bounds.top, inclusive);
        var topRight = this.contains(bounds.right, bounds.top, inclusive);

        return (partial) ? (bottomLeft || bottomRight || topLeft || topRight)
            : (bottomLeft && bottomRight && topLeft && topRight);
    }

    /**
     * @function Bounds.prototype.determineQuadrant
     * @description 判断传入坐标是否在 bounds 范围内的象限。以 bounds 中心点为坐标原点。
     * @example
     * var bounds = new Bounds(-180,-90,100,80);
     * //str = "tr";
     * var str = bounds.determineQuadrant(
     *      new LonLat(20,20)
     *  );
     * @param {LonLat} lonlat - 传入的坐标对象。
     * @returns {string} 传入坐标所在的象限（"br" "tr" "tl" "bl" 分别对应"右下"，"右上"，"左上" "左下"）。
     */
    determineQuadrant(lonlat) {

        var quadrant = "";
        var center = this.getCenterLonLat();

        quadrant += (lonlat.lat < center.lat) ? "b" : "t";
        quadrant += (lonlat.lon < center.lon) ? "l" : "r";

        return quadrant;
    }

    /**
     * @function Bounds.prototype.wrapDateLine
     * @description 将当前 bounds 移动到最大边界范围内部（所谓的内部是相交或者内部）。
     * @example
     * var bounds = new Bounds(380,-40,400,-20);
     * var maxExtent = new Bounds(-180,-90,100,80);
     * //新的bounds
     * var newBounds = bounds.wrapDateLine(maxExtent);
     * @param {Bounds} maxExtent - 最大的边界范围（一般是全球范围）。
     * @param {Object} options - 可选选项参数。
     * @param {number} [options.leftTolerance=0] - left 允许的误差。
     * @param {number} [options.rightTolerance=0] - right 允许的误差。
     * @returns {Bounds} 克隆当前边界。如果当前边界完全在最大范围之外此函数则返回一个不同值的边界，
     *                            若落在最大边界的左边，则给当前的bounds值加上最大范围的宽度，即向右移动，
     *                            若落在右边，则向左移动，即给当前的bounds值加上负的最大范围的宽度。
     */
    wrapDateLine(maxExtent, options) {
        options = options || {};

        var leftTolerance = options.leftTolerance || 0;
        var rightTolerance = options.rightTolerance || 0;

        var newBounds = this.clone();

        if (maxExtent) {
            var width = maxExtent.getWidth();
            //如果 newBounds 在 maxExtent 的左边，那么一直向右移动，直到相交或者包含为止，每次移动width
            //shift right?
            while (newBounds.left < maxExtent.left &&
            newBounds.right - rightTolerance <= maxExtent.left) {
                newBounds = newBounds.add(width, 0);
            }
            //如果 newBounds 在 maxExtent 的右边，那么一直向左移动，直到相交或者包含为止，每次移动width
            //shift left?
            while (newBounds.left + leftTolerance >= maxExtent.right &&
            newBounds.right > maxExtent.right) {
                newBounds = newBounds.add(-width, 0);
            }
            //如果和右边相交，左边又在内部，那么再次向左边移动一次
            // crosses right only? force left
            var newLeft = newBounds.left + leftTolerance;
            if (newLeft < maxExtent.right && newLeft > maxExtent.left &&
                newBounds.right - rightTolerance > maxExtent.right) {
                newBounds = newBounds.add(-width, 0);
            }
        }

        return newBounds;
    }

    /**
     * @function Bounds.prototype.toServerJSONObject
     * @description 转换成对应的 JSON 格式对象。
     * @example
     * var bounds = new Bounds(-180,-90,100,80);
     * var obj = bounds.toServerJSONObject();
     * @returns {Object} JSON 格式的 Object 对象。
     */
    toServerJSONObject() {
        var jsonObject = {
            rightTop: {x: this.right, y: this.top},
            leftBottom: {x: this.left, y: this.bottom},
            left: this.left,
            right: this.right,
            top: this.top,
            bottom: this.bottom
        }
        return jsonObject;
    }

    /**
     *
     * @function Bounds.prototype.destroy
     * @description 销毁此对象。
     * 销毁后此对象的所有属性为 null，而不是初始值。
     * @example
     * var bounds = new Bounds(-180,-90,100,80);
     * bounds.destroy();
     */
    destroy() {
        this.left = null;
        this.right = null;
        this.top = null;
        this.bottom = null;
        this.centerLonLat = null;
    }

    /**
     * @function Bounds.fromString
     * @description 通过字符串参数创建新的 bounds 的构造函数。
     * @example
     * var bounds = Bounds.fromString("-180,-90,100,80");
     * @param {string} str - 边界字符串，用逗号隔开（e.g. <i>"5,42,10,45"</i>）。
     * @param {boolean} [reverseAxisOrder=false] - 是否反转轴顺序。
     * 如果设为true，则倒转顺序（bottom,left,top,right），否则按正常轴顺序（left,bottom,right,top）。
     * @returns {Bounds} 给定的字符串创建的新的边界对象。
     */
    static fromString(str, reverseAxisOrder) {
        var bounds = str.split(",");
        return Bounds.fromArray(bounds, reverseAxisOrder);
    }

    /**
     * @function Bounds.fromArray
     * @description 通过边界框数组创建 Bounds。
     * @example
     * var bounds = Bounds.fromArray([-180,-90,100,80]);
     * @param {Array.<number>} bbox - 边界值数组。（e.g. <i>[5,42,10,45]</i>）。
     * @param {boolean} [reverseAxisOrder=false] - 是否是反转轴顺序。如果设为true，则倒转顺序（bottom,left,top,right），否则按正常轴顺序（left,bottom,right,top）。
     * @returns {Bounds} 根据传入的数组创建的新的边界对象。
     */
    static fromArray(bbox, reverseAxisOrder) {
        return reverseAxisOrder === true ?
            new Bounds(bbox[1], bbox[0], bbox[3], bbox[2]) :
            new Bounds(bbox[0], bbox[1], bbox[2], bbox[3]);
    }

    /**
     * @function Bounds.fromSize
     * @description 通过传入的边界大小来创建新的边界。
     * @example
     * var bounds = Bounds.fromSize(new Size(20,10));
     * @param {Size} size - 边界大小。
     * @returns {Bounds} 根据传入的边界大小的创建新的边界。
     */
    static fromSize(size) {
        return new Bounds(0,
            size.h,
            size.w,
            0);
    }

    /**
     * @function Bounds.oppositeQuadrant
     * @description 反转象限。"t"和"b" 交换，"r"和"l"交换, 如："tl"变为"br"。
     * @param {string} quadrant - 代表象限的字符串，如："tl"。
     * @returns {string} 反转后的象限。
     */
    static oppositeQuadrant(quadrant) {
        var opp = "";

        opp += (quadrant.charAt(0) === 't') ? 'b' : 't';
        opp += (quadrant.charAt(1) === 'l') ? 'r' : 'l';

        return opp;
    }

}

;// CONCATENATED MODULE: ./src/common/commontypes/geometry/Collection.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class GeometryCollection
 * @aliasclass Geometry.Collection
 * @deprecatedclass SuperMap.Geometry.Collection
 * @classdesc 几何对象集合类，存储在本地的 components 属性中（可作为参数传递给构造函数）。<br>
 *            随着新的几何图形添加到集合中，将不能被克隆，当移动几何图形时，需要指定参照物。<br>
 *            getArea 和 getLength 函数只能通过遍历存储几何对象的 components 数组，总计所有几何图形的面积和长度。
 * @category BaseTypes Geometry
 * @extends {Geometry}
 * @param {Array.<Geometry>} components - 几何对象数组。
 * @example
 * var point1 = new GeometryPoint(10,20);
 * var point2 = new GeometryPoint(30,40);
 * var col = new GeometryCollection([point1,point2]);
 * @usage
 */
class Collection extends Geometry_Geometry {


    constructor(components) {
        super();

        /**
         * @description 存储几何对象的数组。
         * @member {Array.<Geometry>} GeometryCollection.prototype.components
         */
        this.components = [];

        /**
         * @member {Array.<string>} GeometryCollection.prototype.componentTypes
         * @description components 存储的几何对象所支持的几何类型数组，为空表示类型不受限制。
         */
        this.componentTypes = null;
        if (components != null) {
            this.addComponents(components);
        }
        this.CLASS_NAME = "SuperMap.Geometry.Collection";
        this.geometryType = "Collection";
    }

    /**
     * @function GeometryCollection.prototype.destroy
     * @description 销毁几何图形。
     */
    destroy() {
        this.components.length = 0;
        this.components = null;
        super.destroy();
    }

    /**
     * @function GeometryCollection.prototype.clone
     * @description 克隆当前几何对象。
     * @returns {GeometryCollection} 克隆的几何对象集合。
     */
    clone() {
        var geometry = new Collection();
        for (var i = 0, len = this.components.length; i < len; i++) {
            geometry.addComponent(this.components[i].clone());
        }

        // catch any randomly tagged-on properties
        Util_Util.applyDefaults(geometry, this);

        return geometry;
    }

    /**
     * @function GeometryCollection.prototype.getComponentsString
     * @description 获取 components 字符串。
     * @returns {string} components 字符串。
     */
    getComponentsString() {
        var strings = [];
        for (var i = 0, len = this.components.length; i < len; i++) {
            strings.push(this.components[i].toShortString());
        }
        return strings.join(",");
    }

    /**
     * @function GeometryCollection.prototype.calculateBounds
     * @description 通过遍历数组重新计算边界，在遍历每一子项中时调用 extend 方法。
     */
    calculateBounds() {
        this.bounds = null;
        var bounds = new Bounds();
        var components = this.components;
        if (components) {
            for (var i = 0, len = components.length; i < len; i++) {
                bounds.extend(components[i].getBounds());
            }
        }
        // to preserve old behavior, we only set bounds if non-null
        // in the future, we could add bounds.isEmpty()
        if (bounds.left != null && bounds.bottom != null &&
            bounds.right != null && bounds.top != null) {
            this.setBounds(bounds);
        }
    }

    /**
     * @function GeometryCollection.prototype.addComponents
     * @description 给几何图形对象添加元素。
     * @param {Array.<Geometry>} components - 几何对象组件。
     * @example
     * var geometryCollection = new GeometryCollection();
     * geometryCollection.addComponents(new SuerpMap.Geometry.Point(10,10));
     */
    addComponents(components) {
        if (!(Util_Util.isArray(components))) {
            components = [components];
        }
        for (var i = 0, len = components.length; i < len; i++) {
            this.addComponent(components[i]);
        }
    }

    /**
     * @function GeometryCollection.prototype.addComponent
     * @description 添加几何对象到集合中。如果设置了 componentTypes 类型，则添加的几何对象必须是 componentTypes 中的类型。
     * @param {Geometry} component - 待添加的几何对象。
     * @param {number} [index] - 几何对象插入的位置。
     * @returns {boolean} 是否添加成功。
     */
    addComponent(component, index) {
        var added = false;
        if (component) {
            if (this.componentTypes == null ||
                (Util_Util.indexOf(this.componentTypes,
                    component.CLASS_NAME) > -1)) {

                if (index != null && (index < this.components.length)) {
                    var components1 = this.components.slice(0, index);
                    var components2 = this.components.slice(index,
                        this.components.length);
                    components1.push(component);
                    this.components = components1.concat(components2);
                } else {
                    this.components.push(component);
                }
                component.parent = this;
                this.clearBounds();
                added = true;
            }
        }
        return added;
    }

    /**
     * @function GeometryCollection.prototype.removeComponents
     * @description 清除几何对象。
     * @param {Array.<Geometry>} components - 需要清除的几何对象。
     * @returns {boolean} 元素是否被删除。
     */
    removeComponents(components) {
        var removed = false;

        if (!(Util_Util.isArray(components))) {
            components = [components];
        }
        for (var i = components.length - 1; i >= 0; --i) {
            removed = this.removeComponent(components[i]) || removed;
        }
        return removed;
    }

    /**
     * @function GeometryCollection.prototype.removeComponent
     * @description 从集合中移除几何对象。
     * @param {Geometry} component - 要移除的几何对象。
     * @returns {boolean} 几何对象是否移除成功。
     */
    removeComponent(component) {
        Util_Util.removeItem(this.components, component);

        // clearBounds() so that it gets recalculated on the next call
        // to this.getBounds();
        this.clearBounds();
        return true;
    }

    /**
     * @function GeometryCollection.prototype.getArea
     * @description 计算几何对象的面积。注意，这个方法在 {@link GeometryPolygon} 类中需要重写。
     * @returns {number} 几何图形的面积，是几何对象中所有组成部分的面积之和。
     */
    getArea() {
        var area = 0.0;
        for (var i = 0, len = this.components.length; i < len; i++) {
            area += this.components[i].getArea();
        }
        return area;
    }

    /**
     * @function GeometryCollection.prototype.equals
     * @description 判断两个几何图形是否相等。如果所有的 components 具有相同的坐标，则认为是相等的。
     * @param {Geometry} geometry - 需要判断的几何图形。
     * @returns {boolean} 输入的几何图形与当前几何图形是否相等。
     */
    equals(geometry) {
        var equivalent = true;
        if (!geometry || !geometry.CLASS_NAME ||
            (this.CLASS_NAME !== geometry.CLASS_NAME)) {
            equivalent = false;
        } else if (!(Util_Util.isArray(geometry.components)) ||
            (geometry.components.length !== this.components.length)) {
            equivalent = false;
        } else {
            for (var i = 0, len = this.components.length; i < len; ++i) {
                if (!this.components[i].equals(geometry.components[i])) {
                    equivalent = false;
                    break;
                }
            }
        }
        return equivalent;
    }


    /**
     * @function GeometryCollection.prototype.getVertices
     * @description 返回几何对象的所有结点的列表。
     * @param {boolean} [nodes] - 对于线来说，仅仅返回作为端点的顶点，如果设为 false，则返回非端点的顶点，如果没有设置此参数，则返回所有顶点。
     * @returns {Array} 几何对象的顶点列表。
     */
    getVertices(nodes) {
        var vertices = [];
        for (var i = 0, len = this.components.length; i < len; ++i) {
            Array.prototype.push.apply(
                vertices, this.components[i].getVertices(nodes)
            );
        }
        return vertices;
    }

}

;// CONCATENATED MODULE: ./src/common/commontypes/geometry/MultiPoint.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class GeometryMultiPoint
 * @aliasclass Geometry.MultiPoint
 * @deprecatedclass SuperMap.Geometry.MultiPoint
 * @classdesc 几何对象多点类。
 * @category BaseTypes Geometry
 * @extends GeometryCollection
 * @param {Array.<GeometryPoint>} components - 点对象数组。
 * @example
 * var point1 = new GeometryPoint(5,6);
 * var poine2 = new GeometryMultiPoint(7,8);
 * var multiPoint = new MultiPoint([point1,point2]);
 * @usage
 */
class MultiPoint extends Collection {


    constructor(components) {
        super(components);
        /**
         * @member {Array.<string>} [GeometryMultiPoint.prototype.componentTypes=["SuperMap.Geometry.Point"]]
         * @description components 存储的几何对象所支持的几何类型数组。
         * @readonly
         */
        this.componentTypes = ["SuperMap.Geometry.Point"];
        this.CLASS_NAME = "SuperMap.Geometry.MultiPoint";
        this.geometryType = "MultiPoint";
    }

    /**
     * @function GeometryMultiPoint.prototype.addPoint
     * @description 添加点，封装了 {@link GeometryCollection|GeometryCollection.addComponent} 方法。
     * @param {GeometryPoint} point - 添加的点。
     * @param {number} [index] - 下标。
     */
    addPoint(point, index) {
        this.addComponent(point, index);
    }


    /**
     * @function GeometryMultiPoint.prototype.removePoint
     * @description 移除点，封装了 {@link GeometryCollection|GeometryCollection.removeComponent} 方法。
     * @param {GeometryPoint} point - 移除的点对象。
     */
    removePoint(point) {
        this.removeComponent(point);
    }


}

;// CONCATENATED MODULE: ./src/common/commontypes/geometry/Curve.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class GeometryCurve
 * @aliasclass Geometry.Curve
 * @deprecatedclass SuperMap.Geometry.Curve
 * @classdesc 几何对象曲线类。
 * @category BaseTypes Geometry
 * @extends GeometryMultiPoint
 * @param {Array.<GeometryPoint>} components - 几何对象数组。
 * @example
 * var point1 = new GeometryPoint(10,20);
 * var point2 = new GeometryPoint(30,40);
 * var curve = new Curve([point1,point2]);
 * @usage
 */
class Curve extends MultiPoint {

    constructor(components) {
        super(components);
        /**
         * @member {Array.<string>} [GeometryCurve.prototype.componentTypes=["SuperMap.Geometry.Point", "SuperMap.PointWithMeasure"]]
         * @description components 存储的几何对象所支持的几何类型数组。
         * @readonly
         */
        this.componentTypes = ["SuperMap.Geometry.Point", "SuperMap.PointWithMeasure"];
        this.CLASS_NAME = "SuperMap.Geometry.Curve";
        this.geometryType = "Curve";

    }


}

;// CONCATENATED MODULE: ./src/common/commontypes/geometry/Point.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class GeometryPoint
 * @aliasclass Geometry.Point
 * @deprecatedclass SuperMap.Geometry.Point
 * @classdesc 点几何对象类。
 * @category BaseTypes Geometry
 * @extends {Geometry}
 * @param {number} x - x 坐标。
 * @param {number} y - y 坐标。
 * @param {string} [type = 'Point'] - 点的类型。
 * @param {number} [tag] - 额外的属性，比如插值分析中的 Z 值。
 * @example
 * var point = new GeometryPoint(-111.04, 45.68);
 * @usage
 */
class Point extends Geometry_Geometry {


    constructor(x, y, type, tag) {
        super(x, y, type, tag);
        /**
         * @member {number} GeometryPoint.prototype.x
         * @description 横坐标。
         */
        this.x = parseFloat(x);

        /**
         * @member {number} GeometryPoint.prototype.y
         * @description 纵坐标。
         */
        this.y = parseFloat(y);

        /**
         * @member {string} GeometryPoint.prototype.tag
         * @description  用来存储额外的属性，比如插值分析中的 Z 值。
         */
        this.tag = (tag || tag == 0) ? parseFloat(tag) : null;

        /**
         * @member {string} GeometryPoint.prototype.type
         * @description  用来存储点的类型。
         */
        this.type = type || "Point";
        this.CLASS_NAME = "SuperMap.Geometry.Point";
        this.geometryType = "Point";
    }

    /**
     * @function GeometryPoint.prototype.clone
     * @description 克隆点对象。
     * @returns {GeometryPoint} 克隆后的点对象。
     */
    clone(obj) {
        if (obj == null) {
            obj = new Point(this.x, this.y);
        }

        // catch any randomly tagged-on properties
        Util_Util.applyDefaults(obj, this);

        return obj;
    }

    /**
     * @function GeometryPoint.prototype.calculateBounds
     * @description 计算点对象的范围。
     */
    calculateBounds() {
        this.bounds = new Bounds(this.x, this.y,
            this.x, this.y);
    }

    /**
     * @function GeometryPoint.prototype.equals
     * @description 判断两个点对象是否相等。如果两个点对象具有相同的坐标，则认为是相等的。
     * @example
     * var point= new GeometryPoint(0,0);
     * var point1={x:0,y:0};
     * var result= point.equals(point1);
     * @param {GeometryPoint} geom - 需要判断的点对象。
     * @returns {boolean} 两个点对象是否相等（true 为相等，false 为不等）。
     */
    equals(geom) {
        var equals = false;
        if (geom != null) {
            equals = ((this.x === geom.x && this.y === geom.y) ||
                (isNaN(this.x) && isNaN(this.y) && isNaN(geom.x) && isNaN(geom.y)));
        }
        return equals;
    }


    /**
     * @function GeometryPoint.prototype.move
     * @description 沿着 x、y 轴的正方向上按照给定的位移移动点对象，move 不仅改变了几何对象的位置并且清理了边界缓存。
     * @param {number} x - x 轴正方向上的偏移量。
     * @param {number} y - y 轴正方向上偏移量。
     */
    move(x, y) {
        this.x = this.x + x;
        this.y = this.y + y;
        this.clearBounds();
    }

    /**
     * @function GeometryPoint.prototype.toShortString
     * @description 将 x/y 坐标转换成简单字符串。
     * @returns {string} 字符串代表点对象。(ex. <i>"5, 42"</i>)
     */
    toShortString() {
        return (this.x + ", " + this.y);
    }

    /**
     * @function GeometryPoint.prototype.destroy
     * @description 释放点对象的资源。
     */
    destroy() {
        this.x = null;
        this.y = null;
        this.tag = null;
        super.destroy();
    }

    /**
     * @function GeometryPoint.prototype.getVertices
     * @description 获取几何图形所有顶点的列表。
     * @returns {Array} 几何图形的顶点列表。
     */
    getVertices() {
        return [this];
    }


}

;// CONCATENATED MODULE: ./src/common/commontypes/geometry/LineString.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class GeometryLineString
 * @aliasclass Geometry.LineString
 * @deprecatedclass SuperMap.Geometry.LineString
 * @classdesc 几何对象线串类。
 * @category BaseTypes Geometry
 * @param {Array.<GeometryPoint>} points - 用来生成线串的点数组。
 * @extends GeometryCurve
 * @example
 * var points = [new GeometryPoint(4933.319287022352, -3337.3849141502124),
 *     new GeometryPoint(4960.9674060199022, -3349.3316322355736),
 *     new GeometryPoint(5006.0235999418364, -3358.8890067038628),
 *     new GeometryPoint(5075.3145648369318, -3378.0037556404409),
 *     new GeometryPoint(5305.19551436013, -3376.9669111768926)],
 * var roadLine = new GeometryLineString(points)；
 * @usage
 */
class LineString extends Curve {

    constructor(points) {
        super(points);
        this.CLASS_NAME = "SuperMap.Geometry.LineString";
        this.geometryType = "LineString";
    }

    /**
     * @function GeometryLineString.prototype.removeComponent
     * @description 只有在线串上有三个或更多的点的时候，才会允许移除点（否则结果将会是单一的点）。
     * @param {GeometryPoint} point - 将被删除的点。
     * @returns {boolean} 删除的点。
     */
    removeComponent(point) { // eslint-disable-line no-unused-vars
        var removed = this.components && (this.components.length > 2);
        if (removed) {
            super.removeComponent.apply(this, arguments);
        }
        return removed;
    }

    /**
     * @function GeometryLineString.prototype.getSortedSegments
     * @description 获取升序排列的点坐标对象数组。
     * @returns {Array} 升序排列的点坐标对象数组。
     */
    getSortedSegments() {
        var numSeg = this.components.length - 1;
        var segments = new Array(numSeg), point1, point2;
        for (var i = 0; i < numSeg; ++i) {
            point1 = this.components[i];
            point2 = this.components[i + 1];
            if (point1.x < point2.x) {
                segments[i] = {
                    x1: point1.x,
                    y1: point1.y,
                    x2: point2.x,
                    y2: point2.y
                };
            } else {
                segments[i] = {
                    x1: point2.x,
                    y1: point2.y,
                    x2: point1.x,
                    y2: point1.y
                };
            }
        }

        // more efficient to define this somewhere static
        function byX1(seg1, seg2) {
            return seg1.x1 - seg2.x1;
        }

        return segments.sort(byX1);
    }

    /**
     * @function GeometryLineString.prototype.getVertices
     * @description 返回几何图形的所有顶点的列表。
     * @param {boolean} [nodes] - 对于线来说，仅仅返回作为端点的顶点，如果设为 false，则返回非端点的顶点，如果没有设置此参数，则返回所有顶点。
     * @returns {Array} 几何图形的顶点列表。
     */
    getVertices(nodes) {
        var vertices;
        if (nodes === true) {
            vertices = [
                this.components[0],
                this.components[this.components.length - 1]
            ];
        } else if (nodes === false) {
            vertices = this.components.slice(1, this.components.length - 1);
        } else {
            vertices = this.components.slice();
        }
        return vertices;
    }

    /**
     * @function GeometryLineString.calculateCircle
     * @description 三点画圆弧。
     * @param {Array.<GeometryPoint>} points - 传入的待计算的初始点串。
     * @returns {Array.<GeometryPoint>} 计算出相应的圆弧控制点。
     * @example
     * var points = [];
     * points.push(new GeometryPoint(-50,30));
     * points.push(new GeometryPoint(-30,50));
     * points.push(new GeometryPoint(2,60));
     * var circle = GeometryLineString.calculateCircle(points);
     */
    static calculateCircle(points) {
        if (points.length < 3) {
            return points
        }
        var centerPoint = {},
            p1 = points[0],
            p2 = points[1],
            p3 = points[2];
        var R = 0,
            dStep = 0,
            direc = true,
            dRotation = 0,
            dRotationBegin = 0,
            dRotationAngle = 0,
            nSegmentCount = 72,
            circlePoints = [];

        var KTan13 = (p3.y - p1.y) / (p3.x - p1.x);
        var B13 = p3.y - KTan13 * p3.x;
        if ((((p3.x != p1.x) && (p3.y != p1.y)) && (p2.y == KTan13 * p2.x + B13)) ||
            ((p3.x == p1.x) && (p2.x == p1.x)) || ((p3.y == p1.y) && (p2.y == p1.y)) ||
            ((p3.x == p1.x) && (p3.y == p1.y)) || ((p3.x == p2.x) && (p3.y == p2.y)) || ((p1.x == p2.x) && (p1.y == p2.y))) {
            circlePoints.push(p1);
            circlePoints.push(p2);
            circlePoints.push(p3);
        } else {
            var D = ((p2.x * p2.x + p2.y * p2.y) - (p1.x * p1.x + p1.y * p1.y)) * (2 * (p3.y - p1.y)) - ((p3.x * p3.x + p3.y * p3.y) -
                (p1.x * p1.x + p1.y * p1.y)) * (2 * (p2.y - p1.y));
            var E = (2 * (p2.x - p1.x)) * ((p3.x * p3.x + p3.y * p3.y) - (p1.x * p1.x + p1.y * p1.y)) -
                (2 * (p3.x - p1.x)) * ((p2.x * p2.x + p2.y * p2.y) - (p1.x * p1.x + p1.y * p1.y));
            var F = 4 * ((p2.x - p1.x) * (p3.y - p1.y) - (p3.x - p1.x) * (p2.y - p1.y));
            centerPoint.x = D / F;
            centerPoint.y = E / F;
            R = Math.sqrt((p1.x - centerPoint.x) * (p1.x - centerPoint.x) + (p1.y - centerPoint.y) * (p1.y - centerPoint.y));

            var dis = (p1.x - p3.x) * (p1.x - p3.x) + (p1.y - p3.y) * (p1.y - p3.y);
            var cons = (2 * R * R - dis) / (2 * R * R);
            cons = cons >= 1 ? 1 : cons;
            cons = cons <= -1 ? -1 : cons;
            dRotationAngle = Math.acos(cons) * 180 / Math.PI;

            if (p3.x == p1.x) {
                dRotationAngle = ((centerPoint.x > p1.x && p2.x > p1.x) || (centerPoint.x < p1.x && p2.x < p1.x)) ? (360 - dRotationAngle) : dRotationAngle;
            } else {
                dRotationAngle = ((centerPoint.y > (KTan13 * centerPoint.x + B13) && p2.y > (KTan13 * p2.x + B13)) ||
                    (centerPoint.y < (KTan13 * centerPoint.x + B13) && p2.y < (KTan13 * p2.x + B13))) ? (360 - dRotationAngle) : dRotationAngle;
            }
            dStep = dRotationAngle / 72;

            if (p3.y != p1.y) {
                if (p3.x == p1.x) {
                    if (p3.y > p1.y) {
                        if (p2.x < p1.x) {
                            direc = false;
                        }
                    } else {
                        if (p2.x > p1.x) {
                            direc = false;
                        }
                    }
                } else if (p3.x < p1.x) {
                    if (p2.y < KTan13 * p2.x + B13) {
                        direc = false;
                    }
                } else {
                    if (p2.y > KTan13 * p2.x + B13) {
                        direc = false;
                    }
                }
            } else {
                if (p3.x > p1.x) {
                    if (p2.y > p1.y) {
                        direc = false;
                    }
                } else {
                    if (p2.y < p1.y) {
                        direc = false;
                    }
                }
            }

            var K10 = (p1.y - centerPoint.y) / (p1.x - centerPoint.x);
            var atan10 = K10 >= 0 ? Math.atan(K10) * 180 / Math.PI : Math.abs(Math.atan(K10) * 180 / Math.PI) + 90;

            var CY = Math.abs(centerPoint.y);
            if ((p1.y == CY) && (CY == p3.y)) {
                if (p1.x < p3.x) {
                    atan10 = atan10 + 180;
                }
            }

            var newPY = p1.y - centerPoint.y;
            circlePoints.push(p1);
            for (var i = 1; i < nSegmentCount; i++) {
                dRotation = dStep * i;
                dRotationBegin = atan10;

                if (direc) {
                    if (newPY >= 0) {
                        if (K10 >= 0) {
                            dRotationBegin = dRotationBegin + dRotation;
                        } else {
                            dRotationBegin = (180 - (dRotationBegin - 90)) + dRotation;
                        }
                    } else {
                        if (K10 > 0) {
                            dRotationBegin = (dRotationBegin - 180) + dRotation;
                        } else {
                            dRotationBegin = (90 - dRotationBegin) + dRotation;
                        }
                    }
                } else {
                    if (newPY >= 0) {
                        if (K10 >= 0) {
                            dRotationBegin = dRotationBegin - dRotation;
                        } else {
                            dRotationBegin = (180 - (dRotationBegin - 90)) - dRotation;
                        }
                    } else {
                        if (K10 >= 0) {
                            dRotationBegin = (dRotationBegin - 180) - dRotation;
                        } else {
                            dRotationBegin = (90 - dRotationBegin) - dRotation;
                        }
                    }
                }

                dRotationBegin = dRotationBegin * Math.PI / 180;
                var x = centerPoint.x + R * Math.cos(dRotationBegin);
                var y = centerPoint.y + R * Math.sin(dRotationBegin);
                circlePoints.push(new Point(x, y));
            }
            circlePoints.push(p3);
        }
        return circlePoints;
    }

    /**
     * @function GeometryLineString.createLineEPS
     * @description 根据点的类型画出不同类型的曲线。
     * 点的类型有三种：LTypeArc，LTypeCurve，NONE。
     * @param {Array.<GeometryPoint>} points - 传入的待计算的初始点串。
     * @returns {Array.<GeometryPoint>} 计算出相应的 lineEPS 控制点。
     * @example
     * var points = [];
     * points.push(new GeometryPoint(-50,30));
     * points.push(new GeometryPoint(-30,50,"LTypeArc"));
     * points.push(new GeometryPoint(2,60));
     * points.push(new GeometryPoint(8,20));
     * var lineEPS = GeometryLineString.createLineEPS(points);
     */
    static createLineEPS(points) {
        var list = [],
            len = points.length;
        if (len < 2) {
            return points;
        }
        for (var i = 0; i < len;) {
            var type = points[i].type;
            if (type == 'LTypeArc') {
                var listObj = LineString.createLineArc(list, i, len, points);
                list = listObj[0];
                i = listObj[1];
            } else {
                list.push(points[i]);
                i++;
            }
        }
        return list;
    }

    static createLineArc(list, i, len, points) {
        if (i == 0) {
            let bezierPtsObj = LineString.addPointEPS(points, i, len, 'LTypeArc');
            Array.prototype.push.apply(list, bezierPtsObj[0]);
            i = bezierPtsObj[1] + 1;
        } else if (i == len - 1) {
            var bezierP = [points[i - 1], points[i]],
                bezierPts = LineString.calculateCircle(bezierP);
            Array.prototype.push.apply(list, bezierPts);
            i++;
        } else {
            let bezierPtsObj = LineString.addPointEPS(points, i, len, 'LTypeArc');
            list.pop();
            Array.prototype.push.apply(list, bezierPtsObj[0]);
            i = bezierPtsObj[1] + 1;
        }
        return [list, i];
    }

    static addPointEPS(points, i, len, type) {
        var bezierP = [], j = i + 1;
        if (i == 0) {
            Array.prototype.push.apply(bezierP, [points[i], points[i + 1]]);
        } else if (i == len - 1) {
            Array.prototype.push.apply(bezierP, [points[i - 1], points[i]]);
        } else {
            Array.prototype.push.apply(bezierP, [points[i - 1], points[i], points[i + 1]]);
        }
        var bezierPts;
        if (type == 'LTypeCurve') {
            bezierPts = LineString.calculatePointsFBZN(bezierP);
        } else if (type == 'LTypeArc') {
            bezierPts = LineString.calculateCircle(bezierP);
        }
        return [bezierPts, j];
    }
}



;// CONCATENATED MODULE: ./src/common/commontypes/geometry/GeoText.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/








/**
 * @class GeometryGeoText
 * @aliasclass Geometry.GeoText
 * @deprecatedclass SuperMap.Geometry.GeoText
 * @classdesc 文本标签类。
 * @category BaseTypes Geometry
 * @extends {Geometry}
 * @param {number} x - x 坐标。
 * @param {number} y - y 坐标。
 * @param {string} text - 标签中的文本内容。
 * @usage
 */
class GeoText extends Geometry_Geometry {

    constructor(x, y, text) {
        super(x, y, text);
        /**
         * @member {number} GeometryGeoText.prototype.x
         * @description 横坐标。
         */
        this.x = parseFloat(x);

        /**
         * @member {number} GeometryGeoText.prototype.y
         * @description 纵坐标。
         */
        this.y = parseFloat(y);

        /**
         * @member {string} GeometryGeoText.prototype.text
         * @description 标签中的文本内容。
         */
        this.text = text.toString();

        /**
         * @member {Object} GeometryGeoText.prototype.bsInfo
         * @description 标签范围的基础信息。
         * @property {number} w - bounds 的宽度。
         * @property {number} h - bounds 的高度。
         */
        this.bsInfo = {
            "h": null,
            "w": null
        };
        this.element = document.createElement('span');
        this.CLASS_NAME = "SuperMap.Geometry.GeoText";
        this.geometryType = "GeoText";
    }

    /**
     * @function GeometryGeoText.prototype.destroy
     * @description 销毁文本标签类。
     */
    destroy() {
        super.destroy();
        this.x = null;
        this.y = null;
        this.text = null;
    }

    /**
     * @function GeometryGeoText.prototype.getCentroid
     * @description 获取标签对象的质心。
     * @returns {GeometryPoint} 标签对象的质心。
     */
    getCentroid() {
        return new Point(this.x, this.y);
    }

    /**
     * @function GeometryGeoText.prototype.clone
     * @description 克隆标签对象。
     * @returns {GeometryGeoText} 克隆后的标签对象。
     */
    clone(obj) {
        if (obj == null) {
            obj = new GeoText(this.x, this.y, this.text);
        }
        Util_Util.applyDefaults(obj, this);
        return obj;
    }

    /**
     * @function GeometryGeoText.prototype.calculateBounds
     * @description 计算标签对象的范围。
     */
    calculateBounds() {
        this.bounds = new Bounds(this.x, this.y,
            this.x, this.y);
    }

    /**
     * @function GeometryGeoText.prototype.getLabelPxBoundsByLabel
     * @description 根据绘制好的标签获取文字标签的像素范围，参数的单位是像素；此方法相对于 getLabelPxBoundsByText 效率较低，但支持所有格式的文本。
     * @param {Object} locationPixel - 标签的位置点，该对象含有属性 x（横坐标），属性 y（纵坐标）。
     * @param {string} labelWidth - 标签的宽度，如：“90px”。
     * @param {string} labelHeight - 标签的高度。
     * @param {Object} style - 标签的 style。
     * @returns {Bounds} 标签的像素范围。
     */
    getLabelPxBoundsByLabel(locationPixel, labelWidth, labelHeight, style) {
        var labelPxBounds, left, bottom, top, right;
        var locationPx = Util_Util.cloneObject(locationPixel);

        //计算文本行数
        var theText = style.label || this.text;
        var textRows = theText.split('\n');
        var laberRows = textRows.length;

        //处理文字对齐
        labelWidth = parseFloat(labelWidth);
        labelHeight = parseFloat(labelHeight);
        if (laberRows > 1) {
            labelHeight = parseFloat(labelHeight) * laberRows;
        }
        if (style.labelAlign && style.labelAlign !== "cm") {
            switch (style.labelAlign) {
                case "lt":
                    locationPx.x += labelWidth / 2;
                    locationPx.y += labelHeight / 2;
                    break;
                case "lm":
                    locationPx.x += labelWidth / 2;
                    break;
                case "lb":
                    locationPx.x += labelWidth / 2;
                    locationPx.y -= labelHeight / 2;
                    break;
                case "ct":
                    locationPx.y += labelHeight / 2;
                    break;
                case "cb":
                    locationPx.y -= labelHeight / 2;
                    break;
                case "rt":
                    locationPx.x -= labelWidth / 2;
                    locationPx.y += labelHeight / 2;
                    break;
                case "rm":
                    locationPx.x -= labelWidth / 2;
                    break;
                case "rb":
                    locationPx.x -= labelWidth / 2;
                    locationPx.y -= labelHeight / 2;
                    break;
                default:
                    break;
            }
        }

        this.bsInfo.h = labelHeight;
        this.bsInfo.w = labelWidth;

        //bounds的四边
        left = locationPx.x - parseFloat(labelWidth) / 2;
        bottom = locationPx.y + parseFloat(labelHeight) / 2;
        right = locationPx.x + parseFloat(labelWidth) / 2;
        top = locationPx.y - parseFloat(labelHeight) / 2;

        labelPxBounds = new Bounds(left, bottom, right, top);

        return labelPxBounds;
    }

    /**
     * @function GeometryGeoText.prototype.getLabelPxBoundsByText
     * @description 根据文本内容获取文字标签的像素范围。
     * @param {Object} locationPixel - 标签的位置点，该对象含有属性 x（横坐标），属性 y（纵坐标）。
     * @param {Object} style - 标签的样式。
     * @returns {Bounds} 标签的像素范围。
     */
    getLabelPxBoundsByText(locationPixel, style) {
        var labelPxBounds, left, bottom, top, right;
        var labelSize = this.getLabelPxSize(style);
        var locationPx = Util_Util.cloneObject(locationPixel);

        //处理文字对齐
        if (style.labelAlign && style.labelAlign !== "cm") {
            switch (style.labelAlign) {
                case "lt":
                    locationPx.x += labelSize.w / 2;
                    locationPx.y += labelSize.h / 2;
                    break;
                case "lm":
                    locationPx.x += labelSize.w / 2;
                    break;
                case "lb":
                    locationPx.x += labelSize.w / 2;
                    locationPx.y -= labelSize.h / 2;
                    break;
                case "ct":
                    locationPx.y += labelSize.h / 2;
                    break;
                case "cb":
                    locationPx.y -= labelSize.h / 2;
                    break;
                case "rt":
                    locationPx.x -= labelSize.w / 2;
                    locationPx.y += labelSize.h / 2;
                    break;
                case "rm":
                    locationPx.x -= labelSize.w / 2;
                    break;
                case "rb":
                    locationPx.x -= labelSize.w / 2;
                    locationPx.y -= labelSize.h / 2;
                    break;
                default:
                    break;
            }
        }

        this.bsInfo.h = labelSize.h;
        this.bsInfo.w = labelSize.w;


        left = locationPx.x - labelSize.w / 2;
        bottom = locationPx.y + labelSize.h / 2;
        //处理斜体字
        if (style.fontStyle && style.fontStyle === "italic") {
            right = locationPx.x + labelSize.w / 2 + parseInt(parseFloat(style.fontSize) / 2);
        } else {
            right = locationPx.x + labelSize.w / 2;
        }
        top = locationPx.y - labelSize.h / 2;

        labelPxBounds = new Bounds(left, bottom, right, top);

        return labelPxBounds;
    }

    /**
     * @function GeometryGeoText.prototype.getLabelPxSize
     * @description 获取 label 的像素大小。
     * @param {Object} style - 标签样式。
     * @returns {Object} 标签大小对象，属性 w 表示标签的宽度，属性 h 表示标签的高度。
     */
    getLabelPxSize(style) {
        var text,//文本内容
            fontSize,//字体大小
            spacing = 1,//两个字符间的间距（单位：px）
            lineSpacing = 0.2,
            bgstrokeWidth = parseFloat(style.strokeWidth);//标签背景框边框的宽度

        text = style.label || this.text;
        if (style.fontSize) {
            fontSize = parseFloat(style.fontSize);
        } else {
            fontSize = parseFloat("12px");
        }

        //标签宽高
        var labelW, labelH;

        var textRows = text.split('\n');
        var numRows = textRows.length;

        if (numRows > 1) {
            labelH = fontSize * numRows + numRows + bgstrokeWidth + lineSpacing * fontSize;
        } else {
            labelH = fontSize + bgstrokeWidth + lineSpacing * fontSize + 1;
        }

        //取最大宽度
        labelW = 0;
        if (this.labelWTmp && labelW < this.labelWTmp) {
            labelW = this.labelWTmp;
        }
        for (var i = 0; i < numRows; i++) {
            var textCharC = this.getTextCount(textRows[i]);
            var labelWTmp = this.labelWTmp = Util_Util.getTextBounds(style, textRows[i], this.element).textWidth + textCharC.textC * spacing + bgstrokeWidth;
            if (labelW < labelWTmp) {
                labelW = labelWTmp;
            }
        }

        var labelSize = new Object(); //标签大小
        labelSize.h = labelH;
        labelSize.w = labelW;

        return labelSize;
    }

    /**
     * @function GeometryGeoText.prototype.getTextCount
     * @description 获取 text 中的字符个数。
     * @param {string} text - 字符串。
     * @returns {Object} 字符个数统计结果，属性 cnC 表示中文字符个数，属性 enC 表示英文字符个数，属性 textC 表示字符总个数。
     */
    getTextCount(text) {
        var textCharCount = {};

        var cnCount = 0;
        var enCount = 0;

        for (var i = 0; i < text.length; i++) {
            if (text.charCodeAt(i) > 255) { //遍历判断字符串中每个字符的Unicode码,大于255则为中文
                cnCount++;
            } else {
                enCount++;
            }
        }
        //中午字符个数
        textCharCount.cnC = cnCount;
        //英文字符个数
        textCharCount.enC = enCount;
        //字符总个数
        textCharCount.textC = text.length;

        return textCharCount;
    }


}

;// CONCATENATED MODULE: ./src/common/commontypes/geometry/LinearRing.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class  GeometryLinearRing
 * @aliasclass Geometry.LinearRing
 * @deprecatedclass SuperMap.Geometry.LinearRing
 * @classdesc 几何对象线环类，是一个特殊的封闭的线串，在每次 addPoint/removePoint 之后会通过添加一个点（此点是复制的第一个点得到的）
 * 作为最后的一个点来自动关闭线环。
 * @category BaseTypes Geometry
 * @extends GeometryLineString
 * @param {Array.<GeometryPoint>} points - 组成线性环的点。
 * @example
 * var points = [new GeometryPoint(4933.319287022352, -3337.3849141502124),
 *      new GeometryPoint(4960.9674060199022, -3349.3316322355736),
 *      new GeometryPoint(5006.0235999418364, -3358.8890067038628),
 *      new GeometryPoint(5075.3145648369318, -3378.0037556404409),
 *      new GeometryPoint(5305.19551436013, -3376.9669111768926)],
 * var linearRing = new GeometryLinearRing(points);
 * @usage
 */
class LinearRing_LinearRing extends LineString {


    constructor(points) {
        super(points);
        /**
         * @member {Array.<string>} [GeometryLinearRing.prototype.componentTypes=["SuperMap.Geometry.Point"]]
         * @description components 存储的几何对象所支持的几何类型数组，为空表示类型不受限制。
         * @readonly
         */
        this.componentTypes = ["SuperMap.Geometry.Point"];
        this.CLASS_NAME = "SuperMap.Geometry.LinearRing";
        this.geometryType = "LinearRing";
    }

    /**
     * @function GeometryLinearRing.prototype.addComponent
     * @description 添加一个点到几何图形数组中，如果这个点将要被添加到组件数组的末端，并且与数组中已经存在的最后一个点相同，
     * 重复的点是不能被添加的。这将影响未关闭环的关闭。
     * 这个方法可以通过将非空索引（组件数组的下标）作为第二个参数重写。
     * @param {GeometryPoint} point - 点对象。
     * @param {number} [index] - 插入组件数组的下标。
     * @returns {boolean} 点对象是否添加成功。
     */
    addComponent(point, index) {
        var added = false;

        //remove last point
        var lastPoint = this.components.pop();

        // given an index, add the point
        // without an index only add non-duplicate points
        if (index != null || !point.equals(lastPoint)) {
            added = super.addComponent.apply(this, arguments);
        }

        //append copy of first point
        var firstPoint = this.components[0];
        super.addComponent.apply(this, [firstPoint]);

        return added;
    }

    /**
     * @function GeometryLinearRing.prototype.removeComponent
     * @description 从几何组件中删除一个点。
     * @param {GeometryPoint} point - 点对象。
     * @returns {boolean} 点对象是否删除。
     */
    removeComponent(point) { // eslint-disable-line no-unused-vars
        var removed = this.components && (this.components.length > 3);
        if (removed) {
            //remove last point
            this.components.pop();

            //remove our point
            super.removeComponent.apply(this, arguments);
            //append copy of first point
            var firstPoint = this.components[0];
            super.addComponent.apply(this, [firstPoint]);
        }
        return removed;
    }

    /**
     * @function GeometryLinearRing.prototype.getArea
     * @description 获得当前几何对象区域大小，如果是沿顺时针方向的环则是正值，否则为负值。
     * @returns {number} 环的面积。
     */
    getArea() {
        var area = 0.0;
        if (this.components && (this.components.length > 2)) {
            var sum = 0.0;
            for (var i = 0, len = this.components.length; i < len - 1; i++) {
                var b = this.components[i];
                var c = this.components[i + 1];
                sum += (b.x + c.x) * (c.y - b.y);
            }
            area = -sum / 2.0;
        }
        return area;
    }

    /**
     * @function GeometryLinearRing.prototype.getVertices
     * @description 返回几何图形的所有点的列表。
     * @param {boolean} [nodes] - 对于线来说，仅仅返回作为端点的顶点，如果设为 false ，则返回非端点的顶点，如果没有设置此参数，则返回所有顶点。
     * @returns {Array} 几何对象所有点的列表。
     */
    getVertices(nodes) {
        return (nodes === true) ? [] : this.components.slice(0, this.components.length - 1);
    }


}

;// CONCATENATED MODULE: ./src/common/commontypes/geometry/MultiLineString.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class GeometryMultiLineString
 * @aliasclass Geometry.MultiLineString
 * @deprecatedclass SuperMap.Geometry.MultiLineString
 * @classdesc 几何对象多线类。
 * @category BaseTypes Geometry
 * @extends GeometryCollection
 * @param {Array.<GeometryLineString>} components - GeometryLineString 数组。
 * @example
 * var multi = new GeometryMultiLineString([
 *      new GeometryLineString([
 *          new GeometryPoint(1, 0),
 *          new GeometryPoint(0, 1)
 *      ])
 *  ]);
 * @usage
 */
class MultiLineString extends Collection {


    constructor(components) {
        super(components);
        /**
         * @member {Array.<string>} [GeometryMultiLineString.prototype.componentTypes=["SuperMap.Geometry.LineString"]]
         * @description components 存储的几何对象所支持的几何类型数组。
         * @readonly
         */
        this.componentTypes = ["SuperMap.Geometry.LineString"];
        this.CLASS_NAME = "SuperMap.Geometry.MultiLineString";
        this.geometryType = "MultiLineString";
    }


}

;// CONCATENATED MODULE: ./src/common/commontypes/geometry/MultiPolygon.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class GeometryMultiPolygon
 * @aliasclass Geometry.MultiPolygon
 * @deprecatedclass SuperMap.Geometry.MultiPolygon
 * @classdesc 几何对象多多边形类。
 * @category BaseTypes Geometry
 * @extends GeometryCollection
 * @param  {Array.<GeometryPolygon>} components - 形成 GeometryMultiPolygon 的多边形数组。
 * @example
 * var points1 = [new GeometryPoint(10,10),new GeometryPoint(0,0)];
 * var points2 = [new GeometryPoint(10,10),new GeometryPoint(0,0),new GeometryPoint(3,3),new GeometryPoint(10,10)];
 *
 * var linearRing1 = new GeometryLinearRing(points1);
 * var linearRing2 = new GeometryLinearRing(points2);
 *
 * var polygon1 = new GeometryPolygon([linearRing1]);
 * var polygon2 = new GeometryPolygon([linearRing2]);
 *
 * var multiPolygon1 = new GeometryMultiPolygon([polygon1,polygon2]);
 * @usage
 */
class MultiPolygon extends Collection {


    constructor(components) {
        super(components);
        /**
         * @member {Array.<string>} [GeometryMultiPolygon.prototype.componentTypes=["SuperMap.Geometry.Polygon"]]
         * @description components 存储的几何对象所支持的几何类型数组。
         * @readonly
         */
        this.componentTypes = ["SuperMap.Geometry.Polygon"];
        this.CLASS_NAME = "SuperMap.Geometry.MultiPolygon";
        this.geometryType = "MultiPolygon";
    }


}

;// CONCATENATED MODULE: ./src/common/commontypes/geometry/Polygon.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class GeometryPolygon
 * @aliasclass Geometry.Polygon
 * @deprecatedclass SuperMap.Geometry.Polygon
 * @classdesc  多边形几何对象类。
 * @category BaseTypes Geometry
 * @extends GeometryCollection
 * @param {Array.<GeometryLinearRing>} components - 多边形的线环数组。
 * @example
 * var points =[new GeometryPoint(0,4010338),
 *      new GeometryPoint(1063524,4010338),
 *      new GeometryPoint(1063524,3150322),
 *      new GeometryPoint(0,3150322)
 *  ],
 *  var linearRings = new GeometryLinearRing(points),
 *  var  region = new GeometryPolygon([linearRings]);
 * @usage
 */
class Polygon_Polygon extends Collection {


    constructor(components) {
        super(components);
        /**
         * @member {Array.<string>} [GeometryPolygon.prototype.componentTypes=["SuperMap.Geometry.LinearRing"]]
         * @description components 存储的几何对象所支持的几何类型数组。
         * @readonly
         */
        this.componentTypes = ["SuperMap.Geometry.LinearRing"];
        this.CLASS_NAME = "SuperMap.Geometry.Polygon";
        this.geometryType = "Polygon";
    }

    /**
     * @function GeometryPolygon.prototype.getArea
     * @description 获得区域面积，从区域的外部口径减去计此区域内部口径算所得的面积。
     * @returns {number} 几何对象的面积。
     */
    getArea() {
        var area = 0.0;
        if (this.components && (this.components.length > 0)) {
            area += Math.abs(this.components[0].getArea());
            for (var i = 1, len = this.components.length; i < len; i++) {
                area -= Math.abs(this.components[i].getArea());
            }
        }
        return area;
    }


}

;// CONCATENATED MODULE: ./src/common/commontypes/geometry/Rectangle.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class  GeometryRectangle
 * @aliasclass Geometry.Rectangle
 * @deprecatedclass SuperMap.Geometry.Rectangle
 * @classdesc 矩形几何对象类。
 * @category BaseTypes Geometry
 * @param {number} x - 矩形左下角点的横坐标。
 * @param {number} y - 矩形左下角点的纵坐标。
 * @param {number} width - 矩形的宽度。
 * @param {number} height -  矩形的高度。
 * @extends {Geometry}
 * @example
 *  //x 为矩形左下角点的横坐标；y 为矩形左下角点的纵坐标；w 为矩形的宽度；h 为矩形的高度
 *  var x = 1;
 *  var y = 2;
 *  var w = 10;
 *  var h = 20;
 *  var recttangle = new GeometryRectangle(x, y, w, h);
 * @usage
 */

class Rectangle extends Geometry_Geometry {


    constructor(x, y, width, height) {
        super(x, y, width, height);
        /**
         * @member {number} GeometryRectangle.prototype.x
         * @description 矩形左下角点的横坐标。
         */
        this.x = x;

        /**
         * @member {number} GeometryRectangle.prototype.y
         * @description 矩形左下角点的纵坐标。
         */
        this.y = y;

        /**
         * @member {number} GeometryRectangle.prototype.width
         * @description 矩形的宽度。
         */
        this.width = width;

        /**
         * @member {number} GeometryRectangle.prototype.height
         * @description 矩形的高度。
         */
        this.height = height;

        this.CLASS_NAME = "SuperMap.Geometry.Rectangle";
        this.geometryType = "Rectangle";
    }

    /**
     * @function GeometryRectangle.prototype.calculateBounds
     * @description 计算出此矩形对象的 bounds。
     */
    calculateBounds() {
        this.bounds = new Bounds(this.x, this.y,
            this.x + this.width,
            this.y + this.height);
    }


    /**
     * @function GeometryRectangle.prototype.getArea
     * @description 获取矩形对象的面积。
     * @returns {number} 矩形对象面积。
     */
    getArea() {
        var area = this.width * this.height;
        return area;
    }


}

;// CONCATENATED MODULE: ./src/common/commontypes/geometry/index.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/























;// CONCATENATED MODULE: ./src/common/commontypes/Credential.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/

/**
 * @class Credential
 * @deprecatedclass SuperMap.Credential
 * @category Security
 * @classdesc SuperMap 的安全证书类，其中包括 token 等安全验证信息。</br>
 * 需要使用用户名和密码在："http://localhost:8090/iserver/services/security/tokens" 下申请 value。</br>
 * 获得形如："2OMwGmcNlrP2ixqv1Mk4BuQMybOGfLOrljruX6VcYMDQKc58Sl9nMHsqQaqeBx44jRvKSjkmpZKK1L596y7skQ.." 的 value。</br>
 * 目前支持的功能包括：地图服务、专题图、量算、查询、公交换乘、空间分析、网络分析，不支持轮询功能。</br>
 * @param {string} value - 访问受安全限制的服务时用于通过安全认证的验证信息。
 * @param {string} [name='token'] - 验证信息前缀，name=value 部分的 name 部分。
 * @example
 * var pixcel = new Credential("valueString","token");
 * pixcel.destroy();
 * @usage
 */
class Credential {

    constructor(value, name) {

        /**
         * @member {string} Credential.prototype.value
         * @description 访问受安全限制的服务时用于通过安全认证的验证信息。
         */
        this.value = value ? value : "";

        /**
         * @member {string} [Credential.prototype.name='token']
         * @description 验证信息前缀，name=value 部分的 name 部分。
         */
        this.name = name ? name : "token";
        this.CLASS_NAME = "SuperMap.Credential";
    }

    /**
     * @function Credential.prototype.getUrlParameters
     * @description 获取 name=value 的表达式。
     * @example
     * var credential = new Credential("valueString","token");
     * //这里 str = "token=valueString";
     * var str = credential.getUrlParameters();
     * @returns {string} 安全信息组成的 url 片段。
     */
    getUrlParameters() {
        //当需要其他安全信息的时候，则需要return this.name + "=" + this.value + "&" + "...";的形式添加。
        return this.name + "=" + this.value;
    }


    /**
     * @function Credential.prototype.getValue
     * @description 获取 value。
     * @example
     * var credential = new Credential("2OMwGmcNlrP2ixqv1Mk4BuQMybOGfLOrljruX6VcYMDQKc58Sl9nMHsqQaqeBx44jRvKSjkmpZKK1L596y7skQ..","token");
     * //这里 str = "2OMwGmcNlrP2ixqv1Mk4BuQMybOGfLOrljruX6VcYMDQKc58Sl9nMHsqQaqeBx44jRvKSjkmpZKK1L596y7skQ..";
     * var str = credential.getValue();
     * @returns {string} value 字符串，在 iServer 服务下该 value 值即为 token 值。
     */
    getValue() {
        return this.value;
    }

    /**
     *
     * @function Credential.prototype.destroy
     * @description 销毁此对象。销毁后此对象的所有属性为 null，而不是初始值。
     * @example
     * var credential = new Credential("valueString","token");
     * credential.destroy();
     */
    destroy() {
        this.value = null;
        this.name = null;
    }

}

/**
 * @member {Credential} Credential.CREDENTIAL
 * @description 这个对象保存一个安全类的实例，在服务端需要安全验证的时候必须进行设置。
 * @example
 * 代码实例:
 *  // 当iServer启用服务安全的时候，下边的代码是必须的。安全证书类能够接收一个value和一个name参数。
 *  var value = "(以iServer为例，这里是申请的token值)";
 *  var name = "token";
 *  // 默认name参数为token，所以当使用iServer服务的时候可以不进行设置。
 *  Credential.CREDENTIAL = new Credential(value, name);
 *
 */

 Credential.CREDENTIAL = null;

;// CONCATENATED MODULE: ./src/common/commontypes/Date.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/
/**
 * @name Date
 * @namespace
 * @category BaseTypes Util
 * @description 包含 parse、toISOString 方法的实现，两个方法用来解析 RFC 3339 日期，遵循 ECMAScript 5 规范。
 * @private
 */
var DateExt = {

    /**
     * @description 生成代表一个具体的日期字符串，该日期遵循 ISO 8601 标准（详情查看{@link http://tools.ietf.org/html/rfc3339}）。
     * @example
     *  var dateString = DateExt.toISOString(new Date());
     * @param {Date} date - 日期对象。
     * @returns {string} 一个代表日期的字符串。（例如 "2010-08-07T16:58:23.123Z"）。
     */
    toISOString: (function () {
        //标准的Date会存在toISOString方法，可以直接调用
        if ("toISOString" in Date.prototype) {
            return function (date) {
                return date.toISOString();
            };
        } else {// 部分浏览器没有，就得自己组合，组合后的字符串规则不变
            function pad(num, len) {
                var str = num + "";
                while (str.length < len) {
                    str = "0" + str;
                }
                return str;
            }

            return function (date) {
                var str;
                if (isNaN(date.getTime())) {
                    // ECMA-262 says throw RangeError, Firefox returns
                    // "Invalid Date"
                    str = "Invalid Date";
                } else {
                    str =
                        date.getUTCFullYear() + "-" +
                        pad(date.getUTCMonth() + 1, 2) + "-" +
                        pad(date.getUTCDate(), 2) + "T" +
                        pad(date.getUTCHours(), 2) + ":" +
                        pad(date.getUTCMinutes(), 2) + ":" +
                        pad(date.getUTCSeconds(), 2) + "." +
                        pad(date.getUTCMilliseconds(), 3) + "Z";
                }
                return str;
            };
        }

    })(),

    /**
     * @description 从一个字符串生成一个日期对象。
     * @example
     *  var date = DateExt.parse("2010-08-07");
     * @param {string} str - 日期的字符串。（例如： "2010", "2010-08", "2010-08-07", "2010-08-07T16:58:23.123Z","2010-08-07T11:58:23.123-06"）。
     * @returns {Date} 日期对象，如果字符串无法被解析，则返回一个无效的日期。(例如 isNaN(date.getTime()))。
     */
    parse: function (str) {
        var date;
        var match = str.match(/^(?:(\d{4})(?:-(\d{2})(?:-(\d{2}))?)?)?(?:(?:T(\d{1,2}):(\d{2}):(\d{2}(?:\.\d+)?)(Z|(?:[+-]\d{1,2}(?::(\d{2}))?)))|Z)?$/);
        if (match && (match[1] || match[7])) { // must have at least year or time
            var year = parseInt(match[1], 10) || 0;
            var month = (parseInt(match[2], 10) - 1) || 0;
            var day = parseInt(match[3], 10) || 1;
            date = new Date(Date.UTC(year, month, day));
            // optional time
            var type = match[7];
            if (type) {
                var hours = parseInt(match[4], 10);
                var minutes = parseInt(match[5], 10);
                var secFrac = parseFloat(match[6]);
                var seconds = secFrac | 0;
                var milliseconds = Math.round(1000 * (secFrac - seconds));
                date.setUTCHours(hours, minutes, seconds, milliseconds);
                // check offset
                if (type !== "Z") {
                    var hoursOffset = parseInt(type, 10);
                    var minutesOffset = parseInt(match[8], 10) || 0;
                    var offset = -1000 * (60 * (hoursOffset * 60) + minutesOffset * 60);
                    date = new Date(date.getTime() + offset);
                }
            }
        } else {
            date = new Date("invalid");
        }
        return date;
    }
};

;// CONCATENATED MODULE: ./src/common/commontypes/Event.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/
 

 /**
  * @name Event
  * @namespace
  * @category BaseTypes Events
  * @description 事件处理函数.
  * @usage
  * ```
  * // 浏览器
  * <script type="text/javascript" src="{cdn}"></script>
  * <script>
  *   const element = {namespace}.Event.element();
  *
  *   // 弃用的写法
  *   const result = SuperMap.Event.element();
  *
  * </script>
  *
  * // ES6 Import
  * import { Event } from '{npm}';
  * 
  * const result = Event.element();
  * ```
  */
 var Event = {

     /**
      * @description  事件观察者列表。
      * @type {Object}
      * @default false
      */
     observers: false,

     /**
      * @description KEY_SPACE
      * @type {number}
      * @default 32
      */
     KEY_SPACE: 32,

     /**
      * @description KEY_BACKSPACE
      * @type {number}
      * @default 8
      */
     KEY_BACKSPACE: 8,

     /**
      * @description KEY_TAB
      * @type {number}
      * @default 9
      */
     KEY_TAB: 9,

     /**
      * @description KEY_RETURN
      * @type {number}
      * @default 13
      */
     KEY_RETURN: 13,

     /**
      * @description KEY_ESC
      * @type {number}
      * @default 27
      */
     KEY_ESC: 27,

     /**
      * @description KEY_LEFT
      * @type {number}
      * @default 37
      */
     KEY_LEFT: 37,

     /**
      * @description KEY_UP
      * @type {number}
      * @default 38
      */
     KEY_UP: 38,

     /**
      * @description KEY_RIGHT
      * @type {number}
      * @default 39
      */
     KEY_RIGHT: 39,

     /**
      * @description KEY_DOWN
      * @type {number}
      * @default 40
      */
     KEY_DOWN: 40,

     /**
      * @description KEY_DELETE
      * @type {number}
      * @default 46
      */
     KEY_DELETE: 46,


     /**
      * @description 监听浏览器 DOM 事件。
      * @param {Event} event - Event 对象。
      * @returns {HTMLElement} 触发事件的 DOM 元素。
      */
     element: function (event) {
         return event.target || event.srcElement;
     },

     /**
      * @description 判断事件是否由单次触摸引起。
      * @param {Event} event - Event 对象。
      * @returns {boolean} 是否有且只有一个当前在与触摸表面接触的 Touch 对象。
      */
     isSingleTouch: function (event) {
         return event.touches && event.touches.length === 1;
     },

     /**
      * @description 判断事件是否由多点触控引起。
      * @param {Event} event - Event 对象。
      * @returns {boolean} 是否存在多个当前在与触摸表面接触的 Touch 对象。
      */
     isMultiTouch: function (event) {
         return event.touches && event.touches.length > 1;
     },

     /**
      * @description 确定事件是否由左键单击引起。
      * @param {Event} event - Event 对象。
      * @returns {boolean} 是否点击鼠标左键。
      */
     isLeftClick: function (event) {
         return (((event.which) && (event.which === 1)) ||
             ((event.button) && (event.button === 1)));
     },

     /**
      * @description 确定事件是否由鼠标右键单击引起。
      * @param {Event} event - Event 对象。
      * @returns {boolean} 是否点击鼠标右键。
      */
     isRightClick: function (event) {
         return (((event.which) && (event.which === 3)) ||
             ((event.button) && (event.button === 2)));
     },

     /**
      * @description 阻止事件冒泡。
      * @param {Event} event - Event 对象。
      * @param {boolean} allowDefault - 默认为 false，表示阻止事件的默认行为。
      */
     stop: function (event, allowDefault) {

         if (!allowDefault) {
             if (event.preventDefault) {
                 event.preventDefault();
             } else {
                 event.returnValue = false;
             }
         }

         if (event.stopPropagation) {
             event.stopPropagation();
         } else {
             event.cancelBubble = true;
         }
     },

     /**
      * @description 查询触发指定事件的 DOM 元素。
      * @param {Event} event - Event 对象。
      * @param {string} tagName - html 标签名。
      * @returns {HTMLElement} DOM 元素。
      */
     findElement: function (event, tagName) {
         var element = Event.element(event);
         while (element.parentNode && (!element.tagName ||
             (element.tagName.toUpperCase() != tagName.toUpperCase()))) {
             element = element.parentNode;
         }
         return element;
     },

     /**
      * @description 监听事件，注册事件处理方法。
      * @param {(HTMLElement|string)} elementParam - 待监听的 DOM 对象或者其 ID 标识。
      * @param {string} name - 监听事件的类别名称。
      * @param {function} observer - 注册的事件处理方法。
      * @param {boolean} [useCapture=false] - 是否捕获。
      */
     observe: function (elementParam, name, observer, useCapture) {
         var element = Util_Util.getElement(elementParam);
         useCapture = useCapture || false;

         if (name === 'keypress' &&
             (navigator.appVersion.match(/Konqueror|Safari|KHTML/)
                 || element.attachEvent)) {
             name = 'keydown';
         }

         //if observers cache has not yet been created, create it
         if (!this.observers) {
             this.observers = {};
         }

         //if not already assigned, make a new unique cache ID
         if (!element._eventCacheID) {
             var idPrefix = "eventCacheID_";
             if (element.id) {
                 idPrefix = element.id + "_" + idPrefix;
             }
             element._eventCacheID = Util_Util.createUniqueID(idPrefix);
         }

         var cacheID = element._eventCacheID;

         //if there is not yet a hash entry for this element, add one
         if (!this.observers[cacheID]) {
             this.observers[cacheID] = [];
         }

         //add a new observer to this element's list
         this.observers[cacheID].push({
             'element': element,
             'name': name,
             'observer': observer,
             'useCapture': useCapture
         });

         //add the actual browser event listener
         if (element.addEventListener) {
             if(name === 'mousewheel'){
                 // https://www.chromestatus.com/features/6662647093133312
                 element.addEventListener(name, observer, {useCapture: useCapture, passive: false} );
             } else {
                 element.addEventListener(name, observer, useCapture);
             }
         } else if (element.attachEvent) {
             element.attachEvent('on' + name, observer);
         }
     },

     /**
      * @description 移除给定 DOM 元素的监听事件。
      * @param {(HTMLElement|string)} elementParam - 待监听的 DOM 对象或者其 ID 标识。
      */
     stopObservingElement: function (elementParam) {
         var element = Util_Util.getElement(elementParam);
         var cacheID = element._eventCacheID;
         this._removeElementObservers(Event.observers[cacheID]);
     },
     _removeElementObservers: function (elementObservers) {
         if (elementObservers) {
             for (var i = elementObservers.length - 1; i >= 0; i--) {
                 var entry = elementObservers[i];
                 var args = new Array(entry.element, entry.name, entry.observer, entry.useCapture);
                 Event.stopObserving.apply(this, args);
             }
         }
     },

     /**
      * @description 移除事件监听和注册的事件处理方法。注意：事件的移除和监听相对应，移除时的各属性信息必须监听时保持一致才能确保事件移除成功。
      * @param {(HTMLElement|string)} elementParam - 被监听的 DOM 元素或者其 ID。
      * @param {string} name - 需要移除的被监听事件名称。
      * @param {function} observer - 需要移除的事件处理方法。
      * @param {boolean} [useCapture=false] - 是否捕获。
      * @returns {boolean} 监听事件是否被移除。
      */
     stopObserving: function (elementParam, name, observer, useCapture) {
         useCapture = useCapture || false;

         var element = Util_Util.getElement(elementParam);
         var cacheID = element._eventCacheID;

         if (name === 'keypress') {
             if (navigator.appVersion.match(/Konqueror|Safari|KHTML/) ||
                 element.detachEvent) {
                 name = 'keydown';
             }
         }

         // find element's entry in this.observers cache and remove it
         var foundEntry = false;
         var elementObservers = Event.observers[cacheID];
         if (elementObservers) {

             // find the specific event type in the element's list
             var i = 0;
             while (!foundEntry && i < elementObservers.length) {
                 var cacheEntry = elementObservers[i];

                 if ((cacheEntry.name === name) &&
                     (cacheEntry.observer === observer) &&
                     (cacheEntry.useCapture === useCapture)) {

                     elementObservers.splice(i, 1);
                     if (elementObservers.length == 0) {
                         delete Event.observers[cacheID];
                     }
                     foundEntry = true;
                     break;
                 }
                 i++;
             }
         }

         //actually remove the event listener from browser
         if (foundEntry) {
             if (element.removeEventListener) {
                 element.removeEventListener(name, observer, useCapture);
             } else if (element && element.detachEvent) {
                 element.detachEvent('on' + name, observer);
             }
         }
         return foundEntry;
     },

     /**
      * @description 移除缓存中的监听事件。
      */
     unloadCache: function () {
         // check for Event before checking for observers, because
         // Event may be undefined in IE if no map instance was
         // created
         if (Event && Event.observers) {
             for (var cacheID in Event.observers) {
                 var elementObservers = Event.observers[cacheID];
                 Event._removeElementObservers.apply(this,
                     [elementObservers]);
             }
             Event.observers = false;
         }
     },

     CLASS_NAME: "SuperMap.Event"
 };
 /* prevent memory leaks in IE */
 Event.observe(window, 'resize', Event.unloadCache, false);

;// CONCATENATED MODULE: ./src/common/commontypes/Events.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class Events
 * @deprecatedclass SuperMap.Events
 * @classdesc 事件类。
 * @category BaseTypes Events
 * @param {Object} object - 当前事件对象被添加到的 JS 对象。
 * @param {HTMLElement} element - 响应浏览器事件的 DOM 元素。
 * @param {Array.<string>} eventTypes - 自定义应用事件的数组。
 * @param {boolean} [fallThrough=false] - 是否允许事件处理之后向上传递（冒泡），为 false 的时候阻止事件冒泡。
 * @param {Object} options - 事件对象选项。
 * @usage
 */
class Events {

    constructor(object, element, eventTypes, fallThrough, options) {
        /**
         * @member {Array.<string>} Events.prototype.BROWSER_EVENTS
         * @description 支持的事件。
         * @constant
         * @default [
         "mouseover", "mouseout","mousedown", "mouseup", "mousemove",
         "click", "dblclick", "rightclick", "dblrightclick","resize",
         "focus", "blur","touchstart", "touchmove", "touchend","keydown",
         "MSPointerDown", "MSPointerUp", "pointerdown", "pointerup",
         "MSGestureStart", "MSGestureChange", "MSGestureEnd","contextmenu"
         ]
         */
        this.BROWSER_EVENTS = [
            "mouseover", "mouseout",
            "mousedown", "mouseup", "mousemove",
            "click", "dblclick", "rightclick", "dblrightclick",
            "resize", "focus", "blur",
            "touchstart", "touchmove", "touchend",
            "keydown", "MSPointerDown", "MSPointerUp", "pointerdown", "pointerup",
            "MSGestureStart", "MSGestureChange", "MSGestureEnd",
            "contextmenu"
        ];

        /**
         * @member {Object} Events.prototype.listeners
         * @description 事件监听器函数。
         */
        this.listeners = {};

        /**
         * @member {Object} Events.prototype.object
         * @description  发布应用程序事件的对象。
         */
        this.object = object;

        /**
         * @member {HTMLElement} Events.prototype.element
         * @description 接受浏览器事件的 DOM 节点。
         */
        this.element = null;

        /**
         * @member {Array.<string>} Events.prototype.eventTypes
         * @description 支持的事件类型列表。
         */
        this.eventTypes = [];

        /**
         * @member {function} Events.prototype.eventHandler
         * @description 绑定在元素上的事件处理器对象。
         */
        this.eventHandler = null;

        /**
         * @member {boolean} [Events.prototype.fallThrough=false]
         * @description 是否允许事件处理之后向上传递（冒泡），为 false 的时候阻止事件冒泡。
         */
        this.fallThrough = fallThrough;

        /**
         * @member {boolean} [Events.prototype.includeXY=false]
         * @description 判断是否让 xy 属性自动创建到浏览器上的鼠标事件，一般设置为 false，如果设置为 true，鼠标事件将会在事件传递过程中自动产生 xy 属性。可根据事件对象的 'evt.object' 属性在相关的事件句柄上调用 getMousePosition 函数。这个选项习惯默认为 false 的原因在于，当创建一个事件对象，其主要目的是管理。在一个 div 的相对定位的鼠标事件，将其设为 true 也是有意义的。这个选项也可以用来控制是否抵消缓存。如果设为 false 不抵消，如果设为 true，用 this.clearMouseCache() 清除缓存偏移（边界元素偏移，元素在页面的位置偏移）。
         * @example
         *  function named(evt) {
         *        this.xy = this.object.events.getMousePosition(evt);
         *  }
         */
        this.includeXY = false;

        /**
         * @member {Object} Events.prototype.extensions
         * @description 事件扩展。Keys 代表事件类型，values 代表事件对象。
         */
        this.extensions = {};

        /**
         * @member {Object} Events.prototype.extensionCount
         * @description 事件扩展数量。
         */
        this.extensionCount = {};
        /**
         * @member {Object} Events.prototype.clearMouseListener
         * @description 待移除的鼠标监听事件。
         */
        this.clearMouseListener = null;

        Util_Util.extend(this, options);

        if (eventTypes != null) {
            for (var i = 0, len = eventTypes.length; i < len; i++) {
                this.addEventType(eventTypes[i]);
            }
        }

        if (element != null) {
            this.attachToElement(element);
        }

        this.CLASS_NAME = "SuperMap.Events";
    }

    /**
     * @function Events.prototype.destroy
     * @description 移除当前要素 element 上的所有事件监听和处理。
     */
    destroy() {
        for (var e in this.extensions) {
            if (typeof this.extensions[e] !== "boolean") {
                this.extensions[e].destroy();
            }
        }
        this.extensions = null;
        if (this.element) {
            Event.stopObservingElement(this.element);
            if (this.element.hasScrollEvent) {
                Event.stopObserving(
                    window, "scroll", this.clearMouseListener
                );
            }
        }
        this.element = null;

        this.listeners = null;
        this.object = null;
        this.eventTypes = null;
        this.fallThrough = null;
        this.eventHandler = null;
    }

    /**
     * @function Events.prototype.addEventType
     * @description 在此事件对象中添加新的事件类型，如果这个事件类型已经添加过了，则不做任何事情。
     * @param {string} eventName - 事件名。
     */
    addEventType(eventName) {
        if (!this.listeners[eventName]) {
            this.eventTypes.push(eventName);
            this.listeners[eventName] = [];
        }
    }

    /**
     * @function Events.prototype.attachToElement
     * @description 给 DOM 元素绑定浏览器事件。
     * @param {HTMLElement} element - 绑定浏览器事件的 DOM 元素。
     */
    attachToElement(element) {
        if (this.element) {
            Event.stopObservingElement(this.element);
        } else {
            // keep a bound copy of handleBrowserEvent() so that we can
            // pass the same function to both Event.observe() and .stopObserving()
            this.eventHandler = FunctionExt.bindAsEventListener(
                this.handleBrowserEvent, this
            );

            // to be used with observe and stopObserving
            this.clearMouseListener = FunctionExt.bind(
                this.clearMouseCache, this
            );
        }
        this.element = element;
        for (var i = 0, len = this.BROWSER_EVENTS.length; i < len; i++) {
            var eventType = this.BROWSER_EVENTS[i];

            // every browser event has a corresponding application event
            // (whether it's listened for or not).
            this.addEventType(eventType);

            // use Prototype to register the event cross-browser
            Event.observe(element, eventType, this.eventHandler);
        }
        // disable dragstart in IE so that mousedown/move/up works normally
        Event.observe(element, "dragstart", Event.stop);
    }


    /**
     * @function Events.prototype.on
     * @description 在一个相同的范围内注册监听器的方法，此方法调用 register 函数。
     * @example
     * // 注册一个 "loadstart" 监听事件
     * events.on({"loadstart": loadStartListener});
     *
     * // 同样注册一个 "loadstart" 监听事件
     * events.register("loadstart", undefined, loadStartListener);
     *
     * // 同时为对象注册多个监听事件
     * events.on({
     *     "loadstart": loadStartListener,
     *     "loadend": loadEndListener,
     *     scope: object
     * });
     *
     * // 同时为对象注册多个监听事件，多次调用 register 方法
     * events.register("loadstart", object, loadStartListener);
     * events.register("loadend", object, loadEndListener);
     *
     *
     * @param {Object} object - 添加监听的对象。
     */
    on(object) {
        for (var type in object) {
            if (type !== "scope" && object.hasOwnProperty(type)) {
                this.register(type, object.scope, object[type]);
            }
        }
    }


    /**
     * @function Events.prototype.register
     * @description 在事件对象上注册一个事件。当事件被触发时，'func' 函数被调用，假设我们触发一个事件，
     *              指定 Bounds 作为 "obj"，当事件被触发时，回调函数的上下文作为 Bounds 对象。
     * @param {string} type - 事件注册者的名字。
     * @param {Object} [obj=this.object] - 对象绑定的回调。
     * @param {function} [func] - 回调函数，如果没有特定的回调，则这个函数不做任何事情。
     * @param {(boolean|Object)} [priority] - 当为 true 时将新的监听加在事件队列的前面。
     */
    register(type, obj, func, priority) {
        if (type in Events && !this.extensions[type]) {
            this.extensions[type] = new Events[type](this);
        }
        if ((func != null) &&
            (Util_Util.indexOf(this.eventTypes, type) !== -1)) {

            if (obj == null) {
                obj = this.object;
            }
            var listeners = this.listeners[type];
            if (!listeners) {
                listeners = [];
                this.listeners[type] = listeners;
                this.extensionCount[type] = 0;
            }
            var listener = {obj: obj, func: func};
            if (priority) {
                listeners.splice(this.extensionCount[type], 0, listener);
                if (typeof priority === "object" && priority.extension) {
                    this.extensionCount[type]++;
                }
            } else {
                listeners.push(listener);
            }
        }
    }

    /**
     * @function Events.prototype.registerPriority
     * @description 相同的注册方法，但是在前面增加新的监听者事件查询而代替到方法的结束。
     * @param {string} type - 事件注册者的名字。
     * @param {Object} [obj=this.object] - 对象绑定的回调。
     * @param {function} [func] - 回调函数，如果没有特定的回调，则这个函数不做任何事情。
     */
    registerPriority(type, obj, func) {
        this.register(type, obj, func, true);
    }


    /**
     * @function Events.prototype.un
     * @description 在一个相同的范围内取消注册监听器的方法，此方法调用 unregister 函数。
     * @example
     * // 移除 "loadstart" 事件监听
     * events.un({"loadstart": loadStartListener});
     *
     * // 使用 "unregister" 方法移除 "loadstart" 事件监听
     * events.unregister("loadstart", undefined, loadStartListener);
     *
     * // 取消对象多个事件监听
     * events.un({
     *     "loadstart": loadStartListener,
     *     "loadend": loadEndListener,
     *     scope: object
     * });
     *
     * // 取消对象多个事件监听，多次调用unregister方法。
     * events.unregister("loadstart", object, loadStartListener);
     * events.unregister("loadend", object, loadEndListener);
     *
     * @param {Object} object - 移除监听的对象。
     */
    un(object) {
        for (var type in object) {
            if (type !== "scope" && object.hasOwnProperty(type)) {
                this.unregister(type, object.scope, object[type]);
            }
        }
    }

    /**
     * @function Events.prototype.unregister
     * @description 取消注册。
     * @param {string} type - 事件类型。
     * @param {Object} [obj=this.object] - 对象绑定的回调。
     * @param {function} [func] - 回调函数，如果没有特定的回调，则这个函数不做任何事情。
     */
    unregister(type, obj, func) {
        if (obj == null) {
            obj = this.object;
        }
        var listeners = this.listeners[type];
        if (listeners != null) {
            for (var i = 0, len = listeners.length; i < len; i++) {
                if (listeners[i].obj === obj && listeners[i].func === func) {
                    listeners.splice(i, 1);
                    break;
                }
            }
        }
    }


    /**
     * @function Events.prototype.remove
     * @description 删除某个事件类型的所有监听，如果该事件类型没有注册，则不做任何操作。
     * @param {string} type - 事件类型。
     */
    remove(type) {
        if (this.listeners[type] != null) {
            this.listeners[type] = [];
        }
    }

    /**
     * @function Events.prototype.triggerEvent
     * @description 触发一个特定的注册事件。
     * @param {string} type - 触发事件类型。
     * @param {Event} evt - 事件对象。
     * @returns {Event|boolean} 监听对象，如果返回是 false，则停止监听。
     */
    triggerEvent(type, evt) {
        var listeners = this.listeners[type];

        // fast path
        if (!listeners || listeners.length == 0) {
            return undefined;
        }

        // prep evt object with object & div references
        if (evt == null) {
            evt = {};
        }
        evt.object = this.object;
        evt.element = this.element;
        if (!evt.type) {
            evt.type = type;
        }

        // execute all callbacks registered for specified type
        // get a clone of the listeners array to
        // allow for splicing during callbacks
        listeners = listeners.slice();
        var continueChain;
        for (var i = 0, len = listeners.length; i < len; i++) {
            var callback = listeners[i];
            // bind the context to callback.obj
            continueChain = callback.func.apply(callback.obj, [evt]);

            if ((continueChain != undefined) && (continueChain === false)) {
                // if callback returns false, execute no more callbacks.
                break;
            }
        }
        // don't fall through to other DOM elements
        if (!this.fallThrough) {
            Event.stop(evt, true);
        }
        return continueChain;
    }


    /**
     * @function Events.prototype.handleBrowserEvent
     * @description 对 triggerEvent 函数的包装，给事件对象设置了 xy 属性（即当前鼠标点的 xy 坐标）。
     * @param {Event} evt - 事件对象。
     */
    handleBrowserEvent(evt) {
        var type = evt.type, listeners = this.listeners[type];
        if (!listeners || listeners.length == 0) {
            // noone's listening, bail out
            return;
        }
        // add clientX & clientY to all events - corresponds to average x, y
        var touches = evt.touches;
        if (touches && touches[0]) {
            var x = 0;
            var y = 0;
            var num = touches.length;
            var touch;
            for (var i = 0; i < num; ++i) {
                touch = touches[i];
                x += touch.clientX;
                y += touch.clientY;
            }
            evt.clientX = x / num;
            evt.clientY = y / num;
        }
        if (this.includeXY) {
            evt.xy = this.getMousePosition(evt);
        }
        this.triggerEvent(type, evt);
    }


    /**
     * @function Events.prototype.clearMouseCache
     * @description 清除鼠标缓存。
     */
    clearMouseCache() {
        this.element.scrolls = null;
        this.element.lefttop = null;
        var body = document.body;
        if (body && !((body.scrollTop != 0 || body.scrollLeft != 0) &&
                navigator.userAgent.match(/iPhone/i))) {
            this.element.offsets = null;
        }
    }

    /**
     * @function Events.prototype.getMousePosition
     * @description 获取当前鼠标的位置。
     * @param {Event} evt - 事件对象。
     * @returns {Pixel} 当前的鼠标的 xy 坐标点。
     */
    getMousePosition(evt) {
        if (!this.includeXY) {
            this.clearMouseCache();
        } else if (!this.element.hasScrollEvent) {
            Event.observe(window, "scroll", this.clearMouseListener);
            this.element.hasScrollEvent = true;
        }

        if (!this.element.scrolls) {
            var viewportElement = Util_Util.getViewportElement();
            this.element.scrolls = [
                viewportElement.scrollLeft,
                viewportElement.scrollTop
            ];
        }

        if (!this.element.lefttop) {
            this.element.lefttop = [
                (document.documentElement.clientLeft || 0),
                (document.documentElement.clientTop || 0)
            ];
        }

        if (!this.element.offsets) {
            this.element.offsets = Util_Util.pagePosition(this.element);
        }

        return new Pixel(
            (evt.clientX + this.element.scrolls[0]) - this.element.offsets[0]
            - this.element.lefttop[0],
            (evt.clientY + this.element.scrolls[1]) - this.element.offsets[1]
            - this.element.lefttop[1]
        );
    }

}
Events.prototype.BROWSER_EVENTS = [
  "mouseover", "mouseout",
  "mousedown", "mouseup", "mousemove",
  "click", "dblclick", "rightclick", "dblrightclick",
  "resize", "focus", "blur",
  "touchstart", "touchmove", "touchend",
  "keydown", "MSPointerDown", "MSPointerUp", "pointerdown", "pointerup",
  "MSGestureStart", "MSGestureChange", "MSGestureEnd",
  "contextmenu"
];

;// CONCATENATED MODULE: ./src/common/commontypes/Feature.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class Feature
 * @deprecatedclass SuperMap.Feature
 * @category BaseTypes Geometry
 * @classdesc 要素类组合了地理和属性，Feature 类同时具有 marker 和 lonlat 属性。
 * @param {SuperMap.Layer} layer - 图层。
 * @param {LonLat} lonlat - 经纬度。
 * @param {Object} data - 数据对象。
 * @usage
 */
class Feature_Feature {


    constructor(layer, lonlat, data) {
        this.CLASS_NAME = "SuperMap.Feature";
        /**
         * @deprecated
         * @member {SuperMap.Layer} Feature.prototype.layer
         * @description 图层。
         */
        this.layer = layer;

        /**
         * @member {string} Feature.prototype.id
         * @description 要素 ID。
         */
        this.id = Util_Util.createUniqueID(this.CLASS_NAME + "_");

        /**
         * @member {LonLat} Feature.prototype.lonlat
         * @description 经纬度。
         *
         */
        this.lonlat = lonlat;

        /**
         * @member {Object} Feature.prototype.data
         * @description 数据对象。
         */
        this.data = (data != null) ? data : {};

    }

    /**
     * @function Feature.prototype.destroy
     * @description 释放相关资源。
     */
    destroy() {
        this.id = null;
        this.lonlat = null;
        this.data = null;
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/ServerColor.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/

/**
 * @class ServerColor
 * @deprecatedclass SuperMap.ServerColor
 * @category iServer Map Theme
 * @classdesc 颜色类。该类使用三原色（ RGB ）来表达颜色。
 * @param {Object} options - 可选参数。
 * @param {number} [options.red=255] - 获取或设置红色值。
 * @param {number} [options.green=0] - 获取或设置绿色值。
 * @param {number} [options.blue=0] - 获取或设置蓝色值。
 * @usage
 */
class ServerColor {

    constructor(red, green, blue) {

        /**
         * @member {number} [ServerColor.prototype.red=255]
         * @description 获取或设置红色值。
         */
        this.red = (!red && red != 0)?255:red;

        /**
         * @member {number} [ServerColor.prototype.green=0]
         * @description 获取或设置绿色值。
         */
        this.green = green||0;

        /**
         * @member {number} [ServerColor.prototype.blue=0]
         * @description 获取或设置蓝色值。
         */
        this.blue = blue||0;

        this.CLASS_NAME = "SuperMap.ServerColor";
    }

    /**
     * @function ServerColor.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.red = null;
        me.green = null;
        me.blue = null;
    }


    /**
     * @function ServerColor.formJson
     * @description 将 JSON 对象转化为 ServerColor 对象。
     * @param {Object} jsonObject - 要转换的 JSON 对象。
     * @returns {ServerColor} 转化后的 ServerColor 对象。
     */
    static fromJson(jsonObject) {
        if (!jsonObject) {
            return;
        }
        var color = new ServerColor();
        var red = 255;
        if (jsonObject.red !== null) {
            red = Number(jsonObject.red);
        }
        color.red = red;

        var green = 0;
        if (jsonObject.green !== null) {
            green = Number(jsonObject.green);
        }
        color.green = green;

        var blue = 0;
        if (jsonObject.blue !== null) {
            blue = Number(jsonObject.blue);
        }
        color.blue = blue;
        return color;
    }

}



;// CONCATENATED MODULE: ./src/common/iServer/ServerStyle.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class ServerStyle
 * @deprecatedclass SuperMap.ServerStyle
 * @category  iServer Map Theme
 * @classdesc 服务端矢量要素风格类。该类用于定义点状符号、线状符号、填充符号风格及其相关属性。
 * @param {Object} options - 参数。
 * @param {FillGradientMode} options.fillGradientMode - 渐变填充风格的渐变类型。
 * @param {ServerColor} [options.fillBackColor=[255,255,255]] - 填充背景颜色。
 * @param {boolean} [options.fillBackOpaque=false] - 背景是否不透明。
 * @param {ServerColor} [options.fillForeColor=[255,0,0]] - 填充颜色。
 * @param {number} [options.fillGradientAngle=0] - 渐变填充的旋转角度。
 * @param {number} [options.fillGradientOffsetRatioX=0] - 渐变填充中心点相对于填充区域范围中心点的水平偏移百分比。
 * @param {number} [options.fillGradientOffsetRatioY=0] - 填充中心点相对于填充区域范围中心点的垂直偏移百分比。
 * @param {number} [options.fillOpaqueRate=100] - 填充不透明度。
 * @param {number} [options.fillSymbolID=0] - 填充符号的编码。
 * @param {ServerColor} [options.lineColor] - 矢量要素的边线颜色。默认 lineColor = new ServerColor(0, 0, 0)。
 * @param {number} [options.lineSymbolID=0] - 线状符号的编码。
 * @param {number} [options.lineWidth=1] - 边线的宽度。
 * @param {number} [options.markerAngle=0] - 点状符号的旋转角度。
 * @param {number} [options.markerSize=1] - 点状符号的大小。
 * @param {number} [options.markerSymbolID=-1] - 点状符号的编码。
 * @usage
 */
class ServerStyle {

    constructor(options) {
        /**
         * @member {ServerColor} ServerStyle.prototype.fillBackColor
         * @description 填充背景颜色。当填充模式为渐变填充时，该颜色为填充终止色。
         */
        this.fillBackColor = new ServerColor(255, 255, 255);

        /**
         * @member {boolean} [ServerStyle.prototype.fillBackOpaque=false]
         * @description 背景是否不透明。false 表示透明。
         */
        this.fillBackOpaque = false;

        /**
         * @member {ServerColor} ServerStyle.prototype.fillForeColor
         * @description 填充颜色。当填充模式为渐变填充时，该颜色为填充起始颜色。
         */
        this.fillForeColor = new ServerColor(255, 0, 0);

        /**
         * @member {FillGradientMode} ServerStyle.prototype.fillGradientMode
         * @description 渐变填充风格的渐变类型。
         */
        this.fillGradientMode = null;

        /**
         * @member {number} ServerStyle.prototype.fillGradientAngle -
         * @description 渐变填充的旋转角度。单位为度，精确到 0.1 度，逆时针方向为正方向。
         */
        this.fillGradientAngle = 0;

        /**
         * @member {number} ServerStyle.prototype.fillGradientOffsetRatioX
         * @description 渐变填充中心点相对于填充区域范围中心点的水平偏移百分比。它们的关系如下：设填充区域范围中心点的坐标为（x0, y0），
         *              填充中心点的坐标为（x, y），填充区域范围的宽度为 a，水平偏移百分比为 dx，则 x=x0 + a*dx/100。
         */
        this.fillGradientOffsetRatioX = 0;

        /**
         * @member {number} ServerStyle.prototype.fillGradientOffsetRatioY
         * @description 填充中心点相对于填充区域范围中心点的垂直偏移百分比。它们的关系如下：<br>
         *              设填充区域范围中心点的坐标为（x0, y0），填充中心点的坐标为（x, y），填充区域范围的高度为 b，垂直偏移百分比为 dy，则 y=y0 + b*dx/100。
         */
        this.fillGradientOffsetRatioY = 0;

        /**
         * @member {number} [ServerStyle.prototype.fillOpaqueRate=100]
         * @description 填充不透明度。合法值为 0 - 100 的数值。其中为 0 表示完全透明；
         *              100 表示完全不透明。赋值小于 0 时按照 0 处理，大于 100 时按照 100 处理。
         */
        this.fillOpaqueRate = 100;

        /**
         * @member {number} ServerStyle.prototype.fillSymbolID
         * @description 填充符号的编码。此编码用于唯一标识各普通填充风格的填充符号。
         *              关于填充符号的样式与对应的 ID 号请在 SuperMap 桌面软件中查找。
         */
        this.fillSymbolID = 0;

        /**
         * @member {ServerColor} ServerStyle.prototype.lineColor
         * @description 矢量要素的边线颜色。如果等级符号是点符号，点符号的颜色由 lineColor 控制。
         */
        this.lineColor = new ServerColor(0, 0, 0);

        /**
         * @member {number} [ServerStyle.prototype.lineSymbolID=0]
         * @description 线状符号的编码。此编码用于唯一标识各普通填充风格的填充符号。
         *              关于线状符号的样式与对应的 ID 号请在 SuperMap 桌面软件中查找。
         */
        this.lineSymbolID = 0;

        /**
         * @member {number} [ServerStyle.prototype.lineWidth=1.0]
         * @description 边线的宽度。单位为毫米，精度到 0.1。
         */
        this.lineWidth = 1;

        /**
         * @member {number} [ServerStyle.prototype.markerAngle=0]
         * @description 点状符号的旋转角度。以度为单位，精确到 0.1 度，逆时针方向为正方向。
         */
        this.markerAngle = 0;

        /**
         * @member {number} [ServerStyle.prototype.markerSize=1.0]
         * @description 点状符号的大小。单位为毫米，精度为 0.1。当该属性设置为0时，采用符号默认大小 1.0 显示。
         *              当该属性设置为非法值时，交由服务器默认处理。
         */
        this.markerSize = 1;

        /**
         * @member {number} [ServerStyle.prototype.markerSymbolID=-1]
         * @description 点状符号的编码。此编码用于唯一标识各点状符号。
         *              关于线状符号的样式与对应的 ID 号请在 SuperMap 桌面软件中查找。
         */
        this.markerSymbolID = -1;
        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.ServerStyle";
    }

    /**
     * @function ServerStyle.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        if (me.fillBackColor) {
            me.fillBackColor.destroy();
            me.fillBackColor = null;
        }
        me.fillBackOpaque = null;

        if (me.fillForeColor) {
            me.fillForeColor.destroy();
            me.fillForeColor = null;
        }
        me.fillGradientMode = null;
        me.fillGradientAngle = null;
        me.fillGradientOffsetRatioX = null;
        me.fillGradientOffsetRatioY = null;
        me.fillOpaqueRate = null;
        me.fillSymbolID = null;
        if (me.lineColor) {
            me.lineColor.destroy();
            me.lineColor = null;
        }
        me.lineSymbolID = null;
        me.lineWidth = null;
        me.markerAngle = null;
        me.markerSize = null;
        me.markerSymbolID = null;
    }

    /**
     * @function ServerStyle.prototype.toServerJSONObject
     * @description 转换成对应的 JSON 格式对象。
     * @returns {Object} 对应的 JSON 格式对象.
     */
    toServerJSONObject() {
        var styleObj = {};
        styleObj = Util_Util.copyAttributes(styleObj, this);
        //暂时先忽略serverColor往Json的转换
        return styleObj;
    }

    /**
     * @function ServerStyle.fromJson
     * @description 将JSON对象转换为 ServerStyle 对象。
     * @param {Object} jsonObject - 要转换的 JSON 对象。
     * @returns {ServerStyle} 转化后的 ServerStyle 对象。
     */
    static fromJson(jsonObject) {
        if (!jsonObject) {
            return;
        }
        return new ServerStyle({
            fillBackColor: ServerColor.fromJson(jsonObject.fillBackColor),
            fillBackOpaque: jsonObject.fillBackOpaque,
            fillForeColor: ServerColor.fromJson(jsonObject.fillForeColor),
            fillGradientMode: jsonObject.fillGradientMode,
            fillGradientAngle: jsonObject.fillGradientAngle,
            fillGradientOffsetRatioX: jsonObject.fillGradientOffsetRatioX,
            fillGradientOffsetRatioY: jsonObject.fillGradientOffsetRatioY,
            fillOpaqueRate: jsonObject.fillOpaqueRate,
            fillSymbolID: jsonObject.fillSymbolID,
            lineColor: ServerColor.fromJson(jsonObject.lineColor),
            lineSymbolID: jsonObject.lineSymbolID,
            lineWidth: jsonObject.lineWidth,
            markerAngle: jsonObject.markerAngle,
            markerSize: jsonObject.markerSize,
            markerSymbolID: jsonObject.markerSymbolID
        });
    }

}



;// CONCATENATED MODULE: ./src/common/iServer/PointWithMeasure.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class PointWithMeasure
 * @deprecatedclass SuperMap.PointWithMeasure
 * @category iServer SpatialAnalyst RouteLocator
 * @classdesc 路由点类。路由点是指具有线性度量值 (Measure) 的二维地理坐标点。
 * @param {Object} options - 参数。
 * @param {number} options.measure - 度量值，即路由对象属性值 M。
 * @param {number} options.x - 地理坐标系下的 X 坐标值。
 * @param {number} options.y - 地理坐标系下的 Y 坐标值。
 * @extends {GeometryPoint}
 * @usage
 */
class PointWithMeasure extends Point {

    constructor(options) {
        super(options);

        /**
         * @member {number} PointWithMeasure.prototype.measure
         * @description 度量值，即路由对象属性值 M。
         */
        this.measure = null;

        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.PointWithMeasure";
    }

    /**
     * @function PointWithMeasure.prototype.equals
     * @description 判断两个路由点对象是否相等。如果两个路由点对象具有相同的坐标以及度量值，则认为是相等的。
     * @param {PointWithMeasure} geom - 需要判断的路由点对象。
     * @returns {boolean} 两个路由点对象是否相等（true 为相等，false 为不等）。
     */
    equals(geom) {
        var equals = false;
        if (geom != null) {
            var isValueEquals = this.x === geom.x && this.y === geom.y && this.measure === geom.measure;
            var isNaNValue = isNaN(this.x) && isNaN(this.y) && isNaN(this.measure);
            var isNaNGeometry = isNaN(geom.x) && isNaN(geom.y) && isNaN(geom.measure);
            equals = ( isValueEquals || ( isNaNValue && isNaNGeometry ));
        }
        return equals;
    }


    /**
     * @function PointWithMeasure.prototype.toJson
     * @description 转换为 JSON 对象。
     * */
     toJson() {
        var result = "{";
        if (this.measure != null && this.measure != undefined) {
            result += "\"measure\":" + this.measure + ",";
        }
        result += "\"x\":" + this.x + ",";
        result += "\"y\":" + this.y;
        result += "}";
        return result;
    }


     /**
     * @function PointWithMeasure.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.measure = null;
        me.x = null;
        me.y = null;
    }

    /**
     * @function PointWithMeasure.fromJson
     * @description 将 JSON 对象转换为{@link PointWithMeasure} 对象。
     * @param {Object} jsonObject - JSON 对象表示的路由点。
     * @returns {PointWithMeasure} 转化后的 PointWithMeasure 对象。
     */
    static fromJson(jsonObject) {
        if (!jsonObject) {
            return;
        }
        return new PointWithMeasure({
            x: jsonObject.x,
            y: jsonObject.y,
            measure: jsonObject.measure
        });
    }

}


;// CONCATENATED MODULE: ./src/common/iServer/Route.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/






/**
 * @class Route
 * @deprecatedclass SuperMap.Route
 * @category iServer SpatialAnalyst RouteCalculateMeasure
 * @classdesc 路由对象类。路由对象为一系列有序的带有属性值 M 的 x，y 坐标对，其中 M 值为该结点的距离属性（到已知点的距离）。
 * @param {Array.<Geometry>} points - 形成路由对象的线数组。
 * @param {Object} options - 参数。
 * @param {number} options.id - 路由对象在数据库中的 ID。
 * @param {number} options.length - 路由对象的长度。单位与数据集的单位相同。
 * @param {number} [options.maxM] - 最大线性度量值，即所有结点到起始点的量算距离中最大值。
 * @param {number} [options.minM] - 最小线性度量值，即所有结点到起始点的量算距离中最小值。
 * @param {string} [options.type] - 数据类型，如："LINEM"。
 * @extends GeometryCollection
 * @usage
 */
class Route extends Collection {

    constructor(points, options) {
        super(points, options);

        /**
         * @member {number} Route.prototype.id
         * @description 路由对象在数据库中的 ID。
         */
        this.id = null;

        /**
         * @member {number} Route.prototype.center
         * @description 路由对象的中心点。
         */
        this.center = null;

        /**
         * @member {string} Route.prototype.style
         * @description 路由对象的样式。
         */
        this.style = null;

        /**
         * @member {number} Route.prototype.length
         * @description 路由对象的长度。单位与数据集的单位相同。
         */
        this.length = null;

        /**
         *  @member {number} Route.prototype.maxM
         *  @description 最大线性度量值，即所有结点到起始点的量算距离中最大值。
         */
        this.maxM = null;

        /**
         * @member {number} Route.prototype.minM
         * @description 最小线性度量值，即所有结点到起始点的量算距离中最小值。
         */
        this.minM = null;

        /**
         * @member {Array.<number>} Route.prototype.parts
         * @description 服务端几何对象中各个子对象所包含的节点个数。
         */
        this.parts = null;

        /**
         * @member {Array.<Object>} Route.prototype.points
         * @description 路由对象的所有路由点。
         * @example
         * (start code)
         * [
         *  {
         *      "measure": 0,
         *      "y": -4377.027184298267,
         *      "x": 4020.0045221720466
         *  },
         *  {
         *      "measure": 37.33288381391519,
         *      "y": -4381.569363260499,
         *      "x": 4057.0600591960642
         *  }
         * ]
         * (end)
         */
        this.points = null;

        /**
         * @member {string} Route.prototype.type
         * @description 服务端几何对象类型。
         */
        this.type = null;

        /**
         * @member {Array.<string>} [Route.prototype.componentTypes=LineString]
         * @description components 存储的几何对象所支持的几何类型数组。
         */
        this.componentTypes = ["SuperMap.Geometry.LinearRing", "SuperMap.Geometry.LineString"];

        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.Route";
        this.geometryType = "LINEM";
    }

    /**
     *
     * @function Route.prototype.toJson
     * @description 转换为 JSON 对象。
     * @returns {Object} JSON 对象。
     */
    toJson() {
        var result = "{";
        if (this.id != null && this.id != undefined) {
            result += "\"id\":" + this.id + ",";
        }
        if (this.center != null && this.center != undefined) {
            result += "\"center\":" + this.center + ",";
        }
        if (this.style != null && this.style != undefined) {
            result += "\"style\":" + this.style + ",";
        }
        if (this.length != null && this.length != undefined) {
            result += "\"length\":" + this.length + ",";
        }
        if (this.maxM != null && this.maxM != undefined) {
            result += "\"maxM\":" + this.maxM + ",";
        }
        if (this.minM != null && this.minM != undefined) {
            result += "\"minM\":" + this.minM + ",";
        }
        if (this.type != null && this.type != undefined) {
            result += "\"type\":\"" + this.type + "\",";
        }
        if (this.parts != null && this.parts != undefined) {
            result += "\"parts\":[" + this.parts[0];

            for (var i = 1; i < this.parts.length; i++) {
                result += "," + this.parts[i];
            }
            result += "],";
        }
        if (this.components != null && this.components.length > 0) {
            result += "\"points\":[";
            for (var j = 0, len = this.components.length; j < len; j++) {
                for (var k = 0, len2 = this.components[j].components.length; k < len2; k++) {
                    result += this.components[j].components[k].toJson() + ",";
                }
            }
            result = result.replace(/,$/g, '');
            result += "]";
        }
        result = result.replace(/,$/g, '');
        result += "}";
        return result;
    }


    /**
     * @function Route.prototype.destroy
     * @override
     */
    destroy() {
        var me = this;
        me.id = null;
        me.center = null;
        me.style = null;
        me.length = null;
        me.maxM = null;
        me.minM = null;
        me.type = null;
        me.parts = null;
        me.components.length = 0;
        me.components = null;
        me.componentTypes = null;
    }


    /**
     * @function Route.fromJson
     * @description 将 JSON 对象转换为 Route 对象。
     * @param {Object} [jsonObject] - JSON 对象表示的路由对象。
     * @returns {Route} 转化后的 Route 对象。
     */
    static fromJson(jsonObject) {
        if (!jsonObject) {
            return;
        }

        var geoParts = jsonObject.parts || [],
            geoPoints = jsonObject.points || [],
            len = geoParts.length,
            lineList = [];
        if (len > 0) {
            for (var i = 0, pointIndex = 0, pointList = []; i < len; i++) {
                for (var j = 0; j < geoParts[i]; j++) {
                    pointList.push(PointWithMeasure.fromJson(geoPoints[pointIndex + j]));
                }
                pointIndex += geoParts[i];
                //判断线是否闭合，如果闭合，则返回LinearRing，否则返回LineString
                if (pointList[0].equals(pointList[geoParts[i] - 1])) {
                    lineList.push(new LinearRing_LinearRing(pointList));
                } else {
                    lineList.push(new LineString(pointList));
                }
                pointList = [];
            }

        } else {
            return null;
        }

        return new Route(lineList, {
            id: jsonObject.id,
            center: jsonObject.center,
            style: jsonObject.style,
            length: jsonObject.length,
            maxM: jsonObject.maxM,
            minM: jsonObject.minM,
            type: jsonObject.type,
            parts: jsonObject.parts
        });
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/ServerGeometry.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/













/**
 * @class ServerGeometry
 * @deprecatedclass SuperMap.ServerGeometry
 * @category  iServer Data Feature
 * @classdesc 服务端几何对象类。该类描述几何对象（矢量）的特征数据（坐标点对、几何对象的类型等）。基于服务端的空间分析、空间关系运算、查询等 GIS 服务功能使用服务端几何对象。
 * @param {Object} options - 参数。
 * @param {string} options.id - 服务端几何对象唯一标识符。
 * @param {Array.<number>} options.parts - 服务端几何对象中各个子对象所包含的节点个数。
 * @param {Array.<GeometryPoint>} options.points - 组成几何对象的节点的坐标对数组。
 * @param {GeometryType} options.type - 几何对象的类型。
 * @param {ServerStyle} [options.style] - 服务端几何对象的风格。
 * @usage
 */
class ServerGeometry {
    constructor(options) {
        /**
         * @member {string} ServerGeometry.prototype.id
         * @description 服务端几何对象唯一标识符。
         */
        this.id = 0;

        /**
         * @member {ServerStyle} [ServerGeometry.prototype.style]
         * @description 服务端几何对象的风格（ServerStyle）。
         */
        this.style = null;

        /**
         * @member {Array.<number>} ServerGeometry.prototype.parts
         * @description 服务端几何对象中各个子对象所包含的节点个数。<br>
         * 1.几何对象从结构上可以分为简单几何对象和复杂几何对象。
         * 简单几何对象与复杂几何对象的区别：简单的几何对象一般为单一对象，
         * 而复杂的几何对象由多个简单对象组成或经过一定的空间运算之后产生，
         * 如：矩形为简单的区域对象，而中空的矩形为复杂的区域对象。<br>
         * 2.通常情况，一个简单几何对象的子对象就是它本身，
         * 因此对于简单对象来说的该字段为长度为1的整型数组，
         * 该字段的值就是这个简单对象节点的个数。
         * 如果一个几何对象是由几个简单对象组合而成的，
         * 例如，一个岛状几何对象由 3 个简单的多边形组成而成，
         * 那么这个岛状的几何对象的 Parts 字段值就是一个长度为 3 的整型数组，
         * 数组中每个成员的值分别代表这三个多边形所包含的节点个数。
         */
        this.parts = null;

        /**
         * @member {Array.<GeometryPoint>} ServerGeometry.prototype.points
         * @description 组成几何对象的节点的坐标对数组。<br>
         * 1.所有几何对象（点、线、面）都是由一些简单的点坐标组成的，
         * 该字段存放了组成几何对象的点坐标的数组。
         * 对于简单的面对象，他的起点和终点的坐标点相同。<br>
         * 2.对于复杂的几何对象，根据 Parts 属性来确定每一个组成复杂几何对象的简单对象所对应的节点的个数，
         * 从而确定 Points 字段中坐标对的分配归属问题。
         */
        this.points = null;

        /**
         * @member {GeometryType} ServerGeometry.prototype.type
         * @description 几何对象的类型（GeometryType）。
         */
        this.type = null;

        /**
         * @member {Object} ServerGeometry.prototype.prjCoordSys
         * @description 投影坐标参数，现仅在缓冲区分析中有效。
         */
        this.prjCoordSys = null;
        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = 'SuperMap.ServerGeometry';
    }

    /**
     * @function ServerGeometry.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.id = null;
        me.style = null;
        me.parts = null;
        me.partTopo = null;
        me.points = null;
        me.type = null;
        me.prjCoordSys = null;
    }

    /**
     * @function ServerGeometry.prototype.toGeometry
     * @description 将服务端几何对象 ServerGeometry 转换为客户端几何对象 Geometry。
     * @returns {Geometry} 转换后的客户端几何对象。
     */
    toGeometry() {
        var me = this,
            geoType = me.type;
        switch (geoType.toUpperCase()) {
            case REST_GeometryType.POINT:
                return me.toGeoPoint();
            case REST_GeometryType.LINE:
                return me.toGeoLine();
            case REST_GeometryType.LINEM:
                return me.toGeoLinem();
            case REST_GeometryType.REGION:
                return me.toGeoRegion();
            case REST_GeometryType.POINTEPS:
                return me.toGeoPoint();
            case REST_GeometryType.LINEEPS:
                return me.toGeoLineEPS();
            case REST_GeometryType.REGIONEPS:
                return me.toGeoRegionEPS();
            case REST_GeometryType.GEOCOMPOUND:
                return me.transformGeoCompound();
        }
    }

    /**
     * @function ServerGeometry.prototype.toGeoPoint
     * @description 将服务端的点几何对象转换为客户端几何对象。包括 Point、MultiPoint。
     * @returns {Geometry} 转换后的客户端几何对象。
     */
    toGeoPoint() {
        var me = this,
            geoParts = me.parts || [],
            geoPoints = me.points || [],
            len = geoParts.length;
        if (len > 0) {
            if (len === 1) {
                return new Point(geoPoints[0].x, geoPoints[0].y);
            } else {
                var pointList = [];
                for (let i = 0; i < len; i++) {
                    pointList.push(new Point(geoPoints[i].x, geoPoints[i].y));
                }
                return new MultiPoint(pointList);
            }
        } else {
            return null;
        }
    }

    /**
     * @function ServerGeometry.prototype.toGeoLine
     * @description 将服务端的线几何对象转换为客户端几何对象。包括 GeometryLinearRing、GeometryLineString、GeometryMultiLineString。
     * @returns {Geometry} 转换后的客户端几何对象。
     */
    toGeoLine() {
        var me = this,
            geoParts = me.parts || [],
            geoPoints = me.points || [],
            len = geoParts.length;
        if (len > 0) {
            if (len === 1) {
                let pointList = [];
                for (let i = 0; i < geoParts[0]; i++) {
                    pointList.push(new Point(geoPoints[i].x, geoPoints[i].y));
                }
                //判断线是否闭合，如果闭合，则返回LinearRing，否则返回LineString
                if (pointList[0].equals(pointList[geoParts[0] - 1])) {
                    return new LinearRing_LinearRing(pointList);
                } else {
                    return new LineString(pointList);
                }
            } else {
                let lineList = [];
                for (let i = 0; i < len; i++) {
                    let pointList = [];
                    for (let j = 0; j < geoParts[i]; j++) {
                        pointList.push(new Point(geoPoints[j].x, geoPoints[j].y));
                    }
                    lineList.push(new LineString(pointList));
                    geoPoints.splice(0, geoParts[i]);
                }
                return new MultiLineString(lineList);
            }
        } else {
            return null;
        }
    }

    /**
     * @function ServerGeometry.prototype.toGeoLineEPS
     * @description 将服务端的线几何对象转换为客户端几何对象。包括 GeometryLinearRing、GeometryLineString、GeometryMultiLineString。
     * @returns {Geometry} 转换后的客户端几何对象。
     */
    toGeoLineEPS() {
        var me = this,
            geoParts = me.parts || [],
            geoPoints = me.points || [],
            i,
            j,
            pointList,
            lineList,
            lineEPS,
            len = geoParts.length;
        if (len > 0) {
            if (len === 1) {
                for (i = 0, pointList = []; i < geoParts[0]; i++) {
                    pointList.push(new Point(geoPoints[i].x, geoPoints[i].y, geoPoints[i].type));
                }
                //判断线是否闭合，如果闭合，则返回LinearRing，否则返回LineString
                if (pointList[0].equals(pointList[geoParts[0] - 1])) {
                    lineEPS = LineString.createLineEPS(pointList);
                    return new LinearRing_LinearRing(lineEPS);
                } else {
                    lineEPS = LineString.createLineEPS(pointList);
                    return new LineString(lineEPS);
                }
            } else {
                for (i = 0, lineList = []; i < len; i++) {
                    for (j = 0, pointList = []; j < geoParts[i]; j++) {
                        pointList.push(new Point(geoPoints[j].x, geoPoints[j].y));
                    }
                    lineEPS = LineString.createLineEPS(pointList);
                    lineList.push(new LineString(lineEPS));
                    geoPoints.splice(0, geoParts[i]);
                }
                return new MultiLineString(lineList);
            }
        } else {
            return null;
        }
    }

    /**
     * @function ServerGeometry.prototype.toGeoLinem
     * @description 将服务端的路由线几何对象转换为客户端几何对象。包括 LinearRing、LineString、MultiLineString。
     * @returns {Geometry} 转换后的客户端几何对象。
     */
    toGeoLinem() {
        var me = this;
        return Route.fromJson(me);
    }

    /**
     * @function ServerGeometry.prototype.toGeoRegion
     * @description 将服务端的面几何对象转换为客户端几何对象。类型为 GeometryPolygon。
     * @returns {Geometry} 转换后的客户端几何对象。
     */
    toGeoRegion() {
        var me = this,
            geoParts = me.parts || [],
            geoTopo = me.partTopo || [],
            geoPoints = me.points || [],
            len = geoParts.length;
        if (len <= 0) {
            return null;
        }
        var polygonArray = [];
        var pointList = [];
        if (len == 1) {
            for (let i = 0; i < geoPoints.length; i++) {
                pointList.push(new Point(geoPoints[i].x, geoPoints[i].y));
            }
            polygonArray.push(new Polygon_Polygon([new LinearRing_LinearRing(pointList)]));
            return new MultiPolygon(polygonArray);
        }
        //处理复杂面
        var CCWArray = [];
        var areaArray = [];
        var polygonArrayTemp = [];
        var polygonBounds = [];
        //polyon岛洞标识数组，初始都是岛。
        var CCWIdent = [];
        for (let i = 0, pointIndex = 0; i < len; i++) {
            for (let j = 0; j < geoParts[i]; j++) {
                pointList.push(new Point(geoPoints[pointIndex + j].x, geoPoints[pointIndex + j].y));
            }
            pointIndex += geoParts[i];
            var polygon = new Polygon_Polygon([new LinearRing_LinearRing(pointList)]);
            pointList = [];
            polygonArrayTemp.push(polygon);
            if (geoTopo.length === 0) {
                polygonBounds.push(polygon.getBounds());
            }
            CCWIdent.push(1);
            areaArray.push(polygon.getArea());
        }
        //iServer 9D新增字段
        if (geoTopo.length === 0) {
            //根据面积排序
            ServerGeometry.bubbleSort(areaArray, polygonArrayTemp, geoTopo, polygonBounds);
            //岛洞底层判断原则：将所有的子对象按照面积排序，面积最大的直接判定为岛（1），从面积次大的开始处理，
            // 如果发现该对象在某个面积大于它的对象之中（即被包含），则根据包含它的对象的标识（1 or -1），指定其标识（-1 or 1），
            // 依次处理完所有对象，就得到了一个标识数组，1表示岛，-1表示洞
            //目标polygon索引列表 -1标示没有被任何polygon包含，
            var targetArray = [];
            for (let i = 1; i < polygonArrayTemp.length; i++) {
                for (let j = i - 1; j >= 0; j--) {
                    targetArray[i] = -1;
                    if (polygonBounds[j].containsBounds(polygonBounds[i])) {
                        CCWIdent[i] = CCWIdent[j] * -1;
                        if (CCWIdent[i] < 0) {
                            targetArray[i] = j;
                        }
                        break;
                    }
                }
            }
            for (let i = 0; i < polygonArrayTemp.length; i++) {
                if (CCWIdent[i] > 0) {
                    polygonArray.push(polygonArrayTemp[i]);
                } else {
                    polygonArray[targetArray[i]].components = polygonArray[targetArray[i]].components.concat(
                        polygonArrayTemp[i].components
                    );
                    //占位
                    polygonArray.push('');
                }
            }
        } else {
            polygonArray = new Array();
            for (let i = 0; i < polygonArrayTemp.length; i++) {
                if (geoTopo[i] && geoTopo[i] == -1) {
                    CCWArray = CCWArray.concat(polygonArrayTemp[i].components);
                } else {
                    if (CCWArray.length > 0 && polygonArray.length > 0) {
                        polygonArray[polygonArray.length - 1].components =
                            polygonArray[polygonArray.length - 1].components.concat(CCWArray);
                        CCWArray = [];
                    }
                    polygonArray.push(polygonArrayTemp[i]);
                }
                if (i == len - 1) {
                    var polyLength = polygonArray.length;
                    if (polyLength) {
                        polygonArray[polyLength - 1].components =
                            polygonArray[polyLength - 1].components.concat(CCWArray);
                    } else {
                        for (let k = 0, length = CCWArray.length; k < length; k++) {
                            polygonArray.push(new Polygon_Polygon(CCWArray));
                        }
                    }
                }
            }
        }
        return new MultiPolygon(polygonArray);
    }

    /**
     * @function ServerGeometry.prototype.toGeoRegionEPS
     * @description 将服务端的面几何对象转换为客户端几何对象。类型为 Polygon。
     * @returns {Geometry} 转换后的客户端几何对象。
     */
    toGeoRegionEPS() {
        var me = this,
            geoParts = me.parts || [],
            geoTopo = me.partTopo || [],
            geoPoints = me.points || [],
            len = geoParts.length;

        if (len <= 0) {
            return null;
        }
        var polygonArray = [];
        var pointList = [];
        var lineEPS;
        if (len == 1) {
            for (var i = 0; i < geoPoints.length; i++) {
                pointList.push(new Point(geoPoints[i].x, geoPoints[i].y));
            }

            lineEPS = LineString.createLineEPS(pointList);
            polygonArray.push(new Polygon_Polygon([new LinearRing_LinearRing(lineEPS)]));
            return new MultiPolygon(polygonArray);
        }
        //处理复杂面
        var CCWArray = [];
        var areaArray = [];
        var polygonArrayTemp = [];
        var polygonBounds = [];
        //polyon岛洞标识数组，初始都是岛。
        var CCWIdent = [];
        for (let i = 0, pointIndex = 0; i < len; i++) {
            for (let j = 0; j < geoParts[i]; j++) {
                pointList.push(new Point(geoPoints[pointIndex + j].x, geoPoints[pointIndex + j].y));
            }
            pointIndex += geoParts[i];

            lineEPS = LineString.createLineEPS(pointList);
            var polygon = new Polygon_Polygon([new LinearRing_LinearRing(lineEPS)]);
            pointList = [];
            polygonArrayTemp.push(polygon);
            if (geoTopo.length === 0) {
                polygonBounds.push(polygon.getBounds());
            }
            CCWIdent.push(1);
            areaArray.push(polygon.getArea());
        }
        //iServer 9D新增字段
        if (geoTopo.length === 0) {
            //根据面积排序
            ServerGeometry.bubbleSort(areaArray, polygonArrayTemp, geoTopo, polygonBounds);
            //岛洞底层判断原则：将所有的子对象按照面积排序，面积最大的直接判定为岛（1），从面积次大的开始处理，
            // 如果发现该对象在某个面积大于它的对象之中（即被包含），则根据包含它的对象的标识（1 or -1），指定其标识（-1 or 1），
            // 依次处理完所有对象，就得到了一个标识数组，1表示岛，-1表示洞
            //目标polygon索引列表 -1标示没有被任何polygon包含，
            var targetArray = [];
            for (let i = 1; i < polygonArrayTemp.length; i++) {
                for (let j = i - 1; j >= 0; j--) {
                    targetArray[i] = -1;
                    if (polygonBounds[j].containsBounds(polygonBounds[i])) {
                        CCWIdent[i] = CCWIdent[j] * -1;
                        if (CCWIdent[i] < 0) {
                            targetArray[i] = j;
                        }
                        break;
                    }
                }
            }
            for (let i = 0; i < polygonArrayTemp.length; i++) {
                if (CCWIdent[i] > 0) {
                    polygonArray.push(polygonArrayTemp[i]);
                } else {
                    polygonArray[targetArray[i]].components = polygonArray[targetArray[i]].components.concat(
                        polygonArrayTemp[i].components
                    );
                    //占位
                    polygonArray.push('');
                }
            }
        } else {
            polygonArray = new Array();
            for (let i = 0; i < polygonArrayTemp.length; i++) {
                if (geoTopo[i] && geoTopo[i] == -1) {
                    CCWArray = CCWArray.concat(polygonArrayTemp[i].components);
                } else {
                    if (CCWArray.length > 0 && polygonArray.length > 0) {
                        polygonArray[polygonArray.length - 1].components =
                            polygonArray[polygonArray.length - 1].components.concat(CCWArray);
                        CCWArray = [];
                    }
                    polygonArray.push(polygonArrayTemp[i]);
                }
                if (i == len - 1) {
                    var polyLength = polygonArray.length;
                    if (polyLength) {
                        polygonArray[polyLength - 1].components =
                            polygonArray[polyLength - 1].components.concat(CCWArray);
                    } else {
                        for (let k = 0, length = CCWArray.length; k < length; k++) {
                            polygonArray.push(new Polygon_Polygon(CCWArray));
                        }
                    }
                }
            }
        }
        return new MultiPolygon(polygonArray);
    }
    transformGeoCompound() {
        const me = this,
            geoParts = me.geoParts || [],
            len = geoParts.length;
        if (len <= 0) {
            return null;
        }
        const geometryList = [];
        for (let index = 0; index < len; index++) {
            const geometry = geoParts[index];
            geometryList.push(new ServerGeometry(geometry).toGeometry());
        }
        return new Collection(geometryList);
    }

    /**
     * @function ServerGeometry.prototype.fromJson
     * @description 将 JSON 对象表示服务端几何对象转换为 ServerGeometry。
     * @param {Object} jsonObject - 要转换的 JSON 对象。
     * @returns {ServerGeometry} 转换后的 ServerGeometry 对象。
     */
    static fromJson(jsonObject) {
        if (!jsonObject) {
            return;
        }
        return new ServerGeometry({
            id: jsonObject.id,
            style: ServerStyle.fromJson(jsonObject.style),
            parts: jsonObject.parts,
            partTopo: jsonObject.partTopo,
            points: jsonObject.points,
            center: jsonObject.center,
            length: jsonObject.length,
            maxM: jsonObject.maxM,
            minM: jsonObject.minM,
            type: jsonObject.type
        });
    }

    /**
     * @function ServerGeometry.prototype.fromGeometry
     * @description 将客户端 Geometry 转换成服务端 ServerGeometry。
     * @param {Geometry} geometry - 要转换的客户端 Geometry 对象。
     * @returns {ServerGeometry} 转换后的 ServerGeometry 对象。
     */
    static fromGeometry(geometry) {
        if (!geometry) {
            return;
        }
        var id = 0,
            parts = [],
            points = [],
            type = null,
            icomponents = geometry.components,
            className = geometry.CLASS_NAME,
            prjCoordSys = { epsgCode: geometry.SRID };

        if (!isNaN(geometry.id)) {
            id = geometry.id;
        }
        //坑爹的改法，没法，为了支持态势标绘，有时间就得全改
        if (
            className != 'SuperMap.Geometry.LinearRing' &&
            className != 'SuperMap.Geometry.LineString' &&
            (geometry instanceof MultiPoint || geometry instanceof MultiLineString)
        ) {
            let ilen = icomponents.length;
            for (let i = 0; i < ilen; i++) {
                const vertices = icomponents[i].getVertices();
                let partPointsCount = vertices.length;
                parts.push(partPointsCount);
                for (let j = 0; j < partPointsCount; j++) {
                    points.push(new Point(vertices[j].x, vertices[j].y));
                }
            }
            //这里className不是多点就全部是算线
            type = className == 'SuperMap.Geometry.MultiPoint' ? REST_GeometryType.POINT : REST_GeometryType.LINE;
        } else if (geometry instanceof MultiPolygon) {
            let ilen = icomponents.length;
            for (let i = 0; i < ilen; i++) {
                let polygon = icomponents[i],
                    linearRingOfPolygon = polygon.components,
                    linearRingOfPolygonLen = linearRingOfPolygon.length;
                for (let j = 0; j < linearRingOfPolygonLen; j++) {
                    const vertices = linearRingOfPolygon[j].getVertices();
                    const partPointsCount = vertices.length + 1;
                    parts.push(partPointsCount);
                    for (let k = 0; k < partPointsCount - 1; k++) {
                        points.push(new Point(vertices[k].x, vertices[k].y));
                    }
                    points.push(
                        new Point(vertices[0].x, vertices[0].y)
                    );
                }
            }
            type = REST_GeometryType.REGION;
        } else if (geometry instanceof Polygon_Polygon) {
            let ilen = icomponents.length;
            for (let i = 0; i < ilen; i++) {
                const vertices = icomponents[i].getVertices();
                let partPointsCount = vertices.length + 1;
                parts.push(partPointsCount);
                for (let j = 0; j < partPointsCount - 1; j++) {
                  points.push(new Point(vertices[j].x, vertices[j].y));
                }
                points.push(new Point(vertices[0].x, vertices[0].y));
            }
            type = REST_GeometryType.REGION;
        } else {
            const vertices = geometry.getVertices();
            let geometryVerticesCount = vertices.length;
            for (let j = 0; j < geometryVerticesCount; j++) {
                points.push(new Point(vertices[j].x, vertices[j].y));
            }
            if (geometry instanceof LinearRing_LinearRing) {
                points.push(new Point(vertices[0].x, vertices[0].y));
                geometryVerticesCount++;
            }
            parts.push(geometryVerticesCount);
            type = geometry instanceof Point ? REST_GeometryType.POINT : REST_GeometryType.LINE;
        }

        return new ServerGeometry({
            id: id,
            style: null,
            parts: parts,
            points: points,
            type: type,
            prjCoordSys: prjCoordSys
        });
    }

    /**
     * @function ServerGeometry.prototype.IsClockWise
     * @description 判断 linearRing 中的点的顺序。返回值大于 0，逆时针；小于 0，顺时针。
     * @param {Geometry} geometry - 要转换的客户端 Geometry 对象。
     * @returns {number} 返回值大于 0，逆时针；小于 0，顺时针。
     */
    static IsClockWise(points) {
        var length = points.length;
        if (length < 3) {
            return 0.0;
        }
        var s = points[0].y * (points[length - 1].x - points[1].x);
        points.push(points[0]);
        for (var i = 1; i < length; i++) {
            s += points[i].y * (points[i - 1].x - points[i + 1].x);
        }
        return s * 0.5;
    }

    static bubbleSort(areaArray, pointList, geoTopo, polygonBounds) {
        for (var i = 0; i < areaArray.length; i++) {
            for (var j = 0; j < areaArray.length; j++) {
                if (areaArray[i] > areaArray[j]) {
                    var d = areaArray[j];
                    areaArray[j] = areaArray[i];
                    areaArray[i] = d;
                    var b = pointList[j];
                    pointList[j] = pointList[i];
                    pointList[i] = b;
                    if (geoTopo && geoTopo.length > 0) {
                        var c = geoTopo[j];
                        geoTopo[j] = geoTopo[i];
                        geoTopo[i] = c;
                    }
                    if (polygonBounds && polygonBounds.length > 0) {
                        var f = polygonBounds[j];
                        polygonBounds[j] = polygonBounds[i];
                        polygonBounds[i] = f;
                    }
                }
            }
        }
    }
}


;// CONCATENATED MODULE: ./src/common/commontypes/Geometry3D.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class Geometry3D
 * @deprecatedclass SuperMap.Geometry3D
 * @category BaseTypes Geometry3D
 * @classdesc 所有三维几何类的基类。
 * @usage
 */
class Geometry3D extends ServerGeometry{

    constructor(options) {
        super(options);

        this.CLASS_NAME = "SuperMap.Geometry3D";
        /**
         * @member {GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|Array.<number>} Geometry3D.prototype.position 
         * @description  三维几何对象的位置。
         *
         */
        this.position = null;

        /**
         * @member {number} Geometry3D.prototype.rotationX 
         * @description 三维几何对象沿 X 轴方向的旋转角度，单位为度。
         */
        this.rotationX = null;

        /**
         * @member {number} Geometry3D.prototype.rotationY
         * @description 三维几何对象沿 Y 轴方向的旋转角度，单位为度。
         */
        this.rotationY = null;

        /**
         * @member {number} Geometry3D.prototype.rotationZ
         * @description 三维几何对象沿 Z 轴方向的旋转角度，单位为度。
         */
        this.rotationZ = null;

        /**
         * @member {number} Geometry3D.prototype.scaleX 
         * @description 三维几何对象沿 X 轴方向的缩放比例。
         */
        this.scaleX = null;

        /**
         * @member {number} Geometry3D.prototype.scaleY
         * @description 三维几何对象沿 Y 轴方向的缩放比例。
         */
        this.scaleY = null;

        /**
         * @member {number} Geometry3D.prototype.scaleZ
         * @description 三维几何对象沿 Z 轴方向的缩放比例。
         */
        this.scaleZ = null;

        Util_Util.extend(this, options);
    }


    /**
     * @function Geometry3D.prototype.destroy
     * @description 释放资源。
     */
    destroy() {
        this.position = null;
        this.rotationX = null;
        this.rotationY = null;
        this.rotationZ = null;
        this.scaleX = null;
        this.scaleY = null;
        this.scaleZ = null;
    }
}

;// CONCATENATED MODULE: ./src/common/commontypes/Vector.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class FeatureVector
 * @aliasclass Feature.Vector
 * @deprecatedclass SuperMap.Feature.Vector
 * @category BaseTypes Geometry
 * @classdesc 矢量要素类。该类具有 Geometry 属性存放几何信息，
 * attributes 属性存放非几何信息，另外还包含了 style 属性，用来定义矢量要素的样式，
 * 其中，默认的样式在 {@link FeatureVector.style} 类中定义，如果没有特别的指定将使用默认的样式。
 * @extends {Feature}
 * @param {Geometry} geometry - 要素的几何信息。
 * @param {Object} [attributes] - 描述要素的任意的可序列化属性，将要映射到 attributes 属性中的对象。
 * @param {Object} [style] - 样式对象。
 * @example
 * var geometry = new GeometryPoint(-115,10);
 *  var style = {
     *      strokeColor:"#339933",
     *      strokeOpacity:1,
     *      strokeWidth:3,
     *      pointRadius:6
     *  }
 *  var pointFeature = new FeatureVector(geometry,null,style);
 *  vectorLayer.addFeatures(pointFeature);
 * @usage
 */
// TRASH THIS
const State = {
  /** states */
  UNKNOWN: 'Unknown',
  INSERT: 'Insert',
  UPDATE: 'Update',
  DELETE: 'Delete'
};
class Vector extends Feature_Feature {


    constructor(geometry, attributes, style) {
        super(null, null, attributes);
        /**
         * @member {string} FeatureVector.prototype.fid
         * @description fid。
         */
        this.fid = null;

        /**
         * @member {Geometry} FeatureVector.prototype.geometry
         * @description 存放几何信息。
         */
        this.geometry = geometry ? geometry : null;

        /**
         * @member {Object} FeatureVector.prototype.attributes
         * @description 描述要素的任意的可序列化属性。
         */
        this.attributes = {};

        if (attributes) {
            this.attributes = Util_Util.extend(this.attributes, attributes);
        }

        /**
         * @member {Bounds} FeatureVector.prototype.bounds
         * @description 限制要素几何的边界。
         */
        this.bounds = null;

        /**
         * @member {string} FeatureVector.prototype.state
         * @description state。
         */
        this.state = null;

        /**
         * @member {Object} FeatureVector.prototype.style
         * @description 要素的样式属性，地图查询返回的 feature 的 style，8C 变为null。
         */
        this.style = style ? style : null;

        /**
         * @member {string} FeatureVector.prototype.url
         * @description 如果设置了这个属性，在更新或者删除要素时需要考虑 {@link HTTP} 。
         */
        this.url = null;

        this.lonlat = null;

        this.CLASS_NAME = "SuperMap.Feature.Vector";

        Vector.style = {
            'default': {
                fillColor: "#ee9900",
                fillOpacity: 0.4,
                hoverFillColor: "white",
                hoverFillOpacity: 0.8,
                strokeColor: "#ee9900",
                strokeOpacity: 1,
                strokeWidth: 1,
                strokeLinecap: "round",
                strokeDashstyle: "solid",
                hoverStrokeColor: "red",
                hoverStrokeOpacity: 1,
                hoverStrokeWidth: 0.2,
                pointRadius: 6,
                hoverPointRadius: 1,
                hoverPointUnit: "%",
                pointerEvents: "visiblePainted",
                cursor: "inherit",
                fontColor: "#000000",
                labelAlign: "cm",
                labelOutlineColor: "white",
                labelOutlineWidth: 3
            },
            'select': {
                fillColor: "blue",
                fillOpacity: 0.4,
                hoverFillColor: "white",
                hoverFillOpacity: 0.8,
                strokeColor: "blue",
                strokeOpacity: 1,
                strokeWidth: 2,
                strokeLinecap: "round",
                strokeDashstyle: "solid",
                hoverStrokeColor: "red",
                hoverStrokeOpacity: 1,
                hoverStrokeWidth: 0.2,
                pointRadius: 6,
                hoverPointRadius: 1,
                hoverPointUnit: "%",
                pointerEvents: "visiblePainted",
                cursor: "pointer",
                fontColor: "#000000",
                labelAlign: "cm",
                labelOutlineColor: "white",
                labelOutlineWidth: 3

            },
            'temporary': {
                fillColor: "#66cccc",
                fillOpacity: 0.2,
                hoverFillColor: "white",
                hoverFillOpacity: 0.8,
                strokeColor: "#66cccc",
                strokeOpacity: 1,
                strokeLinecap: "round",
                strokeWidth: 2,
                strokeDashstyle: "solid",
                hoverStrokeColor: "red",
                hoverStrokeOpacity: 1,
                hoverStrokeWidth: 0.2,
                pointRadius: 6,
                hoverPointRadius: 1,
                hoverPointUnit: "%",
                pointerEvents: "visiblePainted",
                //cursor:"inherit",
                cursor: "default",
                fontColor: "#000000",
                labelAlign: "cm",
                labelOutlineColor: "white",
                labelOutlineWidth: 3

            },
            'delete': {
                display: "none"
            }
        };
    }

    /**
     * @function FeatureVector.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        if (this.layer) {
            this.layer.removeFeatures(this);
            this.layer = null;
        }

        this.geometry = null;
        super.destroy();
    }

    /**
     * @function FeatureVector.prototype.clone
     * @description 复制矢量要素，并返回复制后的新对象。
     * @returns {FeatureVector} 相同要素的新的矢量要素。
     */
    clone() {
        return new Vector(
            this.geometry ? this.geometry.clone() : null,
            this.attributes,
            this.style);
    }

    /**
     * @function FeatureVector.prototype.toState
     * @description 设置新状态。
     * @param {string} state - 状态。
     */
    toState(state) {
        if (state === State.UPDATE) {
            switch (this.state) {
                case State.UNKNOWN:
                case State.DELETE:
                    this.state = state;
                    break;
                case State.UPDATE:
                case State.INSERT:
                    break;
            }
        } else if (state === State.INSERT) {
            switch (this.state) {
                case State.UNKNOWN:
                    break;
                default:
                    this.state = state;
                    break;
            }
        } else if (state === State.DELETE) {
            switch (this.state) {
                case State.INSERT:
                    // the feature should be destroyed
                    break;
                case State.DELETE:
                    break;
                case State.UNKNOWN:
                case State.UPDATE:
                    this.state = state;
                    break;
            }
        } else if (state === State.UNKNOWN) {
            this.state = state;
        }
    }
}
/**
 *
 * @typedef {Object} FeatureVector.style
 * @description Feature 有大量的样式属性，如果没有特别的指定将使用默认的样式，
 * 大部分样式通过 SVG 标准定义属性。
 * - fill properties 资料介绍：{@link http://www.w3.org/TR/SVG/painting.html#FillProperties}
 * - stroke properties 资料介绍：{@link http://www.w3.org/TR/SVG/painting.html#StrokeProperties}
 * @property {boolean} [fill] - 不需要填充则设置为 false。
 * @property {string} [fillColor='#ee9900'] - 十六进制填充颜色。
 * @property {number} [fillOpacity=0.4] - 填充不透明度。
 * @property {boolean} [stroke] - 不需要描边则设为 false。
 * @property {string} [strokeColor='#ee9900'] - 十六进制描边颜色。
 * @property {number} [strokeOpacity=0.4] - 描边的不透明度(0-1)。
 * @property {number} [strokeWidth=1] - 像素描边宽度。
 * @property {string} [strokeLinecap='round'] - strokeLinecap 有三种类型 butt，round，square。
 * @property {string} [strokeDashstyle='solid'] - 有 dot，dash，dashdot，longdash，longdashdot，solid 几种样式。
 * @property {boolean} [graphic] - 不需要则设置为 false。
 * @property {number} [pointRadius=6] - 像素点半径。
 * @property {string} [pointerEvents='visiblePainted'] - pointerEvents。
 * @property {string} [cursor] - cursor。
 * @property {boolean} [allowRotate='false'] - 是否允许图标随着运行方向旋转。用于时空数据图层。
 * @property {string} [externalGraphic] - 连接到用来渲染点的外部的图形。
 * @property {number} [graphicWidth] - 外部图表的像素宽度。
 * @property {number} [graphicHeight] - 外部图表的像素高度。
 * @property {number} [graphicOpacity] - 外部图表的不透明度(0-1)。
 * @property {number} [graphicXOffset] - 外部图表沿着x方向的偏移量。
 * @property {number} [graphicYOffset] - 外部图表沿着y方向的偏移量。
 * @property {number} [rotation] - 一个图表沿着其中心点（或者偏移中心指定点）在顺时针方向旋转。
 * @property {number} [graphicZIndex] - 渲染时使用的索引值。
 * @property {string} [graphicName='circle'] - 渲染点时图标使用的名字。支持"circle" , "square", "star", "x", "cross", "triangle"。
 * @property {string} [graphicTitle] - 外部图表的提示框。
 * @property {string} [backgroundGraphic] - 外部图表的背景。
 * @property {number} [backgroundGraphicZIndex] - 背景图渲染时使用的索引值。
 * @property {number} [backgroundXOffset] - 背景图在 x 轴的偏移量。
 * @property {number} [backgroundYOffset] - 背景图在 y 轴的偏移量。
 * @property {number} [backgroundHeight] - 背景图的高度。如果没有设置，将用 graphicHeight。
 * @property {number} [backgroundWidth] - 背景图的宽度。如果没有设置，将用 graphicWidth。
 * @property {boolean} [isUnicode=false] - 这个属性要配合 label 属性来用，当为 true时，label 就可以使用 unicode 编码，
 * 比如 "a" 的 unicode 十六进制编码为 61，则 label 属性可以为 "&#x61;",其中 "&#" 为前缀，标志这个为 unicode 编码，
 * "x" 是指 16 进制,这时页面显示的是 "a"；当此值为 false 的时候，label 的内容会被直接输出，
 * 比如，label 为 "&#x61;"，这时页面显示的也是 "&#x61;"。
 * @property {string} [label] - 可选的标签文本。
 * @property {string} [labelAlign='cm'] - 标签对齐，是由两个字符组成的字符串，如："lt", "cm", "rb"，
 * 其中第一个字符代表水平方向上的对齐，"l"=left, "c"=center, "r"=right；
 * 第二个字符代表垂直方向上的对齐，"t"=top, "m"=middle, "b"=bottom。
 * @property {number} [labelXOffset] - 标签在 x 轴方向的偏移量。
 * @property {number} [labelYOffset] - 标签在 y 轴方向的偏移量。
 * @property {boolean} [labelSelect=false] - 如果设为 true，标签可以选用 SelectFeature 或者 similar 控件。
 * @property {string} [fontColor='#000000'] - 标签字体颜色。
 * @property {number} [fontOpacity] - 标签透明度 (0-1)。
 * @property {string} [fontFamily] - 标签的字体类型。
 * @property {string} [fontSize] - 标签的字体大小。
 * @property {string} [fontStyle] - 标签的字体样式。
 * @property {string} [fontWeight] - 标签的字体粗细。
 * @property {string} [display] - 如果 display 属性设置为 “none”，符号将没有任何效果。
 * @example
 *  // label的用法如下：
 *  function addGeoTest(){
 *  var geometry = new GeometryPoint(105, 35);
 *  var pointFeature = new FeatureVector(geometry);
 *  var styleTest = {
 *        label:"supermap",
 *        fontColor:"#0000ff",
 *        fontOpacity:"0.5",
 *        fontFamily:"隶书",
 *        fontSize:"8em",
 *        fontWeight:"bold",
 *        fontStyle:"italic",
 *        labelSelect:"true",
 *     }
 *           pointFeature.style = styleTest;
 *          vectorLayer.addFeatures([pointFeature]);
 * }
     */



;// CONCATENATED MODULE: ./src/common/commontypes/index.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/































;// CONCATENATED MODULE: ./src/common/format/Format.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class Format
 * @deprecatedclass SuperMap.Format
 * @classdesc 读写各种格式的格式类基类。其子类应该包含并实现 read 和 write 方法。
 * @category BaseTypes Format
 * @param {Object} options - 可选参数。
 * @param {boolean} [options.keepData=false] - 如果设置为 true， data 属性会指向被解析的对象（例如 JSON 或 xml 数据对象）。
 * @param {Object} [options.data] - 当 keepData 属性设置为 true，这是传递给 read 操作的要被解析的字符串。
 * @usage
 */
class Format {


    constructor(options) {
        /**
         * @member {Object} Format.prototype.data
         * @description 当 keepData 属性设置为 true，这是传递给 read 操作的要被解析的字符串。
         */
        this.data = null;

        /**
         * @member {Object} [Format.prototype.keepData=false]
         * @description 保持最近读到的数据的引用（通过 data 属性）。
         */
        this.keepData = false;

        Util_Util.extend(this, options);
        this.options = options;

        this.CLASS_NAME = "SuperMap.Format";
    }

    /**
     * @function Format.prototype.destroy
     * @description 销毁该格式类，释放相关资源。
     */
    destroy() {
        //用来销毁该格式类，释放相关资源
    }

    /**
     * @function Format.prototype.read
     * @description 来从字符串中读取数据。
     * @param {string} data - 读取的数据。
     */
    read(data) { // eslint-disable-line no-unused-vars
        //用来从字符串中读取数据
    }

    /**
     * @function Format.prototype.write
     * @description 将对象写成字符串。
     * @param {Object} object - 可序列化的对象。
     * @returns {string} 对象转化后的字符串。
     */
    write(object) { // eslint-disable-line no-unused-vars
        //用来写字符串
    }
}

;// CONCATENATED MODULE: ./src/common/format/JSON.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class JSONFormat
 * @aliasclass Format.JSON
 * @deprecatedclass SuperMap.Format.JSON
 * @classdesc 安全的读写 JSON 的解析类。使用 {@link JSONFormat} 构造函数创建新实例。
 * @category BaseTypes Format
 * @param {Object} [options] - 可选参数。
 * @param {string} [options.indent="    "] - 用于格式化输出，indent 字符串会在每次缩进的时候使用一次。
 * @param {string} [options.space=" "] - 用于格式化输出，space 字符串会在名值对的 ":" 后边添加。
 * @param {string} [options.newline="\n"] - 用于格式化输出, newline 字符串会用在每一个名值对或数组项末尾。
 * @param {number} [options.level=0] - 用于格式化输出, 表示的是缩进级别。
 * @param {boolean} [options.pretty=false] - 是否在序列化的时候使用额外的空格控制结构。在 write 方法中使用。
 * @param {boolean} [options.nativeJSON] - 需要被注册的监听器对象。
 * @extends {Format}
 * @usage
 */
class JSONFormat extends Format {

    constructor(options) {
        super(options);
        /**
         * @member {string} [JSONFormat.prototype.indent="    "]
         * @description 用于格式化输出，indent 字符串会在每次缩进的时候使用一次。
         */
        this.indent = "    ";

        /**
         * @member {string} [JSONFormat.prototype.space=" "]
         * @description 用于格式化输出，space 字符串会在名值对的 ":" 后边添加。
         */
        this.space = " ";

        /**
         * @member {string} [JSONFormat.prototype.newline="\n"]
         * @description 用于格式化输出, newline 字符串会用在每一个名值对或数组项末尾。
         */
        this.newline = "\n";

        /**
         * @member {number} [JSONFormat.prototype.level=0]
         * @description 用于格式化输出, 表示的是缩进级别。
         */
        this.level = 0;

        /**
         * @member {boolean} [JSONFormat.prototype.pretty=false]
         * @description 是否在序列化的时候使用额外的空格控制结构。在 write 方法中使用。
         */
        this.pretty = false;

        /**
         * @member {boolean} JSONFormat.prototype.nativeJSON
         * @description 判断浏览器是否原生支持 JSON 格式数据。
         */
        this.nativeJSON = (function () {
            return !!(window.JSON && typeof JSON.parse === "function" && typeof JSON.stringify === "function");
        })();

        this.CLASS_NAME = "SuperMap.Format.JSON";
        /**
         * @member JSONFormat.prototype.serialize
         * @description 提供一些类型对象转 JSON 字符串的方法。
         */
        this.serialize = {
            /**
             * @function JSONFormat.serialize.object
             * @description 把对象转换为 JSON 字符串。
             * @param {Object} object - 可序列化的对象。
             * @returns {string} JSON 字符串。
             */
            'object': function (object) {
                // three special objects that we want to treat differently
                if (object == null) {
                    return "null";
                }
                if (object.constructor === Date) {
                    return this.serialize.date.apply(this, [object]);
                }
                if (object.constructor === Array) {
                    return this.serialize.array.apply(this, [object]);
                }
                var pieces = ['{'];
                this.level += 1;
                var key, keyJSON, valueJSON;

                var addComma = false;
                for (key in object) {
                    if (object.hasOwnProperty(key)) {
                        // recursive calls need to allow for sub-classing
                        keyJSON = this.write.apply(this,
                            [key, this.pretty]);
                        valueJSON = this.write.apply(this,
                            [object[key], this.pretty]);
                        if (keyJSON != null && valueJSON != null) {
                            if (addComma) {
                                pieces.push(',');
                            }
                            pieces.push(this.writeNewline(), this.writeIndent(),
                                keyJSON, ':', this.writeSpace(), valueJSON);
                            addComma = true;
                        }
                    }
                }

                this.level -= 1;
                pieces.push(this.writeNewline(), this.writeIndent(), '}');
                return pieces.join('');
            },

            /**
             * @function JSONFormat.serialize.array
             * @description 把数组转换成 JSON 字符串。
             * @param {Array} array - 可序列化的数组。
             * @returns {string} JSON 字符串。
             */
            'array': function (array) {
                var json;
                var pieces = ['['];
                this.level += 1;

                for (var i = 0, len = array.length; i < len; ++i) {
                    // recursive calls need to allow for sub-classing
                    json = this.write.apply(this,
                        [array[i], this.pretty]);
                    if (json != null) {
                        if (i > 0) {
                            pieces.push(',');
                        }
                        pieces.push(this.writeNewline(), this.writeIndent(), json);
                    }
                }

                this.level -= 1;
                pieces.push(this.writeNewline(), this.writeIndent(), ']');
                return pieces.join('');
            },

            /**
             * @function JSONFormat.serialize.string
             * @description 把字符串转换成 JSON 字符串。
             * @param {string} string - 可序列化的字符串。
             * @returns {string} JSON 字符串。
             */
            'string': function (string) {
                // If the string contains no control characters, no quote characters, and no
                // backslash characters, then we can simply slap some quotes around it.
                // Otherwise we must also replace the offending characters with safe
                // sequences.
                var m = {
                    '\b': '\\b',
                    '\t': '\\t',
                    '\n': '\\n',
                    '\f': '\\f',
                    '\r': '\\r',
                    '"': '\\"',
                    '\\': '\\\\'
                };
                /*eslint-disable no-control-regex*/
                if (/["\\\x00-\x1f]/.test(string)) {
                    return '"' + string.replace(/([\x00-\x1f\\"])/g, function (a, b) {
                        var c = m[b];
                        if (c) {
                            return c;
                        }
                        c = b.charCodeAt();
                        return '\\u00' +
                            Math.floor(c / 16).toString(16) +
                            (c % 16).toString(16);
                    }) + '"';
                }
                return '"' + string + '"';
            },

            /**
             * @function JSONFormat.serialize.number
             * @description 把数字转换成 JSON 字符串。
             * @param {number} number - 可序列化的数字。
             * @returns {string} JSON 字符串。
             */
            'number': function (number) {
                return isFinite(number) ? String(number) : "null";
            },

            /**
             * @function JSONFormat.serialize.boolean
             * @description Transform a boolean into a JSON string.
             * @param {boolean} bool - The boolean to be serialized.
             * @returns {string} A JSON string representing the boolean.
             */
            'boolean': function (bool) {
                return String(bool);
            },

            /**
             * @function JSONFormat.serialize.object
             * @description 将日期对象转换成 JSON 字符串。
             * @param {Date} date - 可序列化的日期对象。
             * @returns {string} JSON 字符串。
             */
            'date': function (date) {
                function format(number) {
                    // Format integers to have at least two digits.
                    return (number < 10) ? '0' + number : number;
                }

                return '"' + date.getFullYear() + '-' +
                    format(date.getMonth() + 1) + '-' +
                    format(date.getDate()) + 'T' +
                    format(date.getHours()) + ':' +
                    format(date.getMinutes()) + ':' +
                    format(date.getSeconds()) + '"';
            }
        };
    }

    /**
     * @function JSONFormat.prototype.read
     * @description 将一个符合 JSON 结构的字符串进行解析。
     * @param {string} json - 符合 JSON 结构的字符串。
     * @param {function} filter - 过滤方法，最终结果的每一个键值对都会调用该过滤方法，并在对应的值的位置替换成该方法返回的值。
     * @returns {(Object|string|Array|number|boolean)} 对象，数组，字符串或数字。
     */
    read(json, filter) {
        var object;
        if (this.nativeJSON) {
            try {
                object = JSON.parse(json, filter);
            } catch (e) {
                // Fall through if the regexp test fails.
                return { data: json}
            }
        }

        if (this.keepData) {
            this.data = object;
        }

        return object;
    }

    /**
     * @function JSONFormat.prototype.write
     * @description 序列化一个对象到一个符合 JSON 格式的字符串。
     * @param {Object|string|Array|number|boolean} value - 需要被序列化的对象，数组，字符串，数字，布尔值。
     * @param {boolean} [pretty=false] - 是否在序列化的时候使用额外的空格控制结构。在 write 方法中使用。
     * @returns {string} 符合 JSON 格式的字符串。
     *
     */
    write(value, pretty) {
        this.pretty = !!pretty;
        var json = null;
        var type = typeof value;
        if (this.serialize[type]) {
            try {
                json = (!this.pretty && this.nativeJSON) ?
                    JSON.stringify(value) :
                    this.serialize[type].apply(this, [value]);
            } catch (err) {
                //console.error("Trouble serializing: " + err);
            }
        }
        return json;
    }

    /**
     * @function JSONFormat.prototype.writeIndent
     * @description 根据缩进级别输出一个缩进字符串。
     * @private
     * @returns {string} 一个适当的缩进字符串。
     */
    writeIndent() {
        var pieces = [];
        if (this.pretty) {
            for (var i = 0; i < this.level; ++i) {
                pieces.push(this.indent);
            }
        }
        return pieces.join('');
    }

    /**
     * @function JSONFormat.prototype.writeNewline
     * @description 在格式化输出模式情况下输出代表新一行的字符串。
     * @private
     * @returns {string} 代表新的一行的字符串。
     */
    writeNewline() {
        return (this.pretty) ? this.newline : '';
    }

    /**
     * @function JSONFormat.prototype.writeSpace
     * @private
     * @description 在格式化输出模式情况下输出一个代表空格的字符串。
     * @returns {string} 空格字符串。
     */
    writeSpace() {
        return (this.pretty) ? this.space : '';
    }

}

;// CONCATENATED MODULE: ./src/common/format/GeoJSON.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/














/**
 * @class GeoJSONFormat
 * @aliasclass Format.GeoJSON
 * @deprecatedclass SuperMap.Format.GeoJSON
 * @classdesc  GeoJSON 的读和写。使用 {@link GeoJSONObject} 构造器创建一个 GeoJSON 解析器。
 * @category BaseTypes Format
 * @param {Object} [options] - 可选参数。
 * @param {string} [options.indent="    "] - 用于格式化输出，indent 字符串会在每次缩进的时候使用一次。
 * @param {string} [options.space=" "] - 用于格式化输出，space 字符串会在名值对的 ":" 后边添加。
 * @param {string} [options.newline="\n"] - 用于格式化输出, newline 字符串会用在每一个名值对或数组项末尾。
 * @param {number} [options.level=0] - 用于格式化输出, 表示的是缩进级别。
 * @param {boolean} [options.pretty=false] - 是否在序列化的时候使用额外的空格控制结构。在 write 方法中使用。
 * @param {boolean} [options.nativeJSON] - 需要被注册的监听器对象。
 * @param {boolean} [options.ignoreExtraDims=true] - 忽略维度超过 2 的几何要素。
 * @extends {JSONFormat}
 * @usage
 */
class GeoJSON extends JSONFormat {


    constructor(options) {
        super(options);
        /**
         * @member {boolean} [GeoJSONFormat.prototype.ignoreExtraDims=true]
         * @description 忽略维度超过 2 的几何要素。
         */
        this.ignoreExtraDims = true;

        this.CLASS_NAME = "SuperMap.Format.GeoJSON";
        /**
         * @member {Object} GeoJSONFormat.prototype.parseCoords
         * @private
         * @description 一个属性名对应着 GeoJSON 对象的几何类型的对象。每个属性其实都是一个实际上做解析用的方法。
         */
        this.parseCoords = {
            /**
             * @function GeoJSONFormat.parseCoords.point
             * @description 将一组坐标转成一个 {@link Geometry} 对象。
             * @param {Object} array - GeoJSON 片段中的一组坐标。
             * @returns {Geometry} 一个几何对象。
             */
            "point": function (array) {
                if (this.ignoreExtraDims === false &&
                    array.length != 2) {
                    throw "Only 2D points are supported: " + array;
                }
                return new Point(array[0], array[1]);
            },

            /**
             * @function GeoJSONFormat.parseCoords.multipoint
             * @description 将坐标组数组转化成为一个 {@link Geometry} 对象。
             * @param {Object} array - GeoJSON 片段中的坐标组数组。
             * @returns {Geometry} 一个几何对象。
             */
            "multipoint": function (array) {
                var points = [];
                var p = null;
                for (var i = 0, len = array.length; i < len; ++i) {
                    try {
                        p = this.parseCoords["point"].apply(this, [array[i]]);
                    } catch (err) {
                        throw err;
                    }
                    points.push(p);
                }
                return new MultiPoint(points);
            },

            /**
             * @function GeoJSONFormat.parseCoords.linestring
             * @description 将坐标组数组转化成为一个 {@link Geometry} 对象。
             * @param {Object} array - GeoJSON 片段中的坐标组数组。
             * @returns {Geometry} 一个几何对象。
             */
            "linestring": function (array) {
                var points = [];
                var p = null;
                for (var i = 0, len = array.length; i < len; ++i) {
                    try {
                        p = this.parseCoords["point"].apply(this, [array[i]]);
                    } catch (err) {
                        throw err;
                    }
                    points.push(p);
                }
                return new LineString(points);
            },

            /**
             * @function GeoJSONFormat.parseCoords.multilinestring
             * @description 将坐标组数组转化成为一个 {@link Geometry} 对象。
             * @param {Object} array - GeoJSON 片段中的坐标组数组。
             * @returns {Geometry} 一个几何对象。
             */
            "multilinestring": function (array) {
                var lines = [];
                var l = null;
                for (var i = 0, len = array.length; i < len; ++i) {
                    try {
                        l = this.parseCoords["linestring"].apply(this, [array[i]]);
                    } catch (err) {
                        throw err;
                    }
                    lines.push(l);
                }
                return new MultiLineString(lines);
            },

            /**
             * @function GeoJSONFormat.parseCoords.polygon
             * @description 将坐标组数组转化成为一个 {@link Geometry} 对象。
             * @returns {Geometry} 一个几何对象。
             */
            "polygon": function (array) {
                var rings = [];
                var r, l;
                for (var i = 0, len = array.length; i < len; ++i) {
                    try {
                        l = this.parseCoords["linestring"].apply(this, [array[i]]);
                    } catch (err) {
                        throw err;
                    }
                    r = new LinearRing_LinearRing(l.components);
                    rings.push(r);
                }
                return new Polygon_Polygon(rings);
            },

            /**
             * @function GeoJSONFormat.parseCoords.multipolygon
             * @description 将坐标组数组转化成为一个 {@link Geometry} 对象。
             * @param {Object} array - GeoJSON 片段中的坐标组数组。
             * @returns {Geometry} 一个几何对象。
             */
            "multipolygon": function (array) {
                var polys = [];
                var p = null;
                for (var i = 0, len = array.length; i < len; ++i) {
                    try {
                        p = this.parseCoords["polygon"].apply(this, [array[i]]);
                    } catch (err) {
                        throw err;
                    }
                    polys.push(p);
                }
                return new MultiPolygon(polys);
            },

            /**
             * @function GeoJSONFormat.parseCoords.box
             * @description 将坐标组数组转化成为一个 {@link Geometry} 对象。
             * @param {Array} array - GeoJSON 片段中的坐标组数组。
             * @returns {Geometry} 一个几何对象。
             */
            "box": function (array) {
                if (array.length != 2) {
                    throw "GeoJSON box coordinates must have 2 elements";
                }
                return new Polygon_Polygon([
                    new LinearRing_LinearRing([
                        new Point(array[0][0], array[0][1]),
                        new Point(array[1][0], array[0][1]),
                        new Point(array[1][0], array[1][1]),
                        new Point(array[0][0], array[1][1]),
                        new Point(array[0][0], array[0][1])
                    ])
                ]);
            }

        };
        /**
         * @member {Object} GeoJSONFormat.prototype.extract
         * @private
         * @description 一个属性名对应着GeoJSON类型的对象。其值为相应的实际的解析方法。
         */
        this.extract = {
            /**
             * @function GeoJSONFormat.extract.feature
             * @description 返回一个表示单个要素对象的 GeoJSON 的一部分。
             * @param {SuperMap.ServerFeature} feature - iServer 要素对象。
             * @returns {Object} 一个表示点的对象。
             */
            'feature': function (feature) {
                var geom = this.extract.geometry.apply(this, [feature.geometry]);
                var json = {
                    "type": "Feature",
                    "properties": this.createAttributes(feature),
                    "geometry": geom
                };

                if (feature.geometry && feature.geometry.type === 'TEXT') {
                    json.properties.texts = feature.geometry.texts;
                    json.properties.textStyle = feature.geometry.textStyle;
                }
                if (feature.fid) {
                    json.id = feature.fid;
                }
                if (feature.ID) {
                    json.id = feature.ID;
                }
                return json;
            },


            /**
             * @function GeoJSONFormat.extract.geometry
             * @description 返回一个表示单个几何对象的 GeoJSON 的一部分。
             * @param {Object} geometry - iServer 几何对象。
             * @returns {Object} 一个表示几何体的对象。
             */
            'geometry': function (geometry) {
                if (geometry == null) {
                    return null;
                }
                if (!geometry.parts && geometry.points) {
                    geometry.parts = [geometry.points.length];
                }
                var geo = geometry.hasOwnProperty('geometryType')
                    ? geometry
                    : new ServerGeometry(geometry).toGeometry() || geometry;
                var geometryType = geo.geometryType || geo.type;
                var data;
                if (geometryType === "LinearRing") {
                    geometryType = "LineString";
                }
                if (geometryType === "LINEM") {
                    geometryType = "MultiLineString";
                }
                data = this.extract[geometryType.toLowerCase()].apply(this, [geo]);
                geometryType = geometryType === 'TEXT' ? 'Point' : geometryType;
                var json;
                if (geometryType === "Collection") {
                    json = {
                        "type": "GeometryCollection",
                        "geometries": data
                    };
                } else {
                    json = {
                        "type": geometryType,
                        "coordinates": data
                    };
                }
                return json;
            },


            /**
             * @function GeoJSONFormat.extract.point
             * @description 从一个点对象中返回一个坐标组。
             * @param {GeometryPoint} point - 一个点对象。
             * @returns {Array} 一个表示一个点的坐标组。
             */
            'point': function (point) {
                var p = [point.x, point.y];
                for (var name in point) {
                    if (name !== "x" && name !== "y" && point[name] !== null && !isNaN(point[name])) {
                        p.push(point[name]);
                    }
                }
                return p;
            },

            /**
             * @function GeoJSONFormat.extract.point
             * @description 从一个文本对象中返回一个坐标组。
             * @param {Object} geo - 一个文本对象。
             * @returns {Array} 一个表示一个点的坐标组。
             */
            'text': function (geo) {
                return [geo.points[0].x, geo.points[0].y];
            },

            /**
             * @function GeoJSONFormat.extract.multipoint
             * @description 从一个多点对象中返一个坐标组数组。
             * @param {GeometryMultiPoint} multipoint - 多点对象。
             * @returns {Array} 一个表示多点的坐标组数组。
             */
            'multipoint': function (multipoint) {
                var array = [];
                for (var i = 0, len = multipoint.components.length; i < len; ++i) {
                    array.push(this.extract.point.apply(this, [multipoint.components[i]]));
                }
                return array;
            },

            /**
             * @function GeoJSONFormat.extract.linestring
             * @description 从一个线对象中返回一个坐标组数组。
             * @param {Linestring} linestring - 线对象。
             * @returns {Array} 一个表示线对象的坐标组数组。
             */
            'linestring': function (linestring) {
                var array = [];
                for (var i = 0, len = linestring.components.length; i < len; ++i) {
                    array.push(this.extract.point.apply(this, [linestring.components[i]]));
                }
                return array;
            },

            /**
             * @function GeoJSONFormat.extract.multilinestring
             * @description 从一个多线对象中返回一个线数组。
             * @param {GeometryMultiLineString} multilinestring - 多线对象。
             *
             * @returns {Array} 一个表示多线的线数组。
             */
            'multilinestring': function (multilinestring) {
                var array = [];
                for (var i = 0, len = multilinestring.components.length; i < len; ++i) {
                    array.push(this.extract.linestring.apply(this, [multilinestring.components[i]]));
                }
                return array;
            },

            /**
             * @function GeoJSONFormat.extract.polygon
             * @description 从一个面对象中返回一组线环。
             * @param {GeometryPolygon} polygon - 面对象。
             * @returns {Array} 一组表示面的线环。
             */
            'polygon': function (polygon) {
                var array = [];
                for (var i = 0, len = polygon.components.length; i < len; ++i) {
                    array.push(this.extract.linestring.apply(this, [polygon.components[i]]));
                }
                return array;
            },

            /**
             * @function GeoJSONFormat.extract.multipolygon
             * @description 从一个多面对象中返回一组面。
             * @param {GeometryMultiPolygon} multipolygon - 多面对象。
             * @returns {Array} 一组表示多面的面。
             */
            'multipolygon': function (multipolygon) {
                var array = [];
                for (var i = 0, len = multipolygon.components.length; i < len; ++i) {
                    array.push(this.extract.polygon.apply(this, [multipolygon.components[i]]));
                }
                return array;
            },

            /**
             * @function GeoJSONFormat.extract.collection
             * @description 从一个几何要素集合中一组几何要素数组。
             * @param {GeometryCollection} collection - 几何要素集合。
             * @returns {Array} 一组表示几何要素集合的几何要素数组。
             */
            'collection': function (collection) {
                var len = collection.components.length;
                var array = new Array(len);
                for (var i = 0; i < len; ++i) {
                    array[i] = this.extract.geometry.apply(this, [collection.components[i]]);
                }
                return array;
            }
        };
    }

    /**
     * @function GeoJSONFormat.prototype.read
     * @description 将 GeoJSON 对象或者GeoJSON 对象字符串转换为 SuperMap Feature 对象。
     * @param {GeoJSONObject} json - GeoJSON 对象。
     * @param {string} [type='FeaureCollection'] - 可选的字符串，它决定了输出的格式。支持的值有："Geometry","Feature"，和 "FeatureCollection"，如果此值为null。
     * @param {function} filter - 对象中每个层次每个键值对都会调用此函数得出一个结果。每个值都会被 filter 函数的结果所替换掉。这个函数可被用来将某些对象转化成某个类相应的对象，或者将日期字符串转化成Date对象。
     * @returns {Object}  返回值依赖于 type 参数的值。
     *     -如果 type 等于 "FeatureCollection"，返回值将会是 {@link FeatureVector} 数组。
     *     -如果 type 为 "Geometry",输入的 JSON 对象必须表示一个唯一的几何体，然后返回值就会是 {@link Geometry}。
     *     -如果 type 为 "Feature"，输入的 JSON 对象也必须表示的一个要素，这样返回值才会是 {@link FeatureVector}。
     */

    read(json, type, filter) {
        type = (type) ? type : "FeatureCollection";
        var results = null;
        var obj = null;
        if (typeof json == "string") {
            obj = super.read(json, filter);
        } else {
            obj = json;
        }
        if (!obj) {
            //console.error("Bad JSON: " + json);
        } else if (typeof (obj.type) != "string") {
            //console.error("Bad GeoJSON - no type: " + json);
        } else if (this.isValidType(obj, type)) {
            switch (type) {
                case "Geometry":
                    try {
                        results = this.parseGeometry(obj);
                    } catch (err) {
                        //console.error(err);
                    }
                    break;
                case "Feature":
                    try {
                        results = this.parseFeature(obj);
                        results.type = "Feature";
                    } catch (err) {
                        //console.error(err);
                    }
                    break;
                case "FeatureCollection":
                    // for type FeatureCollection, we allow input to be any type
                    results = [];
                    switch (obj.type) {
                        case "Feature":
                            try {
                                results.push(this.parseFeature(obj));
                            } catch (err) {
                                results = null;
                                //console.error(err);
                            }
                            break;
                        case "FeatureCollection":
                            for (var i = 0, len = obj.features.length; i < len; ++i) {
                                try {
                                    results.push(this.parseFeature(obj.features[i]));
                                } catch (err) {
                                    results = null;
                                    // console.error(err);
                                }
                            }
                            break;
                        default:
                            try {
                                var geom = this.parseGeometry(obj);
                                results.push(new Vector(geom));
                            } catch (err) {
                                results = null;
                                //console.error(err);
                            }
                    }
                    break;
                default:
                    break;
            }
        }
        return results;
    }

    /**
     * @function GeoJSONFormat.prototype.write
     * @description iServer Geometry JSON 对象 转 GeoJSON对象字符串。
     * @param {Object} obj - iServer Geometry JSON 对象。
     * @param {boolean} [pretty=false] - 是否使用换行和缩进来控制输出。
     * @returns {GeoJSONObject} 一个 GeoJSON 字符串，它表示了输入的几何对象，要素对象，或者要素对象数组。
     */
    write(obj, pretty) {
        return super.write(this.toGeoJSON(obj), pretty);
    }
    /**
     * @function GeoJSONFormat.prototype.fromGeoJSON
     * @version 9.1.1
     * @description 将 GeoJSON 对象或者GeoJSON 对象字符串转换为iServer Feature JSON。
     * @param {GeoJSONObject} json - GeoJSON 对象。
     * @param {string} [type='FeaureCollection'] - 可选的字符串，它决定了输出的格式。支持的值有："Geometry","Feature"，和 "FeatureCollection"，如果此值为null。
     * @param {function} filter - 对象中每个层次每个键值对都会调用此函数得出一个结果。每个值都会被 filter 函数的结果所替换掉。这个函数可被用来将某些对象转化成某个类相应的对象，或者将日期字符串转化成Date对象。
     * @returns {Object}  iServer Feature JSON。
     */
    fromGeoJSON(json, type, filter) {
        let feature = this.read(json, type, filter);
        if (!Util_Util.isArray(feature)) {
            return this._toiSevrerFeature(feature);
        }
        return feature.map((element) => {
            return this._toiSevrerFeature(element);
        })
    }

    /**
     * @function GeoJSONFormat.prototype.toGeoJSON
     * @version 9.1.1
     * @description 将 iServer Feature JSON 对象转换为 GeoJSON 对象。
     * @param {Object} obj - iServer Feature JSON。
     * @returns {GeoJSONObject}  GeoJSON 对象。
     */
    toGeoJSON(obj) {
        var geojson = {
            "type": null
        };
        if (Util_Util.isArray(obj)) {
            geojson.type = "FeatureCollection";
            var numFeatures = obj.length;
            geojson.features = new Array(numFeatures);
            for (var i = 0; i < numFeatures; ++i) {
                var element = obj[i];
                if (isGeometry(element)) {
                    let feature = {};
                    feature.geometry = element;
                    geojson.features[i] = this.extract.feature.apply(this, [feature]);
                } else {
                    geojson.features[i] = this.extract.feature.apply(this, [element]);
                }
            }
        } else if (isGeometry(obj)) {
            let feature = {};
            feature.geometry = obj;
            geojson = this.extract.feature.apply(this, [feature]);
        } else {
            geojson = this.extract.feature.apply(this, [obj]);
        }

        function isGeometry(input) {
            return (input.hasOwnProperty("parts") && input.hasOwnProperty("points")) || input.hasOwnProperty("geoParts");
        }

        return geojson;

    }
    /**
     *  @function GeoJSONFormat.prototype.isValidType
     *  @description 检查一个 GeoJSON 对象是否和给定的类型相符的合法的对象。
     *  @returns {boolean} GeoJSON 是否是给定类型的合法对象。
     *  @private
     */
    isValidType(obj, type) {
        var valid = false;
        switch (type) {
            case "Geometry":
                if (Util_Util.indexOf(
                    ["Point", "MultiPoint", "LineString", "MultiLineString",
                        "Polygon", "MultiPolygon", "Box", "GeometryCollection"
                    ],
                    obj.type) == -1) {
                    // unsupported geometry type
                    //console.error("Unsupported geometry type: " +
                    // obj.type);
                } else {
                    valid = true;
                }
                break;
            case "FeatureCollection":
                // allow for any type to be converted to a feature collection
                valid = true;
                break;
            default:
                // for Feature types must match
                if (obj.type == type) {
                    valid = true;
                } else {
                    //console.error("Cannot convert types from " +
                    //obj.type + " to " + type);
                }
        }
        return valid;
    }

    /**
     * @function GeoJSONFormat.prototype.parseFeature
     * @description 将一个 GeoJSON 中的 feature 转化成 {@link FeatureVector}> 对象。
     * @private
     * @param {GeoJSONObject} obj - 从 GeoJSON 对象中创建一个对象。
     * @returns {FeatureVector} 一个要素。
     */
    parseFeature(obj) {
        var feature, geometry, attributes, bbox;
        attributes = (obj.properties) ? obj.properties : {};
        bbox = (obj.geometry && obj.geometry.bbox) || obj.bbox;
        try {
            geometry = this.parseGeometry(obj.geometry);
        } catch (err) {
            // deal with bad geometries
            throw err;
        }
        feature = new Vector(geometry, attributes);
        if (bbox) {
            feature.bounds = Bounds.fromArray(bbox);
        }
        if (obj.id) {
            feature.geometry.id = obj.id;
            feature.fid = obj.id;
        }
        return feature;
    }


    /**
     * @function GeoJSONFormat.prototype.parseGeometry
     * @description 将一个 GeoJSON 中的几何要素转化成 {@link Geometry} 对象。
     * @param {GeoJSONObject} obj - 从 GeoJSON 对象中创建一个对象。
     * @returns {Geometry} 一个几何要素。
     * @private
     */
    parseGeometry(obj) {
        if (obj == null) {
            return null;
        }
        var geometry;
        if (obj.type == "GeometryCollection") {
            if (!(Util_Util.isArray(obj.geometries))) {
                throw "GeometryCollection must have geometries array: " + obj;
            }
            var numGeom = obj.geometries.length;
            var components = new Array(numGeom);
            for (var i = 0; i < numGeom; ++i) {
                components[i] = this.parseGeometry.apply(
                    this, [obj.geometries[i]]
                );
            }
            geometry = new Collection(components);
        } else {
            if (!(Util_Util.isArray(obj.coordinates))) {
                throw "Geometry must have coordinates array: " + obj;
            }
            if (!this.parseCoords[obj.type.toLowerCase()]) {
                throw "Unsupported geometry type: " + obj.type;
            }
            try {
                geometry = this.parseCoords[obj.type.toLowerCase()].apply(
                    this, [obj.coordinates]
                );
            } catch (err) {
                // deal with bad coordinates
                throw err;
            }
        }
        return geometry;
    }


    /**
     * @function GeoJSONFormat.prototype.createCRSObject
     * @description 从一个要素对象中创建一个坐标参考系对象。
     * @param {FeatureVector} object - 要素对象。
     * @private
     * @returns {GeoJSONObject} 一个可作为 GeoJSON 对象的 CRS 属性使用的对象。
     */
    createCRSObject(object) {
        var proj = object.layer.projection.toString();
        var crs = {};
        if (proj.match(/epsg:/i)) {
            var code = parseInt(proj.substring(proj.indexOf(":") + 1));
            if (code == 4326) {
                crs = {
                    "type": "name",
                    "properties": {
                        "name": "urn:ogc:def:crs:OGC:1.3:CRS84"
                    }
                };
            } else {
                crs = {
                    "type": "name",
                    "properties": {
                        "name": "EPSG:" + code
                    }
                };
            }
        }
        return crs;
    }
    _toiSevrerFeature(feature) {
        const attributes = feature.attributes;
        const attrNames = [];
        const attrValues = [];
        for (var attr in attributes) {
            attrNames.push(attr);
            attrValues.push(attributes[attr]);
        }
        const newFeature = {
            fieldNames: attrNames,
            fieldValues: attrValues,
            geometry: ServerGeometry.fromGeometry(feature.geometry)
        };
        newFeature.geometry.id = feature.fid;
        return newFeature;
    }
    createAttributes(feature) {
        if (!feature) {
            return null;
        }
        var attr = {};
        processFieldsAttributes(feature, attr);
        var exceptKeys = ["fieldNames", "fieldValues", "geometry", "stringID", "ID"];
        for (var key in feature) {
            if (exceptKeys.indexOf(key) > -1) {
                continue;
            }
            attr[key] = feature[key];
        }

        function processFieldsAttributes(feature, attributes) {
            if (!(feature.hasOwnProperty("fieldNames") && feature.hasOwnProperty("fieldValues"))) {
                return;
            }
            var names = feature.fieldNames,
                values = feature.fieldValues;
            for (var i in names) {
                attributes[names[i]] = values[i];
            }
        }

        return attr;
    }
}

;// CONCATENATED MODULE: ./src/common/format/WKT.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/











/**
 * @class WKTFormat
 * @aliasclass Format.WKT
 * @deprecatedclass SuperMap.Format.WKT
 * @classdesc 用于读写常见文本的类。通过 {@link WKTFormat} 构造器来创建一个新的实例。
 * @category BaseTypes Format
 * @extends {Format}
 * @param {Object} options - 可选的选项对象，其属性将被设置到实例。option 具体配置项继承自 {@link Format}。
 * @usage
 */
class WKT extends Format {

    constructor(options) {
        super(options);
        this.regExes = {
            'typeStr': /^\s*(\w+)\s*\(\s*(.*)\s*\)\s*$/,
            'spaces': /\s+/,
            'parenComma': /\)\s*,\s*\(/,
            'doubleParenComma': /\)\s*\)\s*,\s*\(\s*\(/,  // can't use {2} here
            'trimParens': /^\s*\(?(.*?)\)?\s*$/
        };
        this.CLASS_NAME = "SuperMap.Format.WKT"; /**
         * @private
         * @description Object with properties corresponding to the geometry types.
         * Property values are functions that do the actual data extraction.
         */
        this.extract = {
            /**
             * @description Return a space delimited string of point coordinates.
             * @param {GeometryPoint} point
             * @returns  {string} A string of coordinates representing the point
             */
            'point': function (point) {
                return point.x + ' ' + point.y;
            },

            /**
             * @description  Return a comma delimited string of point coordinates from a multipoint.
             * @param {GeometryMultiPoint} multipoint
             * @returns  {string} A string of point coordinate strings representing
             *                  the multipoint
             */
            'multipoint'(multipoint) {
                var array = [];
                for (var i = 0, len = multipoint.components.length; i < len; ++i) {
                    array.push('(' +
                        this.extract.point.apply(this, [multipoint.components[i]]) +
                        ')');
                }
                return array.join(',');
            },

            /**
             * @description  Return a comma delimited string of point coordinates from a line.
             * @param {GeometryLineString} linestring
             * @returns  {string} A string of point coordinate strings representing
             *                  the linestring
             */
            'linestring'(linestring) {
                var array = [];
                for (var i = 0, len = linestring.components.length; i < len; ++i) {
                    array.push(this.extract.point.apply(this, [linestring.components[i]]));
                }
                return array.join(',');
            },

            /**
             * @description  Return a comma delimited string of linestring strings from a multilinestring.
             * @param {GeometryMultiLineString} multilinestring
             * @returns  {string} A string of of linestring strings representing
             *                  the multilinestring
             */
            'multilinestring'(multilinestring) {
                var array = [];
                for (var i = 0, len = multilinestring.components.length; i < len; ++i) {
                    array.push('(' +
                        this.extract.linestring.apply(this, [multilinestring.components[i]]) +
                        ')');
                }
                return array.join(',');
            },

            /**
             * @description  Return a comma delimited string of linear ring arrays from a polygon.
             * @param {GeometryPolygon} polygon
             * @returns  {string} An array of linear ring arrays representing the polygon
             */
            'polygon'(polygon) {
                var array = [];
                for (var i = 0, len = polygon.components.length; i < len; ++i) {
                    array.push('(' +
                        this.extract.linestring.apply(this, [polygon.components[i]]) +
                        ')');
                }
                return array.join(',');
            },

            /**
             * @description  Return an array of polygon arrays from a multipolygon.
             * @param {GeometryMultiPolygon} multipolygon
             * @returns  {string} An array of polygon arrays representing
             *                  the multipolygon
             */
            'multipolygon'(multipolygon) {
                var array = [];
                for (var i = 0, len = multipolygon.components.length; i < len; ++i) {
                    array.push('(' +
                        this.extract.polygon.apply(this, [multipolygon.components[i]]) +
                        ')');
                }
                return array.join(',');
            },

            /**
             * @description  Return the WKT portion between 'GEOMETRYCOLLECTION(' and ')' for an <GeometryCollection>
             * @param {GeometryCollection} collection
             * @returns  {string} internal WKT representation of the collection
             */
            'collection'(collection) {
                var array = [];
                for (var i = 0, len = collection.components.length; i < len; ++i) {
                    array.push(this.extractGeometry.apply(this, [collection.components[i]]));
                }
                return array.join(',');
            }

        };

        /**
         * @private
         * @description Object with properties corresponding to the geometry types.
         * Property values are functions that do the actual parsing.
         */
        this.parse = {
            /**
             * @private
             * @description  Return point feature given a point WKT fragment.
             * @param {string} str A WKT fragment representing the point
             * @returns  {FeatureVector} A point feature
             *
             */
            'point': function (str) {
                var coords = StringExt.trim(str).split(this.regExes.spaces);
                return new Vector(new Point(coords[0], coords[1])
                );
            },

            /**
             * @description  Return a multipoint feature given a multipoint WKT fragment.
             * @param {string} A WKT fragment representing the multipoint
             * @returns  {FeatureVector} A multipoint feature
             * @private
             */
            'multipoint': function (str) {
                var point;
                var points = StringExt.trim(str).split(',');
                var components = [];
                for (var i = 0, len = points.length; i < len; ++i) {
                    point = points[i].replace(this.regExes.trimParens, '$1');
                    components.push(this.parse.point.apply(this, [point]).geometry);
                }
                return new Vector(
                    new MultiPoint(components)
                );
            },

            /**
             * @description  Return a linestring feature given a linestring WKT fragment.
             * @param {string} A WKT fragment representing the linestring
             * @returns  {FeatureVector} A linestring feature
             * @private
             */
            'linestring': function (str) {
                var points = StringExt.trim(str).split(',');
                var components = [];
                for (var i = 0, len = points.length; i < len; ++i) {
                    components.push(this.parse.point.apply(this, [points[i]]).geometry);
                }
                return new Vector(
                    new LineString(components)
                );
            },

            /**
             * @description  Return a multilinestring feature given a multilinestring WKT fragment.
             * @param {string} A WKT fragment representing the multilinestring
             * @returns  {FeatureVector} A multilinestring feature
             * @private
             */
            'multilinestring': function (str) {
                var line;
                var lines = StringExt.trim(str).split(this.regExes.parenComma);
                var components = [];
                for (var i = 0, len = lines.length; i < len; ++i) {
                    line = lines[i].replace(this.regExes.trimParens, '$1');
                    components.push(this.parse.linestring.apply(this, [line]).geometry);
                }
                return new Vector(
                    new MultiLineString(components)
                );
            },

            /**
             * @description  Return a polygon feature given a polygon WKT fragment.
             * @param {string} A WKT fragment representing the polygon
             * @returns  {FeatureVector} A polygon feature
             * @private
             */
            'polygon': function (str) {
                var ring, linestring, linearring;
                var rings = StringExt.trim(str).split(this.regExes.parenComma);
                var components = [];
                for (var i = 0, len = rings.length; i < len; ++i) {
                    ring = rings[i].replace(this.regExes.trimParens, '$1');
                    linestring = this.parse.linestring.apply(this, [ring]).geometry;
                    linearring = new LinearRing_LinearRing(linestring.components);
                    components.push(linearring);
                }
                return new Vector(
                    new Polygon_Polygon(components)
                );
            },

            /**
             * @private
             * @description  Return a multipolygon feature given a multipolygon WKT fragment.
             * @param {string} A WKT fragment representing the multipolygon
             * @returns  {FeatureVector} A multipolygon feature
             *
             */
            'multipolygon': function (str) {
                var polygon;
                var polygons = StringExt.trim(str).split(this.regExes.doubleParenComma);
                var components = [];
                for (var i = 0, len = polygons.length; i < len; ++i) {
                    polygon = polygons[i].replace(this.regExes.trimParens, '$1');
                    components.push(this.parse.polygon.apply(this, [polygon]).geometry);
                }
                return new Vector(
                    new MultiPolygon(components)
                );
            },


            /**
             * @description  Return an array of features given a geometrycollection WKT fragment.
             * @param {string} A WKT fragment representing the geometrycollection
             * @returns  {Array} An array of FeatureVector
             * @private
             */
            'geometrycollection': function (str) {
                // separate components of the collection with |
                str = str.replace(/,\s*([A-Za-z])/g, '|$1');
                var wktArray = StringExt.trim(str).split('|');
                var components = [];
                for (var i = 0, len = wktArray.length; i < len; ++i) {
                    components.push(this.read(wktArray[i]));
                }
                return components;
            }

        };

    }


    /**
     * @function WKTFormat.prototype.read
     * @description 反序列化 WKT 字符串并返回向量特征或向量特征数组。支持 POINT、MULTIPOINT、LINESTRING、MULTILINESTRING、POLYGON、MULTIPOLYGON 和 GEOMETRYCOLLECTION 的 WKT。
     * @param {string} wkt - WKT 字符串。
     * @returns {FeatureVector|Array} GEOMETRYCOLLECTION WKT 的矢量要素或者矢量要素数组。
     */
    read(wkt) {
        var features, type, str;
        wkt = wkt.replace(/[\n\r]/g, " ");
        var matches = this.regExes.typeStr.exec(wkt);
        if (matches) {
            type = matches[1].toLowerCase();
            str = matches[2];
            if (this.parse[type]) {
                features = this.parse[type].apply(this, [str]);
            }
        }
        return features;
    }


    /**
     * @function WKTFormat.prototype.write
     * @description 将矢量要素或矢量要素数组序列化为 WKT 字符串。
     * @param {(FeatureVector|Array)} features - 矢量要素或矢量要素数组。
     * @returns {string} 表示几何的 WKT 字符串。
     */
    write(features) {
        var collection, geometry, isCollection;
        if (features.constructor === Array) {
            collection = features;
            isCollection = true;
        } else {
            collection = [features];
            isCollection = false;
        }
        var pieces = [];
        if (isCollection) {
            pieces.push('GEOMETRYCOLLECTION(');
        }
        for (var i = 0, len = collection.length; i < len; ++i) {
            if (isCollection && i > 0) {
                pieces.push(',');
            }
            geometry = collection[i].geometry;
            pieces.push(this.extractGeometry(geometry));
        }
        if (isCollection) {
            pieces.push(')');
        }
        return pieces.join('');
    }

    /**
     * @function WKTFormat.prototype.extractGeometry
     * @description 为单个 Geometry 对象构造 WKT 的入口点。
     * @param {Geometry} geometry - Geometry 对象。
     * @returns {string} 表示几何的 WKT 字符串。
     */
    extractGeometry(geometry) {
        var type = geometry.CLASS_NAME.split('.')[2].toLowerCase();
        if (!this.extract[type]) {
            return null;
        }
        var wktType = type === 'collection' ? 'GEOMETRYCOLLECTION' : type.toUpperCase();
        var data = wktType + '(' + this.extract[type].apply(this, [geometry]) + ')';
        return data;
    }
}

;// CONCATENATED MODULE: ./src/common/format/index.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/









;// CONCATENATED MODULE: ./src/common/control/TimeControlBase.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class TimeControlBase
 * @deprecatedclass SuperMap.TimeControlBase
 * @classdesc 时间控制基类。
 * @modulecategory Control
 * @category  Control
 * @param {Object} options - 可选参数。
 * @param {number} [options.speed=1] - 速度。不能小于 0，（每帧渲染的数据之间的间隔为1），设置越大速度越快。
 * @param {number} [options.frequency=1000] - 刷新频率（单位 ms），服务器刷新的时间间隔。
 * @param {number} [options.startTime=0] - 起始时间，必须为数字，且小于等于 endTime。如果不设置，初始化时为 0，建议设置。
 * @param {number} [options.endTime] - 结束时间，必须为数字，且大于等于 startTime。如果不设置，初始化时以当前时间进行设置，建议设置。
 * @param {boolean} [options.repeat=true] - 是否重复循环。
 * @param {boolean} [options.reverse=false] - 是否反向。
 * @usage
 */
class TimeControlBase {


    constructor(options) {
        //设置步长，刷新频率、开始结束时间、是否循环、是否反向
        var me = this;
        options = options || {};


        /**
         * @member {number} [TimeControlBase.prototype.speed=1]
         * @description 步长，必须为非负数，默认为1（表示前后两次渲染的数据之间的间隔为1）。
         */
        this.speed = (options.speed && options.speed >= 0) ? options.speed : 1;

        /**
         * @member {number} [TimeControlBase.prototype.frequency=1000]
         * @description 刷新频率(单位ms)，服务器刷新的时间间隔。
         */
        this.frequency = (options.speed && options.frequency >= 0) ? options.frequency : 1000;

        /**
         * @member {number} [TimeControlBase.prototype.startTime=0]
         * @description 记录的起始时间，必须为数字，
         *              如果不设置，初始化时为0，建议设置。
         */
        this.startTime = (options.startTime && options.startTime != null) ? options.startTime : 0;

        /**
         * @member {number} TimeControlBase.prototype.endTime
         * @description 记录的结束时间，必须为数字，
         *              如果不设置，初始化时以当前时间进行设置，建议设置。
         */
        this.endTime = (options.endTime && options.endTime != null && options.endTime >= me.startTime) ? options.endTime : +new Date();

        /**
         * @member {boolean} [TimeControlBase.prototype.repeat=true]
         * @description 是否重复循环。
         */
        this.repeat = (options.repeat !== undefined) ? options.repeat : true;

        /**
         * @member {boolean} [TimeControlBase.prototype.reverse=false]
         * @description 是否反向。
         */
        this.reverse = (options.reverse !== undefined) ? options.reverse : false;

        /**
         * @member {number} TimeControlBase.prototype.currentTime
         * @description 记录近期的时间，也就是当前帧运行到的时间。
         */
        this.currentTime = null;

        /**
         * @member {number} TimeControlBase.prototype.oldTime
         * @description 记录上一帧的时间，也就是之前运行到的时间。
         */
        this.oldTime = null;

        /**
         * @member {boolean} [TimeControlBase.prototype.running=false]
         * @description 记录当前是否处于运行中。
         */
        this.running = false;


        /**
         * @private
         * @member {Array.<string>} TimeControlBase.prototype.EVENT_TYPES
         * @description 此类支持的事件类型。
         *
         */
        this.EVENT_TYPES = ["start", "pause", "stop"];

        /**
         * @private
         * @member {Events} TimeControlBase.prototype.events
         * @description 事件
         */
        me.events = new Events(this, null, this.EVENT_TYPES);

        me.speed = Number(me.speed);
        me.frequency = Number(me.frequency);
        me.startTime = Number(me.startTime);
        me.endTime = Number(me.endTime);

        me.startTime = Date.parse(new Date(me.startTime));
        me.endTime = Date.parse(new Date(me.endTime));

        //初始化当前时间
        me.currentTime = me.startTime;

        this.CLASS_NAME = "SuperMap.TimeControlBase";
    }


    /**
     * @function TimeControlBase.prototype.updateOptions
     * @description 更新参数。
     * @param {Object} options - 设置参数的可选参数。设置步长，刷新频率、开始结束时间、是否循环、是否反向。
     */
    updateOptions(options) {
        //设置步长，刷新频率、开始结束时间、是否循环、是否反向
        var me = this;
        options = options || {};
        if (options.speed && options.speed >= 0) {
            me.speed = options.speed;
            me.speed = Number(me.speed);
        }

        if (options.speed && options.frequency >= 0) {
            me.frequency = options.frequency;
            me.frequency = Number(me.frequency);
        }

        if (options.startTime && options.startTime != null) {
            me.startTime = options.startTime;
            me.startTime = Date.parse(new Date(me.startTime));
        }

        if (options.endTime && options.endTime != null && options.endTime >= me.startTime) {
            me.endTime = options.endTime;
            me.endTime = Date.parse(new Date(me.endTime));
        }

        if (options.repeat != null) {
            me.repeat = options.repeat;
        }

        if (options.reverse != null) {
            me.reverse = options.reverse;
        }
    }


    /**
     * @function TimeControlBase.prototype.start
     * @description 开始。
     */
    start() {
        var me = this;

        if (!me.running) {
            me.running = true;
            me.tick();
            me.events.triggerEvent('start', me.currentTime);
        }
    }


    /**
     * @function TimeControlBase.prototype.pause
     * @description 暂停。
     */
    pause() {
        var me = this;
        me.running = false;
        me.events.triggerEvent('pause', me.currentTime);
    }


    /**
     * @function TimeControlBase.prototype.stop
     * @description 停止，停止后返回起始状态。
     */
    stop() {
        var me = this;
        //停止时 时间设置为开始时间
        me.currentTime = me.startTime;
        //如果正在运行，修改为初始时间即可绘制一帧
        if (me.running) {
            me.running = false;
        }
        me.events.triggerEvent('stop', me.currentTime);
    }


    /**
     * @function TimeControlBase.prototype.toggle
     * @description 开关切换，切换的是开始和暂停。
     */
    toggle() {
        var me = this;

        if (me.running) {
            me.pause();
        } else {
            me.start();
        }
    }


    /**
     * @function TimeControlBase.prototype.setSpeed
     * @description 设置步长。
     * @param {number} [speed=1] - 步长，必须为非负数。
     * @returns {boolean} true 代表设置成功，false 设置失败（speed 小于 0 时失败）。
     */
    setSpeed(speed) {
        var me = this;
        if (speed >= 0) {
            me.speed = speed;
            return true;
        }
        return false;
    }


    /**
     * @function TimeControlBase.prototype.getSpeed
     * @description 获取步长。
     * @returns {number} 返回当前的步长。
     */
    getSpeed() {
        return this.speed;
    }


    /**
     * @function TimeControlBase.prototype.setFrequency
     * @description 设置刷新频率。
     * @param {number} [frequency=1000] - 刷新频率，单位为 ms。
     * @returns {boolean} true 代表设置成功，false 设置失败（frequency 小于 0 时失败）。
     */
    setFrequency(frequency) {
        var me = this;
        if (frequency >= 0) {
            me.frequency = frequency;
            return true;
        }
        return false;
    }


    /**
     * @function TimeControlBase.prototype.getFrequency
     * @description 获取刷新频率。
     * @returns {number} 返回当前的刷新频率。
     */
    getFrequency() {
        return this.frequency;
    }


    /**
     * @function TimeControlBase.prototype.setStartTime
     * @description 设置起始时间，设置完成后如果当前时间小于起始时间，则从起始时间开始。
     * @param {number} startTime - 需要设置的起始时间。
     * @returns {boolean} true 代表设置成功，false 设置失败（startTime 大于结束时间时失败）。
     */
    setStartTime(startTime) {
        var me = this;
        startTime = Date.parse(new Date(startTime));
        //起始时间不得大于结束时间
        if (startTime > me.endTime) {
            return false;
        }
        me.startTime = startTime;
        //如果当前时间小于了起始时间，则从当前起始时间开始
        if (me.currentTime < me.startTime) {
            me.currentTime = me.startTime;
            me.tick();
        }
        return true;
    }


    /**
     * @function TimeControlBase.prototype.getStartTime
     * @description 获取起始时间。
     * @returns {number} 返回当前的起始时间。
     */
    getStartTime() {
        return this.startTime;
    }


    /**
     * @function TimeControlBase.prototype.setEndTime
     * @description 设置结束时间，设置完成后如果当前时间大于结束，则从起始时间开始。
     * @param {number} endTime - 需要设置的结束时间。
     * @returns {boolean} true 代表设置成功，false 设置失败（endTime 小于开始时间时失败）。
     */
    setEndTime(endTime) {
        var me = this;
        me.endTime = Date.parse(new Date(me.endTime));
        //结束时间不得小于开始时间
        if (endTime < me.startTime) {
            return false;
        }
        me.endTime = endTime;
        //如果当前时间大于了结束时间，则从起始时间开始
        if (me.currentTime >= me.endTime) {
            me.currentTime = me.startTime;
            me.tick();
        }
        return true;
    }


    /**
     * @function TimeControlBase.prototype.getEndTime
     * @description 获取结束时间。
     * @returns {number} 返回当前的结束时间。
     */
    getEndTime() {
        return this.endTime;
    }


    /**
     * @function TimeControlBase.prototype.setCurrentTime
     * @description 设置当前时间。
     * @param {number} currentTime - 需要设置的当前时间。
     * @returns {boolean} true 代表设置成功，false 设置失败。
     */
    setCurrentTime(currentTime) {
        var me = this;
        me.currentTime = Date.parse(new Date(me.currentTime));
        //结束时间不得小于开始时间
        if (currentTime >= me.startTime && currentTime <= me.endTime) {
            me.currentTime = currentTime;
            me.startTime = me.currentTime;
            me.tick();
            return true;
        }
        return false;
    }


    /**
     * @function TimeControlBase.prototype.getCurrentTime
     * @description 获取当前时间。
     * @returns {number} 返回当前时间。
     */
    getCurrentTime() {
        return this.currentTime;
    }


    /**
     * @function TimeControlBase.prototype.setRepeat
     * @description 设置是否重复循环。
     * @param {boolean} [repeat=true] - 是否重复循环。
     */
    setRepeat(repeat) {
        this.repeat = repeat;
    }


    /**
     * @function TimeControlBase.prototype.getRepeat
     * @description 获取是否重复循环，默认是 true。
     * @returns {boolean} 返回是否重复循环。
     */
    getRepeat() {
        return this.repeat;
    }


    /**
     * @function TimeControlBase.prototype.setReverse
     * @description 设置是否反向。
     * @param {boolean} [reverse=false] - 是否反向。
     */
    setReverse(reverse) {
        this.reverse = reverse;
    }


    /**
     * @function TimeControlBase.prototype.getReverse
     * @description 获取是否反向，默认是false。
     * @returns {boolean} 返回是否反向。
     */
    getReverse() {
        return this.reverse;
    }


    /**
     * @function TimeControlBase.prototype.getRunning
     * @description 获取运行状态。
     * @returns {boolean} true 代表正在运行，false 发表没有运行。
     */
    getRunning() {
        return this.running;
    }


    /**
     * @function TimeControlBase.prototype.destroy
     * @description 销毁 Animator 对象，释放资源。
     */
    destroy() {
        var me = this;
        me.speed = null;
        me.frequency = null;
        me.startTime = null;
        me.endTime = null;
        me.currentTime = null;
        me.repeat = null;
        me.running = false;
        me.reverse = null;
    }


    tick() {
        //TODO 每次刷新执行的操作。子类实现
    }

}


;// CONCATENATED MODULE: ./src/common/control/TimeFlowControl.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class TimeFlowControl
 * @deprecatedclass SuperMap.TimeFlowControl
 * @category  Control
 * @classdesc 时间管理类。此类只负责时间上的控制，具体执行的操作需要用户在初始化时的回调函数内部进行实现。
 * 如设置起始时间为 1000，结束时间是 2000，步长设置为 1，
 * 那么表示按照每次1年（可以通过 setSpeed 进行修改）的变化从公元 1000 年开始到公元 2000 年为止，默认每 1 秒会变化 1 次（通过 setFrequency 修改）
 * @modulecategory Control
 * @extends {TimeControlBase}
 * @param {function} callback - 每次刷新回调函数。具体的效果需要用户在此回调函数里面实现。
 * @param {Object} options - 可选参数。
 * @param {number} [options.speed=1] - 步长（单位 ms）。不能小于 0，（每次刷新的数据之间的间隔为 1ms）。
 * @param {number} [options.frequency=1000] - 刷新频率（单位 ms）。
 * @param {number} [options.startTime=0] - 起始时间，必须为数字，且小于等于 endTime。如果不设置，初始化时为 0，建议设置。
 * @param {number} [options.endTime] - 结束时间，必须为数字，且大于等于 startTime。如果不设置，初始化时使用 new Date() 以当前时间进行设置，建议设置。
 * @param {boolean} [options.repeat=true] - 是否重复循环。
 * @param {boolean} [options.reverse=false] - 是否反向。
 * @usage
 */
class TimeFlowControl extends TimeControlBase {


    constructor(callback, options) {
        super(options);
        var me = this;
        /**
         * @member TimeFlowControl.prototype.callback -{function}
         * @description 每次刷新执行的回调函数。
         */
        me.callback = callback;

        //先让IE下支持bind方法
        if (!Function.prototype.bind) {
            Function.prototype.bind = function (oThis) {
                if (typeof this !== "function") {
                    throw new TypeError("Function.prototype.bind - what is trying to be bound is not callable");
                }
                var aArgs = Array.prototype.slice.call(arguments, 1),
                    fToBind = this,
                    fNOP = function () {
                        //empty Function
                    },
                    fBound = function () {
                        return fToBind.apply(this instanceof fNOP && oThis
                            ? this
                            : oThis,
                            aArgs.concat(Array.prototype.slice.call(arguments)));
                    };
                fNOP.prototype = this.prototype;
                fBound.prototype = new fNOP();
                return fBound;
            };
        }
        //保证 this.tick 的上下文还是 TimeControl 这个对象
        me.update = me.update.bind(me);

        me.oldTime = me.currentTime;

        me.CLASS_NAME = "SuperMap.TimeFlowControl";
    }


    /**
     * @function TimeFlowControl.prototype.updateOptions
     * @override
     */
    updateOptions(options) {
        options = options || {};
        super.updateOptions(options);
    }


    /**
     * @function TimeFlowControl.prototype.start
     * @override
     */
    start() {
        var me = this;
        if (me.running) {
            return;
        }
        me.running = true;
        if (me.reverse) {
            if (me.currentTime === me.startTime) {
                me.oldTime = me.endTime;
                me.currentTime = me.oldTime;
            }
        } else {
            if (me.oldTime === me.endTime) {
                me.currentTime = me.startTime;
                me.oldTime = me.currentTime;
            }
        }
        me.tick();
    }


    /**
     * @function TimeFlowControl.prototype.stop
     * @override
     */
    stop() {
        super.stop();
        var me = this;
        me.oldTime = me.currentTime;

        if (me.running) {
            me.running = false;
        }
        //清除定时tick
        me.intervalId && window.clearTimeout(me.intervalId);
    }


    /**
     * @function TimeFlowControl.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        me.oldTime = null;
        me.callback = null;
    }


    /**
     * @function TimeFlowControl.prototype.tick
     * @description 定时刷新。
     */
    tick() {
        var me = this;
        me.intervalId && window.clearInterval(me.intervalId);
        me.intervalId = null;
        me.update();
        me.intervalId = window.setInterval(me.update, me.frequency);
    }

    /**
     * @function TimeFlowControl.prototype.update
     * @description 更新控件。
     */
    update() {
        var me = this;

        //判定是否还需要继续
        if (!me.running) {
            return;
        }
        //调用回调函数
        me.callback && me.callback(me.currentTime);    //destroy之后callback就为空，所以需要判定一下

        if (!me.reverse) {
            //如果相等，则代表上一帧已经运行到了最后，下一帧运行初始化的状态
            if (me.currentTime === me.endTime) {
                //不循环时
                if (!me.repeat) {
                    me.running = false;
                    me.stop();
                    return null;
                }
                me.stop();
                me.currentTime = me.startTime;
                me.oldTime = me.currentTime;
                me.start();
            } else {//否则时间递增
                me.oldTime = me.currentTime;
                me.currentTime += me.speed;
            }

            if (me.currentTime >= me.endTime) {
                me.currentTime = me.endTime;
            }

        } else {
            //如果相等，则代表上一帧已经运行到了最前，下一帧运行结束的状态
            if (me.currentTime === me.startTime) {
                //不循环时
                if (!me.repeat) {
                    me.running = false;
                    return null;
                }

                me.oldTime = me.endTime;
                me.currentTime = me.oldTime;
            } else {//否则时间递减
                me.currentTime = me.oldTime;
                me.oldTime -= me.speed;
            }

            if (me.oldTime <= me.startTime) {
                me.oldTime = me.startTime;
            }
        }

    }


}


;// CONCATENATED MODULE: ./src/common/control/index.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/








// EXTERNAL MODULE: ./node_modules/promise-polyfill/dist/polyfill.js
var polyfill = __webpack_require__(107);
// EXTERNAL MODULE: ./node_modules/fetch-ie8/fetch.js
var fetch_ie8_fetch = __webpack_require__(693);
// EXTERNAL MODULE: ./node_modules/fetch-jsonp/build/fetch-jsonp.js
var fetch_jsonp = __webpack_require__(144);
var fetch_jsonp_default = /*#__PURE__*/__webpack_require__.n(fetch_jsonp);
;// CONCATENATED MODULE: ./src/common/util/FetchRequest.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





let FetchRequest_fetch = window.fetch;
var setFetch = function (newFetch) {
    FetchRequest_fetch = newFetch;
}
var RequestJSONPPromise = {
  limitLength: 1500,
  queryKeys: [],
  queryValues: [],
  supermap_callbacks: {},
  addQueryStrings: function (values) {
      var me = this;
      for (var key in values) {
          me.queryKeys.push(key);
          if (typeof values[key] !== 'string') {
              values[key] = Util_Util.toJSON(values[key]);
          }
          var tempValue = encodeURIComponent(values[key]);
          me.queryValues.push(tempValue);
      }
  },
  issue: function (config) {
      var me = this,
          uid = me.getUid(),
          url = config.url,
          splitQuestUrl = [];

      // me.addQueryStrings({
      //     callback: "RequestJSONPPromise.supermap_callbacks[" + uid + "]"
      // });
      var sectionURL = url,
          keysCount = 0; //此次sectionURL中有多少个key
      var length = me.queryKeys ? me.queryKeys.length : 0;
      for (var i = 0; i < length; i++) {
          if (sectionURL.length + me.queryKeys[i].length + 2 >= me.limitLength) {
              //+2 for ("&"or"?")and"="
              if (keysCount == 0) {
                  return false;
              }
              splitQuestUrl.push(sectionURL);
              sectionURL = url;
              keysCount = 0;
              i--;
          } else {
              if (sectionURL.length + me.queryKeys[i].length + 2 + me.queryValues[i].length > me.limitLength) {
                  var leftValue = me.queryValues[i];
                  while (leftValue.length > 0) {
                      var leftLength = me.limitLength - sectionURL.length - me.queryKeys[i].length - 2; //+2 for ("&"or"?")and"="
                      if (sectionURL.indexOf('?') > -1) {
                          sectionURL += '&';
                      } else {
                          sectionURL += '?';
                      }
                      var tempLeftValue = leftValue.substring(0, leftLength);
                      //避免 截断sectionURL时，将类似于%22这样的符号截成两半，从而导致服务端组装sectionURL时发生错误
                      if (tempLeftValue.substring(leftLength - 1, leftLength) === '%') {
                          leftLength -= 1;
                          tempLeftValue = leftValue.substring(0, leftLength);
                      } else if (tempLeftValue.substring(leftLength - 2, leftLength - 1) === '%') {
                          leftLength -= 2;
                          tempLeftValue = leftValue.substring(0, leftLength);
                      }

                      sectionURL += me.queryKeys[i] + '=' + tempLeftValue;
                      leftValue = leftValue.substring(leftLength);
                      if (tempLeftValue.length > 0) {
                          splitQuestUrl.push(sectionURL);
                          sectionURL = url;
                          keysCount = 0;
                      }
                  }
              } else {
                  keysCount++;
                  if (sectionURL.indexOf('?') > -1) {
                      sectionURL += '&';
                  } else {
                      sectionURL += '?';
                  }
                  sectionURL += me.queryKeys[i] + '=' + me.queryValues[i];
              }
          }
      }
      splitQuestUrl.push(sectionURL);
      return me.send(
          splitQuestUrl,
          'SuperMapJSONPCallbacks_' + uid,
          config && config.proxy
      );
  },

  getUid: function () {
      var uid = new Date().getTime(),
          random = Math.floor(Math.random() * 1e17);
      return uid * 1000 + random;
  },

  send: function (splitQuestUrl, callback, proxy) {
      var len = splitQuestUrl.length;
      if (len > 0) {
         return new Promise((resolve) => {
          var jsonpUserID = new Date().getTime();
          for (var i = 0; i < len; i++) {
              var url = splitQuestUrl[i];
              if (url.indexOf('?') > -1) {
                  url += '&';
              } else {
                  url += '?';
              }
              url += 'sectionCount=' + len;
              url += '&sectionIndex=' + i;
              url += '&jsonpUserID=' + jsonpUserID;
              if (proxy) {
                  url = decodeURIComponent(url);
                  url = proxy + encodeURIComponent(url);
              }
              fetch_jsonp_default()(url, {
                  jsonpCallbackFunction: callback,
                  timeout: 30000
              }).then((result) => {
                resolve(result.json());
              });
          }
         })
      }
  },

  GET: function (config) {
      var me = this;
      me.queryKeys.length = 0;
      me.queryValues.length = 0;
      me.addQueryStrings(config.params);
      return me.issue(config);
  },

  POST: function (config) {
      var me = this;
      me.queryKeys.length = 0;
      me.queryValues.length = 0;
      me.addQueryStrings({
          requestEntity: config.data
      });
      return me.issue(config);
  },

  PUT: function (config) {
      var me = this;
      me.queryKeys.length = 0;
      me.queryValues.length = 0;
      me.addQueryStrings({
          requestEntity: config.data
      });
      return me.issue(config);
  },
  DELETE: function (config) {
      var me = this;
      me.queryKeys.length = 0;
      me.queryValues.length = 0;
      me.addQueryStrings({
          requestEntity: config.data
      });
      return me.issue(config);
  }
};

var CORS;
var RequestTimeout;
/**
 * @function setCORS
 * @description 设置是否允许跨域请求，全局配置，优先级低于 service 下的 crossOring 参数。
 * @category BaseTypes Util
 * @param {boolean} cors - 是否允许跨域请求。
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   {namespace}.setCORS(cors);
 *
 *   // 弃用的写法
 *   SuperMap.setCORS(cors);
 *
 * </script>
 *
 * // ES6 Import
 * import { setCORS } from '{npm}';
 *
 * setCORS(cors);
 * ```
 */
var setCORS = function (cors) {
    CORS = cors;
}
/**
 * @function isCORS
 * @description 是是否允许跨域请求。
 * @category BaseTypes Util
 * @returns {boolean} 是否允许跨域请求。
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.isCORS();
 *
 *   // 弃用的写法
 *   const result = SuperMap.isCORS();
 *
 * </script>
 *
 * // ES6 Import
 * import { isCORS } from '{npm}';
 *
 * const result = isCORS();
 * ```
 */
var isCORS = function () {
    if (CORS != undefined) {
        return CORS;
    }
    return window.XMLHttpRequest && 'withCredentials' in new window.XMLHttpRequest();
}
/**
 * @function setRequestTimeout
 * @category BaseTypes Util
 * @description 设置请求超时时间。
 * @param {number} [timeout=45] - 请求超时时间，单位秒。
 * @usage
 * ```
 * // 浏览器
  <script type="text/javascript" src="{cdn}"></script>
  <script>
    {namespace}.setRequestTimeout(timeout);

    // 弃用的写法
    SuperMap.setRequestTimeout(timeout);

  </script>

  // ES6 Import
  import { setRequestTimeout } from '{npm}';

  setRequestTimeout(timeout);
 * ```
 */
var setRequestTimeout = function (timeout) {
    return RequestTimeout = timeout;
}
/**
 * @function getRequestTimeout
 * @category BaseTypes Util
 * @description 获取请求超时时间。
 * @returns {number} 请求超时时间。
 * @usage
 * ```
 * // 浏览器
  <script type="text/javascript" src="{cdn}"></script>
  <script>
    {namespace}.getRequestTimeout();

    // 弃用的写法
    SuperMap.getRequestTimeout();

  </script>

  // ES6 Import
  import { getRequestTimeout } from '{npm}';

  getRequestTimeout();
 * ```
 */
var getRequestTimeout = function () {
    return RequestTimeout || 45000;
}

/**
 * @name FetchRequest
 * @namespace
 * @category BaseTypes Util
 * @description 获取请求。
 * @usage
 * ```
 * // 浏览器
 * <script type="text/javascript" src="{cdn}"></script>
 * <script>
 *   const result = {namespace}.FetchRequest.commit(method, url, params, options);
 *
 * </script>
 *
 * // ES6 Import
 * import { FetchRequest } from '{npm}';
 *
 * const result = FetchRequest.commit(method, url, params, options);
 *
 * ```
 */
var FetchRequest = {
    /**
     * @function FetchRequest.commit
     * @description commit 请求。
     * @param {string} method - 请求方法。
     * @param {string} url - 请求地址。
     * @param {string} params - 请求参数。
     * @param {Object} options - 请求的配置属性。
     * @returns {Promise} Promise 对象。
     */
    commit: function (method, url, params, options) {
        method = method ? method.toUpperCase() : method;
        switch (method) {
            case 'GET':
                return this.get(url, params, options);
            case 'POST':
                return this.post(url, params, options);
            case 'PUT':
                return this.put(url, params, options);
            case 'DELETE':
                return this.delete(url, params, options);
            default:
                return this.get(url, params, options);
        }
    },
    /**
     * @function FetchRequest.supportDirectRequest
     * @description supportDirectRequest 请求。
     * @param {string} url - 请求地址。
     * @param {Object} options - 请求的配置属性。
     * @returns {boolean} 是否允许跨域请求。
     */
    supportDirectRequest: function (url, options) {
        if (Util_Util.isInTheSameDomain(url)) {
            return true;
        }
        if (options.crossOrigin != undefined) {
            return options.crossOrigin;
        } else {
            return isCORS() || options.proxy;
        }
    },
    /**
     * @function FetchRequest.get
     * @description get 请求。
     * @param {string} url - 请求地址。
     * @param {string} params - 请求参数。
     * @param {Object} options - 请求的配置属性。
     * @returns {Promise} Promise 对象。
     */
    get: function (url, params, options) {
        options = options || {};
        var type = 'GET';
        url = Util_Util.urlAppend(url, this._getParameterString(params || {}));
        url = this._processUrl(url, options);
        if (!this.supportDirectRequest(url, options)) {
            url = url.replace('.json', '.jsonp');
            var config = {
                url: url,
                data: params
            };
            return RequestJSONPPromise.GET(config);
        }
        if (!this.urlIsLong(url)) {
            return this._fetch(url, params, options, type);
        } else {
            return this._postSimulatie(type, url.substring(0, url.indexOf('?')), params, options);
        }
    },
    /**
     * @function FetchRequest.delete
     * @description delete 请求。
     * @param {string} url - 请求地址。
     * @param {string} params - 请求参数。
     * @param {Object} options -请求的配置属性。
     * @returns {Promise} Promise 对象。
     */
    delete: function (url, params, options) {
        options = options || {};
        var type = 'DELETE';
        url = Util_Util.urlAppend(url, this._getParameterString(params || {}));
        url = this._processUrl(url, options);
        if (!this.supportDirectRequest(url, options)) {
            url = url.replace('.json', '.jsonp');
            var config = {
                url: url += "&_method=DELETE",
                data: params
            };
            return RequestJSONPPromise.DELETE(config);
        }
        if (this.urlIsLong(url)) {
            return this._postSimulatie(type, url.substring(0, url.indexOf('?')), params, options);
        }
        return this._fetch(url, params, options, type);
    },
    /**
     * @function FetchRequest.post
     * @description post 请求。
     * @param {string} url - 请求地址。
     * @param {string} params - 请求参数。
     * @param {Object} options - 请求的配置属性。
     * @returns {Promise} Promise 对象。
     */
    post: function (url, params, options) {
        options = options || {};
        url = this._processUrl(url, options);
        if (!this.supportDirectRequest(url, options)) {
            url = url.replace('.json', '.jsonp');
            var config = {
                url: Util_Util.urlAppend(url, "_method=POST"),
                data: params
            };
            return RequestJSONPPromise.POST(config);
        }
        return this._fetch(url, params, options, 'POST');
    },
    /**
     * @function FetchRequest.put
     * @description put 请求。
     * @param {string} url - 请求地址。
     * @param {string} params - 请求参数。
     * @param {Object} options - 请求的配置属性。
     * @returns {Promise} Promise 对象。
     */
    put: function (url, params, options) {
        options = options || {};
        url = this._processUrl(url, options);
        if (!this.supportDirectRequest(url, options)) {
            url = url.replace('.json', '.jsonp');
            var config = {
                url: url += "&_method=PUT",
                data: params
            };
            return RequestJSONPPromise.PUT(config);
        }
        return this._fetch(url, params, options, 'PUT');
    },
    /**
     * @function FetchRequest.urlIsLong
     * @description URL 的字节长度是否太长。
     * @param {string} url - 请求地址。
     * @returns {boolean} URL 的字节长度是否太长。
     */
    urlIsLong: function (url) {
        //当前url的字节长度。
        var totalLength = 0,
            charCode = null;
        for (var i = 0, len = url.length; i < len; i++) {
            //转化为Unicode编码
            charCode = url.charCodeAt(i);
            if (charCode < 0x007f) {
                totalLength++;
            } else if ((0x0080 <= charCode) && (charCode <= 0x07ff)) {
                totalLength += 2;
            } else if ((0x0800 <= charCode) && (charCode <= 0xffff)) {
                totalLength += 3;
            }
        }
        return totalLength < 2000 ? false : true;
    },
    _postSimulatie: function (type, url, params, options) {
        var separator = url.indexOf('?') > -1 ? '&' : '?';
        url += separator + '_method=' + type;
        if (typeof params !== 'string') {
            params = JSON.stringify(params);
        }
        return this.post(url, params, options);
    },

    _processUrl: function (url, options) {
        if (this._isMVTRequest(url)) {
            return url;
        }

        if (url.indexOf('.json') === -1 && !options.withoutFormatSuffix) {
            if (url.indexOf('?') < 0) {
                url += '.json';
            } else {
                var urlArrays = url.split('?');
                if (urlArrays.length === 2) {
                    url = urlArrays[0] + '.json?' + urlArrays[1];
                }
            }
        }
        if (options && options.proxy) {
            if (typeof options.proxy === 'function') {
                url = options.proxy(url);
            } else {
                url = decodeURIComponent(url);
                url = options.proxy + encodeURIComponent(url);
            }
        }
        return url;
    },

    _fetch: function (url, params, options, type) {
        options = options || {};
        options.headers = options.headers || {};
        if (!options.headers['Content-Type'] && !FormData.prototype.isPrototypeOf(params)) {
            options.headers['Content-Type'] = 'application/x-www-form-urlencoded;charset=UTF-8';
        }
        if (options.timeout) {
            return this._timeout(
                options.timeout,
                FetchRequest_fetch(url, {
                    method: type,
                    headers: options.headers,
                    body: type === 'PUT' || type === 'POST' ? params : undefined,
                    credentials: this._getWithCredentials(options),
                    mode: 'cors',
                    timeout: getRequestTimeout()
                }).then(function (response) {
                    return response;
                })
            );
        }
        return FetchRequest_fetch(url, {
            method: type,
            body: type === 'PUT' || type === 'POST' ? params : undefined,
            headers: options.headers,
            credentials: this._getWithCredentials(options),
            mode: 'cors',
            timeout: getRequestTimeout()
        }).then(function (response) {
            return response;
        });
    },

    _getWithCredentials: function (options) {
        if (options.withCredentials === true) {
            return 'include';
        }
        if (options.withCredentials === false) {
            return 'omit';
        }
        return 'same-origin';
    },

    _fetchJsonp: function (url, options) {
        options = options || {};
        return fetch_jsonp_default()(url, {
            method: 'GET',
            timeout: options.timeout
        }).then(function (response) {
            return response;
        });
    },

    _timeout: function (seconds, promise) {
        return new Promise(function (resolve, reject) {
            setTimeout(function () {
                reject(new Error('timeout'));
            }, seconds);
            promise.then(resolve, reject);
        });
    },

    _getParameterString: function (params) {
        var paramsArray = [];
        for (var key in params) {
            var value = params[key];
            if (value != null && typeof value !== 'function') {
                var encodedValue;
                if (Array.isArray(value) || value.toString() === '[object Object]') {
                    encodedValue = encodeURIComponent(JSON.stringify(value));
                } else {
                    encodedValue = encodeURIComponent(value);
                }
                paramsArray.push(encodeURIComponent(key) + '=' + encodedValue);
            }
        }
        return paramsArray.join('&');
    },

    _isMVTRequest: function (url) {
        return url.indexOf('.mvt') > -1 || url.indexOf('.pbf') > -1;
    }
}

;// CONCATENATED MODULE: ./src/common/security/SecurityManager.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class SecurityManager
 * @deprecatedclass SuperMap.SecurityManager
 * @category Security
 * @classdesc 安全管理中心，提供 iServer,iPortal,Online 统一权限认证管理。
 *  > 使用说明：
 *  > 创建任何一个服务之前调用 {@link SecurityManager.registerToken}或
 *  > {@link SecurityManager.registerKey}注册凭据。
 *  > 发送请求时根据 URL 或者服务 ID 获取相应的 key 或者 token 并自动添加到服务地址中。
 * @usage
 */
class SecurityManager {
    /**
     * @description 从服务器获取一个token,在此之前要注册服务器信息。
     * @function SecurityManager.generateToken
     * @param {string} url - 服务器域名+端口，如：http://localhost:8092。
     * @param {TokenServiceParameter} tokenParam - token 申请参数。
     * @returns {Promise} 包含 token 信息的 Promise 对象。
     */

    static generateToken(url, tokenParam) {
        var serverInfo = this.servers[url];
        if (!serverInfo) {
            return;
        }
        return FetchRequest.post(serverInfo.tokenServiceUrl, JSON.stringify(tokenParam.toJSON())).then(function(
            response
        ) {
            return response.text();
        });
    }

    /**
     * @description 注册安全服务器相关信息。
     * @function SecurityManager.registerServers
     * @param {ServerInfo} serverInfos - 服务器信息。
     */
    static registerServers(serverInfos) {
        this.servers = this.servers || {};
        if (!Util_Util.isArray(serverInfos)) {
            serverInfos = [serverInfos];
        }
        for (var i = 0; i < serverInfos.length; i++) {
            var serverInfo = serverInfos[i];
            this.servers[serverInfo.server] = serverInfo;
        }
    }

    /**
     * @description 服务请求都会自动带上这个 token。
     * @function SecurityManager.registerToken
     * @param {string} url -服务器域名+端口：如http://localhost:8090。
     * @param {string} token - token。
     */
    static registerToken(url, token) {
        this.tokens = this.tokens || {};
        if (!url || !token) {
            return;
        }
        var domain = this._getTokenStorageKey(url);
        this.tokens[domain] = token;
    }

    /**
     * @description 注册 key,ids 为数组(存在一个 key 对应多个服务)。
     * @function SecurityManager.registerKey
     * @param {Array} ids - 可以是服务 ID 数组或者 URL 地址数组或者 webAPI 类型数组。
     * @param {string} key - key。
     */
    static registerKey(ids, key) {
        this.keys = this.keys || {};
        if (!ids || ids.length < 1 || !key) {
            return;
        }

        ids = Util_Util.isArray(ids) ? ids : [ids];
        for (var i = 0; i < ids.length; i++) {
            var id = this._getUrlRestString(ids[0]) || ids[0];
            this.keys[id] = key;
        }
    }

    /**
     * @description 获取服务器信息。
     * @function SecurityManager.getServerInfo
     * @param {string} url - 服务器域名+端口，如：http://localhost:8092。
     * @returns {ServerInfo} 服务器信息。
     */
    static getServerInfo(url) {
        this.servers = this.servers || {};
        return this.servers[url];
    }

    /**
     * @description 根据 URL 获取token。
     * @function SecurityManager.getToken
     * @param {string} url - 服务器域名+端口，如：http://localhost:8092。
     * @returns {string} token。
     */
    static getToken(url) {
        if (!url) {
            return;
        }
        this.tokens = this.tokens || {};
        var domain = this._getTokenStorageKey(url);
        return this.tokens[domain];
    }

    /**
     * @description 根据 URL 获取 key。
     * @function SecurityManager.getKey
     * @param {string} id - ID。
     * @returns {string} key。
     */
    static getKey(id) {
        this.keys = this.keys || {};
        var key = this._getUrlRestString(id) || id;
        return this.keys[key];
    }

    /**
     * @description iServer 登录验证。
     * @function SecurityManager.loginiServer
     * @param {string} url - iServer 首页地址，如：http://localhost:8090/iserver。
     * @param {string} username - 用户名。
     * @param {string} password - 密码。
     * @param {boolean} [rememberme=false] - 是否记住。
     * @returns {Promise} 包含 iServer 登录请求结果的 Promise 对象。
     */
    static loginiServer(url, username, password, rememberme) {
        url = Util_Util.urlPathAppend(url, 'services/security/login');
        var loginInfo = {
            username: username && username.toString(),
            password: password && password.toString(),
            rememberme: rememberme
        };
        loginInfo = JSON.stringify(loginInfo);
        var requestOptions = {
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8'
            }
        };
        return FetchRequest.post(url, loginInfo, requestOptions).then(function(response) {
            return response.json();
        });
    }

    /**
     * @description iServer登出。
     * @function SecurityManager.logoutiServer
     * @param {string} url - iServer 首页地址,如：http://localhost:8090/iserver。
     * @returns {Promise} 是否登出成功。
     */
    static logoutiServer(url) {
        url = Util_Util.urlPathAppend(url, 'services/security/logout');
        var requestOptions = {
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8'
            },
            withoutFormatSuffix: true
        };
        return FetchRequest.get(url, '', requestOptions)
            .then(function() {
                return true;
            })
            .catch(function() {
                return false;
            });
    }

    /**
     * @description Online 登录验证。
     * @function SecurityManager.loginOnline
     * @param {string} callbackLocation - 跳转位置。
     * @param {boolean} [newTab=true] - 是否新窗口打开。
     */
    static loginOnline(callbackLocation, newTab) {
        var loginUrl = SecurityManager.SSO + '/login?service=' + callbackLocation;
        this._open(loginUrl, newTab);
    }

    /**
     * @description iPortal登录验证。
     * @function SecurityManager.loginiPortal
     * @param {string} url - iportal 首页地址,如：http://localhost:8092/iportal。
     * @param {string} username - 用户名。
     * @param {string} password - 密码。
     * @returns {Promise} 包含 iPortal 登录请求结果的 Promise 对象。
     */
    static loginiPortal(url, username, password) {
        url = Util_Util.urlPathAppend(url, 'web/login');
        var loginInfo = {
            username: username && username.toString(),
            password: password && password.toString()
        };
        loginInfo = JSON.stringify(loginInfo);
        var requestOptions = {
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8'
            },
            withCredentials: false
        };
        return FetchRequest.post(url, loginInfo, requestOptions).then(function(response) {
            return response.json();
        });
    }

    /**
     * @description iPortal 登出。
     * @function SecurityManager.logoutiPortal
     * @param {string} url - iportal 首页地址，如：http://localhost:8092/iportal。
     * @returns {Promise} 如果登出成功，返回 true;否则返回 false。
     */
    static logoutiPortal(url) {
        url = Util_Util.urlPathAppend(url, 'services/security/logout');
        var requestOptions = {
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8'
            },
            withCredentials: true,
            withoutFormatSuffix: true
        };
        return FetchRequest.get(url, '', requestOptions)
            .then(function() {
                return true;
            })
            .catch(function() {
                return false;
            });
    }

    /**
     * @description iManager 登录验证。
     * @function SecurityManager.loginManager
     * @param {string} url - iManager 地址。地址参数为 iManager 首页地址，如： http://localhost:8390/imanager。
     * @param {Object} [loginInfoParams] - iManager 登录参数。
     * @param {string} loginInfoParams.userName - 用户名。
     * @param {string} loginInfoParams.password - 密码。
     * @param {Object} options
     * @param {boolean} [options.isNewTab=true] - 不同域时是否在新窗口打开登录页面。
     * @returns {Promise} 包含 iManager 登录请求结果的 Promise 对象。
     */
    static loginManager(url, loginInfoParams) {
        var requestUrl = Util_Util.urlPathAppend(url, '/security/tokens');
        var params = loginInfoParams || {};
        var loginInfo = {
            username: params.userName && params.userName.toString(),
            password: params.password && params.password.toString()
        };
        loginInfo = JSON.stringify(loginInfo);
        var requestOptions = {
            headers: {
                Accept: '*/*',
                'Content-Type': 'application/json; charset=UTF-8'
            }
        };
        var me = this;
        return FetchRequest.post(requestUrl, loginInfo, requestOptions).then(function(response) {
          return response.text();
        }).then(function(result) {
          me.imanagerToken = result;
          return result;
        });
    }

    /**
     * @description 清空全部验证信息。
     * @function SecurityManager.destroyAllCredentials
     */
    static destroyAllCredentials() {
        this.keys = null;
        this.tokens = null;
        this.servers = null;
    }

    /**
     * @description 清空令牌信息。
     * @function SecurityManager.destroyToken
     * @param {string} url - iportal 首页地址，如：http://localhost:8092/iportal。
     */
    static destroyToken(url) {
        if (!url) {
            return;
        }
        var domain = this._getTokenStorageKey(url);
        this.tokens = this.tokens || {};
        if (this.tokens[domain]) {
            delete this.tokens[domain];
        }
    }

    /**
     * @description 清空服务授权码。
     * @function SecurityManager.destroyKey
     * @param {string} url - iServer 首页地址,如：http://localhost:8090/iserver。
     */
    static destroyKey(url) {
        if (!url) {
            return;
        }
        this.keys = this.keys || {};
        var key = this._getUrlRestString(url) || url;
        if (this.keys[key]) {
            delete this.keys[key];
        }
    }

    /**
     * @description 服务URL追加授权信息，授权信息需先通过SecurityManager.registerKey或SecurityManager.registerToken注册。
     * @version 10.1.2
     * @function SecurityManager.appendCredential
     * @param {string} url - 服务URL。
     * @returns {string} 绑定了token或者key的服务URL。
     */
    static appendCredential(url) {
        var newUrl = url;
        var value = this.getToken(url);
        var credential = value ? new Credential(value, 'token') : null;
		if (!credential) {
            value = this.getKey(url);
            credential = value ? new Credential(value, 'key') : null;
          }
        if (credential) {
            newUrl = Util_Util.urlAppend(newUrl, credential.getUrlParameters());
        }
        return newUrl;
    }

    static _open(url, newTab) {
        newTab = newTab != null ? newTab : true;
        var offsetX = window.screen.availWidth / 2 - this.INNER_WINDOW_WIDTH / 2;
        var offsetY = window.screen.availHeight / 2 - this.INNER_WINDOW_HEIGHT / 2;
        var options =
            'height=' +
            this.INNER_WINDOW_HEIGHT +
            ', width=' +
            this.INNER_WINDOW_WIDTH +
            ',top=' +
            offsetY +
            ', left=' +
            offsetX +
            ',toolbar=no, menubar=no, scrollbars=no, resizable=no, location=no, status=no';
        if (newTab) {
            window.open(url, 'login');
        } else {
            window.open(url, 'login', options);
        }
    }

    static _getTokenStorageKey(url) {
        var patten = /(.*?):\/\/([^\/]+)/i;
        var result = url.match(patten);
        if (!result) {
            return url;
        }
        return result[0];
    }

    static _getUrlRestString(url) {
        if (!url) {
            return url;
        }
        // var patten = /http:\/\/(.*\/rest)/i;
        var patten = /(http|https):\/\/(.*\/rest)/i;
        var result = url.match(patten);
        if (!result) {
            return url;
        }
        return result[0];
    }
}
SecurityManager.INNER_WINDOW_WIDTH = 600;
SecurityManager.INNER_WINDOW_HEIGHT = 600;
SecurityManager.SSO = 'https://sso.supermap.com';
SecurityManager.ONLINE = 'https://www.supermapol.com';

;// CONCATENATED MODULE: ./src/common/iManager/iManagerServiceBase.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class IManagerServiceBase
 * @aliasclass iManagerServiceBase
 * @deprecatedclass SuperMap.iManagerServiceBase
 * @classdesc iManager 服务基类（有权限限制的类需要实现此类）。
 * @category iManager
 * @param {string} url - iManager 首页地址，如：http://localhost:8390/imanager。
 * @param {Object} options - 可选参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class IManagerServiceBase {

    constructor(url,options) {
        if (url) {
            var end = url.substr(url.length - 1, 1);
            this.serviceUrl = end === "/" ? url.substr(0, url.length - 2) : url;
        }
        this.options = options || {};
        this.CLASS_NAME = "SuperMap.iManagerServiceBase";
    }

    /**
     * @function IManagerServiceBase.prototype.request
     * @description 子类统一通过该方法发送请求。
     * @param {string} url - 请求 URL。
     * @param {string} [method='GET'] - 请求类型。
     * @param {Object} [requestOptions] - 请求选项。
     * @param {Object} param - 请求参数。
     * @description 发送请求。
     * @returns {Promise} Promise 对象。
     */
    request(method, url, param, requestOptions) {
        requestOptions = requestOptions || {
            headers: {
                'Accept': '*/*',
                'Content-Type': 'application/json'
            }
        };
        requestOptions['crossOrigin'] = this.options.crossOrigin;
        requestOptions['headers'] = this.options.headers;
        var token = SecurityManager.imanagerToken;
        if (token) {
            if (!requestOptions.headers) {
                requestOptions.headers = {};
            }
            requestOptions.headers['X-Auth-Token'] = token;
        }
        if (param) {
            param = JSON.stringify(param);
        }
        return FetchRequest.commit(method, url, param, requestOptions).then(function (response) {
            return response.json();
        });
    }

}


;// CONCATENATED MODULE: ./src/common/iManager/iManagerCreateNodeParam.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class IManagerCreateNodeParam
 * @aliasclass iManagerCreateNodeParam
 * @deprecatedclass SuperMap.iManagerCreateNodeParam
 * @classdesc iManager 创建节点参数。
 * @category iManager
 * @param {Object} [params] - 节点参数。
 * @usage
 */
class IManagerCreateNodeParam {

    constructor(params) {
        params = params || {};
        this.nodeSpec = 'SMALL';              //取值范围: ['SMALL','MEDIUM','LARGE'] 以及自定义的环境规格名称
        this.nodeCount = 1;                   //要创建vm的个数
        this.nodeName = '';                   //vm名称
        this.password = '';                   //vm的密码,空表示随机分配
        this.description = '';                //描述信息
        this.physicalMachineName = '';        //vm所属的物理机名称.
        this.ips = [];                        //vm的ip,空数组表示随机分配
        this.userName = '';                   //vm所属用户
        Util_Util.extend(this, params);
    }

}

;// CONCATENATED MODULE: ./src/common/iManager/iManager.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class IManager
 * @aliasclass iManager
 * @deprecatedclass SuperMap.iManager
 * @classdesc iManager 服务类。
 * @category iManager
 * @modulecategory Services
 * @param {string} serviceUrl - iManager 首页地址。
 * @usage
 */
class IManager extends IManagerServiceBase {

    constructor(iManagerUrl) {
        super(iManagerUrl);
    }

    /**
     * @function IManager.prototype.load
     * @description 获取所有服务接口，验证是否已登录授权。
     * @returns {Promise} Promise 对象。
     */
    load() {
        return this.request("GET", this.serviceUrl + '/web/api/service.json');
    }

    /**
     * @function IManager.prototype.createIServer
     * @param {IManagerCreateNodeParam} createParam - 创建参数。
     * @description 创建 iServer。
     * @returns {Promise} Promise 对象。
     */
    createIServer(createParam) {
        return this.request("POST", this.serviceUrl + '/cloud/web/nodes/server.json', new IManagerCreateNodeParam(createParam));
    }

    /**
     * @function IManager.prototype.createIPortal
     * @param {IManagerCreateNodeParam} createParam - 创建参数。
     * @description 创建 iPortal。
     * @returns {Promise} Promise 对象。
     */
    createIPortal(createParam) {
        return this.request("POST", this.serviceUrl + '/icloud/web/nodes/portal.json', new IManagerCreateNodeParam(createParam));
    }

    /**
     * @function IManager.prototype.iServerList
     * @description 获取所有创建的 iServer。
     * @returns {Promise} Promise 对象。
     */
    iServerList() {
        return this.request("GET", this.serviceUrl + '/cloud/web/nodes/server.json');
    }

    /**
     * @function IManager.prototype.iPortalList
     * @description 获取所有创建的 iPortal。
     * @returns {Promise} Promise 对象。
     */
    iPortalList() {
        return this.request("GET", this.serviceUrl + '/icloud/web/nodes/portal.json');
    }

    /**
     * @function IManager.prototype.startNodes
     * @param {Array.<string>} ids - 需要启动节点的 ID 数组。e.g:['1']。
     * @description 启动节点。
     * @returns {Promise} Promise 对象。
     */
    startNodes(ids) {
        return this.request("POST", this.serviceUrl + '/cloud/web/nodes/started.json', ids);
    }

    /**
     * @function IManager.prototype.stopNodes
     * @param {Array.<string>} ids - 需要停止节点的 ID 数组。e.g:['1']。
     * @description 停止节点。
     * @returns {Promise} Promise 对象。
     */
    stopNodes(ids) {
        return this.request("POST", this.serviceUrl + '/cloud/web/nodes/stopped.json', ids);
    }
}


;// CONCATENATED MODULE: ./src/common/iManager/index.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/








;// CONCATENATED MODULE: ./src/common/iPortal/iPortalServiceBase.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class IPortalServiceBase
 * @aliasclass iPortalServiceBase
 * @deprecatedclass SuperMap.iPortalServiceBase
 * @classdesc iPortal 服务基类（有权限限制的类需要实现此类）。
 * @category iPortal/Online Core
 * @param {string} url - 服务地址。
 * @param {Object} options - 可选参数。
 * @param {boolean} [options.withCredentials=false] - 请求是否携带 cookie。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class IPortalServiceBase {

    constructor(url, options) {
        options = options || {};
        this.serviceUrl = url;
        this.CLASS_NAME = "SuperMap.iPortalServiceBase";
        this.withCredentials = options.withCredentials || false;
        this.crossOrigin = options.crossOrigin
        this.headers = options.headers
    }

    /**
     * @function IPortalServiceBase.prototype.request
     * @description 子类统一通过该方法发送请求。
     * @param {string} [method='GET'] - 请求类型。
     * @param {string} url - 服务地址。
     * @param {Object} param - 请求参数。
     * @param {Object} [requestOptions] - fetch 请求配置项。
     * @returns {Promise} 返回包含请求结果的 Promise 对象。
     */

    request(method, url, param, requestOptions = {headers: this.headers, crossOrigin: this.crossOrigin, withCredentials: this.withCredentials }) {
        url = SecurityManager.appendCredential(url);
        return FetchRequest.commit(method, url, param, requestOptions).then(function (response) {
            return response.json();
        });
    }


}


;// CONCATENATED MODULE: ./src/common/iPortal/iPortalQueryParam.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class IPortalQueryParam
 * @aliasclass iPortalQueryParam
 * @deprecatedclass SuperMap.iPortalQueryParam
 * @classdesc iPortal 资源查询参数。
 * @version 10.0.1
 * @category iPortal/Online Resources ResourcesQuery
 * @param {Object} params - 可选参数。
 * @param {ResourceType} [params.resourceType] - 资源类型。
 * @param {number} [params.pageSize] - 分页中每页大小。
 * @param {number} [params.currentPage] - 分页页码。
 * @param {OrderBy} [params.orderBy] - 排序字段。
 * @param {OrderType} [params.orderType] - 根据升序还是降序过滤。
 * @param {SearchType} [params.searchType] - 根据查询的范围进行过滤。
 * @param {Array} [params.tags] - 标签。
 * @param {Array} [params.dirIds] - 目录 ID。
 * @param {Array} [params.resourceSubTypes] - 根据资源的子类型进行过滤。
 * @param {AggregationTypes} [params.aggregationTypes] - 聚合查询的类型。
 * @param {string} [params.text] - 	搜索的关键词。
 * @param {Array} [params.groupIds] - 	根据群组进行过滤。
 * @param {Array} [params.departmentIds] - 根据部门进行过滤。
 * @usage
 */
class IPortalQueryParam {

    constructor(params) {
        params = params || {};
        this.resourceType = ""; // 空为全部 MAP SERVICE SCENE DATA INSIGHTS_WORKSPACE MAP_DASHBOARD
        this.pageSize = 12; // 每页多少条
        this.currentPage = 1; // 第几页
        this.orderBy = "UPDATETIME"; // UPDATETIME HEATLEVEL
        this.orderType = "DESC"; // DESC ASC
        this.searchType = "PUBLIC"; // PUBLIC SHARETOME_RES MYDEPARTMENT_RES MYGROUP_RES MY_RES
        this.tags = [];  // 标签
        this.dirIds = []; // 类别
        this.resourceSubTypes = []; // 类型
        this.aggregationTypes = []; // TAG TYPE SUBTYPE
        this.text = ""; // 搜索字段
        this.groupIds = []; // 群组Id过滤
        this.departmentIds = []; // 部门Id过滤
        Util_Util.extend(this, params);
    }
}


;// CONCATENATED MODULE: ./src/common/iPortal/iPortalQueryResult.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class IPortalQueryResult
 * @aliasclass iPortalQueryResult
 * @deprecatedclass SuperMap.iPortalQueryResult
 * @classdesc iPortal 资源结果集封装类。
 * @version 10.0.1
 * @category iPortal/Online Resources ResourcesQuery
 * @param {Object} queryResult - 可选参数。
 * @param {Array} [queryResult.content] - 页面内容。
 * @param {number} [queryResult.total] - 总记录数。
 * @param {number} [queryResult.currentPage] - 当前第几页。
 * @param {number} [queryResult.pageSize] - 每页大小。
 * @param {Object} [queryResult.aggregations] - 聚合查询的结果。
 * @usage
 */
class IPortalQueryResult {
    constructor(queryResult) {
        queryResult = queryResult || {};
        this.content = [];
        this.total = 0;
        this.currentPage = 1;
        this.pageSize = 12;
        this.aggregations = null;
        Util_Util.extend(this, queryResult);
    }

}



;// CONCATENATED MODULE: ./src/common/iPortal/iPortalResource.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class IPortalResource
 * @aliasclass iPortalResource
 * @deprecatedclass SuperMap.iPortalResource
 * @classdesc iPortal 资源详情类。
 * @version 10.0.1
 * @category iPortal/Online Resources
 * @param {string} portalUrl - 服务地址。
 * @param {Object} resourceInfo - 可选参数。
 * @param {Array} [resourceInfo.authorizeSetting] - 资源的授权信息。
 * @param {string} [resourceInfo.bounds] - 资源的坐标范围。
 * @param {string} [resourceInfo.bounds4326] - 资源的坐标范围，转换为EPSG 4326坐标系统后的地理范围。
 * @param {string} [resourceInfo.checkStatus] - 资源的审核状态，可以是：空,SUCCESSFUL,UNCHECKED,FAILED。
 * @param {Date} [resourceInfo.createTime] - 资源的创建时间。
 * @param {string} [resourceInfo.description] - 资源描述。
 * @param {number} [resourceInfo.dirId] - 资源所在的门户目录的ID。
 * @param {number} [resourceInfo.epsgCode] - 门户资源基于的坐标系的EPSG值。
 * @param {number} [resourceInfo.heatLevel] - 记录资源的访问量或下载量。
 * @param {string} [resourceInfo.id] - 资源存储到ElasticSearch中的文档ID。
 * @param {string} [resourceInfo.name] - 资源名称。
 * @param {number} [resourceInfo.personalDirId] - 资源所在的个人目录的ID。
 * @param {number} [resourceInfo.resourceId] - 资源表(maps,services等)里的ID。
 * @param {string} [resourceInfo.resourceSubType] - 某类资源的具体子类型。
 * @param {ResourceType} [resourceInfo.resourceType] - 资源类型。
 * @param {number} [resourceInfo.serviceRootUrlId] - 批量注册服务时，服务根地址的ID。
 * @param {Array} [resourceInfo.tags] - 资源的标签。
 * @param {string} [resourceInfo.thumbnail] - 资源的缩略图。
 * @param {Date} [resourceInfo.updateTime] - 资源的更新时间。
 * @param {string} [resourceInfo.userName] - 搜索的关键词。
 * @param {Object} [resourceInfo.sourceJSON] - 提供了门户项目返回的所有信息。
 * @extends {IPortalServiceBase}
 * @usage
 */
class IPortalResource extends IPortalServiceBase {
    constructor(portalUrl, resourceInfo) {
        super(portalUrl);
        resourceInfo = resourceInfo || {};
        this.authorizeSetting = [];
        this.bounds = "";
        this.bounds4326 = "";
        this.checkStatus = "";
        this.createTime = 0;
        this.description = null;
        this.dirId = null;
        this.epsgCode = 0;
        this.heatLevel = 0;
        this.id = 0;
        this.name = "";
        this.personalDirId = null;
        this.resourceId = 0;
        this.resourceSubType = null;
        this.resourceType = null;
        this.serviceRootUrlId = null;
        this.tags = null;
        this.thumbnail = null;
        this.updateTime = 0;
        this.userName = "";
        this.sourceJSON = {};//返回门户资源详细信息
        Util_Util.extend(this, resourceInfo); // INSIGHTS_WORKSPACE MAP_DASHBOARD
        this.resourceUrl = portalUrl + "/web/"+this.resourceType.replace("_","").toLowerCase()+"s/" + this.resourceId;
        if (this.withCredentials) {
            this.resourceUrl = portalUrl + "/web/mycontent/"+this.resourceType.replace("_","").toLowerCase()+"s/" + this.resourceId;
        }
        // if (this.id) {
        //     this.mapUrl = mapUrl + "/" + this.id;
        // }
    }

    /**
     * @function IPortalResource.prototype.load
     * @description 加载资源信息。
     * @returns {Promise} 返回 Promise 对象。如果成功，Promise 没有返回值，请求返回结果自动填充到该类的属性中；如果失败，Promise 返回值包含错误信息。
     */
    load() {
        var me = this;
        return me.request("GET", me.resourceUrl + ".json")
            .then(function (resourceInfo) {
                if (resourceInfo.error) {
                    return resourceInfo;
                }
                me.sourceJSON = resourceInfo;
            });
    }

    /**
     * @function IPortalResource.prototype.update
     * @description 更新资源属性信息。
     * @returns {Promise} 返回包含更新操作状态的 Promise 对象。
     */
    update() {
        var resourceName = this.resourceType.replace("_","").toLowerCase();
        var options = {
            headers: {'Content-Type': 'application/x-www-form-urlencoded'}
        };
        if( resourceName === 'data') {
            this.resourceUrl = this.resourceUrl + "/attributes.json";
        }
        var entity = JSON.stringify(this.sourceJSON);
        //对服务资源进行编辑时，请求体内容只留关键字字段（目前如果是全部字段 更新返回成功 但其实没有真正的更新）
        if( resourceName === 'service') {
            var serviceInfo = {
                authorizeSetting:this.sourceJSON.authorizeSetting,
                metadata:this.sourceJSON.metadata,
                tags:this.sourceJSON.tags,
                thumbnail:this.sourceJSON.thumbnail,
                tokenRefreshUrl:this.sourceJSON.tokenRefreshUrl
            };
            entity = JSON.stringify(serviceInfo);
        }
        return this.request("PUT", this.resourceUrl, entity, options);
    }

}



;// CONCATENATED MODULE: ./src/common/iPortal/iPortalShareParam.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class IPortalShareParam
 * @aliasclass iPortalShareParam
 * @deprecatedclass SuperMap.iPortalShareParam
 * @classdesc iPortal 资源共享参数。
 * @version 10.0.1
 * @category iPortal/Online Resources ResourcesShare
 * @param {Object} params - 可选参数。
 * @param {ResourceType} [params.resourceType] - 资源类型。
 * @param {Array} [params.ids] - 资源的ID数组。
 * @param {IPortalShareEntity} [params.entities] - 资源的实体共享参数。
 * @usage
 */
class IPortalShareParam {

    constructor(params) {
        params = params || {};
        this.ids = [];
        this.entities = [];
        this.resourceType = ""; // MAP SERVICE SCENE DATA INSIGHTS_WORKSPACE MAP_DASHBOARD
        Util_Util.extend(this, params);
    }
}


;// CONCATENATED MODULE: ./src/common/iPortal/iPortal.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/







/**
 * @class IPortal
 * @aliasclass iPortal
 * @deprecatedclass SuperMap.iPortal
 * @classdesc 对接 SuperMap iPortal 基础服务。
 * @category iPortal/Online Resources
 * @modulecategory Services
 * @extends {IPortalServiceBase}
 * @param {string} iportalUrl - 服务地址。
 * @param {Object} options - 可选参数。
 * @param {boolean} [options.withCredentials] - 请求是否携带 cookie。
 * @usage
 */
class IPortal extends IPortalServiceBase {
    constructor(iportalUrl, options) {
        super(iportalUrl, options);
        this.iportalUrl = iportalUrl;
        options = options || {};
        this.withCredentials = options.withCredentials || false;
    }

    /**
     * @function IPortal.prototype.load
     * @description 加载页面。
     * @returns {Promise} 包含 iportal web 资源信息的 Promise 对象。
     */
    load() {
        return FetchRequest.get(this.iportalUrl + "/web");
    }

    /**
     * @function IPortal.prototype.queryResources
     * @description 查询资源。
     * @version 10.0.1
     * @param {IPortalQueryParam} queryParams - 查询参数。
     * @returns {Promise} 包含所有资源结果的 Promise 对象。
     */
    queryResources(queryParams) {
        if (!(queryParams instanceof IPortalQueryParam)) {
            return new Promise( function(resolve){
                resolve(
                    "queryParams is not instanceof iPortalQueryParam !"
                );
            });
        }
        var me = this;
        var resourceUrl = this.iportalUrl + "/gateway/catalog/resource/search.json";
        queryParams.t = new Date().getTime();
        return this.request("GET", resourceUrl, queryParams).then(function(result) {
            var content = [];
            (result.content || []).forEach(function(item) {
                content.push(new IPortalResource(me.iportalUrl, item));
            });
            let queryResult = new IPortalQueryResult();
            queryResult.content = content;
            queryResult.total = result.total;
            queryResult.currentPage = result.currentPage;
            queryResult.pageSize = result.pageSize;
            queryResult.aggregations = result.aggregations;
            return queryResult;
        });
    }


    /**
     * @function IPortal.prototype.updateResourcesShareSetting
     * @description 更新共享设置。
     * @version 10.0.1
     * @param {IPortalShareParam} shareParams - 共享的参数。
     * @returns {Promise} 包含共享资源结果的 Promise 对象。
     */
    updateResourcesShareSetting(shareParams) {
        if (!(shareParams instanceof IPortalShareParam)) {
            return new Promise( function(resolve){
                resolve(
                    "shareParams is not instanceof iPortalShareParam !"
                );
            });
        }
        var resourceUrlName = shareParams.resourceType.replace("_","").toLowerCase()+"s";
        if(resourceUrlName === "datas"){
            resourceUrlName = "mycontent/"+resourceUrlName;
        }
        var cloneShareParams = {
            ids: shareParams.ids,
            entities: shareParams.entities
        }
        var shareUrl = this.iportalUrl + "/web/"+resourceUrlName+"/sharesetting.json";
        return this.request("PUT", shareUrl, JSON.stringify(cloneShareParams)).then(function(result) {
            return result;
        });
    }
}

;// CONCATENATED MODULE: ./src/common/iPortal/iPortalShareEntity.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class IPortalShareEntity
 * @aliasclass iPortalShareEntity
 * @deprecatedclass SuperMap.iPortalShareEntity
 * @classdesc iPortal 资源共享实体参数。
 * @version 10.0.1
 * @category iPortal/Online Resources ResourcesShare
 * @param {Object} shareEntity - 可选参数。
 * @param {PermissionType} [shareEntity.permissionType] - 权限类型。
 * @param {EntityType} [shareEntity.entityType] - 实体类型。
 * @param {string} [shareEntity.entityName] - 实体 Name。对应的 USER（用户）、ROLE（角色）、GROUP（用户组）、IPORTALGROUP（群组）的名称。
 * @param {number} [shareEntity.entityId] - 实体的 ID。用于群组的授权。
 * @usage
 */
class IPortalShareEntity {

    constructor(shareEntity) {
        shareEntity = shareEntity || {};
        this.permissionType = ""; // SEARCH READ READWRITE DOWNLOAD
        this.entityType = ""; // USER DEPARTMENT IPORTALGROUP
        this.entityName = "GUEST"; // GUEST or 具体用户 name
        this.entityId = null;
        Util_Util.extend(this, shareEntity);
    }
}


;// CONCATENATED MODULE: ./src/common/iPortal/iPortalAddResourceParam.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/
 

 /**
  * @class IPortalAddResourceParam
  * @aliasclass iPortalAddResourceParam
  * @deprecatedclass SuperMap.iPortalAddResourceParam
  * @classdesc iPortal 添加资源参数。
  * @version 10.0.1
  * @category iPortal/Online Resources ResourcesShare
  * @param {Object} params - 可选参数。
  * @param {string} [params.rootUrl] - 服务地址。
  * @param {Array} [params.tags] - 标签。
  * @param {IPortalShareEntity} [params.entities] - 资源的实体共享参数。
  * @usage
  */
 class IPortalAddResourceParam {

     constructor(params) {
         params = params || {};
         this.rootUrl = "";
         this.tags = [];
         this.entities = [];
         Util_Util.extend(this, params);
     }
 }


;// CONCATENATED MODULE: ./src/common/iPortal/iPortalRegisterServiceParam.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class IPortalRegisterServiceParam
 * @aliasclass iPortalRegisterServiceParam
 * @deprecatedclass SuperMap.iPortalRegisterServiceParam
 * @classdesc iPortal 注册服务参数。
 * @version 10.0.1
 * @category iPortal/Online Resources Data
 * @param {Object} params - 可选参数。
 * @param {string} [params.type] - 服务类型。
 * @param {Array} [params.tags] - 服务标签。
 * @param {IPortalShareEntity} [params.entities] - 资源的实体共享参数。
 * @param {Object} [params.metadata] - 服务元信息。
 * @param {Array} [params.addedMapNames] - 地图服务列表。
 * @param {Array} [params.addedSceneNames] - 场景服务列表。
 * @usage
 */
class IPortalRegisterServiceParam {

    constructor(params) {
        params = params || {};
        this.type = ""; // SUPERMAP_REST ARCGIS_REST WMS WFS WCS WPS WMTS OTHERS
        this.tags = [];
        this.entities = [];
        this.metadata = {};
        this.addedMapNames = [];
        this.addedSceneNames = [];
        Util_Util.extend(this, params);
    }
}


;// CONCATENATED MODULE: ./src/common/iPortal/iPortalAddDataParam.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/
 

 /**
  * @class IPortalAddDataParam
  * @aliasclass iPortalAddDataParam
  * @deprecatedclass SuperMap.iPortalAddDataParam
  * @classdesc iPortal 上传/注册数据所需的参数。
  * @version 10.0.1
  * @category iPortal/Online Resources Data
  * @param {Object} params - 参数。
  * @param {string} params.fileName - 文件名称。
  * @param {DataItemType} params.type - 数据类型。
  * @param {Array} [params.tags] - 数据的标签。
  * @param {IPortalDataMetaInfoParam} [params.dataMetaInfo] - 数据元信息。
  * @usage
  */
 class IPortalAddDataParam {

     constructor(params) {
        params = params || {};
        this.fileName = "";
        this.type = "";
        this.tags = [];
        this.dataMetaInfo = {};
        Util_Util.extend(this, params);
     }
 }


;// CONCATENATED MODULE: ./src/common/iPortal/iPortalDataMetaInfoParam.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/
 

 /**
  * @class IPortalDataMetaInfoParam
  * @aliasclass iPortalDataMetaInfoParam
  * @deprecatedclass SuperMap.iPortalDataMetaInfoParam
  * @classdesc iPortal 上传数据/注册数据元信息所需的参数。
  * @version 10.0.1
  * @category iPortal/Online Resources Data
  * @param {Object} params - 参数。
  * @param {string} params.xField - X 坐标字段。
  * @param {string} params.yField - Y 坐标字段。
  * @param {number} params.xIndex - x所在列（关系型存储下CSV或EXCEL数据时必填）。
  * @param {number} params.yIndex - y所在列（关系型存储下CSV或EXCEL数据时必填）。
  * @param {Array.<string>} [params.fieldTypes] - 设置字段类型（关系型存储下CSV或EXCEL数据时可选填）。默认类型为：WTEXT。该参数按照CSV文件字段顺序从左到右依次设置，其中默认字段类型可省略不设置。例如，CSV文件中有10个字段，如果只需设定第1，2，4个字段，可设置为['a','b',,'c']。
  * @param {string} params.separator - 分隔符（关系型存储下CSV数据时必填）。
  * @param {boolean} params.firstRowIsHead - 是否带表头（关系型存储下CSV数据时必填）。
  * @param {boolean} params.url - HDFS注册目录地址。
  * @param {IPortalDataStoreInfoParam} params.dataStoreInfo - 注册数据时的数据存储信息。
  * @usage
  */
 class IPortalDataMetaInfoParam {

     constructor(params) {
        params = params || {};
        this.xField = "";
        this.yField = "";
        this.fileEncoding = "UTF-8";

        this.xIndex = 1;
        this.yIndex = 1;
        this.fieldTypes = [];
        this.separator = "";
        this.firstRowIsHead = true;

        this.url = "";
        this.dataStoreInfo = {};
        Util_Util.extend(this, params);
     }
 }


;// CONCATENATED MODULE: ./src/common/iPortal/iPortalDataStoreInfoParam.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/
 

 /**
  * @class IPortalDataStoreInfoParam
  * @aliasclass iPortalDataStoreInfoParam
  * @deprecatedclass SuperMap.iPortalDataStoreInfoParam
  * @classdesc iPortal 注册一个HBASE HDFS数据存储类。
  * @version 10.0.1
  * @category iPortal/Online Resources Data
  * @param {Object} params - 参数。
  * @param {string} params.type - 大数据文件共享类型和空间数据库类型，包括大数据文件共享HDFS 目录(HDFS)和空间数据库HBASE。
  * @param {string} params.url - HDFS数据存储目录地址。
  * @param {IPortalDataConnectionInfoParam} [params.connectionInfo] - HBASE空间数据库服务的连接信息。
  * @usage
  */
 class IPortalDataStoreInfoParam {

     constructor(params) {
        params = params || {};
        this.type = "";
        this.url = "";
        this.connectionInfo = {};
        Util_Util.extend(this, params);
     }
 }


;// CONCATENATED MODULE: ./src/common/iPortal/iPortalDataConnectionInfoParam.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/
 

 /**
  * @class IPortalDataConnectionInfoParam
  * @aliasclass iPortalDataConnectionInfoParam
  * @deprecatedclass SuperMap.iPortalDataConnectionInfoParam
  * @classdesc iPortal HBASE数据源连接信息类。
  * @version 10.0.1
  * @category iPortal/Online Resources Data
  * @param {Object} params - 参数。
  * @param {string} params.dataBase - 数据源连接的数据库名。
  * @param {string} params.server - 服务地址。
  * @usage
  */
 class IPortalDataConnectionInfoParam {

     constructor(params) {
        params = params || {};
        this.dataBase = "";
        this.server = "";
        Util_Util.extend(this, params);
     }
 }


;// CONCATENATED MODULE: ./src/common/iPortal/iPortalUser.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/







/**
 * @class IPortalUser
 * @aliasclass iPortalUser
 * @deprecatedclass SuperMap.iPortalUser
 * @classdesc iPortal 门户中用户信息的封装类。用于管理用户资源，包括可删除，添加资源。
 * @version 10.0.1
 * @category iPortal/Online Resources
 * @param {string} iportalUrl - 服务地址。
 * @extends {IPortalServiceBase}
 * @usage
 */
class IPortalUser extends IPortalServiceBase {
    constructor(iportalUrl) {
        super(iportalUrl);
        this.iportalUrl = iportalUrl;
    }

    /**
     * @function IPortalUser.prototype.deleteResources
     * @description 删除资源。
     * @param {Object} params - 删除资源所需的参数对象：{ids,resourceType}。
     * @returns {Promise} 返回包含删除操作状态的 Promise 对象。
     */
    deleteResources(params) {
        var resourceName = params.resourceType.replace("_","").toLowerCase();
        var deleteResourceUrl = this.iportalUrl+"/web/" + resourceName +"s.json?ids=" + encodeURI(JSON.stringify(params.ids));
        if( resourceName === 'data') {
            deleteResourceUrl = this.iportalUrl + "/web/mycontent/datas/delete.json";
            return this.request("POST", deleteResourceUrl, JSON.stringify(params.ids));
        }
        return this.request("DELETE", deleteResourceUrl);
    }

    /**
     * @function IPortalUser.prototype.addMap
     * @description 添加地图。
     * @version 10.1.0
     * @param {IPortalAddResourceParam} addMapParams - 添加地图的参数。
     * @returns {Promise} 返回包含添加地图结果的 Promise 对象。
     */
    addMap(addMapParams) {
        if (!(addMapParams instanceof IPortalAddResourceParam)) {
            return this.getErrMsgPromise("addMapParams is not instanceof IPortalAddResourceParam !");
        }
        let cloneAddMapParams = {
            rootUrl: addMapParams.rootUrl,
            tags: addMapParams.tags,
            authorizeSetting: addMapParams.entities
        }
        let addMapUrl = this.iportalUrl + "/web/maps/batchaddmaps.json";
        return this.request("POST", addMapUrl, JSON.stringify(cloneAddMapParams)).then(function(result) {
            return result;
        });
    }

    /**
     * @function IPortalUser.prototype.addScene
     * @description 添加场景。
     * @version 10.1.0
     * @param {IPortalAddResourceParam} addSceneParams - 添加场景的参数。
     * @returns {Promise} 返回包含添加场景结果的 Promise 对象。
     */
    addScene(addSceneParams) {
        if (!(addSceneParams instanceof IPortalAddResourceParam)) {
            return this.getErrMsgPromise("addSceneParams is not instanceof IPortalAddResourceParam !");
        }
        let cloneAddSceneParams = {
            rootUrl: addSceneParams.rootUrl,
            tags: addSceneParams.tags,
            authorizeSetting: addSceneParams.entities
        }
        let addSceneUrl = this.iportalUrl + "/web/scenes/batchaddscenes.json";
        return this.request("POST", addSceneUrl, JSON.stringify(cloneAddSceneParams)).then(function(result) {
            return result;
        });
    }

    /**
     * @function IPortalUser.prototype.registerService
     * @description 注册服务。
     * @version 10.1.0
     * @param {IPortalRegisterServiceParam} registerParams - 注册服务的参数。
     * @returns {Promise} 返回包含注册服务结果的 Promise 对象。
     */
    registerService(registerParams) {
        if(!(registerParams instanceof IPortalRegisterServiceParam)) {
            return this.getErrMsgPromise("registerParams is not instanceof IPortalRegisterServiceParam !");
        }
        let cloneRegisterParams = {
            type: registerParams.type,
            tags: registerParams.tags,
            authorizeSetting: registerParams.entities,
            metadata: registerParams.metadata,
            addedMapNames: registerParams.addedMapNames,
            addedSceneNames: registerParams.addedSceneNames
        }
        let registerUrl = this.iportalUrl + "/web/services.json";
        return this.request("POST", registerUrl, JSON.stringify(cloneRegisterParams)).then(result => {
            return result;
        });
    }

    /**
     * @function IPortalUser.prototype.getErrMsgPromise
     * @description 获取包含错误信息的Promise对象。
     * @version 10.1.0
     * @param {string} errMsg - 传入的错误信息。
     * @returns {Promise} 返回包含错误信息的 Promise 对象。
     */
    getErrMsgPromise(errMsg) {
        return new Promise(resolve => {
            resolve(errMsg);
        })
    }

    /**
     * @function IPortalUser.prototype.uploadDataRequest
     * @description 上传数据。
     * @version 10.1.0
     * @param {number} id - 上传数据的资源ID。
     * @param {Object} formData - 请求体为文本数据流。
     * @returns {Promise} 返回包含上传数据操作的 Promise 对象。
     */
    uploadDataRequest(id,formData) {
        var uploadDataUrl = this.iportalUrl + "/web/mycontent/datas/"+id+"/upload.json";
        return this.request("POST",uploadDataUrl,formData);
    }

    /**
     * @function IPortalUser.prototype.addData
     * @description 上传/注册数据。
     * @version 10.1.0
     * @param {IPortalAddDataParam} params - 上传/注册数据所需的参数。
     * @param {Object} [formData] - 请求体为文本数据流(上传数据时传入)。
     * @returns {Promise} 返回上传/注册数据的 Promise 对象。
     */
    addData(params,formData) {
        if(!(params instanceof IPortalAddDataParam)){
            return this.getErrMsgPromise("params is not instanceof iPortalAddDataParam !");
        }
        var datasUrl = this.iportalUrl + "/web/mycontent/datas.json";
        var entity = {
            fileName:params.fileName,
            tags:params.tags,
            type:params.type
        };
        var type = params.type.toLowerCase();
        var dataMetaInfo;
        if(type === "excel" || type === "csv"){
            if(!(params.dataMetaInfo instanceof IPortalDataMetaInfoParam)){
                return  this.getErrMsgPromise("params.dataMetaInfo is not instanceof iPortalDataMetaInfoParam !");
            }
            dataMetaInfo = {
                xField:params.dataMetaInfo.xField,
                yField:params.dataMetaInfo.yField
            }
            if(type === 'csv') {
                dataMetaInfo.fileEncoding = params.dataMetaInfo.fileEncoding
            }
            entity.coordType = "WGS84";
            entity.dataMetaInfo = dataMetaInfo;
        }else if(type === "hdfs" || type === "hbase") {
            if(!(params.dataMetaInfo instanceof IPortalDataMetaInfoParam)){
                return this.getErrMsgPromise("params.dataMetaInfo is not instanceof iPortalDataMetaInfoParam !");
            }
            if(!(params.dataMetaInfo.dataStoreInfo instanceof IPortalDataStoreInfoParam)){
                return this.getErrMsgPromise("params.dataMetaInfo.dataStoreInfo is not instanceof iPortalDataStoreInfoParam !");
            }
            var dataStoreInfo = {
                type:params.dataMetaInfo.dataStoreInfo.type
            }
            switch (type) {
                case "hdfs":
                    dataStoreInfo.url = params.dataMetaInfo.dataStoreInfo.url;
                    dataMetaInfo = {
                        url: params.dataMetaInfo.url,
                        dataStoreInfo:dataStoreInfo
                    }
                    break;
                case "hbase":
                    if(!(params.dataMetaInfo.dataStoreInfo.connectionInfo instanceof IPortalDataConnectionInfoParam)){
                        return this.getErrMsgPromise("params.dataMetaInfo.dataStoreInfo.connectionInfo is not instanceof iPortalDataConnectionInfoParam !");
                    }
                    dataStoreInfo.connectionInfo = {
                        dataBase:params.dataMetaInfo.dataStoreInfo.connectionInfo.dataBase,
                        server:params.dataMetaInfo.dataStoreInfo.connectionInfo.server,
                        engineType:'HBASE'
                    }
                    dataStoreInfo.datastoreType = "SPATIAL";//该字段SPATIAL表示HBASE注册
                    dataMetaInfo = {
                        dataStoreInfo:dataStoreInfo
                    }
                    break;
            }
            entity.dataMetaInfo = dataMetaInfo;
        }
        return this.request("POST",datasUrl,JSON.stringify(entity)).then(res=>{
            if(type === "hdfs" || type === "hbase"){
                return res;
            }else {
                if(res.childID) {
                    return this.uploadDataRequest(res.childID,formData);
                }else {
                    return res.customResult;
                }
            }
        })
    }

    /**
     * @function IPortalUser.prototype.publishOrUnpublish
     * @description 发布/取消发布。
     * @version 10.1.0
     * @param {Object} options - 发布/取消发布数据服务所需的参数。
     * @param {Object} options.dataId - 数据项ID。
     * @param {Object} options.serviceType - 发布的服务类型，目前支持发布的服务类型包括：RESTDATA, RESTMAP, RESTREALSPACE, RESTSPATIALANALYST。
     * @param {Object} [options.dataServiceId] - 发布的服务 ID。
     * @param {boolean} forPublish - 是否取消发布。
     * @returns {Promise} 返回发布/取消发布数据服务的 Promise 对象。
     */
    publishOrUnpublish(option,forPublish){
        if(!option.dataId || !option.serviceType) {
            return this.getErrMsgPromise("option.dataID and option.serviceType are Required!");
        }
        var dataId = option.dataId;
        var dataServiceId = option.dataServiceId;
        var serviceType = option.serviceType;
        var publishUrl = this.iportalUrl + "/web/mycontent/datas/" + dataId + "/publishstatus.json?serviceType=" + serviceType;
        if (dataServiceId) {
            publishUrl += "&dataServiceId=" + dataServiceId;
        }
        return this.request("PUT",publishUrl,JSON.stringify(forPublish)).then(res=>{
            // 发起服务状态查询
            if(forPublish) {
                // 发布服务的结果异步处理
                //  var publishStateUrl = this.iportalUrl + "web/mycontent/datas/" + dataId + "/publishstatus.rjson";
                if (!dataServiceId) { // 发布服务时会回传serviceIDs，发布服务之前serviceIDs为空
                    dataServiceId = res.customResult;
                }
                return dataServiceId;
            }else {
                // 取消发布的结果同步处理
                return res;
            }
        });
    }

    /**
     * @function IPortalUser.prototype.getDataPublishedStatus
     * @description 查询服务状态，发起服务状态查询。
     * @version 10.1.0
     * @param {number} dataId - 查询服务状态的数据项ID。
     * @param {string} dataServiceId - 发布的服务ID。
     * @returns {Promise} 返回查询服务状态的 Promise 对象。
     */
    getDataPublishedStatus(dataId,dataServiceId){
        var publishStateUrl = this.iportalUrl + "/web/mycontent/datas/" + dataId + "/publishstatus.json?dataServiceId="+dataServiceId+"&forPublish=true";
        return this.request("GET",publishStateUrl);
    }

    /**
     * @function IPortalUser.prototype.unPublishedDataService
     * @description 取消发布。
     * @version 10.1.0
     * @param {Object} options - 取消发布服务具体参数。
     * @param {Object} options.dataId - 数据项ID。
     * @param {Object} options.serviceType - 发布的服务类型，目前支持发布的服务类型包括：RESTDATA, RESTMAP, RESTREALSPACE, RESTSPATIALANALYST。
     * @param {Object} [options.dataServiceId] - 发布的服务 ID。
     * @returns {Promise} 返回取消发布数据服务的 Promise 对象。
     */
    unPublishDataService(option){
        return this.publishOrUnpublish(option,false);
    }

    /**
     * @function IPortalUser.prototype.publishedDataService
     * @description 发布数据服务。
     * @version 10.1.0
     * @param {Object} options - 发布数据服务具体参数。
     * @param {Object} options.dataId - 数据项ID。
     * @param {Object} options.serviceType - 发布的服务类型，目前支持发布的服务类型包括：RESTDATA, RESTMAP, RESTREALSPACE, RESTSPATIALANALYST。
     * @param {Object} [options.dataServiceId] - 发布的服务 ID。
     * @returns {Promise} 返回发布数据服务的 Promise 对象。
     */
    publishDataService(option){
        return this.publishOrUnpublish(option,true);
    }
}

;// CONCATENATED MODULE: ./src/common/iPortal/index.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





























;// CONCATENATED MODULE: ./src/common/iServer/CommonServiceBase.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/








/**
 * @class CommonServiceBase
 * @deprecatedclass SuperMap.CommonServiceBase
 * @category  iServer Core
 * @classdesc 对接 iServer 各种服务的 Service 的基类。
 * @param {string} url - 服务地址。
 * @param {Object} options - 参数。
 * @param {string} [options.proxy] - 服务代理地址。
 * @param {boolean} [options.withCredentials=false] - 请求是否携带 cookie。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class CommonServiceBase {
    constructor(url, options) {
        let me = this;

        this.EVENT_TYPES = ['processCompleted', 'processFailed'];

        this.events = null;

        this.eventListeners = null;

        this.url = null;

        this.urls = null;

        this.proxy = null;

        this.index = null;

        this.length = null;

        this.totalTimes = null;

        this.POLLING_TIMES = 3;

        this.isInTheSameDomain = null;

        this.withCredentials = false;

        if (Util_Util.isArray(url)) {
            me.urls = url;
            me.length = url.length;
            me.totalTimes = me.length;
            if (me.length === 1) {
                me.url = url[0];
            } else {
                me.index = parseInt(Math.random() * me.length);
                me.url = url[me.index];
            }
        } else {
            me.totalTimes = 1;
            me.url = url;
        }

        if (Util_Util.isArray(url) && !me.isServiceSupportPolling()) {
            me.url = url[0];
            me.totalTimes = 1;
        }

        options = options || {};
        this.crossOrigin = options.crossOrigin;
        this.headers = options.headers;
        Util_Util.extend(this, options);

        me.isInTheSameDomain = Util_Util.isInTheSameDomain(me.url);

        me.events = new Events(me, null, me.EVENT_TYPES, true);
        if (me.eventListeners instanceof Object) {
            me.events.on(me.eventListeners);
        }

        this.CLASS_NAME = 'SuperMap.CommonServiceBase';
    }

    /**
     * @function CommonServiceBase.prototype.destroy
     * @description 释放资源，将引用的资源属性置空。
     */
    destroy() {
        let me = this;
        if (Util_Util.isArray(me.urls)) {
            me.urls = null;
            me.index = null;
            me.length = null;
            me.totalTimes = null;
        }
        me.url = null;
        me.isInTheSameDomain = null;
        me.EVENT_TYPES = null;
        if (me.events) {
            me.events.destroy();
            me.events = null;
        }
        if (me.eventListeners) {
            me.eventListeners = null;
        }
    }

    /**
     * @function CommonServiceBase.prototype.request
     * @description: 该方法用于向服务发送请求。
     * @param {Object} options - 参数。
     * @param {string} [options.method='GET'] - 请求方式，包括 "GET"，"POST"，"PUT"，"DELETE"。
     * @param {string} [options.url] - 发送请求的地址。
     * @param {Object} [options.params] - 作为查询字符串添加到 URL 中的一组键值对，此参数只适用于 GET 方式发送的请求。
     * @param {string} [options.data] - 发送到服务器的数据。
     * @param {function} options.success - 请求成功后的回调函数。
     * @param {function} options.failure - 请求失败后的回调函数。
     * @param {Object} [options.scope] - 如果回调函数是对象的一个公共方法，设定该对象的范围。
     * @param {boolean} [options.isInTheSameDomain] - 请求是否在当前域中。
     * @param {boolean} [options.withCredentials=false] - 请求是否携带 cookie。
     * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
     * @param {Object} [options.headers] - 请求头。
     */
    request(options) {
        let format = options.scope.format;
        // 兼容 callback 未传，dataFormat 传入的情况
        if (typeof options.success === 'string') {
          options.scope.format = options.success;
          format = options.success;
          options.success = null;
          options.failure = null;
        }
       
        if (format && !this.supportDataFormat(format)) {
          throw new Error(`${this.CLASS_NAME} is not surport ${format} format!`);
        }
       
        let me = this;
        options.url = options.url || me.url;
        if (this._returnContent(options) && !options.url.includes('returnContent=true')) {
          options.url = Util_Util.urlAppend(options.url, 'returnContent=true');
        }
        options.proxy = options.proxy || me.proxy;
        options.withCredentials = options.withCredentials != undefined ? options.withCredentials : me.withCredentials;
        options.crossOrigin = options.crossOrigin != undefined ? options.crossOrigin : me.crossOrigin;
        options.headers = options.headers || me.headers;
        options.isInTheSameDomain = me.isInTheSameDomain;
        options.withoutFormatSuffix = options.scope.withoutFormatSuffix || false;
        //为url添加安全认证信息片段
        options.url = SecurityManager.appendCredential(options.url);

        me.calculatePollingTimes();
        options.scope = me;
        if (me.totalTimes > 0) {
          me.totalTimes--;
          return me.ajaxPolling(options);
        }
        return me._commit(options);
    }


    /**
     *
     * @function CommonServiceBase.prototype.ajaxPolling
     * @description 请求失败后，如果剩余请求失败次数不为 0，重新获取 URL 发送请求。
     * @param {Object} options - 请求参数对象。
     * @private
     */
    ajaxPolling(options) {
        let me = this,
            url = options.url,
            re = /^http:\/\/([a-z]{9}|(\d+\.){3}\d+):\d{0,4}/;
        me.index = parseInt(Math.random() * me.length);
        me.url = me.urls[me.index];
        url = url.replace(re, re.exec(me.url)[0]);
        options.url = url;
        options.isInTheSameDomain = Util_Util.isInTheSameDomain(url);
        return me._commit(options);
    }

    /**
     * @function CommonServiceBase.prototype.calculatePollingTimes
     * @description 计算剩余请求失败执行次数。
     */
    calculatePollingTimes() {
        let me = this;
        if (me.times) {
            if (me.totalTimes > me.POLLING_TIMES) {
                if (me.times > me.POLLING_TIMES) {
                    me.totalTimes = me.POLLING_TIMES;
                } else {
                    me.totalTimes = me.times;
                }
            } else {
                if (me.times < me.totalTimes) {
                    me.totalTimes = me.times;
                }
            }
        } else {
            if (me.totalTimes > me.POLLING_TIMES) {
                me.totalTimes = me.POLLING_TIMES;
            }
        }
        me.totalTimes--;
    }

    /**
     * @function CommonServiceBase.prototype.isServiceSupportPolling
     * @description 判断服务是否支持轮询。
     */
    isServiceSupportPolling() {
        let me = this;
        return !(
            me.CLASS_NAME === 'SuperMap.REST.ThemeService' || me.CLASS_NAME === 'SuperMap.REST.EditFeaturesService'
        );
    }

    /**
     * @function CommonServiceBase.prototype.transformResult
     * @description 状态完成时转换结果。
     * @param {Object} result - 服务器返回的结果对象。
     * @param {Object} options - 请求参数。
     * @return {Object} 转换结果。
     * @private
     */
    transformResult(result, options) {
        result = Util_Util.transformResult(result);
        return { result, options };
    }

    /**
     * @function CommonServiceBase.prototype.transformErrorResult
     * @description 状态失败时转换结果。
     * @param {Object} result - 服务器返回的结果对象。
     * @param {Object} options - 请求参数。
     * @return {Object} 转换结果。
     * @private
     */
    transformErrorResult(result, options) {
        result = Util_Util.transformResult(result);
        let error = result.error || result;
        return { error, options };
    }

    /**
    * @function CommonServiceBase.prototype.serviceProcessCompleted
    * @description 状态完成，执行此方法。
    * @param {Object} result - 服务器返回的结果对象。
    * @param {Object} options - 请求参数对象。
    * @private
    */
    serviceProcessCompleted(result, options) {
        result = this.transformResult(result).result;
        this.events.triggerEvent('processCompleted', {
            result: result,
            options: options
        });
    }

    /**
     * @function CommonServiceBase.prototype.serviceProcessFailed
     * @description 状态失败，执行此方法。
     * @param {Object} result - 服务器返回的结果对象。
     * @param {Object} options - 请求参数对象。对象
     * @private
     */
    serviceProcessFailed(result, options) {
      result = this.transformErrorResult(result).error;
      let error = result.error || result;
      this.events.triggerEvent('processFailed', {
          error: error,
          options: options
      });
    }

    _returnContent(options) {
      if (options.scope.format === DataFormat.FGB) {
        return false;
      }
      if (options.scope.returnContent) {
        return true;
      }
      return false;
    }

    supportDataFormat(foramt) {
      return this.dataFormat().includes(foramt);
    }

    dataFormat() {
      return [DataFormat.GEOJSON, DataFormat.ISERVER];
    }

    _commit(options) {
        if (options.method === 'POST' || options.method === 'PUT' || options.method === 'PATCH') {
            if (options.params) {
                options.url = Util_Util.urlAppend(options.url, Util_Util.getParameterString(options.params || {}));
            }
            if (typeof options.data === 'object') {
                try {
                    options.params = Util_Util.toJSON(options.data);
                } catch (e) {
                    console.log('不是json对象');
                }
            } else {
                options.params = options.data;
            }
        }
        return FetchRequest.commit(options.method, options.url, options.params, {
            headers: options.headers,
            withoutFormatSuffix: options.withoutFormatSuffix,
            withCredentials: options.withCredentials,
            crossOrigin: options.crossOrigin,
            timeout: options.async ? 0 : null,
            proxy: options.proxy
        })
            .then(function (response) {
                if (response.text) {
                    return response.text();
                }
                if (response.json) {
                    return response.json();
                }
                return response;
            })
            .then(function (text) {
                let requestResult = text;
                if (typeof text === 'string') {
                    requestResult = new JSONFormat().read(text);
                }
                if (
                    !requestResult ||
                    requestResult.error ||
                    (requestResult.code >= 300 && requestResult.code !== 304)
                ) {
                    if (requestResult && requestResult.error) {
                        requestResult = {
                            error: requestResult.error
                        };
                    } else {
                        requestResult = {
                            error: requestResult
                        };
                    }
                }
                if (requestResult && options.scope.format === DataFormat.FGB) {
                  requestResult.newResourceLocation = requestResult.newResourceLocation.replace('.json', '') + '.fgb';
                }
                return requestResult;
            })
            .catch(function (e) {
                return { error: e };
            })
            .then((requestResult) => {
                let response = {
                  object: this
                };
                if (requestResult.error) {
                  const type = 'processFailed';
                  // 兼容服务在构造函数中使用 eventListeners 的老用法
                  if (this.events && this.events.listeners[type] && this.events.listeners[type].length) {
                    var failure = options.failure && (options.scope ? FunctionExt.bind(options.failure, options.scope) : options.failure);
                    failure ? failure(requestResult, options) : this.serviceProcessFailed(requestResult, options);
                  } else {
                    response = {...response, ...this.transformErrorResult(requestResult, options)};
                    response.type = type;
                    options.failure && options.failure(response);
                  }
                } else {
                  const type = 'processCompleted';
                  if (this.events && this.events.listeners[type] && this.events.listeners[type].length) {
                    var success = options.success && (options.scope ? FunctionExt.bind(options.success, options.scope) : options.success);
                    success ? success(requestResult, options) : this.serviceProcessCompleted(requestResult, options);
                  } else {
                    requestResult.succeed = requestResult.succeed == undefined ? true : requestResult.succeed;
                    response = {...response, ...this.transformResult(requestResult, options)};
                    response.type = type;
                    options.success && options.success(response);
                  }
                }
                return response;
            });
    }
}


/**
 * 服务器请求回调函数。
 * @callback RequestCallback
 * @category BaseTypes Util
 * @example
 * var requestCallback = function (serviceResult){
 *      console.log(serviceResult.result);
 * }
 * new QueryService(url).queryByBounds(param, requestCallback);
 * @param {Object} serviceResult
 * @param {Object} serviceResult.result 服务器返回结果。
 * @param {Object} serviceResult.object 发布应用程序事件的对象。
 * @param {Object} serviceResult.type 事件类型。
 * @param {Object} serviceResult.options 请求参数。
 */

;// CONCATENATED MODULE: ./src/common/iServer/GeoCodingParameter.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class GeoCodingParameter
 * @deprecatedclass SuperMap.GeoCodingParameter
 * @category  iServer AddressMatch
 * @classdesc 地理正向匹配参数类。
 * @param {Object} options - 参数。
 * @param {string} options.address - 地点关键词。
 * @param {number} [options.fromIndex] - 设置返回对象的起始索引值。
 * @param {number} [options.toIndex] - 设置返回对象的结束索引值。
 * @param {Array.<string>} [options.filters] - 过滤字段，限定查询区域。
 * @param {string} [options.prjCoordSys] - 查询结果的坐标系。
 * @param {number} [options.maxReturn] - 最大返回结果数。
 * @usage
 */
class GeoCodingParameter {
    constructor(options) {
        if (options.filters && typeof(options.filters) === 'string') {
            options.filters =  options.filters.split(',');
        }
        /**
         * @member {string} GeoCodingParameter.prototype.address
         * @description 地点关键词。
         */
        this.address = null;

        /**
         * @member {number} [GeoCodingParameter.prototype.fromIndex]
         * @description 设置返回对象的起始索引值。
         */
        this.fromIndex = null;

        /**
         * @member {number} [GeoCodingParameter.prototype.toIndex]
         * @description 设置返回对象的结束索引值。
         */
        this.toIndex = null;

        /**
         * @member {Array.<string>} [GeoCodingParameter.prototype.filters]
         * @description 过滤字段，限定查询区域。
         */
        this.filters = null;

        /**
         * @member {string} [GeoCodingParameter.prototype.prjCoordSys]
         * @description  查询结果的坐标系。
         */
        this.prjCoordSys = null;

        /**
         * @member {number} [GeoCodingParameter.prototype.maxReturn]
         * @description 最大返回结果数。
         */
        this.maxReturn = null;
        Util_Util.extend(this, options);
    }

    /**
     * @function GeoCodingParameter.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        this.address = null;
        this.fromIndex = null;
        this.toIndex = null;
        this.filters = null;
        this.prjCoordSys = null;
        this.maxReturn = null;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/GeoDecodingParameter.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class GeoDecodingParameter
 * @deprecatedclass SuperMap.GeoDecodingParameter
 * @category iServer AddressMatch
 * @classdesc 地理反向匹配参数类。
 * @param {Object} options - 参数。
 * @param {number} options.x - 查询位置的横坐标。
 * @param {number} options.y - 查询位置的纵坐标。
 * @param {number} [options.fromIndex] - 设置返回对象的起始索引值。
 * @param {number} [options.toIndex] - 设置返回对象的结束索引值。
 * @param {Array.<string>} [options.filters] - 过滤字段，限定查询区域。
 * @param {string} [options.prjCoordSys] - 查询结果的坐标系。
 * @param {number} [options.maxReturn] - 最大返回结果数。
 * @param {number} [options.geoDecodingRadius] - 查询半径。
 * @usage
 */
class GeoDecodingParameter {


    constructor(options) {

        if (options.filters) {
            options.filters = options.filters.split(',');
        }
        /**
         * @member {number} GeoDecodingParameter.prototype.x
         * @description 查询位置的横坐标。
         */
        this.x = null;

        /**
         * @member {number} GeoDecodingParameter.prototype.y
         * @description 查询位置的纵坐标。
         */
        this.y = null;
        /**
         * @member {number} [GeoDecodingParameter.prototype.fromIndex]
         * @description  设置返回对象的起始索引值。
         */
        this.fromIndex = null;

        /**
         * @member {number} [GeoDecodingParameter.prototype.toIndex]
         * @description 设置返回对象的结束索引值。
         */
        this.toIndex = null;

        /**
         * @member {Array.<string>} [GeoDecodingParameter.prototype.filters]
         * @description 过滤字段，限定查询区域。
         */
        this.filters = null;

        /**
         * @member {string} [GeoDecodingParameter.prototype.prjCoordSys]
         * @description 查询结果的坐标系。
         */
        this.prjCoordSys = null;

        /**
         *  @member {number} [GeoDecodingParameter.prototype.maxReturn]
         *  @description 最大返回结果数。
         */
        this.maxReturn = null;

        /**
         * @member {number} GeoDecodingParameter.prototype.geoDecodingRadius
         * @description 查询半径。
         */
        this.geoDecodingRadius = null;
        Util_Util.extend(this, options);
    }

    /**
     * @function GeoDecodingParameter.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        this.x = null;
        this.y = null;
        this.fromIndex = null;
        this.toIndex = null;
        this.filters = null;
        this.prjCoordSys = null;
        this.maxReturn = null;
        this.geoDecodingRadius = null;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/AddressMatchService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class AddressMatchService
 * @deprecatedclass SuperMap.AddressMatchService
 * @category iServer AddressMatch
 * @classdesc 地址匹配服务，包括正向匹配和反向匹配。
 * @param {string} url - 服务地址。
 * @param {Object} options - 可选参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class AddressMatchService extends CommonServiceBase {
    constructor(url, options) {
        super(url, options);
        this.options = options || {};
        this.CLASS_NAME = 'SuperMap.AddressMatchService';
    }

    /**
     * @function AddressMatchService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function AddressMatchService.prototype.code
     * @param {string} url - 正向地址匹配服务地址。
     * @param {GeoCodingParameter} params - 正向地址匹配服务参数。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    code(url, params, callback) {
        if (!(params instanceof GeoCodingParameter)) {
            return;
        }
        return this.processAsync(url, params, callback);
    }

    /**
     * @function AddressMatchService.prototype.decode
     * @param {string} url - 反向地址匹配服务地址。
     * @param {GeoDecodingParameter} params - 反向地址匹配服务参数。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    decode(url, params, callback) {
        if (!(params instanceof GeoDecodingParameter)) {
            return;
        }
        return this.processAsync(url, params, callback);
    }

    /**
     * @function AddressMatchService.prototype.processAsync
     * @description 负责将客户端的动态分段服务参数传递到服务端。
     * @param {string} url - 服务地址。
     * @param {Object} params - 参数。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */

    processAsync(url, params, callback) {
      return this.request({
          method: 'GET',
          url,
          params,
          scope: this,
          success: callback,
          failure: callback
      });
    }
    /**
     * @function AddressMatchService.prototype.transformResult
     * @param {Object} result - 服务器返回的结果对象。
     * @param {Object} options - 请求参数。
     * @return {Object} 转换结果。
     * @description 状态完成时转换结果。
     */
    transformResult(result, options) {
        if (result.succeed) {
            delete result.succeed;
        }
        return { result, options };
    }
}


;// CONCATENATED MODULE: ./src/common/iServer/AggregationParameter.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class AggregationParameter
 * @deprecatedclass SuperMap.AggregationParameter
 * @classdesc 聚合查询参数设置，该参数仅支持数据来源 Elasticsearch 服务的数据服务。
 * @category iServer Data FeatureResults
 * @param {Object} options - 参数。
 * @param {string} options.aggName - 聚合名称。
 * @param {string} options.aggFieldName - 聚合字段。
 * @usage
 */
class AggregationParameter {
    constructor(options) {
        /**
         * @member {string} AggregationParameter.prototype.aggName
         * @description 聚合名称。
         */
        this.aggName = null;
        /**
         * @member {string} AggregationParameter.prototype.aggFieldName
         * @description 聚合字段。
         */
        this.aggFieldName = null;

        this.CLASS_NAME = 'SuperMap.AggregationParameter';
        Util_Util.extend(this, options);
    }

    destroy() {
        var me = this;
        me.aggName = null;
        me.aggFieldName = null;
        me.aggType = null;
    }
}


;// CONCATENATED MODULE: ./src/common/iServer/BucketAggParameter.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class BucketAggParameter
 * @deprecatedclass SuperMap.BucketAggParameter
 * @classdesc 子聚合类查询参数设置，该参数仅支持数据来源 Elasticsearch 服务的Supermap iServer的rest数据服务。
 * @category iServer Data FeatureResults
 * @param {Object} options - 参数。
 * @param {Array.<MetricsAggParameter>} options.subAggs - 子聚合类集合。
 * @extends {AggregationParameter}
 * @usage
 */
class BucketAggParameter extends AggregationParameter {
    constructor(options) {
        super();
        /**
         * @member {Array.<MetricsAggParameter>} BucketAggParameter.prototype.subAggs
         * @description 子聚合类集合。
         */
        this.subAggs = null;
        this.aggType = null;

        this.CLASS_NAME = 'SuperMap.BucketAggParameter';
        Util_Util.extend(this, options);
    }

    destroy() {
        var me = this;
        if (me.subAggs) {
            me.subAggs = null;
        }
    }
}


;// CONCATENATED MODULE: ./src/common/iServer/MetricsAggParameter.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class MetricsAggParameter
 * @deprecatedclass SuperMap.MetricsAggParameter
 * @classdesc 指标聚合查询参数类，该参数仅支持数据来源 Elasticsearch 服务的Supermap iServer的rest数据服务。
 * @category iServer Data FeatureResults
 * @param {Object} options - 可选参数。
 * @param {MetricsAggType} [options.aggType = 'avg'] - 聚合类型。
 * @extends {AggregationParameter}
 * @usage
 */
class MetricsAggParameter extends AggregationParameter {
    constructor(option) {
        super();
        /**
         * @member {MetricsAggType} [MetricsAggParameter.prototype.aggType=MetricsAggType.AVG]
         * @description 指标聚合类型。
         */
        this.aggType = MetricsAggType.AVG;

        Util_Util.extend(this, option);
        this.CLASS_NAME = 'SuperMap.MetricsAggParameter';
    }

    destroy() {
        super.destroy();
        var me = this;
        me.aggType = null;
    }
}


;// CONCATENATED MODULE: ./src/common/iServer/AreaSolarRadiationParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class AreaSolarRadiationParameters
 * @deprecatedclass SuperMap.AreaSolarRadiationParameters
 * @category iServer SpatialAnalyst SolarRadiationAnalyst
 * @classdesc 地区太阳辐射参数类。
 * @param {Object} options - 参数。
 * @param {string} options.dataset - 要用来做地区太阳辐射数据源中数据集的名称。该名称用形如"数据集名称@数据源别名"的形式来表示，例如：JingjinTerrain@Jingjin。
 * @param {string} options.targetDatasourceName - 指定的存储结果数据集的数据源名称, 例如："Jingjin"。
 * @param {string} options.totalGridName - 指定地区太阳辐射总辐射量数据集的名称。
 * @param {string} options.diffuseDatasetGridName - 指定地区太阳辐射散射辐射量数据集的名称。
 * @param {string} options.durationDatasetGridName - 指定地区太阳辐射太阳直射持续时间数据集的名称。
 * @param {string} options.directDatasetGridName - 指定地区太阳辐射直射辐射量数据集的名称。
 * @param {number} options.latitude - 待计算区域的纬度值。
 * @param {string} [options.timeMode = 'MULTIDAYS'] - 时间模式。可选值"WITHINDAY"（单日）或"MULTIDAYS"（多日）。
 * @param {number} options.dayStart - 起始日期（年内的第几天）。
 * @param {number} options.dayEnd - 结束日期（年内的第几天）。
 * @param {number} [options.hourStart] - 起始时间（一天中的第几个小时）。
 * @param {number} [options.hourEnd] - 结束时间（一天中的第几个小时）。
 * @param {number} [options.transmittance] - 太阳辐射穿过大气的透射率。
 * @param {number} [options.hourInterval=0.5] - 计算时的小时间隔（设置的越小计算量越大并且计算结果更精确，如果修改此参数，必须使用整数）。
 * @param {number} [options.dayInterval=5] - 计算时的天数间隔（设置的越小计算量越大并且计算结果更精确，必须使用整数）。
 * @param {boolean} [options.deleteExistResultDataset=false] - 如果用户命名的结果数据集名称与已有的数据集重名，是否删除已有的数据集。
 * @usage
 */
class AreaSolarRadiationParameters {

    constructor(options) {
        /**
         *  @member {string} AreaSolarRadiationParameters.prototype.dataset
         *  @description 要用来做地区太阳辐射数据源中数据集的名称。该名称用形如“数据集名称@数据源别名”形式来表示，例如：JingjinTerrain@Jingjin。注：地区太阳辐射数据必须为栅格数据集。
         */
        this.dataset = null;

        /**
         * @member {string} AreaSolarRadiationParameters.prototype.targetDatasourceName
         * @description 指定的存储结果数据集的数据源名称，例如："Jingjin"。
         */
        this.targetDatasourceName = null;

        /**
         * @member {string} AreaSolarRadiationParameters.prototype.totalGridName
         * @description 指定地区太阳辐射总辐射量数据集的名称。
         */
        this.totalGridName = null;

        /**
         * @member {string} AreaSolarRadiationParameters.prototype.diffuseDatasetGridName
         * @description 指定地区太阳辐射散射辐射量数据集的名称。
         */
        this.diffuseDatasetGridName = null;

        /**
         * @member {string} AreaSolarRadiationParameters.prototype.durationDatasetGridName
         * @description  指定地区太阳辐射太阳直射持续时间数据集的名称。
         */
        this.durationDatasetGridName = null;

        /**
         * @member {string} AreaSolarRadiationParameters.prototype.directDatasetGridName
         * @description 指定地区太阳辐射直射辐射量数据集的名称。
         */
        this.directDatasetGridName = null;

        /**
         * @member {number} AreaSolarRadiationParameters.prototype.latitude
         * @description 待计算区域的纬度值。
         */
        this.latitude = null;

        /**
         *  @member {string} [AreaSolarRadiationParameters.prototype.timeMode='MULTIDAYS']
         *  @description 时间模式。可选值"WITHINDAY"（单日）或"MULTIDAYS"（多日）。
         */
        this.timeMode = "MULTIDAYS";

        /**
         *  @member {number} AreaSolarRadiationParameters.prototype.dayStart
         *  @description 起始日期（年内的第几天）。
         */
        this.dayStart = null;

        /**
         *  @member {number} AreaSolarRadiationParameters.prototype.dayEnd
         * @description 结束日期（年内的第几天）。
         */
        this.dayEnd = null;

        /**
         * @member {number} [AreaSolarRadiationParameters.prototype.hourStart]
         * @description 起始时间（一天中的第几个小时）。
         */
        this.hourStart = null;

        /**
         * @member {number} [AreaSolarRadiationParameters.prototype.hourEnd]
         * @description 结束时间（一天中的第几个小时）。
         */
        this.hourEnd = null;

        /**
         * @member {number} [AreaSolarRadiationParameters.prototype.transmittance]
         * @description 太阳辐射穿过大气的透射率。
         */
        this.transmittance = null;

        /**
         *  @member {number} [AreaSolarRadiationParameters.prototype.hourInterval=0.5]
         *  @description 计算时的小时间隔（设置的越小计算量越大并且计算结果更精确, 如果修改此参数，必须使用整数）
         */
        this.hourInterval = null;

        /**
         * @member {number} [AreaSolarRadiationParameters.prototype.dayInterval=5]
         * @description 计算时的天数间隔（设置的越小计算量越大并且计算结果更精确, 必须使用整数）
         */
        this.dayInterval = null;

        /**
         * @member {boolean} [AreaSolarRadiationParameters.prototype.deleteExistResultDataset=false]
         * @description 如果用户命名的结果数据集名称与已有的数据集重名，是否删除已有的数据集。
         */
        this.deleteExistResultDataset = false;

        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.AreaSolarRadiationParameters";
    }

    /**
     * @function AreaSolarRadiationParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.dataset = null;
        me.zFactor = 1.0;
        me.averageCurvatureName = null;
        me.profileCurvatureName = null;
        me.planCurvatureName = null;
        me.deleteExistResultDataset = true;
    }

    /**
     * @function AreaSolarRadiationParameters.toObject
     * @param {AreaSolarRadiationParameters} param - 地区太阳辐射参数类。
     * @param {AreaSolarRadiationParameters} tempObj - 地区太阳辐射参数对象。
     * @returns {Object} JSON对象。
     * @description 将AreaSolarRadiationParameters对象转换成JSON对象。
     */
    static toObject(param, tempObj) {
        var parameter = {};
        for (var name in param) {
            if (name !== "dataset") {
                var name1 = (name === "latitude" || name === "timeMode" || name === "dayStart");
                var name2 = (name === "dayEnd" || name === "hourStart" || name === "hourEnd");
                var name3 = (name === "transmittance" || name === "hourInterval" || name === "dayInterval");
                if (name1 || name2 || name3) {
                    parameter[name] = param[name];
                } else {
                    tempObj[name] = param[name];
                }
            }
        }
        tempObj["parameter"] = parameter;
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/SpatialAnalystBase.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class SpatialAnalystBase
 * @deprecatedclass SuperMap.SpatialAnalystBase
 * @category  iServer Core
 * @classdesc 空间分析服务基类。
 * @param {string} url - 地址。
 * @param {Object} options - 可选参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @extends {CommonServiceBase}
 * @usage
 */
class SpatialAnalystBase extends CommonServiceBase {

    constructor(url, options) {
        super(url, options);
        /**
         * @member {DataFormat} [SpatialAnalystBase.prototype.format=DataFormat.GEOJSON]
         * @description 查询结果返回格式，目前支持 iServerJSON、GeoJSON、FGB 三种格式。参数格式为 "ISERVER"，"GEOJSON"，"FGB"。
         */
        this.format = (options && options.format) || DataFormat.GEOJSON;
        this.CLASS_NAME = "SuperMap.SpatialAnalystBase";
    }

    /**
     * @function SpatialAnalystBase.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        this.format = null;
    }

    /**
     * @function SpatialAnalystBase.prototype.transformResult
     * @description 状态完成时转换结果。
     * @param {Object} result - 服务器返回的结果对象。
     * @param {Object} options - 请求参数。
     * @return {Object} 转换结果。
     */
    transformResult(result, options) {
        var me = this, analystResult;
        result = Util_Util.transformResult(result);
        if (result && me.format === DataFormat.GEOJSON && typeof me.toGeoJSONResult === 'function') {
            //批量分析时会返回多个结果
            if (Util_Util.isArray(result)) {
                for (var i = 0; i < result.length; i++) {
                    result[i] = me.toGeoJSONResult(result[i])
                }
                analystResult = result;

            } else {
                analystResult = me.toGeoJSONResult(result);
            }
        }
        if (!analystResult) {
            analystResult = result;
        }
        return { result: analystResult, options };
    }

    /**
     * @function SpatialAnalystBase.prototype.toGeoJSONResult
     * @description 将含有 geometry 的数据转换为 GeoJSON 格式。
     * @param {Object} result - 服务器返回的结果对象。
     *
     */
    toGeoJSONResult(result) {
        if (!result) {
            return null;
        }
        //批量叠加分析时结果这样处理
        if (result.result && result.result.resultGeometry) {
            result = result.result
        }
        var geoJSONFormat = new GeoJSON();
        if (result.recordsets) {
            for (var i = 0, recordsets = result.recordsets, len = recordsets.length; i < len; i++) {
                if (recordsets[i].features) {
                    recordsets[i].features = geoJSONFormat.toGeoJSON(recordsets[i].features);
                }
            }
        } else if (result.recordset && result.recordset.features) {
            result.recordset.features =geoJSONFormat.toGeoJSON(result.recordset.features);
        }
        if (result.resultGeometry) {
            result.resultGeometry = geoJSONFormat.toGeoJSON(result.resultGeometry);
        }
        if (result.regions) {
            result.regions = geoJSONFormat.toGeoJSON(result.regions);
        }

        return result;
    }

}


;// CONCATENATED MODULE: ./src/common/iServer/AreaSolarRadiationService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class AreaSolarRadiationService
 * @deprecatedclass SuperMap.AreaSolarRadiationService
 * @category iServer SpatialAnalyst SolarRadiationAnalyst
 * @classdesc 地区太阳辐射服务类。
 * @param {string} url - 服务的访问地址。如：</br>http://localhost:8090/iserver/services/spatialanalyst-sample/restjsr/spatialanalyst。</br>
 * @param {Object} options - 参数。</br>
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @extends {SpatialAnalystBase}
 * @example 例如：
 * (start code)
 * var myAreaSolarRadiationService = new AreaSolarRadiationService(url);
 * (end)
 * @usage
 */
class AreaSolarRadiationService extends SpatialAnalystBase {

    constructor(url, options) {
        super(url, options);
        this.CLASS_NAME = "SuperMap.AreaSolarRadiationService";
    }

    /**
     * @function AreaSolarRadiationService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function AreaSolarRadiationService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {AreaSolarRadiationParameters} parameter - 地区太阳辐射参数。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(parameter, callback) {
        if (!(parameter instanceof AreaSolarRadiationParameters)) {
            return;
        }
        var me = this;
        var parameterObject = {};

        if (parameter instanceof AreaSolarRadiationParameters) {
            me.url = Util_Util.urlPathAppend(me.url, `datasets/${parameter.dataset}/solarradiation`);
        }
        me.url = Util_Util.urlAppend(me.url, 'returnContent=true');
        AreaSolarRadiationParameters.toObject(parameter, parameterObject);
        var jsonParameters = Util_Util.toJSON(parameterObject);

        return me.request({
            method: 'POST',
            data: jsonParameters,
            scope: me,
            success: callback,
            failure: callback
        });
    }
}



;// CONCATENATED MODULE: ./src/common/iServer/BufferDistance.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class BufferDistance
 * @deprecatedclass SuperMap.BufferDistance
 * @category iServer SpatialAnalyst BufferAnalyst
 * @classdesc 缓冲区分析的缓冲距离类。通过该类可以设置缓冲区分析的缓冲距离，距离可以是数值也可以是数值型的字段表达式。
 * @param {Object} options - 可选参数。
 * @param {string} [options.exp] - 以数值型的字段表达式作为缓冲区分析的距离值。
 * @param {number} [options.value=100] - 以数值作为缓冲区分析的距离值。单位：米。
 * @usage
 */
class BufferDistance {


    constructor(options) {
        /**
         * @member {string} [BufferDistance.prototype.exp]
         * @description 以数值型的字段表达式作为缓冲区分析的距离值。
         */
        this.exp = null;

        /**
         * @member {number} [BufferDistance.prototype.value=100]
         * @description 以数值作为缓冲区分析的距离值。单位：米。
         */
        this.value = 100;

        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.BufferDistance";
    }

    /**
     * @function BufferDistance.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        this.exp = null;
        this.value = null;
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/BufferSetting.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class BufferSetting
 * @deprecatedclass SuperMap.BufferSetting
 * @category iServer SpatialAnalyst BufferAnalyst
 * @classdesc 缓冲区分析通用设置类。
 * @param {Object} options - 可选参数。
 * @param {BufferEndType} [options.endType=BufferEndType.FLAT] - 缓冲区端点枚举值。
 * @param {BufferDistance} [options.leftDistance=100] - 左侧缓冲距离。
 * @param {BufferDistance} [options.rightDistance=100] - 右侧缓冲距离。
 * @param {number} [options.semicircleLineSegment=4] - 圆头缓冲圆弧处线段的个数。
 * @param {BufferRadiusUnit} [options.radiusUnit=BufferRadiusUnit.METER] - 缓冲半径单位。
 * @usage
 */
class BufferSetting {


    constructor(options) {
        /**
         * @member {BufferEndType} [BufferSetting.prototype.endType = BufferEndType.FLAT]
         * @description 缓冲区端点枚举值。分为平头和圆头两种。当设置缓冲区端点为平头时，左侧、右侧缓冲距离需为相同数值。
         */
        this.endType = BufferEndType.FLAT;

        /**
         * @member {BufferDistance} [BufferSetting.prototype.leftDistance=100]
         * @description 左侧缓冲距离。
         * 当为 GeometryBufferAnalyst 时，单位为默认地图的投影系的单位（如3857为米，4326为度），
         * 当为 DatasetBufferAnalyst 时，单位通过{@link BufferSetting.radiusUnit}设置（默认全部为米）。
         */
        this.leftDistance = new BufferDistance();

        /**
         * @member {BufferDistance} [BufferSetting.prototype.rightDistance=100]
         * @description 右侧缓冲距离。
         * 当为 GeometryBufferAnalyst 时，单位为默认地图的投影系的单位（如3857为米，4326为度），
         * 当为 DatasetBufferAnalyst 时，单位通过{@link BufferSetting#radiusUnit}设置（默认全部为米）。
         */
        this.rightDistance = new BufferDistance();

        /**
         * @member {number} [BufferSetting.prototype.semicircleLineSegment=4]
         * @description 圆头缓冲圆弧处线段的个数。即用多少个线段来模拟一个半圆。
         */
        this.semicircleLineSegment = 4;

        /**
         * @member {BufferRadiusUnit} [BufferSetting.prototype.radiusUnit = BufferRadiusUnit.METER]
         * @description 缓冲半径单位，可以是{@link BufferRadiusUnit.METER}、{@link BufferRadiusUnit.MILLIMETER}、
         * {@link BufferRadiusUnit.CENTIMETER}、{@link BufferRadiusUnit.DECIMETER}、{@link BufferRadiusUnit.KILOMETER}、
         * {@link BufferRadiusUnit.FOOT}、{@link BufferRadiusUnit.INCH}、{@link BufferRadiusUnit.MILE}、{@link BufferRadiusUnit.YARD}。
         * 仅对BufferAnalyst有效。
         */
        this.radiusUnit = BufferRadiusUnit.METER;

        if (options) {
            Util_Util.extend(this, options);
        }
        this.CLASS_NAME = "SuperMap.BufferSetting";
    }


    /**
     * @function BufferSetting.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        let me = this;
        me.endType = null;
        if (me.leftDistance) {
            me.leftDistance.destroy();
            me.leftDistance = null;
        }
        if (me.rightDistance) {
            me.rightDistance.destroy();
            me.rightDistance = null;
        }
        me.semicircleLineSegment = null;
        me.radiusUnit = null;
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/BufferAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class BufferAnalystParameters
 * @deprecatedclass SuperMap.BufferAnalystParameters
 * @category iServer SpatialAnalyst BufferAnalyst
 * @classdesc 缓冲区分析参数基类。
 * @param {Object} options - 参数。
 * @param {BufferSetting} [options.bufferSetting] - 设置缓冲区通用参数。为缓冲区分析提供必要的参数信息，包括左缓冲距离、右缓冲距离、端点类型、圆头缓冲圆弧处线段的个数信息。
 * @usage
 */
class BufferAnalystParameters {


    constructor(options) {
        var me = this;
        /**
         * @member {BufferSetting} [BufferAnalystParameters.prototype.bufferSetting]
         * @description 设置缓冲区通用参数。为缓冲区分析提供必要的参数信息，包括左缓冲距离、右缓冲距离、端点类型、圆头缓冲圆弧处线段的个数信息。
         */
        me.bufferSetting = new BufferSetting();
        Util_Util.extend(this, options);
        this.CLASS_NAME = "SuperMap.BufferAnalystParameters";
    }


    /**
     * @function BufferAnalystParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        if (me.bufferSetting) {
            me.bufferSetting.destroy();
            me.bufferSetting = null;
        }
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/DataReturnOption.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class DataReturnOption
 * @deprecatedclass SuperMap.DataReturnOption
 * @category iServer Data Dataset
 * @classdesc 数据返回设置类。
 * @param {Object} options - 参数。
 * @param {number} [options.expectCount=1000] - 设置返回的最大记录数，小于或者等于 0 时表示返回所有记录数。
 * @param {string} [options.dataset] - 设置结果数据集标识，当 dataReturnMode 为 {@link DataReturnMode.DATASET_ONLY}或{@link DataReturnMode.DATASET_AND_RECORDSET}时有效，
 *                                   作为返回数据集的名称。该名称用形如“数据集名称@数据源别名”形式来表示。
 * @param {DataReturnMode} [options.dataReturnMode=DataReturnMode.RECORDSET_ONLY] - 数据返回模式。
 * @param {boolean} [options.deleteExistResultDataset=true] - 如果用户命名的结果数据集名称与已有的数据集重名，是否删除已有的数据集。
 * @usage
 */
class DataReturnOption {


    constructor(options) {

        /**
         * @member {number} [DataReturnOption.prototype.expectCount=1000]
         * @description 设置返回的最大记录数，小于或者等于0时表示返回所有记录数。
         */
        this.expectCount = 1000;

        /**
         * @member {string} [DataReturnOption.prototype.dataset]
         * @description 设置结果数据集标识，当dataReturnMode为 {@link DataReturnMode.DATASET_ONLY}
         * 或{@link DataReturnMode.DATASET_AND_RECORDSET}时有效，
         * 作为返回数据集的名称。该名称用形如"数据集名称@数据源别名"形式来表示。
         */
        this.dataset = null;

        /**
         * @member {DataReturnMode} [DataReturnOption.prototype.dataReturnMode=DataReturnMode.RECORDSET_ONLY]
         * @description 数据返回模式。
         */
        this.dataReturnMode = DataReturnMode.RECORDSET_ONLY;

        /**
         * @member {boolean} [DataReturnOption.prototype.deleteExistResultDataset=true]
         * @description 如果用户命名的结果数据集名称与已有的数据集重名，是否删除已有的数据集。
         */
        this.deleteExistResultDataset = true;

        Util_Util.extend(this, options);
        this.CLASS_NAME = "SuperMap.DataReturnOption";
    }

    /**
     * @function DataReturnOption.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.expectCount = null;
        me.dataset = null;
        me.dataReturnMode = null;
        me.deleteExistResultDataset = null;
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/FilterParameter.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class FilterParameter
 * @deprecatedclass SuperMap.FilterParameter
 * @category iServer Data FeatureResults
 * @classdesc 查询过滤条件参数类。该类用于设置查询数据集的查询过滤参数。
 * @param {Object} options - 参数。
 * @param {string} options.attributeFilter - 属性过滤条件。
 * @param {string} options.name - 查询数据集名称或者图层名称。
 * @param {Array.<JoinItem>} [options.joinItems] - 与外部表的连接信息 JoinItem 数组。
 * @param {Array.<LinkItem>} [options.linkItems] - 与外部表的关联信息 LinkItem 数组。
 * @param {Array.<string>} [options.ids] - 查询 id 数组，即属性表中的 SmID 值。
 * @param {string} [options.orderBy] - 查询排序的字段，orderBy 的字段须为数值型的。
 * @param {string} [options.groupBy] - 查询分组条件的字段。
 * @param {Array.<string>} [options.fields] - 查询字段数组。
 * @usage
 */
class FilterParameter {


    constructor(options) {
        /**
         * @member {string} FilterParameter.prototype.attributeFilter
         * @description 属性过滤条件。
         * 相当于 SQL 语句中的 WHERE 子句，其格式为：WHERE <条件表达式>，
         * attributeFilter 就是其中的“条件表达式”。
         * 该字段的用法为 attributeFilter = "过滤条件"。
         * 例如，要查询字段 fieldValue 小于100的记录，设置 attributeFilter = "fieldValue < 100"；
         * 要查询字段 name 的值为“酒店”的记录，设置 attributeFilter = "name like '%酒店%'"，等等。
         */
        this.attributeFilter = null;

        /**
         * @member {string} FilterParameter.prototype.name
         * @description 查询数据集名称或者图层名称，根据实际的查询对象而定。
         * 一般情况下该字段为数据集名称，但在进行与地图相关功能的操作时，
         * 需要设置为图层名称（图层名称格式：数据集名称@数据源别名）。
         * 因为一个地图的图层可能是来自于不同数据源的数据集，
         * 而不同的数据源中可能存在同名的数据集，
         * 使用数据集名称不能唯一的确定数据集，
         * 所以在进行与地图相关功能的操作时，该值需要设置为图层名称。
         */
        this.name = null;

        /**
         * @member {Array.<JoinItem>} [FilterParameter.prototype.joinItems]
         * @description 与外部表的连接信息 JoinItem 数组。
         */
        this.joinItems = null;

        /**
         * @member {Array.<LinkItem>} [FilterParameter.prototype.linkItems]
         * @description 与外部表的关联信息 LinkItem 数组。
         */
        this.linkItems = null;

        /**
         * @member {Array.<string>} [FilterParameter.prototype.ids]
         * @description 查询 id 数组，即属性表中的 SmID 值。
         */
        this.ids = null;

        /**
         * @member {string} [FilterParameter.prototype.orderBy]
         * @description 查询排序的字段，orderBy的字段须为数值型的。
         * 相当于 SQL 语句中的 ORDER BY 子句，其格式为：ORDER BY <列名>，
         * 列名即属性表中每一列的名称，列又可称为属性，在 SuperMap 中又称为字段。
         * 对单个字段排序时，该字段的用法为 orderBy = "字段名"；
         * 对多个字段排序时，字段之间以英文逗号进行分割，用法为 orderBy = "字段名1, 字段名2"。
         * 例如，现有一个国家数据集，它有两个字段分别为“SmArea”和“pop_1994”，
         * 分别表示国家的面积和1994年的各国人口数量。
         * 如果要按照各国人口数量对记录进行排序，则 orderBy = "pop_1994"；
         * 如果要以面积和人口进行排序，则 orderBy = "SmArea, pop_1994"。
         */
        this.orderBy = null;


        /**
         * @member {string} [FilterParameter.prototype.groupBy]
         * @description 查询分组条件的字段。
         * 相当于 SQL 语句中的 GROUP BY 子句，其格式为：GROUP BY <列名>，
         * 列名即属性表中每一列的名称，列又可称为属性，在 SuperMap 中又称为字段。
         * 对单个字段分组时，该字段的用法为 groupBy = "字段名"；
         * 对多个字段分组时，字段之间以英文逗号进行分割，用法为 groupBy = "字段名1, 字段名2"。
         * 例如，现有一个全球城市数据集，该数据集有两个字段分别为“Continent”和“Country”，
         * 分别表示某个城市所属的洲和国家。
         * 如果要按照国家对全球的城市进行分组，可以设置 groupBy = "Country"；
         * 如果以洲和国家对城市进行分组，设置 groupBy = "Continent, Country"。
         */
        this.groupBy = null;

        /**
         * @member {Array.<string>} [FilterParameter.prototype.fields]
         * @description 查询字段数组，如果不设置则使用系统返回的所有字段。
         */
        this.fields = null;
        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.FilterParameter";
    }


    /**
     * @function FilterParameter.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.attributeFilter = null;
        me.name = null;
        if (me.joinItems) {
            for (let i = 0, joinItems = me.joinItems, len = joinItems.length; i < len; i++) {
                joinItems[i].destroy();
            }
            me.joinItems = null;
        }
        if (me.linkItems) {
            for (let i = 0, linkItems = me.linkItems, len = linkItems.length; i < len; i++) {
                linkItems[i].destroy();
            }
            me.linkItems = null;
        }
        me.ids = null;
        me.orderBy = null;
        me.groupBy = null;
        me.fields = null;
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/DatasetBufferAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class DatasetBufferAnalystParameters
 * @deprecatedclass SuperMap.DatasetBufferAnalystParameters
 * @category iServer SpatialAnalyst BufferAnalyst
 * @classdesc 数据集缓冲区分析参数类。
 * @param {Object} options - 参数。
 * @param {string} options.dataset - 要用来做缓冲区分析的数据源中数据集的名称。该名称用形如“数据集名称@数据源别名”形式来表示。
 * @param {FilterParameter} [options.filterQueryParameter] - 设置数据集中几何对象的过滤条件。只有满足此条件的几何对象才参与缓冲区分析。
 * @param {DataReturnOption} [options.resultSetting] - 结果返回设置类。
 * @param {boolean} [options.isAttributeRetained=true] - 是否保留进行缓冲区分析的对象的字段属性。当 isUnion 字段为 false 时该字段有效。
 * @param {boolean} [options.isUnion=false] - 是否将缓冲区与源记录集中的对象合并后返回。对于面对象而言，要求源数据集中的面对象不相交。
 * @param {BufferSetting} [options.bufferSetting] - 设置缓冲区通用参数。
 *
 * @extends {BufferAnalystParameters}
 * @usage
 */
class DatasetBufferAnalystParameters extends BufferAnalystParameters {


    constructor(options) {
        super(options);
        /**
         * @member {string} DatasetBufferAnalystParameters.prototype.dataset
         * @description 要用来做缓冲区分析的数据源中数据集的名称。该名称用形如“数据集名称@数据源别名”形式来表示。
         */
        this.dataset = null;

        /**
         *  @member {FilterParameter} [DatasetBufferAnalystParameters.prototype.filterQueryParameter]
         *  @description 设置数据集中几何对象的过滤条件。只有满足此条件的几何对象才参与缓冲区分析。
         */
        this.filterQueryParameter = new FilterParameter();

        /**
         * @member {DataReturnOption} [DatasetBufferAnalystParameters.prototype.resultSetting]
         * @description 结果返回设置类。
         */
        this.resultSetting = new DataReturnOption();

        /**
         * @member {boolean} [DatasetBufferAnalystParameters.prototype.isAttributeRetained=true]
         * @description 是否保留进行缓冲区分析的对象的字段属性。当 isUnion 字段为 false 时该字段有效。
         */
        this.isAttributeRetained = true;

        /**
         * @member {boolean} [DatasetBufferAnalystParameters.prototype.isUnion=false]
         * @description 是否将缓冲区与源记录集中的对象合并后返回。对于面对象而言，要求源数据集中的面对象不相交。
         */
        this.isUnion = false;

        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.DatasetBufferAnalystParameters";
    }


    /**
     * @function DatasetBufferAnalystParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        me.dataset = null;
        if (me.filterQueryParameter) {
            me.filterQueryParameter.destroy();
            me.filterQueryParameter = null;
        }
        if (me.resultSetting) {
            me.resultSetting.destroy();
            me.resultSetting = null;
        }
        me.isAttributeRetained = null;
        me.isUnion = null;
    }

    /**
     * @function DatasetBufferAnalystParameters.toObject
     * @param {DatasetBufferAnalystParameters} datasetBufferAnalystParameters - 数据集缓冲区分析参数类。
     * @param {DatasetBufferAnalystParameters} tempObj - 数据集缓冲区分析参数对象。
     * @description 将数据集缓冲区分析参数对象转换为 JSON 对象。
     * @returns {Object} JSON 对象。
     */
    static toObject(datasetBufferAnalystParameters, tempObj) {
        for (var name in datasetBufferAnalystParameters) {
            if (name === "bufferSetting") {
                datasetBufferAnalystParameters.bufferSetting.radiusUnit = datasetBufferAnalystParameters.bufferSetting.radiusUnit.toUpperCase();
                tempObj.bufferAnalystParameter = datasetBufferAnalystParameters.bufferSetting;
            } else if (name === "resultSetting") {
                tempObj.dataReturnOption = datasetBufferAnalystParameters.resultSetting;
            } else if (name === "dataset") {
                continue;
            } else {
                tempObj[name] = datasetBufferAnalystParameters[name];
            }
        }
    }

}


;// CONCATENATED MODULE: ./src/common/iServer/GeometryBufferAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class GeometryBufferAnalystParameters
 * @deprecatedclass SuperMap.GeometryBufferAnalystParameters
 * @category  iServer SpatialAnalyst BufferAnalyst
 * @classdesc 几何对象缓冲区分析参数类。
 * 对指定的某个几何对象做缓冲区分析。通过该类可以指定要做缓冲区分析的几何对象、缓冲区参数等。
 * @param {Object} options - 参数。
 * @param {GeoJSONObject} options.sourceGeometry - 要做缓冲区分析的几何对象。
 * @param {number} options.sourceGeometrySRID - 缓冲区几何对象投影坐标参数, 如 4326，3857。
 * @param {BufferSetting} [options.bufferSetting] - 设置缓冲区通用参数。
 * @extends {BufferAnalystParameters}
 * @usage
 */
class GeometryBufferAnalystParameters extends BufferAnalystParameters {


    constructor(options) {
        super(options);
        /**
         * @member {GeoJSONObject} GeometryBufferAnalystParameters.prototype.sourceGeometry
         * @description 要做缓冲区分析的几何对象。<br>
         * 点类型可以是：{@link GeometryPoint}|{@link L.Marker}|{@link L.CircleMarker}|{@link L.Circle}|{@link L.GeoJSON}|{@link ol.geom.Point}|{@link ol.format.GeoJSON}。</br>
         * 线类型可以是：{@link GeometryLineString}|{@link GeometryLinearRing}|{@link L.Polyline}|{@link L.GeoJSON}|{@link ol.geom.LineString}|{@link ol.format.GeoJSON}。</br>
         * 面类型可以是：{@link GeometryPolygon}|{@link L.Polygon}|{@link L.GeoJSON}|{@link ol.geom.Polygon}|{@link ol.format.GeoJSON}。
         */
        this.sourceGeometry = null;

        /**
         * @member {number} GeometryBufferAnalystParameters.prototype.sourceGeometrySRID
         * @description 缓冲区几何对象投影坐标参数, 如 4326，3857。
         */
        this.sourceGeometrySRID = null;

        if (options) {
            Util_Util.extend(this, options);
        }
        this.CLASS_NAME = " SuperMap.GeometryBufferAnalystParameters";
    }

    /**
     * @function GeometryBufferAnalystParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        if (me.sourceGeometry) {
            me.sourceGeometry.destroy();
            me.sourceGeometry = null;
        }
    }

    /**
     * @function GeometryBufferAnalystParameters.toObject
     * @param {GeometryBufferAnalystParameters} geometryBufferAnalystParameters - 几何对象缓冲区分析参数类。
     * @param {GeometryBufferAnalystParameters} tempObj - 几何对象缓冲区分析参数对象。
     * @description 将几何对象缓冲区分析参数对象转换为 JSON 对象。
     * @returns {Object} JSON 对象。
     */
    static toObject(geometryBufferAnalystParameters, tempObj) {
        for (var name in geometryBufferAnalystParameters) {
            if (name === "bufferSetting") {
                var tempBufferSetting = {};
                for (var key in geometryBufferAnalystParameters.bufferSetting) {
                    tempBufferSetting[key] = geometryBufferAnalystParameters.bufferSetting[key];
                }
                tempObj.analystParameter = tempBufferSetting;
            } else if (name === "sourceGeometry") {
                tempObj.sourceGeometry = ServerGeometry.fromGeometry(geometryBufferAnalystParameters.sourceGeometry);

            } else {
                tempObj[name] = geometryBufferAnalystParameters[name];
            }
        }
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/BufferAnalystService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/






/**
 * @class BufferAnalystService
 * @deprecatedclass SuperMap.BufferAnalystService
 * @category iServer SpatialAnalyst BufferAnalyst
 * @classdesc 缓冲区分析服务类。
 * 该类负责将客户设置的缓冲区分析参数传递给服务端，并接收服务端返回的缓冲区分析结果数据。
 * 缓冲区分析结果通过该类支持的事件的监听函数参数获取。
 * @param {string} url - 服务的访问地址。如：http://localhost:8090/iserver/services/spatialanalyst-changchun/restjsr/spatialanalyst。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @extends {SpatialAnalystBase}
 * @example 例如：
 * (start code)
 * var myBufferAnalystService = new BufferAnalystService(url);
 * (end)
 * @usage
 */
class BufferAnalystService extends SpatialAnalystBase {


    constructor(url, options) {
        super(url, options);
        /**
         * @member {string} BufferAnalystService.prototype.mode
         * @description 缓冲区分析类型
         */
        this.mode = null;
        if (options) {
            Util_Util.extend(this, options);
        }
        this.CLASS_NAME = "SuperMap.BufferAnalystService";
    }


    /**
     * @function BufferAnalystService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        this.mode = null;
    }


    /**
     * @method BufferAnalystService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {BufferAnalystParameters} parameter - 缓冲区分析参数
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(parameter, callback) {
        var parameterObject = {};
        var me = this;
        if (parameter instanceof DatasetBufferAnalystParameters) {
            me.mode = 'datasets';
            me.url = Util_Util.urlPathAppend(me.url, 'datasets/' + parameter.dataset + '/buffer');
            DatasetBufferAnalystParameters.toObject(parameter, parameterObject);
        } else if (parameter instanceof GeometryBufferAnalystParameters) {
            me.mode = 'geometry';
            me.url = Util_Util.urlPathAppend(me.url, 'geometry/buffer');
            GeometryBufferAnalystParameters.toObject(parameter, parameterObject);
        }

        var jsonParameters = Util_Util.toJSON(parameterObject);
        this.returnContent = true;
        return me.request({
            method: "POST",
            data: jsonParameters,
            scope: me,
            success: callback,
            failure: callback
        });
    }

    dataFormat() {
      return [DataFormat.GEOJSON, DataFormat.ISERVER, DataFormat.FGB];
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/DatasourceConnectionInfo.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/

 // eslint-disable-line no-unused-vars

/**
 * @class DatasourceConnectionInfo
 * @deprecatedclass SuperMap.DatasourceConnectionInfo
 * @category  iServer Data Datasource
 * @classdesc 数据源连接信息类。该类包括了进行数据源连接的所有信息，如所要连接的服务器名称、数据库名称、用户名以及密码等。
 *            当保存为工作空间时，工作空间中的数据源的连接信息都将存储到工作空间文件中。对于不同类型的数据源，其连接信息有所区别。
 *            所以在使用该类所包含的成员时，请注意该成员所适用的数据源类型。对于从数据源对象中返回的数据连接信息对象，只有 connect 方法可以被修改，
 *            其他内容是不可以被修改的。对于用户创建的数据源连接信息对象，其内容都可以修改。
 * @param {Object} options - 参数。
 * @param {string} options.alias - 数据源别名。
 * @param {string} options.dataBase - 数据源连接的数据库名。
 * @param {boolean} [options.connect] - 数据源是否自动连接数据。
 * @param {string} [options.driver] - 使用 ODBC(Open Database Connectivity，开放数据库互连)的数据库的驱动程序名。
 * @param {EngineType} [options.engineType] - 数据源连接的引擎类型。
 * @param {boolean} [options.exclusive] - 是否以独占方式打开数据源。
 * @param {boolean} [options.OpenLinkTable] - 是否把数据库中的其他非 SuperMap 数据表作为 LinkTable 打开。
 * @param {string} [options.password] - 登录数据源连接的数据库或文件的密码。
 * @param {boolean} [options.readOnly] - 是否以只读方式打开数据源。
 * @param {string} [options.server] - 数据库服务器名或 SDB 文件名。
 * @param {string} [options.user] - 登录数据库的用户名。
 * @usage
 */
class DatasourceConnectionInfo {


    constructor(options) {

        /**
         * @member {string} DatasourceConnectionInfo.prototype.alias
         * @description 数据源别名。
         */
        this.alias = null;

        /**
         * @member {boolean} [DatasourceConnectionInfo.prototype.connect]
         * @description 数据源是否自动连接数据。
         */
        this.connect = null;

        /**
         * @member {string} DatasourceConnectionInfo.prototype.dataBase
         * @description 数据源连接的数据库名。
         */
        this.dataBase = null;

        /**
         * @member {string} [DatasourceConnectionInfo.prototype.driver]
         * @description 使用 ODBC(Open Database Connectivity，开放数据库互连) 的数据库的驱动程序名。
         * 其中，对于 SQL Server 数据库与 iServer 发布的 WMTS 服务，此为必设参数。
         * 对于 SQL Server 数据库，它使用 ODBC 连接，所设置的驱动程序名为 "SQL Server" 或 "SQL Native Client"；
         * 对于 iServer 发布的 WMTS 服务，设置的驱动名称为 "WMTS"。
         */
        this.driver = null;

        /**
         * @member {EngineType} [DatasourceConnectionInfo.prototype.engineType]
         * @description 数据源连接的引擎类型。
         */
        this.engineType = null;

        /**
         * @member {boolean} [DatasourceConnectionInfo.prototype.exclusive]
         * @description 是否以独占方式打开数据源。
         */
        this.exclusive = null;

        /**
         * @member {boolean} [DatasourceConnectionInfo.prototype.OpenLinkTable]
         * @description 是否把数据库中的其他非 SuperMap 数据表作为 LinkTable 打开。
         */
        this.OpenLinkTable = null;

        /**
         * @member {string} [DatasourceConnectionInfo.prototype.password]
         * @description 登录数据源连接的数据库或文件的密码。
         */
        this.password = null;

        /**
         * @member {boolean} [DatasourceConnectionInfo.prototype.readOnly]
         * @description 是否以只读方式打开数据源。
         */
        this.readOnly = null;

        /**
         * @member {string} [DatasourceConnectionInfo.prototype.server]
         * @description 数据库服务器名、文件名或服务地址。
         * 1.对于 SDB 和 UDB 文件，为其文件的绝对路径。注意：当绝对路径的长度超过 UTF-8 编码格式的 260 字节长度，该数据源无法打开。
         * 2.对于 Oracle 数据库，其服务器名为其 TNS 服务名称。
         * 3.对于 SQL Server 数据库，其服务器名为其系统的 DSN(Database Source Name) 名称。
         * 4.对于 PostgreSQL 数据库，其服务器名为 “IP:端口号”，默认的端口号是 5432。
         * 5.对于 DB2 数据库，已经进行了编目，所以不需要进行服务器的设置。
         * 6.对于 Kingbase 数据库，其服务器名为其 IP 地址。
         * 7.对于 GoogleMaps 数据源，其服务器地址，默认设置为 “{@link http://maps.google.com}”，且不可更改。
         * 8.对于 SuperMapCould 数据源，为其服务地址。
         * 9.对于 MAPWORLD 数据源，为其服务地址，默认设置为 “{@link http://www.tianditu.cn}”，且不可更改。
         * 10.对于 OGC 和 REST 数据源，为其服务地址。
         */
        this.server = null;

        /**
         * @member {string} DatasourceConnectionInfo.prototype.user
         * @description 登录数据库的用户名。
         */
        this.user = null;

        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.DatasourceConnectionInfo";
    }

    /**
     * @function DatasourceConnectionInfo.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.alias = null;
        me.connect = null;
        me.dataBase = null;
        me.driver = null;
        me.engineType = null;
        me.exclusive = null;
        me.OpenLinkTable = null;
        me.password = null;
        me.readOnly = null;
        me.server = null;
        me.user = null;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/OutputSetting.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class OutputSetting
 * @deprecatedclass SuperMap.OutputSetting
 * @category  iServer ProcessingService
 * @classdesc 分布式分析输出类型设置类。
 * @param {Object} options - 参数。
 * @param {DatasourceConnectionInfo} options.datasourceInfo - 数据源连接信息。
 * @param {string} [options.datasetName='analystResult'] - 结果数据集名称。
 * @param {OutputType} [options.type=OutputType.UDB] - 输出类型。
 * @param {string} [options.outputPath] - 分析结果输出路径。
 * @usage
 */
class OutputSetting {

    constructor(options) {

        /**
         * @member {OutputType} OutputSetting.prototype.type
         * @description 分布式分析的输出类型。
         */
        this.type = OutputType.UDB;

        /**
         * @member {string} [OutputSetting.prototype.datasetName='analystResult']
         * @description 分布式分析的输出结果数据集名称。
         */
        this.datasetName = "analystResult";

        /**
         * @member {DatasourceConnectionInfo} OutputSetting.prototype.datasourceInfo
         * @description 分布式分析的输出结果数据源连接信息。
         */
        this.datasourceInfo = null;

        /**
         * @member {string} [OutputSetting.prototype.outputPath]
         * @description 分布式分析的分析结果输出路径。
         */
        this.outputPath = "";

        Util_Util.extend(this, options);
        this.CLASS_NAME = "SuperMap.OutputSetting";
    }

    /**
     * @function OutputSetting.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.type = null;
        me.datasetName = null;
        me.outputPath = null;
        if (me.datasourceInfo instanceof DatasourceConnectionInfo) {
            me.datasourceInfo.destroy();
            me.datasourceInfo = null;
        }
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/MappingParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class MappingParameters
 * @deprecatedclass SuperMap.MappingParameters
 * @category  iServer ProcessingService
 * @classdesc 分析后结果可视化的参数类。
 * @param {Object} options - 参数。
 * @param {Array.<ThemeGridRangeItem>} [options.items] - 栅格分段专题图子项数组。
 * @param {number} [options.numericPrecision=1] - 精度，此字段用于设置分析结果标签专题图中标签数值的精度，如“1”表示精确到小数点的后一位。
 * @param {RangeMode} [options.rangeMode=RangeMode.EQUALINTERVAL] - 专题图分段模式。
 * @param {number} [options.rangeCount] - 专题图分段个数。
 * @param {ColorGradientType} [options.colorGradientType=ColorGradientType.YELLOW_RED] - 专题图颜色渐变模式。
 * @usage
 */
class MappingParameters {

    constructor(options) {

        /**
         * @member {Array.<ThemeGridRangeItem>} [MappingParameters.prototype.items]
         * @description 栅格分段专题图子项数组。
         */
        this.items = null;

        /**
         * @member {number} [MappingParameters.prototype.numericPrecision=1]
         * @description 精度，此字段用于设置分析结果标签专题图中标签数值的精度，如“1”表示精确到小数点的后一位。
         */
        this.numericPrecision = 1;

        /**
         * @member {RangeMode} [MappingParameters.prototype.RangeMode=RangeMode.EQUALINTERVAL]
         * @description 专题图分段模式。
         */
        this.rangeMode = RangeMode.EQUALINTERVAL;

        /**
         * @member {number} [MappingParameters.prototype.rangeCount]
         * @description 专题图分段个数。
         */
        this.rangeCount = "";

        /**
         * @member {ColorGradientType} [MappingParameters.prototype.colorGradientType=ColorGradientType.YELLOW_RED]
         * @description 专题图颜色渐变模式。
         */
        this.colorGradientType = ColorGradientType.YELLOW_RED;

        Util_Util.extend(this, options);
        this.CLASS_NAME = "SuperMap.MappingParameters";
    }

    /**
     * @function MappingParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        if (me.items) {
            if (me.items.length > 0) {
                for (var item in me.items) {
                    me.items[item].destroy();
                    me.items[item] = null;
                }
            }
            me.items = null;
        }
        me.numericPrecision = null;
        me.rangeMode = null;
        me.rangeCount = null;
        me.colorGradientType = null;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/BuffersAnalystJobsParameter.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class BuffersAnalystJobsParameter
 * @deprecatedclass SuperMap.BuffersAnalystJobsParameter
 * @category iServer ProcessingService BufferAnalyst
 * @classdesc 缓冲区分析任务参数类。
 * @param {Object} options - 参数。
 * @param {string} options.datasetName - 数据集名。
 * @param {(SuperMap.Bounds|L.Bounds|L.LatLngBounds|ol.extent|mapboxgl.LngLatBounds|GeoJSONObject)} [options.bounds] - 分析范围（默认为全图范围）。
 * @param {string} [options.distance='15'] - 缓冲距离，或缓冲区半径。
 * @param {string} [options.distanceField='pickup_latitude'] - 缓冲区分析距离字段。
 * @param {AnalystSizeUnit} [options.distanceUnit=AnalystSizeUnit.METER] - 缓冲距离单位单位。
 * @param {OutputSetting} [options.output] - 输出参数设置。
 * @param {MappingParameters} [options.mappingParameters] - 分析后结果可视化的参数类。
 * @usage
 */
class BuffersAnalystJobsParameter {
    constructor(options) {
        /**
         * @member {string} BuffersAnalystJobsParameter.prototype.datasetName
         * @description 数据集名。
         */
        this.datasetName = '';

        /**
         * @member {(SuperMap.Bounds|L.Bounds|L.LatLngBounds|ol.extent|mapboxgl.LngLatBounds|GeoJSONObject)} BuffersAnalystJobsParameter.prototype.bounds
         * @description 分析范围。
         */
        this.bounds = '';

        /**
         * @member {string} [BuffersAnalystJobsParameter.prototype.distance='15']
         * @description 缓冲距离，或称为缓冲区半径。当缓冲距离字段为空时，此参数有效。
         */
        this.distance = '';

        /**
         * @member {string} [BuffersAnalystJobsParameter.prototype.distanceField='pickup_latitude']
         * @description 缓冲距离字段。
         */
        this.distanceField = '';

        /**
         * @member {AnalystSizeUnit} [BuffersAnalystJobsParameter.prototype.distanceUnit=AnalystSizeUnit.METER]
         * @description 缓冲距离单位。
         */
        this.distanceUnit = AnalystSizeUnit.METER;

        /**
         * @member {string} BuffersAnalystJobsParameter.prototype.dissolveField
         * @description 融合字段，根据字段值对缓冲区结果面对象进行融合。
         */
        this.dissolveField = '';

        /**
         * @member {OutputSetting} [BuffersAnalystJobsParameter.prototype.output]
         * @description 输出参数设置类。
         */
        this.output = null;

        /**
         * @member {MappingParameters} [BuffersAnalystJobsParameter.prototype.mappingParameters]
         * @description 分析后结果可视化的参数类。
         */
        this.mappingParameters = null;

        if (!options) {
            return this;
        }
        Util_Util.extend(this, options);

        this.CLASS_NAME = 'SuperMap.BuffersAnalystJobsParameter';
    }

    /**
     * @function BuffersAnalystJobsParameter.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        this.datasetName = null;
        this.bounds = null;
        this.distance = null;
        this.distanceField = null;
        this.distanceUnit = null;
        this.dissolveField = null;
        if (this.output instanceof OutputSetting) {
            this.output.destroy();
            this.output = null;
        }
        if (this.mappingParameters instanceof MappingParameters) {
            this.mappingParameters.destroy();
            this.mappingParameters = null;
        }
    }

    /**
     * @function BuffersAnalystJobsParameter.toObject
     * @param {BuffersAnalystJobsParameter} BuffersAnalystJobsParameter - 缓冲区分析任务参数。
     * @param {Object} tempObj - 目标对象。
     * @description 生成缓冲区分析任务对象。
     */
    static toObject(BuffersAnalystJobsParameter, tempObj) {
        for (var name in BuffersAnalystJobsParameter) {
            if (name === 'datasetName') {
                tempObj['input'] = tempObj['input'] || {};
                tempObj['input'][name] = BuffersAnalystJobsParameter[name];
                continue;
            }
            if (name === 'output') {
                tempObj['output'] = tempObj['output'] || {};
                tempObj['output'] = BuffersAnalystJobsParameter[name];
                continue;
            }

            tempObj['analyst'] = tempObj['analyst'] || {};
            if (name === 'bounds' && BuffersAnalystJobsParameter[name]) {
                tempObj['analyst'][name] = BuffersAnalystJobsParameter[name].toBBOX();
            } else {
                tempObj['analyst'][name] = BuffersAnalystJobsParameter[name];
            }
            if (name === 'mappingParameters') {
                tempObj['analyst'][name] = tempObj['analyst'][name] || {};
                tempObj['analyst']['mappingParameters'] = BuffersAnalystJobsParameter[name];
            }
        }
    }
}


;// CONCATENATED MODULE: ./src/common/iServer/ProcessingServiceBase.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class ProcessingServiceBase
 * @deprecatedclass SuperMap.ProcessingServiceBase
 * @category  iServer Core
 * @classdesc 分布式分析服务基类
 * @extends {CommonServiceBase}
 * @param {string} url - 服务地址。
 * @param {Object} options - 参数。
 * @param {Events} options.events - 处理所有事件的对象。
 * @param {number} options.index - 服务访问地址在数组中的位置。
 * @param {number} options.length - 服务访问地址数组长度。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class ProcessingServiceBase extends CommonServiceBase {

    constructor(url, options) {
        options = options || {};
        super(url, options);

        this.CLASS_NAME = "SuperMap.ProcessingServiceBase";
    }

    /**
     * @function ProcessingServiceBase.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function ProcessingServiceBase.prototype.getJobs
     * @description 获取分布式分析任务。
     * @param {string} url - 资源地址。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    getJobs(url, callback) {
        var me = this;
        return FetchRequest.get(SecurityManager.appendCredential(url), null, {
            proxy: me.proxy
        }).then(function (response) {
            return response.json();
        }).then(function (result) {
            const res = { result, object: me, type: 'processCompleted' };
            callback(res);
            return res;
        }).catch(function (e) {
          const res = { error: e, object: me, type: 'processFailed' };
          callback(res);
          return res;
        });
    }

    /**
     * @function ProcessingServiceBase.prototype.addJob
     * @description 添加分布式分析任务。
     * @param {string} url - 资源根地址。
     * @param {Object} params - 创建一个空间分析的请求参数。
     * @param {string} paramType - 请求参数类型。
     * @param {number} seconds - 开始创建后，获取创建成功结果的时间间隔。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @param {RequestCallback} [processRunningCallback] - 回调函数。
     * @returns {Promise} Promise 对象。
     */
    addJob(url, params, paramType, seconds, callback, processRunningCallback) {
        var me = this,
            parameterObject = null;
        if (params && params instanceof paramType) {
            parameterObject = new Object();
            paramType.toObject(params, parameterObject);
        }
        let headers = Object.assign({
          'Content-Type': 'application/x-www-form-urlencoded'
        }, me.headers || {})
        var options = {
            proxy: me.proxy,
            headers,
            withCredentials: me.withCredentials,
            crossOrigin: me.crossOrigin,
            isInTheSameDomain: me.isInTheSameDomain
        };
        return FetchRequest.post(SecurityManager.appendCredential(url), JSON.stringify(parameterObject), options).then(function (response) {
            return response.json();
        }).then(function (result) {
            if (result.succeed) {
                return me.transformResult(result, seconds, callback, processRunningCallback);
            } else {
              result = me.transformErrorResult(result);
              result.options = me;
              result.type = 'processFailed';
              callback(result);
              return result;
            }
        }).catch(function (e) {
            e = me.transformErrorResult({ error: e });
            e.options = me;
            e.type = 'processFailed';
            callback(e);
            return e;
        });
    }

    transformResult(result, seconds, callback, processRunningCallback) {
        result = Util_Util.transformResult(result);
        seconds = seconds || 1000;
        var me = this;
        if (result) {
           return new Promise((resolve) => {
              var id = setInterval(function () {
                FetchRequest.get(SecurityManager.appendCredential(result.newResourceLocation), {
                        _t: new Date().getTime()
                    })
                    .then(function (response) {
                        return response.json();
                    }).then(function (job) {
                        resolve({
                          object: me,
                          id: job.id,
                          state: job.state
                        });
                        processRunningCallback({
                            id: job.id,
                            state: job.state,
                            object: me
                        });
                        if (job.state.runState === 'LOST' || job.state.runState === 'KILLED' || job.state.runState === 'FAILED') {
                            clearInterval(id);
                            const res = {
                              error: job.state.errorMsg,
                              state: job.state.runState,
                              object: me,
                              type: 'processFailed'
                            };
                            resolve(res);
                            callback(res);
                        }
                        if (job.state.runState === 'FINISHED' && job.setting.serviceInfo) {
                            clearInterval(id);
                            const res = {
                              result: job,
                              object: me, 
                              type: 'processCompleted'
                            };
                            resolve(res);
                            callback(res);
                        }
                    }).catch(function (e) {
                        clearInterval(id);
                        const res = {
                          error: e,
                          object: me,
                          type: 'processFailed'
                        };
                        resolve(res);
                        callback(res);
                    });
            }, seconds);
           });
        }
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/BuffersAnalystJobsService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class BuffersAnalystJobsService
 * @deprecatedclass SuperMap.BuffersAnalystJobsService
 * @category iServer ProcessingService BufferAnalyst
 * @classdesc 缓冲区分析服务类。
 * @extends {ProcessingServiceBase}
 * @param {string} url - 服务地址。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class BuffersAnalystJobsService extends ProcessingServiceBase {
    constructor(url, options) {
        super(url, options);
        this.url = Util_Util.urlPathAppend(this.url, 'spatialanalyst/buffers');
        this.CLASS_NAME = 'SuperMap.BuffersAnalystJobsService';
    }

    /**
     *@override
     */
    destroy() {
      super.destroy();
    }

    /**
     * @function BuffersAnalystJobsService.prototype.getBufferJobs
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @description 获取缓冲区分析所有任务
     * @returns {Promise} Promise 对象。
     */
    getBuffersJobs(callback) {
      return super.getJobs(this.url, callback);
    }

    /**
     * @function BuffersAnalystJobsService.prototype.getBufferJob
     * @description 获取指定id的缓冲区分析服务
     * @param {string} id - 指定要获取数据的id。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    getBuffersJob(id, callback) {
        return super.getJobs(Util_Util.urlPathAppend(this.url, id), callback);
    }

    /**
     * @function BuffersAnalystJobsService.prototype.addBufferJob
     * @description 新建缓冲区分析服务
     * @param {BuffersAnalystJobsParameter} params - 创建一个空间分析的请求参数。
     * @param {number} seconds - 开始创建后，获取创建成功结果的时间间隔。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @param {RequestCallback} [processRunningCallback] - 回调函数。
     * @returns {Promise} Promise 对象。
     */
    addBuffersJob(params, seconds, callback, processRunningCallback) {
        return super.addJob(this.url, params, BuffersAnalystJobsParameter, seconds, callback, processRunningCallback);
    }
}


;// CONCATENATED MODULE: ./src/common/iServer/BurstPipelineAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class BurstPipelineAnalystParameters
 * @deprecatedclass SuperMap.BurstPipelineAnalystParameters
 * @category iServer NetworkAnalyst BurstAnalyse
 * @classdesc 爆管分析参数类。
 * @param {Object} options - 参数。
 * @param {Array.<number>} options.sourceNodeIDs - 指定的设施点 ID 数组。
 * @param {number} [options.edgeID] - 指定的弧段ID，edgeID 与 nodeID 必须指定一个。
 * @param {number} [options.nodeID] - 指定的结点ID，edgeID 与 nodeID 必须指定一个。
 * @param {boolean} [options.isUncertainDirectionValid=false] - 指定不确定流向是否有效。
 * @usage
 */
class BurstPipelineAnalystParameters {


    constructor(options) {
        var me = this;
        /**
         * @member {Array.<number>} BurstPipelineAnalystParameters.prototype.sourceNodeIDs
         * @description 指定的设施点 ID 数组。
         */
        this.sourceNodeIDs = null;

        /**
         * @member {number} [BurstPipelineAnalystParameters.prototype.edgeID]
         * @description 指定的弧段 ID，edgeID 与 nodeID 必须指定一个。
         */
        this.edgeID = null;

        /**
         * @member {number} [BurstPipelineAnalystParameters.prototype.nodeID]
         * @description 指定的结点 ID，edgeID 与 nodeID 必须指定一个。
         */
        this.nodeID = null;

        /**
         * @member {boolean} [BurstPipelineAnalystParameters.prototype.isUncertainDirectionValid=false]
         * @description 指定不确定流向是否有效。
         * 指定为 true，表示不确定流向有效，遇到不确定流向时分析继续进行。
         * 指定为 false，表示不确定流向无效，遇到不确定流向将停止在该方向上继续查找。
         */
        this.isUncertainDirectionValid = false;

        Util_Util.extend(me, options);

        this.CLASS_NAME = "SuperMap.BurstPipelineAnalystParameters";
    }

    /**
     * @function BurstPipelineAnalystParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.sourceNodeIDs = null;
        me.edgeID = null;
        me.nodeID = null;
        me.isUncertainDirectionValid = null;
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/NetworkAnalystServiceBase.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class NetworkAnalystServiceBase
 * @deprecatedclass SuperMap.NetworkAnalystServiceBase
 * @category iServer Core
 * @classdesc 网络分析服务基类。
 * @extends {CommonServiceBase}
 * @param {string} url - 服务地址。
 * @param {Object} options - 可选参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class NetworkAnalystServiceBase extends CommonServiceBase {

    constructor(url, options) {
        super(url, options);
        /**
         * @member {DataFormat} [NetworkAnalystServiceBase.prototype.format=DataFormat.GEOJSON]
         * @description 查询结果返回格式，目前支持 iServerJSON 和 GeoJSON 两种格式，参数格式为 "ISERVER","GEOJSON"
         */
        this.format = DataFormat.GEOJSON;

        this.CLASS_NAME = "SuperMap.NetworkAnalystServiceBase";
    }

    /**
     * @function NetworkAnalystServiceBase.prototype.destroy
     * @description 释放资源，将引用的资源属性置空。
     */
    destroy() {
        super.destroy();
        this.format = null;
    }

    /**
     * @function NetworkAnalystServiceBase.prototype.transformResult
     * @description 状态完成时转换结果。
     * @param {Object} result - 服务器返回的结果对象。
     * @param {Object} options - 请求参数。
     * @return {Object} 转换结果。
     */
    transformResult(result, options) {
        var me = this, analystResult;
        result = Util_Util.transformResult(result);
        if (result && me.format === DataFormat.GEOJSON && typeof me.toGeoJSONResult === 'function') {
            analystResult = me.toGeoJSONResult(result);
        }
        if (!analystResult) {
            analystResult = result;
        }
        return { result: analystResult, options };
    }

    /**
     * @function NetworkAnalystServiceBase.prototype.toGeoJSONResult
     * @description 将含有 geometry 的数据转换为 GeoJSON 格式。只处理结果中的路由，由子类实现。
     * @param {Object} result - 服务器返回的结果对象。
     * @returns {GeoJSONObject} GeoJSON 对象。
     */
    toGeoJSONResult(result) { // eslint-disable-line no-unused-vars
        return null;
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/BurstPipelineAnalystService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class BurstPipelineAnalystService
 * @deprecatedclass SuperMap.BurstPipelineAnalystService
 * @category iServer NetworkAnalyst BurstAnalyse
 * @classdesc 爆管分析服务类。即将给定弧段或节点作为爆管点来进行分析，返回关键结点 ID 数组、普通结点 ID 数组及其上下游弧段 ID 数组。
 * @extends {NetworkAnalystServiceBase}
 * @param {string} url - 网络分析服务地址。请求网络分析服务，URL应为：
 *                       http://{服务器地址}:{服务端口号}/iserver/services/{网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}，
 *                       例如: "http://localhost:8090/iserver/services/test/rest/networkanalyst/WaterNet@FacilityNet"。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class BurstPipelineAnalystService extends NetworkAnalystServiceBase {

    constructor(url, options) {
        super(url, options);
        this.CLASS_NAME = "SuperMap.BurstPipelineAnalystService";
    }

    /**
     * @function BurstPipelineAnalystService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function BurstPipelineAnalystService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @params {BurstPipelineAnalystParameters} params - 爆管分析参数类
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof BurstPipelineAnalystParameters)) {
            return null;
        }
        var me = this, jsonObject;
        me.url = Util_Util.urlPathAppend(me.url, 'burstAnalyse');
        jsonObject = {
            sourceNodeIDs: params.sourceNodeIDs,
            isUncertainDirectionValid: params.isUncertainDirectionValid
        };

        //必传参数不正确，就终止
        if (params.edgeID !== null && params.nodeID !== null) {
            throw new Error('edgeID and nodeID cannot be null at the same time.');
        }
        if (params.edgeID === null && params.nodeID === null) {
            throw new Error('edgeID and nodeID cannot be null at the same time.');
        }
        if (params.edgeID !== null) {
            jsonObject.edgeID = params.edgeID;
        } else {
            jsonObject.nodeID = params.nodeID;
        }

        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/ChartFeatureInfoSpecsService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class ChartFeatureInfoSpecsService
 * @deprecatedclass SuperMap.ChartFeatureInfoSpecsService
 * @category  iServer Map Chart
 * @classdesc 海图物标信息服务类，通过该服务类可以查询到服务端支持的所有海图物标信息。
 *            用户可以通过两种方式获取查询结果：
 *            一种是通过监听 ChartFeatureInfoSpecsEvent.PROCESS_COMPLETE 事件；
 *            另一种是使用 AsyncResponder 类实现异步处理。
 * @extends {CommonServiceBase}
 * @param {string} url - 地图（特指海图）服务地址。
 *        如："http://localhost:8090/iserver/services/map-ChartW/rest/maps/海图"。
 *        发送请求格式类似于："http://localhost:8090/iserver/services/map-ChartW/rest/maps/海图/chartFeatureInfoSpecs.json"。
 * @param {Object} options - 参数。
 * @param {DataFormat} [options.format] - 查询结果返回格式，目前支持 iServerJSON 和 GeoJSON 两种格式，参数格式为"ISERVER","GEOJSON"。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class ChartFeatureInfoSpecsService extends CommonServiceBase {

    constructor(url, options) {
        super(url, options);
        this.CLASS_NAME = "SuperMap.ChartFeatureInfoSpecsService";
    }

    /**
     * @function ChartFeatureInfoSpecsService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        Util_Util.reset(this);
    }

    /**
     * @function ChartFeatureInfoSpecsService.prototype.processAsync
     * @description 根据地图（特指海图）服务地址与服务端完成异步通讯，获取物标信息。
     *              当查询物标信息成功时，将触发 ChartFeatureInfoSpecsEvent.PROCESS_COMPLETE
     *              事件。用可以通过户两种方式获取图层信息:
     *              1. 通过 AsyncResponder 类获取（推荐使用）；
     *              2. 通过监听 ChartFeatureInfoSpecsEvent.PROCESS_COMPLETE 事件获取。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(callback) {
        var me = this, method = "GET";
        if (!me.isTempLayers) {
            Util_Util.urlPathAppend(me.url,'chartFeatureInfoSpecs');
        }
        return me.request({
            method: method,
            params: null,
            scope: me,
            success: callback,
            failure: callback
        });
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/ChartQueryFilterParameter.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class ChartQueryFilterParameter
 * @deprecatedclass SuperMap.ChartQueryFilterParameter
 * @category  iServer Map Chart
 * @classdesc 海图查询过滤参数类，用于设置海图查询的过滤参数。包括：物标代码、物标可应用对象的选择（是否查询点、线或面）、属性字段过滤条件。
 * @param {Object} options - 参数。
 * @param {string} options.attributeFilter - 属性字段过滤条件。
 * @param {number} options.chartFeatureInfoSpecCode - 查询的物标代号。
 * @param {boolean} [options.isQueryPoint] - 是否查询点。
 * @param {boolean} [options.isQueryLine] - 是否查询线。
 * @param {boolean} [options.isQueryRegion] - 是否查询面。
 * @usage
 */
class ChartQueryFilterParameter {


    constructor(options) {
        /**
         * @member {boolean} [ChartQueryFilterParameter.prototype.isQueryPoint]
         * @description 是否查询点。
         */
        this.isQueryPoint = null;

        /**
         * @member {boolean} [ChartQueryFilterParameter.prototype.isQueryLine]
         * @description 是否查询线。
         */
        this.isQueryLine = null;

        /**
         * @member {boolean} [ChartQueryFilterParameter.prototype.isQueryRegion]
         * @description 是否查询面。
         */
        this.isQueryRegion = null;

        /**
         * @member {string} ChartQueryFilterParameter.prototype.attributeFilter
         * @description 属性字段过滤条件。
         */
        this.attributeFilter = null;

        /**
         * @member {number} ChartQueryFilterParameter.prototype.chartFeatureInfoSpecCode
         * @description 查询的物标代号。
         */
        this.chartFeatureInfoSpecCode = null;

        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.ChartQueryFilterParameter";
    }

    /**
     * @function ChartQueryFilterParameter.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.isQueryPoint = null;
        me.isQueryLine = null;
        me.isQueryRegion = null;
        me.attributeFilter = null;
        me.chartFeatureInfoSpecCode = null;
    }

    /**
     * @function ChartQueryFilterParameter.prototype.toJson
     * @description 将属性信息转化成 JSON 格式字符串。
     */
    toJson() {
        var json = "";
        json += "\"isQueryPoint\":" + this.isQueryPoint + ",";
        json += "\"isQueryLine\":" + this.isQueryLine + ",";
        json += "\"isQueryRegion\":" + this.isQueryRegion + ",";
        if (this.attributeFilter) {
            json += "\"attributeFilter\": \"" + this.attributeFilter + "\",";
        }
        json += "\"chartFeatureInfoSpecCode\":" + this.chartFeatureInfoSpecCode;
        json = "{" + json + "}";
        return json;
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/ChartQueryParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class ChartQueryParameters
 * @deprecatedclass SuperMap.ChartQueryParameters
 * @category iServer Map Chart
 * @classdesc 海图查询参数类，该类用于设置海图查询时的相关参数，海图查询分为海图属性查询和海图范围查询两类，通过属性 queryMode 指定查询模式。
 *            必设属性有：queryMode、chartLayerNames、chartQueryFilterParameters。当进行海图范围查询时，必设属性还包括 bounds。
 * @param {Object} options - 参数。
 * @param {string} options.queryMode - 海图查询模式类型，支持两种查询方式：海图属性查询（"ChartAttributeQuery"）和海图空间查询（"ChartBoundsQuery"）。
 * @param {Array.<string>} options.chartLayerNames - 查询的海图图层的名称。
 * @param {Array.<ChartQueryFilterParameter>} options.chartQueryFilterParameters - 海图查询过滤参数。包括：物标代码、物标可应用对象的选择（是否查询点、线或面）、属性字段过滤条件。
 * @param {(SuperMap.Bounds|L.Bounds|L.LatLngBounds|ol.extent|mapboxgl.LngLatBounds|GeoJSONObject)} options.bounds - 海图查询范围。当进行海图范围查询时，此为必选参数。
 * @param {boolean} [options.returnContent=true] - 获取或设置是返回查询结果记录集 recordsets，还是返回查询结果的资源 resourceInfo。
 * @param {number} [options.startRecord=0] - 查询起始记录位置。
 * @param {number} [options.expectCount] - 期望查询结果返回的记录数，该值大于0。
 * @usage
 */
class ChartQueryParameters {


    constructor(options) {
        /**
         * @member {string} ChartQueryParameters.prototype.queryMode
         * @description 海图查询模式类型，支持两种查询方式：海图属性查询（"ChartAttributeQuery"）和海图空间查询（"ChartBoundsQuery"）。
         */
        this.queryMode = null;

        /**
         * @member {(SuperMap.Bounds|L.Bounds|L.LatLngBounds|ol.extent|mapboxgl.LngLatBounds|GeoJSONObject)} ChartQueryParameters.prototype.bounds
         * @description 海图查询范围。
         */
         this.bounds = null;

        /**
         * @member {Array.<string>} ChartQueryParameters.prototype.chartLayerNames
         * @description 查询的海图图层的名称。
         */
        this.chartLayerNames = null;

        /**
         * @member {Array.<ChartQueryFilterParameter>} ChartQueryParameters.prototype.chartQueryFilterParameters
         * @description 海图查询过滤参数。包括：物标代码、物标可应用对象的选择（是否查询点、线或面）、属性字段过滤条件。
         */
        this.chartQueryFilterParameters = null;

        /**
         * @member {boolean} [ChartQueryParameters.prototype.returnContent=true]
         * @description 获取或设置是返回查询结果记录集 recordsets，还是返回查询结果的资源 resourceInfo。
         */
        this.returnContent = true;

        /**
         * @member {number} [ChartQueryParameters.prototype.startRecord=0]
         * @description 查询起始记录位置。
         */
        this.startRecord = 0;

        /**
         * @member {number} [ChartQueryParameters.prototype.expectCount]
         * @description 期望查询结果返回的记录数，该值大于0。
         */
        this.expectCount = null;
        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.ChartQueryParameters";
    }

    /**
     * @function ChartQueryParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.queryMode = null;
        me.bounds = null;
        me.chartLayerNames = null;
        me.chartQueryFilterParameters = null;
        me.returnContent = true;
        me.startRecord = 0;
        me.expectCount = null;
    }

    /**
     * @function ChartQueryParameters.prototype.getVariablesJson
     * @description 将属性信息转换成能够被服务识别的 JSON 格式字符串。
     * @returns {string} JSON 字符串。
     */
    getVariablesJson() {
        var json = "";

        json += "\"queryMode\":\"" + this.queryMode + "\",";

        if (this.chartLayerNames && this.chartLayerNames.length) {
            var chartLayersArray = [];
            var layerLength = this.chartLayerNames.length;
            for (var i = 0; i < layerLength; i++) {
                chartLayersArray.push("\"" + this.chartLayerNames[i] + "\"");
            }
            var layerNames = "[" + chartLayersArray.join(",") + "]";
            json += "\"chartLayerNames\":" + layerNames + ",";
        }

        if (this.queryMode === "ChartBoundsQuery" && this.bounds) {
            json += "\"bounds\":" + "{" + "\"leftBottom\":" + "{" + "\"x\":" + this.bounds.left + "," +
                "\"y\":" + this.bounds.bottom + "}" + "," + "\"rightTop\":" + "{" + "\"x\":" + this.bounds.right + "," +
                "\"y\":" + this.bounds.top + "}" + "},";
        }

        if (this.chartQueryFilterParameters && this.chartQueryFilterParameters.length) {
            var chartParamArray = [];
            var chartLength = this.chartQueryFilterParameters.length;
            for (var j = 0; j < chartLength; j++) {
                var chartQueryFilterParameter = this.chartQueryFilterParameters[j];
                if (!(chartQueryFilterParameter instanceof ChartQueryFilterParameter)) {
                    continue;
                }
                chartParamArray.push(chartQueryFilterParameter.toJson());
            }
            var chartParamsJson = "[" + chartParamArray.join(",") + "]";
            chartParamsJson = "\"chartQueryParams\":" + chartParamsJson + ",";
            chartParamsJson += "\"startRecord\":" + this.startRecord + ",";
            chartParamsJson += "\"expectCount\":" + this.expectCount;
            chartParamsJson = "{" + chartParamsJson + "}";
            json += "\"chartQueryParameters\":" + chartParamsJson;
        }
        json = "{" + json + "}";
        return json;
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/QueryParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class QueryParameters
 * @deprecatedclass SuperMap.QueryParameters
 * @category  iServer Map QueryResults
 * @classdesc 查询参数基类。距离查询、SQL 查询、几何地物查询等各自的参数均继承此类。
 * @param {Object} options - 参数。
 * @param {Array.<FilterParameter>} options.queryParams - 查询过滤条件参数数组。
 * @param {string} [options.customParams] - 自定义参数，供扩展使用。
 * @param {Object} [options.prjCoordSys] - 自定义参数，供 SuperMap Online 提供的动态投影查询扩展使用。如 {"epsgCode":3857}。
 * @param {number} [options.expectCount=100000] - 期望返回结果记录个数。
 * @param {GeometryType} [options.networkType=GeometryType.LINE] - 网络数据集对应的查询类型。
 * @param {QueryOption} [options.queryOption=QueryOption.ATTRIBUTEANDGEOMETRY] - 查询结果类型枚举类。
 * @param {number} [options.startRecord=0] - 查询起始记录号。
 * @param {number} [options.holdTime=10] - 资源在服务端保存的时间,单位为分钟。
 * @param {boolean} [options.returnCustomResult=false] - 仅供三维使用。
 * @param {boolean} [options.returnFeatureWithFieldCaption = false] - 返回的查询结果要素字段标识是否为字段别名。为 false 时，返回的是字段名；为 true 时，返回的是字段别名。
 * @usage
 */
class QueryParameters {


    constructor(options) {
        if (!options) {
            return;
        }
        /**
         * @member {string} [QueryParameters.prototype.customParams]
         * @description 自定义参数，供扩展使用。
         */
        this.customParams = null;

        /**
         * @member {Object} [QueryParameters.prototype.prjCoordSys]
         * @description 自定义参数，供 SuperMap Online 提供的动态投影查询扩展使用。如 {"epsgCode":3857}
         */
        this.prjCoordSys = null;

        /**
         * @member {number} [QueryParameters.prototype.expectCount=100000]
         * @description 期望返回结果记录个数，默认返回100000条查询记录，
         *              如果实际不足100000条则返回实际记录条数。
         */
        this.expectCount = 100000;

        /**
         * @member {GeometryType} [QueryParameters.prototype.networkType=GeometryType.LINE]
         * @description 网络数据集对应的查询类型，分为点和线两种类型。
         */
        this.networkType = REST_GeometryType.LINE;

        /**
         * @member {QueryOption} [QueryParameters.prototype.queryOption=QueryOption.ATTRIBUTEANDGEOMETRY]
         * @description 查询结果类型枚举类。
         *              该类描述查询结果返回类型，包括只返回属性、
         *              只返回几何实体以及返回属性和几何实体。
         */
        this.queryOption = QueryOption.ATTRIBUTEANDGEOMETRY;

        /**
         * @member {Array.<FilterParameter>} QueryParameters.prototype.queryParams
         * @description 查询过滤条件参数数组。
         *              该类用于设置查询数据集的查询过滤参数。
         */
        this.queryParams = null;

        /**
         * @member {number} [QueryParameters.prototype.startRecord=0]
         * @description 查询起始记录号。
         */
        this.startRecord = 0;

        /**
         * @member {number} [QueryParameters.prototype.holdTime=10]
         * @description 资源在服务端保存的时间，单位为分钟。
         */
        this.holdTime = 10;

        /**
         * @member {boolean} [QueryParameters.prototype.returnCustomResult=false]
         * @description 仅供三维使用。
         */
        this.returnCustomResult = false;
        /**
         * @member {boolean} [QueryParameters.prototype.returnFeatureWithFieldCaption=false]
         * @description 返回的查询结果要素字段标识是否为字段别名。为 false 时，返回的是字段名；为 true 时，返回的是字段别名。
         */
        this.returnFeatureWithFieldCaption = false;
        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.QueryParameters";
    }

    /**
     * @function QueryParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.customParams = null;
        me.expectCount = null;
        me.networkType = null;
        me.queryOption = null;
        if (me.queryParams) {
            for (var i = 0, qps = me.queryParams, len = qps.length; i < len; i++) {
                qps[i].destroy();
            }
            me.queryParams = null;
        }
        me.startRecord = null;
        me.holdTime = null;
        me.returnCustomResult = null;
        me.prjCoordSys = null;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/ChartQueryService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/







/**
 * @class ChartQueryService
 * @deprecatedclass SuperMap.ChartQueryService
 * @category  iServer Map Chart
 * @classdesc 海图查询服务类。该类负责将海图查询所需参数（ChartQueryParameters）传递至服务端，并获取服务端的返回结果。
 *      用户可以通过两种方式获取查询结果：
 *      1.通过 AsyncResponder 类获取（推荐使用）；
 *      2.通过监听 QueryEvent.PROCESS_COMPLETE 事件获取。
 * @extends {CommonServiceBase}
 * @param {string} url - 地图查询服务访问地址。如："http://localhost:8090/iserver/services/map-ChartW/rest/maps/海图"。
 * @param {Object} options - 参数。
 * @param {DataFormat} [options.format] - 查询结果返回格式，目前支持 iServerJSON 和 GeoJSON 两种格式。参数格式为"ISERVER","GEOJSON"。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @example
 * 下面示例显示了如何进行海图属性查询：
 * var nameArray = ["GB4X0000_52000"];
 * var chartQueryFilterParameter = new ChartQueryFilterParameter({
 *       isQueryPoint:true,
 *        isQueryLine:true,
 *        isQueryRegion:true,
 *        attributeFilter:"SmID<10",
 *        chartFeatureInfoSpecCode:1
 *    });
 *
 * var chartQueryParameters = new ChartQueryParameters({
 *        queryMode:"ChartAttributeQuery",
 *        chartLayerNames:nameArray,
 *        returnContent:true,
 *        chartQueryFilterParameters:[chartQueryFilterParameter]
 *    });
 *
 * var chartQueryService = new ChartQueryService(url);
 *
 * chartQueryService.events.on({
 *        "processCompleted":processCompleted,
 *        "processFailed":processFailed
 *    });
 * chartQueryService.processAsync(chartQueryParameters);
 * @usage
 */
class ChartQueryService extends CommonServiceBase {


    constructor(url, options) {
        super(url, options);
        options = options || {};

        /**
         * @member {boolean} ChartQueryService.prototype.returnContent
         * @description 是否立即返回新创建资源的表述还是返回新资源的URI。
         */
        this.returnContent = null;

        /**
         * @member {DataFormat} ChartQueryService.prototype.format
         * @description 查询结果返回格式，目前支持iServerJSON 和GeoJSON两种格式
         *              参数格式为"ISERVER","GEOJSON",GEOJSON
         */
        this.format = DataFormat.GEOJSON;

        Util_Util.extend(this, options);
        var me = this;
        if (options.format) {
            me.format = options.format.toUpperCase();
        }

        if (!me.url) {
            return;
        }
        me.url = Util_Util.urlPathAppend(me.url, 'queryResults');

        this.CLASS_NAME = "SuperMap.ChartQueryService";
    }


    /**
     * @function ChartQueryService.prototype.destroy
     * @override
     */
    destroy() {
        var me = this;
        CommonServiceBase.prototype.destroy.apply(this, arguments);
        me.returnContent = null;
        me.format = null;
    }


    /**
     * @function ChartQueryService.prototype.processAsync
     * @description 使用服务地址 URL 实例化 ChartQueryService 对象。
     * @param {ChartQueryParameters} params - 查询参数。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        //todo重点需要添加代码的地方
        if (!(params instanceof ChartQueryParameters)) {
            return;
        }
        var me = this, jsonParameters;
        me.returnContent = params.returnContent;
        jsonParameters = params.getVariablesJson();
        if (me.returnContent) {
            me.url = Util_Util.urlAppend(me.url, 'returnContent=true');
        }
        return me.request({
            method: "POST",
            data: jsonParameters,
            scope: me,
            success: callback,
            failure: callback
        });
    }


    /**
     * @function ChartQueryService.prototype.transformResult
     * @description 状态完成时转换结果。
     * @param {Object} result - 服务器返回的结果对象。
     * @param {Object} options - 请求参数。
     * @return {Object} 转换结果。
     */
    transformResult(result, options) {
        var me = this;
        result = Util_Util.transformResult(result);
        if (result && result.recordsets && me.format === DataFormat.GEOJSON) {
            for (var i = 0, recordsets = result.recordsets, len = recordsets.length; i < len; i++) {
                if (recordsets[i].features) {
                    var geoJSONFormat = new GeoJSON();
                    recordsets[i].features = geoJSONFormat.toGeoJSON(recordsets[i].features);
                }
            }

        }
        return { result, options };
    }

    /**
     * @function ChartQueryService.prototype.getQueryParameters
     * @description 将 JSON 对象表示的查询参数转化为 QueryParameters 对象。
     * @param {Object} params - JSON 字符串表示的查询参数。
     * @returns {QueryParameters} 返回查询结果
     */
    getQueryParameters(params) {
        return new QueryParameters({
            queryMode: params.queryMode,
            bounds: params.bounds,
            chartLayerNames: params.chartLayerNames,
            chartQueryFilterParameters: params.chartQueryFilterParameters,
            returnContent: params.returnContent
        });
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/ClipParameter.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class ClipParameter
 * @deprecatedclass SuperMap.ClipParameter
 * @category iServer SpatialAnalyst InterpolationAnalyst
 * @classdesc 用于裁剪的参数。优先使用用户指定的裁剪区域多边形进行裁剪，也可以通过指定数据源和数据集名，从而使用指定数据集的边界多边形进行裁剪。
 * @param {Object} options - 可选参数。
 * @param {string} [options.clipDatasetName] - 裁剪的数据集名。
 * @param {string} [options.clipDatasourceName] - 裁剪的数据集所在数据源的名字。
 * @param {GeometryPolygon|L.Polygon|L.GeoJSON|ol.geom.Polygon|ol.format.GeoJSON|GeoJSONObject} [options.clipRegion] - 用户指定的裁剪区域。
 * @param {boolean} [options.isClipInRegion=true] - 是否对裁剪区内的数据集进行裁剪。
 * @param {boolean} [options.isExactClip=true] - 是否使用精确裁剪。
 * @usage
 */
class ClipParameter {


    constructor(options) {
        /**
         * @member {string} ClipParameter.prototype.clipDatasetName
         * @description 用于裁剪的数据集名，clipDatasetName 与 clipRegion 必须设置一个。
         */
        this.clipDatasetName = null;

        /**
         * @member {string} ClipParameter.prototype.clipDatasourceName
         * @description 用于裁剪的数据集所在数据源的名字。当 clipRegion 不设置时起作用。
         */
        this.clipDatasourceName = null;

        /**
         * @member {GeometryPolygon|L.Polygon|L.GeoJSON|ol.geom.Polygon|ol.format.GeoJSON|GeoJSONObject} ClipParameter.prototype.clipRegion
         * @description 用户指定的裁剪区域，优先使用，clipDatasetName 与 clipRegion 必须设置一个。
         */
        this.clipRegion = null;

        /**
         * @member {boolean} [ClipParameter.prototype.isClipInRegion=true]
         * @description 是否对裁剪区内的数据集进行裁剪。若为 true，则对裁剪区域内的结果进行裁剪，若为 false，则对裁剪区域外的结果进行裁剪。
         */
        this.isClipInRegion = true;

        /**
         * @member {boolean} [ClipParameter.prototype.isExactClip=true]
         * @description 是否使用精确裁剪。
         */
        this.isExactClip = null;

        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.ClipParameter";
    }


    /**
     * @function ClipParameter.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.clipDatasetName = null;
        me.clipDatasourceName = null;
        me.clipRegion = null;
        me.isClipInRegion = null;
        me.isExactClip = null;
    }


    /**
     * @function ClipParameter.prototype.toJSON
     * @description 将 ClipParameter 对象转化为 JSON 字符串。
     * @returns {string} 返回转换后的 JSON 字符串。
     */
    toJSON() {
        return Util_Util.toJSON({
            isClipInRegion: this.isClipInRegion,
            clipDatasetName: this.clipDatasetName,
            clipDatasourceName: this.clipDatasourceName,
            isExactClip: this.isExactClip,
            clipRegion: ServerGeometry.fromGeometry(this.clipRegion)
        });
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/ColorDictionary.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class ColorDictionary
 * @deprecatedclass SuperMap.ColorDictionary
 * @category  iServer Map Theme
 * @classdesc 颜色对照表类。颜色对照表中的键名为具体的高程值，键值表示该高程值要显示的颜色。
 * 对于栅格图层中高程值小于颜色对照表中高程最小值的点使用颜色对照表中高程最小值对应的颜色，
 * 对于栅格图层中高程值大于颜色对照表中高程最大值的点使用颜色对照表中高程最大值对应的颜色，
 * 对于栅格图层中高程值在颜色对照表中没有对应颜色的点，则查找颜色对照表中与当前高程值相邻的两个高程对应的颜色，
 * 然后通过渐变运算要显示的颜色。如果设置了颜色对照表的话，则颜色表设置无效。
 * @param {Object} options - 参数。
 * @param {number} options.elevation - 高程值。
 * @param {ServerColor} options.color - 服务端颜色类。
 * @usage
 */
class ColorDictionary {


    constructor(options) {
        options = options || {};

        /**
         * @member {number} ColorDictionary.prototype.elevation
         * @description 高程值。
         */
        this.elevation = null;

        /**
         * @member {ServerColor} ColorDictionary.prototype.color
         * @description 服务端颜色类。
         */
        this.color = null;

        Util_Util.extend(this, options);

        var me = this,
            c = me.color;
        if (c) {
            me.color = new ServerColor(c.red, c.green, c.blue);
        }

        this.CLASS_NAME = "SuperMap.ColorDictionary";
    }

    /**
     * @function ColorDictionary.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        Util_Util.reset(this);
    }

    /**
     * @function ColorDictionary.prototype.toServerJSONObject
     * @description 转换成对应的 JSON 格式对象。
     * @returns {Object} JSON 对象。
     */
    toServerJSONObject() {
        var dataObj = {};
        dataObj = Util_Util.copyAttributes(dataObj, this);
        return dataObj;
    }
}



;// CONCATENATED MODULE: ./src/common/iServer/TransportationAnalystResultSetting.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class TransportationAnalystResultSetting
 * @deprecatedclass SuperMap.TransportationAnalystResultSetting
 * @category  iServer NetworkAnalyst
 * @classdesc 交通网络分析结果参数类。通过该类设置交通网络分析返回的结果，包括是否返回图片、是否返回弧段空间信息、是否返回结点空间信息等。
 * @param {Object} options - 可选参数。
 * @param {boolean} [options.returnEdgeFeatures=false] - 是否在分析结果中包含弧段要素集合。
 * @param {boolean} [options.returnEdgeGeometry=false] - 返回的弧段要素集合中是否包含几何对象信息。
 * @param {boolean} [options.returnEdgeIDs=false] - 返回结果中是否包含经过弧段 ID 集合。
 * @param {boolean} [options.returnNodeFeatures=false] - 是否在分析结果中包含结点要素集合。
 * @param {boolean} [options.returnNodeGeometry=false] - 返回的结点要素集合中是否包含几何对象信息。
 * @param {boolean} [options.returnNodeIDs=false] - 返回结果中是否包含经过结点 ID 集合。
 * @param {boolean} [options.returnPathGuides=false] - 返回分析结果中是否包含行驶导引集合。
 * @param {boolean} [options.returnRoutes=false] - 返回分析结果中是否包含路由对象的集合。
 * @usage
 */
class TransportationAnalystResultSetting {

    constructor(options) {
        if (!options) {
            return;
        }

        /**
         * @member {boolean} TransportationAnalystResultSetting.prototype.returnEdgeFeatures
         * @description 是否在分析结果中包含弧段要素集合。弧段要素包括弧段的空间信息和属性信息。
         */
        this.returnEdgeFeatures = false;

        /**
         * @member {boolean} [TransportationAnalystResultSetting.prototype.returnEdgeGeometry=false]
         * @description 返回的弧段要素集合中是否包含几何对象信息。
         */
        this.returnEdgeGeometry = false;

        /**
         * @member {boolean} [TransportationAnalystResultSetting.prototype.returnEdgeIDs=false]
         * @description 返回结果中是否包含经过弧段 ID 集合。
         */
        this.returnEdgeIDs = false;

        /**
         * @member {boolean} [TransportationAnalystResultSetting.prototype.returnNodeFeatures=false]
         * @description 是否在分析结果中包含结点要素集合。
         * 结点要素包括结点的空间信息和属性信息。其中返回的结点要素是否包含空间信息可通过 returnNodeGeometry 字段设置。
         */
        this.returnNodeFeatures = false;

        /**
         * @member {boolean} [TransportationAnalystResultSetting.prototype.returnNodeGeometry=false]
         * @description 返回的结点要素集合中是否包含几何对象信息。
         */
        this.returnNodeGeometry = false;

        /**
         * @member {boolean} [TransportationAnalystResultSetting.prototype.returnNodeIDs=false]
         * @description 返回结果中是否包含经过结点 ID 集合。
         */
        this.returnNodeIDs = false;

        /**
         * @member {boolean} TransportationAnalystResultSetting.prototype.returnPathGuides
         * @description 返回分析结果中是否包含行驶导引集合。
         */
        this.returnPathGuides = false;

        /**
         * @member {boolean} TransportationAnalystResultSetting.prototype.returnRoutes
         * @description 返回分析结果中是否包含路由对象的集合。
         */
        this.returnRoutes = false;

        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.TransportationAnalystResultSetting";
    }

    /**
     * @function TransportationAnalystResultSetting.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.returnEdgeFeatures = null;
        me.returnEdgeGeometry = null;
        me.returnEdgeIDs = null;
        me.returnNodeFeatures = null;
        me.returnNodeGeometry = null;
        me.returnNodeIDs = null;
        me.returnPathGuides = null;
        me.returnRoutes = null;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/TransportationAnalystParameter.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class TransportationAnalystParameter
 * @deprecatedclass SuperMap.TransportationAnalystParameter
 * @category  iServer NetworkAnalyst
 * @classdesc 交通网络分析通用参数类。该类主要用来提供交通网络分析所需的通用参数。
 * 通过本类可以设置障碍边、障碍点、权值字段信息的名称标识、转向权值字段等信息，还可以对分析结果包含的内容进行一些设置。
 * @param {Object} options - 参数。
 * @param {Array.<number>} options.barrierEdgeIDs - 网络分析中障碍弧段的 ID 数组。
 * @param {Array.<number>} options.barrierNodeIDs - 网络分析中障碍点的 ID 数组。
 * @param {string} options.turnWeightField - 转向权重字段的名称。
 * @param {TransportationAnalystResultSetting} options.resultSetting - 分析结果返回内容。
 * @param {Array.<GeometryPoint|L.Point|L.LatLng|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>} [options.barrierPoints] - 网络分析中 Point2D 类型的障碍点数组。
 * @param {string} [options.weightFieldName] - 阻力字段的名称。
 * @usage
 */
class TransportationAnalystParameter {

    constructor(options) {

        if (!options) {
            return;
        }
        /**
         * @member {Array.<number>} TransportationAnalystParameter.prototype.barrierEdgeIDs
         * @description 网络分析中障碍弧段的 ID 数组。弧段设置为障碍边之后，表示双向都不通。
         */
        this.barrierEdgeIDs = null;

        /**
         * @member {Array.<number>} TransportationAnalystParameter.prototype.barrierNodeIDs
         * @description 网络分析中障碍点的 ID 数组。结点设置为障碍点之后，表示任何方向都不能通过此结点。
         */
        this.barrierNodeIDs = null;

        /**
         * @member {Array.<GeometryPoint|L.Point|L.LatLng|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>}  TransportationAnalystParameter.prototype.barrierPoints
         * @description 网络分析中 Point2D 类型的障碍点数组。障碍点表示任何方向都不能通过此点。</br>
         * 当各网络分析参数类中的 isAnalyzeById 属性设置为 false 时，该属性才生效。
         */
        this.barrierPoints = null;

        /**
         * @member {string} [TransportationAnalystParameter.prototype.weightFieldName]
         * @description 阻力字段的名称，标识了进行网络分析时所使用的阻力字段，例如表示时间、长度等的字段都可以用作阻力字段。
         * 该字段默值为服务器发布的所有耗费字段的第一个字段。
         */
        this.weightFieldName = null;

        /**
         * @member {string} TransportationAnalystParameter.prototype.turnWeightField
         * @description 转向权重字段的名称。
         */
        this.turnWeightField = null;

        /**
         *  @member {TransportationAnalystResultSetting} TransportationAnalystParameter.prototype.resultSetting
         *  @description 分析结果返回内容。
         */
        this.resultSetting = new TransportationAnalystResultSetting();

        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.TransportationAnalystParameter";
    }

    /**
     * @function TransportationAnalystParameter.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.barrierEdgeIDs = null;
        me.barrierNodeIDs = null;
        me.weightFieldName = null;
        me.turnWeightField = null;
        if (me.resultSetting) {
            me.resultSetting.destroy();
            me.resultSetting = null;
        }
        if (me.barrierPoints && me.barrierPoints.length) {
            for (var i in me.barrierPoints) {
                me.barrierPoints[i].destroy();
            }
        }
        me.barrierPoints = null;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/ComputeWeightMatrixParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class ComputeWeightMatrixParameters
 * @deprecatedclass SuperMap.ComputeWeightMatrixParameters
 * @category iServer NetworkAnalyst WeightMatrix
 * @classdesc 耗费矩阵分析参数类。根据交通网络分析参数中的耗费字段返回一个耗费矩阵。该矩阵是一个二维数组，用来存储任意两点间的资源消耗。
 * @param {Object} options - 参数。
 * @param {boolean} [options.isAnalyzeById=false] - 是否通过节点 ID 指定路径分析的结点。
 * @param {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|Array.<number>>} options.nodes - 要计算耗费矩阵的点数组。
 * @param {TransportationAnalystParameter} [options.parameter] - 交通网络分析通用参数。
 * @usage
 */
class ComputeWeightMatrixParameters {


    constructor(options) {
        /**
         * @member {boolean} [ComputeWeightMatrixParameters.prototype.isAnalyzeById=false]
         * @description 是否通过节点 ID 指定路径分析的结点，即通过坐标点指定。
         */
        this.isAnalyzeById = false;

        /**
         * @member {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|Array.<number>>} ComputeWeightMatrixParameters.prototype.nodes
         * @description 要计算耗费矩阵的点数组。
         *              当 {@link ComputeWeightMatrixParameters.isAnalyzeById} = false 时，nodes 应为点的坐标数组；
         *              当 {@link ComputeWeightMatrixParameters.isAnalyzeById} = true 时，nodes 应为点的 ID 数组。
         */
        this.nodes = null;

        /**
         * @member {TransportationAnalystParameter} ComputeWeightMatrixParameters.prototype.parameter
         * @description 交通网络分析通用参数。
         */
        this.parameter = new TransportationAnalystParameter();

        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.ComputeWeightMatrixParameters";
    }

    /**
     * @function ComputeWeightMatrixParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.isAnalyzeById = null;
        me.nodes = null;
        if (me.parameter) {
            me.parameter.destroy();
            me.parameter = null;
        }
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/ComputeWeightMatrixService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class ComputeWeightMatrixService
 * @deprecatedclass SuperMap.ComputeWeightMatrixService
 * @category  iServer NetworkAnalyst WeightMatrix
 * @classdesc 耗费矩阵分析服务类。
 *            耗费矩阵是根据交通网络分析参数中的耗费字段来计算一个二维数组，
 *            用来存储指定的任意两点间的资源消耗。
 *            耗费矩阵分析结果通过该类支持的事件的监听函数参数获取
 * @extends {NetworkAnalystServiceBase}
 * @example
 * var mycomputeWeightMatrixService = new ComputeWeightMatrixService(url);
 * @param {string} url - 耗费矩阵分析服务地址。请求服务的URL应为：
 *                       http://{服务器地址}:{服务端口号}/iserver/services/{网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}；
 *                       例如："http://localhost:8090/iserver/services/components-rest/rest/networkanalyst/RoadNet@Changchun"。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class ComputeWeightMatrixService extends NetworkAnalystServiceBase {

    constructor(url, options) {
        super(url, options);
        this.CLASS_NAME = "SuperMap.ComputeWeightMatrixService";
    }


    /**
     * @function ComputeWeightMatrixService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }


    /**
     * @function ComputeWeightMatrixService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {ComputeWeightMatrixParameters} params - 耗费矩阵分析参数类
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof ComputeWeightMatrixParameters)) {
            return;
        }
        var me = this,
            jsonObject;
        me.url = Util_Util.urlPathAppend(me.url, 'weightmatrix');
        jsonObject = {
            parameter: Util_Util.toJSON(params.parameter),
            nodes: me.getJson(params.isAnalyzeById, params.nodes)
        };
        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }

    /**
     * @function ComputeWeightMatrixService.prototype.getJson
     * @description 将对象转化为JSON字符串。
     * @param {boolean} isAnalyzeById - 是否通过id分析
     * @param {Array.<ComputeWeightMatrixParameters>} params - 分析参数数组
     * @returns {string} 转化后的JSON字符串。
     */
    getJson(isAnalyzeById, params) {
        var jsonString = "[",
            len = params ? params.length : 0;

        if (isAnalyzeById === false) {
            for (let i = 0; i < len; i++) {
                if (i > 0) {
                    jsonString += ",";
                }
                jsonString += '{"x":' + params[i].x + ',"y":' + params[i].y + '}';
            }
        } else if (isAnalyzeById === true) {
            for (let i = 0; i < len; i++) {
                if (i > 0) {
                    jsonString += ",";
                }
                jsonString += params[i];
            }
        }
        jsonString += ']';
        return jsonString;
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/DataFlowService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class DataFlowService
 * @deprecatedclass SuperMap.DataFlowService
 * @category iServer DataFlow
 * @classdesc 数据流服务类。
 * @extends {CommonServiceBase}
 * @param {string} url - 数据流服务地址。
 * @param {Object} options - 参数。
 * @param {function} options.style - 设置数据加载样式。
 * @param {function} [options.onEachFeature] - 设置每个数据加载popup等。
 * @param {GeoJSONObject} [options.geometry] - 指定几何范围，该范围内的要素才能被订阅。
 * @param {Object} [options.excludeField] - 排除字段。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class DataFlowService extends CommonServiceBase {


    constructor(url, options) {
        options = options || {};
        /*
         * @constant EVENT_TYPES
         * {Array.<string>}
         * 此类支持的事件类型
         */
        options.EVENT_TYPES = ["broadcastSocketConnected", "broadcastSocketClosed", "broadcastSocketError", "broadcastFailed", "broadcastSucceeded", "subscribeSocketConnected", "subscribeSocketClosed", "subscribeSocketError", "messageSucceeded", "setFilterParamSucceeded"]
        super(url, options);

        /**
         * @member {GeoJSONObject} DataFlowService.prototype.geometry
         * @description 指定几何范围，该范围内的要素才能被订阅。
         */
        this.geometry = null;

        /**
         * @member {Object} DataFlowService.prototype.prjCoordSys
         * @description 动态投影参数。
         */
        this.prjCoordSys = null;

        /**
         * @member {Object} DataFlowService.prototype.excludeField
         * @description 排除字段。
         */
        this.excludeField = null;

        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.DataFlowService";
    }

    /**
     * @function DataFlowService.prototype.initBroadcast
     * @description 初始化广播。
     * @returns {DataFlowService}
     */
    initBroadcast() {
        var me = this;
        this.broadcastWebSocket = this._connect(Util_Util.urlPathAppend(me.url, 'broadcast'));
        this.broadcastWebSocket.onopen = function (e) {
            me.broadcastWebSocket.isOpen = true;
            e.eventType = 'broadcastSocketConnected';
            me.events.triggerEvent('broadcastSocketConnected', e);
        };
        this.broadcastWebSocket.onclose = function (e) {
            if (me.broadcastWebSocket) {
                me.broadcastWebSocket.isOpen = false;
            }
            e.eventType = 'broadcastSocketClosed';
            me.events.triggerEvent('broadcastSocketClosed', e);
        };
        this.broadcastWebSocket.onerror = function (e) {
            e.eventType = 'broadcastSocketError';
            me.events.triggerEvent('broadcastSocketError', e);
        };
        return this;
    }

    /**
     * @function DataFlowService.prototype.broadcast
     * @description 加载广播数据。
     * @param {GeoJSONObject} geoJSONFeature - JSON 格式的要素数据。
     */
    broadcast(geoJSONFeature) {
        if (!this.broadcastWebSocket||!this.broadcastWebSocket.isOpen) {
            this.events.triggerEvent('broadcastFailed');
            return;
        }
        this.broadcastWebSocket.send(JSON.stringify(geoJSONFeature));
        this.events.triggerEvent('broadcastSucceeded');
    }

    /**
     * @function DataFlowService.prototype.initSubscribe
     * @description 初始化订阅数据。
     * @returns {DataFlowService} DataFlowService的实例对象。
     */
    initSubscribe() {
        var me = this;
        this.subscribeWebSocket = this._connect(Util_Util.urlPathAppend(me.url, 'subscribe'));
        this.subscribeWebSocket.onopen = function (e) {
            me.subscribeWebSocket.send(me._getFilterParams());
            e.eventType = 'subscribeSocketConnected';
            me.events.triggerEvent('subscribeSocketConnected', e);
        };
        this.subscribeWebSocket.onclose = function (e) {
            e.eventType = 'subscribeWebSocketClosed';
            me.events.triggerEvent('subscribeWebSocketClosed', e);
        };
        this.subscribeWebSocket.onerror = function (e) {
            e.eventType = 'subscribeSocketError';
            me.events.triggerEvent('subscribeSocketError', e);
        };
        this.subscribeWebSocket.onmessage = function (e) {
            me._onMessage(e);
        };
        return this;
    }


    /**
     * @function DataFlowService.prototype.setExcludeField
     * @description 设置排除字段。
     * @param {Object} excludeField - 排除字段。
     * @returns {DataFlowService} DataFlowService的实例对象。
     */
    setExcludeField(excludeField) {
        this.excludeField = excludeField;
        this.subscribeWebSocket.send(this._getFilterParams());
        return this;
    }

    /**
     * @function DataFlowService.prototype.setGeometry
     * @description 设置添加的几何要素数据。
     * @param {GeoJSONObject} geometry - 指定几何范围，该范围内的要素才能被订阅。
     * @returns {DataFlowService} DataFlowService的实例对象。
     */
    setGeometry(geometry) {
        this.geometry = geometry;
        this.subscribeWebSocket.send(this._getFilterParams());
        return this;
    }

    /**
     * @function DataFlowService.prototype.unSubscribe
     * @description 结束订阅数据。
     */
    unSubscribe() {
        if (!this.subscribeWebSocket) {
            return;
        }
        this.subscribeWebSocket.close();
        this.subscribeWebSocket = null;
    }

    /**
     * @function DataFlowService.prototype.unBroadcast
     * @description 结束加载广播。
     */
    unBroadcast() {
        if (!this.broadcastWebSocket) {
            return;
        }
        this.broadcastWebSocket.close();
        this.broadcastWebSocket = null;
    }

    /**
     * @function DataFlowService.prototype.destroy
     * @override
     */
    destroy() {
        CommonServiceBase.prototype.destroy.apply(this, arguments);
        var me = this;
        me.geometry = null;
        me.prjCoordSys = null;
        me.excludeField = null;
        this.unBroadcast();
        this.unSubscribe();

    }


    _getFilterParams() {
        var filter = {
            filterParam: {
                prjCoordSys: this.prjCoordSys,
                excludeField: this.excludeField,
                geometry: this.geometry
            }
        };
        return Util_Util.toJSON(filter);
    }


    _onMessage(e) {
        if (e.data && e.data.indexOf("filterParam") >= 0) {
            var filterParam = JSON.parse(e.data);
            e.filterParam = filterParam;
            e.eventType = 'setFilterParamSucceeded';
            this.events.triggerEvent('setFilterParamSucceeded', e);
            return;
        }
        var feature = JSON.parse(e.data);
        e.featureResult = feature;
        e.eventType = 'messageSucceeded';
        this.events.triggerEvent('messageSucceeded', e);
    }


    _connect(url) {
        url = SecurityManager.appendCredential(url);
        if ("WebSocket" in window) {
            return new WebSocket(url);
        } else if ("MozWebSocket" in window) {
            var mozWebSocket = window.MozWebSocket;
            return new mozWebSocket(url);
        } else {
            console.log("no WebSocket");
            return null;
        }
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/DatasetInfo.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class DatasetInfo
 * @deprecatedclass SuperMap.DatasetInfo
 * @category  iServer Data Dataset
 * @classdesc 数据集信息类。
 * 数据集一般为存储在一起的相关数据的集合；根据数据类型的不同，分为矢量数据集、栅格数据集（griddataset）和
 * 影像数据集（image dataset），以及为了处理特定问题而设计的数据集，如拓扑数据集，网络数据集等。
 * 数据集是 GIS 数据组织的最小单位。其中矢量数据集是由同种类型空间要素组成的集合，
 * 所以也可以称为要素集。根据要素的空间特征的不同，矢量数据集又分为点数据集，
 * 线数据集，面数据集等，各矢量数据集是空间特征和性质相同的数据组织起来的集合。
 * 目前版本支持的数据集主要有点数据集，线数据集，面数据集，文本数据集，复合数据集（CAD 数据集）、
 * 网络数据集，栅格数据集（grid dataset）和影像数据集（image dataset）。
 * @param {Object} options - 参数。
 * @param {Bounds} [options.bounds] - 数据集范围。
 * @param {string} [options.dataSourceName] - 数据源名称。
 * @param {string} [options.description] - 数据集的描述信息。
 * @param {string} [options.encodeType] - 数据集存储时的压缩编码方式。
 * @param {boolean} [options.isReadOnly] - 数据集是否为只读。
 * @param {string} options.name - 数据集名称。
 * @param {Object} [options.prjCoordSys] - 数据集的投影信息。如：prjCoordSys={"epsgCode":3857}。
 * @param {string} [options.tableName] - 表名。
 * @param {string} options.type - 数据集类型。主要有点数据集，线数据集，面数据集，文本数据集，复合数据集（CAD 数据集）、网络数据集，栅格数据集（grid dataset）和影像数据集（image dataset）。
 * @usage
 */
class DatasetInfo {


    constructor(options) {
        options = options || {};

        /**
         * @member {Bounds} [DatasetInfo.prototype.bounds]
         * @description 数据集范围，该字段只读。
         */
        this.bounds = null;

        /**
         * @member {string} [DatasetInfo.prototype.dataSourceName]
         * @description 数据源名称，该字段只读。
         */
        this.dataSourceName = null;

        /**
         * @member {string} [DatasetInfo.prototype.description]
         * @description 数据集的描述信息。
         */
        this.description = null;

        /**
         * @member {string} [DatasetInfo.prototype.encodeType]
         * @description 数据集存储时的压缩编码方式，该字段只读。
         */
        this.encodeType = null;

        /**
         * @member {boolean} [DatasetInfo.prototype.isReadOnly]
         * @description 数据集是否为只读。
         */
        this.isReadOnly = null;

        /**
         * @member {string} DatasetInfo.prototype.name
         * @description 数据集名称，该字段必须且只读。
         */
        this.name = null;

        /**
         * @member {Object} [DatasetInfo.prototype.prjCoordSys]
         * @description 数据集的投影信息。
         */
        this.prjCoordSys = null;

        /**
         * @member {string} [DatasetInfo.prototype.tableName]
         * @description 表名，该字段只读。
         */
        this.tableName = null;

        /**
         * @member {string} DatasetInfo.prototype.type
         * @description 数据集类型，该字段必设。主要有点数据集，线数据集，面数据集，文本数据集，复合数据集（CAD 数据集）、网络数据集，栅格数据集（grid dataset）和影像数据集（image dataset）。
         */
        this.type = null;

        Util_Util.extend(this, options);

        var b = this.bounds;
        if (b) {
            this.bounds = new Bounds(b.leftBottom.x, b.leftBottom.y, b.rightTop.x, b.rightTop.y);
        }
        this.CLASS_NAME = "SuperMap.DatasetInfo";
    }

    /**
     * @function DatasetInfo.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        Util_Util.reset(this);
    }

    /**
     * @function DatasetInfo.prototype.toServerJSONObject
     * @description 转换成对应的 JSON 格式对象。
     * @returns {Object} JSON 对象。
     */
    toServerJSONObject() {
        var dataObj = {};
        dataObj = Util_Util.copyAttributes(dataObj, this);
        if (dataObj.bounds) {
            if (dataObj.bounds.toServerJSONObject) {
                dataObj.bounds = dataObj.bounds.toServerJSONObject();
            }
        }
        return dataObj;
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/OverlayAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class OverlayAnalystParameters
 * @deprecatedclass SuperMap.OverlayAnalystParameters
 * @category  iServer SpatialAnalyst OverlayAnalyst
 * @classdesc 叠加分析参数基类。数据集叠加分析参数和几何对象叠加分析参数均继承此基类。
 * @param {Object} options - 参数。
 * @usage
 */
class OverlayAnalystParameters {


    constructor(options) {
        /**
         * @member {OverlayOperationType} [OverlayAnalystParameters.prototype.operation=OverlayOperationType.UNION]
         * @description 指定叠加分析操作类型。
         */
        this.operation = OverlayOperationType.UNION;

        if (options) {
            Util_Util.extend(this, options);
        }
        this.CLASS_NAME = "SuperMap.OverlayAnalystParameters";
    }

    /**
     * @function OverlayAnalystParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.operation = null;
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/DatasetOverlayAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/






/**
 * @class DatasetOverlayAnalystParameters
 * @deprecatedclass SuperMap.DatasetOverlayAnalystParameters
 * @category  iServer SpatialAnalyst OverlayAnalyst
 * @classdesc 数据集叠加分析参数类。
 * @param {Object} options - 参数。
 * @param {string} options.operateDataset -数据集名称。
 * @param {string} options.sourceDataset - 源数据集名称。
 * @param {Array.<string>} [options.operateDatasetFields] - 叠加分析中操作数据集保留在结果数据集中的字段名列表。
 * @param {FilterParameter} [options.operateDatasetFilter] - 设置操作数据集中空间对象过滤条件。
 * @param {Array.<GeometryPolygon|L.Polygon|ol.geom.Polygon|GeoJSONObject>} [options.operateRegions] - 操作面对象集合，表示与这些面对象进行叠加分析。与 operateDataset 参数互斥，冲突时以 operateDataset 为准。
 * @param {Array.<string>} [options.sourceDatasetFields] - 叠加分析中源数据集保留在结果数据集中的字段名列表。
 * @param {FilterParameter} [options.sourceDatasetFilter] - 设置源数据集中空间对象过滤条件。
 * @param {number} [options.tolerance=0] - 容限。
 * @param {OverlayOperationType} options.operation - 叠加操作枚举值。
 * @param {DataReturnOption} [options.resultSetting] - 结果返回设置类。
 * @extends {GetFeaturesParametersBase}
 * @usage
 */
class DatasetOverlayAnalystParameters extends OverlayAnalystParameters {


    constructor(options) {
        super(options);
        /**
         * @member {string} DatasetOverlayAnalystParameters.prototype.operateDataset
         * @description 叠加分析中操作数据集的名称。
         */
        this.operateDataset = null;

        /**
         * @member {Array.<string>} [DatasetOverlayAnalystParameters.prototype.operateDatasetFields]
         * @description 叠加分析中操作数据集保留在结果数据集中的字段名列表。
         */
        this.operateDatasetFields = [];

        /**
         * @member {FilterParameter} DatasetOverlayAnalystParameters.prototype.operateDatasetFilter
         * @description 设置操作数据集中空间对象过滤条件。
         */
        this.operateDatasetFilter = new FilterParameter();

        /**
         * @member {Array.<GeometryPolygon|L.Polygon|ol.geom.Polygon|GeoJSONObject>} [DatasetOverlayAnalystParameters.prototype.operateRegions]
         * @description 操作面对象集合，表示与这些面对象进行叠加分析。与 operateDataset 参数互斥，冲突时以 operateDataset 为准。
         */
        this.operateRegions = [];


        /**
         * @member {string} DatasetOverlayAnalystParameters.prototype.sourceDataset
         * @description 叠加分析中源数据集的名称。
         */
        this.sourceDataset = null;


        /**
         * @member {Array.<string>} [DatasetOverlayAnalystParameters.prototype.sourceDatasetFields]
         * @description 叠加分析中源数据集保留在结果数据集中的字段名列表。
         */
        this.sourceDatasetFields = [];


        /**
         * @member {FilterParameter} [DatasetOverlayAnalystParameters.prototype.filterQueryParameter]
         * @description 设置源数据集中空间对象过滤条件。
         */
        this.sourceDatasetFilter = new FilterParameter();

        /**
         * @member {number} [DatasetOverlayAnalystParameters.prototype.tolerance=0]
         * @description 容限。
         */
        this.tolerance = 0;

        /**
         * @member {DataReturnOption} [DatasetOverlayAnalystParameters.prototype.resultSetting]
         * @description 结果返回设置类。
         */
        this.resultSetting = new DataReturnOption();

        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.DatasetOverlayAnalystParameters";
    }


    /**
     * @function DatasetOverlayAnalystParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();

        var me = this;
        me.operateDataset = null;
        me.operateDatasetFields = null;
        if (me.operateDatasetFilter) {
            me.operateDatasetFilter.destroy();
            me.operateDatasetFilter = null;
        }
        if (me.operateRegions) {
            for (var i = 0, opRegions = me.operateRegions, len = opRegions.length; i < len; i++) {
                opRegions[i].destroy();
            }
            me.operateRegions = null;
        }
        me.sourceDataset = null;
        me.sourceDatasetFields = null;
        if (me.sourceDatasetFilter) {
            me.sourceDatasetFilter.destroy();
            me.sourceDatasetFilter = null;
        }
        me.tolerance = null;
        if (me.resultSetting) {
            me.resultSetting.destroy();
            me.resultSetting = null;
        }
    }

    /**
     * @function DatasetOverlayAnalystParameters.toObject
     * @param {DatasetOverlayAnalystParameters} datasetOverlayAnalystParameters - 数据集叠加分析参数类。
     * @param {DatasetOverlayAnalystParameters} tempObj - 数据集叠加分析参数对象。
     * @description 将数据集叠加分析参数类转换为 JSON 对象。
     * @returns {Object} JSON 对象。
     */
    static toObject(datasetOverlayAnalystParameters, tempObj) {
        for (var name in datasetOverlayAnalystParameters) {
            if (name === "sourceDataset") {
                continue;
            } else if (name === "operateRegions") {
                tempObj.operateRegions = [];
                var ors = datasetOverlayAnalystParameters.operateRegions;
                for (var index in ors) {
                    if (ors.hasOwnProperty(index)) {    //icl542
                        tempObj.operateRegions[index] = ServerGeometry.fromGeometry(ors[index]);
                    }
                }
            } else if (name === "resultSetting") {
                tempObj.dataReturnOption = datasetOverlayAnalystParameters.resultSetting;
            } else {
                tempObj[name] = datasetOverlayAnalystParameters[name];
            }
        }
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/SurfaceAnalystParametersSetting.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class SurfaceAnalystParametersSetting
 * @deprecatedclass SuperMap.SurfaceAnalystParametersSetting
 * @category  iServer SpatialAnalyst SurfaceAnalyst
 * @classdesc 表面分析参数设置类。
 * 通过该类可以设置表面分析提取等值线、提取等值面的一些参数，包括基准值、等值距、光滑度、光滑方法等。
 * @param {Object} options - 参数。
 * @param {GeometryPolygon|L.Polygon|ol.geom.Polygon|GeoJSONObject} [options.clipRegion] - 裁剪面对象，如果不需要对操作结果进行裁剪，可以使用 null 值取代该参数。
 * @param {number} [options.datumValue=0] - 提取等值线、提取等值面的基准值。
 * @param {Array.<number>} options.expectedZValues - 期望分析结果的 Z 值集合。
 * @param {number} [options.interval=0] - 等值距。等值距是两条等值线之间的间隔值。
 * @param {number} [options.resampleTolerance=0] - 重采样容限。
 * @param {SmoothMethod} [options.smoothMethod=SmoothMethod.BSPLINE] - 光滑处理所使用的方法。
 * @param {number} [options.smoothness=0] - 等值线或等值面的边界线的光滑度。
 * @usage
 */
class SurfaceAnalystParametersSetting {

    constructor(options) {
        /**
         * @member {GeometryPolygon|L.Polygon|ol.geom.Polygon|GeoJSONObject} [SurfaceAnalystParametersSetting.prototype.clipRegion]
         * @description 获取或设置裁剪面对象，如果不需要对操作结果进行裁剪，可以使用 null 值取代该参数。
         */
        this.clipRegion = null;

        /**
         * @member {number} [SurfaceAnalystParametersSetting.prototype.datumValue=0]
         * @description 获取或设置表面分析中提取等值线、提取等值面的基准值。
         * 基准值是作为一个生成等值线的初始起算值，并不一定是最小等值线的值。例如，高程范围为 220 -1550 的 DEM 栅格数据，
         * 如果设基准值为 0，等值距为 50，则提取等值线时，以基准值 0 为起点，等值距 50 为间隔提取等值线，
         * 因为给定高程的最小值是 220，所以，在给定范围内提取等值线的最小高程是 250。
         * 提取等值线的结果是：最小等值线值为 250，最大等值线值为 1550。
         */
        this.datumValue = 0;
        /**
         * @member {Array.<number>} SurfaceAnalystParametersSetting.prototype.expectedZValues
         * @description 获取或设置期望分析结果的 Z 值集合。
         *              Z 值集合存储一系列数值，该数值为待提取等值线的值。即仅高程值在 Z 值集合中的等值线会被提取。
         */
        this.expectedZValues = null;

        /**
         *  @member {number} [SurfaceAnalystParametersSetting.prototype.interval=0]
         *  @description 获取或设置等值距。等值距是两条等值线之间的间隔值。
         */
        this.interval = 0;

        /**
         * @member {number} [SurfaceAnalystParametersSetting.prototype.resampleTolerance=0]
         * @description 获取或设置重采样容限。
         *              容限值越大，采样结果数据越简化。当分析结果出现交叉时，可通过调整重采样容限为较小的值来处理。
         */
        this.resampleTolerance = 0;

        /**
         * @member {SmoothMethod} [SurfaceAnalystParametersSetting.prototype.smoothMethod=SmoothMethod.BSPLINE]
         * @description 获取或设置光滑处理所使用的方法。
         */
        this.smoothMethod = SmoothMethod.BSPLINE;

        /**
         * @member {number} [SurfaceAnalystParametersSetting.prototype.smoothness=0]
         * @description 获取或设置表面分析中等值线或等值面的边界线的光滑度。
         * 以为 0-5 为例，光滑度为 0 表示不进行光滑操作，值越大表示光滑度越高。
         * 随着光滑度的增加，提取的等值线越光滑，当然光滑度越大，
         * 计算所需的时间和占用的内存也就越大。而且，当等值距较小时，
         * 光滑度太高会出现等值线相交的问题。
         */
        this.smoothness = 0;

        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.SurfaceAnalystParametersSetting";
    }


    /**
     * @function SurfaceAnalystParametersSetting.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        if (me.clipRegion) {
            me.clipRegion.destroy();
            me.clipRegion = null;
        }

        me.datumValue = null;
        me.expectedZValues = null;
        me.interval = null;
        me.resampleTolerance = null;
        me.smoothMethod = null;
        me.smoothness = null;
    }
    /**
     * @function SurfaceAnalystParametersSetting.prototype.toJSON
     * @description 将对象转化为 JSON 字符串。
     * @returns {string} 对象 JSON 字符串。
     */
    toJSON() {
        let json = "'datumValue':" + Util_Util.toJSON(this.datumValue);
        json += ",'interval':" + Util_Util.toJSON(this.interval);
        json += ",'resampleTolerance':" + Util_Util.toJSON(this.resampleTolerance);
        json += ",'smoothMethod':" + Util_Util.toJSON(this.smoothMethod);
        json += ",'smoothness':" + Util_Util.toJSON(this.smoothness);
        if (this.expectedZValues != null) {
            json += "," + "'expectedZValues':" + Util_Util.toJSON(this.expectedZValues);
        }
        if (this.clipRegion != null) {
            var serverGeometry = this.clipRegion;
            if (this.clipRegion instanceof Geometry_Geometry && this.clipRegion.components) {
                serverGeometry = ServerGeometry.fromGeometry(this.clipRegion)
            }
            json += ",'clipRegion':" + Util_Util.toJSON(serverGeometry);
        }
        return "{" + json + "}";
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/SurfaceAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class SurfaceAnalystParameters
 * @deprecatedclass SuperMap.SurfaceAnalystParameters
 * @category  iServer SpatialAnalyst SurfaceAnalyst
 * @classdesc 表面分析提取操作参数类。通过该类可以为进行表面分析提供参数信息，包括表面分析的方法提取等值线、提取等值面和中间结果的分辨率，
 * {@link DatasetSurfaceAnalystParameters} 和 {@link GeometrySurfaceAnalystParameters} 继承自该类。
 * @param {Object} options - 参数。
 * @param {SurfaceAnalystParametersSetting} options.extractParameter - 表面分析参数设置类。
 * @param {number} options.resolution - 指定中间结果（栅格数据集）的分辨率。
 * @param {DataReturnOption} options.resultSetting - 结果返回设置类。
 * @param {SurfaceAnalystMethod} [options.surfaceAnalystMethod=SurfaceAnalystMethod.ISOLINE] - 获取或设置表面分析的提取方法，提取等值线和提取等值面。
 * @usage
 */
class SurfaceAnalystParameters {

    constructor(options) {
        /**
         * @member {number} SurfaceAnalystParameters.prototype.resolution
         * @description 获取或设置指定中间结果（栅格数据集）的分辨率。
         */
        this.resolution = 0;

        /**
         * @member {SurfaceAnalystParametersSetting} SurfaceAnalystParameters.prototype.extractParameter
         * @description 获取或设置表面分析参数。
         * 在进行点数据集进行提取等值面分析时，暂时不支持 SurfaceAnalystParametersSetting 类中的 expectedZValues 字段。
         */
        this.extractParameter = new SurfaceAnalystParametersSetting();

        /**
         * @member {DataReturnOption} SurfaceAnalystParameters.prototype.resultSetting
         * @description 结果返回设置类。
         */
        this.resultSetting = new DataReturnOption();

        /**
         * @member {SurfaceAnalystMethod} [SurfaceAnalystParameters.prototype.surfaceAnalystMethod=SurfaceAnalystMethod.ISOLINE]
         * @description 获取或设置表面分析的提取方法，提取等值线和提取等值面。
         */
        this.surfaceAnalystMethod = SurfaceAnalystMethod.ISOLINE;

        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.SurfaceAnalystParameters";
    }

    /**
     * @function SurfaceAnalystParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.resolution = null;
        if (me.extractParameter) {
            me.extractParameter.destroy();
            me.extractParameter = null;
        }
        if (me.resultSetting) {
            me.resultSetting.destroy();
            me.resultSetting = null;
        }
        me.surfaceAnalystMethod = null;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/DatasetSurfaceAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/






/**
 * @class DatasetSurfaceAnalystParameters
 * @deprecatedclass SuperMap.DatasetSurfaceAnalystParameters
 * @category iServer SpatialAnalyst SurfaceAnalyst
 * @classdesc 数据集表面分析参数类。该类对数据集表面分析所用到的参数进行设置。
 * @param {Object} options - 参数。
 * @param {string} options.dataset - 数据集名称。
 * @param {string} options.zValueFieldName - 字段名称。
 * @param {number} options.resolution - 指定中间结果（栅格数据集）的分辨率。
 * @param {SurfaceAnalystParametersSetting} options.extractParameter - 表面分析参数设置类。获取或设置表面分析参数。
 * @param {FilterParameter} [options.filterQueryParameter] - 查询过滤条件参数。
 * @param {DataReturnOption} [options.resultSetting] - 结果返回设置类。
 * @param {SurfaceAnalystMethod} [options.surfaceAnalystMethod=SurfaceAnalystMethod.ISOLINE] - 表面分析的提取方法，提取等值线和提取等值面。
 * @extends {SurfaceAnalystParameters}
 * @usage
 */
class DatasetSurfaceAnalystParameters extends SurfaceAnalystParameters {


    constructor(options) {
        super(options);
        /**
         * @member {string} DatasetSurfaceAnalystParameters.prototype.dataset
         * @description 要用来做数据集表面分析的数据源中数据集的名称。该名称用形如 "数据集名称@数据源别名" 形式来表示，例如：Country@World。
         */
        this.dataset = null;

        /**
         *  @member {FilterParameter} DatasetSurfaceAnalystParameters.prototype.filterQueryParameter
         *  @description 获取或设置查询过滤条件参数。
         */
        this.filterQueryParameter = new FilterParameter();

        /**
         * @member {string} DatasetSurfaceAnalystParameters.prototype.zValueFieldName
         * @description 获取或设置用于提取操作的字段名称。提取等值线时，将使用该字段中的值，对点记录集中的点数据进行插值分析，得到栅格数据集（中间结果），接着从栅格数据集提取等值线。
         */
        this.zValueFieldName = null;

        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.DatasetSurfaceAnalystParameters";
    }

    /**
     * @function DatasetSurfaceAnalystParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        me.dataset = null;
        if (me.filterQueryParameter) {
            me.filterQueryParameter.destroy();
            me.filterQueryParameter = null;
        }
        me.zValueFieldName = null;
    }

    /**
     * @function DatasetSurfaceAnalystParameters.toObject
     * @param {DatasetSurfaceAnalystParameters} datasetSurfaceAnalystParameters - 数据集表面分析参数类。
     * @param {DatasetSurfaceAnalystParameters} tempObj - 数据集表面分析参数对象。
     * @description 将数据集表面分析参数对象转换为 JSON 对象。
     * @returns JSON 对象。
     */
    static toObject(datasetSurfaceAnalystParameters, tempObj) {
        for (var name in datasetSurfaceAnalystParameters) {
            if (name === "filterQueryParameter") {
                tempObj.filterQueryParameter = datasetSurfaceAnalystParameters.filterQueryParameter;
            }
            if (name === "extractParameter") {
                if (datasetSurfaceAnalystParameters.extractParameter.clipRegion instanceof Geometry_Geometry && datasetSurfaceAnalystParameters.extractParameter.clipRegion.components) {
                    datasetSurfaceAnalystParameters.extractParameter.clipRegion = ServerGeometry.fromGeometry(datasetSurfaceAnalystParameters.extractParameter.clipRegion);
                }
                tempObj.extractParameter = datasetSurfaceAnalystParameters.extractParameter;
            } else if (name === "dataset") {
                continue;
            } else if (name === "surfaceAnalystMethod") {
                continue;
            } else {
                tempObj[name] = datasetSurfaceAnalystParameters[name];
            }
        }
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/ThiessenAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class ThiessenAnalystParameters
 * @deprecatedclass SuperMap.ThiessenAnalystParameters
 * @category  iServer SpatialAnalyst ThiessenPolygonAnalyst
 * @classdesc 泰森多边形分析参数基类。
 * @param {Object} options - 可选参数。
 * @param {GeometryPolygon|L.Polygon|ol.geom.Polygon|GeoJSONObject} [options.clipRegion] - 结果数据裁剪区域，可以为 null，表示不对结果进行裁剪。
 * @param {boolean} [options.createResultDataset=false] - 是否返回结果数据集。
 * @param {string} [options.resultDatasetName] - 指定结果数据集名称。
 * @param {string} [options.resultDatasourceName] - 指定结果数据集所在数据源，默认为当前数据源。
 * @param {boolean} [options.returnResultRegion=true] - 是否返回分析得到的多边形面数组。
 * @usage
 */
class ThiessenAnalystParameters {

    constructor(options) {
        if (!options) {
            return;
        }
        /**
         * @member {GeometryPolygon|L.Polygon|ol.geom.Polygon|GeoJSONObject} [ThiessenAnalystParameters.prototype.clipRegion]
         * @description 结果数据裁剪区域，可以为 null，表示不对结果进行裁剪。
         */
        this.clipRegion = null;

        /**
         *  @member {boolean} [ThiessenAnalystParameters.prototype.createResultDataset=false]
         *  @description 是否返回结果数据集。如果为 true，则必须设置属性 resultDatasetName 和 resultDatasourceName。
         */
        this.createResultDataset = false;

        /**
         * @member {string} ThiessenAnalystParameters.prototype.resultDatasetName
         * @description 指定结果数据集名称。
         */
        this.resultDatasetName = null;

        /**
         * @member {string} ThiessenAnalystParameters.prototype.resultDatasourceName
         * @description 指定结果数据集所在数据源。
         */
        this.resultDatasourceName = null;

        /**
         * @member {boolean} ThiessenAnalystParameters.prototype.returnResultRegion
         * @description 是否返回分析得到的多边形面数组。
         */
        this.returnResultRegion = true;

        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.ThiessenAnalystParameters";
    }

    /**
     * @function ThiessenAnalystParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        if (me.clipRegion) {
            me.clipRegion.destroy();
            me.clipRegion = null;
        }
        me.createResultDataset = null;
        me.resultDatasetName = null;
        me.resultDatasourceName = null;
        me.returnResultRegion = null;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/DatasetThiessenAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class DatasetThiessenAnalystParameters
 * @deprecatedclass SuperMap.DatasetThiessenAnalystParameters
 * @category iServer SpatialAnalyst ThiessenAnalyst
 * @classdesc 数据集泰森多边形分析参数类。
 * @param {Object} options - 参数。
 * @param {(GeometryPolygon|L.Polygon|ol.geom.Polygon|GeoJSONObject)} [options.clipRegion] - 结果数据裁剪区域，可以为 null，表示不对结果进行裁剪。
 * @param {boolean} [options.createResultDataset] - 是否返回结果数据集。如果为 true，则必须设置属性 resultDatasetName 和 resultDatasourceName。
 * @param {string} [options.dataset] - 数据集名称待分析的数据集名称，请使用 "datasetName@datasourceName" 格式来表示。
 * @param {FilterParameter} [options.filterQueryParameter] - 过滤参数类，即对数据集中的所有点进行分析。
 * @param {string} [options.resultDatasetName] - 指定结果数据集名称。
 * @param {boolean} [options.resultDatasourceName] - 指定结果数据集所在数据源。
 * @param {boolean} [options.returnResultRegion] - 是否返回分析得到的多边形面数组。
 * @extends {ThiessenAnalystParameters}
 * @usage
 */
class DatasetThiessenAnalystParameters extends ThiessenAnalystParameters {


    constructor(options) {
        super(options);
        /**
         * @member {FilterParameter} [DatasetThiessenAnalystParameters.prototype.filterQueryParameter]
         * @description 过滤条件，对待分析数据集中的点进行过滤，即对数据集中的所有点进行分析。
         * @example
         *  var filterQueryParameter = new FilterParameter({
         *   name: "Countries@World",
         *   attributeFilter: "SmID>100"
         *  });
         */
        this.filterQueryParameter = null;

        /**
         * @member {string} DatasetThiessenAnalystParameters.prototype.dataset
         * @description 数据集名称待分析的数据集名称，请使用 "datasetName@datasourceName" 格式来表示。
         */
        this.dataset = null;

        if (options) {
            Util_Util.extend(this, options);
        }


        this.CLASS_NAME = "SuperMap.DatasetThiessenAnalystParameters";
    }

    /**
     * @function DatasetThiessenAnalystParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        if (me.filterQueryParameter) {
            me.filterQueryParameter.destroy();
            me.filterQueryParameter = null;
        }
    }

    /**
     * @function DatasetThiessenAnalystParameters.toObject
     * @param {DatasetThiessenAnalystParameters} datasetThiessenAnalystParameters - 泰森多边形分析服务参数类。
     * @param {DatasetThiessenAnalystParameters} tempObj - 泰森多边形分析服务参数对象。
     * @description 将泰森多边形分析服务参数对象转换为 JSON 对象。
     * @returns JSON 对象。
     */
    static toObject(datasetThiessenAnalystParameters, tempObj) {
        for (var name in datasetThiessenAnalystParameters) {
            if (name === "clipRegion") {
                tempObj.clipRegion = ServerGeometry.fromGeometry(datasetThiessenAnalystParameters.clipRegion);
            } else {
                tempObj[name] = datasetThiessenAnalystParameters[name];
            }
        }
    }

}


;// CONCATENATED MODULE: ./src/common/iServer/DatasourceService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class DatasourceService
 * @deprecatedclass SuperMap.DatasourceService
 * @category iServer Data Datasource
 * @classdesc 数据源查询服务类。
 * @param {string} url - 服务地址。如访问World Data服务，只需将url设为：http://localhost:8090/iserver/services/data-world/rest/data 即可。
 * @param {Object} options - 参数。
 * @param {DataFormat} [options.format=DataFormat.GEOJSON] - 查询结果返回格式，目前支持 iServerJSON 和 GeoJSON 两种格式。参数格式为 "ISERVER"，"GEOJSON"。
 * @param {string} options.datasource - 要查询的数据集所在的数据源名称。
 * @param {string} options.dataset - 要查询的数据集名称。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @extends {CommonServiceBase}
 * @usage
 */

class DatasourceService extends CommonServiceBase {

    constructor(url, options) {
        super(url, options);
        if (options) {
            Util_Util.extend(this, options);
        }
        this.CLASS_NAME = "SuperMap.DatasourceService";
    }


    /**
     * @function DatasourceService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }


    /**
     * @function DatasourceService.prototype.getDatasourceService
     * @description 获取指定数据源信息。
     * @param {string} datasourceName - 数据源名称。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    getDatasourceService(datasourceName, callback) {
        let url = Util_Util.urlPathAppend(this.url,`datasources/name/${datasourceName}`);
        return this.processAsync(url, "GET", callback);
    }

    /**
     * @function DatasourceService.prototype.getDatasourcesService
     * @description 获取所有数据源信息。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    getDatasourcesService(callback) {
        let url = Util_Util.urlPathAppend(this.url,`datasources`);
        return this.processAsync(url, "GET", callback);
    }
    /**
     * @function DatasourceService.prototype.setDatasourceService
     * @description 更新数据源信息。
     * @param {Object} params 请求参数信息。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    setDatasourceService(params, callback) {
        if (!params) {
            return;
        }
        const url = Util_Util.urlPathAppend(this.url,`datasources/name/${params.datasourceName}`);
        return this.processAsync(url, "PUT", callback, params);
    }

    processAsync(url, method, callback, params) {
       var me = this;
       let requestConfig = {
          url,
          method,
          scope: me,
          success: callback,
          failure: callback
        }
        params && (requestConfig.data = Util_Util.toJSON(params));
        return me.request(requestConfig);
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/DensityKernelAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class DensityKernelAnalystParameters
 * @deprecatedclass SuperMap.DensityKernelAnalystParameters
 * @category  iServer SpatialAnalyst DensityAnalyst
 * @classdesc 核密度分析参数类。
 * @param {Object} options - 参数。
 * @param {string} options.dataset - 要用来做核密度分析数据源中数据集的名称。该名称用形如 "数据集名称@数据源别名" 形式来表示，例如：BaseMap_P@Jingjin。
 * @param {string} options.fieldName - 用于进行核密度分析的测量值的字段名称，核密度分析不支持文本类型的字段。
 * @param {string} options.resultGridName - 指定结果数据集名称。
 * @param {(SuperMap.Bounds|L.Bounds|L.LatLngBounds|ol.extent|mapboxgl.LngLatBounds|GeoJSONObject)} [options.bounds] - 核密度分析的范围，用于确定结果栅格数据集的范围。如果缺省，则默认为原数据集的范围。
 * @param {number} [options.searchRadius] - 栅格邻域内用于计算密度的查找半径，单位与当前数据集相同。默认值为当前数据集的长宽中的最大值除30。
 * @param {number} [options.resultGridDatasetResolution] - 密度分析结果栅格数据的分辨率，单位与当前数据集相同。默认值为当前数据集的长宽中的最小值除500。
 * @param {string} [options.targetDatasource] - 指定的存储结果数据集的数据源，默认为当前分析的数据集所在的数据源。
 * @param {boolean} [options.deleteExistResultDataset=false] - 如果用户命名的结果数据集名称与已有的数据集重名，是否删除已有的数据集。
 * @usage
 */
class DensityKernelAnalystParameters {


    constructor(options) {
        /**
         * @member {string} DensityKernelAnalystParameters.prototype.dataset
         * @description 要用来做核密度分析数据源中数据集的名称。
         * 该名称用形如 "数据集名称@数据源别名" 形式来表示，例如：Railway@Changchun。
         * 注：核密度分析支持点数据集和线数据集。
         */
        this.dataset = null;

        /**
         * @member {(SuperMap.Bounds|L.Bounds|L.LatLngBounds|ol.extent|mapboxgl.LngLatBounds|GeoJSONObject)} [DensityKernelAnalystParameters.prototype.bounds]
         * @description 核密度分析的范围，用于确定结果栅格数据集的范围。
         * 如果缺省，则默认为原数据集的范围。
         */
        this.bounds = null;

        /**
         * @member {string} DensityKernelAnalystParameters.prototype.fieldName
         * @description 用于进行核密度分析的测量值的字段名称，核密度分析不支持文本类型的字段。
         */
        this.fieldName = null;

        /**
         * @member {number} [DensityKernelAnalystParameters.prototype.resultGridDatasetResolution]
         * @description 密度分析结果栅格数据的分辨率，单位与当前数据集相同。默认值为当前数据集的长宽中的最小值除500。
         */
        this.resultGridDatasetResolution = null;

        /**
         * @member {number} [DensityKernelAnalystParameters.prototype.searchRadius]
         * @description 栅格邻域内用于计算密度的查找半径，单位与当前数据集相同。默认值为当前数据集的长宽中的最大值除30。
         */
        this.searchRadius = null;

        /**
         * @member {string} [DensityKernelAnalystParameters.prototype.targetDatasource]
         * @description 指定的存储结果数据集的数据源，默认为当前分析的数据集所在的数据源。
         */
        this.targetDatasource = null;

        /**
         * @member {string} DensityKernelAnalystParameters.prototype.resultGridName
         * @description 指定结果数据集名称。
         */
        this.resultGridName = null;

        /**
         * @member {boolean} [DensityKernelAnalystParameters.prototype.deleteExistResultDataset=false]
         * @description 如果用户命名的结果数据集名称与已有的数据集重名，是否删除已有的数据集。
         */
        this.deleteExistResultDataset = false;
        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.DensityKernelAnalystParameters";

    }


    /**
     * @function DensityKernelAnalystParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.dataset = null;
        me.bounds = null;
        me.fieldName = null;
        me.resultGridDatasetResolution = null;
        me.searchRadius = null;

        me.targetDatasource = null;
        me.resultGridName = null;
        me.deleteExistResultDataset = null;
    }

    /**
     * @function DensityKernelAnalystParameters.toObject
     * @param {DensityKernelAnalystParameters} densityKernelAnalystParameters -核密度分析参数类。
     * @param {DensityKernelAnalystParameters} tempObj - 核密度分析参数对象。
     * @description 将核密度分析参数对象转换成 JSON 对象。
     * @returns JSON 对象。
     */
    static toObject(densityKernelAnalystParameters, tempObj) {
        for (var name in densityKernelAnalystParameters) {
            if (name !== "dataset") {
                tempObj[name] = densityKernelAnalystParameters[name];
            }
        }
    }

}


;// CONCATENATED MODULE: ./src/common/iServer/DensityAnalystService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class DensityAnalystService
 * @deprecatedclass SuperMap.DensityAnalystService
 * @category iServer SpatialAnalyst DensityAnalyst
 * @classdesc
 * 密度分析服务类，密度分析可计算每个输出栅格像元周围圆形邻域内输入的点或线对象的密度。
 * 密度分析，在某种意义上来说，相当于在表面上将输入的点线对象的测量值散开来，将每个点或线对象的测量量分布在整个研究区域，并计算输出栅格中每个像元的密度值。目前提供1种密度分析：核密度分析（Kernel）。
 * @param {string} url - 服务地址。如 http://localhost:8090/iserver/services/spatialanalyst-changchun/restjsr/spatialanalyst 。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @extends {SpatialAnalystBase}
 * @example  例如：
 *  var myDensityAnalystService = new DensityAnalystService(url);
 *  myDensityAnalystService.on({
 *     "processCompleted": processCompleted,
 *     "processFailed": processFailed
 *     }
 *  );
 * @usage
 */
class DensityAnalystService extends SpatialAnalystBase {


    constructor(url, options) {
        super(url, options);
        /**
         * @member {string} DensityAnalystService.prototype.mode
         * @description 密度分析类型。
         */
        this.mode = null;

        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.DensityAnalystService";
    }

    /**
     * @function DensityAnalystService.prototype.destroy
     * @description 释放资源,将引用资源的属性置空。
     */
    destroy() {
        super.destroy();
        this.mode = null;
    }

    /**
     * @function DensityAnalystService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {DensityKernelAnalystParameters} parameter - 核密度分析参数。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(parameter, callback) {
        var me = this;
        var parameterObject = new Object();

        if (parameter instanceof DensityKernelAnalystParameters) {
            me.url = Util_Util.urlPathAppend(me.url, 'datasets/' + parameter.dataset + '/densityanalyst/kernel');
            me.mode = "kernel";
        }

        DensityKernelAnalystParameters.toObject(parameter, parameterObject);
        var jsonParameters = Util_Util.toJSON(parameterObject);
        me.url = Util_Util.urlAppend(me.url, 'returnContent=true');

        return me.request({
            method: "POST",
            data: jsonParameters,
            scope: me,
            success: callback,
            failure: callback
        });
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/EditFeaturesParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class EditFeaturesParameters
 * @deprecatedclass SuperMap.EditFeaturesParameters
 * @category iServer Data Feature
 * @classdesc 数据集添加、修改、删除参数类。
 * @param {Object} options - 参数。
 * @param {Array.<FeatureVector|GeoJSONObject|ol.Feature>} options.features - 当前需要创建或者是修改的要素集。
 * @param {boolean} [options.returnContent=false] - 是否返回要素内容。如果为true则返回创建要素的 ID 数组，否则返回 featureResult 资源的 URI。
 * @param {EditType} [options.editType=EditType.ADD] - POST 动作类型 (ADD、UPDATE、DELETE)。
 * @param {Array.<string|number>} [options.IDs] - 删除要素时的要素的 ID 数组。
 * @usage
 */
class EditFeaturesParameters {


    constructor(options) {
        /**
         * @member {string} EditFeaturesParameters.prototype.dataSourceName
         * @description 当前需要创建或者是修改的要素的数据源。
         */
        this.dataSourceName = null;

        /**
         * @member {string} EditFeaturesParameters.prototype.dataSetName
         * @description 当前需要创建或者是修改的要素的数据集。
         */
        this.dataSetName = null;

        /**
         * @member {Array.<FeatureVector|GeoJSONObject|ol.Feature>} EditFeaturesParameters.prototype.features
         * @description 当前需要创建或者是修改的要素集。
         */
        this.features = null;

        /**
         * @member {EditType} [EditFeaturesParameters.prototype.editType=EditType.ADD]
         * @description 要素集更新类型 (add、update、delete)。
         */
        this.editType = EditType.ADD;

        /**
         * @member {Array.<string|number>} [EditFeaturesParameters.prototype.IDs]
         * @description 执行删除时要素集 ID 集合。
         */
        this.IDs = null;

        /**
         * @member {boolean} [EditFeaturesParameters.prototype.returnContent=false]
         * @description 要素添加时，isUseBatch 不传或传为 false 的情况下有效。
         *              true 表示直接返回新创建的要素的 ID 数组；false 表示返回创建的 featureResult 资源的 URI。
         */
        this.returnContent = false;

        /**
         * @member {boolean} [EditFeaturesParameters.prototype.isUseBatch=false]
         * @description 是否使用批量添加要素功能，要素添加时有效。批量添加能够提高要素编辑效率。true 表示批量添加；false 表示不使用批量添加。
         */
        this.isUseBatch = false;
        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.EditFeaturesParameters";
    }


    /**
     * @function EditFeaturesParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.dataSourceName = null;
        me.dataSetName = null;
        me.features = null;
        me.editType = null;
        me.IDs = null;
        me.returnContent = null;
    }

    /**
     * @function EditFeaturesParameters.prototype.toJsonParameters
     * @description 将 EditFeaturesParameters 对象参数转换为 JSON 字符串。
     * @param {EditFeaturesParameters} params - 地物编辑参数。
     * @returns {string} JSON 字符串。
     */
    static toJsonParameters(params) {
        var feature,
            len,
            features,
            editType = params.editType;

        if (editType === EditType.DELETE) {
            if (params.IDs === null) {
                return;
            }

            features = {ids: params.IDs};
        } else {
            features = [];
            if (params.features) {
                len = params.features.length;
                for (var i = 0; i < len; i++) {
                    feature = params.features[i];
                    feature.geometry = ServerGeometry.fromGeometry(feature.geometry);
                    features.push(feature);
                }
            }
        }

        return Util_Util.toJSON(features);
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/EditFeaturesService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/






/**
 * @class EditFeaturesService
 * @deprecatedclass SuperMap.EditFeaturesService
 * @category  iServer Data Feature
 * @classdesc 数据服务中数据集添加、更新、删除服务类。
 * @extends {CommonServiceBase}
 * @param {string} url - 服务端的数据服务资源地址。请求数据服务中数据集编辑服务，URL 应为：</br>
 * http://{服务器地址}:{服务端口号}/iserver/services/{数据服务名}/rest/data/datasources/name/{数据源名}/datasets/name/{数据集名} 。</br>
 * 例如：http://localhost:8090/iserver/services/data-jingjin/rest/data/datasources/name/Jingjin/datasets/name/Landuse_R
 * @param {Object} options - 参数。
 * @param {DataFormat} [format] -查询结果返回格式，目前支持iServerJSON 和GeoJSON两种格式。参数格式为"ISERVER","GEOJSON"。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @example
 * var myService = new EditFeaturesService(url);
 * @usage
 */
class EditFeaturesService extends CommonServiceBase {

    constructor(url, options) {
        super(url, options);
        /**
         * @member {boolean} [EditFeaturesService.prototype.returnContent=false]
         * @description要素添加时，isUseBatch 不传或传为 false 的情况下有效。true 表示直接返回新创建的要素的 ID 数组;false 表示返回创建的 featureResult 资源的 URI。
         */
        this.returnContent = false;

        /**
         * @member {boolean} [EditFeaturesService.prototype.isUseBatch=false]
         * @description 是否使用批量添加要素功能，要素添加时有效。
         *              批量添加能够提高要素编辑效率。
         *              true 表示批量添加；false 表示不使用批量添加。
         */
        this.isUseBatch = false;

        if (options) {
            Util_Util.extend(this, options);
        }
        this.url = Util_Util.urlPathAppend(this.url, 'features');

        this.CLASS_NAME = "SuperMap.EditFeaturesService";
    }


    /**
     * @function EditFeaturesService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        me.returnContent = null;
        me.isUseBatch = null;
        me.fromIndex = null;
        me.toIndex = null;
    }


    /**
     * @function EditFeaturesService.prototype.processAsync
     * @description 负责将客户端的更新参数传递到服务端。
     * @param {EditFeaturesParameters} params - 编辑要素参数。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof EditFeaturesParameters)) {
            return;
        }
        var me = this,
            method = "POST",
            ids = "",
            editType = params.editType,
            jsonParameters = null;

        me.returnContent = params.returnContent;
        me.isUseBatch = params.isUseBatch;
        jsonParameters = EditFeaturesParameters.toJsonParameters(params);
        if (editType === EditType.DELETE) {
            ids = Util_Util.toJSON(params.IDs);
            jsonParameters = ids;
            var urlWithIds = Util_Util.urlAppend(me.url, Util_Util.getParameterString({ids}))
            if(FetchRequest.urlIsLong(urlWithIds)) {
                me.url = Util_Util.urlAppend(me.url, Util_Util.getParameterString({_method: 'DELETE'}));
                method = "POST";
            } else{
                me.url = urlWithIds;
                method = "DELETE";
            }
           
        } else if (editType === EditType.UPDATE) {
            method = "PUT";
        } else {
            if (me.isUseBatch) {
                me.url = Util_Util.urlAppend(me.url, `isUseBatch=${me.isUseBatch}`);
                me.returnContent = false;
            }
            if (me.returnContent) {
                me.url = Util_Util.urlAppend(me.url, 'returnContent=true');
                method = "POST";
            }
        }

        return me.request({
            method: method,
            data: jsonParameters,
            scope: me,
            success: callback,
            failure: callback
        });
    }

    /**
     * @function EditFeaturesService.prototype.getMetaData
     * @description 获取地理要素元信息。
     * @version 11.1.1
     */
    getMetaData(params, callback){
      var me = this;
      var featureId = params.featureId;
      me.url = Util_Util.urlPathAppend(me.url, featureId +'/metadata');
      return me.request({
        method: "GET",
        data: null,
        scope: me,
        success: callback,
        failure: callback
      });
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalyst3DParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class FacilityAnalyst3DParameters
 * @deprecatedclass SuperMap.FacilityAnalyst3DParameters
 * @category  iServer FacilityAnalyst3D
 * @classdesc 最近设施分析参数基类。最近设施分析是指在网络上给定一个事件点和一组设施点，查找从事件点到设施点（或从设施点到事件点）以最小耗费能到达的最佳路径。
 * 设施点一般为学校、超市、加油站等服务设施；事件点为需要服务设施的事件位置。例如事件发生点是一起交通事故，要求查找在 10 分钟内能到达的最近医院，
 * 超过 10 分钟能到达的都不予考虑。此例中，事故发生地即是一个事件点，周边的医院则是设施点。最近设施查找实际上也是一种路径分析，因此对路径分析起作用的障碍边、障碍点、转向表、耗费等属性在最近设施分析时同样可设置。
 * @param {Object} options - 参数。
 * @param {string} options.weightName - 指定的权值字段信息对象的名称。
 * @param {number} [options.edgeID] - 指定的弧段 ID，edgeID 与 nodeID 必须指定一个。
 * @param {number} [options.nodeID] - 指定的结点 ID，edgeID 与 nodeID 必须指定一个。
 * @param {boolean} [options.isUncertainDirectionValid=false] - 指定不确定流向是否有效。指定为 true，表示不确定流向有效，遇到不确定流向时分析继续进行；
 *                                                      指定为 false，表示不确定流向无效，遇到不确定流向将停止在该方向上继续查找。
 * @usage
 */
class FacilityAnalyst3DParameters {


    constructor(options) {
        /**
         * @member {number} [FacilityAnalyst3DParameters.prototype.edgeID]
         * @description 指定的弧段 ID，edgeID 与 nodeID 必须指定一个。
         */
        this.edgeID = null;

        /**
         * @member {number} [FacilityAnalyst3DParameters.prototype.nodeID]
         * @description 指定的结点 ID，edgeID 与 nodeID 必须指定一个。
         */
        this.nodeID = null;

        /**
         * @member {string} FacilityAnalyst3DParameters.prototype.weightName
         * @description 指定的权值字段信息对象的名称。
         */
        this.weightName = null;

        /**
         * @member {boolean} [FacilityAnalyst3DParameters.prototype.isUncertainDirectionValid=false]
         * @description 指定不确定流向是否有效。指定为 true，表示不确定流向有效，遇到不确定流向时分析继续进行；
         *              指定为 false，表示不确定流向无效，遇到不确定流向将停止在该方向上继续查找。
         */
        this.isUncertainDirectionValid = false;
        Util_Util.extend(this, options);
        this.CLASS_NAME = "SuperMap.FacilityAnalyst3DParameters";

    }

    /**
     * @function FacilityAnalyst3DParameters.prototype.destroy
     * @description 释放资源，将资源的属性置空。
     */
    destroy() {
        var me = this;
        me.edgeID = null;
        me.nodeID = null;
        me.weightName = null;
        me.isUncertainDirectionValid = null;
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalystSinks3DParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class FacilityAnalystSinks3DParameters
 * @deprecatedclass SuperMap.FacilityAnalystSinks3DParameters
 * @category  iServer FacilityAnalyst3D Sinks
 * @classdesc 最近设施分析参数类(汇查找资源)。最近设施分析是指在网络上给定一个事件点和一组设施点，查找从事件点到设施点(或从设施点到事件点)以最小耗费能到达的最佳路径。
 *            设施点一般为学校、超市、加油站等服务设施；事件点为需要服务设施的事件位置。例如事件发生点是一起交通事故，要求查找在10分钟内能到达的最近医院，超过10分钟能到达的都不予考虑。此例中，事故发生地即是一个事件点，周边的医院则是设施点。最近设施查找实际上也是一种路径分析，因此对路径分析起作用的障碍边、障碍点、转向表、耗费等属性在最近设施分析时同样可设置。
 * @extends {FacilityAnalyst3DParameters}
 * @param {Object} options - 参数。
 * @param {string} options.weightName - 指定的权值字段信息对象的名称。
 * @param {number} [options.edgeID] - 指定的弧段 ID，edgeID 与 nodeID 必须指定一个。
 * @param {number} [options.nodeID] - 指定的结点 ID，edgeID 与 nodeID 必须指定一个。
 * @param {boolean} [options.isUncertainDirectionValid=false] - 指定不确定流向是否有效。指定为 true，表示不确定流向有效，遇到不确定流向时分析继续进行；
 * 指定为 false，表示不确定流向无效，遇到不确定流向将停止在该方向上继续查找。
 * @usage
 */
class FacilityAnalystSinks3DParameters extends FacilityAnalyst3DParameters {

    constructor(options) {
        super(options);
        this.CLASS_NAME = "SuperMap.FacilityAnalystSinks3DParameters";
    }


    /**
     * @function FacilityAnalystSinks3DParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalystSinks3DService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class FacilityAnalystSinks3DService
 * @deprecatedclass SuperMap.FacilityAnalystSinks3DService
 * @category iServer FacilityAnalyst3D Sinks
 * @classdesc  最近设施分析服务类(汇查找资源)<br>
 *             最近设施分析是指在网络上给定一个事件点和一组设施点，
 *             查找从事件点到设施点(或从设施点到事件点)以最小耗费能到达的最佳路径。
 *             该类负责将客户端指定的最近设施分析参数传递给服务端，并接收服务端返回的结果数据。
 *             最近设施分析结果通过该类支持的事件的监听函数参数获取
 * @extends {CommonServiceBase}
 * @example
 * var myFacilityAnalystSinks3DService = new FacilityAnalystSinks3DService(url);
 * @param {string} url - 网络分析服务地址。请求网络分析服务，URL应为：<br>
 *                       http://{服务器地址}:{服务端口号}/iserver/services/{网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}；<br>
 *                       例如:"http://localhost:8090/iserver/services/components-rest/rest/networkanalyst/RoadNet@Changchun"。<br>
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class FacilityAnalystSinks3DService extends CommonServiceBase {

    constructor(url, options) {
        super(url, options);
        this.CLASS_NAME = "SuperMap.FacilityAnalystSinks3DService";
    }


    /**
     * @function FacilityAnalystSinks3DService.prototype.destroy
     * @override
     */
    destroy() {
        CommonServiceBase.prototype.destroy.apply(this, arguments);
    }


    /**
     * @function FacilityAnalystSinks3DService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {FacilityAnalystSinks3DParameters} params - 最近设施分析参数类(汇查找资源)
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof FacilityAnalystSinks3DParameters)) {
            return;
        }
        var me = this, jsonObject;
        me.url = Util_Util.urlPathAppend(me.url, 'sinks');
        jsonObject = {
            edgeID: params.edgeID,
            nodeID: params.nodeID,
            weightName: params.weightName,
            isUncertainDirectionValid: params.isUncertainDirectionValid
        };
        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalystSources3DParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class FacilityAnalystSources3DParameters
 * @deprecatedclass SuperMap.FacilityAnalystSources3DParameters
 * @category  iServer FacilityAnalyst3D Sources
 * @classdesc 最近设施分析参数类(源查找资源)。最近设施分析是指在网络上给定一个事件点和一组设施点，查找从事件点到设施点(或从设施点到事件点)以最小耗费能到达的最佳路径。
 *            设施点一般为学校、超市、加油站等服务设施；事件点为需要服务设施的事件位置。例如事件发生点是一起交通事故，要求查找在10分钟内能到达的最近医院，超过10分钟能到达的都不予考虑。此例中，事故发生地即是一个事件点，周边的医院则是设施点。最近设施查找实际上也是一种路径分析，因此对路径分析起作用的障碍边、障碍点、转向表、耗费等属性在最近设施分析时同样可设置。
 * @extends {FacilityAnalyst3DParameters}
 * @param {Object} options - 参数。
 * @param {string} options.weightName - 指定的权值字段信息对象的名称。
 * @param {number} [options.edgeID] - 指定的弧段 ID，edgeID 与 nodeID 必须指定一个。
 * @param {number} [options.nodeID] - 指定的结点 ID，edgeID 与 nodeID 必须指定一个。
 * @param {boolean} [options.isUncertainDirectionValid=false] - 指定不确定流向是否有效。指定为 true，表示不确定流向有效，遇到不确定流向时分析继续进行；
 * 指定为 false，表示不确定流向无效，遇到不确定流向将停止在该方向上继续查找。
 * @usage
 */
class FacilityAnalystSources3DParameters extends FacilityAnalyst3DParameters {


    constructor(options) {
        super(options);
        this.CLASS_NAME = "SuperMap.FacilityAnalystSources3DParameters";
    }

    /**
     * @function FacilityAnalystSources3DParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalystSources3DService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class FacilityAnalystSources3DService
 * @deprecatedclass SuperMap.FacilityAnalystSources3DService
 * @category  iServer FacilityAnalyst3D Sources
 * @classdesc 最近设施分析服务类(源查找资源)
 *            最近设施分析是指在网络上给定一个事件点和一组设施点，
 *            查找从事件点到设施点(或从设施点到事件点)以最小耗费能到达的最佳路径。
 *            该类负责将客户端指定的最近设施分析参数传递给服务端，并接收服务端返回的结果数据。
 *            最近设施分析结果通过该类支持的事件的监听函数参数获取。
 * @extends {CommonServiceBase}
 * @param {string} url - 服务地址。请求网络分析服务，URL应为：
 *                       http://{服务器地址}:{服务端口号}/iserver/services/{网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}；
 *                       例如:"http://localhost:8090/iserver/services/components-rest/rest/networkanalyst/RoadNet@Changchun"。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class FacilityAnalystSources3DService extends CommonServiceBase {

    constructor(url, options) {
        super(url, options);

        this.CLASS_NAME = "SuperMap.FacilityAnalystSources3DService";
    }


    /**
     * @function FacilityAnalystSources3DService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }


    /**
     * @function FacilityAnalystSources3DService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {FacilityAnalystSources3DParameters} params - 最近设施分析参数类（源查找资源）
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof FacilityAnalystSources3DParameters)) {
            return;
        }
        var me = this, jsonObject;
        me.url = Util_Util.urlPathAppend(me.url, 'sources');
        jsonObject = {
            edgeID: params.edgeID,
            nodeID: params.nodeID,
            weightName: params.weightName,
            isUncertainDirectionValid: params.isUncertainDirectionValid
        };
        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalystStreamParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class FacilityAnalystStreamParameters
 * @deprecatedclass SuperMap.FacilityAnalystStreamParameters
 * @category iServer NetworkAnalyst UpstreamCirticalFaclilities
 * @classdesc 上游/下游关键设施查找资源参数类。
 * @param {Object} options - 参数。
 * @param {Array.<number>} options.sourceNodeIDs - 指定的设施点 ID 数组。
 * @param {number} options.queryType - 分析类型，只能是 0 (上游关键设施查询) 或者是 1（下游关键设施查询）。
 * @param {number} [options.edgeID] - 指定的弧段 ID，edgeID 与 nodeID 必须指定一个。
 * @param {number} [options.nodeID] - 指定的结点 ID，edgeID 与 nodeID 必须指定一个。
 * @param {boolean} [options.isUncertainDirectionValid=false] - 指定不确定流向是否有效。
 * @usage
 */
class FacilityAnalystStreamParameters {


    constructor(options) {
        /**
         * @member {Array.<number>} [FacilityAnalystStreamParameters.prototype.sourceNodeIDs]
         * @description 指定的设施点 ID 数组。
         */
        this.sourceNodeIDs = null;

        /**
         * @member {number} [FacilityAnalystStreamParameters.prototype.edgeID]
         * @description 指定的弧段 ID，edgeID 与 nodeID 必须指定一个。
         */
        this.edgeID = null;

        /**
         * @member {number} [FacilityAnalystStreamParameters.prototype.nodeID]
         * @description 指定的结点 ID，edgeID 与 nodeID 必须指定一个。
         */
        this.nodeID = null;

        /**
         * @member {boolean} [FacilityAnalystStreamParameters.prototype.isUncertainDirectionValid=false]
         * @description 指定不确定流向是否有效。指定为 true，表示不确定流向有效，遇到不确定流向时分析继续进行；
         *              指定为 false，表示不确定流向无效，遇到不确定流向将停止在该方向上继续查找。
         */
        this.isUncertainDirectionValid = false;

        /**
         * @member {number} FacilityAnalystStreamParameters.prototype.queryType
         * @description 分析类型，只能是 0 (上游关键设施查询) 或者是 1（下游关键设施查询）。
         */
        this.queryType = null;
        Util_Util.extend(this, options);
        this.CLASS_NAME = "SuperMap.FacilityAnalystStreamParameters";
    }


    /**
     * @function FacilityAnalystStreamParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.edgeID = null;
        me.nodeID = null;
        me.weightName = null;
        me.isUncertainDirectionValid = null;
        me.type = null;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalystStreamService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class FacilityAnalystStreamService
 * @deprecatedclass SuperMap.FacilityAnalystStreamService
 * @category iServer NetworkAnalyst UpstreamCirticalFaclilities
 * @classdesc 上游/下游 关键设施查找资源服务类：即查找给定弧段或节点的上游/下游中的关键设施结点，返回关键结点 ID 数组及其下游弧段 ID 数组。
 * @extends NetworkAnalystServiceBase
 * @param {string} url - 服务地址。请求网络分析服务，URL应为：
 *                       http://{服务器地址}:{服务端口号}/iserver/services/{网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}；
 *                       例如: "http://localhost:8090/iserver/services/test/rest/networkanalyst/WaterNet@FacilityNet";
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class FacilityAnalystStreamService extends NetworkAnalystServiceBase {

    constructor(url, options) {
        super(url, options);
        this.CLASS_NAME = "SuperMap.FacilityAnalystStreamService";
    }


    /**
     * @function FacilityAnalystStreamService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }


    /**
     * @function FacilityAnalystStreamService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {FacilityAnalystStreamParameters} params - 上游/下游关键设施查找资源参数类。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof FacilityAnalystStreamParameters)) {
            return;
        }
        var me = this,
            jsonObject;
        //URL 通过参数类型来判断是 上游 还是下游 查询
        if (params.queryType === 0) {
            me.url = Util_Util.urlPathAppend(me.url, 'upstreamcirticalfaclilities');
        } else if (params.queryType === 1) {
            me.url = Util_Util.urlPathAppend(me.url, 'downstreamcirticalfaclilities');
        } else {
            return;
        }

        jsonObject = {
            sourceNodeIDs: params.sourceNodeIDs,
            isUncertainDirectionValid: params.isUncertainDirectionValid
        };

        if (params.edgeID !== null && params.nodeID !== null) {
            return;
        }
        if (params.edgeID === null && params.nodeID === null) {
            return;
        }
        if (params.edgeID !== null) {
            jsonObject.edgeID = params.edgeID;
        } else {
            jsonObject.nodeID = params.nodeID;
        }

        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalystTracedown3DParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class FacilityAnalystTracedown3DParameters
 * @deprecatedclass SuperMap.FacilityAnalystTracedown3DParameters
 * @category iServer FacilityAnalyst3D TraceDownResult
 * @classdesc 下游追踪资源参数类。
 * @extends {FacilityAnalyst3DParameters}
 * @param {Object} options - 参数。
 * @param {string} options.weightName - 指定的权值字段信息对象的名称。
 * @param {number} [options.edgeID] - 指定的弧段 ID，edgeID 与 nodeID 必须指定一个。
 * @param {number} [options.nodeID] - 指定的结点 ID，edgeID 与 nodeID 必须指定一个。
 * @param {boolean} [options.isUncertainDirectionValid=false] - 指定不确定流向是否有效。指定为 true，表示不确定流向有效，遇到不确定流向时分析继续进行；
 * 指定为 false，表示不确定流向无效，遇到不确定流向将停止在该方向上继续查找。
 * @usage
 */
class FacilityAnalystTracedown3DParameters extends FacilityAnalyst3DParameters {

    constructor(options) {
        super(options);
        this.CLASS_NAME = "SuperMap.FacilityAnalystTracedown3DParameters";
    }

    /**
     * @function FacilityAnalystTracedown3DParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalystTracedown3DService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class FacilityAnalystTracedown3DService
 * @deprecatedclass SuperMap.FacilityAnalystTracedown3DService
 * @category iServer FacilityAnalyst3D TraceDownResult
 * @classdesc 下游追踪资源服务类
 * @extends {CommonServiceBase}
 * @param {string} url - 服务地址。请求网络分析服务，URL应为：
 *                       http://{服务器地址}:{服务端口号}/iserver/services/{网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}；
 *                       例如:"http://localhost:8090/iserver/services/components-rest/rest/networkanalyst/RoadNet@Changchun"。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class FacilityAnalystTracedown3DService extends CommonServiceBase {

    constructor(url, options) {
        super(url, options);
        this.CLASS_NAME = "SuperMap.FacilityAnalystTracedown3DService";
    }

    /**
     * @function FacilityAnalystTracedown3DService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function FacilityAnalystTracedown3DService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {FacilityAnalystTracedown3DParameters} params - 下游追踪资源参数类。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof FacilityAnalystTracedown3DParameters)) {
            return;
        }
        var me = this, jsonObject;
        me.url = Util_Util.urlPathAppend(me.url, 'tracedownresult');

        jsonObject = {
            edgeID: params.edgeID,
            nodeID: params.nodeID,
            weightName: params.weightName,
            isUncertainDirectionValid: params.isUncertainDirectionValid
        };
        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalystTraceup3DParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class FacilityAnalystTraceup3DParameters
 * @deprecatedclass SuperMap.FacilityAnalystTraceup3DParameters
 * @category iServer FacilityAnalyst3D TraceUpResult
 * @classdesc 上游追踪资源参数类。
 * @extends {FacilityAnalyst3DParameters}
 * @param {Object} options - 参数。
 * @param {string} options.weightName - 指定的权值字段信息对象的名称。
 * @param {number} [options.edgeID] - 指定的弧段ID，edgeID 与 nodeID 必须指定一个。
 * @param {number} [options.nodeID] - 指定的结点ID，edgeID 与 nodeID 必须指定一个。
 * @param {boolean} [options.isUncertainDirectionValid=false] - 指定不确定流向是否有效。指定为 true，表示不确定流向有效，遇到不确定流向时分析继续进行；
 * 指定为 false，表示不确定流向无效，遇到不确定流向将停止在该方向上继续查找。
 * @usage
 */
class FacilityAnalystTraceup3DParameters extends FacilityAnalyst3DParameters {

    constructor(options) {
        super(options);
        this.CLASS_NAME = "SuperMap.FacilityAnalystTraceup3DParameters";
    }

    /**
     * @function FacilityAnalystTraceup3DParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalystTraceup3DService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class FacilityAnalystTraceup3DService
 * @deprecatedclass SuperMap.FacilityAnalystTraceup3DService
 * @category iServer FacilityAnalyst3D TraceUpResult
 * @classdesc 上游追踪资源服务类
 * @extends {CommonServiceBase}
 * @param {string} url - 服务地址。请求网络分析服务，URL应为：
 *                       http://{服务器地址}:{服务端口号}/iserver/services/{网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}；
 *                       例如:"http://localhost:8090/iserver/services/components-rest/rest/networkanalyst/RoadNet@Changchun"。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class FacilityAnalystTraceup3DService extends CommonServiceBase {
    constructor(url, options) {
        super(url, options);
        this.CLASS_NAME = "SuperMap.FacilityAnalystTraceup3DService";
    }

    /**
     * @function FacilityAnalystTraceup3DService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function FacilityAnalystTraceup3DService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {FacilityAnalystTraceup3DParameters} params - 上游追踪资源参数类
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof FacilityAnalystTraceup3DParameters)) {
            return;
        }
        var me = this, jsonObject;
        me.url = Util_Util.urlPathAppend(me.url, 'traceupresult');
        jsonObject = {
            edgeID: params.edgeID,
            nodeID: params.nodeID,
            weightName: params.weightName,
            isUncertainDirectionValid: params.isUncertainDirectionValid
        };
        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalystUpstream3DParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class FacilityAnalystUpstream3DParameters
 * @deprecatedclass SuperMap.FacilityAnalystUpstream3DParameters
 * @category iServer FacilityAnalyst3D UpstreamCirticalFaclilities
 * @classdesc 上游关键设施查找资源参数类。
 * @extends {FacilityAnalyst3DParameters}
 * @param {Object} options - 参数。
 * @param {Array.<number>} options.sourceNodeIDs - 指定的设施点 ID 数组。
 * @param {number} [options.edgeID] - 指定的弧段ID。edgeID 与 nodeID 必须指定一个。
 * @param {number} [options.nodeID] - 指定的结点ID。edgeID 与 edgeID 必须指定一个。
 * @param {boolean} [options.isUncertainDirectionValid=false] - 指定不确定流向是否有效。指定为 true，表示不确定流向有效，遇到不确定流向时分析继续进行；
 * 指定为 false，表示不确定流向无效，遇到不确定流向将停止在该方向上继续查找。
 * @usage
 */
class FacilityAnalystUpstream3DParameters extends FacilityAnalyst3DParameters {


    constructor(options) {
        super(options);
        options = options || {};
        this.sourceNodeIDs = null;
        Util_Util.extend(this, options);
        this.CLASS_NAME = "SuperMap.FacilityAnalystUpstream3DParameters";
    }


    /**
     * @function FacilityAnalystUpstream3DParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        this.sourceNodeIDs = null;
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/FacilityAnalystUpstream3DService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class FacilityAnalystUpstream3DService
 * @deprecatedclass SuperMap.FacilityAnalystUpstream3DService
 * @category  iServer FacilityAnalyst3D UpstreamCirticalFaclilities
 * @classdesc 上游关键设施查找资源服务类
 * @extends {CommonServiceBase}
 * @param {string} url - 服务地址。请求网络分析服务，URL应为：
 *                       http://{服务器地址}:{服务端口号}/iserver/services/{网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}；
 *                       例如:"http://localhost:8090/iserver/services/components-rest/rest/networkanalyst/RoadNet@Changchun"。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class FacilityAnalystUpstream3DService extends CommonServiceBase {

    constructor(url, options) {
        super(url, options);
        this.CLASS_NAME = "SuperMap.FacilityAnalystUpstream3DService";
    }

    /**
     * @function FacilityAnalystUpstream3DService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function FacilityAnalystUpstream3DService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {FacilityAnalystUpstream3DParameters} params - 上游关键设施查找资源参数类
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof FacilityAnalystUpstream3DParameters)) {
            return;
        }
        var me = this, jsonObject;
        me.url = Util_Util.urlPathAppend(me.url, 'upstreamcirticalfaclilities');
        jsonObject = {
            sourceNodeIDs: params.sourceNodeIDs,
            edgeID: params.edgeID,
            nodeID: params.nodeID,
            isUncertainDirectionValid: params.isUncertainDirectionValid
        };
        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }
}


;// CONCATENATED MODULE: ./src/common/iServer/FieldParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class FieldParameters
 * @deprecatedclass SuperMap.FieldParameters
 * @category iServer Data Field
 * @classdesc 字段信息查询参数类。
 * @param {Object} options - 参数。
 * @param {string} options.datasource - 数据源名称。
 * @param {string} options.dataset - 数据集名称。
 * @usage
 */
class FieldParameters {


    constructor(options) {
        /**
         * @member {string} FieldParameters.prototype.datasource
         * @description 要查询的数据集所在的数据源名称。
         */
        this.datasource = null;

        /**
         *  @member {string} FieldParameters.prototype.dataset
         *  @description 要查询的数据集名称。
         */
        this.dataset = null;

        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.FieldParameters";
    }

    /**
     * @function FieldParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.datasource = null;
        me.dataset = null;
    }

}


;// CONCATENATED MODULE: ./src/common/iServer/FieldStatisticsParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class FieldStatisticsParameters
 * @deprecatedclass SuperMap.FieldStatisticsParameters
 * @category iServer Data Field
 * @classdesc 字段统计信息查询参数类。
 * @param {Object} options - 参数。
 * @param {string} options.datasource - 数据源名称。
 * @param {string} options.dataset - 数据集名称。
 * @param {string} options.fieldName - 字段名。
 * @param {(string.<StatisticMode>|Array.<string.<StatisticMode>>)} statisticMode - 字段统计方法类型。
 * @extends {FieldParameters}
 * @usage
 */
class FieldStatisticsParameters extends FieldParameters {


    constructor(options) {
        super(options);
        /**
         * @member {string} FieldStatisticsParameters.prototype.fieldName
         * @description 字段名。
         */
        this.fieldName = null;

        /**
         * @member {(string.<StatisticMode>|Array.<string.<StatisticMode>>)} FieldStatisticsParameters.prototype.statisticMode
         * @description 字段统计方法类型。
         */
        this.statisticMode = null;

        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.FieldStatisticsParameters";
    }

    /**
     * @function FieldStatisticsParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.fieldName = null;
        me.statisticMode = null;
    }

}


;// CONCATENATED MODULE: ./src/common/iServer/FieldStatisticService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class FieldStatisticService
 * @deprecatedclass SuperMap.FieldStatisticService
 * @category iServer Data Field
 * @classdesc 字段查询统计服务类。用来完成对指定数据集指定字段的查询统计分析，即求平均值，最大值等。
 * @extends {CommonServiceBase}
 * @param {string} url - 服务地址。如访问 World Map 服务，只需将 url 设为：http://localhost:8090/iserver/services/data-world/rest/data 即可。
 * @param {Object} options - 参数。
 * @param {DataFormat} [options.format] - 查询结果返回格式，目前支持 iServerJSON 和GeoJSON 两种格式。参数格式为 "ISERVER","GEOJSON"。
 * @param {string} options.datasource - 数据集所在的数据源名称。
 * @param {string} options.dataset - 数据集名称。
 * @param {string} options.field - 查询统计的目标字段名称。
 * @param {StatisticMode} options.statisticMode - 字段查询统计的方法类型。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @example
 * var myService = new FieldStatisticService(url,
 *     datasource: "World",
 *     dataset: "Countries",
 *     field: "SmID",
 *     statisticMode: StatisticMode.AVERAGE
 * };
 * @usage
 */


class FieldStatisticService extends CommonServiceBase {


    constructor(url, options) {
        super(url, options);
        /**
         * @member {string} FieldStatisticService.prototype.datasource
         * @description 数据集所在的数据源名称。
         */
        this.datasource = null;


        /**
         * @member {string} FieldStatisticService.prototype.dataset
         * @description 数据集名称。
         */
        this.dataset = null;

        /**
         * @member {string} FieldStatisticService.prototype.field
         * @description 查询统计的目标字段名称。
         */
        this.field = null;

        /**
         * @member {StatisticMode} FieldStatisticService.prototype.statisticMode
         * @description 字段查询统计的方法类型。
         */
        this.statisticMode = null;
        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.FieldStatisticService";
    }


    /**
     * @function FieldStatisticService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        me.datasource = null;
        me.dataset = null;
        me.field = null;
        me.statisticMode = null;
    }


    /**
     * @function FieldStatisticService.prototype.processAsync
     * @description 执行服务，进行指定字段的查询统计。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(callback) {
        var me = this,
            fieldStatisticURL = "datasources/" + me.datasource + "/datasets/" + me.dataset + "/fields/" + me.field + "/" + me.statisticMode;
        me.url = Util_Util.urlPathAppend(me.url, fieldStatisticURL);

        return me.request({
            method: "GET",
            data: null,
            scope: me,
            success: callback,
            failure: callback
        });
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FindClosestFacilitiesParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class FindClosestFacilitiesParameters
 * @deprecatedclass SuperMap.FindClosestFacilitiesParameters
 * @category iServer NetworkAnalyst ClosestFacility
 * @classdesc 最近设施分析参数类。
 * @param {Object} options - 参数。
 * @param {GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>} options.event - 事件点，一般为需要获得服务设施服务的事件位置。
 * @param {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>}  options.facilities - 设施点集合，一般为提供服务的服务设施位置。
 * @param {number} [options.expectFacilityCount=1] - 要查找的设施点数量。
 * @param {boolean} [options.fromEvent=false] - 是否从事件点到设施点进行查找。
 * @param {boolean} [options.isAnalyzeById=false] - 事件点和设施点是否通过节点 ID 号来指定。
 * @param {number} [options.maxWeight=0] - 权值的最大限值。单位与该类中 parameter 字段（交通网络分析通用参数）中设置的耗费字段一致。
 * @param {TransportationAnalystParameter} [options.parameter] - 交通网络分析通用参数。
 * @usage
 */
class FindClosestFacilitiesParameters {


    constructor(options) {
        /**
         * @member {GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>} FindClosestFacilitiesParameters.prototype.event
         * @description 事件点，一般为需要获得服务设施服务的事件位置。
         *              可以通过两种方式赋予事件点：当该类中字段 isAnalyzeById = true 时，应输入事件点 ID 号；当 isAnalyzeById = false 时，应输入事件点坐标。
         */
        this.event = null;

        /**
         * @member {number} [FindClosestFacilitiesParameters.prototype.expectFacilityCount=1]
         * @description 要查找的设施点数量。
         */
        this.expectFacilityCount = 1;

        /**
         * @member {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>} [FindClosestFacilitiesParameters.prototype.facilities=false]
         * @description 设施点集合，一般为提供服务的服务设施位置。
         *              可以通过两种方式赋予设施点：当该类中字段 isAnalyzeById = true 时，应输入设施点 ID 号；当 isAnalyzeById = false 时，应输入设施点坐标。
         */
        this.facilities = null;

        /**
         * @member {boolean} [FindClosestFacilitiesParameters.prototype.fromEvent=false]
         * @description 是否从事件点到设施点进行查找。最近设施分析主要是通过设施点和事件点之间最优的路线来分析在一定范围内哪个或哪些设施与事件点有最优路线的关系。
         *              这个行走线路是通过网络图层进行网络分析算法计算出来的两点间的最优路线。由于存在从 A 点到 B 点与从 B 点到 A 点的耗费不一样的情况，因此起止点不同可能会得到不同的最优路线。因此在进行最近设施分析之前，需要设置获取的最优路线的方向，即是以事件点作为起点到最近设施点的方向分析，还是以最近设施点为起点到事件点的方向分析。如果需要以事件点作为起点到设施点方向进行查找，设置该字段值为 true；设置为 false，表示从设施点到事件点进行查找。
         */
        this.fromEvent = false;

        /**
         * @member {boolean} [FindClosestFacilitiesParameters.prototype.isAnalyzeById=false]
         * @description 事件点和设施点是否通过节点 ID 号来指定，设置为 false，表示通过坐标点指定事件点和设施点。
         */
        this.isAnalyzeById = false;

        /**
         * @member {number} [FindClosestFacilitiesParameters.prototype.maxWeight=0]
         * @description 权值的最大限值。单位与该类中 parameter 字段（交通网络分析通用参数）中设置的耗费字段一致。
         *              例如事件发生点是一起交通事故，要求查找在 10 分钟内能到达的最近医院，超过 10 分钟能到达的都不予考虑。
         *              那么需要将网络分析参数中 parameter.weightFieldName 设置为表示时间的字段，然后设置查找范围的半径值为10。
         */
        this.maxWeight = 0;

        /**
         * @member {TransportationAnalystParameter} [FindClosestFacilitiesParameters.prototype.parameter]
         * @description 交通网络分析通用参数。通过本类可以设置障碍边、障碍点、权值字段信息的名称标识、转向权值字段等信息。
         *              它为 TransportationAnalystParameter 类型，虽然为可选参数，但是如果不设置其中的 resultSetting 字段，
         *              则返回结果空间信息等都为空。
         */
        this.parameter = new TransportationAnalystParameter();
        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.FindClosestFacilitiesParameters";
    }

    /**
     * @function FindClosestFacilitiesParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.event = null;
        me.expectFacilityCount = null;
        me.facilities = null;
        me.fromEvent = null;
        me.isAnalyzeById = null;
        me.maxWeight = null;
        if (me.parameter) {
            me.parameter.destroy();
            me.parameter = null;
        }
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FindClosestFacilitiesService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class FindClosestFacilitiesService
 * @deprecatedclass SuperMap.FindClosestFacilitiesService
 * @category  iServer NetworkAnalyst ClosestFacility
 * @classdesc 最近设施分析服务类。
 *            最近设施分析是指在网络上给定一个事件点和一组设施点，
 *            查找从事件点到设施点(或从设施点到事件点)以最小耗费能到达的最佳路径。
 *            该类负责将客户端指定的最近设施分析参数传递给服务端，并接收服务端返回的结果数据。
 *            最近设施分析结果通过该类支持的事件的监听函数参数获取
 * @extends {NetworkAnalystServiceBase}
 * @example
 * var myfindClosestFacilitiesService = new FindClosestFacilitiesService(url);
 * @param {string} url - 服务地址。请求网络分析服务，URL应为：
 *                       http://{服务器地址}:{服务端口号}/iserver/services/{网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}；
 *                       例如:"http://localhost:8090/iserver/services/components-rest/rest/networkanalyst/RoadNet@Changchun"。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class FindClosestFacilitiesService extends NetworkAnalystServiceBase {
    constructor(url, options) {
        super(url, options);

        this.CLASS_NAME = "SuperMap.FindClosestFacilitiesService";
    }

    /**
     * @function FindClosestFacilitiesService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function FindClosestFacilitiesService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {FindClosestFacilitiesParameters} params - 最近设施分析服务参数类
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof FindClosestFacilitiesParameters)) {
            return;
        }
        var me = this,
            jsonObject;
        me.url = Util_Util.urlPathAppend(me.url, 'closestfacility');
        jsonObject = {
            expectFacilityCount: params.expectFacilityCount,
            fromEvent: params.fromEvent,
            maxWeight: params.maxWeight,
            parameter: Util_Util.toJSON(params.parameter),
            event: Util_Util.toJSON(params.event),
            facilities: me.getJson(params.isAnalyzeById, params.facilities)
        };
        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }

    /**
     * @function FindClosestFacilitiesService.prototype.getJson
     * @description 将对象转化为JSON字符串。
     * @param {boolean} isAnalyzeById - 是否通过ID来分析
     * @param {Array.<Object>} params - 分析参数数组
     * @returns {Object} 转化后的JSON字符串。
     */
    getJson(isAnalyzeById, params) {
        var jsonString = "[",
            len = params ? params.length : 0;

        if (isAnalyzeById === false) {
            for (let i = 0; i < len; i++) {
                if (i > 0) {
                    jsonString += ",";
                }
                jsonString += '{"x":' + params[i].x + ',"y":' + params[i].y + '}';
            }
        } else if (isAnalyzeById === true) {
            for (let i = 0; i < len; i++) {
                if (i > 0) {
                    jsonString += ",";
                }
                jsonString += params[i];
            }
        }
        jsonString += ']';
        return jsonString;
    }

    /**
     * @function FindClosestFacilitiesService.prototype.toGeoJSONResult
     * @description 将含有 geometry 的数据转换为 GeoJSON 格式。
     * @param {Object} result - 服务器返回的结果对象。
     */
    toGeoJSONResult(result) {
        if (!result || !result.facilityPathList) {
            return result;
        }

        var geoJSONFormat = new GeoJSON();
        result.facilityPathList.map(function (path) {
            if (path.route) {
                path.route = geoJSONFormat.toGeoJSON(path.route);
            }
            if (path.pathGuideItems) {
                path.pathGuideItems = geoJSONFormat.toGeoJSON(path.pathGuideItems);

            }
            if (path.edgeFeatures) {
                path.edgeFeatures = geoJSONFormat.toGeoJSON(path.edgeFeatures);
            }
            if (path.nodeFeatures) {
                path.nodeFeatures = geoJSONFormat.toGeoJSON(path.nodeFeatures);
            }
            return path;
        });
        return result;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FindLocationParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class FindLocationParameters
 * @deprecatedclass SuperMap.FindLocationParameters
 * @category iServer NetworkAnalyst Location
 * @classdesc 选址分区分析参数类。
 * @param {Object} options - 参数。
 * @param {string} options.turnWeightField - 转向权值字段的名称。
 * @param {string} options.weightName - 阻力字段的名称，标识了进行网络分析时所使用的阻力字段。
 * @param {Array.<SupplyCenter>} options.supplyCenters - 资源供给中心集合。
 * @param {number} [options.expectedSupplyCenterCount=1] - 期望用于最终设施选址的资源供给中心数量。
 * @param {boolean} [options.isFromCenter=false] - 是否从中心点开始分配资源。
 * @usage
 */
class FindLocationParameters {


    constructor(options) {
        /**
         * @member {number} [FindLocationParameters.prototype.expectedSupplyCenterCount=1]
         * @description 期望用于最终设施选址的资源供给中心数量。
         *              当输入值为 0 时，最终设施选址的资源供给中心数量默认为覆盖分析区域内的所需最少的供给中心数。
         */
        this.expectedSupplyCenterCount = null;

        /**
         * @member {boolean} [FindLocationParameters.prototype.isFromCenter=false]
         * @description 是否从中心点开始分配资源。
         *              由于网路数据中的弧段具有正反阻力，即弧段的正向阻力值与其反向阻力值可能不同，
         *              因此，在进行分析时，从资源供给中心开始分配资源到需求点与从需求点向资源供给中心分配这两种分配形式下，所得的分析结果会不同。
         */
        this.isFromCenter = false;


        /**
         * @member {Array.<SupplyCenter>} FindLocationParameters.prototype.supplyCenters
         * @description 资源供给中心集合。
         *              资源供给中心是提供资源和服务的设施，对应于网络结点，
         *              资源供给中心的相关信息包括资源量、最大阻力值、资源供给中心类型，资源供给中心在网络中所处结点的 ID 等，以便在进行选址分区分析时使用。
         */
        this.supplyCenters = null;

        /**
         * @member {string} FindLocationParameters.prototype.turnWeightField
         * @description 转向权值字段的名称。
         */
        this.turnWeightField = null;

        /**
         * @member {string} FindLocationParameters.prototype.weightName
         * @description 阻力字段的名称，标识了进行网络分析时所使用的阻力字段。
         */
        this.weightName = null;
        Util_Util.extend(this, options);
        this.CLASS_NAME = "SuperMap.FindLocationParameters";
    }

    /**
     * @function FindLocationParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.expectedSupplyCenterCount = null;
        me.isFromCenter = null;
        me.turnWeightField = null;
        me.weightName = null;
        if (me.supplyCenters) {
            for (var i = 0, supplyCenters = me.supplyCenters, len = supplyCenters.length; i < len; i++) {
                supplyCenters[i].destroy();
            }
            me.supplyCenters = null;
        }
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/FindLocationService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class FindLocationService
 * @deprecatedclass SuperMap.FindLocationService
 * @category iServer NetworkAnalyst Location
 * @classdesc 选址分区分析服务类。
 *            选址分区分析是为了确定一个或多个待建设施的最佳或最优位置，使得设施可以用一种最经济有效的方式为需求方提供服务或者商品。
 *            选址分区不仅仅是一个选址过程，还要将需求点的需求分配到相应的新建设施的服务区中，因此称之为选址与分区。
 *            选址分区分析结果通过该类支持的事件的监听函数参数获取
 * @extends {NetworkAnalystServiceBase}
 * @example
 * (start code)
 * var findLocationService = new FindLocationService(url);
 * (end)
 * @param {string} url - 服务地址。
 *                       如 http://localhost:8090/iserver/services/transportationanalyst-sample/rest/networkanalyst/RoadNet@Changchun 。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class FindLocationService extends NetworkAnalystServiceBase {

    constructor(url, options) {
        super(url, options);

        this.CLASS_NAME = "SuperMap.FindLocationService";
    }

    /**
     * @function FindLocationService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function FindLocationService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {FindLocationParameters} params - 选址分区分析服务参数类
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof FindLocationParameters)) {
            return;
        }
        var me = this,
            jsonObject;
        me.url = Util_Util.urlPathAppend(me.url, 'location');
        jsonObject = {
            isFromCenter: params.isFromCenter,
            expectedSupplyCenterCount: params.expectedSupplyCenterCount,
            weightName: params.weightName,
            turnWeightField: params.turnWeightField,
            returnEdgeFeature: true,
            returnEdgeGeometry: true,
            returnNodeFeature: true,
            mapParameter: Util_Util.toJSON(params.mapParameter),
            supplyCenters: me.getCentersJson(params.supplyCenters)
        };
        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }

    /**
     * @function FindLocationService.prototype.getCentersJson
     * @description 将数组对象转化为JSON字符串。
     * @param {Array} params - 需要转换的参数
     * @returns {string} 转化后的JSON字符串。
     */
    getCentersJson(params) {
        var json = "[",
            len = params ? params.length : 0;
        for (var i = 0; i < len; i++) {
            if (i > 0) {
                json += ",";
            }
            json += Util_Util.toJSON(params[i]);
        }
        json += "]";
        return json;
    }

    /**
     * @function FindLocationService.prototype.toGeoJSONResult
     * @description 将含有 geometry 的数据转换为 GeoJSON 格式。
     * @param {Object} result - 服务器返回的结果对象。
     */
    toGeoJSONResult(result) {
        if (!result) {
            return null;
        }
        var geoJSONFormat = new GeoJSON();
        if (result.demandResults) {
            result.demandResults = geoJSONFormat.toGeoJSON(result.demandResults);
        }
        if (result.supplyResults) {
            result.supplyResults = geoJSONFormat.toGeoJSON(result.supplyResults);
        }

        return result;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FindMTSPPathsParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class FindMTSPPathsParameters
 * @deprecatedclass SuperMap.FindMTSPPathsParameters
 * @category iServer NetworkAnalyst MTSPPath
 * @classdesc 多旅行商分析参数类。
 * @param {Object} options - 参数。
 * @param {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>} options.centers - 配送中心集合。
 * @param {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>} options.nodes - 配送目标集合。
 * @param {boolean} [options.hasLeastTotalCost=false] - 配送模式是否为总花费最小方案。
 * @param {boolean} [options.isAnalyzeById=false] - 是否通过节点 ID 号来指定配送中心点和配送目的点，即通过坐标点指定。
 * @param {TransportationAnalystParameter} [options.parameter] - 交通网络分析通用参数。
 * @usage
 */
class FindMTSPPathsParameters {


    constructor(options) {
        /**
         * @member {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>} FindMTSPPathsParameters.prototype.centers
         * @description 配送中心集合。
         *              当 FindMTSPPathsParameters.isAnalyzeById = false 时，centers 应为点的坐标数组；
         *              当 FindMTSPPathsParameters.isAnalyzeById = true 时，centers 应为点的 ID 数组。
         */
        this.centers = null;

        /**
         * @member {boolean} [FindMTSPPathsParameters.prototype.hasLeastTotalCost=false]
         * @description 配送模式是否为总花费最小方案。
         *              若为 true，则按照总花费最小的模式进行配送，此时可能会出现某几个配送中心点配送的花费较多而其他配送中心点的花费很少的情况。
         *              若为 false，则为局部最优，此方案会控制每个配送中心点的花费，使各个中心点花费相对平均，此时总花费不一定最小。
         */
        this.hasLeastTotalCost = false;

        /**
         * @member {boolean} [FindMTSPPathsParameters.prototype.isAnalyzeById=false]
         * @description 是否通过节点 ID 号来指定配送中心点和配送目的点，即通过坐标点指定。
         */
        this.isAnalyzeById = false;

        /**
         * @member {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>} FindMTSPPathsParameters.prototype.nodes
         * @description 配送目标集合。
         *              当 FindMTSPPathsParameters.isAnalyzeById = false 时，nodes 应为点的坐标数组；
         *              当 FindMTSPPathsParameters.isAnalyzeById = true 时，nodes 应为点的 ID 数组。
         */
        this.nodes = null;

        /**
         * @member {TransportationAnalystParameter} [FindMTSPPathsParameters.prototype.parameter]
         * @description 交通网络分析通用参数。
         *              通过本类可以设置障碍边、障碍点、权值字段信息的名称标识、转向权值字段等信息。
         *              TransportationAnalystParameter 类型，它虽然为可选参数，但是如果不设置其中的 resultSetting 字段，则返回结果空间信息等都为空。
         */
        this.parameter = new TransportationAnalystParameter();
        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.FindMTSPPathsParameters";
    }

    /**
     * @function FindMTSPPathsParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.centers = null;
        me.hasLeastTotalCost = null;
        me.isAnalyzeById = null;
        me.nodes = null;
        me.maxWeight = null;
        if (me.parameter) {
            me.parameter.destroy();
            me.parameter = null;
        }
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FindMTSPPathsService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class FindMTSPPathsService
 * @deprecatedclass SuperMap.FindMTSPPathsService
 * @category  iServer NetworkAnalyst MTSPPath
 * @classdesc 多旅行商分析服务类
 *            多旅行商分析也称为物流配送，是指在网络数据集中，给定 M 个配送中心点和 N 个配送目的地（M，N 为大于零的整数）。
 *            查找经济有效的配送路径，并给出相应的行走路线。
 *            物流配送功能就是解决如何合理分配配送次序和送货路线，使配送总花费达到最小或每个配送中心的花费达到最小。
 *            该类负责将客户端指定的多旅行商分析参数传递给服务端，并接收服务端返回的结果数据。
 *            多旅行商分析结果通过该类支持的事件的监听函数参数获取。
 * @extends {NetworkAnalystServiceBase}
 * @example
 * var myFindMTSPPathsService = new FindMTSPPathsService(url);
 * @param {string} url - 服务地址。请求网络分析服务，URL应为：
 *                       http://{服务器地址}:{服务端口号}/iserver/services/网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}；
 *                       例如:"http://localhost:8090/iserver/services/components-rest/rest/networkanalyst/RoadNet@Changchun"。
 * @param {Object} options - 互服务时所需可选参数。如：
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class FindMTSPPathsService extends NetworkAnalystServiceBase {

    constructor(url, options) {
        super(url, options);

        this.CLASS_NAME = "SuperMap.FindMTSPPathsService";
    }

    /**
     * @function FindMTSPPathsService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function FindMTSPPathsService..prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {FindMTSPPathsParameters} params - 多旅行商分析服务参数类
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof FindMTSPPathsParameters)) {
            return;
        }
        var me = this,
            jsonObject,
            //end = me.url.substr(me.url.length - 1, 1),
            centers = me.getJson(params.isAnalyzeById, params.centers),
            nodes = me.getJson(params.isAnalyzeById, params.nodes);
        me.url = Util_Util.urlPathAppend(me.url, 'mtsppath');
        jsonObject = {
            centers: centers,
            nodes: nodes,
            parameter: Util_Util.toJSON(params.parameter),
            hasLeastTotalCost: params.hasLeastTotalCost
        };
        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }

    /**
     * @function FindMTSPPathsService.prototype.getJson
     * @description 将对象转化为JSON字符串。
     * @param {boolean} isAnalyzeById - 是否通过id分析
     * @param {Array} params - 需要转换的数字
     * @returns {Object} 转化后的JSON字符串。
     */
    getJson(isAnalyzeById, params) {
        var jsonString = "[",
            len = params ? params.length : 0;

        if (isAnalyzeById === false) {
            for (let i = 0; i < len; i++) {
                if (i > 0) {
                    jsonString += ",";
                }
                jsonString += '{"x":' + params[i].x + ',"y":' + params[i].y + '}';
            }
        } else if (isAnalyzeById === true) {
            for (let i = 0; i < len; i++) {
                if (i > 0) {
                    jsonString += ",";
                }
                jsonString += params[i];
            }
        }
        jsonString += ']';
        return jsonString;
    }

    /**
     * @function FindMTSPPathsService.prototype.toGeoJSONResult
     * @description 将含有 geometry 的数据转换为 GeoJSON 格式。
     * @param {Object} result - 服务器返回的结果对象。
     */
    toGeoJSONResult(result) {
        if (!result || !result.pathList) {
            return null;
        }
        var geoJSONFormat = new GeoJSON();
        result.pathList.map(function (path) {
            if (path.route) {
                path.route = geoJSONFormat.toGeoJSON(path.route);
            }
            if (path.pathGuideItems) {
                path.pathGuideItems = geoJSONFormat.toGeoJSON(path.pathGuideItems);
            }
            if (path.edgeFeatures) {
                path.edgeFeatures = geoJSONFormat.toGeoJSON(path.edgeFeatures);
            }
            if (path.nodeFeatures) {
                path.nodeFeatures = geoJSONFormat.toGeoJSON(path.nodeFeatures);
            }
            return path;
        });
        return result;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FindPathParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class FindPathParameters
 * @deprecatedclass SuperMap.FindPathParameters
 * @category  iServer NetworkAnalyst Path
 * @classdesc 最佳路径分析参数类。最佳路径是在网络数据集中指定一些结点，按照顺序访问结点从而求解起止点之间阻抗最小的路径。
 * 例如如果要顺序访问 1、2、3、4 四个结点，则需要分别找到1、2结点间的最佳路径 R1—2，2、3 间的最佳路径 R2—3 和 3、4 结点间的最佳路径 R3—4，
 * 顺序访问 1、2、3、4 四个结点的最佳路径就是 R = R1—2 + R2—3 + R3—4。
 * 阻抗就是指从一点到另一点的耗费，在实际应用中我们可以将距离、时间、花费等作为阻抗条件。
 * 阻抗最小也就可以理解为从一点到另一点距离最短、时间最少、花费最低等。当两点间距离最短时为最短路径，它是最佳路径问题的一个特例。
 * 阻抗值通过 {@link TransportationAnalystParameter#weightFieldName}设置。
 * 计算最佳路径除了受阻抗影响外，还受转向字段的影响。转向值通过 {@link TransportationAnalystParameter#turnWeightField} 设置。
 *
 * @param {Object} options - 参数。
 * @param {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>} options.nodes - 最佳路径分析经过的结点或设施点数组。该字段至少包含两个点。
 * @param {boolean} [options.isAnalyzeById=false] - 是否通过节点 ID 指定路径分析的结点。
 * @param {boolean} [options.hasLeastEdgeCount=false] - 是否按照弧段数最少的进行最佳路径分析。
 * @param {TransportationAnalystParameter} [options.parameter] - 交通网络分析通用参数。
 * @usage
 */
class FindPathParameters {

    constructor(options) {

        /**
         * @member {boolean} [FindPathParameters.prototype.isAnalyzeById=false]
         * @description 是否通过节点 ID 指定路径分析的结点。
         *              指定路径分析经过的结点或设施点有两种方式：输入结点 ID 号或直接输入点坐标。
         *              当该字段为 true 时，表示通过结点 ID 指定途经点，即 FindPathParameters.nodes = [ID1,ID2,...]；
         *              反之表示通过结点坐标指定途经点，即 FindPathParameters.nodes = [{x1,y1},{x2,y2},...] 。
         */
        this.isAnalyzeById = false;

        /**
         * @member {boolean} [FindPathParameters.prototype.hasLeastEdgeCount=false]
         * @description 是否按照弧段数最少的进行最佳路径分析。
         *              true 表示按照弧段数最少进行分析，返回弧段数最少的路径中一个阻抗最小的最佳路径；
         *              false 表示直接返回阻抗最小的路径，而不考虑弧段的多少。
         */
        this.hasLeastEdgeCount = null;

        /**
         * @member {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>} FindPathParameters.prototype.nodes
         * @description 最佳路径分析经过的结点或设施点数组，必设字段。该字段至少包含两个点。
         *              当 FindPathParameters.isAnalyzeById = false 时，nodes 应为点的坐标数组；
         *              当 FindPathParameters.isAnalyzeById = true 时，nodes 应为点的 ID 数组。
         */
        this.nodes = null;

        /**
         * @member {TransportationAnalystParameter} FindPathParameters.prototype.parameter
         * @description 交通网络分析通用参数。
         */
        this.parameter = new TransportationAnalystParameter();
        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.FindPathParameters";
    }


    /**
     * @function FindPathParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.isAnalyzeById = null;
        me.hasLeastEdgeCount = null;
        me.nodes = null;
        if (me.parameter) {
            me.parameter.destroy();
            me.parameter = null;
        }
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FindPathService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class FindPathService
 * @deprecatedclass SuperMap.FindPathService
 * @category iServer NetworkAnalyst Path
 * @classdesc 最佳路径分析服务类。
 *            最佳路径是在网络数据集中指定一些节点，按照节点的选择顺序，
 *            顺序访问这些节点从而求解起止点之间阻抗最小的路经。
 *            该类负责将客户端指定的最佳路径分析参数传递给服务端，并接收服务端返回的结果数据。
 *            最佳路径分析结果通过该类支持的事件的监听函数参数获取
 * @extends {NetworkAnalystServiceBase}
 * @example
 * var myFindPathService = new FindPathService(url);
 * @param {string} url - 服务地址。请求网络分析服务，URL应为：
 *                       http://{服务器地址}:{服务端口号}/iserver/services/{网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}；
 *                       例如:"http://localhost:8090/iserver/services/components-rest/rest/networkanalyst/RoadNet@Changchun"。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class FindPathService extends NetworkAnalystServiceBase {

    constructor(url, options) {
        super(url, options);

        this.CLASS_NAME = "SuperMap.FindPathService";
    }

    /**
     * @function FindPathService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function FindPathService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {FindPathParameters} params - 最佳路径分析服务参数类
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof FindPathParameters)) {
            return;
        }
        var me = this, jsonObject;
        me.url = Util_Util.urlPathAppend(me.url, 'path');
        jsonObject = {
            hasLeastEdgeCount: params.hasLeastEdgeCount,
            parameter: Util_Util.toJSON(params.parameter),
            nodes: me.getJson(params.isAnalyzeById, params.nodes)
        };
        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }

    /**
     * @function FindPathService.prototype.getJson
     * @description 将对象转化为JSON字符串。
     * @param {boolean} isAnalyzeById - 是否通过id分析
     * @param {Array} params - 需要转换的数字
     * @returns {Object} 转化后的JSON字符串。
     */
    getJson(isAnalyzeById, params) {
        var jsonString = "[",
            len = params ? params.length : 0;

        if (isAnalyzeById === false) {
            for (let i = 0; i < len; i++) {
                if (i > 0) {
                    jsonString += ",";
                }
                jsonString += '{"x":' + params[i].x + ',"y":' + params[i].y + '}';
            }
        } else if (isAnalyzeById === true) {
            for (let i = 0; i < len; i++) {
                if (i > 0) {
                    jsonString += ",";
                }
                jsonString += params[i];
            }
        }
        jsonString += ']';
        return jsonString;
    }

    /**
     * @function FindMTSPPathsService.prototype.toGeoJSONResult
     * @description 将含有 geometry 的数据转换为 GeoJSON 格式。
     * @param {Object} result - 服务器返回的结果对象。
     */
    toGeoJSONResult(result) {
        if (!result || !result.pathList || result.pathList.length < 1) {
            return null;
        }
        var geoJSONFormat = new GeoJSON();
        result.pathList.forEach(function (path) {
            if (path.route) {
                path.route = geoJSONFormat.toGeoJSON(path.route);
            }
            if (path.pathGuideItems) {
                path.pathGuideItems = geoJSONFormat.toGeoJSON(path.pathGuideItems);

            }
            if (path.edgeFeatures) {
                path.edgeFeatures = geoJSONFormat.toGeoJSON(path.edgeFeatures);
            }
            if (path.nodeFeatures) {
                path.nodeFeatures = geoJSONFormat.toGeoJSON(path.nodeFeatures);
            }
        });
        return result;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FindServiceAreasParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class FindServiceAreasParameters
 * @deprecatedclass SuperMap.FindServiceAreasParameters
 * @category iServer NetworkAnalyst ServiceArea
 * @classdesc 服务区分析参数类。
 *            服务区分析是以指定服务站点为中心，在一定服务范围内查找网络上服务站点能够提供服务的区域范围。
 *            例如：计算某快餐店能够在30分钟内送达快餐的区域。
 * @param {Object} options - 参数。
 * @param {Array.<number>} options.weights - 每个服务站点提供服务的阻力半径，超过这个阻力半径的区域不予考虑，其单位与阻力字段一致。
 * @param {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>} options.centers - 服务站点数组。
 * @param {boolean} [options.isAnalyzeById=false] - 是否通过节点 ID 指定路径分析的结点。
 * @param {boolean} [options.isCenterMutuallyExclusive=false] - 是否中心点互斥。
 * @param {boolean} [options.isFromCenter=false] - 是否从中心点开始分析。
 * @param {TransportationAnalystParameter} [options.parameter] - 交通网络分析通用参数。
 * @usage
 */
class FindServiceAreasParameters {


    constructor(options) {
        /**
         * @member {boolean} [FindServiceAreasParameters.prototype.isAnalyzeById=false]
         * @description 是否通过节点 ID 指定路径分析的结点。
         *              指定路径分析经过的结点或设施点有两种方式：输入结点 ID 号或直接输入点坐标。
         *              当该字段为 true 时，表示通过结点 ID 指定途经点，即 FindServiceAreasParameters.centers = [ID1,ID2,...]；
         *              反之表示通过结点坐标指定途经点，即 FindServiceAreasParameters.centers = [{x1,y1},{x2,y2},...]。
         */
        this.isAnalyzeById = false;

        /**
         * @member {boolean} [FindServiceAreasParameters.prototype.isCenterMutuallyExclusive=false]
         * @description 是否中心点互斥，即按照中心点的距离进行判断是否要进行互斥处理。
         *              若分析出的服务区有重叠的部分，则通过设置该参数进行互斥处理。
         */
        this.isCenterMutuallyExclusive = false;

        /**
         * @member {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>} FindServiceAreasParameters.prototype.centers
         * @description 服务站点数组。
         *              当该类的 iSAnalyzeById = true 时，通过结点 ID 号指定服务站点；当 iSAnalyzeById = false 时，通过点坐标指定服务站点。
         */
        this.centers = null;

        /**
         * @member {boolean} [FindServiceAreasParameters.prototype.isFromCenter=false]
         * @description 是否从中心点开始分析。
         *              从中心点开始分析和不从中心点开始分析，体现了服务中心和需要该服务的需求地的关系模式。
         *              从中心点开始分析，是一个服务中心向服务需求地提供服务；
         *              而不从中心点开始分析，是一个服务需求地主动到服务中心获得服务。
         */
        this.isFromCenter = false;

        /**
         * APIProperty: weights
         * @member {Array.<number>} FindServiceAreasParameters.prototype.weights
         * @description 每个服务站点提供服务的阻力半径，即超过这个阻力半径的区域不予考虑，其单位与阻力字段一致。
         *              该字段为一个数组，数组长度跟服务中心个数一致，按照索引顺序与站点一一对应，每个元素表示了在对每个服务中心进行服务区分析时，所用的服务半径。
         */
        this.weights = null;

        /**
         * @member {TransportationAnalystParameter} FindServiceAreasParameters.prototype.parameter
         * @description 交通网络分析通用参数。
         */
        this.parameter = new TransportationAnalystParameter();

        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.FindServiceAreasParameters";
    }

    /**
     * @function FindServiceAreasParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.isAnalyzeById = null;
        me.isCenterMutuallyExclusive = null;
        me.centers = null;
        me.isFromCenter = null;
        me.weights = null;
        if (me.parameter) {
            me.parameter.destroy();
            me.parameter = null;
        }
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FindServiceAreasService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class FindServiceAreasService
 * @deprecatedclass SuperMap.FindServiceAreasService
 * @category iServer NetworkAnalyst ServiceArea
 * @classdesc 服务区分析服务类。
 *            服务区分析是以指定服务站点为中心，
 *            在一定服务范围内查找网络上服务站点能够提供服务的区域范围。
 *            该类负责将客户端指定的服务区分析参数传递给服务端，并接收服务端返回的结果数据。
 *            服务区分析结果通过该类支持的事件的监听函数参数获取
 * @extends {NetworkAnalystServiceBase}
 * @example
 * var myFindServiceAreasService = new FindServiceAreasService(url);
 * @param {string} url - 服务地址。请求网络分析服务，URL应为：
 *                       http://{服务器地址}:{服务端口号}/iserver/services/{网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}；
 *                       例如:"http://localhost:8090/iserver/services/components-rest/rest/networkanalyst/RoadNet@Changchun"。
 * @param {Object} options - 互服务时所需可选参数。如：
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class FindServiceAreasService extends NetworkAnalystServiceBase {

    constructor(url, options) {
        super(url, options);

        this.CLASS_NAME = "SuperMap.FindServiceAreasService";
    }

    /**
     * @function FindServiceAreasService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function FindServiceAreasService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {FindServiceAreasParameters} params - 服务区分析服务参数类
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof FindServiceAreasParameters)) {
            return;
        }
        var me = this, jsonObject;
        me.url = Util_Util.urlPathAppend(me.url, 'servicearea');
        jsonObject = {
            isFromCenter: params.isFromCenter,
            isCenterMutuallyExclusive: params.isCenterMutuallyExclusive,
            parameter: Util_Util.toJSON(params.parameter),
            centers: me.getJson(params.isAnalyzeById, params.centers),
            weights: me.getJson(true, params.weights)
        };
        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }

    /**
     * @function FindServiceAreasService.prototype.getJson
     * @description 将对象转化为JSON字符串。
     * @param {boolean} isAnalyzeById - 是否通过id分析
     * @param {Array} params - 需要转换的数字
     * @returns {Object} 转化后的JSON字符串。
     */
    getJson(isAnalyzeById, params) {
        var jsonString = "[",
            len = params ? params.length : 0;

        if (isAnalyzeById === false) {
            for (let i = 0; i < len; i++) {
                if (i > 0) {
                    jsonString += ",";
                }
                jsonString += '{"x":' + params[i].x + ',"y":' + params[i].y + '}';
            }
        } else if (isAnalyzeById === true) {
            for (let i = 0; i < len; i++) {
                if (i > 0) {
                    jsonString += ",";
                }
                jsonString += params[i];
            }
        }
        jsonString += ']';
        return jsonString;
    }

    /**
     * @function FindServiceAreasService.prototype.toGeoJSONResult
     * @description 将含有 geometry 的数据转换为 GeoJSON 格式。
     * @param {Object} result - 服务器返回的结果对象。
     */
    toGeoJSONResult(result) {
        if (!result || !result.serviceAreaList) {
            return result;
        }
        var geoJSONFormat = new GeoJSON();
        result.serviceAreaList.map(function (serviceArea) {
            if (serviceArea.serviceRegion) {
                serviceArea.serviceRegion = geoJSONFormat.toGeoJSON(serviceArea.serviceRegion);
            }
            if (serviceArea.edgeFeatures) {
                serviceArea.edgeFeatures = geoJSONFormat.toGeoJSON(serviceArea.edgeFeatures);
            }
            if (serviceArea.nodeFeatures) {
                serviceArea.nodeFeatures = geoJSONFormat.toGeoJSON(serviceArea.nodeFeatures);
            }
            if (serviceArea.routes) {
                serviceArea.routes = geoJSONFormat.toGeoJSON(serviceArea.routes);
            }
            return serviceArea;
        });

        return result;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FindTSPPathsParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class FindTSPPathsParameters
 * @deprecatedclass SuperMap.FindTSPPathsParameters
 * @category iServer NetworkAnalyst TSPPath
 * @classdesc 旅行商分析参数类。
 * 旅行商分析是路径分析的一种，它从起点开始（默认为用户指定的第一点）查找能够遍历所有途经点且花费最小的路径。
 * 旅行商分析也可以指定到达的终点，这时查找从起点能够遍历所有途经点最后到达终点，且花费最小的路径。
 * 旅行商分析和最佳路径分析都是在网络中寻找遍历所有站点的最经济的路径，区别是在遍历网络所有站点的过程中对结点访问顺序不同。
 * 最佳路径分析必须按照指定顺序对站点进行访问，而旅行商分析是无序的路径分析。
 * @param {Object} options - 参数。
 * @param {boolean} [options.endNodeAssigned=false] - 是否指定终止点，将指定的途经点的最后一个点作为终止点。true 表示指定终止点，则旅行商必须最后一个访问终止点。
 * @param {boolean} [options.isAnalyzeById=false] - 是否通过节点 ID 号来指定配送中心点和配送目的点。
 * @param {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>} options.nodes - 配送目标集合。
 * @param {TransportationAnalystParameter} [options.parameter] - 交通网络分析通用参数。
 * @usage
 */
class FindTSPPathsParameters {


    constructor(options) {
        /**
         * @member {boolean} [FindTSPPathsParameters.prototype.endNodeAssigned=false]
         * @description 是否指定终止点，将指定的途经点的最后一个点作为终止点。
         *              true 表示指定终止点，则旅行商必须最后一个访问终止点。
         */
        this.endNodeAssigned = false;

        /**
         * @member {boolean} [FindTSPPathsParameters.prototype.isAnalyzeById=false]
         * @description 是否通过节点 ID 号来指定途经点。
         */
        this.isAnalyzeById = false;

        /**
         * @member {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|mapboxgl.Point|Array.<number>>} FindTSPPathsParameters.prototype.nodes
         * @description 旅行商分析途经点数组。
         *              当 FindTSPPathsParameters.isAnalyzeById = false 时，nodes 应为点的坐标数组；
         *              当 FindTSPPathsParameters.isAnalyzeById = true 时，nodes 应为点的 ID 数组。
         */
        this.nodes = null;

        /**
         * @member {TransportationAnalystParameter} [FindTSPPathsParameters.prototype.parameter]
         * @description 交通网络分析通用参数。通过本类可以设置障碍边、障碍点、权值字段信息的名称标识、转向权值字段等信息。
         *              TransportationAnalystParameter 类型，它虽然为可选参数，但是如果不设置其中的 resultSetting
         *              字段，则返回结果空间信息等都为空。
         */
        this.parameter = new TransportationAnalystParameter();
        Util_Util.extend(this, options);

        this.CLASS_NAME = "SuperMap.FindTSPPathsParameters";
    }

    /**
     * @function FindTSPPathsParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.endNodeAssigned = null;
        me.isAnalyzeById = null;
        me.nodes = null;
        if (me.parameter) {
            me.parameter.destroy();
            me.parameter = null;
        }
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/FindTSPPathsService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class FindTSPPathsService
 * @deprecatedclass SuperMap.FindTSPPathsService
 * @category  iServer NetworkAnalyst TSPPath
 * @classdesc 旅行商分析服务类
 *            旅行商分析是路径分析的一种，它从起点开始（默认为用户指定的第一点）查找能够遍历所有途经点且花费最小的路径。
 *            旅行商分析也可以指定到达的终点，这时查找从起点能够遍历所有途经点最后到达终点，且花费最小的路径。
 *            该类负责将客户端指定的旅行商分析参数传递给服务端，并接收服务端返回的结果数据。
 *            旅行商分析结果通过该类支持的事件的监听函数参数获取
 * @extends {NetworkAnalystServiceBase}
 * @example
 * (start code)
 * var myFindTSPPathsService = new FindTSPPathsService(url);
 * (end)
 * @param {string} url - 网络分析服务地址。请求网络分析服务，URL应为：
 *                       http://{服务器地址}:{服务端口号}/iserver/services/{网络分析服务名}/rest/networkanalyst/{网络数据集@数据源}；
 *                       例如:"http://localhost:8090/iserver/services/components-rest/rest/networkanalyst/RoadNet@Changchun"。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class FindTSPPathsService extends NetworkAnalystServiceBase {

    constructor(url, options) {
        super(url, options);

        this.CLASS_NAME = "SuperMap.FindTSPPathsService";
    }

    /**
     * @function FindTSPPathsService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function FindTSPPathsService.prototype.processAsync
     * @description 负责将客户端的查询参数传递到服务端。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @param {FindTSPPathsParameters} params - 旅行商分析服务参数类。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof FindTSPPathsParameters)) {
            return;
        }
        var me = this, jsonObject;
        me.url = Util_Util.urlPathAppend(me.url, 'tsppath');
        jsonObject = {
            parameter: Util_Util.toJSON(params.parameter),
            endNodeAssigned: params.endNodeAssigned,
            nodes: me.getNodesJson(params)
        };
        return me.request({
            method: "GET",
            params: jsonObject,
            scope: me,
            success: callback,
            failure: callback
        });
    }

    /**
     * @function FindTSPPathsService.prototype.getNodesJson
     * @description 将节点对象转化为JSON字符串。
     * @param {FindTSPPathsParameters} params - 旅行商分析服务参数类。
     * @returns {string} 转化后的JSON字符串。
     */
    getNodesJson(params) {
        var jsonParameters = "", nodesString, i, len, nodes;
        if (params.isAnalyzeById === false) {
            for (nodesString = "[", i = 0, nodes = params.nodes, len = nodes.length; i < len; i++) {
                if (i > 0) {
                    nodesString += ",";
                }
                nodesString += '{"x":' + nodes[i].x + ',"y":' + nodes[i].y + '}';
            }
            nodesString += ']';
            jsonParameters += nodesString;
        } else if (params.isAnalyzeById === true) {
            let nodeIDsString = "[", nodes = params.nodes, len = nodes.length;
            for (let i = 0; i < len; i++) {
                if (i > 0) {
                    nodeIDsString += ",";
                }
                nodeIDsString += nodes[i];
            }
            nodeIDsString += ']';
            jsonParameters += nodeIDsString;
        }
        return jsonParameters;
    }

    /**
     * @function FindTSPPathsService.prototype.toGeoJSONResult
     * @description 将含有 geometry 的数据转换为 GeoJSON 格式。
     * @param {Object} result - 服务器返回的结果对象。
     */
    toGeoJSONResult(result) {
        if (!result || !result.tspPathList) {
            return null;
        }
        var geoJSONFormat = new GeoJSON();
        result.tspPathList.forEach(function (path) {
            if (path.route) {
                path.route = geoJSONFormat.toGeoJSON(path.route);
            }
            if (path.pathGuideItems) {
                path.pathGuideItems = geoJSONFormat.toGeoJSON(path.pathGuideItems);
            }
            if (path.edgeFeatures) {
                path.edgeFeatures = geoJSONFormat.toGeoJSON(path.edgeFeatures);
            }
            if (path.nodeFeatures) {
                path.nodeFeatures = geoJSONFormat.toGeoJSON(path.nodeFeatures);
            }
        });
        return result;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/GenerateSpatialDataParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class GenerateSpatialDataParameters
 * @deprecatedclass SuperMap.GenerateSpatialDataParameters
 * @category iServer SpatialAnalyst GenerateSpatialData
 * @classdesc 动态分段操作参数类。通过该类可以为动态分段提供参数信息。
 * @param {Object} options - 参数。
 * @param {string} options.routeTable - 路由数据集。
 * @param {string} options.routeIDField - 路由数据集的标识字段。
 * @param {string} options.eventTable - 用于生成空间数据的事件表名。
 * @param {DataReturnOption} options.dataReturnOption - 设置数据返回选项。
 * @param {string} [options.attributeFilter] - 属性过滤条件。
 * @param {string} options.eventRouteIDField - 用于生成空间数据的事件表的路由标识字段。
 * @param {string} [options.measureField] - 用于生成空间数据的事件表的刻度字段，只有当事件为点事件的时候该属性才有意义。
 * @param {string} [options.measureStartField] - 用于生成空间数据的事件表的起始刻度字段，只有当事件为线事件的时候该属性才有意义。
 * @param {string} [options.measureEndField] - 用于生成空间数据的事件表的终止刻度字段，只有当事件为线事件的时候该属性才有意义。
 * @param {string} [options.measureOffsetField] - 刻度偏移量字段。
 * @param {string} [options.errorInfoField] - 错误信息字段，直接写入原事件表，用于描述事件未能生成对应的点或线时的错误信息。
 * @param {Array.<string>} [options.retainedFields] - 欲保留到结果空间数据中的字段集合（系统字段除外）。
 * @usage
 */
class GenerateSpatialDataParameters {


    constructor(options) {
        /**
         * @member {string} GenerateSpatialDataParameters.prototype.routeTable
         * @description 路由数据集。
         */
        this.routeTable = null;

        /**
         * @member {string} GenerateSpatialDataParameters.prototype.routeIDField
         * @description 路由数据集的标识字段。
         */
        this.routeIDField = null;
        /**
         * @member {string} [GenerateSpatialDataParameters.prototype.attributeFilter]
         * @description 属性过滤条件。
         * 当 {@link GenerateSpatialDataParameters.prototype.dataReturnOption.dataReturnMode}  为 {@link DataReturnMode.DATASET_AND_RECORDSET} 或 {@link DataReturnMode.RECORDSET_ONLY} 时有效。
         */
        this.attributeFilter = null;
        /**
         * @member {string} GenerateSpatialDataParameters.prototype.eventTable
         * @description 用于生成空间数据的事件表名。
         */
        this.eventTable = null;

        /**
         * @member {string} GenerateSpatialDataParameters.prototype.eventRouteIDField
         * @description 用于生成空间数据的事件表的路由标识字段。
         */
        this.eventRouteIDField = null;

        /**
         * @member {string} [GenerateSpatialDataParameters.prototype.measureField]
         * @description 用于生成空间数据的事件表的刻度字段，只有当事件为点事件的时候该属性才有意义。
         */
        this.measureField = null;

        /**
         * @member {string} [GenerateSpatialDataParameters.prototype.measureStartField]
         * @description 用于生成空间数据的事件表的起始刻度字段，只有当事件为线事件的时候该属性才有意义。
         */
        this.measureStartField = null;

        /**
         * @member {string} [GenerateSpatialDataParameters.prototype.measureEndField]
         * @description 用于生成空间数据的事件表的终止刻度字段，只有当事件为线事件的时候该属性才有意义。
         */
        this.measureEndField = null;

        /**
         * @member {string} [GenerateSpatialDataParameters.prototype.measureOffsetField]
         * @description 刻度偏移量字段。
         */
        this.measureOffsetField = null;

        /**
         * @member {string} [GenerateSpatialDataParameters.prototype.errorInfoField]
         * @description 错误信息字段，直接写入原事件表，用于描述事件未能生成对应的点或线时的错误信息。
         */
        this.errorInfoField = null;

        /**
         * @member {Array.<string>} [GenerateSpatialDataParameters.prototype.retainedFields]
         * @description 欲保留到结果空间数据中的字段集合（系统字段除外）。
         * 生成空间数据时，无论是否指定保留字段，路由 ID 字段、刻度偏移量字段、刻度值字段（点事件为刻度字段，线事件是起始和终止刻度字段）都会保留到结果空间数据中；
         * 如果没有指定 retainedFields 参数或者 retainedFields 参数数组长度为 0，则返回所有用户字段。
         */
        this.retainedFields = null;

        /**
         * @member {DataReturnOption} GenerateSpatialDataParameters.prototype.dataReturnOption
         * @description 设置数据返回的选项。
         */
        this.dataReturnOption = null;
        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.GenerateSpatialDataParameters";
    }


    /**
     * @function GenerateSpatialDataParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        if (me.routeTable) {
            me.routeTable = null;
        }
        me.routeIDField = null;
        me.attributeFilter = null;
        me.eventTable = null;
        me.eventRouteIDField = null;
        me.measureField = null;
        me.measureStartField = null;
        me.measureEndField = null;
        me.measureOffsetField = null;
        me.errorInfoField = null;
        if (me.dataReturnOption) {
            me.dataReturnOption.destroy();
            me.dataReturnOption = null;
        }
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/GenerateSpatialDataService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class GenerateSpatialDataService
 * @deprecatedclass SuperMap.GenerateSpatialDataService
 * @category iServer SpatialAnalyst GenerateSpatialData
 * @classdesc 动态分段分析服务类。该类负责将客户设置的动态分段分析服务参数传递给服务端，并接收服务端返回的动态分段分析结果数据。
 * 获取的结果数据包括 originResult 、result 两种，其中，originResult 为服务端返回的用 JSON 对象表示的动态分段分析结果数据，result 为服务端返回的动态分段分析结果数据。
 * @param {string} url - 服务地址。如 http://localhost:8090/iserver/services/spatialanalyst-changchun/restjsr/spatialanalyst。
 * @param {Object} options - 参数。</br>
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @extends {SpatialAnalystBase}
 * @example 实例化该类如下例所示：
 * (start code)
 *  function GenerateSpatialData(){
     *
     *  //配置数据返回选项(option)
     *  var option = new DataReturnOption({
     *      expectCount: 1000,
     *      dataset: "generateSpatialData",
     *      deleteExistResultDataset: true,
     *      dataReturnMode: DataReturnMode.DATASET_ONLY
     *  }),
     *  //配置动态分段参数(Parameters)
     *  parameters = new GenerateSpatialDataParameters({
     *      routeTable: "RouteDT_road@Changchun",
     *      routeIDField: "RouteID",
     *      eventTable: "LinearEventTabDT@Changchun",
     *      eventRouteIDField: "RouteID",
     *      measureField: "",
     *      measureStartField: "LineMeasureFrom",
     *      measureEndField: "LineMeasureTo",
     *      measureOffsetField: "",
     *      errorInfoField: "",
     *      retainedFields:[],
     *      dataReturnOption: option
     *  }),
     *  //配置动态分段iService
     *  iService = new GenerateSpatialDataService(Changchun_spatialanalyst);
     *  //执行
     *  iService.processAsync(parameters, generateCompleted);
     *  function Completed(generateSpatialDataEventArgs){//todo};
     *  function Error(generateSpatialDataEventArgs){//todo};
     * (end)
     * @usage
     */
class GenerateSpatialDataService extends SpatialAnalystBase {

    constructor(url, options) {
        super(url, options);

        this.CLASS_NAME = "SuperMap.GenerateSpatialDataService";
    }

    /**
     * @function GenerateSpatialDataService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }


    /**
     * @function GenerateSpatialDataService.prototype.processAsync
     * @description 负责将客户端的动态分段服务参数传递到服务端。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @param {GenerateSpatialDataParameters} params - 动态分段操作参数类。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!(params instanceof GenerateSpatialDataParameters)) {
            return;
        }
        var me = this,
            jsonParameters;

        jsonParameters = me.getJsonParameters(params);

        return me.request({
            method: "POST",
            data: jsonParameters,
            scope: me,
            success: callback,
            failure: callback
        });
    }


    /**
     * @function GenerateSpatialDataService.prototype.getJsonParameters
     * @description 将参数转化为 JSON 字符串。
     * @param {GenerateSpatialDataParameters} params - 动态分段操作参数类。
     * @returns {string}转化后的JSON字符串。
     */
    getJsonParameters(params) {
        var jsonParameters = "",
            jsonStr = "datasets/" + params.routeTable + "/linearreferencing/generatespatialdata",
            me = this;

        me.url = Util_Util.urlPathAppend(me.url, jsonStr);
        me.url = Util_Util.urlAppend(me.url, 'returnContent=true');
        jsonParameters = Util_Util.toJSON(params);
        return jsonParameters;
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/GeoHashGridAggParameter.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class GeoHashGridAggParameter
 * @deprecatedclass SuperMap.GeoHashGridAggParameter
 * @classdesc 格网聚合查询参数类，该参数仅支持数据来源 Elasticsearch 服务的Supermap iServer的rest数据服务。
 * @category iServer Data FeatureResults
 * @param {Object} options - 可选参数。
 * @param {number} [options.precision=5] - 精度。
 * @extends {BucketAggParameter}
 * @usage
 */
class GeoHashGridAggParameter extends BucketAggParameter {
    constructor(options) {
        super();
        /**
         * @member {number} [GeoHashGridAggParameter.prototype.precision=5]
         * @description 网格中数字的精度。
         */
        this.precision = 5;
        Util_Util.extend(this, options);
        /**
         * @member {BucketAggType} [GeoHashGridAggParameter.prototype.aggType=BucketAggType.GEOHASH_GRID]
         * @description 格网聚合类型。
         */
        this.aggType = BucketAggType.GEOHASH_GRID;

        this.CLASS_NAME = 'SuperMap.GeoHashGridAggParameter';
    }

    destroy() {
        super.destroy();
        this.aggType = null;
        this.precision = null;
    }

    /**
     * @function GeoHashGridAggParameter.toJsonParameters
     * @description 将对象转为 JSON 格式。
     * @param param 转换对象。
     * @returns {Object}
     */
    static toJsonParameters(param) {
        var parameters = {
            aggName: param.aggName,
            aggFieldName: param.aggFieldName,
            aggType: param.aggType,
            precision: param.precision
        };
        return Util_Util.toJson(parameters);
    }
}


;// CONCATENATED MODULE: ./src/common/iServer/GeometryOverlayAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class GeometryOverlayAnalystParameters
 * @deprecatedclass SuperMap.GeometryOverlayAnalystParameters
 * @category  iServer SpatialAnalyst OverlayAnalyst
 * @classdesc
 * 几何对象叠加分析参数类。对指定的某两个几何对象做叠加分析。通过该类可以指定要做叠加分析的几何对象、叠加操作类型。
 * @param {Object} options - 参数。
 * @param {GeoJSONObject} options.operateGeometry - 叠加分析的操作几何对象。<br>
 * 点类型可以是：{@link GeometryPoint}|{@link L.Marker}|{@link L.CircleMarker}|{@link L.Circle}|{@link L.GeoJSON}|{@link ol.geom.Point}|{@link ol.format.GeoJSON}|{@link GeoJSONObject}。<br>
 * 线类型可以是：{@link GeometryLineString}|{@link GeometryLinearRing}|{@link L.Polyline}|{@link L.GeoJSON}|{@link ol.geom.LineString}|{@link GeoJSONObject}。<br>
 * 面类型可以是：{@link GeometryPolygon}|{@link L.Polygon}|{@link L.GeoJSON}|{@link ol.geom.Polygon}|{@link GeoJSONObject}。<br>
 * @param {GeoJSONObject} options.sourceGeometry - 叠加分析的源几何对象。
 * @param {Array.<GeoJSONFeature>} [options.operateGeometries] - 批量叠加分析的操作几何对象数组。
 * @param {Array.<GeoJSONFeature>} [options.sourceGeometries] -批量叠加分析的源几何对象数组。
 * @param {OverlayOperationType} [options.operation] - 叠加操作枚举值。
 * @extends {OverlayAnalystParameters}
 * @usage
 */
class GeometryOverlayAnalystParameters extends OverlayAnalystParameters {

    constructor(options) {
        super(options);
        if (options && options.operateGeometry) {
            this.operateGeometry = options.operateGeometry;
        }
        if (options && options.sourceGeometry) {
            this.sourceGeometry = options.sourceGeometry;
        }
        if (options && options.operateGeometries) {
            this.operateGeometries = options.operateGeometries;
        }
        if (options && options.sourceGeometries) {
            this.sourceGeometries = options.sourceGeometries;
        }

        if (options) {
            Util_Util.extend(this, options);
        }
        this.CLASS_NAME = "SuperMap.GeometryOverlayAnalystParameters";
    }

    /**
     * @function GeometryOverlayAnalystParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        if (me.sourceGeometry) {
            me.sourceGeometry.destroy();
            me.sourceGeometry = null;
        }

        if (me.sourceGeometries) {
            me.sourceGeometries.destroy();
            me.sourceGeometries = null;
        }
        if (me.sourceGeometry) {
            me.sourceGeometry.destroy();
            me.sourceGeometry = null;
        }

        if (me.operateGeometries) {
            me.operateGeometries.destroy();
            me.operateGeometries = null;
        }
    }

    /**
     * @function GeometryOverlayAnalystParameters.toObject
     * @param {GeometryOverlayAnalystParameters} geometryOverlayAnalystParameters - 几何对象叠加分析参数类。
     * @param {GeometryOverlayAnalystParameters} tempObj - 几何对象叠加分析参数对象。
     * @description 将几何对象叠加分析参数对象转换为 JSON 对象。
     * @returns {Object} JSON 对象。
     */
    static toObject(geometryOverlayAnalystParameters, tempObj) {
        for (var name in geometryOverlayAnalystParameters) {
            if (name === "sourceGeometry") {
                tempObj.sourceGeometry = ServerGeometry.fromGeometry(geometryOverlayAnalystParameters.sourceGeometry);

            } else if (name === "sourceGeometries") {
                var sourceGeometries = [];
                for (var i = 0; i < geometryOverlayAnalystParameters.sourceGeometries.length; i++) {
                    sourceGeometries.push(ServerGeometry.fromGeometry(geometryOverlayAnalystParameters.sourceGeometries[i]));
                }
                tempObj.sourceGeometries = sourceGeometries;

            } else if (name === "operateGeometry") {
                tempObj.operateGeometry = ServerGeometry.fromGeometry(geometryOverlayAnalystParameters.operateGeometry);

            } else if (name === "operateGeometries") {
                var operateGeometries = [];
                for (var j = 0; j < geometryOverlayAnalystParameters.operateGeometries.length; j++) {
                    operateGeometries.push(ServerGeometry.fromGeometry(geometryOverlayAnalystParameters.operateGeometries[j]));
                }
                tempObj.operateGeometries = operateGeometries;

            } else {
                tempObj[name] = geometryOverlayAnalystParameters[name];
            }
        }
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/GeometrySurfaceAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class GeometrySurfaceAnalystParameters
 * @deprecatedclass SuperMap.GeometrySurfaceAnalystParameters
 * @category  iServer SpatialAnalyst SurfaceAnalyst
 * @classdesc 几何对象表面分析参数类。该类对几何对象表面分析所用到的参数进行设置。
 * @param {Object} options - 参数。
 * @param {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|Array.<number>>} options.points - 表面分析的坐标点数组。
 * @param {Array.<number>} options.zValues - 表面分析的坐标点的 Z 值数组。
 * @param {number} [options.resolution] - 获取或设置指定中间结果（栅格数据集）的分辨率。
 * @param {DataReturnOption} [options.resultSetting] - 结果返回设置类。
 * @param {SurfaceAnalystParametersSetting} options.extractParameter - 获取或设置表面分析参数。
 * @param {SurfaceAnalystMethod} [options.surfaceAnalystMethod = SurfaceAnalystMethod.ISOLINE] - 获取或设置表面分析的提取方法，提取等值线和提取等值面。
 * @extends {SurfaceAnalystParameters}
 * @usage
 */
class GeometrySurfaceAnalystParameters extends SurfaceAnalystParameters {


    constructor(options) {
        super(options);
        /**
         * @member {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|Array.<number>>} GeometrySurfaceAnalystParameters.prototype.points
         * @description 获取或设置用于表面分析的坐标点数组。
         */
        this.points = null;

        /**
         * @member {Array.<number>} GeometrySurfaceAnalystParameters.prototype.zValues
         * @description 获取或设置用于提取操作的值。提取等值线时，将使用该数组中的值，
         * 对几何对象中的坐标点数组进行插值分析，得到栅格数据集（中间结果），接着从栅格数据集提取等值线。
         */
        this.zValues = null;
        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.GeometrySurfaceAnalystParameters";
    }

    /**
     * @function GeometrySurfaceAnalystParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        if (me.points) {
            for (var i = 0, points = me.points, len = points.length; i < len; i++) {
                points[i].destroy();
            }
            me.points = null;
        }
        me.zValues = null;
    }

}


;// CONCATENATED MODULE: ./src/common/iServer/GeometryThiessenAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class GeometryThiessenAnalystParameters
 * @deprecatedclass SuperMap.GeometryThiessenAnalystParameters
 * @constructs GeometryThiessenAnalystParameters
 * @category iServer SpatialAnalyst ThiessenPolygonAnalyst
 * @classdesc 几何对象泰森多边形分析参数类。对指定的某个几何对象做泰森多边形分析。通过该类可以指定要做泰森多边形分析的几何对象、返回数据集名称等。
 * @param {Object} options - 参数。
 * @param {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|Array.<number>>} options.points - 使用点数组进行分析时使用的几何对象。
 * @extends {ThiessenAnalystParameters}
 * @usage
 */

class GeometryThiessenAnalystParameters extends ThiessenAnalystParameters {


    constructor(options) {
        super(options);
        /**
         * @member {Array.<GeometryPoint|L.LatLng|L.Point|ol.geom.Point|mapboxgl.LngLat|Array.<number>>} GeometryThiessenAnalystParameters.prototype.points
         * @description 使用点数组进行分析时使用的几何对象。
         */
        this.points = null;
        if (options) {
            Util_Util.extend(this, options);
        }
        this.CLASS_NAME = "SuperMap.GeometryThiessenAnalystParameters";
    }

    /**
     * @function GeometryThiessenAnalystParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        if (me.points) {
            for (var i = me.points.length - 1; i >= 0; i--) {
                me.points[i].destroy();
            }
            me.points = null;
        }
    }

    /**
     * @function GeometryThiessenAnalystParameters.toObject
     * @param {GeometryThiessenAnalystParameters} geometryThiessenAnalystParameters - 几何对象泰森多边形分析参数类。
     * @param {GeometryThiessenAnalystParameters} tempObj - 几何对象泰森多边形分析参数对象。
     * @description 将几何对象泰森多边形分析参数对象转换为 JSON 对象。
     * @returns {Object} JSON 对象。
     */
    static toObject(geometryThiessenAnalystParameters, tempObj) {
        for (var name in geometryThiessenAnalystParameters) {
            if (name === "clipRegion") {
                tempObj.clipRegion = ServerGeometry.fromGeometry(geometryThiessenAnalystParameters.clipRegion);
            } else {
                tempObj[name] = geometryThiessenAnalystParameters[name];
            }
        }
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/GeoprocessingService.js


/**
 * @class GeoprocessingService
 * @deprecatedclass SuperMap.GeoprocessingService
 * @category  iServer ProcessingAutomationService
 * @classdesc 处理自动化服务接口的基类。
 * @version 10.1.0
 * @extends {CommonServiceBase}
 * @param {string} url - 服务地址。
 * @param {Object} options - 参数。
 * @param {Events} options.events - 处理所有事件的对象。
 * @usage
 */
class GeoprocessingService extends CommonServiceBase {
    constructor(url, options) {
        options = options || {};
        super(url, options);
        this.CLASS_NAME = 'SuperMap.GeoprocessingService';
        this.headers = {};
        this.crossOrigin = true;
    }
    /**
     * @function GeoprocessingService.prototype.getTools
     * @description 获取处理自动化工具列表。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    getTools(callback) {
      return this._processAsync({ url: `${this.url}/list`, callback });
    }
    /**
     * @function GeoprocessingService.prototype.getTool
     * @description 获取处理自动化工具的ID、名称、描述、输入参数、环境参数和输出结果等相关参数。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    getTool(identifier, callback) {
      return this._processAsync({ url: `${this.url}/${identifier}`, callback });
    }
    /**
     * @function GeoprocessingService.prototype.execute
     * @description 同步执行处理自动化工具。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {Object} parameter - 处理自动化工具的输入参数。
     * @param {Object} environment - 处理自动化工具的环境参数。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    execute(identifier, parameter, environment, callback) {
        parameter = parameter ? parameter : null;
        environment = environment ? environment : null;
        const executeParamter = { parameter, environment };
        return this._processAsync({ url: `${this.url}/${identifier}/execute`, executeParamter, callback });
    }
    /**
     * @function GeoprocessingService.prototype.submitJob
     * @description 异步执行处理自动化工具。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {Object} parameter - 处理自动化工具的输入参数。
     * @param {Object} environments - 处理自动化工具的环境参数。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    submitJob(identifier, parameter, environments, callback) {
        parameter = parameter ? parameter : null;
        environments = environments ? environments : null;
        const asyncParamter = JSON.stringify({ parameter: parameter, environments: environments });
        return this._processAsync({ url: `${this.url}/${identifier}/jobs`, method: 'POST', callback, params: asyncParamter });
    }

    /**
     * @function GeoprocessingService.prototype.waitForJobCompletion
     * @description 获取处理自动化异步执行状态信息。
     * @param {string} jobId - 处理自动化任务ID。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {Object} options - 状态信息参数。
     * @param {number} options.interval - 定时器时间间隔。
     * @param {function} options.statusCallback - 任务状态的回调函数。
     */
    waitForJobCompletion(jobId, identifier, options, callback) {
        const me = this;
        const timer = setInterval(function () {
            const transformResult = function (serverResult) {
                const state = serverResult.result.state.runState;
                if (serverResult.options.statusCallback) {
                    serverResult.options.statusCallback(state);
                }
                if (['FINISHED', 'FAILED', 'CANCELED'].indexOf(state) !== -1) {
                  clearInterval(timer);
                  callback(serverResult);
                }
            };
            me._processAsync({ url: `${me.url}/${identifier}/jobs/${jobId}`, callback: transformResult });
        }, options.interval);
    }

    /**
     * @function GeoprocessingService.prototype.getJobInfo
     * @description 获取处理自动化任务的执行信息。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {string} jobId - 处理自动化任务ID。
     * @returns {Promise} Promise 对象。
     */
    getJobInfo(identifier, jobId, callback) {
      return this._processAsync({ url: `${this.url}/${identifier}/jobs/${jobId}`, callback });
    }

    /**
     * @function GeoprocessingService.prototype.cancelJob
     * @description 取消处理自动化任务的异步执行。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {string} jobId - 处理自动化任务ID。
     * @returns {Promise} Promise 对象。
     */
    cancelJob(identifier, jobId, callback) {
      return this._processAsync({ url: `${this.url}/${identifier}/jobs/${jobId}/cancel`, callback });
    }
    /**
     * @function GeoprocessingService.prototype.getJobs
     * @description 获取处理自动化服务任务列表。
     * @param {string} identifier - 处理自动化工具ID。(传参代表identifier算子的任务列表，不传参代表所有任务的列表)
     * @returns {Promise} Promise 对象。
     */
    getJobs(identifier, callback) {
        let url = `${this.url}/jobs`;
        if (identifier && typeof identifier === 'string') {
            url = `${this.url}/${identifier}/jobs`;
        } else {
          callback = identifier;
        }
        this._processAsync({ url, callback });
    }
    /**
     * @function GeoprocessingService.prototype.getResults
     * @description 处理自动化工具执行的结果等,支持结果过滤。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {string} jobId - 处理自动化任务ID。
     * @param {string} filter - 输出异步结果的ID。(可选，传入filter参数时对该处理自动化工具执行的结果进行过滤获取，不填参时显示所有的执行结果)
     * @returns {Promise} Promise 对象。
     */
    getResults(identifier, jobId, filter, callback) {
        let url = `${this.url}/${identifier}/jobs/${jobId}/results`;
        if (filter) {
          if (typeof filter === 'string') {
            url = `${url}/${filter}`;
          } else {
            callback = filter;
          }
        } 
        return this._processAsync({ url, callback });
    }
   
    _processAsync({ url, method, callback, paramter }) {
          return this.request({
              url: url,
              method: method || 'GET',
              params: paramter,
              headers: { 'Content-type': 'application/json' },
              scope: this,
              success: callback,
              failure: callback
          });
      }
}

;// CONCATENATED MODULE: ./src/common/iServer/GeoRelationAnalystParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class GeoRelationAnalystParameters
 * @deprecatedclass SuperMap.GeoRelationAnalystParameters
 * @category iServer SpatialAnalyst GeoRelationAnalyst
 * @classdesc 空间关系分析服务参数类。使用该类可以为空间关系分析服务提供所需的参数信息。
 * @param {Object} options - 参数。
 * @param {FilterParameter} options.sourceFilter - 空间关系分析中的源数据集查询参数。仅 name, ids, attributeFilter 和 fields 字段有效。
 * @param {FilterParameter} options.referenceFilter - 空间关系分析中的参考数据集查询参数。仅 name, ids, attributeFilter 和 fields 字段有效。
 * @param {SpatialRelationType} options.spatialRelationType - 指定的空间关系类型。
 * @param {boolean} [options.isBorderInside] - 边界处理方式，即位于面边线上的点是否被面包含。此参数仅用于空间关系为包含或被包含的情况。
 * @param {boolean} [options.returnFeature] - 是否返回 Feature 信息。
 * @param {boolean} [options.returnGeoRelatedOnly=true] - 仅返回满足指定空间关系的空间对象。
 * @param {number} [options.startRecord=0] - 分析结果起始记录位置。
 * @param {number} [options.expectCount=500] - 空间关系分析期望返回结果记录数，如果实际不足500条结果则返回所有分析结果。
 * @usage
 */
class GeoRelationAnalystParameters {


    constructor(options) {
        /**
         *  @member {string} GeoRelationAnalystParameters.prototype.dataset
         *  @description 源数据集名称。
         */
        this.dataset = null;

        /**
         * @member {FilterParameter} GeoRelationAnalystParameters.prototype.sourceFilter
         * @description 空间关系分析中的源数据集查询参数。仅 ids、attributeFilter 和 fields 字段有效。
         */
        this.sourceFilter = null;

        /**
         * @member {FilterParameter} GeoRelationAnalystParameters.prototype.referenceFilter
         * @description 空间关系分析中的参考数据集查询参数。仅 name，ids，attributeFilter 和 fields 字段有效。
         */
        this.referenceFilter = null;

        /**
         * @member {SpatialRelationType} GeoRelationAnalystParameters.prototype.spatialRelationType
         * @description 指定的空间关系类型。
         */
        this.spatialRelationType = null;

        /**
         * @member {boolean} [GeoRelationAnalystParameters.prototype.isBorderInside]
         * @description 边界处理方式，即位于面边线上的点是否被面包含。此参数仅用于空间关系为包含或被包含的情况。
         */
        this.isBorderInside = null;

        /**
         * @member {boolean} [GeoRelationAnalystParameters.prototype.returnFeature]
         * @description 是否返回 Feature 信息。
         */
        this.returnFeature = null;

        /**
         * @member {boolean} [GeoRelationAnalystParameters.prototype.returnGeoRelatedOnly=true]
         * @description 是否仅返回满足指定空间关系的空间对象。
         */
        this.returnGeoRelatedOnly = null;

        /**
         * @member {number} [GeoRelationAnalystParameters.prototype.returnGeoRelatedOnly=0]
         * @description 分析结果起始记录位置。
         */
        this.startRecord = 0;

        /**
         * @member {number} [GeoRelationAnalystParameters.prototype.expectCount=500]
         * @description 空间关系分析期望返回结果记录数，如果实际不足 500 条结果则返回所有分析结果。
         */
        this.expectCount = 500;
        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.GeoRelationAnalystParameters";
    }

    /**
     * @function GeoRelationAnalystParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        if (me.sourceFilter) {
            me.sourceFilter.destroy();
        }
        me.sourceFilter = null;

        if (me.referenceFilter) {
            me.referenceFilter.destroy();
        }
        me.referenceFilter = null;

        me.dataset = null;
        me.spatialRelationType = null;
        me.isBorderInside = null;
        me.returnFeature = null;
        me.returnGeoRelatedOnly = null;
        me.startRecord = null;
        me.expectCount = null;
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/GeoRelationAnalystService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class GeoRelationAnalystService
 * @deprecatedclass SuperMap.GeoRelationAnalystService
 * @category iServer SpatialAnalyst GeoRelationAnalyst
 * @classdesc 空间关系分析服务类。该类负责将客户设置的空间关系分析服务参数传递给服务端，并接收服务端返回的空间关系分析结果数据。
 * @param {string} url - 服务地址。如 http://localhost:8090/iserver/services/spatialanalyst-changchun/restjsr/spatialanalyst。
 * @param {Object} options - 参数。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @extends {SpatialAnalystBase}
 * @example 实例化该类如下例所示：
 * (start code)
 *  function datasetGeoRelationAnalystProcess() {
     *      var referenceFilter = new FilterParameter({
     *                              name:"Frame_R@Changchun",
     *                              attributeFilter:"SmID>0"});
     *      var sourceFilter = new FilterParameter({
     *                          attributeFilter:"SmID>0"});
     *      //初始化服务类
     *      var datasetGeoRelationService = new GeoRelationAnalystService(
     *          "http://localhost:8090/iserver/services/spatialanalyst-changchun/restjsr/spatialanalyst/"),
     *      //构建参数类
     *      datasetGeoRelationParameters = new GeoRelationAnalystParameters({
     *          dataset: "Park@Changchun",
     *          startRecord: 0,
     *          expectCount: 20,
     *          sourceFilter: sourceFilter,
     *          referenceFilter: referenceFilter,
     *          spatialRelationType: SpatialRelationType.INTERSECT,
     *          isBorderInside: true,
     *          returnFeature: true,
     *          returnGeoRelatedOnly: true
     *      });
     *      datasetGeoRelationService.events.on({
     *          "processCompleted": datasetGeoRelationAnalystCompleted,
     *          "processFailed": datasetGeoRelationAnalystFailed});
     *      //执行
     *      datasetGeoRelationService.processAsync(datasetGeoRelationParameters);
     *  }
 *  function Completed(datasetGeoRelationAnalystCompleted){//todo};
 *  function Error(datasetGeoRelationAnalystFailed){//todo};
 * (end)
 * @usage
 */
class GeoRelationAnalystService extends SpatialAnalystBase {

    constructor(url, options) {
        super(url, options);
        this.CLASS_NAME = "SuperMap.GeoRelationAnalystService";
    }

    /**
     * @function GeoRelationAnalystService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function GeoRelationAnalystService.prototype.processAsync
     * @description 负责将客户端的空间关系分析参数传递到服务端
     * @param {GeoRelationAnalystParameters} parameter - 空间关系分析所需的参数信息。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(parameter, callback) {
        if (!(parameter instanceof GeoRelationAnalystParameters)) {
            return;
        }
        var me = this;
        me.url = Util_Util.urlPathAppend(me.url, 'datasets/' + parameter.dataset + '/georelation');
        var jsonParameters = Util_Util.toJSON(parameter);

        me.url = Util_Util.urlAppend(me.url, 'returnContent=true');

        return me.request({
            method: "POST",
            data: jsonParameters,
            scope: me,
            success: callback,
            failure: callback
        });
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/DatasetService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class DatasetService
 * @deprecatedclass SuperMap.DatasetService
 * @category iServer Data Dataset
 * @classdesc 数据集查询服务。
 * @param {string} url - 服务的访问地址。如访问World Data服务，只需将url设为：http://localhost:8090/iserver/services/data-world/rest/data 即可。
 * @param {Object} options - 参数。
 * @param {DataFormat} [options.format=DataFormat.GEOJSON] - 查询结果返回格式，目前支持 iServerJSON 和 GeoJSON 两种格式。参数格式为 "ISERVER"，"GEOJSON"。
 * @param {string}options.datasource - 数据源名称。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @usage
 */
class DatasetService extends CommonServiceBase {

    constructor(url, options) {
        super(url, options);
        if(!options){
            return;
        }
        /**
         * @member {string} DatasetService.prototype.datasource
         * @description 要查询的数据集所在的数据源名称。
         */
        this.datasource = null;

        /**
         *  @member {string} DatasetService.prototype.dataset
         *  @description 要查询的数据集名称。
         */
        this.dataset = null;
        if (options) {
            Util_Util.extend(this, options);
        }

        this.CLASS_NAME = "SuperMap.DatasetService";
    }

    /**
     * @function DatasetService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        me.datasource = null;
        me.dataset = null;
    }

    /**
     * @function DatasetService.prototype.getDatasetsService
     * @description 执行服务，查询数据集服务。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    getDatasetsService(params, callback) {
        const url = Util_Util.urlPathAppend(this.url,`datasources/name/${params}/datasets`);
        return this.processAsync(url, 'GET', callback);
    }

    /**
     * @function DatasetService.prototype.getDatasetService
     * @description 执行服务，查询数据集信息服务。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    getDatasetService(datasourceName, datasetName, callback) {
        const url = Util_Util.urlPathAppend(this.url,`datasources/name/${datasourceName}/datasets/name/${datasetName}`);
        return this.processAsync(url, 'GET', callback);
    }

    /**
     * @function DatasetService.prototype.setDatasetService
     * @description 执行服务，更改数据集信息服务。
     * @returns {Promise} Promise 对象。
     */
    setDatasetService(params, callback) {
        if (!params) {
            return;
        }
        const url = Util_Util.urlPathAppend(this.url, `datasources/name/${params.datasourceName}/datasets/name/${params.datasetName}`);
        delete params.datasourceName;
        return this.processAsync(url, 'PUT', callback, params);
    }

     /**
     * @function DatasetService.prototype.deleteDatasetService
     * @description 执行服务，删除数据集信息服务。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    deleteDatasetService(datasourceName, datasetName, callback) {
      const url = Util_Util.urlPathAppend(this.url, `datasources/name/${datasourceName}/datasets/name/${datasetName}`);
      return this.processAsync(url, 'DELETE', callback);
    }

    processAsync(url, method, callback, params) {
       var me = this;
       let requestConfig = {
          url,
          method,
          scope: me,
          success: callback,
          failure: callback
        }
        params && (requestConfig.data = Util_Util.toJSON(params));
        return me.request(requestConfig);
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/GetFeaturesParametersBase.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/


/**
 * @class GetFeaturesParametersBase
 * @deprecatedclass SuperMap.GetFeaturesParametersBase
 * @category  iServer Data FeatureResults
 * @classdesc 要素查询参数基类。
 * @param {Object} options - 参数。
 * @param {Array.<string>} options.datasetNames - 数据集名称列表。
 * @param {boolean} [options.returnContent=true] - 是否直接返回查询结果。
 * @param {number} [options.fromIndex=0] - 查询结果的最小索引号。
 * @param {number} [options.toIndex=19] - 查询结果的最大索引号。
 * @param {string|number} [options.targetEpsgCode] - 动态投影的目标坐标系对应的 EPSG Code，使用此参数时，returnContent 参数需为 true。
 * @param {Object} [options.targetPrj] - 动态投影的目标坐标系。使用此参数时，returnContent 参数需为 true。如：prjCoordSys={"epsgCode":3857}。当同时设置 targetEpsgCode 参数时，此参数不生效。
 * @param {MetricsAggParameter|GeoHashGridAggParameter} [options.aggregations] - 聚合查询参数。该参数仅支持数据来源 Elasticsearch 服务的Supermap iServer的rest数据服务。
 * @usage
 */
class GetFeaturesParametersBase {
    constructor(options) {
        /**
         * @member {Array.<string>} GetFeaturesParametersBase.prototype.datasetName
         * @description 数据集集合中的数据集名称列表。
         */
        this.datasetNames = null;

        /**
         * @member {string} GetFeaturesParametersBase.prototype.targetEpsgCode
         * @description 动态投影的目标坐标系对应的 EPSG Code，使用时需设置 returnContent 参数为 true。
         */
        this.targetEpsgCode = null;

        /**
         * @member {Object} GetFeaturesParametersBase.prototype.targetPrj
         * @description 动态投影的目标坐标系。使用时需设置 returnContent 参数为 true。如：prjCoordSys={"epsgCode":3857}。当同时设置 targetEpsgCode 参数时，此参数不生效。
         */
        this.targetPrj = null;

        /**
         * @member {boolean} [GetFeaturesParametersBase.prototype.returnContent=true]
         * @description 是否立即返回新创建资源的表述还是返回新资源的 URI。
         *              如果为 true，则直接返回新创建资源，即查询结果的表述。
         *              如果为 false，则返回的是查询结果资源的 URI。
         */
        this.returnContent = true;

        /**
         * @member {number} [GetFeaturesParametersBase.prototype.fromIndex=0]
         * @description 查询结果的最小索引号。如果该值大于查询结果的最大索引号，则查询结果为空。
         */
        this.fromIndex = 0;

        /**
         * @member {number} [GetFeaturesParametersBase.prototype.toIndex=19]
         * @description 查询结果的最大索引号。如果该值大于查询结果的最大索引号，则以查询结果的最大索引号为终止索引号。
         */
        this.toIndex = 19;

        /**
         * @member {boolean} [GetFeaturesParametersBase.prototype.returnCountOnly=false]
         * @description 只返回查询结果的总数。
         */
        this.returnCountOnly = false;

        /**
         * @member {number} [GetFeaturesParametersBase.prototype.maxFeatures=1000]
         * @description 进行 SQL 查询时，用于设置服务端返回查询结果条目数量。
         */
        this.maxFeatures = null;

        /**
         * @member {number} [GetFeaturesParametersBase.prototype.hasGeometry=true]
         * @description 返回结果是否包含Geometry。
         */
        this.hasGeometry = true;

        /**
         * @member {MetricsAggParameter|GeoHashGridAggParameter} GetFeaturesParametersBase.prototype.aggregations
         * @description 聚合查询参数，该参数仅支持数据来源 Elasticsearch 服务的Supermap iServer的rest数据服务。
         */
        this.aggregations = null;

        Util_Util.extend(this, options);

        this.CLASS_NAME = 'SuperMap.GetFeaturesParametersBase';
    }

    /**
     *
     * @function GetFeaturesParametersBase.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        var me = this;
        me.datasetNames = null;
        me.returnContent = null;
        me.fromIndex = null;
        me.toIndex = null;
        me.hasGeometry = null;
        me.maxFeatures = null;
        me.targetEpsgCode = null;
        me.targetPrj = null;
        if (me.aggregation) {
            me.aggregation = null;
        }
    }
}

;// CONCATENATED MODULE: ./src/common/iServer/GetFeaturesByBoundsParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class GetFeaturesByBoundsParameters
 * @deprecatedclass SuperMap.GetFeaturesByBoundsParameters
 * @category iServer Data FeatureResults
 * @classdesc 数据集范围查询参数类，该类用于设置数据集范围查询的相关参数。
 * @param {Object} options - 参数。
 * @param {(SuperMap.Bounds|L.Bounds|L.LatLngBounds|ol.extent|mapboxgl.LngLatBounds|GeoJSONObject)} options.bounds - 查询的范围对象。
 * @param {Array.<string>} options.datasetNames - 数据集名称列表。
 * @param {string} [options.attributeFilter] - 范围查询属性过滤条件。
 * @param {Array.<string>} [options.fields] - 设置查询结果返回字段。默认返回所有字段。
 * @param {SpatialQueryMode} [options.spatialQueryMode=SpatialQueryMode.CONTAIN] - 空间查询模式常量。
 * @param {boolean} [options.returnContent=true] - 是否直接返回查询结果。
 * @param {number} [options.fromIndex=0] - 查询结果的最小索引号。
 * @param {number} [options.toIndex=19] - 查询结果的最大索引号。
 * @param {string|number} [options.targetEpsgCode] - 动态投影的目标坐标系对应的 EPSG Code，使用此参数时，returnContent 参数需为 true。
 * @param {Object} [options.targetPrj] - 动态投影的目标坐标系。使用此参数时，returnContent 参数需为 true。如：prjCoordSys={"epsgCode":3857}。当同时设置 targetEpsgCode 参数时，此参数不生效。
 * @param {MetricsAggParameter|GeoHashGridAggParameter} [options.aggregations] - 聚合查询参数。该参数仅支持数据来源 Elasticsearch 服务的Supermap iServer的rest数据服务。
 * @extends {GetFeaturesParametersBase}
 * @usage
 */

class GetFeaturesByBoundsParameters extends GetFeaturesParametersBase {
    constructor(options) {
        super(options);
        /**
         * @member {string} GetFeaturesByBoundsParameters.prototype.getFeatureMode
         * @description 数据集查询模式。范围查询有 "BOUNDS"，"BOUNDS_ATTRIBUTEFILTER" 两种，当用户设置 attributeFilter 时会自动切换到 BOUNDS_ATTRIBUTEFILTER 访问服务。
         */
        this.getFeatureMode = GetFeaturesByBoundsParameters.getFeatureMode.BOUNDS;

        /**
         * @member {(SuperMap.Bounds|L.Bounds|L.LatLngBounds|ol.extent|mapboxgl.LngLatBounds|GeoJSONObject)} GetFeaturesByBoundsParameters.prototype.bounds
         * @description 用于查询的范围对象。
         *
         */
        this.bounds = null;

        /**
         * @member {Array.<string>} GetFeaturesByBoundsParameters.prototype.fields
         * @description 设置查询结果返回字段。当指定了返回结果字段后，则 GetFeaturesResult 中的 features 的属性字段只包含所指定的字段。不设置即返回全部字段。
         */
        this.fields = null;

        /**
         * @member {string} GetFeaturesByBoundsParameters.prototype.attributeFilter
         * @description 范围查询属性过滤条件。
         */
        this.attributeFilter = null;

        /**
         * @member {SpatialQueryMode} [GetFeaturesByBoundsParameters.prototype.spatialQueryMode=SpatialQueryMode.CONTAIN]
         * @description 空间查询模式常量。
         */
        this.spatialQueryMode = SpatialQueryMode.CONTAIN;

        Util_Util.extend(this, options);
        this.CLASS_NAME = 'SuperMap.GetFeaturesByBoundsParameters';
    }

    /**
     * @function GetFeaturesByBoundsParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        if (me.bounds) {
            me.bounds.destroy();
            me.bounds = null;
        }
        if (me.fields) {
            while (me.fields.length > 0) {
                me.fields.pop();
            }
            me.fields = null;
        }
        me.attributeFilter = null;
        me.spatialQueryMode = null;
        me.getFeatureMode = null;
    }

    /**
     * @function GetFeaturesByBoundsParameters.toJsonParameters
     * @description 将 {@link GetFeaturesByBoundsParameters} 对象参数转换为 JSON 字符串。
     * @param {GetFeaturesByBoundsParameters} params - 范围查询参数。
     * @returns {string} 转化后的 JSON 字符串。
     *
     */
    static toJsonParameters(params) {
        var filterParameter, bounds, parasByBounds;

        bounds = {
            leftBottom: { x: params.bounds.left, y: params.bounds.bottom },
            rightTop: { x: params.bounds.right, y: params.bounds.top }
        };
        parasByBounds = {
            datasetNames: params.datasetNames,
            getFeatureMode: GetFeaturesByBoundsParameters.getFeatureMode.BOUNDS,
            bounds: bounds,
            spatialQueryMode: params.spatialQueryMode
        };
        if (params.fields) {
            filterParameter = new FilterParameter();
            filterParameter.name = params.datasetNames;
            filterParameter.fields = params.fields;
            parasByBounds.queryParameter = filterParameter;
        }
        if (params.attributeFilter) {
            parasByBounds.attributeFilter = params.attributeFilter;
            parasByBounds.getFeatureMode = GetFeaturesByBoundsParameters.getFeatureMode.BOUNDS_ATTRIBUTEFILTER;
        }
        if (params.maxFeatures && !isNaN(params.maxFeatures)) {
            parasByBounds.maxFeatures = params.maxFeatures;
        }

        if (typeof params.hasGeometry === 'boolean') {
          parasByBounds.hasGeometry = params.hasGeometry;
        }

        if (params.targetEpsgCode) {
            parasByBounds.targetEpsgCode = params.targetEpsgCode;
        }
        if (!params.targetEpsgCode && params.targetPrj) {
            parasByBounds.targetPrj = params.targetPrj;
        }
        if (params.aggregations) {
            parasByBounds.aggregations = params.aggregations;
        }

        return Util_Util.toJSON(parasByBounds);
    }
}

GetFeaturesByBoundsParameters.getFeatureMode = {
    BOUNDS: 'BOUNDS',
    BOUNDS_ATTRIBUTEFILTER: 'BOUNDS_ATTRIBUTEFILTER'
};


;// CONCATENATED MODULE: ./src/common/iServer/GetFeaturesServiceBase.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class GetFeaturesServiceBase
 * @deprecatedclass SuperMap.GetFeaturesServiceBase
 * @category  iServer Data FeatureResults
 * @classdesc 数据服务中数据集查询服务基类。获取结果数据类型为 Object。包含 result 属性，result 的数据格式根据 format 参数决定为 GeoJSON 或者 iServerJSON。
 * @extends CommonServiceBase
 * @param {string} url - 服务地址。请求数据服务中数据集查询服务，
 * URL应为：http://{服务器地址}:{服务端口号}/iserver/services/{数据服务名}/rest/data/
 * 例如："http://localhost:8090/iserver/services/data-jingjin/rest/data/"
 * @param {Object} options - 参数。
 * @param {DataFormat} [options.format=DataFormat.GEOJSON] - 查询结果返回格式，目前支持 iServerJSON、GeoJSON、FGB 三种格式。参数格式为 "ISERVER"，"GEOJSON"，"FGB"。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @example
 * var myService = new GetFeaturesServiceBase(url);
 * @usage
 */
class GetFeaturesServiceBase extends CommonServiceBase {

    constructor(url, options) {
        super(url, options);
        options = options || {};

        /**
         * @member {boolean} [GetFeaturesServiceBase.prototype.returnContent=true]
         * @description 是否立即返回新创建资源的表述还是返回新资源的 URI。
         * 如果为 true，则直接返回新创建资源，即查询结果的表述。
         * 如果为 false，则返回的是查询结果资源的 URI。
         */
        this.returnContent = true;

        /**
         * @member {number} [GetFeaturesServiceBase.prototype.fromIndex=0]
         * @description 查询结果的最小索引号。如果该值大于查询结果的最大索引号，则查询结果为空。
         */
        this.fromIndex = 0;

        /**
         * @member {number} [GetFeaturesServiceBase.prototype.toIndex=19]
         * @description 查询结果的最大索引号。
         * 如果该值大于查询结果的最大索引号，则以查询结果的最大索引号为终止索引号。
         */
        this.toIndex = 19;

         /**
         * @member {number} [GetFeaturesServiceBase.prototype.hasGeometry=true]
         * @description 返回结果是否包含Geometry。
         */
        this.hasGeometry = true;

        /**
         * @member {number} [GetFeaturesServiceBase.prototype.maxFeatures=1000]
         * @description 进行 SQL 查询时，用于设置服务端返回查询结果条目数量。
         */
        this.maxFeatures = null;

        /**
         * @member {string} [GetFeaturesServiceBase.prototype.format=DataFormat.GEOJSON]
         * @description 查询结果返回格式，目前支持 iServerJSON、GeoJSON、FGB 三种格式。
         * 参数格式为 "ISERVER"，"GEOJSON"，"FGB"。
         */
        this.format = DataFormat.GEOJSON;

        Util_Util.extend(this, options);
        this.url = Util_Util.urlPathAppend(this.url, 'featureResults');
        this.CLASS_NAME = "SuperMap.GetFeaturesServiceBase";
    }

    /**
     * @function GetFeaturesServiceBase.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        super.destroy();
        var me = this;
        me.returnContent = null;
        me.fromIndex = null;
        me.toIndex = null;
        me.maxFeatures = null;
        me.format = null;
        me.hasGeometry = null;
    }

    /**
     * @function GetFeaturesServiceBase.prototype.processAsync
     * @description 将客户端的查询参数传递到服务端。
     * @param {Object} params - 查询参数。
     * @param {RequestCallback} [callback] - 回调函数，该参数未传时可通过返回的 promise 获取结果。
     * @returns {Promise} Promise 对象。
     */
    processAsync(params, callback) {
        if (!params) {
            return;
        }
        var me = this,
            jsonParameters = null,
            firstPara = true;

        me.returnContent = params.returnContent;
        me.fromIndex = params.fromIndex;
        me.toIndex = params.toIndex;
        me.maxFeatures = params.maxFeatures;
        me.hasGeometry = params.hasGeometry;
        if (me.returnContent) {
          firstPara = false;
        }
        var isValidNumber = me.fromIndex != null && me.toIndex != null && !isNaN(me.fromIndex) && !isNaN(me.toIndex);
        if (isValidNumber && me.fromIndex >= 0 && me.toIndex >= 0 && !firstPara) {
            me.url = Util_Util.urlAppend(me.url, `fromIndex=${me.fromIndex}&toIndex=${me.toIndex}`);
        }

        if (params.returnCountOnly) {
            me.url = Util_Util.urlAppend(me.url, "&returnCountOnly=" + params.returnContent)
        }

        jsonParameters = me.getJsonParameters(params);
        return me.request({
            method: "POST",
            data: jsonParameters,
            scope: me,
            success: callback,
            failure: callback
        });
    }

    /**
     * @function GetFeaturesServiceBase.prototype.transformResult
     * @description 状态完成时转换结果。
     * @param {Object} result - 服务器返回的结果对象。
     * @param {Object} options - 请求参数。
     * @return {Object} 转换结果。
     */
    transformResult(result, options) {
        var me = this;
        result = Util_Util.transformResult(result);
        if (me.format === DataFormat.GEOJSON && result.features) {
            var geoJSONFormat = new GeoJSON();
            result.features = geoJSONFormat.toGeoJSON(result.features);
        }
       return { result, options };
    }

    dataFormat() {
      return [DataFormat.GEOJSON, DataFormat.ISERVER, DataFormat.FGB];
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/GetFeaturesByBoundsService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class GetFeaturesByBoundsService
 * @deprecatedclass SuperMap.GetFeaturesByBoundsService
 * @category iServer Data FeatureResults
 * @classdesc 数据集范围查询服务类,查询与指定范围对象符合一定空间关系的矢量要素。
 * @description 数据集范围查询服务类构造函数。
 * @extends {GetFeaturesServiceBase}
 * @param {string} url - 服务地址。请求数据服务中数据集查询服务，URL 应为：http://{服务器地址}:{服务端口号}/iserver/services/{数据服务名}/rest/data/；
 * 例如："http://localhost:8090/iserver/services/data-jingjin/rest/data/"
 * @param {Object} options - 参数。
 * @param {DataFormat} [options.format=DataFormat.GEOJSON] - 查询结果返回格式，目前支持 iServerJSON、GeoJSON、FGB 三种格式。参数格式为 "ISERVER"，"GEOJSON"，"FGB"。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @example
 * var myGetFeaturesByBoundsService = new SuperMa.GetFeaturesByBoundsService(url);
 * @usage
 */

class GetFeaturesByBoundsService extends GetFeaturesServiceBase {

    constructor(url, options) {
        super(url, options);

        this.CLASS_NAME = "SuperMap.GetFeaturesByBoundsService";
    }

    /**
     * @function GetFeaturesByBoundsService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function GetFeaturesByBoundsService.prototype.getJsonParameters
     * @description 将查询参数转化为 JSON 字符串。在本类中重写此方法，可以实现不同种类的查询（ID, SQL, Buffer, Geometry,Bounds等）。
     * @param params {GetFeaturesByBoundsParameters}
     * @returns {string} 转化后的 JSON 字符串。
     *
     */
    getJsonParameters(params) {
        return GetFeaturesByBoundsParameters.toJsonParameters(params);
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/GetFeaturesByBufferParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/





/**
 * @class GetFeaturesByBufferParameters
 * @deprecatedclass SuperMap.GetFeaturesByBufferParameters
 * @category iServer Data FeatureResults
 * @classdesc 数据集缓冲区查询参数类。
 * @param {Object} options - 参数。
 * @param {number} options.bufferDistance - buffer 距离，单位与所查询图层对应的数据集单位相同。
 * @param {GeoJSONObject} options.geometry - 空间查询条件。
 * @param {Array.<string>} options.dataSetNames - 数据集集合中的数据集名称列表。
 * @param {Array.<string>} [options.fields] - 设置查询结果返回字段。默认返回所有字段。
 * @param {string} [options.attributeFilter] - 属性查询条件。
 * @param {boolean} [options.returnContent=true] - 是否直接返回查询结果。
 * @param {number} [options.fromIndex=0] - 查询结果的最小索引号。
 * @param {number} [options.toIndex=19] - 查询结果的最大索引号。
 * @param {string|number} [options.targetEpsgCode] - 动态投影的目标坐标系对应的 EPSG Code，使用此参数时，returnContent 参数需为 true。
 * @param {Object} [options.targetPrj] - 动态投影的目标坐标系。使用此参数时，returnContent 参数需为 true。如：prjCoordSys={"epsgCode":3857}。当同时设置 targetEpsgCode 参数时，此参数不生效。
 * @extends {GetFeaturesParametersBase}
 * @usage
 */
class GetFeaturesByBufferParameters extends GetFeaturesParametersBase {
    constructor(options) {
        super(options);
        /**
         * @member {number} GetFeaturesByBufferParameters.prototype.bufferDistance
         * @description buffer 距离，单位与所查询图层对应的数据集单位相同。
         */
        this.bufferDistance = null;

        /**
         * @member {string} GetFeaturesByBufferParameters.prototype.attributeFilter
         * @description 属性查询条件。
         */
        this.attributeFilter = null;

        /**
         * @member {GeoJSONObject} GetFeaturesByBufferParameters.prototype.geometry
         * @description 空间查询条件。<br>
         * 点类型可以是：{@link GeometryPoint}|{@link L.Marker}|{@link L.CircleMarker}|{@link L.Circle}|{@link L.GeoJSON}|{@link ol.geom.Point}|{@link ol.format.GeoJSON}|{@link mapboxgl.LngLat}|{@link mapboxgl.Point}|{@link GeoJSONObject}。</br>
         * 线类型可以是：{@link GeometryLineString}|{@link GeometryLinearRing}|{@link L.Polyline}|{@link L.GeoJSON}|{@link ol.geom.LineString}|{@link ol.format.GeoJSON}|{@link GeoJSONObject}。</br>
         * 面类型可以是：{@link GeometryPolygon}|{@link L.Polygon}|{@link L.GeoJSON}|{@link ol.geom.Polygon}|{@link ol.format.GeoJSON}|{@link mapboxgl.LngLatBounds}|{@link GeoJSONObject}。
         */
        this.geometry = null;

        /**
         * @member {Array.<string>} GetFeaturesByBufferParameters.prototype.fields
         * @description 设置查询结果返回字段。当指定了返回结果字段后，则 GetFeaturesResult 中的 features 的属性字段只包含所指定的字段。不设置即返回全部字段。
         */
        this.fields = null;
        Util_Util.extend(this, options);
        this.CLASS_NAME = 'SuperMap.GetFeaturesByBufferParameters';
    }

    /**
     * @function GetFeaturesByBufferParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        me.bufferDistance = null;
        me.attributeFilter = null;
        if (me.fields) {
            while (me.fields.length > 0) {
                me.fields.pop();
            }
            me.fields = null;
        }
        if (me.geometry) {
            me.geometry.destroy();
            me.geometry = null;
        }
    }

    /**
     * @function GetFeaturesByBufferParameters.toJsonParameters
     * @description 将 GetFeaturesByBufferParameters 对象转换为 JSON 字符串。
     * @param {GetFeaturesByBufferParameters} params - 数据集缓冲区查询参数对象。
     * @returns {string} 转化后的 JSON 字符串。
     */
    static toJsonParameters(params) {
        var filterParameter, paramsByBuffer, geometry;
        geometry = ServerGeometry.fromGeometry(params.geometry);
        paramsByBuffer = {
            datasetNames: params.datasetNames,
            getFeatureMode: 'BUFFER',
            bufferDistance: params.bufferDistance,
            geometry: geometry
        };
        if (params.fields) {
            filterParameter = new FilterParameter();
            filterParameter.name = params.datasetNames;
            filterParameter.fields = params.fields;
            paramsByBuffer.queryParameter = filterParameter;
        }
        if (params.attributeFilter) {
            paramsByBuffer.attributeFilter = params.attributeFilter;
            paramsByBuffer.getFeatureMode = 'BUFFER_ATTRIBUTEFILTER';
        }
        if (params.maxFeatures && !isNaN(params.maxFeatures)) {
            paramsByBuffer.maxFeatures = params.maxFeatures;
        }

        if (typeof params.hasGeometry === 'boolean') {
            paramsByBuffer.hasGeometry = params.hasGeometry;
        }

        if (params.targetEpsgCode) {
            paramsByBuffer.targetEpsgCode = params.targetEpsgCode;
        }
        if (!params.targetEpsgCode && params.targetPrj) {
            paramsByBuffer.targetPrj = params.targetPrj;
        }
        return Util_Util.toJSON(paramsByBuffer);
    }
}


;// CONCATENATED MODULE: ./src/common/iServer/GetFeaturesByBufferService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class GetFeaturesByBufferService
 * @deprecatedclass SuperMap.GetFeaturesByBufferService
 * @category iServer Data FeatureResults
 * @classdesc 数据服务中数据集缓冲区查询服务类。
 * @param {string} url - 服务地址。请求数据服务中数据集查询服务，
 * URL 应为：http://{服务器地址}:{服务端口号}/iserver/services/{数据服务名}/rest/data/；
 * 例如："http://localhost:8090/iserver/services/data-jingjin/rest/data/"
 * @param {Object} options - 参数。
 * @param {DataFormat} [options.format=DataFormat.GEOJSON] - 查询结果返回格式，目前支持 iServerJSON、GeoJSON、FGB 三种格式。参数格式为 "ISERVER"，"GEOJSON"，"FGB"。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @extends {GetFeaturesServiceBase}
 * @example
 * var myGetFeaturesByBufferService = new GetFeaturesByBufferService(url);
 * function GetFeaturesCompleted(object){//todo};
 * function GetFeaturesError(object){//todo};
 * @usage
 */

class GetFeaturesByBufferService extends GetFeaturesServiceBase {

    constructor(url, options) {
        super(url, options);

        this.CLASS_NAME = "SuperMap.GetFeaturesByBufferService";
    }

    /**
     * @function GetFeaturesByBufferService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function GetFeaturesByBufferService.prototype.getJsonParameters
     * @description 将查询参数转化为 JSON 字符串。在本类中重写此方法，可以实现不同种类的查询（IDs, SQL, Buffer, Geometry等）。
     * @param {GetFeaturesByBufferParameters} params - 数据集缓冲区查询参数类。
     * @returns {Object} 转化后的 JSON 字符串。
     */
    getJsonParameters(params) {
        if (!(params instanceof GetFeaturesByBufferParameters)) {
            return;
        }
        return GetFeaturesByBufferParameters.toJsonParameters(params);
    }

}

;// CONCATENATED MODULE: ./src/common/iServer/GetFeaturesByGeometryParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/






/**
 * @class GetFeaturesByGeometryParameters
 * @deprecatedclass SuperMap.GetFeaturesByGeometryParameters
 * @category iServer Data FeatureResults
 * @classdesc 数据集几何查询参数类。该类用于设置数据集几何查询的相关参数。
 * @param {Object} options - 参数。
 * @param {GeoJSONObject} options.geometry - 查询的几何对象。
 * @param {Array.<string>} options.datasetNames - 数据集集合中的数据集名称列表。
 * @param {string} [options.attributeFilter] - 几何查询属性过滤条件。
 * @param {Array.<string>} [options.fields] - 设置查询结果返回字段。默认返回所有字段。
 * @param {SpatialQueryMode} [options.spatialQueryMode=SpatialQueryMode.CONTAIN] - 空间查询模式常量。
 * @param {boolean} [options.returnContent=true] - 是否直接返回查询结果。
 * @param {number} [options.fromIndex=0] - 查询结果的最小索引号。
 * @param {number} [options.toIndex=19] - 查询结果的最大索引号。
 * @param {string|number} [options.targetEpsgCode] - 动态投影的目标坐标系对应的 EPSG Code，使用此参数时，returnContent 参数需为 true。
 * @param {Object} [options.targetPrj] - 动态投影的目标坐标系。使用此参数时，returnContent 参数需为 true。如：prjCoordSys={"epsgCode":3857}。当同时设置 targetEpsgCode 参数时，此参数不生效。
 * @param {MetricsAggParameter|GeoHashGridAggParameter} [options.aggregations] - 聚合查询参数。该参数仅支持数据来源 Elasticsearch 服务的Supermap iServer的rest数据服务。
 * @extends {GetFeaturesParametersBase}
 * @usage
 */
class GetFeaturesByGeometryParameters extends GetFeaturesParametersBase {
    constructor(options) {
        super(options);
        /**
         * @member {string} GetFeaturesByGeometryParameters.prototype.getFeatureMode
         * @description 数据集查询模式。几何查询有 "SPATIAL"，"SPATIAL_ATTRIBUTEFILTER" 两种，当用户设置 attributeFilter 时会自动切换到 SPATIAL_ATTRIBUTEFILTER 访问服务。
         */
        this.getFeatureMode = 'SPATIAL';

        /**
         * @member {GeoJSONObject} GetFeaturesByGeometryParameters.prototype.geometry
         * @description 用于查询的几何对象。 </br>
         * 点类型可以是：{@link GeometryPoint}|{@link L.Marker}|{@link L.CircleMarker}|{@link L.Circle}|{@link L.GeoJSON}|{@link ol.geom.Point}|{@link ol.format.GeoJSON}|{@link mapboxgl.LngLat}|{@link mapboxgl.Point}|{@link GeoJSONObject}。</br>
         * 线类型可以是：{@link GeometryLineString}|{@link GeometryLinearRing}|{@link L.Polyline}|{@link L.GeoJSON}|{@link ol.geom.LineString}|{@link ol.format.GeoJSON}|{@link GeoJSONObject}。</br>
         * 面类型可以是：{@link GeometryPolygon}|{@link L.Polygon}|{@link L.GeoJSON}|{@link ol.geom.Polygon}|{@link ol.format.GeoJSON}|{@link mapboxgl.LngLatBounds}|{@link GeoJSONObject}。
         */
        this.geometry = null;

        /**
         * @member {Array.<string>} GetFeaturesByGeometryParameters.prototype.fields
         * @description 设置查询结果返回字段。当指定了返回结果字段后，则 GetFeaturesResult 中的 features 的属性字段只包含所指定的字段。不设置即返回全部字段。
         */
        this.fields = null;

        /**
         * @member {string} GetFeaturesByGeometryParameters.prototype.attributeFilter
         *  @description 几何查询属性过滤条件。
         */
        this.attributeFilter = null;

        /**
         * @member {SpatialQueryMode} [GetFeaturesByGeometryParameters.prototype.spatialQueryMode=SpatialQueryMode.CONTAIN]
         * @description 空间查询模式常量。
         */
        this.spatialQueryMode = SpatialQueryMode.CONTAIN;
        Util_Util.extend(this, options);

        this.CLASS_NAME = 'SuperMap.GetFeaturesByGeometryParameters';
    }

    /**
     * @function GetFeaturesByGeometryParameters.prototype.destroy
     * @description 释放资源，将引用资源的属性置空。
     */
    destroy() {
        super.destroy();
        var me = this;
        if (me.geometry) {
            me.geometry.destroy();
            me.geometry = null;
        }
        if (me.fields) {
            while (me.fields.length > 0) {
                me.fields.pop();
            }
            me.fields = null;
        }
        me.attributeFilter = null;
        me.spatialQueryMode = null;
        me.getFeatureMode = null;
    }

    /**
     * @function GetFeaturesByGeometryParameters.toJsonParameters
     * @description 将 GetFeaturesByGeometryParameters 对象参数转换为 JSON 字符串。
     * @param {GetFeaturesByGeometryParameters} params - 查询参数对象。
     * @returns {string} 转化后的 JSON 字符串。
     */
    static toJsonParameters(params) {
        var filterParameter, geometry, parasByGeometry;

        geometry = ServerGeometry.fromGeometry(params.geometry);
        parasByGeometry = {
            datasetNames: params.datasetNames,
            getFeatureMode: 'SPATIAL',
            geometry: geometry,
            spatialQueryMode: params.spatialQueryMode
        };
        if (params.fields) {
            filterParameter = new FilterParameter();
            filterParameter.name = params.datasetNames;
            filterParameter.fields = params.fields;
            parasByGeometry.queryParameter = filterParameter;
        }
        if (params.attributeFilter) {
            parasByGeometry.attributeFilter = params.attributeFilter;
            parasByGeometry.getFeatureMode = 'SPATIAL_ATTRIBUTEFILTER';
        }

        if (params.maxFeatures && !isNaN(params.maxFeatures)) {
            parasByGeometry.maxFeatures = params.maxFeatures;
        }

        if (typeof params.hasGeometry === 'boolean') {
            parasByGeometry.hasGeometry = params.hasGeometry;
        }

        if (params.targetEpsgCode) {
            parasByGeometry.targetEpsgCode = params.targetEpsgCode;
        }
        if (!params.targetEpsgCode && params.targetPrj) {
            parasByGeometry.targetPrj = params.targetPrj;
        }
        if (params.aggregations) {
            parasByGeometry.aggregations = params.aggregations;
        }

        return Util_Util.toJSON(parasByGeometry);
    }
}


;// CONCATENATED MODULE: ./src/common/iServer/GetFeaturesByGeometryService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class GetFeaturesByGeometryService
 * @deprecatedclass SuperMap.GetFeaturesByGeometryService
 * @category iServer Data FeatureResults
 * @classdesc 数据集几何查询服务类，查询与指定几何对象符合一定空间关系的矢量要素。
 * @param {string} url - 服务地址。请求数据服务中数据集查询服务。
 * URL 应为：http://{服务器地址}:{服务端口号}/iserver/services/{数据服务名}/rest/data；
 * 例如："http://localhost:8090/iserver/services/data-jingjin/rest/data"
 * @param {Object} options - 参数。
 * @param {DataFormat} [options.format=DataFormat.GEOJSON] - 查询结果返回格式，目前支持 iServerJSON、GeoJSON、FGB 三种格式。参数格式为 "ISERVER"，"GEOJSON"，"FGB"。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @extends {GetFeaturesServiceBase}
 * @example
 * var myService = new GetFeaturesByGeometryService(url);
 * function getFeatureCompleted(object){//todo};
 * function getFeatureError(object){//todo}
 * @usage
 */
class GetFeaturesByGeometryService extends GetFeaturesServiceBase {

    constructor(url, options) {
        super(url, options);
        this.CLASS_NAME = "SuperMap.GetFeaturesByGeometryService";
    }

    /**
     * @function GetFeaturesByGeometryService.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
    }

    /**
     * @function GetFeaturesByGeometryService.prototype.getJsonParameters
     * @param {GetFeaturesByGeometryParameters} params - 数据集几何查询参数类。
     * @description 将查询参数转化为 JSON 字符串。
     * 在本类中重写此方法，可以实现不同种类的查询（ID, SQL, Buffer, Geometry等）。
     * @returns {Object} 转化后的 JSON 字符串。
     */
    getJsonParameters(params) {
        return GetFeaturesByGeometryParameters.toJsonParameters(params);
    }


}

;// CONCATENATED MODULE: ./src/common/iServer/GetFeaturesByIDsParameters.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/




/**
 * @class GetFeaturesByIDsParameters
 * @deprecatedclass SuperMap.GetFeaturesByIDsParameters
 * @category  iServer Data FeatureResults
 * @classdesc ID 查询参数类。
 * @param {Object} options - 参数。
 * @param {Array.<number>} options.IDs - 指定查询的元素 ID 信息。
 * @param {Array.<string>} [options.fields] - 设置查询结果返回字段。默认返回所有字段。
 * @param {Array.<string>} options.dataSetNames - 数据集集合中的数据集名称列表。
 * @param {boolean} [options.returnContent=true] - 是否直接返回查询结果。
 * @param {number} [options.fromIndex=0] - 查询结果的最小索引号。
 * @param {number} [options.toIndex=19] - 查询结果的最大索引号。
 * @param {string|number} [options.targetEpsgCode] - 动态投影的目标坐标系对应的 EPSG Code，使用此参数时，returnContent 参数需为 true。
 * @param {Object} [options.targetPrj] - 动态投影的目标坐标系。使用此参数时，returnContent 参数需为 true。如：prjCoordSys={"epsgCode":3857}。当同时设置 targetEpsgCode 参数时，此参数不生效。
 * @param {MetricsAggParameter|GeoHashGridAggParameter} [options.aggregations] - 聚合查询参数。该参数仅支持数据来源 Elasticsearch 服务的Supermap iServer的rest数据服务。
 * @extends {GetFeaturesParametersBase}
 * @usage
 */
class GetFeaturesByIDsParameters extends GetFeaturesParametersBase {
    constructor(options) {
        super(options);
        /**
         * @member {string} GetFeaturesByIDsParameters.prototype.getFeatureMode
         * @description 数据集查询模式。
         */
        this.getFeatureMode = 'ID';

        /**
         * @member {Array.<number>} GetFeaturesByIDsParameters.prototype.IDs
         * @description 所要查询指定的元素 ID 信息。
         */
        this.IDs = null;

        /**
         *  @member {Array.<string>} GetFeaturesByIDsParameters.prototype.fields
         *  @description 设置查询结果返回字段。当指定了返回结果字段后，则 GetFeaturesResult 中的 features 的属性字段只包含所指定的字段。不设置即返回全部字段。
         */
        this.fields = null;
        Util_Util.extend(this, options);

        this.CLASS_NAME = 'SuperMap.GetFeaturesByIDsParameters';
    }

    /**
     * @function GetFeaturesByIDsParameters.prototype.destroy
     * @override
     */
    destroy() {
        super.destroy();
        var me = this;
        me.IDs = null;
        me.getFeatureMode = null;
        if (me.fields) {
            while (me.fields.length > 0) {
                me.fields.pop();
            }
            me.fields = null;
        }
    }

    /**
     * @function GetFeaturesByIDsParameters.toJsonParameters
     * @description 将 GetFeaturesByIDsParameters 对象转换为 JSON 字符串。
     * @param {GetFeaturesByIDsParameters} params - ID 查询参数对象。
     * @returns {string} 转化后的 JSON 字符串。
     */
    static toJsonParameters(params) {
        var parasByIDs, filterParameter;

        parasByIDs = {
            datasetNames: params.datasetNames,
            getFeatureMode: 'ID',
            ids: params.IDs
        };
        if (params.fields) {
            filterParameter = new FilterParameter();
            filterParameter.name = params.datasetNames;
            filterParameter.fields = params.fields;
            parasByIDs.queryParameter = filterParameter;
        }
        if (params.targetEpsgCode) {
            parasByIDs.targetEpsgCode = params.targetEpsgCode;
        }

        if (typeof params.hasGeometry === 'boolean') {
            parasByIDs.hasGeometry = params.hasGeometry;
        }

        if (!params.targetEpsgCode && params.targetPrj) {
            parasByIDs.targetPrj = params.targetPrj;
        }
        if (params.aggregations) {
            parasByIDs.aggregations = params.aggregations;
        }

        return Util_Util.toJSON(parasByIDs);
    }
}


;// CONCATENATED MODULE: ./src/common/iServer/GetFeaturesByIDsService.js
/* Copyright© 2000 - 2023 SuperMap Software Co.Ltd. All rights reserved.
 * This program are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at http://www.apache.org/licenses/LICENSE-2.0.html.*/



/**
 * @class GetFeaturesByIDsService
 * @deprecatedclass SuperMap.GetFeaturesByIDsService
 * @category iServer Data FeatureResults
 * @classdesc 数据集ID查询服务类。在数据集集合中查找指定 ID 号对应的空间地物要素。
 * @param {string} url - 服务地址。请求数据服务中数据集查询服务。
 *                       URL 应为：http://{服务器地址}:{服务端口号}/iserver/services/{数据服务名}/rest/data/；</br>
 *                       例如："http://localhost:8090/iserver/services/data-jingjin/rest/data/"
 * @param {Object} options - 参数。
 * @param {DataFormat} [options.format=DataFormat.GEOJSON] - 查询结果返回格式，目前支持 iServerJSON、GeoJSON、FGB 三种格式。参数格式为 "ISERVER"，"GEOJSON"，"FGB"。
 * @param {boolean} [options.crossOrigin] - 是否允许跨域请求。
 * @param {Object} [options.headers] - 请求头。
 * @extends {GetFeaturesServiceBase}
 * @example
 * var myGetFeaturesByIDsService = new GetFeaturesByIDsService(url);
 * function getFeatureCompleted(object){//todo};
 * function getFeatureError(object){//todo}
 * @usage
 */
class GetFeaturesByIDsService extends GetFeaturesServiceBase {


    constructor(url, options) {
        super(url, options);

        this.CLASS_NAME = "SuperMap.GetFeaturesByIDsService";
    }

    /**
     * @function